/*
 * Argus Software.  Argus files - Input processing
 * Copyright (c) 2000-2006 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * ArgusSource.c - Argus packet source routines.
 *
 * written by Carter Bullard
 * QoSient, LLC
 * Tue Aug  8 08:13:36 EDT 2000
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */


#if !defined(ArgusSource)
#define ArgusSource
#endif

#if defined(__APPLE_CC__) || defined(__APPLE__)
#define PCAP_DONT_INCLUDE_PCAP_BPF_H
#include <sys/ioctl.h>
#include <net/bpf.h>
#endif

#include <ArgusModeler.h>

#include <sys/mman.h>
#include <net/ppp.h>
#if !defined(PPP_HDRLEN)
#define PPP_HDRLEN      4       /* length of PPP header */
#endif


void ArgusGetInterfaceStatus (struct ArgusSourceStruct *src);


struct ArgusSourceStruct *
ArgusNewSource(struct ArgusModelerStruct *model)
{
   struct ArgusSourceStruct *retn =  NULL;
 
   if ((retn = (struct ArgusSourceStruct *) ArgusCalloc (1, sizeof (struct ArgusSourceStruct))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewSource: ArgusCalloc error %s\n", strerror(errno));

   retn->ArgusModel = model;
   retn->Nflag = -1;
   retn->ArgusSnapLen = ARGUS_MINSNAPLEN;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusNewSource() returning 0x%x\n", retn);
#endif

   return (retn);
}


#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


void
ArgusInitSource (struct ArgusSourceStruct *src)
{
   char errbuf[PCAP_ERRBUF_SIZE];
   char *cmdbuf = NULL;
   int type, i;

   src->ArgusInterfaces = 0;
   for (i = 0; i < ARGUS_MAXINTERFACE; i++)
      bzero ((char *)&src->ArgusInterface[i], sizeof(struct ArgusInterfaceStruct));

   if (src->ArgusRfileList != NULL) {
      setuid(getuid());
      bzero(errbuf, sizeof(errbuf));

      if (ArgusOpenInputPacketFile(src, errbuf) == 0) {
      }

   } else {
      if (src->ArgusDeviceList == NULL)
         setArgusDevice (src, pcap_lookupdev (errbuf));

      if (src->ArgusDeviceList) {
         for (i = 0; i < src->ArgusDeviceList->count; i++) {
            struct ArgusDeviceStruct *device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_NOLOCK);

            if ((src->ArgusInterface[i].ArgusPd = pcap_open_live(device->name, src->ArgusSnapLen, !src->Arguspflag, 100, errbuf)) != NULL) {
#if defined(__APPLE_CC__) || defined(__APPLE__)
               int v = 1;
               ioctl(pcap_fileno(src->ArgusInterface[i].ArgusPd), BIOCIMMEDIATE, &v);
#endif
#ifdef ARGUSDEBUG
               ArgusDebug (1, "ArgusInitSource() pcap_open_live(%s) returned 0x%x\n", device->name, src->ArgusInterface[i].ArgusPd);
#endif
               src->ArgusInputPacketFileType = ARGUSLIBPPKTFILE;
               src->ArgusInterface[i].ArgusInterfaceType = ARGUSLIBPPKTFILE;
               src->ArgusInterface[i].ArgusDevice = device->name;
               memset((char *)&src->ArgusInterface[i].ifr, 0, sizeof(src->ArgusInterface[i].ifr));
               strncpy(src->ArgusInterface[i].ifr.ifr_name, device->name, sizeof(src->ArgusInterface[i].ifr.ifr_name));

               if (!((pcap_lookupnet (device->name, (u_int *)&src->ArgusInterface[i].ArgusLocalNet,
                                                      (u_int *)&src->ArgusInterface[i].ArgusNetMask, errbuf)) < 0)) {
#if defined(_LITTLE_ENDIAN)
                  src->ArgusInterface[i].ArgusLocalNet = ntohl(src->ArgusInterface[i].ArgusLocalNet);
                  src->ArgusInterface[i].ArgusNetMask  = ntohl(src->ArgusInterface[i].ArgusNetMask);
#endif
               }

               src->ArgusInterface[i].ArgusCallBack = Arguslookup_pcap_callback(type = pcap_datalink(src->ArgusInterface[i].ArgusPd));

            } else
               ArgusLog (LOG_ERR, "ArgusInitSource: pcap_open_live %s\n", errbuf);

            src->ArgusInterfaces++;
            ArgusPushBackList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_NOLOCK);
         }
      }

      setuid(getuid());
   }

   if (src->ArgusInterface[0].ArgusPd == NULL)
      ArgusLog (LOG_ERR, "ArgusInitSource: no input source\n");

   cmdbuf = ArgusCopyArgv(&src->ArgusArgv[src->ArgusOptind]);

   if (cmdbuf) {
      if (src->ArgusInputFilter)
         ArgusFree(src->ArgusInputFilter);

      src->ArgusInputFilter = cmdbuf;
   }

   bzero ((char *) &src->ArgusInterface[0].ArgusFilter, sizeof (struct bpf_program));

   if (pcap_compile (src->ArgusInterface[0].ArgusPd, &src->ArgusInterface[0].ArgusFilter, src->ArgusInputFilter, getArgusOflag(src), src->ArgusInterface[0].ArgusNetMask) < 0)
      ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));

   if (src->Argusbpflag) {
      Argusbpf_dump (&src->ArgusInterface[0].ArgusFilter, src->Argusbpflag);
      exit(0);
   }

   for (i = 0; i < src->ArgusInterfaces; i++) {
      if (src->ArgusInterface[i].ArgusPd) {
         if (src->ArgusInputPacketFileType == ARGUSLIBPPKTFILE) {
            if (src->ArgusInputFilter != NULL) {
               if (pcap_setfilter (src->ArgusInterface[i].ArgusPd, &src->ArgusInterface[0].ArgusFilter) < 0)
                  ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[i].ArgusPd));
            }
         }
      }
   }

   if (src->ArgusWriteOutPacketFile) {
      if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
         ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusInitSource() returning\n");
#endif
}


int
ArgusCloseSource(struct ArgusSourceStruct *src)
{

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseSource(0x%x) starting\n", src);
#endif

   if (src) {
      if (src->ArgusPcapOutFile)
         pcap_dump_close(src->ArgusPcapOutFile);

      if (src->ArgusInputFilter)
         ArgusFree (src->ArgusInputFilter);

      if (src->ArgusDeviceList)
         ArgusDeleteList(src->ArgusDeviceList, ARGUS_DEVICE_LIST);

      if (src->ArgusRfileList != NULL)
         ArgusDeleteList (src->ArgusRfileList, ARGUS_RFILE_LIST);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseSource(0x%x) deleting source\n", src);
#endif
   return (0);
}


unsigned char
getArgusInterfaceStatus(struct ArgusSourceStruct *src) {
   return (src->ArgusInterfaceStatus);
}
 
void
setArgusInterfaceStatus(struct ArgusSourceStruct *src, unsigned char value)
{
   src->ArgusInterfaceStatus = value;
 
#ifdef ARGUSDEBUG
   ArgusDebug (1, "setArgusInterfaceStatus(%d)\n", value);
#endif
}

 
int
getArgusSnapLen(struct ArgusSourceStruct *src)
{
   return (src->ArgusSnapLen);
}
 
void
setArgusSnapLen(struct ArgusSourceStruct *src, int value)
{
   src->ArgusSnapLen = value;
}

int
getArgusbpflag(struct ArgusSourceStruct *src)
{
   return (src->Argusbpflag);
}

int
getArguspflag(struct ArgusSourceStruct *src)
{
   return (src->Arguspflag);
}

int
getArgusOflag(struct ArgusSourceStruct *src)
{
   return (src->ArgusOflag);
}

void
setArgusbpflag(struct ArgusSourceStruct *src, int value)
{
   src->Argusbpflag = value;
}

void
setArguspflag(struct ArgusSourceStruct *src, int value)
{
   src->Arguspflag = value;
}

void
setArgusOflag(struct ArgusSourceStruct *src, int value)
{
   src->ArgusOflag = value;
}

void
setArgusArgv(struct ArgusSourceStruct *src, char **value)
{
   src->ArgusArgv = value;
}

void
setArgusOptind (struct ArgusSourceStruct *src, int value)
{
   src->ArgusOptind = value;
}

char *
getArgusDevice (struct ArgusSourceStruct *src)
{
   struct ArgusDeviceStruct *device = NULL;
   char *retn = NULL;

   if (src->ArgusDeviceList != NULL)
      if ((device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_NOLOCK)) != NULL)
         ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_NOLOCK);

   if (device != NULL)
      retn = device->name;
#ifdef ARGUSDEBUG
   ArgusDebug (3, "getArgusDevice() returning %s\n", retn);
#endif
   return (retn);
}


void
setArgusDevice (struct ArgusSourceStruct *src, char *value)
{
   if (src->ArgusDeviceList == NULL)
      src->ArgusDeviceList = ArgusNewList();

   if (value) {
      struct ArgusDeviceStruct *device = NULL;
      if ((device = (struct ArgusDeviceStruct *) ArgusCalloc(1, sizeof(*device))) == NULL)
         ArgusLog (LOG_ERR, "setArgusDevice ArgusCalloc %s\n", strerror(errno));

      device->name = strdup(value);
      ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_NOLOCK);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "setArgusDevice(%s) returning\n", value);
#endif
}

void
clearArgusDevice (struct ArgusSourceStruct *src)
{
   if (src->ArgusDeviceList != NULL) {
      ArgusDeleteList(src->ArgusDeviceList, ARGUS_DEVICE_LIST);
      src->ArgusDeviceList = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "clearArgusDevice(0x%x) returning\n", src);
#endif
}


char *
getArgusrfile (struct ArgusSourceStruct *src)
{
   struct ArgusRfileStruct *rfile = NULL;
   char *retn = NULL;

   if (src->ArgusRfileList != NULL) {
      rfile = (struct ArgusRfileStruct *) src->ArgusRfileList->start;
      retn = rfile->name;
   }
   return (retn);
}

void
setArgusrfile (struct ArgusSourceStruct *src, char *value)
{
   if (src->ArgusRfileList == NULL)
      src->ArgusRfileList = ArgusNewList();

   if (value) {
      struct ArgusRfileStruct *rfile;

      if ((rfile = (struct ArgusRfileStruct *) ArgusCalloc(1, sizeof(*rfile))) == NULL)
         ArgusLog (LOG_ERR, "setArgusrfile ArgusCalloc %s\n", strerror(errno));

      rfile->name = strdup(value);
      ArgusPushFrontList(src->ArgusRfileList, (struct ArgusListRecord *) rfile, ARGUS_NOLOCK);
   }
}

int
getArgusMoatTshFile (struct ArgusSourceStruct *src)
{
   return(src->Argustflag);
}

void
setArgusMoatTshFile (struct ArgusSourceStruct *src, int value)
{
   src->Argustflag = value;
}

float
getArgusRealTime (struct ArgusSourceStruct *src)
{
   return(src->Tflag);
}


void
setArgusRealTime (struct ArgusSourceStruct *src, float value)
{
   src->Tflag = value;
}


void
setArgusWriteOutPacketFile (struct ArgusSourceStruct *src, char *file)
{
   src->ArgusWriteOutPacketFile = strdup(file);
}


#define ARGUSMOATLEN      44
#define ARGUSMOATTSHTCPLEN   40

int ArgusMoatTshRead (struct ArgusSourceStruct *);

int
ArgusMoatTshRead (struct ArgusSourceStruct *src)
{
   struct ArgusMoatTshPktHdr MoatTshBuffer[2], *ArgusMoatPktHdr = &MoatTshBuffer[0];
   int retn = 0, length = 0;
   struct ip *iphdr = NULL;

   bzero (ArgusMoatPktHdr, sizeof(MoatTshBuffer));
 
   if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), ArgusMoatPktHdr, ARGUSMOATLEN)) == ARGUSMOATLEN) {
      ArgusMoatPktHdr->interface = 0;
#if defined(_LITTLE_ENDIAN)
      src->ArgusModel->ArgusGlobalTime.tv_sec  = ntohl(ArgusMoatPktHdr->sec);
      src->ArgusModel->ArgusGlobalTime.tv_usec = ntohl(*((int *)&ArgusMoatPktHdr->interface));
#else
      src->ArgusModel->ArgusGlobalTime.tv_sec  = ArgusMoatPktHdr->sec;
      src->ArgusModel->ArgusGlobalTime.tv_usec = *((int *)&ArgusMoatPktHdr->interface);
#endif

      iphdr = &ArgusMoatPktHdr->ip;

#if defined(_LITTLE_ENDIAN)
      length = ntohs(iphdr->ip_len);
#else
      length = iphdr->ip_len;
#endif
      src->ArgusThisLength  = length;

      switch (iphdr->ip_p) {
         case IPPROTO_ICMP:
         case IPPROTO_TCP:
         default:
            src->ArgusSnapLength  = ARGUSMOATTSHTCPLEN;
            break;
      }

      src->ArgusThisSnapEnd = (((unsigned char *)iphdr) + src->ArgusSnapLength);

      if ((src->ArgusInputFilter == NULL) ||
           (bpf_filter(src->ArgusInterface[0].ArgusFilter.bf_insns, (u_char *)iphdr, src->ArgusSnapLength, src->ArgusSnapLen))) {

         ArgusProcessIpPacket (src->ArgusModel, iphdr, length, &src->ArgusModel->ArgusGlobalTime);
      }

   } else
      close(pcap_fileno(src->ArgusInterface[0].ArgusPd));

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusMoatTshRead() returning %d\n", retn);
#endif

   return (retn);
}


int
ArgusSnoopRead (struct ArgusSourceStruct *src)
{
   int retn = 0, len = 0;
   struct pcap_pkthdr pcap_pkthdr;
   struct ArgusSnoopPktHdr SnoopPktHdr;
   unsigned char ArgusPacketBuf[1540];

   if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), &SnoopPktHdr, sizeof(SnoopPktHdr))) == sizeof(SnoopPktHdr)) {
#if defined(_LITTLE_ENDIAN)
      SnoopPktHdr.len            = ntohl(SnoopPktHdr.len);
      SnoopPktHdr.tlen           = ntohl(SnoopPktHdr.tlen);
      SnoopPktHdr.argtvp.tv_sec  = ntohl(SnoopPktHdr.argtvp.tv_sec);
      SnoopPktHdr.argtvp.tv_usec = ntohl(SnoopPktHdr.argtvp.tv_usec);
#endif
      if ((len = ((SnoopPktHdr.tlen + 3) & 0xFFFFFFC)) < 1500) {
         if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), ArgusPacketBuf, len)) == len) {
            pcap_pkthdr.ts.tv_sec  = SnoopPktHdr.argtvp.tv_sec;
            pcap_pkthdr.ts.tv_usec = SnoopPktHdr.argtvp.tv_usec;
            pcap_pkthdr.caplen = SnoopPktHdr.tlen;
            pcap_pkthdr.len    = SnoopPktHdr.len;

            if ((src->ArgusInputFilter == NULL) ||
               (bpf_filter(src->ArgusInterface[0].ArgusFilter.bf_insns, ArgusPacketBuf, SnoopPktHdr.tlen, src->ArgusSnapLen))) {
 
               src->ArgusInterface[0].ArgusCallBack (NULL, &pcap_pkthdr, ArgusPacketBuf);
            }
         }
      }

   } else
      close(pcap_fileno(src->ArgusInterface[0].ArgusPd));

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusSnoopRead() returning %d\n", retn);
#endif

   return (retn);
}


pcap_handler
Arguslookup_pcap_callback (int type)
{
   pcap_handler retn = NULL;
   struct callback *callback;
   char *name = NULL;
 
   for (callback = callbacks; callback->function; ++callback)
      if (type == callback->type) {
         retn = callback->function;
         name = callback->fname;
         break;
      }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "Arguslookup_pcap_callback(%d) returning %s: 0x%x\n", type, name, retn);
#endif

   return (retn);
}

int ArgusProcessPacket (struct ArgusModelerStruct *, char *, int, struct timeval *, int);

void
ArgusEtherPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;
   struct stat statbuf;

   if (p != NULL) {
      unsigned int ind = src->ArgusThisIndex;

      if ((src->ArgusInterfaces > 1) && ((tvp->tv_sec   < src->lasttime.tv_sec) ||
                                        ((tvp->tv_sec  == src->lasttime.tv_sec) &&
                                        (tvp->tv_usec  < src->lasttime.tv_usec)))) {
         src->ArgusInterface[ind].state |= ARGUS_HOLDING;
      }

      src->lasttime = *tvp;

      if (src->ArgusWriteOutPacketFile) {
         if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
            if (src->ArgusPcapOutFile != NULL) {
               pcap_dump_close(src->ArgusPcapOutFile);
               src->ArgusPcapOutFile = NULL;
            }
   
            if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
               ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
         }

         if (src->ArgusPcapOutFile != NULL)
            pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
      }

      if (p && (length >= sizeof(struct ether_header))) {
         struct ether_header *ep;
 
         src->ArgusInterface[ind].ArgusTotalPkts++;
         src->ArgusInterface[ind].ArgusTotalBytes += length;
         src->ArgusInterface[ind].ArgusPacketBuffer = (u_char *) p;
         ep = (struct ether_header *) p;
 
         src->ArgusModel->ArgusGlobalTime  = *tvp;
         src->ArgusModel->ArgusThisLength  = length;
         src->ArgusModel->ArgusSnapLength  = caplen;
         src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;

         ArgusProcessPacket (src->ArgusModel, (char *)ep, length, tvp, ARGUS_ETHER_HDR);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusEtherPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#define TYPE_LEGACY       0
#define TYPE_HDLC_POS     1
#define TYPE_ETH          2
#define TYPE_ATM          3
#define TYPE_AAL5         4

void
ArgusTokenPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusTokenPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#include <net/arcnet.h>

void
ArgusArcnetPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct ether_header *ep = (struct ether_header *) src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBufferBuffer;
   struct arc_header *ap = (struct arc_header *) p;
   u_char arc_type = ap->arc_type;
   int flag = 0, archdrlen = 0;
   struct stat statbuf;

   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   if (src->ArgusModel->ArgusGlobalTime.tv_sec < 0) {
      ArgusLog (LOG_ERR, "ArgusArcnetPacket (0x%x, 0x%x, 0x%x) libpcap timestamp out of range %d.%d\n",
              user, h, p, src->ArgusModel->ArgusGlobalTime.tv_sec, src->ArgusModel->ArgusGlobalTime.tv_usec);
   }

   src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBuffer = src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBufferBuffer; 

   switch (arc_type) {
      case ARCTYPE_IP_OLD:
      case ARCTYPE_ARP_OLD:
      case ARCTYPE_DIAGNOSE:
         archdrlen = ARC_HDRLEN;
         break;

      default:
         if (ap->arc_flag == 0xff) {
            archdrlen = ARC_HDRNEWLEN_EXC;
            flag = ap->arc_flag2;
         } else {
            archdrlen = ARC_HDRNEWLEN;
            flag = ap->arc_flag;
         }
         break;
   }

   length -= archdrlen;
   caplen -= archdrlen;
   p += archdrlen;
   
   bcopy (p, (char *)ep, caplen);

   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;

   ArgusProcessPacket (src->ArgusModel, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusArcnetPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusAtmClipPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusAtmClipPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusLoopPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusLoopPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusHdlcPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusHdlcPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusPppHdlcPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;
   int ind = src->ArgusThisIndex;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusInterface[ind].ArgusTotalPkts++;
   src->ArgusInterface[ind].ArgusTotalBytes += length;
   src->ArgusInterface[ind].ArgusPacketBuffer = (u_char *) p;

   src->ArgusModel->ArgusGlobalTime = *tvp;

   if (p && length) {
      int offset = 0, link = 0;
      unsigned short proto = 0;
      unsigned short value;

#if defined(_LITTLE_ENDIAN)
      value = ntohs(*(unsigned short *)p);
#else
      value = (*(unsigned short *)p);
#endif
      if (value == 0xFF03) {
         link = DLT_PPP_SERIAL;
         if ((p[0] == PPP_ADDRESS) && (p[1] == PPP_CONTROL)) {
            p += 2; length -= 2;
         }
         if (*p & 01) {
            proto = *p; p++; length -= 1;
         } else {
#if defined(_LITTLE_ENDIAN)
            proto = ntohs(*(u_short *)p);
#else
            proto = *(u_short *)p;
#endif
            p += 2; length -= 2;
         }
      } else {
         link = DLT_CHDLC;
#if defined(_LITTLE_ENDIAN)
         proto = ntohs(*(u_short *)&p[2]);
#else
         proto = *(u_short *)p;
#endif

#if !defined(CHDLC_HDRLEN)
#define CHDLC_HDRLEN		4
#endif
         offset = CHDLC_HDRLEN;
         p = (unsigned char *) (src->ArgusInterface[ind].ArgusPacketBuffer + offset);
         length -= offset;
      }

      src->ArgusModel->ArgusThisLength  = length;
      src->ArgusModel->ArgusSnapLength  = caplen;
      src->ArgusModel->ArgusThisSnapEnd = (src->ArgusInterface[ind].ArgusPacketBuffer + caplen);

      switch (proto) {
         case ETHERTYPE_IP:      /*XXX*/
         case PPP_IP:
            ArgusProcessPacket (src->ArgusModel, (char *)p, length, tvp, ETHERTYPE_IP);
            break;

         case PPP_IPV6:
         case ETHERTYPE_IPV6:    /*XXX*/
            ArgusProcessPacket (src->ArgusModel, (char *)p, length, tvp, ETHERTYPE_IPV6);
            break;

         case PPP_MPLS_UCAST:
         case ETHERTYPE_MPLS:
            ArgusProcessPacket (src->ArgusModel, (char *)p, length, tvp, ETHERTYPE_MPLS);
            break;

         case PPP_MPLS_MCAST:
         case ETHERTYPE_MPLS_MULTI:
            ArgusProcessPacket (src->ArgusModel, (char *)p, length, tvp, ETHERTYPE_MPLS_MULTI);
            break;

         case PPP_OSI:
         case PPP_NS:
         case PPP_DECNET:
         case PPP_APPLE:
         case PPP_IPX:
         case PPP_VJC:
         case PPP_VJNC:
         case PPP_BRPDU:
         case PPP_STII:
         case PPP_VINES:
         case PPP_COMP:
         case PPP_HELLO:
         case PPP_LUXCOM:
         case PPP_SNS:
         case PPP_IPCP:
         case PPP_OSICP:
         case PPP_NSCP:
         case PPP_DECNETCP:
         case PPP_APPLECP:
         case PPP_IPXCP:
         case PPP_STIICP:
         case PPP_VINESCP:
         case PPP_IPV6CP:
         case PPP_CCP:
         case PPP_PAP:
         case PPP_LQM:
         case PPP_CHAP:
         case PPP_BACP:
         case PPP_BAP:
         case PPP_MP:

         default:
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppHdlcPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusPppEtherPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppEtherPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
Argus802_11Packet (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "Argus802_11Packet (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusLtalkPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusLtalkPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


void
ArgusIpPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   struct ip *ip = (struct ip *) p;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *) ip) + caplen;

   if (p) {
      src->ArgusModel->ArgusThisLength  = length;
      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }


#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusIpPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}



#if defined(ultrix) || defined(__alpha)
static int   fddi_bitswap = 0;
#else
static int   fddi_bitswap = 1;
#endif

int fddipad = FDDIPAD;

#define FDDI_HDRLEN (sizeof(struct fddi_header))

static u_char fddi_bit_swap[] = {
   0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
   0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
   0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
   0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
   0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
   0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
   0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
   0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
   0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
   0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
   0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
   0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
   0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
   0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
   0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
   0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
   0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
   0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
   0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
   0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
   0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
   0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
   0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
   0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
   0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
   0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
   0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
   0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
   0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
   0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
   0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
   0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff,
};

static inline void
Argusextract_fddi_addrs(const struct fddi_header *fp, struct ether_header *ehdr)
{
   char *fsrc = (char *)&ehdr->ether_shost;
   char *fdst = (char *)&ehdr->ether_dhost;
   int i;

   if (fddi_bitswap) {
      for (i = 0; i < 6; ++i)
         fdst[i] = fddi_bit_swap[fp->fddi_dhost[i]];
      for (i = 0; i < 6; ++i)
         fsrc[i] = fddi_bit_swap[fp->fddi_shost[i]];
   }
   else {
      bcopy ((char *) fp->fddi_dhost, fdst, 6);
      bcopy ((char *) fp->fddi_shost, fsrc, 6);
   }
}

int
ArgusCreatePktFromFddi(const struct fddi_header *fp, struct ether_header *ep, int length)
{
   unsigned char *ptr;
   unsigned int retn = 0;
   struct llc *llc;
 
   if ((fp->fddi_fc & FDDIFC_CLFF) == FDDIFC_LLC_ASYNC) {
      Argusextract_fddi_addrs (fp, ep);

      llc = (struct llc *)(fp + 1);
 
      if (llc->ssap == LLCSAP_SNAP && llc->dsap == LLCSAP_SNAP && llc->llcui == LLC_UI) {
         ((struct ether_header *) ep)->ether_type = EXTRACT_16BITS(&llc->ethertype[0]);
         ptr = (unsigned char *)(llc + 1);
         length -= (sizeof(struct fddi_header) + sizeof(struct llc));
         bcopy ((char *)ptr, (char *)(ep + 1), length);
         retn = length + sizeof(struct ether_header);
      }
   }

   return (retn);
}

void
ArgusFddiPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   const struct fddi_header *fp = (struct fddi_header *)p;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   int ind = src->ArgusThisIndex;
   struct ether_header *ep;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   ep = (struct ether_header *) src->ArgusInterface[ind].ArgusPacketBuffer;

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;

   src->ArgusInterface[ind].ArgusPacketBuffer = src->ArgusInterface[ind].ArgusPacketBufferBuffer;
   if (p && (length = ArgusCreatePktFromFddi(fp, ep, length))) {
      if (p && length)
         ArgusProcessPacket (src->ArgusModel, (char *)ep, length, tvp, ARGUS_ETHER_HDR);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusFddiPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#define ARGUS_802_6_MAC_HDR_LEN      20
#define ARGUS_ATM_HDR_OFFSET      8

void
ArgusATMPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;
   int ind = src->ArgusThisIndex;
   struct ether_header *ep;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   ep = (struct ether_header *) src->ArgusInterface[ind].ArgusPacketBuffer;

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusInterface[ind].ArgusPacketBuffer = src->ArgusInterface[ind].ArgusPacketBufferBuffer;

   if (caplen > 8) {
      if (p[0] != 0xaa || p[1] != 0xaa || p[2] != 0x03) {
         if (caplen > 28) {
            p += ARGUS_802_6_MAC_HDR_LEN;
            length -= ARGUS_802_6_MAC_HDR_LEN;
            caplen -= ARGUS_802_6_MAC_HDR_LEN;
         } else
            return;
      }
   } else
      return;
   
   ep->ether_type = ((p[6] << 8) | p[7]);
   length -= ARGUS_ATM_HDR_OFFSET;
   caplen -= ARGUS_ATM_HDR_OFFSET;
   p += ARGUS_ATM_HDR_OFFSET;
   
   bcopy (p, (char *)(ep + 1), caplen);

   length += sizeof(*ep);
   caplen += sizeof(*ep);

   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;

   ArgusProcessPacket (src->ArgusModel, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusATMPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


void
ArgusPppPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   struct ip *ip = (struct ip *) (p + PPP_HDRLEN);
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ip) + (caplen - PPP_HDRLEN);

   if (p && (length > PPP_HDRLEN)) {
      length -= PPP_HDRLEN;

      src->ArgusModel->ArgusThisLength  = length;

      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}



#define ARGUS_PPPBSDOS_HDR_LEN       24


void
ArgusPppBsdosPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned int length = h->len, hdrlen = 0;
   unsigned int caplen = h->caplen;
   unsigned short ptype = 0;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = (u_char *) p + caplen;

   if (p[0] == PPP_ADDRESS && p[1] == PPP_CONTROL) {
      p += 2;
      hdrlen = 2;
   }

   if (*p & 01) {                  /* Retrieve the protocol type */
      ptype = *p;                  /* Compressed protocol field */
      p++;
      hdrlen += 1;
   } else {
#if defined(_LITTLE_ENDIAN)
      ptype = ntohs(*(u_short *)p);
#else
      ptype = *(u_short *)p;
#endif
      p += 2;
      hdrlen += 2;
   }

   length -= hdrlen;
   if (ptype == PPP_IP)
      if (p && (length > 0)) {
         src->ArgusModel->ArgusThisLength  = length;
         ArgusProcessIpPacket (src->ArgusModel, (struct ip *) p, length, tvp);
      }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppBsdosPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#if defined(__NetBSD__) || defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/mbuf.h>
#endif

#include <net/slcompress.h>
#include <net/slip.h>


/* XXX BSD/OS 2.1 compatibility */

#if !defined(ARGUS_SLIP_HDR_LEN) && defined(SLC_BPFHDR)
#define SLIP_HDRLEN SLC_BPFHDR
#define SLX_DIR 0
#define SLX_CHDR (SLC_BPFHDRLEN - 1)
#define CHDR_LEN (SLC_BPFHDR - SLC_BPFHDRLEN)
#else

#endif


void
ArgusSlipPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval *tvp = (struct timeval *) &h->ts;
   struct ip *ip = (struct ip *) (p + SLIP_HDRLEN);
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ip) + (caplen - SLIP_HDRLEN);

   if (p && (length > SLIP_HDRLEN)) {
      length -= SLIP_HDRLEN;

      src->ArgusModel->ArgusThisLength  = length;

      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }

 
#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusSlipPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#include <sll.h>

#if !defined(ETHER_ADDR_LEN)
#define ETHER_ADDR_LEN  6
#endif


void
ArgusSllPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   const struct sll_header *sllp = NULL;
   u_short pkttype;
   struct timeval *tvp = (struct timeval *) &h->ts;
   unsigned char buf[2048];
   struct ether_header *ep = (struct ether_header *)buf;
   struct stat statbuf;

   src->ArgusModel->ArgusGlobalTime  = *tvp;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   sllp = (const struct sll_header *)p;
   memcpy((void *)&ep->ether_shost, sllp->sll_addr, ETHER_ADDR_LEN);

#if defined(_LITTLE_ENDIAN)
   pkttype = ntohs(sllp->sll_pkttype);
#else
   pkttype = ntohs(sllp->sll_pkttype);
#endif

   if (pkttype != LINUX_SLL_OUTGOING) {
      if (pkttype == LINUX_SLL_BROADCAST)
         memset((void *)&ep->ether_dhost, 0xFF, ETHER_ADDR_LEN);
      else {
         memset((void *)&ep->ether_dhost, 0, ETHER_ADDR_LEN);
         if (pkttype == LINUX_SLL_MULTICAST)
#if defined(HAVE_SOLARIS)
            ep->ether_dhost.ether_addr_octet[0] = 0x01;
#else
            ep->ether_dhost[0] = 0x01;
#endif
         else
#if defined(HAVE_SOLARIS)
            ep->ether_dhost.ether_addr_octet[ETHER_ADDR_LEN-1] = 0x01;
#else
            ep->ether_dhost[ETHER_ADDR_LEN-1] = 0x01;
#endif
      }
   } else {
      /*
       * We sent this packet; we don't know whether it's
       * broadcast, multicast, or unicast, so just make
       * the destination address all 0's.
       */
      memset((void *)&ep->ether_dhost, 0, ETHER_ADDR_LEN);
   }

   length -= SLL_HDR_LEN;
   caplen -= SLL_HDR_LEN;
   p += SLL_HDR_LEN;
 
   ep->ether_type = sllp->sll_protocol;
 
   memcpy((ep + 1), p, caplen);

   src->ArgusModel->ArgusThisSnapEnd = (unsigned char *)(ep + caplen);
   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;

   ArgusProcessPacket (src->ArgusModel, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusSllPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#include <sys/ioctl.h>

#if defined(HAVE_SOLARIS)
#include <sys/sockio.h>
#endif



void
ArgusGetPackets (struct ArgusSourceStruct *src)
{
   fd_set ArgusReadMask, ArgusWriteMask, ArgusExceptMask;
   int tmp, i, width = 0, noerror = 1, fd;
   struct timeval wait;

   if (src != NULL) {
#ifdef ARGUSDEBUG
      ArgusDebug (4, "ArgusGetPackets (0x%x) starting\n", src);
#endif

#if defined(HAVE_SOLARIS)
      sigignore (SIGPIPE);
#else
      signal (SIGPIPE, SIG_IGN);
#endif

      FD_ZERO(&ArgusReadMask);
      FD_ZERO(&ArgusWriteMask);
      FD_ZERO(&ArgusExceptMask);

      wait.tv_sec = 0; wait.tv_usec = 20000;

      ArgusGetInterfaceStatus(src);
      gettimeofday (&src->ArgusStartTime, 0L);

      if (src->ArgusInterface[0].ArgusPd) {
         int found = 0, up = 0;

         for (i = 0; i < src->ArgusInterfaces; i++) {
            if (src->ArgusInterface[i].ArgusPd && (fd = pcap_fileno(src->ArgusInterface[i].ArgusPd) >= 0)) {
               found++;

               if ((src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) != 0) {
                  up++;
                  setArgusInterfaceStatus(src, 1);
                  FD_SET(pcap_fileno(src->ArgusInterface[i].ArgusPd), &ArgusReadMask);
                  if (width < fd)
                     width = fd;
               } else
                  setArgusInterfaceStatus(src, 0);
            }
         }

         if (!(src->ArgusReadingOffLine)) {
            if (found) {
               do {
                  if (up && ((tmp = select (width + 1, &ArgusReadMask, NULL, NULL, &wait)) >= 0)) {
                     found = 0;
#ifdef ARGUSDEBUG
                     ArgusDebug (10, "ArgusGetPackets: select() returned %d\n", tmp);
#endif
                     if (tmp > 0) {
                        for (i = 0; i < src->ArgusInterfaces; i++) {
                           if ((src->ArgusInterface[i].ArgusPd) && ((fd = pcap_fileno(src->ArgusInterface[i].ArgusPd)) >= 0)) {
                              if (FD_ISSET(fd, &ArgusReadMask)) {
                                 found++;
                                 src->ArgusThisIndex = i;
                                 switch (src->ArgusInterface[i].ArgusInterfaceType) {
                                    case ARGUSLIBPPKTFILE:
                                       if ((pcap_dispatch (src->ArgusInterface[i].ArgusPd, 1, src->ArgusInterface[i].ArgusCallBack, (u_char *)src)) < 0) {
                                          if (!(strncmp (pcap_geterr(src->ArgusInterface[i].ArgusPd), "recvfrom", 8))) {
#ifdef ARGUSDEBUG
                                             ArgusDebug (3, "ArgusGetPackets: pcap_dispatch() returned %s\n", pcap_geterr(src->ArgusInterface[i].ArgusPd));
#endif

                                          } else
                                             noerror = 0;
                                       } else {
#ifdef ARGUSDEBUG
                                          ArgusDebug (9, "ArgusGetPackets: pcap_dispatch() interface %s %d up\n", src->ArgusInterface[i].ArgusDevice, up);
#endif
                                       }
                                       break;
          
                                    case ARGUSSNOOPKTFILE:
                                       if (ArgusSnoopRead (src) < 0)
                                          noerror = 0;
                                       break;

                                    case ARGUSMOATTSHPKTFILE:
                                       if (ArgusMoatTshRead (src) < 0)
                                          noerror = 0;
                                       break;

                                 }
                                 if (src->Nflag > 0)
                                    src->Nflag--;
                              }
                           }
                        }

                     } else {
                        gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
#ifdef ARGUSDEBUG
                        ArgusDebug (9, "ArgusGetPackets: select() timeout %d up interfaces\n", up);
#endif
                     }

                  } else {
                     gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
                     if (up) {
#ifdef ARGUSDEBUG
                        ArgusDebug (3, "ArgusGetPackets: select() returned %s\n", strerror(errno));
#endif
                     } else {
#ifdef ARGUSDEBUG
                        ArgusDebug (5, "ArgusGetPackets: no interfaces up: sleeping\n");
#endif
                        usleep(20000);
                     }
                  }
         
                  width = 0;
                  found = 0;
                  up = 0;
                  FD_ZERO(&ArgusReadMask);

                  for (i = 0; i < src->ArgusInterfaces; i++) {
                     if (src->ArgusInterface[i].ArgusPd && (pcap_fileno(src->ArgusInterface[i].ArgusPd) >= 0)) {
                        found++;

                        if (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) {
                           up++;
                           FD_SET(pcap_fileno(src->ArgusInterface[i].ArgusPd), &ArgusReadMask);
                           if (width < pcap_fileno(src->ArgusInterface[i].ArgusPd))
                              width = pcap_fileno(src->ArgusInterface[i].ArgusPd);
                        }
                     }
                  }
         
                  if (!found)
                     break;

                  wait.tv_sec = 0; wait.tv_usec = 20000;

                  if (ArgusUpdateTime (src->ArgusModel)) {
                     ArgusGetInterfaceStatus(src);
                     ArgusQueueManager(ArgusModel);
#if !defined(ARGUS_THREADS)
                     ArgusOutputProcess(ArgusOutputTask);
#endif
                  }

               } while (noerror && (src->Nflag != 0) && (!(ArgusShutDownStarted)));
            }
      
         } else {
            for (i = 0; i < src->ArgusInterfaces; i++) {
               src->ArgusThisIndex = i;
               pcap_offline_read (src->ArgusInterface[i].ArgusPd, src->Nflag,
                          src->ArgusInterface[i].ArgusCallBack, (u_char *) src);
            }
         }
      }

      setArgusFarReportInterval (ArgusModel, "0");
      ArgusQueueManager(ArgusModel);
#if !defined(ARGUS_THREADS)
      ArgusOutputProcess(ArgusOutputTask);
#endif
   }

   gettimeofday (&src->ArgusEndTime, 0L);

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusGetPackets () returning\n");
#endif
}


void
Argusbpf_dump(struct bpf_program *p, int option)
{
   struct bpf_insn *insn;
   int i, n = p->bf_len;

   insn = p->bf_insns;
   if (option > 2) {
      fprintf(stdout, "%d\n", n);
      for (i = 0; i < n; ++insn, ++i) {
         fprintf(stdout, "%lu %lu %lu %lu\n", (long) insn->code,
                (long) insn->jt, (long) insn->jf, (long) insn->k);
      }
      return;
   }
   if (option > 1) {
      for (i = 0; i < n; ++insn, ++i) {
         fprintf(stdout, "{ 0x%x, %d, %d, 0x%08x },\n",
                insn->code, insn->jt, insn->jf, (int) insn->k);
      }
      return;
   }

   for (i = 0; i < n; ++insn, ++i)
      fprintf (stdout, "%s\n", bpf_image(insn, i));
}




#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
 
#define ARGUSSNOOPTAG  "snoop"
 
int
ArgusOpenInputPacketFile(struct ArgusSourceStruct *src, char *errbuf)
{
   int retn = 0, i = src->ArgusInterfaces;
   struct ArgusRfileStruct *rfile;
   int ch, rlen, type;
   char readbuf[256];

   if (src->ArgusRfileList != NULL) {
      rfile = (struct ArgusRfileStruct *) src->ArgusRfileList->start;
      while (rfile) {
         if ((src->ArgusInterface[i].ArgusPd = pcap_open_offline(rfile->name, errbuf)) != NULL) {
            src->ArgusInputPacketFileType = ARGUSLIBPPKTFILE;
            src->ArgusInterface[i].ArgusInterfaceType = ARGUSLIBPPKTFILE;
            src->ArgusInterface[i].ArgusDevice = rfile->name;
            type = pcap_datalink(src->ArgusInterface[i].ArgusPd);
            src->ArgusInterface[i].ArgusCallBack = Arguslookup_pcap_callback(type);
            src->ArgusReadingOffLine++;
            retn++;

         } else {
#ifdef ARGUSDEBUG
            ArgusDebug (3, "ArgusOpenInputPacketFile(0x%x) pcap_open_offline %s\n", src, errbuf);
#endif
            if (strcmp(rfile->name, "-")) {
               if ((src->ArgusPacketInput = fopen(rfile->name, "r")) == NULL) {
                  snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "%s: %s\n", rfile->name, strerror(errno));
               }
            } else
               src->ArgusPacketInput = stdin;
    
            if (src->ArgusPacketInput) {
               if (getArgusMoatTshFile(src)) {
                  if (src->ArgusPacketInput == stdin) {
                     src->ArgusInterface[i].ArgusPcap.fd = 0;
                     src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                     src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                     src->ArgusInterface[i].ArgusInterfaceType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInputPacketFileType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInterface[i].ArgusDevice = rfile->name;
                     src->ArgusReadingOffLine++;
                     retn++;
                  } else
                  if ((src->ArgusInterface[i].ArgusPcap.fd = open (rfile->name, O_RDONLY, NULL)) >= 0) {
                     src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                     src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                     src->ArgusInterface[i].ArgusInterfaceType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInputPacketFileType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInterface[i].ArgusDevice = rfile->name;
                     src->ArgusReadingOffLine++;
                     retn++;
   
                  } else
                     ArgusLog(LOG_ERR, "ArgusOpenInputPacketFile(%s) error. %s\n", rfile->name, strerror(errno));
               } else
               if ((ch = fgetc(src->ArgusPacketInput)) != EOF) {
                  ungetc(ch, src->ArgusPacketInput);
                  if ((rlen = fread ((char *)readbuf, 1, sizeof(ARGUSSNOOPTAG),
                                           src->ArgusPacketInput)) == sizeof(ARGUSSNOOPTAG)) {
                     if ((strncmp((char *)readbuf, ARGUSSNOOPTAG, sizeof(ARGUSSNOOPTAG)) == 0)) {
                        fclose(src->ArgusPacketInput);
                        if ((src->ArgusInterface[i].ArgusPcap.fd = open (rfile->name, O_RDONLY, NULL)) >= 0) {
                           lseek(src->ArgusInterface[i].ArgusPcap.fd, 16, SEEK_SET);
                           src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                           src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                           src->ArgusInputPacketFileType = ARGUSSNOOPKTFILE;
                           src->ArgusReadingOffLine++;
                           retn++;
                        }
         
                     } else {
                        snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "Unknown packet file format\n");
                     }
                  } else {
                     snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "Error reading %s. Read %d bytes\n", rfile->name, rlen);
                  }
               } else {
                  snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "Error reading %s. Stream Empty\n", rfile->name);
               }
            }
         }

         rfile = (struct ArgusRfileStruct *) rfile->nxt;
         i++;
      }

      src->ArgusInterfaces = i;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusOpenInputPacketFile(0x%x) returning %d\n", errbuf, retn);
#endif

   return (retn);
}


char *
ArgusCopyArgv (char **argv)
{
   char **p;
   int len = 0;
   char *buf = NULL, *src, *dst;
 
   p = argv;
   if (*p == 0) return 0;
 
   while (*p) len += (int) strlen (*p++) + 1;
 
   if ((buf = (char *) ArgusMalloc (len)) != NULL) {
      p = argv;
      dst = buf;
      while ((src = *p++) != NULL) {
         if (*src != '-') {
            while ((*dst++ = *src++) != '\0') ;
            dst[-1] = ' ';
         }
      }
 
      dst[-1] = '\0';
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusCopyArgv(0x%x) returning 0x%x\n", argv, buf);
#endif
 
   return buf;
}


#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/ioctl.h>

int ArgusGetInterfaceFD = -1;

void
ArgusGetInterfaceStatus (struct ArgusSourceStruct *src)
{
   struct ArgusDeviceStruct *device = NULL;
   char errbuf[PCAP_ERRBUF_SIZE];
   struct ifreq ifr;
   int fd, i;

   if (src && src->ArgusDeviceList)
      if ((device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_NOLOCK)) != NULL)
         ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_NOLOCK);

   if (device == NULL)
      return;

   if (strstr(device->name, "dag")) {
      for (i = 0; i < src->ArgusInterfaces; i++) {
         if (src->ArgusInterface[i].ArgusPd && (pcap_fileno(src->ArgusInterface[i].ArgusPd) > 0))
            bzero ((char *)&src->ArgusInterface[i].ifr, sizeof(ifr));

         src->ArgusInterface[i].ifr.ifr_flags |= IFF_UP;
         setArgusInterfaceStatus(src, 1);
      }
      return;
   }

   if (ArgusGetInterfaceFD < 0)
      if ((ArgusGetInterfaceFD = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
         ArgusLog(LOG_ERR, "ArgusGetInterfaceStatus: socket %s", strerror(errno));

   fd = ArgusGetInterfaceFD;

   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      if (src->ArgusInterface[i].ArgusPd && (pcap_fileno(src->ArgusInterface[i].ArgusPd) > 0)) {
         memcpy ((char *)&ifr, (char *)&src->ArgusInterface[i].ifr, sizeof(ifr));

         if (ioctl(fd, SIOCGIFFLAGS, (char *)&src->ArgusInterface[i].ifr) < 0)
            ArgusLog(LOG_ERR, "ArgusGetInterfaceStatus: ioctl %s", strerror(errno));

         if ((ifr.ifr_flags & IFF_UP) != (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP)) {
            setArgusInterfaceStatus(src, (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) ? 1 : 0);
 
            if (!((pcap_lookupnet (src->ArgusInterface[i].ArgusDevice, 
                         (u_int *)&src->ArgusInterface[i].ArgusLocalNet,
                         (u_int *)&src->ArgusInterface[i].ArgusNetMask, errbuf)) < 0)) {
#if defined(_LITTLE_ENDIAN)
               src->ArgusInterface[i].ArgusLocalNet = ntohl(src->ArgusInterface[i].ArgusLocalNet);
               src->ArgusInterface[i].ArgusNetMask  = ntohl(src->ArgusInterface[i].ArgusNetMask);
#endif
            }
            ArgusLog (LOG_WARNING, "ArgusGetInterfaceStatus: interface %s is %s\n", src->ArgusInterface[i].ifr.ifr_name,
               (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) ? "up" : "down");
         }

         pcap_stats (src->ArgusInterface[i].ArgusPd, &src->ArgusInterface[i].ArgusStat);
      }
   }

   return;
}
