/*
 * Argus Software.  Argus files - Output processing
 * Copyright (c) 2000-2006 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */


#if !defined(ArgusOutput)
#define ArgusOutput
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
 
#include <ArgusModeler.h>
#include <ArgusUtil.h>
#include <argus_parser.h>
#include <argus_filter.h>


#if defined(ARGUS_THREADS)
#include <pthread.h>
#endif

void *ArgusOutputProcess(void *);


struct timeval *getArgusMarReportInterval(struct ArgusOutputStruct *);
void setArgusMarReportInterval(struct ArgusOutputStruct *, char *);

struct ArgusRecordStruct *ArgusGenerateStatusMarRecord (struct ArgusOutputStruct *, unsigned char);


struct ArgusOutputStruct *
ArgusNewOutput (struct ArgusSourceStruct *src, struct ArgusModelerStruct *model)
{
   struct ArgusOutputStruct *retn = NULL;
   int i;

   if ((retn = (struct ArgusOutputStruct *) ArgusCalloc (1, sizeof (struct ArgusOutputStruct))) == NULL)
     ArgusLog (LOG_ERR, "ArgusNewOutput() ArgusCalloc error %s\n", strerror(errno));

   for (i = 0; i < ARGUS_MAXLISTEN; i++)
      retn->clients[i].fd = -1;

   gettimeofday (&retn->ArgusGlobalTime, 0L);
   retn->ArgusStartTime = retn->ArgusGlobalTime;

   retn->ArgusReportTime.tv_sec   = retn->ArgusGlobalTime.tv_sec + retn->ArgusMarReportInterval.tv_sec;
   retn->ArgusReportTime.tv_usec += retn->ArgusMarReportInterval.tv_usec;
   retn->ArgusLastMarUpdateTime   = retn->ArgusGlobalTime;

   retn->ArgusSrc   = src;
   retn->ArgusModel = model;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusNewOutput() returning retn 0x%x\n", retn);
#endif

   return (retn);
}


struct ArgusRecord *ArgusGenerateInitialMar (struct ArgusOutputStruct *);

void
ArgusInitOutput (struct ArgusOutputStruct *output)
{
   struct ArgusWfileStruct *wfile;
   extern char *chroot_dir;
   extern uid_t new_uid;
   extern gid_t new_gid;
   int len = 0;
#if defined(ARGUS_THREADS)
   pthread_attr_t attrbuf, *attr = &attrbuf;
#endif /* ARGUS_THREADS */

   if ((ArgusOutputTask->ArgusOutputList = ArgusNewList()) == NULL)
      ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusList %s", strerror(errno));

   if (output->ArgusInitMar != NULL)
      ArgusFree (output->ArgusInitMar);

   if ((output->ArgusInitMar = ArgusGenerateInitialMar(output)) == NULL)
      ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusGenerateInitialMar error %s", strerror(errno));

   len = ntohs(output->ArgusInitMar->hdr.len) * 4;

   output->ArgusLfd = -1;

   if (output->ArgusPortNum != 0) {
      char errbuf[256];
      if (ArgusEstablishListen (output, errbuf) < 0) 
         ArgusLog (LOG_ERR, "%s", errbuf);
   }

   if (chroot_dir != NULL)
      ArgusSetChroot(chroot_dir);

   if (new_uid > 0) {
      if (setuid(new_uid) < 0)
         ArgusLog (LOG_ERR, "ArgusInitOutput: setuid error %s", strerror(errno));
   }
   if (new_gid > 0) {
      if (setgid(new_gid) < 0)
         ArgusLog (LOG_ERR, "ArgusInitOutput: setgid error %s", strerror(errno));
   }

   if (output->ArgusWfileList) {
      struct ArgusListRecord *sfile= output->ArgusWfileList->start->obj;
      do {
         if ((wfile = (struct ArgusWfileStruct *) ArgusPopFrontList(output->ArgusWfileList, ARGUS_NOLOCK)) != NULL) {
            int ind = output->clientindex++;

            if (strcmp (wfile->filename, "-")) {
               if ((output->clients[ind].fd = open (wfile->filename, O_WRONLY|O_APPEND|O_CREAT|O_NONBLOCK, 0x1a4)) < 0)
                  ArgusLog (LOG_ERR, "ArgusInitOutput: open %s: %s", wfile->filename, strerror(errno));
            } else {
               output->clients[ind].fd = 1;
               output->ArgusWriteStdOut++;
            }

            output->clientnum++;

            if (wfile->filter != NULL) {
               if (ArgusFilterCompile (&output->clients[ind].ArgusBPFcode, wfile->filter, 1) < 0) 
                  ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusFilter syntax error: %s", wfile->filter);
               output->clients[ind].ArgusFilterInitialized++;
            }

            if ((output->clients[ind].sock = ArgusNewSocket(output->clients[ind].fd)) == NULL)
               ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusNewSocket error %s", strerror(errno));

            if (write (output->clients[ind].fd, (char *) output->ArgusInitMar, len) != len) {
               if (!output->ArgusWriteStdOut) {
                  close (output->clients[ind].fd);
                  unlink (wfile->filename);
               }
               ArgusLog (LOG_ERR, "ArgusInitOutput: write(): %s", strerror(errno));
            }

            if (strcmp(wfile->filename, "/dev/null"))
               output->clients[ind].sock->filename = strdup(wfile->filename);

            output->clients[ind].ArgusClientStart++;
            ArgusPushBackList (output->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_NOLOCK);
         }
      } while (output->ArgusWfileList->start->obj != sfile);

      ArgusDeleteList(output->ArgusWfileList, ARGUS_WFILE_LIST);
      output->ArgusWfileList = NULL;
   }

#if defined(ARGUS_THREADS)
   pthread_attr_init(attr); 

   if (getuid == 0)
      pthread_attr_setschedpolicy(attr, SCHED_RR);
   else
      attr = NULL;

   if ((pthread_create(&output->thread, attr, ArgusOutputProcess, (void *) output)) != 0)
      ArgusLog (LOG_ERR, "ArgusNewOutput() pthread_create error %s\n", strerror(errno));

#endif /* ARGUS_THREADS */

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusInitOutput() done\n");
#endif
}


void
ArgusCloseOutput(struct ArgusOutputStruct *output)
{
#if defined(ARGUS_THREADS)
   void *retn = NULL;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseOutput() scheduling closure after %d records\n", output->ArgusModel->ArgusOutputList->count);
#endif
   if (output != NULL) 
      pthread_join(output->thread, &retn);
#else
   if (output != NULL) {
      output->status |= ARGUS_SHUTDOWN;
#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusCloseOutput() scheduling closure after writing records\n");
#endif
      ArgusOutputProcess(output);
   }
#endif /* ARGUS_THREADS */

   ArgusDeleteList(output->ArgusModel->ArgusOutputList, ARGUS_OUTPUT_LIST);
   ArgusDeleteList(output->ArgusOutputList, ARGUS_OUTPUT_LIST);

   if (output->ArgusInitMar != NULL)
      ArgusFree (output->ArgusInitMar);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseOutput(0x%x) done\n", output);
#endif
}


void ArgusCheckClientStatus (struct ArgusOutputStruct *);
int ArgusCheckClientMessage (struct ArgusOutputStruct *, int);
int ArgusCongested = 0;

int ArgusOutputStatusTime(struct ArgusOutputStruct *);

int
ArgusOutputStatusTime(struct ArgusOutputStruct *output)
{
   int retn = 0;

   gettimeofday (&output->ArgusGlobalTime, 0L);
   if ((output->ArgusReportTime.tv_sec  < output->ArgusGlobalTime.tv_sec) ||
      ((output->ArgusReportTime.tv_sec == output->ArgusGlobalTime.tv_sec) &&
       (output->ArgusReportTime.tv_usec < output->ArgusGlobalTime.tv_usec))) {

      output->ArgusReportTime.tv_sec  += getArgusMarReportInterval(output)->tv_sec;
      output->ArgusReportTime.tv_usec += getArgusMarReportInterval(output)->tv_usec;

      if (output->ArgusReportTime.tv_usec > 1000000) {
         output->ArgusReportTime.tv_sec++;
         output->ArgusReportTime.tv_usec -= 1000000;
      }

      retn++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusOutputStatusTime(0x%x) done", output);
#endif
   return (retn);
}


#define ARGUS_MAXPROCESS		0x10000

void *
ArgusOutputProcess(void *arg)
{
   struct ArgusOutputStruct *output = (struct ArgusOutputStruct *) arg;
   struct timeval ArgusUpDate = {0, 500000}, ArgusNextUpdate = {0,0};
   int i, val, count;
   void *retn = NULL;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusOutputProcess(0x%x) starting\n", output);
#endif

#if defined(ARGUS_THREADS)
   while (!(output->status & ARGUS_STOP)) {
#else
   {
#endif
      struct ArgusRecordStruct *rec = NULL;

      if (output && output->ArgusModel && output->ArgusModel->ArgusOutputList) {
         gettimeofday (&output->ArgusGlobalTime, 0L);

    /* check to see if there are any new clients */
         
         if ((output->ArgusPortNum != 0) &&
            ((output->ArgusGlobalTime.tv_sec >  ArgusNextUpdate.tv_sec) ||
            ((output->ArgusGlobalTime.tv_sec == ArgusNextUpdate.tv_sec) &&
             (output->ArgusGlobalTime.tv_usec > ArgusNextUpdate.tv_usec)))) {
         
            if (output->ArgusLfd != -1) {
               struct timeval wait = {0, 0}; 
               fd_set readmask;
               int width = output->ArgusLfd;
 
               FD_ZERO(&readmask);
               FD_SET(output->ArgusLfd, &readmask);
               if (output->clientnum)
                  for (i = 0; i < output->clientnum; i++)
                     if ((output->clients[i].fd != -1) && (!(output->clients[i].sock->filename))) {
                        FD_SET(output->clients[i].fd, &readmask);
                        width = (output->clients[i].fd > width) ? output->clients[i].fd : width;
                     }

               if ((val = select (width + 1, &readmask, NULL, NULL, &wait)) >= 0) {
                  if (val > 0) {
#ifdef ARGUSDEBUG
                     ArgusDebug (3, "ArgusOutputProcess() select returned with tasks\n");
#endif
                     if (FD_ISSET(output->ArgusLfd, &readmask))
                        ArgusCheckClientStatus(output);
                     else {
                        for (i = 0; i < ARGUS_MAXLISTEN; i++) {
                           if (output->clients[i].fd != -1)
                              if (FD_ISSET(output->clients[i].fd, &readmask)) {
                                 if (ArgusCheckClientMessage(output, i) < 0) {
                                    int x;
                                    ArgusDeleteSocket(output, &output->clients[i]);
                                    for (x = i; x < output->clientnum - 1; x++)
                                       output->clients[x] = output->clients[x + 1];
                                    output->clients[output->clientnum].fd = -1;
                                    output->clientnum--;
                                 }
                           }
                        }
                     }
                  }
               }

               ArgusNextUpdate.tv_usec += ArgusUpDate.tv_usec;
               ArgusNextUpdate.tv_sec  += ArgusUpDate.tv_sec;

               if (ArgusNextUpdate.tv_usec > 1000000) {
                  ArgusNextUpdate.tv_sec++;
                  ArgusNextUpdate.tv_usec -= 1000000;
               }
            }
         }

#if defined(ARGUS_THREADS)
         if (ArgusListEmpty(output->ArgusModel->ArgusOutputList))
            usleep(50000);
#endif
         if (ArgusOutputStatusTime(output)) {
            if ((rec = ArgusGenerateStatusMarRecord(output, ARGUS_STATUS)) != NULL) {
               count = 0;
               for (i = 0; i < output->clientnum; i++) {
                  if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL) && output->clients[i].ArgusClientStart)
                     count++;
               }

               if (count > 0) {
                  for (i = 0; i < output->clientnum; i++) {
                     if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL) && output->clients[i].ArgusClientStart) {
                        if (ArgusWriteSocket (output, &output->clients[i], rec, count--) < 0) {
                           int x;
                           ArgusDeleteSocket(output, &output->clients[i]);
                           for (x = i; x < output->clientnum - 1; x++)
                              output->clients[x] = output->clients[x + 1];
                           output->clients[output->clientnum].fd = -1;
                           output->clientnum--;
                        }
                     }
                  }

               } else
                  ArgusFreeListRecord (rec);
            }
            output->ArgusLastMarUpdateTime   = output->ArgusGlobalTime;
         }

         if (output->ArgusOutputList && !(ArgusListEmpty(output->ArgusModel->ArgusOutputList))) {
            int done = 0;
            ArgusLoadList(output->ArgusModel->ArgusOutputList, output->ArgusOutputList);

            while (!done && ((rec = (struct ArgusRecordStruct *) ArgusPopFrontList(output->ArgusOutputList, ARGUS_NOLOCK)) != NULL)) {
               output->ArgusTotalRecords++;
               output->ArgusOutputSequence = rec->canon.trans.seqnum;
               count = 0;

               if (((rec->hdr.type & 0xF0) == ARGUS_MAR) && ((rec->hdr.cause & 0xF0) == ARGUS_STOP)) {
                  done++;
                  output->status |= ARGUS_STOP;
                  ArgusFreeListRecord(rec);
                  if ((rec = ArgusGenerateStatusMarRecord(output, ARGUS_STOP)) == NULL)
                     if (output->clientnum > 0)
                        ArgusLog (LOG_ERR, "ArgusGenerateStatusMarRecord() error %s", strerror(errno));
                     
#ifdef ARGUSDEBUG
                  ArgusDebug (2, "ArgusOutputProcess() received stop record %d records on the list\n", output->ArgusOutputList->count);
#endif
               }

               for (i = 0; i < output->clientnum; i++)
                  if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL) && output->clients[i].ArgusClientStart)
                     count++;

               if (count > 0) {
                  for (i = 0; i < output->clientnum; i++) {
                     if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL) && output->clients[i].ArgusClientStart) {
                        int arguswriterecord = 1;
#ifdef ARGUSDEBUG
                     ArgusDebug (5, "ArgusOutputProcess() output %d has %d queued\n", i, output->clients[i].sock->ArgusOutputList->count);
#endif

                        if (output->clients[i].ArgusFilterInitialized) {
                           if (!(ArgusFilterRecord ((struct nff_insn *)output->clients[i].ArgusBPFcode.bf_insns, &rec->canon)))
                              arguswriterecord = 0;
                        }

                        if (arguswriterecord) {
                           if (ArgusWriteSocket (output, &output->clients[i], rec, count--) < 0) {
                              int x;
                              ArgusDeleteSocket(output, &output->clients[i]);
                              for (x = i; x < output->clientnum - 1; x++)
                                 output->clients[x] = output->clients[x + 1];
                              output->clients[output->clientnum].fd = -1;
                              output->clientnum--;

                           } else {
                              if (ArgusWriteOutSocket (output, &output->clients[i]) < 0) {
                                 int x;
                                 ArgusDeleteSocket(output, &output->clients[i]);
                                 for (x = i; x < output->clientnum - 1; x++)
                                    output->clients[x] = output->clients[x + 1];
                                 output->clients[output->clientnum - 1].fd = -1;
                                 output->clientnum--;
                              }
                           }

                        } else {
                           if (--count == 0)
                              ArgusFreeListRecord(rec);
                        }
                     }
                  }

               } else
                  ArgusFreeListRecord(rec);
            }

            if (output->ArgusWriteStdOut)
               fflush (stdout);
         }

         if ((output->ArgusPortNum != 0) && (output->clientnum)) {
            int x;
            for (i = 0; i < output->clientnum; i++) {
               if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL)) {
                  if ((output->status & ARGUS_STOP) || (output->status & ARGUS_SHUTDOWN)) {
                     ArgusWriteOutSocket (output, &output->clients[i]);
                     ArgusDeleteSocket(output, &output->clients[i]);
                     for (x = i; x < output->clientnum - 1; x++)
                        output->clients[x] = output->clients[x + 1];
                     output->clients[output->clientnum - 1].fd = -1;
                     output->clientnum--;
                  } else {
                     if (ArgusWriteOutSocket (output, &output->clients[i]) < 0) {
                        ArgusDeleteSocket(output, &output->clients[i]);
                        for (x = i; x < output->clientnum - 1; x++)
                           output->clients[x] = output->clients[x + 1];
                        output->clients[output->clientnum - 1].fd = -1;
                        output->clientnum--;
                     }
                  }
               }
            }
         }

      } else {
#ifdef ARGUSDEBUG
         ArgusDebug (6, "ArgusOutputProcess() waiting for ArgusOutputList 0x%x\n", output);
#endif
         usleep (150000);
      }
#if !defined(ARGUS_THREADS)
   }
#else
   }
#endif /* ARGUS_THREADS */

   if (output->status & ARGUS_SHUTDOWN) {
      int x;
      for (i = 0; i < output->clientnum; i++) {
          if ((output->clients[i].fd != -1) && (output->clients[i].sock != NULL)) {
             ArgusWriteOutSocket (output, &output->clients[i]);
             ArgusDeleteSocket(output, &output->clients[i]);
             for (x = i; x < output->clientnum - 1; x++)
                output->clients[x] = output->clients[x + 1];
             output->clients[output->clientnum - 1].fd = -1;
             output->clientnum--;
          }
       }
    }

#ifdef ARGUSDEBUG
   ArgusDebug (9, "ArgusOutputProcess() exiting\n");
#endif

#if defined(ARGUS_THREADS)
   pthread_exit(retn);
#else
   return (retn);
#endif /* ARGUS_THREADS */
}


#include <netdb.h>

int
ArgusEstablishListen (struct ArgusOutputStruct *output, char *errbuf)
{
   int s = -1;
   int port = output->ArgusPortNum;
   char *bindIP = output->ArgusBindAddr;
   struct sockaddr_in sin;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   struct hostent *host;

   gettimeofday (tvp, 0L);

   if (port) {
      sin.sin_addr.s_addr = INADDR_ANY;
      if (bindIP) {
#ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusEstablishListen(%d, %s, 0x%x)\n", port, bindIP, errbuf);
#endif
         if ((host = gethostbyname (bindIP)) != NULL) {
            if ((host->h_addrtype == AF_INET) && (host->h_length == 4)) {
               bcopy ((char *) *host->h_addr_list, (char *)&sin.sin_addr.s_addr, host->h_length);
            } else
               ArgusLog (LOG_ERR, "ArgusEstablishListen() unsupported bind address %s", bindIP);
         } else
            ArgusLog (LOG_ERR, "ArgusEstablishListen() bind address %s error %s", bindIP, strerror(errno));
      }

      sin.sin_port = htons((u_short) port);
      sin.sin_family = AF_INET;

#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusEstablishListen(%d, 0x%x) binding: %d\n", port, errbuf, sin.sin_addr.s_addr);
#endif

      if ((s = socket (AF_INET, SOCK_STREAM, 0)) != -1) {
         if ((fcntl (s, F_SETFL, O_NDELAY)) >= 0) {
            if (!(bind (s, (struct sockaddr *)&sin, sizeof(sin)))) {
               if ((listen (s, ARGUS_MAXLISTEN)) >= 0) {
                  output->ArgusLfd = s;
               } else {
                  close (s);
                  s = -1;
                  snprintf(errbuf, 1024, "%s: ArgusEstablishListen: listen() failure", ArgusProgramName);
               }
            } else {
               close (s);
               s = -1;
               snprintf(errbuf, 256, "%s: ArgusEstablishListen: bind() error", ArgusProgramName);
            }
         } else
            snprintf(errbuf, 256, "%s: ArgusEstablishListen: fcntl() error", ArgusProgramName);
      } else
         snprintf(errbuf, 256, "%s: ArgusEstablishListen: socket() error", ArgusProgramName);
   }
     
#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusEstablishListen(%d, 0x%x) returning %d\n", port, errbuf, s);
#endif

   return (s);
}


int ArgusAuthenticateClient (struct ArgusClientData *);

void
ArgusCheckClientStatus (struct ArgusOutputStruct *output)
{
   struct sockaddr from;
   int len = sizeof (from);
   int fd;

#ifdef ARGUS_SASL
#define SASL_SEC_MASK   0x0fff
   struct sockaddr_in localaddr, remoteaddr;
   char localhostname[1024];
   sasl_conn_t *conn = NULL;
   int salen, maxbufprops = 4096;
   sasl_security_properties_t secprops;
   sasl_external_properties_t extprops;
   int SASLOpts = 0;
   int retn;
#endif

   if ((fd = accept (output->ArgusLfd, (struct sockaddr *)&from, (socklen_t *)&len)) > 0) {
      if ((fcntl (fd, F_SETFL, O_NONBLOCK)) >= 0) {
         if (ArgusTcpWrapper (fd, &from) >= 0) {
            if (output->clientnum < ARGUS_MAXLISTEN) {
               int i = output->clientnum++;

               bzero(&output->clients[i], sizeof(output->clients[i]));
               output->clients[i].fd = fd;
#ifdef ARGUSDEBUG
               ArgusDebug (2, "ArgusCheckClientStatus() new client %d\n", i);
#endif
               if ((output->clients[i].sock = ArgusNewSocket(fd)) == NULL)
                  ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusNewSocket error %s", strerror(errno));

               if (output->ArgusInitMar != NULL)
                  ArgusFree(output->ArgusInitMar);

               if ((output->ArgusInitMar = ArgusGenerateInitialMar(output)) == NULL)
                  ArgusLog (LOG_ERR, "ArgusInitOutput: ArgusGenerateInitialMar error %s", strerror(errno));

#ifdef ARGUS_SASL
#ifdef ARGUSDEBUG
               ArgusDebug (2, "ArgusAuthenticateClient: SASL enabled\n");
#endif
               if (ArgusMaxSsf > 0) {
                  gethostname(localhostname, 1024);
                  if (!strchr (localhostname, '.')) {
                     strcat (localhostname, ".");
                     getdomainname (&localhostname[strlen(localhostname)], 1024 - strlen(localhostname));
                  }

                  if ((retn = sasl_server_init(NULL, "argus")) != SASL_OK)
                     ArgusLog (LOG_ERR, "ArgusAuthenticateClient: sasl_server_init %d", retn);

                  if ((retn = sasl_server_new("argus", localhostname, NULL, NULL,
                                  SASL_SECURITY_LAYER, &output->clients[i].sasl_conn)) != SASL_OK)
                     ArgusLog (LOG_ERR, "ArgusAuthenticateClient: sasl_server_init %d", retn);

                  conn = output->clients[i].sasl_conn;

                 /* set required security properties here */

                  secprops.min_ssf = ArgusMinSsf;
                  secprops.max_ssf = ArgusMaxSsf;
                  secprops.security_flags = SASLOpts & SASL_SEC_MASK;
                  sasl_setprop(conn, SASL_SEC_PROPS, &secprops);

                 /* set external properties here */
                  extprops.ssf = 0;
                  extprops.auth_id = NULL;
                  sasl_setprop(conn, SASL_SSF_EXTERNAL, &extprops);

                  sasl_setprop(conn, SASL_MAXOUTBUF, &maxbufprops);

                 /* set ip addresses */
                  salen = sizeof(localaddr);
                  if (getsockname(output->clients[i].fd, (struct sockaddr *)&localaddr, &salen) < 0)
                     ArgusLog (LOG_ERR, "getsockname");

                  salen = sizeof(remoteaddr);
                  if (getpeername(output->clients[i].fd, (struct sockaddr *)&remoteaddr, &salen) < 0)
                     ArgusLog (LOG_ERR, "getpeername");

                  if ((retn = sasl_setprop(conn, SASL_IP_LOCAL, &localaddr)) != SASL_OK)
                     ArgusLog (LOG_ERR, "ArgusAuthenticateClient: Error setting local IP address");

                  if ((retn = sasl_setprop(conn, SASL_IP_REMOTE, &remoteaddr)) != SASL_OK)
                     ArgusLog (LOG_ERR, "ArgusAuthenticateClient: Error setting remote IP address");

                  output->ArgusInitMar->argus_mar.status |= htonl(ARGUS_SASL_AUTHENTICATE);
               }
#endif
               len = ntohs(output->ArgusInitMar->hdr.len) * 4;

               if (write (output->clients[i].fd, (char *) output->ArgusInitMar, len) != len) {
                  close (output->clients[i].fd);
                  ArgusLog (LOG_ERR, "ArgusInitOutput: write(): %s", strerror(errno));
               }

#ifdef ARGUS_SASL
               if (ArgusMaxSsf > 0) {
                  if (!(ArgusAuthenticateClient (&output->clients[i]))) {
                     ArgusDeleteSocket(output, &output->clients[i]);
                     output->clientnum--;
                     ArgusLog (LOG_ALERT, "ArgusInitClientProcess: ArgusAuthenticateClient failed\n");
                  }

               } else {
               }
#endif
            } else {
               char buf[MAXARGUSRECORD];
               struct ArgusRecord *argus = (struct ArgusRecord *) &buf;
               if ((argus = ArgusGenerateRecord (output->ArgusModel, NULL, ARGUS_ERROR, (struct ArgusRecord *) &buf)) != NULL) {
                  len = ntohs(argus->hdr.len)  * 4;
                  argus->hdr.cause |= ARGUS_MAXLISTENEXCD;
                  if (write (fd, (char *) argus, len) != len)
                     ArgusLog (LOG_ERR, "ArgusInitOutput: write(): %s", strerror(errno));
                  close(fd);
               }
            }

         } else {
            ArgusLog (LOG_WARNING, "ArgusCheckClientStatus: ArgusTcpWrapper rejects");
            close (fd);
         }
         
      } else {
         ArgusLog (LOG_WARNING, "ArgusCheckClientStatus: fcntl: %s", strerror(errno));
         close (fd);
      }
   } else {
      ArgusLog (LOG_WARNING, "ArgusCheckClientStatus: accept: %s", strerror(errno));
      close (fd);
   }
     
#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusCheckClientStatus() returning\n");
#endif
}


 
#define ARGUSMAXCLIENTCOMMANDS           6
#define RADIUM_START                    0
#define RADIUM_DONE                     1
#define RADIUM_FILTER                   2
#define RADIUM_MODEL                    3
#define RADIUM_PROJECT                  4
#define RADIUM_FILE                     5
 
char *ArgusClientCommands[ARGUSMAXCLIENTCOMMANDS] =
{
   "START: ",
   "DONE: ",
   "FILTER: ",
   "MODEL: ",
   "PROJECT: ",
   "FILE: ",
};


int
ArgusCheckClientMessage (struct ArgusOutputStruct *output, int ind)
{
   struct ArgusClientData *client = &output->clients[ind];
   int retn = 0, cnt = 0, i, found, fd = output->clients[ind].fd;
   char buf[MAXSTRLEN], *ptr = buf;
   unsigned int value = 0;
    
#ifdef ARGUS_SASL
   unsigned int *pvalue = &value;
   char *outputbuf = NULL;
   unsigned int outputlen = 0;

   if (client->sasl_conn)
      if ((retn = sasl_getprop(client->sasl_conn, SASL_MAXOUTBUF, (void **) &pvalue)) != SASL_OK)
         ArgusLog (LOG_ERR, "ArgusCheckClientMessage(0x%x, %d) sasl_getprop %s\n", client, fd, strerror(errno));
#endif /* ARGUS_SASL */

   bzero(buf, MAXSTRLEN);

   if (value == 0)
      value = MAXSTRLEN;

   if ((cnt = recv (fd, buf, value, 0)) <= 0) {
      if (cnt < 0) {
#ifdef ARGUSDEBUG
         ArgusDebug (3, "ArgusCheckClientMessage (0x%x, %d) recv() returned error %s\n", client, fd, strerror(errno));
#endif
         return (-1);

      } else {
#ifdef ARGUSDEBUG
         ArgusDebug (3, "ArgusCheckClientMessage (0x%x, %d) recv() returned %d bytes\n", client, fd, cnt);
#endif
         return(-3);
      }

   } else {
#ifdef ARGUSDEBUG
      ArgusDebug (3, "ArgusCheckClientMessage (0x%x, %d) recv() returned %d bytes\n", client, fd, cnt);
#endif
   }

#ifdef ARGUS_SASL
   if ((client->sasl_conn)) {
      if (sasl_decode (client->sasl_conn, buf, cnt, &outputbuf, &outputlen) != SASL_OK) {
         ArgusLog (LOG_WARNING, "ArgusCheckClientMessage(0x%x, %d) sasl_decode (0x%x, 0x%x, %d, 0x%x, %d) failed",
                    client, fd, client->sasl_conn, buf, cnt, &outputbuf, outputlen);
         return(-1);
      } else
#ifdef ARGUSDEBUG
         ArgusDebug (3, "ArgusCheckClientMessage (0x%x, %d) sasl_decode() returned %d bytes\n", client, fd, outputlen);
#endif
      if (outputlen > 0) {
         if (outputlen < MAXSTRLEN) {
            bzero (buf, MAXSTRLEN);
            bcopy (outputbuf, buf, outputlen);
            cnt = outputlen;
         } else
            ArgusLog (LOG_ERR, "ArgusCheckClientMessage(0x%x, %d) sasl_decode returned %d bytes\n", client, fd, outputlen);
        
         free(outputbuf);
      } else {
         return (0);
      }
   }

#endif /* ARGUS_SASL */

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusCheckClientMessage (0x%x, %d) read '%s' from remote\n", client, fd, ptr);
#endif

   for (i = 0, found = 0; i < ARGUSMAXCLIENTCOMMANDS; i++) {
      if (!(strncmp (ptr, ArgusClientCommands[i], strlen(ArgusClientCommands[i])))) {
         found++;
         switch (i) {
            case RADIUM_START: client->ArgusClientStart++; retn = 0; break;
            case RADIUM_DONE:  retn = -4; break; 
            case RADIUM_FILTER: {
               if (ArgusFilterCompile (&client->ArgusBPFcode, &ptr[7], 1) < 0) {
                  retn = -2;
#ifdef ARGUSDEBUG
                  ArgusDebug (3, "ArgusCheckClientMessage: ArgusFilter syntax error: %s\n", &ptr[7]);
#endif
               } else {
#ifdef ARGUSDEBUG
                  ArgusDebug (3, "ArgusCheckClientMessage: ArgusFilter %s\n", &ptr[7]);
#endif
                  client->ArgusFilterInitialized++;
                  if ((cnt = send (fd, "OK", 2, 0)) != 2) {
                     retn = -3;
#ifdef ARGUSDEBUG
                     ArgusDebug (3, "ArgusCheckClientMessage: send error %s\n", strerror(errno));
#endif
                  } else {
                     retn = 0;
#ifdef ARGUSDEBUG
                     ArgusDebug (3, "ArgusCheckClientMessage: ArgusFilter %s initialized.\n", &ptr[7]);
#endif
                  }
               }
               break;
            }

            case RADIUM_PROJECT: 
            case RADIUM_MODEL: 
               break;

            case RADIUM_FILE: {
#ifdef ARGUSDEBUG
                  ArgusDebug (3, "ArgusCheckClientMessage: ArgusFile %s initialized.\n", &ptr[6]);
#endif
                  retn = 0;
               }
               break;

            default:
               ArgusLog (LOG_WARNING, "ArgusCheckClientMessage: received %s",  ptr);
               break;
         }

         break;
      }
   }

   if (!found)
      ArgusLog (LOG_WARNING, "ArgusCheckClientMessage: received %s",  ptr);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusCheckClientMessage: returning %d\n", retn);
#endif

   return (retn);
}


struct ArgusRecord *
ArgusGenerateInitialMar (struct ArgusOutputStruct *output)
{
   struct ArgusRecord *retn;
   struct timeval now;

   if ((retn = (struct ArgusRecord *) ArgusCalloc (1, sizeof(struct ArgusRecord))) == NULL)
     ArgusLog (LOG_ERR, "ArgusGenerateInitialMar(0x%x) ArgusCalloc error %s\n", output, strerror(errno));
   
   retn->hdr.type  = ARGUS_MAR | ARGUS_VERSION;
   retn->hdr.cause = ARGUS_START;
   retn->hdr.len   = sizeof(struct ArgusRecord) / 4;

   retn->argus_mar.argusid = ARGUS_COOKIE;
   retn->argus_mar.thisid = getArgusID(ArgusModel);


   retn->argus_mar.startime.tv_sec  = output->ArgusSrc->ArgusStartTime.tv_sec;
   retn->argus_mar.startime.tv_usec = output->ArgusSrc->ArgusStartTime.tv_usec;

   gettimeofday (&now, 0L);

   retn->argus_mar.now.tv_sec  = now.tv_sec;
   retn->argus_mar.now.tv_usec = now.tv_usec;

   retn->argus_mar.major_version = VERSION_MAJOR;
   retn->argus_mar.minor_version = VERSION_MINOR;
   retn->argus_mar.reportInterval = getArgusFarReportInterval(output->ArgusModel)->tv_sec;
   retn->argus_mar.argusMrInterval = getArgusMarReportInterval(output)->tv_sec;

   retn->argus_mar.localnet = output->ArgusSrc->ArgusInterface[0].ArgusLocalNet;
   retn->argus_mar.netmask = output->ArgusSrc->ArgusInterface[0].ArgusNetMask;

   retn->argus_mar.nextMrSequenceNum = output->ArgusOutputSequence;
   retn->argus_mar.record_len = -1;

#if defined(_LITTLE_ENDIAN)
   ArgusHtoN(retn);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusGenerateInitialMar() returning\n");
#endif

   return (retn);
}


struct ArgusRecordStruct *
ArgusGenerateStatusMarRecord (struct ArgusOutputStruct *output, unsigned char status)
{
   struct ArgusRecordStruct *retn = NULL;
   struct ArgusRecord *rec = NULL;

   while ((retn = (struct ArgusRecordStruct *) ArgusMallocListRecord (sizeof(*retn))) == NULL) {
      if (output && output->ArgusModel && output->ArgusModel->ArgusOutputList) {
         if ((retn = (struct ArgusRecordStruct *)
                     ArgusPopFrontList(output->ArgusModel->ArgusOutputList, ARGUS_NOLOCK)) != NULL) {
            ArgusFreeListRecord (retn);
         } else
            break;
      } else
         break;
   }

   if (retn) {
      extern int ArgusAllocTotal, ArgusFreeTotal, ArgusAllocBytes;
      struct ArgusSourceStruct *ArgusSrc = NULL;
      struct timeval now;

      retn->hdr.type  = ARGUS_MAR | ARGUS_VERSION;
      retn->hdr.cause = status;
      retn->hdr.len   = ((unsigned short) sizeof(struct ArgusRecord)/4);

      rec = (struct ArgusRecord *) &retn->canon;

      rec->hdr = retn->hdr;
      rec->argus_mar.argusid = getArgusID(ArgusModel);

      gettimeofday (&now, 0L);

      rec->argus_mar.startime.tv_sec  = output->ArgusStartTime.tv_sec;
      rec->argus_mar.startime.tv_usec = output->ArgusStartTime.tv_usec;

      rec->argus_mar.now.tv_sec  = now.tv_sec;
      rec->argus_mar.now.tv_usec = now.tv_usec;

      rec->argus_mar.major_version = VERSION_MAJOR;
      rec->argus_mar.minor_version = VERSION_MINOR;
      rec->argus_mar.reportInterval = getArgusFarReportInterval(output->ArgusModel)->tv_sec;
      rec->argus_mar.argusMrInterval = getArgusMarReportInterval(ArgusOutputTask)->tv_sec;

      rec->argus_mar.localnet = output->ArgusSrc->ArgusInterface[0].ArgusLocalNet;
      rec->argus_mar.netmask = output->ArgusSrc->ArgusInterface[0].ArgusNetMask;
    
      rec->argus_mar.nextMrSequenceNum = output->ArgusOutputSequence;
      rec->argus_mar.record_len = -1;

      if ((ArgusSrc = output->ArgusSrc) != NULL) {
         int i;
         rec->argus_mar.interfaceType = ArgusSrc->ArgusInterface[0].ArgusPcap.linktype;
         rec->argus_mar.interfaceStatus = getArgusInterfaceStatus(ArgusSrc);

         rec->argus_mar.pktsRcvd  = 0;
         rec->argus_mar.bytesRcvd = 0;
         rec->argus_mar.dropped   = 0;

         for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
            rec->argus_mar.pktsRcvd  += ArgusSrc->ArgusInterface[i].ArgusTotalPkts - 
                                       ArgusSrc->ArgusInterface[i].ArgusLastPkts;
            rec->argus_mar.bytesRcvd += ArgusSrc->ArgusInterface[i].ArgusTotalBytes -
                                       ArgusSrc->ArgusInterface[i].ArgusLastBytes;
            rec->argus_mar.dropped   += ArgusSrc->ArgusInterface[i].ArgusStat.ps_drop - 
                                       ArgusSrc->ArgusInterface[i].ArgusLastDrop;

            ArgusSrc->ArgusInterface[i].ArgusLastPkts  = ArgusSrc->ArgusInterface[i].ArgusTotalPkts;
            ArgusSrc->ArgusInterface[i].ArgusLastDrop  = ArgusSrc->ArgusInterface[i].ArgusStat.ps_drop;
            ArgusSrc->ArgusInterface[i].ArgusLastBytes = ArgusSrc->ArgusInterface[i].ArgusTotalBytes;
         }
      }

      rec->argus_mar.records = output->ArgusTotalRecords - output->ArgusLastRecords;
      output->ArgusLastRecords = output->ArgusTotalRecords;

      rec->argus_mar.flows = output->ArgusModel->ArgusTotalNewFlows - output->ArgusModel->ArgusLastNewFlows;
      output->ArgusModel->ArgusLastNewFlows = output->ArgusModel->ArgusTotalNewFlows;

      if (output->ArgusModel && output->ArgusModel->ArgusStatusQueue)
         rec->argus_mar.queue   = output->ArgusModel->ArgusStatusQueue->count;
      else
         rec->argus_mar.queue   = 0;

      if (output->ArgusOutputList)
         rec->argus_mar.output  = output->ArgusOutputList->count;
      else
         rec->argus_mar.output  = 0;

      rec->argus_mar.clients = output->clientnum;

      rec->argus_mar.bufs  = ArgusAllocTotal - ArgusFreeTotal;
      rec->argus_mar.bytes = ArgusAllocBytes;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusGenerateStatusMarRecord(0x%x, %d) returning 0x%x", output, status, retn);
#endif

   return (retn);
}


struct timeval *
getArgusMarReportInterval(struct ArgusOutputStruct *output) {
   return (&output->ArgusMarReportInterval);
}


#include <ctype.h>
#include <math.h>

void
setArgusMarReportInterval(struct ArgusOutputStruct *output, char *value)
{
   struct timeval *tvp = getArgusMarReportInterval(output);

   struct timeval ovalue, now;
   double thisvalue = 0.0, iptr, fptr;
   int ivalue = 0;
   char *ptr = NULL;;

   if (tvp != NULL) {
      ovalue = *tvp;
      tvp->tv_sec  = 0;
      tvp->tv_usec = 0;
   } else {
      ovalue.tv_sec  = 0;
      ovalue.tv_usec = 0;
   }

   if (((ptr = strchr (value, '.')) != NULL) || isdigit((int)*value)) {
      if (ptr != NULL) {
         thisvalue = atof(value);
      } else {
         if (isdigit((int)*value)) {
            ivalue = atoi(value);
            thisvalue = ivalue * 1.0;
         }
      }

      fptr =  modf(thisvalue, &iptr);

      tvp->tv_sec = iptr;
      tvp->tv_usec =  fptr * 1000000;

      gettimeofday(&now, 0L);

      output->ArgusReportTime.tv_sec  = now.tv_sec + tvp->tv_sec;
      output->ArgusReportTime.tv_usec = tvp->tv_usec;

   } else
      *tvp = ovalue;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "setArgusMarReportInterval(%s) returning\n", value);
#endif
}


#if defined HAVE_TCP_WRAPPER

#include <syslog.h>
#include <tcpd.h>

#if !defined(MAXPATHNAMELEN)
#define MAXPATHNAMELEN   BUFSIZ
#endif

#define PARANOID		1
#define KILL_IP_OPTIONS		1
#define HOSTS_ACCESS		1

int allow_severity = LOG_INFO;     /* run-time adjustable */
int deny_severity  = LOG_WARNING;   /* ditto */

void fix_options(struct request_info *);

#endif

int
ArgusTcpWrapper (int fd, struct sockaddr *from)
{
#if defined(HAVE_TCP_WRAPPER)
   int retn = 0;
   struct request_info request;

   /*
    * Find out the endpoint addresses of this conversation. Host name
    * lookups and double checks will be done on demand.
    */
 
   request_init(&request, RQ_DAEMON, ArgusProgramName, RQ_FILE, STDIN_FILENO, 0);
   request.fd = fd;
   fromhost(&request);

   /*
    * Optionally look up and double check the remote host name. Sites
    * concerned with security may choose to refuse connections from hosts
    * that pretend to have someone elses host name.
    */
 
#ifdef PARANOID
   if (STR_EQ(eval_hostname(request.client), paranoid)) {
      ArgusLog (deny_severity, "refused connect from %s", eval_client(&request)); 
      if (request.sink)
         request.sink(request.fd);
      return -1;
   }
#endif

    /*
     * The BSD rlogin and rsh daemons that came out after 4.3 BSD disallow
     * socket options at the IP level. They do so for a good reason.
     * Unfortunately, we cannot use this with SunOS 4.1.x because the
     * getsockopt() system call can panic the system.
     */  

#if defined(KILL_IP_OPTIONS)
   fix_options(&request);
#endif /* KILL_IP_OPTIONS */

    /*
     * Find out and verify the remote host name. Sites concerned with
     * security may choose to refuse connections from hosts that pretend to
     * have someone elses host name.
     */  

#ifdef HOSTS_ACCESS
   if (!hosts_access(&request)) {
      ArgusLog  (deny_severity, "refused connect from %s", eval_client(&request));
      if (request.sink)
         request.sink(request.fd);
      return -1;
   } else
#endif

    /* Report remote client */
   ArgusLog  (allow_severity, "connect from %s", eval_client(&request));
   return (retn);

#else
   return (1);
#endif /* HAVE_TCP_WRAPPER */
}
