/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * argus client library
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: //depot/argus/argus-3.0/clients/common/argus_client.c#135 $
 * $DateTime: 2006/05/24 09:59:00 $
 * $Change: 857 $
 */


#ifndef ArgusClient
#define ArgusClient
#endif

#ifndef ArgusSort
#define ArgusSort
#endif

#ifndef ArgusMetric
#define ArgusMetric
#endif

#ifndef _REENTRANT
#define _REENTRANT
#endif

#include <stdlib.h>
#include <syslog.h>
#include <errno.h>

#include <math.h>
#include <ctype.h>

#include <sys/types.h>
#include <compat.h>

#define ARGUS_MAIN

#include <argus_def.h>
#include <argus_out.h>
#include <argus_util.h>
#include <argus_client.h>
#include <argus_sort.h>
#include <argus_metric.h>
#include <argus_histo.h>
#include <argus_labeler.h>

#include <rasplit.h>

#if defined(__OpenBSD__)
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif
 
#include <netinet/ip_icmp.h>
#include <netinet/igmp.h>
#include <netinet/tcp.h>

#include <argus_main.h>

#define RA_HASHSIZE		256

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifndef ETH_ALEN
#define ETH_ALEN  6
#endif

#ifndef AF_INET6
#define AF_INET6	23
#endif

int ArgusConnect(int, const struct sockaddr *, socklen_t, int);

struct ArgusRecord *ArgusGenerateRecord (struct ArgusRecordStruct *, unsigned char, char *);
struct ArgusRecordStruct *ArgusCopyRecordStruct (struct ArgusRecordStruct *);
void ArgusDeleteRecordStruct (struct ArgusParserStruct *, struct ArgusRecordStruct *);

#define ARGUS_MAX_BUFFER_READ	262144

#ifdef ARGUS_SASL
#include <saslint.h>

/*
   ArgusReadSaslStreamSocket() - this routine needs to keep reading data from the
                                 socket, decrypt it and then copy it into the
                                 standard ArgusReadBuffer, and then use the standard
                                 processing logic to read records.
*/


int ArgusReadSaslStreamSocket (struct ArgusParserStruct *parser, struct ArgusInput *);

int
ArgusReadSaslStreamSocket (struct ArgusParserStruct *parser, struct ArgusInput *input)
{
   int retn = 0, fd = input->fd, cnt = 0;
   u_int val = 0, *pval = &val;
   char *output = NULL, *ptr = NULL;
   u_int outputlen = 0;
   const int *ssfp;
   int result;

   if ((retn = sasl_getprop(input->sasl_conn, SASL_MAXOUTBUF, (const void **) &pval)) != SASL_OK)
      ArgusLog (LOG_ERR, "ArgusReadSaslStreamSocket: sasl_getprop %s", strerror(errno));

   if (val == 0) 
      val = ARGUS_MAX_BUFFER_READ;

   cnt = (input->ArgusBufferLen - input->ArgusSaslBufCnt);
   val = (cnt > val) ? val : cnt;

   if ((cnt = read (fd, input->ArgusSaslBuffer + input->ArgusSaslBufCnt, val)) > 0) {
      ptr = (char *) input->ArgusSaslBuffer;
      input->ArgusSaslBufCnt += cnt;

#ifdef ARGUSDEBUG
      ArgusDebug (5, "ArgusReadSaslStreamSocket (0x%x) read returned %d bytes\n", input, cnt);
#endif

      if ((result = sasl_getprop(input->sasl_conn, SASL_SSF, (const void **) &ssfp)) != SASL_OK)
         ArgusLog (LOG_ERR, "sasl_getprop: error %s\n", sasl_errdetail(input->sasl_conn));

      if (ssfp && (*ssfp > 0)) {
         if ((retn = sasl_decode (input->sasl_conn, ptr, input->ArgusSaslBufCnt, (const char **) &output, &outputlen)) == SASL_OK) {
#ifdef ARGUSDEBUG
            ArgusDebug (5, "ArgusReadSaslStreamSocket (0x%x) sasl_decoded %d bytes\n", input, outputlen);
#endif

         } else
            ArgusLog (LOG_ERR, "ArgusReadSaslStreamSocket: sasl_decode (0x%x, 0x%x, %d, 0x%x, 0x%x) failed %d",
                input->sasl_conn, ptr, cnt, &output, &outputlen, retn);
      } else {
         output = ptr;
         outputlen = input->ArgusSaslBufCnt;
      }

      if (outputlen) {
         while (input->ArgusSaslBufCnt) {
            int bytes, done = 0;
            bytes = (input->ArgusBufferLen - input->ArgusReadSocketCnt);
            bytes = (bytes > outputlen) ? outputlen : bytes;

            bcopy (output, input->ArgusReadPtr + input->ArgusReadSocketCnt, bytes);

            cnt = (input->ArgusBufferLen - input->ArgusSaslBufCnt);

            if (bytes > 0) {
               struct ArgusRecord *rec = NULL;
               input->ArgusReadSocketCnt += bytes;
               retn = 0;
               
               while (!retn && !done && !parser->RaParseDone) {
                  unsigned short length = 0;

                  switch (input->mode) {
                     case ARGUS_V2_DATA_SOURCE: {
                        struct ArgusV2Record *recv2 = (struct ArgusV2Record *)input->ArgusReadPtr;
                        if (input->ArgusReadSocketCnt >= sizeof(recv2->ahdr)) {
                           if ((length = ntohs(recv2->ahdr.length)) > 0) {
                              if (input->ArgusReadSocketCnt >= length)
                                 rec = (struct ArgusRecord *) ArgusConvertRecord (input, (char *)input->ArgusReadPtr);
                           } else {
                              ArgusLog (LOG_ALERT, "ArgusReadStreamSocket (0x%x) record length is zero");
                              retn = 1;
                           }
                        }
                        break;
                     }
                     case ARGUS_DATA_SOURCE: {
                        struct ArgusRecordHeader *recv3 = (struct ArgusRecordHeader *) input->ArgusReadPtr;
                        if (input->ArgusReadSocketCnt >= sizeof(*recv3)) {
                           if ((length = ntohs(recv3->len) * 4) > 0) {
                              if (input->ArgusReadSocketCnt >= length)
                                 rec = (struct ArgusRecord *) input->ArgusReadPtr;
                           } else {
                              ArgusLog (LOG_ALERT, "ArgusReadStreamSocket (0x%x) record length is zero");
                              retn = 1;
                           }
                        }
                        break;
                     }
                  }

                  if (rec) {
                     if (ArgusHandleDatum (ArgusParser, input, rec, &ArgusParser->ArgusFilterCode) == 1)
                        retn = 1;

                     input->offset += length;
                     input->ArgusReadPtr += length;
                     input->ArgusReadSocketCnt -= length;

                     if (input->ostop != -1)
                        if (input->offset > input->ostop)
                           retn = 1;

                     rec = NULL;

                  } else
                     done = 1;
               }

               if (input->ArgusReadPtr != input->ArgusReadBuffer) {
                  if (input->ArgusReadSocketCnt > 0)
                     memmove(input->ArgusReadBuffer, input->ArgusReadPtr, input->ArgusReadSocketCnt);
                  input->ArgusReadPtr = input->ArgusReadBuffer;
               }
            }
            input->ArgusSaslBufCnt = 0;
         }
      }

   } else {
      retn = 1;

      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      } else
         ArgusLog (LOG_ERR, "ArgusReadSaslStreamSocket: read (%d, 0x%x, %d) failed '%s'",
                       fd, input->ArgusSaslBuffer, val, strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadSaslStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}

#endif /* ARGUS_SASL */

struct ArgusRecord *ArgusNetFlowCallRecord (struct ArgusInput *, u_char **);
struct ArgusRecord *ArgusNetFlowDetailInt  (struct ArgusInput *, u_char **);
struct ArgusRecord *ArgusParseCiscoRecord (struct ArgusInput *, u_char **);

struct ArgusRecord *ArgusParseCiscoRecordV1 (struct ArgusInput *, u_char **);
struct ArgusRecord *ArgusParseCiscoRecordV5 (struct ArgusInput *, u_char **);
struct ArgusRecord *ArgusParseCiscoRecordV6 (struct ArgusInput *, u_char **);

unsigned char *ArgusNetFlowRecordHeader = NULL;

unsigned char ArgusNetFlowArgusRecordBuf[1024];
struct ArgusRecord *ArgusNetFlowArgusRecord = (struct ArgusRecord *) ArgusNetFlowArgusRecordBuf;

struct ArgusRecord * 
ArgusParseCiscoRecordV1 (struct ArgusInput *input, u_char **ptr)
{
   CiscoFlowEntryV1_t  *entryPtrV1 = (CiscoFlowEntryV1_t *) *ptr;
   CiscoFlowHeaderV1_t *hdrPtrV1   = (CiscoFlowHeaderV1_t *) ArgusNetFlowRecordHeader;
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) &ArgusNetFlowArgusRecordBuf[4];
   int i;

   *ptr += sizeof(CiscoFlowEntryV1_t);
   bzero ((char *) argus, sizeof(ArgusNetFlowArgusRecordBuf));
   argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
   argus->hdr.cause   = ARGUS_STATUS;
   argus->hdr.len     = 1;

   if (hdrPtrV1) {
      for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
         int ind = (1 << i);
         switch (ind) {
            case ARGUS_FLOW_INDEX: {
               struct ArgusFlow *flow = (struct ArgusFlow *) dsr;
               flow->hdr.type              = ARGUS_FLOW_DSR;
               flow->hdr.subtype           = ARGUS_FLOW_CLASSIC5TUPLE;
               flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_IPV4;
               flow->hdr.argus_dsrvl8.len  = 5;
               flow->ip_flow.ip_src = ntohl(entryPtrV1->srcaddr);
               flow->ip_flow.ip_dst = ntohl(entryPtrV1->dstaddr);

               switch (flow->ip_flow.ip_p = entryPtrV1->prot) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     flow->ip_flow.sport  = ntohs(entryPtrV1->srcport);
                     flow->ip_flow.dport  = ntohs(entryPtrV1->dstport);
                  break;
         
                  case IPPROTO_ICMP:
                     flow->icmp_flow.type  = ((char *)&entryPtrV1->dstport)[0];
                     flow->icmp_flow.code  = ((char *)&entryPtrV1->dstport)[1];
                  break;
               }
               dsr += flow->hdr.argus_dsrvl8.len;
               argus->hdr.len += flow->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TIME_INDEX: {
               struct ArgusTimeObject *time = (struct ArgusTimeObject *) dsr;
               long timeval;

               time->hdr.type               = ARGUS_TIME_DSR;
               time->hdr.subtype            = ARGUS_TIME_ABSOLUTE_RANGE;
               time->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_UTC_MICROSECONDS;
               time->hdr.argus_dsrvl8.len   = 5;               

               timeval = ntohl(entryPtrV1->first);
               time->src.start.tv_sec   = (timeval - (long)hdrPtrV1->sysUptime)/1000; 
               time->src.start.tv_sec  += hdrPtrV1->unix_secs;

               time->src.start.tv_usec  = ((timeval - (long)hdrPtrV1->sysUptime)%1000) * 1000; 
               time->src.start.tv_usec += hdrPtrV1->unix_nsecs/1000;

               if (time->src.start.tv_usec >= 1000000) {
                  time->src.start.tv_sec++;
                  time->src.start.tv_usec -= 1000000;
               }
               if (time->src.start.tv_usec < 0) {
                  time->src.start.tv_sec--;
                  time->src.start.tv_usec += 1000000;
               }

               timeval = ntohl(entryPtrV1->last);
               time->src.end.tv_sec   = (timeval - (long)hdrPtrV1->sysUptime)/1000;
               time->src.end.tv_sec  += hdrPtrV1->unix_secs;

               time->src.end.tv_usec  = ((timeval - (long)hdrPtrV1->sysUptime)%1000) * 1000;
               time->src.end.tv_usec += hdrPtrV1->unix_nsecs/1000;

               if (time->src.end.tv_usec >= 1000000) {
                  time->src.end.tv_sec++;
                  time->src.end.tv_usec -= 1000000;
               }
               if (time->src.end.tv_usec < 0) {
                  time->src.end.tv_sec--;
                  time->src.end.tv_usec += 1000000;
               }

               time->src.start.tv_usec = (time->src.start.tv_usec / 1000) * 1000;
               time->src.end.tv_usec  = (time->src.end.tv_usec / 1000) * 1000;
               dsr += time->hdr.argus_dsrvl8.len;
               argus->hdr.len += time->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TRANSPORT_INDEX: {
               if (input->addr.s_addr != 0) {
                  struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;
                  trans->hdr.type               = ARGUS_TRANSPORT_DSR;
                  trans->hdr.subtype            = ARGUS_SRC;
                  trans->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_IPV4;
                  trans->hdr.argus_dsrvl8.len   = 2;
                  trans->srcid.a_un.ipv4        = input->addr.s_addr;

                  dsr += trans->hdr.argus_dsrvl8.len;
                  argus->hdr.len += trans->hdr.argus_dsrvl8.len;
               }
               break;
            }
            case ARGUS_IPATTR_INDEX: {
               struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
               attr->hdr.type               = ARGUS_IPATTR_DSR;
               attr->hdr.subtype            = 0;
               attr->hdr.argus_dsrvl8.qual  = ARGUS_IPATTR_SRC;
               attr->hdr.argus_dsrvl8.len   = 2;
               attr->src.tos                = entryPtrV1->tos; 
               attr->src.ttl                = 0;
               attr->src.ip_id              = 0;
               dsr += attr->hdr.argus_dsrvl8.len;
               argus->hdr.len += attr->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_METRIC_INDEX: {
               struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *) dsr;
               long long *ptr;
                                    
               metric->hdr.type              = ARGUS_METER_DSR;
               metric->hdr.subtype           = ARGUS_METER_PKTS_BYTES;
               metric->hdr.argus_dsrvl8.qual = ARGUS_SRC_LONGLONG;
               metric->hdr.argus_dsrvl8.len  = 5;
               ptr    = &metric->src.pkts;
               *ptr++ = ntohl(entryPtrV1->pkts);
               *ptr++ = ntohl(entryPtrV1->bytes);
               dsr += metric->hdr.argus_dsrvl8.len;
               argus->hdr.len += metric->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_MAC_INDEX: {
               struct ArgusMacStruct *mac = (struct ArgusMacStruct *) dsr;
               mac->hdr.type              = ARGUS_MAC_DSR;
               mac->hdr.subtype           = 0;
               mac->hdr.argus_dsrvl8.len  = 5;
               entryPtrV1->input = ntohs(entryPtrV1->input);
               entryPtrV1->output = ntohs(entryPtrV1->output);
#if defined(HAVE_SOLARIS)
               bcopy((char *)&entryPtrV1->input, (char *)&mac->mac_union.ether.ehdr.ether_shost.ether_addr_octet[4], 2);
               bcopy((char *)&entryPtrV1->output,(char *)&mac->mac_union.ether.ehdr.ether_dhost.ether_addr_octet[4], 2);
#else
               bcopy((char *)&entryPtrV1->input, (char *)&mac->mac_union.ether.ehdr.ether_shost[4], 2);
               bcopy((char *)&entryPtrV1->output,(char *)&mac->mac_union.ether.ehdr.ether_dhost[4], 2);
#endif

               dsr += mac->hdr.argus_dsrvl8.len;
               argus->hdr.len += mac->hdr.argus_dsrvl8.len;
               break;
            }

            case ARGUS_NETWORK_INDEX: {
               if (entryPtrV1->prot == IPPROTO_TCP) {
                  struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *) dsr;
                  net->hdr.type              = ARGUS_NETWORK_DSR;
                  net->hdr.subtype           = ARGUS_TCP_STATUS;
                  net->hdr.argus_dsrvl8.len  = 3;
                  net->net_union.tcpstatus.src = entryPtrV1->flags;

                  dsr += net->hdr.argus_dsrvl8.len;
                  argus->hdr.len += net->hdr.argus_dsrvl8.len;
               }
            }
         }
      }
   }

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecordV1 (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return(argus);
}


struct ArgusRecord * 
ArgusParseCiscoRecordV5 (struct ArgusInput *input, u_char **ptr)
{
   CiscoFlowEntryV5_t  *entryPtrV5 = (CiscoFlowEntryV5_t *) *ptr;
   CiscoFlowHeaderV5_t *hdrPtrV5   = (CiscoFlowHeaderV5_t *) ArgusNetFlowRecordHeader;
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) &ArgusNetFlowArgusRecordBuf[4];
   int i;

   *ptr += sizeof(CiscoFlowEntryV5_t);
   bzero ((char *) argus, sizeof(ArgusNetFlowArgusRecordBuf));
   argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
   argus->hdr.cause   = ARGUS_STATUS;
   argus->hdr.len     = 1;

   if (hdrPtrV5) {
      for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
         switch (i) {
            case ARGUS_FLOW_INDEX: {
               struct ArgusFlow *flow = (struct ArgusFlow *) dsr;
               flow->hdr.type              = ARGUS_FLOW_DSR;
               flow->hdr.subtype           = ARGUS_FLOW_CLASSIC5TUPLE;
               flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_IPV4;
               flow->hdr.argus_dsrvl8.len  = 5;
               flow->ip_flow.ip_src = ntohl(entryPtrV5->srcaddr);
               flow->ip_flow.ip_dst = ntohl(entryPtrV5->dstaddr);

               switch (flow->ip_flow.ip_p = entryPtrV5->prot) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     flow->ip_flow.sport  = ntohs(entryPtrV5->srcport);
                     flow->ip_flow.dport  = ntohs(entryPtrV5->dstport);
                  break;
         
                  case IPPROTO_ICMP:
                     flow->icmp_flow.type  = ((char *)&entryPtrV5->dstport)[0];
                     flow->icmp_flow.code  = ((char *)&entryPtrV5->dstport)[1];
                  break;
               }
               dsr += flow->hdr.argus_dsrvl8.len;
               argus->hdr.len += flow->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TIME_INDEX: {
               struct ArgusTimeObject *time = (struct ArgusTimeObject *) dsr;
               long timeval;

               time->hdr.type               = ARGUS_TIME_DSR;
               time->hdr.subtype            = ARGUS_TIME_ABSOLUTE_RANGE;
               time->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_UTC_MICROSECONDS;
               time->hdr.argus_dsrvl8.len   = 5;               

               timeval = ntohl(entryPtrV5->first);
               time->src.start.tv_sec   = (timeval - (long)hdrPtrV5->sysUptime)/1000; 
               time->src.start.tv_sec  += hdrPtrV5->unix_secs;

               time->src.start.tv_usec  = ((timeval - (long)hdrPtrV5->sysUptime)%1000) * 1000; 
               time->src.start.tv_usec += hdrPtrV5->unix_nsecs/1000;

               if (time->src.start.tv_usec >= 1000000) {
                  time->src.start.tv_sec++;
                  time->src.start.tv_usec -= 1000000;
               }
               if (time->src.start.tv_usec < 0) {
                  time->src.start.tv_sec--;
                  time->src.start.tv_usec += 1000000;
               }

               timeval = ntohl(entryPtrV5->last);
               time->src.end.tv_sec   = (timeval - (long)hdrPtrV5->sysUptime)/1000;
               time->src.end.tv_sec  += hdrPtrV5->unix_secs;

               time->src.end.tv_usec  = ((timeval - (long)hdrPtrV5->sysUptime)%1000) * 1000;
               time->src.end.tv_usec += hdrPtrV5->unix_nsecs/1000;

               if (time->src.end.tv_usec >= 1000000) {
                  time->src.end.tv_sec++;
                  time->src.end.tv_usec -= 1000000;
               }
               if (time->src.end.tv_usec < 0) {
                  time->src.end.tv_sec--;
                  time->src.end.tv_usec += 1000000;
               }

               time->src.start.tv_usec = (time->src.start.tv_usec / 1000) * 1000;
               time->src.end.tv_usec  = (time->src.end.tv_usec / 1000) * 1000;
               dsr += time->hdr.argus_dsrvl8.len;
               argus->hdr.len += time->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TRANSPORT_INDEX: {
               if (input->addr.s_addr != 0) {
                  struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;
                  trans->hdr.type               = ARGUS_TRANSPORT_DSR;
                  trans->hdr.subtype            = ARGUS_SRC;
                  trans->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_IPV4;
                  trans->hdr.argus_dsrvl8.len   = 2;
                  trans->srcid.a_un.ipv4        = input->addr.s_addr;

                  dsr += trans->hdr.argus_dsrvl8.len;
                  argus->hdr.len += trans->hdr.argus_dsrvl8.len;
               }
               break;
            }
            case ARGUS_IPATTR_INDEX: {
               struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
               attr->hdr.type               = ARGUS_IPATTR_DSR;
               attr->hdr.subtype            = 0;
               attr->hdr.argus_dsrvl8.qual  = ARGUS_IPATTR_SRC;
               attr->hdr.argus_dsrvl8.len   = 2;
               attr->src.tos                = entryPtrV5->tos; 
               attr->src.ttl                = 0;
               attr->src.ip_id              = 0;
               dsr += attr->hdr.argus_dsrvl8.len;
               argus->hdr.len += attr->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_METRIC_INDEX: {
               struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *) dsr;
               long long *ptr;
                                    
               metric->hdr.type              = ARGUS_METER_DSR;
               metric->hdr.subtype           = ARGUS_METER_PKTS_BYTES;
               metric->hdr.argus_dsrvl8.qual = ARGUS_SRC_LONGLONG;
               metric->hdr.argus_dsrvl8.len  = 5;
               ptr    = &metric->src.pkts;

               *ptr++ = ntohl(entryPtrV5->pkts);
               *ptr++ = ntohl(entryPtrV5->bytes);
               dsr += metric->hdr.argus_dsrvl8.len;
               argus->hdr.len += metric->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_MAC_INDEX: {
               struct ArgusMacStruct *mac = (struct ArgusMacStruct *) dsr;
               mac->hdr.type              = ARGUS_MAC_DSR;
               mac->hdr.subtype           = 0;
               mac->hdr.argus_dsrvl8.len  = 5;
               entryPtrV5->input = ntohs(entryPtrV5->input);
               entryPtrV5->output = ntohs(entryPtrV5->output);
#if defined(HAVE_SOLARIS)
               bcopy((char *)&entryPtrV5->input, (char *)&mac->mac_union.ether.ehdr.ether_shost.ether_addr_octet[4], 2);
               bcopy((char *)&entryPtrV5->output,(char *)&mac->mac_union.ether.ehdr.ether_dhost.ether_addr_octet[4], 2);
#else
               bcopy((char *)&entryPtrV5->input, (char *)&mac->mac_union.ether.ehdr.ether_shost[4], 2);
               bcopy((char *)&entryPtrV5->output,(char *)&mac->mac_union.ether.ehdr.ether_dhost[4], 2);
#endif

               dsr += mac->hdr.argus_dsrvl8.len;
               argus->hdr.len += mac->hdr.argus_dsrvl8.len;
               break;
            }

            case ARGUS_NETWORK_INDEX: {
               if (entryPtrV5->prot == IPPROTO_TCP) {
                  struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *) dsr;
                  struct ArgusTCPStatus *tcp = (struct ArgusTCPStatus *)&net->net_union.tcpstatus;

                  net->hdr.type              = ARGUS_NETWORK_DSR;
                  net->hdr.subtype           = ARGUS_TCP_STATUS;
                  net->hdr.argus_dsrvl8.len  = 3;
                  net->net_union.tcpstatus.src = entryPtrV5->tcp_flags;

                  if (entryPtrV5->tcp_flags & TH_RST) 
                     tcp->status |= ARGUS_RESET;
          
                  if (entryPtrV5->tcp_flags & TH_FIN)
                     tcp->status |= ARGUS_FIN;
          
                  if ((entryPtrV5->tcp_flags & TH_ACK) || (entryPtrV5->tcp_flags & TH_PUSH) || (entryPtrV5->tcp_flags & TH_URG))
                     tcp->status |= ARGUS_CON_ESTABLISHED;
          
                  switch (entryPtrV5->tcp_flags & (TH_SYN|TH_ACK)) {
                     case (TH_SYN):  
                        tcp->status |= ARGUS_SAW_SYN;
                        break;
             
                     case (TH_SYN|TH_ACK): 
                        tcp->status |= ARGUS_SAW_SYN_SENT;  
                        if (ntohl(entryPtrV5->pkts) > 1)
                           tcp->status &= ~(ARGUS_CON_ESTABLISHED);
                        break;
                  }

                  dsr += net->hdr.argus_dsrvl8.len;
                  argus->hdr.len += net->hdr.argus_dsrvl8.len;
               }
            }
         }
      }
   }

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecordV5 (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return(argus);
}




struct ArgusRecord * 
ArgusParseCiscoRecordV6 (struct ArgusInput *input, u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
/*
   CiscoFlowEntryV6_t  *entryPtrV6 = (CiscoFlowEntryV6_t *) *ptr;
   CiscoFlowHeaderV6_t *hdrPtrV6   = (CiscoFlowHeaderV6_t *) ArgusNetFlowRecordHeader;
   struct ArgusMacStruct mac;
*/

   *ptr += sizeof(CiscoFlowEntryV6_t);
   bzero ((char *) argus, sizeof (*argus));
   argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
   argus->hdr.cause   = ARGUS_STATUS;
   argus->hdr.len     = sizeof(argus->hdr) + sizeof(argus->argus_far);

/*
   argus->ahdr.status |= ETHERTYPE_IP;

   argus->argus_far.type   = ARGUS_FAR;
   argus->argus_far.length = sizeof(argus->argus_far);

   if (hdrPtrV6) {
      long time; 
      time = ntohl(entryPtrV6->first);
      argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
      argus->argus_far.time.start.tv_sec += hdrPtrV6->unix_secs;

      argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
      argus->argus_far.time.start.tv_usec += hdrPtrV6->unix_nsecs/1000;

      if (argus->argus_far.time.start.tv_usec >= 1000000) {
         argus->argus_far.time.start.tv_sec++;
         argus->argus_far.time.start.tv_usec -= 1000000;
      }
      if (argus->argus_far.time.start.tv_usec < 0) {
         argus->argus_far.time.start.tv_sec--;
         argus->argus_far.time.start.tv_usec += 1000000;
      }
      
      time = ntohl(entryPtrV6->last);
      argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
      argus->argus_far.time.last.tv_sec += hdrPtrV6->unix_secs;
      
      argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
      argus->argus_far.time.last.tv_usec += hdrPtrV6->unix_nsecs/1000;

      if (argus->argus_far.time.last.tv_usec >= 1000000) {
         argus->argus_far.time.last.tv_sec++;
         argus->argus_far.time.last.tv_usec -= 1000000;
      }
      if (argus->argus_far.time.last.tv_usec < 0) {
         argus->argus_far.time.last.tv_sec--;
         argus->argus_far.time.last.tv_usec += 1000000;
      }

      argus->argus_far.time.start.tv_usec = (argus->argus_far.time.start.tv_usec / 1000) * 1000;
      argus->argus_far.time.last.tv_usec  = (argus->argus_far.time.last.tv_usec / 1000) * 1000;
   }

   argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV6->srcaddr);
   argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV6->dstaddr);
   argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV6->srcport);
   argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV6->dstport);
   argus->argus_far.flow.ip_flow.ip_p   = entryPtrV6->prot;
   argus->argus_far.attr_ip.stos        = entryPtrV6->tos;
   argus->argus_far.src.count    = ntohl(entryPtrV6->pkts);
   argus->argus_far.src.bytes    = ntohl(entryPtrV6->bytes);
   argus->argus_far.src.appbytes = 0;

   switch (argus->argus_far.flow.ip_flow.ip_p) {
      case IPPROTO_TCP: {
         struct ArgusTCPObject tcpbuf, *tcp = &tcpbuf;

         bzero ((char *) tcp, sizeof(*tcp));
         tcp->type = ARGUS_TCP_DSR;
         tcp->length = sizeof(struct ArgusTCPObject);
         tcp->src.flags    = entryPtrV6->tcp_flags;

         if (tcp->src.flags & TH_RST)
            tcp->status |= ARGUS_RESET;

         if (tcp->src.flags & TH_FIN)
            tcp->status |= ARGUS_FIN;

         if ((tcp->src.flags & TH_ACK) || (tcp->src.flags & TH_PUSH) || (tcp->src.flags & TH_URG))
            tcp->status |= ARGUS_CON_ESTABLISHED;

         switch (tcp->src.flags & (TH_SYN|TH_ACK)) {
            case (TH_SYN):
               tcp->status |= ARGUS_SAW_SYN;
               break;

            case (TH_SYN|TH_ACK):
               tcp->state |= ARGUS_SAW_SYN_SENT;
               if ((argus->argus_far.src.count + argus->argus_far.dst.count) == 1)
                  tcp->state &= ~(ARGUS_CON_ESTABLISHED);
               break;
         }

         bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
         argus->ahdr.length += sizeof(*tcp);
      }
      break;
   }

   bzero ((char *)&mac, sizeof (mac));
   mac.type   = ARGUS_MAC_DSR;
   mac.length = sizeof(mac);
   mac.status = 0;
   entryPtrV6->input = ntohs(entryPtrV6->input);
   entryPtrV6->output = ntohs(entryPtrV6->output);

   bcopy((char *)&entryPtrV6->input, (char *)&mac.phys_union.ether.ethersrc[4], 2);
   bcopy((char *)&entryPtrV6->output,(char *)&mac.phys_union.ether.etherdst[4], 2);

   bcopy ((char *)&mac, &((char *)argus)[argus->ahdr.length], sizeof(mac));
   argus->ahdr.length += sizeof(mac);
*/

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return(argus);
}

struct ArgusRecord *
ArgusParseCiscoRecord (struct ArgusInput *input, u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   unsigned short *sptr = (unsigned short *) *ptr;

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) version %h\n", *ptr, *sptr);
#endif

   switch (*sptr) {
      case Version1: {
/*
         CiscoFlowHeaderV1_t *hdrPtrV1   = (CiscoFlowHeaderV1_t *) *ptr;
         CiscoFlowEntryV1_t  *entryPtrV1 = (CiscoFlowEntryV1_t *) (hdrPtrV1 + 1);
*/

         bzero ((char *) argus, sizeof (*argus));
         argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
         argus->hdr.cause   = ARGUS_STATUS;
         argus->hdr.len     = sizeof(argus->hdr) + sizeof(argus->argus_far);
/*
         argus->ahdr.status |= ETHERTYPE_IP;

         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV1) {
            long time; 
            time = ntohl(entryPtrV1->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV1->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV1->unix_nsecs/1000;

            if (argus->argus_far.time.start.tv_usec >= 1000000) {
               argus->argus_far.time.start.tv_sec++;
               argus->argus_far.time.start.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.start.tv_usec < 0) {
               argus->argus_far.time.start.tv_sec--;
               argus->argus_far.time.start.tv_usec += 1000000;
            }

            time = ntohl(entryPtrV1->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV1->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV1->unix_nsecs/1000;

            if (argus->argus_far.time.last.tv_usec >= 1000000) {
               argus->argus_far.time.last.tv_sec++;
               argus->argus_far.time.last.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.last.tv_usec < 0) {
               argus->argus_far.time.last.tv_sec--;
               argus->argus_far.time.last.tv_usec += 1000000;
            }

            argus->argus_far.time.start.tv_usec = (argus->argus_far.time.start.tv_usec / 1000) * 1000;
            argus->argus_far.time.last.tv_usec  = (argus->argus_far.time.last.tv_usec / 1000) * 1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV1->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV1->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV1->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV1->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV1->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV1->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV1->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV1->bytes);
*/

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version5: {
/*
         CiscoFlowHeaderV5_t *hdrPtrV5   = (CiscoFlowHeaderV5_t *) ptr;
         CiscoFlowEntryV5_t  *entryPtrV5 = (CiscoFlowEntryV5_t *) (hdrPtrV5 + 1);
*/

         bzero ((char *) argus, sizeof (*argus));
         argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
         argus->hdr.cause   = ARGUS_STATUS;
         argus->hdr.len     = sizeof(argus->hdr) + sizeof(argus->argus_far);
/*
         argus->ahdr.status |= ETHERTYPE_IP; 
   
         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV5) {
            long time;
            time = ntohl(entryPtrV5->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV5->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV5->unix_nsecs/1000;

            if (argus->argus_far.time.start.tv_usec >= 1000000) {
               argus->argus_far.time.start.tv_sec++;
               argus->argus_far.time.start.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.start.tv_usec < 0) {
               argus->argus_far.time.start.tv_sec--;
               argus->argus_far.time.start.tv_usec += 1000000;
            }
      
            time = ntohl(entryPtrV5->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV5->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV5->unix_nsecs/1000;

            if (argus->argus_far.time.last.tv_usec >= 1000000) {
               argus->argus_far.time.last.tv_sec++;
               argus->argus_far.time.last.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.last.tv_usec < 0) {
               argus->argus_far.time.last.tv_sec--;
               argus->argus_far.time.last.tv_usec += 1000000;
            }

            argus->argus_far.time.start.tv_usec = (argus->argus_far.time.start.tv_usec / 1000) * 1000;
            argus->argus_far.time.last.tv_usec  = (argus->argus_far.time.last.tv_usec / 1000) * 1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV5->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV5->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV5->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV5->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV5->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV5->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV5->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV5->bytes);
         argus->argus_far.src.appbytes = 0;
*/

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version6: {
/*
         CiscoFlowHeaderV6_t *hdrPtrV6   = (CiscoFlowHeaderV6_t *) ptr;
         CiscoFlowEntryV6_t  *entryPtrV6 = (CiscoFlowEntryV6_t *) (hdrPtrV6 + 1);
*/

         bzero ((char *) argus, sizeof (*argus));
         argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
         argus->hdr.cause   = ARGUS_STATUS;
         argus->hdr.len     = sizeof(argus->hdr) + sizeof(argus->argus_far);
/*
         argus->ahdr.status |= ETHERTYPE_IP; 
   
         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV6) {
            long time;
            time = ntohl(entryPtrV6->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV6->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV6->unix_nsecs/1000;
      
            if (argus->argus_far.time.start.tv_usec >= 1000000) {
               argus->argus_far.time.start.tv_sec++;
               argus->argus_far.time.start.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.start.tv_usec < 0) {
               argus->argus_far.time.start.tv_sec--;
               argus->argus_far.time.start.tv_usec += 1000000;
            }

            time = ntohl(entryPtrV6->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV6->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV6->unix_nsecs/1000;

            if (argus->argus_far.time.last.tv_usec >= 1000000) {
               argus->argus_far.time.last.tv_sec++;
               argus->argus_far.time.last.tv_usec -= 1000000;
            }
            if (argus->argus_far.time.last.tv_usec < 0) {
               argus->argus_far.time.last.tv_sec--;
               argus->argus_far.time.last.tv_usec += 1000000;
            }

            argus->argus_far.time.start.tv_usec = (argus->argus_far.time.start.tv_usec / 1000) * 1000;
            argus->argus_far.time.last.tv_usec  = (argus->argus_far.time.last.tv_usec / 1000) * 1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV6->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV6->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV6->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV6->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV6->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV6->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV6->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV6->bytes);
         argus->argus_far.src.appbytes = 0;
*/

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version8: {
         break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


struct ArgusRecord *
ArgusNetFlowCallRecord (struct ArgusInput *input, u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   BinaryRecord_CallRecord_V1 *call = (BinaryRecord_CallRecord_V1 *) *ptr;
   struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) &ArgusNetFlowArgusRecordBuf[4];
   int i;

   if (*ptr) {
      bzero ((char *) argus, sizeof (*argus));
      argus->hdr.type    = ARGUS_FAR | ARGUS_NETFLOW | ARGUS_VERSION;
      argus->hdr.cause   = ARGUS_STATUS;
      argus->hdr.len     = 1;

      for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
         switch (i) {
            case ARGUS_FLOW_INDEX: {
               struct ArgusFlow *flow = (struct ArgusFlow *) dsr;
               flow->hdr.type              = ARGUS_FLOW_DSR;
               flow->hdr.subtype           = ARGUS_FLOW_CLASSIC5TUPLE;
               flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_IPV4;
               flow->hdr.argus_dsrvl8.len  = 5;
               flow->ip_flow.ip_src = ntohl(call->srcaddr);
               flow->ip_flow.ip_dst = ntohl(call->dstaddr);

               switch (flow->ip_flow.ip_p = call->prot) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP:
                     flow->ip_flow.sport  = ntohs(call->srcport);
                     flow->ip_flow.dport  = ntohs(call->dstport);
                  break;
         
                  case IPPROTO_ICMP:
                     flow->icmp_flow.type  = ((char *)&call->dstport)[0];
                     flow->icmp_flow.code  = ((char *)&call->dstport)[1];
                  break;
               }
               dsr += flow->hdr.argus_dsrvl8.len;
               argus->hdr.len += flow->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TIME_INDEX: {
               struct ArgusTimeObject *time = (struct ArgusTimeObject *) dsr;

               time->hdr.type               = ARGUS_TIME_DSR;
               time->hdr.subtype            = ARGUS_TIME_ABSOLUTE_RANGE;
               time->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_UTC_MICROSECONDS;
               time->hdr.argus_dsrvl8.len   = 5;               

               time->src.start.tv_sec  = ntohl(call->starttime);
               time->src.end.tv_sec  = ntohl(call->endtime);

               time->src.end.tv_usec = ntohl(call->activetime) % 1000000;
               time->src.end.tv_sec += ntohl(call->activetime) / 1000000;

               time->src.start.tv_usec = (time->src.start.tv_usec / 1000) * 1000;
               time->src.end.tv_usec  = (time->src.end.tv_usec / 1000) * 1000;
               dsr += time->hdr.argus_dsrvl8.len;
               argus->hdr.len += time->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_TRANSPORT_INDEX: {
               if (input->addr.s_addr != 0) {
                  struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;
                  trans->hdr.type               = ARGUS_TRANSPORT_DSR;
                  trans->hdr.subtype            = ARGUS_SRC;
                  trans->hdr.argus_dsrvl8.qual  = ARGUS_TYPE_IPV4;
                  trans->hdr.argus_dsrvl8.len   = 2;
                  trans->srcid.a_un.ipv4        = input->addr.s_addr;

                  dsr += trans->hdr.argus_dsrvl8.len;
                  argus->hdr.len += trans->hdr.argus_dsrvl8.len;
               }
               break;
            }
            case ARGUS_IPATTR_INDEX: {
               struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
               attr->hdr.type               = ARGUS_IPATTR_DSR;
               attr->hdr.subtype            = 0;
               attr->hdr.argus_dsrvl8.qual  = ARGUS_IPATTR_SRC;
               attr->hdr.argus_dsrvl8.len   = 2;
               attr->src.tos                = call->tos; 
               attr->src.ttl                = 0;
               attr->src.ip_id              = 0;
               dsr += attr->hdr.argus_dsrvl8.len;
               argus->hdr.len += attr->hdr.argus_dsrvl8.len;
               break;
            }
            case ARGUS_METRIC_INDEX: {
               struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *) dsr;
               long long *ptr;
                                    
               metric->hdr.type              = ARGUS_METER_DSR;
               metric->hdr.subtype           = ARGUS_METER_PKTS_BYTES;
               metric->hdr.argus_dsrvl8.qual = ARGUS_SRC_LONGLONG;
               metric->hdr.argus_dsrvl8.len  = 7;
               ptr    = &metric->src.pkts;
               *ptr++ = ntohl(call->pkts);
               *ptr++ = ntohl(call->octets);
               dsr += metric->hdr.argus_dsrvl8.len;
               argus->hdr.len += metric->hdr.argus_dsrvl8.len;
               break;
            }
         }
      }

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif
   }
   
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusNetFlowCallRecord (0x%x) returns 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


struct ArgusRecord *
ArgusNetFlowDetailInt (struct ArgusInput *input, u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   BinaryRecord_DetailInterface_V1  *dint = (BinaryRecord_DetailInterface_V1 *) *ptr;

   if (*ptr) {
      dint = NULL;
      bzero ((char *) argus, sizeof (*argus));
   }


#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusNetFlowDetailInt (0x%x) returns 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


ArgusNetFlowHandler ArgusLookUpNetFlow(struct ArgusInput *, int);

struct ArgusNetFlowParsers {
   int type, size;
   ArgusNetFlowHandler proc;
};

struct ArgusNetFlowParsers ArgusNetFlowParsers [] = {
   { SourceNode, 0, NULL },
   { DestNode, 0, NULL },
   { HostMatrix, 0, NULL },
   { SourcePort, 0, NULL },
   { DestPort, 0, NULL },
   { Protocol, 0, NULL },
   { DetailDestNode, 0, NULL },
   { DetailHostMatrix, 0, NULL },
   { DetailInterface, sizeof(BinaryRecord_DetailInterface_V1), ArgusNetFlowDetailInt },
   { CallRecord, sizeof(BinaryRecord_CallRecord_V1), ArgusNetFlowCallRecord },
   { ASMatrix, 0, NULL },
   { NetMatrix, 0, NULL },
   { DetailSourceNode, 0, NULL },
   { DetailASMatrix, 0, NULL },
   { ASHostMatrix, 0, NULL },
   { HostMatrixInterface, 0, NULL },
   { DetailCallRecord, 0, NULL },
   { RouterAS, 0, NULL },
   { RouterProtoPort, 0, NULL },
   { RouterSrcPrefix, 0, NULL },
   { RouterDstPrefix, 0, NULL },
   { RouterPrefix, 0, NULL },
   { -1, 0, NULL },
};
   

ArgusNetFlowHandler 
ArgusLookUpNetFlow(struct ArgusInput *input, int type)
{
   ArgusNetFlowHandler retn = NULL;
   struct ArgusNetFlowParsers *p = ArgusNetFlowParsers;

   do {
      if (type == p->type) {
         retn = p->proc;
         input->ArgusReadSize = p->size;
         input->ArgusReadSocketSize = p->size;
         break;
      }
      p++;
   } while (p->type != -1);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusLookUpNetFlow (0x%x, %d) returning 0x%x\n", input, type, retn);
#endif

   return (retn);
}


#define CISCO_VERSION_1		1
#define CISCO_VERSION_5		5
#define CISCO_VERSION_6		6
#define CISCO_VERSION_8		8

int ArgusReadCiscoStreamSocket (struct ArgusParserStruct *, struct ArgusInput *);

int
ArgusReadCiscoStreamSocket (struct ArgusParserStruct *parser, struct ArgusInput *input)
{
   int retn = 0, cnt = 0, bytes = 0, done = 0;

   if (!(input))
      return (retn);

   bytes = (input->ArgusBufferLen - input->ArgusReadSocketCnt);
   bytes = (bytes > ARGUS_MAX_BUFFER_READ) ? ARGUS_MAX_BUFFER_READ : bytes;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) starting\n", input);
#endif

   if (input->file != NULL) {
      if (input->file == stdin) {
         int sretn;
         fd_set readmask;
         struct timeval wait;

         FD_ZERO (&readmask);
         FD_SET (fileno(stdin), &readmask);
         wait.tv_sec = 0;
         wait.tv_usec = 150000;

         if (!((sretn = select (fileno(stdin)+1, &readmask, NULL, NULL, &wait)) > 0))
            return (sretn);
          else
            cnt = fread (input->ArgusReadPtr + input->ArgusReadSocketCnt, 1, bytes, input->file);
      } else
         cnt = fread (input->ArgusReadPtr + input->ArgusReadSocketCnt, 1, bytes, input->file);
   } else
      cnt = read (input->fd, input->ArgusReadPtr + input->ArgusReadSocketCnt, bytes);

   if (cnt > 0) {
      input->ArgusReadSocketCnt += cnt;

#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) read %d bytes, total %d need %d\n",
                      input, cnt, input->ArgusReadSocketCnt, input->ArgusReadSocketSize);
#endif

      while ((input->ArgusReadSocketCnt >= input->ArgusReadSocketSize) && !done) {
         unsigned int size = input->ArgusReadSocketSize;

         switch (input->ArgusReadSocketState) {
            case ARGUS_READINGPREHDR: {
               unsigned short *sptr = (unsigned short *) input->ArgusReadPtr;

               input->ArgusReadCiscoVersion = ntohs(*sptr++);
               input->ArgusReadSocketNum  = ntohs(*sptr);

               switch (input->ArgusReadCiscoVersion) {
                  case CISCO_VERSION_1:
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowHeaderV1_t) - 4;
                     input->ArgusReadPtr = &input->ArgusReadBuffer[size];
                     break;

                  case CISCO_VERSION_5:
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowHeaderV5_t) - 4;
                     input->ArgusReadPtr = &input->ArgusReadBuffer[size];
                     break;

                  default: {
#ifdef ARGUSDEBUG
                     ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) read version %d preheader num %d\n",
                                       input, input->ArgusReadCiscoVersion, input->ArgusReadSocketNum);
#endif
                  }
               }

               input->ArgusReadSocketState = ARGUS_READINGHDR;
               input->ArgusReadSocketCnt  -= size;
               break;
            }

            case ARGUS_READINGHDR: {
#ifdef ARGUSDEBUG
               ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read record header\n", input);
#endif
               switch (input->ArgusReadCiscoVersion) {
                  case CISCO_VERSION_1: {
                     CiscoFlowHeaderV1_t *ArgusNetFlow = (CiscoFlowHeaderV1_t *) input->ArgusReadBuffer;
                     CiscoFlowHeaderV1_t *nfptr = ArgusNetFlow;

                     input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV1;
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowEntryV1_t);
                     input->ArgusReadPtr = &input->ArgusReadBuffer[sizeof(CiscoFlowHeaderV1_t)];

                     ArgusNetFlow->version    = ntohs(nfptr->version);
                     ArgusNetFlow->count      = ntohs(nfptr->count);
                     ArgusNetFlow->sysUptime  = ntohl(nfptr->sysUptime);
                     ArgusNetFlow->unix_secs  = ntohl(nfptr->unix_secs);
                     ArgusNetFlow->unix_nsecs = ntohl(nfptr->unix_nsecs);
                     ArgusNetFlowRecordHeader = (u_char *)ArgusNetFlow;
                     break;
                  }

                  case CISCO_VERSION_5: {
                     CiscoFlowHeaderV5_t *ArgusNetFlow = (CiscoFlowHeaderV5_t *) input->ArgusReadBuffer;
                     CiscoFlowHeaderV5_t *nfptr = ArgusNetFlow;
 
                     input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV5;
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowEntryV5_t);
                     input->ArgusReadPtr = &input->ArgusReadBuffer[sizeof(CiscoFlowHeaderV5_t)];

                     ArgusNetFlow->version       = ntohs(nfptr->version);
                     ArgusNetFlow->count         = ntohs(nfptr->count);
                     ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
                     ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
                     ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
                     ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
                     ArgusNetFlowRecordHeader = (u_char *)ArgusNetFlow;
                     break;
                  }

                  default: {
#ifdef ARGUSDEBUG
                     ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read header\n", input);
#endif
                  }
               }
               
               input->ArgusReadSocketState = ARGUS_READINGBLOCK;
               input->ArgusReadBlockPtr = input->ArgusReadPtr;
               input->ArgusReadSocketCnt -= size;
               break;
            }

            default: {
#ifdef ARGUSDEBUG
               ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read record complete\n", input);
#endif
               if (ArgusHandleDatum (ArgusParser, input, input->ArgusCiscoNetFlowParse (input, &input->ArgusReadPtr), &ArgusParser->ArgusFilterCode))
                  return(1);

               input->ArgusReadSocketCnt -= size;

               switch (input->ArgusReadCiscoVersion) {
                  case CISCO_VERSION_1:
                     input->ArgusReadPtr += sizeof(CiscoFlowHeaderV1_t);
                     input->ArgusReadSocketCnt -= sizeof(CiscoFlowHeaderV1_t);
                     break;

                  case CISCO_VERSION_5:
                     input->ArgusReadPtr += sizeof(CiscoFlowHeaderV5_t);
                     input->ArgusReadSocketCnt -= sizeof(CiscoFlowHeaderV5_t);
                     break;

                  default: {
                     input->ArgusReadPtr += size;
#ifdef ARGUSDEBUG
                     ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read header\n", input);
#endif
                  }
               }
               break;
            }
         }
      }

      if (input->ArgusReadPtr != input->ArgusReadBuffer) {
         if (input->ArgusReadSocketCnt > 0)
            memmove(input->ArgusReadBuffer, input->ArgusReadPtr, input->ArgusReadSocketCnt);
         input->ArgusReadPtr = input->ArgusReadBuffer;
      }

   } else {
#ifdef ARGUSDEBUG
     if (cnt < 0)
        ArgusDebug (3, "ArgusReadCiscoStreamSocket (0x%x) read returned %d error %s\n", input, cnt, strerror(errno));
     else
        ArgusDebug (6, "ArgusReadCiscoStreamSocket (0x%x) read returned %d\n", input, cnt);
#endif

      retn = 1;

      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadCiscoStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}

int ArgusCiscoDatagramSocketStart = 1;
int ArgusReadCiscoDatagramSocket (struct ArgusParserStruct *, struct ArgusInput *);

int
ArgusReadCiscoDatagramSocket (struct ArgusParserStruct *parser, struct ArgusInput *input)
{
   int retn = 0, cnt = 0, count = 0, i = 0;
   struct sockaddr from;
   socklen_t fromlen = sizeof(from);
   struct sockaddr_in *sin = (struct sockaddr_in *)&from;
   unsigned short *sptr = NULL;
   unsigned char *ptr = NULL;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusReadCiscoDatagramSocket (0x%x) starting\n", input);
#endif

   if ((cnt = recvfrom (input->fd, input->ArgusReadPtr, input->ArgusReadSocketSize, 0L, &from, &fromlen)) > 0) {
      input->ArgusReadSocketCnt = cnt;
      sptr = (unsigned short *) input->ArgusReadPtr;
      ptr = (unsigned char *) input->ArgusReadPtr;

      if (from.sa_family == AF_INET)
         input->addr.s_addr = ntohl(sin->sin_addr.s_addr);
      else
         input->addr.s_addr = 0;


#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusReadCiscoDatagramSocket (0x%x) read %d bytes, capacity %d\n",
                      input, cnt, input->ArgusReadSocketCnt, input->ArgusReadSocketSize);
#endif

      switch (input->ArgusReadCiscoVersion = ntohs(*sptr)) {
         case CISCO_VERSION_1: {
            CiscoFlowHeaderV1_t *ArgusNetFlow = (CiscoFlowHeaderV1_t *) ptr;
            CiscoFlowHeaderV1_t *nfptr = (CiscoFlowHeaderV1_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV1;
            ArgusNetFlow->version    = ntohs(nfptr->version);
            ArgusNetFlow->count      = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime  = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs  = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs = ntohl(nfptr->unix_nsecs);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }
         break;

         case CISCO_VERSION_5: {
            CiscoFlowHeaderV5_t *ArgusNetFlow = (CiscoFlowHeaderV5_t *) ptr;
            CiscoFlowHeaderV5_t *nfptr = (CiscoFlowHeaderV5_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV5;
            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }
         break;

         case CISCO_VERSION_6: {
            CiscoFlowHeaderV6_t *ArgusNetFlow = (CiscoFlowHeaderV6_t *) ptr;
            CiscoFlowHeaderV6_t *nfptr = (CiscoFlowHeaderV6_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV6;
            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }
         break;

         case CISCO_VERSION_8: {
            CiscoFlowHeaderV8_t *ArgusNetFlow = (CiscoFlowHeaderV8_t *) ptr;
            CiscoFlowHeaderV8_t *nfptr = (CiscoFlowHeaderV8_t *) sptr;

            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;

            if ((input->ArgusCiscoNetFlowParse =
                   ArgusLookUpNetFlow(input, ArgusNetFlow->agg_method)) != NULL) {
            }
         }
         break;
      }
               
      for (i = 0; i < count; i++) {
         if (ArgusHandleDatum (ArgusParser, input, input->ArgusCiscoNetFlowParse (input, &ptr), &ArgusParser->ArgusFilterCode))
            return(1);
      }

   } else {
#ifdef ARGUSDEBUG
     ArgusDebug (3, "ArgusReadCiscoDatagramSocket (0x%x) read returned %d error %s\n", input, cnt, strerror(errno));
#endif


      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      } else
         retn = 1;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadCiscoDatagramSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}


int
ArgusReadStreamSocket (struct ArgusParserStruct *parser, struct ArgusInput *input)
{
   int retn = 0, cnt = 0, bytes = 0, done = 0;

   if (!(input))
      return (retn);

   bytes = (input->ArgusBufferLen - input->ArgusReadSocketCnt);
   bytes = (bytes > ARGUS_MAX_BUFFER_READ) ? ARGUS_MAX_BUFFER_READ : bytes;

   if (input->file != NULL) {
      clearerr(input->file);

      if (input->file == stdin) {
         int sretn;
         fd_set readmask;
         struct timeval wait;

         FD_ZERO (&readmask);
         FD_SET (fileno(stdin), &readmask);
         wait.tv_sec  = 1;
         wait.tv_usec = 0;

         if (!((sretn = select (fileno(stdin)+1, &readmask, NULL, NULL, &wait)) > 0)) {
#ifdef ARGUSDEBUG
            ArgusDebug (4, "ArgusReadStreamSocket (0x%x) select returned %d\n", input, sretn);
#endif
            return (sretn);
         } else {
            if ((cnt = fread (input->ArgusReadPtr + input->ArgusReadSocketCnt, 1, bytes, input->file)) == 0) {
               if ((retn = ferror(input->file))) {
                  if ((retn == EAGAIN) || (retn == EINTR))
                     retn = 0;
                  else
                     retn = 1;
               } else {
                  if ((retn = feof(input->file))) {
                     done++;
                     retn = 1;
                  }
               }
            }
         }
      } else {
         if ((cnt = fread (input->ArgusReadPtr + input->ArgusReadSocketCnt, 1, bytes, input->file)) == 0) {
            if ((retn = ferror(input->file))) {
               if ((retn == EAGAIN) || (retn == EINTR))
                  retn = 0;
               else
                  retn = 1;
            } else {
               if ((retn = feof(input->file))) {
                  done++;
                  retn = 1;
               }
            }
         }
      }

   } else {
      if ((cnt = read (input->fd, input->ArgusReadPtr + input->ArgusReadSocketCnt, bytes)) < 0) {
         if ((errno == EAGAIN) || (errno == EINTR))
            retn = 0;
         else
            retn = 1;
      } else {
         if (cnt == 0) {
            retn = 1;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusReadStreamSocket (0x%x) read %d bytes\n", input, cnt);
#endif

   if (cnt > 0) {
      struct ArgusRecord *rec = NULL;
      input->ArgusReadSocketCnt += cnt;
      
      while (!done) {
         unsigned short length = 0;

         switch (input->mode) {
            case ARGUS_V2_DATA_SOURCE: {
               struct ArgusV2Record *recv2 = (struct ArgusV2Record *)input->ArgusReadPtr;
               if (input->ArgusReadSocketCnt >= sizeof(recv2->ahdr)) {
                  if ((length = ntohs(recv2->ahdr.length)) > 0) {
                     if (input->ArgusReadSocketCnt >= length)
                        rec = (struct ArgusRecord *) ArgusConvertRecord (input, (char *)input->ArgusReadPtr);
                  } else {
                     ArgusLog (LOG_ALERT, "ArgusReadStreamSocket (0x%x) record length is zero");
                     retn = 1;
                  }
               }
               break;
            }
            case ARGUS_DATA_SOURCE: {
               struct ArgusRecordHeader *recv3 = (struct ArgusRecordHeader *) input->ArgusReadPtr;
               if (input->ArgusReadSocketCnt >= sizeof(*recv3)) {
                  if ((length = ntohs(recv3->len) * 4) > 0) {
                     if (input->ArgusReadSocketCnt >= length)
                        rec = (struct ArgusRecord *) input->ArgusReadPtr;
                  } else {
                     ArgusLog (LOG_ALERT, "ArgusReadStreamSocket (0x%x) record length is zero");
                     retn = 1;
                  }
               }
               break;
            }
         }

         if (rec && !done && !parser->RaParseDone) {
            if (ArgusHandleDatum (ArgusParser, input, rec, &ArgusParser->ArgusFilterCode) == 1)
               retn = 1;

            input->offset += length;
            input->ArgusReadPtr += length;
            input->ArgusReadSocketCnt -= length;

            if (input->ostop != -1)
               if (input->offset > input->ostop)
                  retn = 1;

            rec = NULL;

         } else
            done = 1;
      }

      if (input->ArgusReadPtr != input->ArgusReadBuffer) {
         if (input->ArgusReadSocketCnt > 0)
            memmove(input->ArgusReadBuffer, input->ArgusReadPtr, input->ArgusReadSocketCnt);
         input->ArgusReadPtr = input->ArgusReadBuffer;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusReadStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}


void
ArgusReadFileStream (struct ArgusParserStruct *parser, struct ArgusInput *input)
{
   int retn = 0, done = 0;
   struct timeval timeoutValue;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusReadFileStream() starting\n");
#endif
      
   timeoutValue.tv_sec = 0;

   while (input && !done) {
      
      switch (input->mode) {
         case ARGUS_DATA_SOURCE:
         case ARGUS_V2_DATA_SOURCE:
            if ((retn = ArgusReadStreamSocket (parser, input)) > 0) {
               ArgusCloseInput(parser, input);
               done++;
            }
            break;

         case ARGUS_CISCO_DATA_SOURCE:
            if ((retn = ArgusReadCiscoStreamSocket (parser, input)) > 0) {
               ArgusCloseInput(parser, input);
               done++;
            }
            break;

      }

      if (timeoutValue.tv_sec == 0) {
         timeoutValue = ArgusParser->ArgusRealTime;

         timeoutValue.tv_sec  += ArgusParser->RaClientTimeout.tv_sec;
         timeoutValue.tv_usec += ArgusParser->RaClientTimeout.tv_usec;

         while (timeoutValue.tv_usec >= 1000000) {
            timeoutValue.tv_sec  += 1;
            timeoutValue.tv_usec -= 1000000;
         }
      }

      if ((ArgusParser->ArgusRealTime.tv_sec  > timeoutValue.tv_sec) ||
         ((ArgusParser->ArgusRealTime.tv_sec == timeoutValue.tv_sec) &&
          (ArgusParser->ArgusRealTime.tv_usec > timeoutValue.tv_usec))) {

         ArgusClientTimeout ();

         if (ArgusParser->Tflag) {
            if ((ArgusParser->Tflag - 1) == 0) {
               ArgusShutDown(0);
            }
            ArgusParser->Tflag--;
         }

         timeoutValue = ArgusParser->ArgusRealTime;
         timeoutValue.tv_sec  += ArgusParser->RaClientTimeout.tv_sec;
         timeoutValue.tv_usec += ArgusParser->RaClientTimeout.tv_usec;

         while (timeoutValue.tv_usec >= 1000000) {
            timeoutValue.tv_sec  += 1;
            timeoutValue.tv_usec -= 1000000;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadFileStream() returning\n");
#endif
}


/*
   The ArgusParser input list is the primary serializer for argus records
   coming in from multiple argus data sources, and is the point where
   global processing occurs, such as general filtering, aggregation,
   holding, sorting, etc.....

   The input list processor basically gets records off the single input
   queue(list), and passes its records off to the client specific
   RaProcessRecord processor.  To support good memory management
   at this point, ArgusProcessInputList will delete any record that
   it gets from the queue.  Clients already are designed to copy
   records that it needs, so this should be cool.


#if defined(ARGUS_THREADS)
long long ArgusInputRecords = 0;

void *
ArgusProcessInputList (void *arg)
{
   struct ArgusParserStruct *parser = (struct ArgusParserStruct *) arg;
   struct timeval timeoutValue = {0, 0};
   void *retn = NULL;

   timeoutValue = parser->ArgusRealTime;
   timeoutValue.tv_sec  += ArgusParser->RaClientTimeout.tv_sec;
   timeoutValue.tv_usec += ArgusParser->RaClientTimeout.tv_usec;

   while (timeoutValue.tv_usec >= 1000000) {
      timeoutValue.tv_sec  += 1;
      timeoutValue.tv_usec -= 1000000;
   }

   while (!parser->RaParseDone) {
      struct ArgusListStruct *olist = parser->ArgusOutputList;
      struct ArgusListStruct *ilist = parser->ArgusInputList;
      struct ArgusRecordStruct *argus = NULL;

      if (ArgusListEmpty(olist)) {
         struct timespec tsbuf, *ts = &tsbuf;

         if (parser->RaDonePending) {
            parser->RaParseDone++;

         } else {
            struct timeval tvp;

            gettimeofday (&tvp, 0L);
            ts->tv_sec = tvp.tv_sec;
            ts->tv_nsec = tvp.tv_usec * 1000;
            ts->tv_nsec += 20000000;
            if (ts->tv_nsec > 1000000000) {
               ts->tv_sec++;
               ts->tv_nsec -= 1000000000;
            }
            pthread_mutex_lock(&olist->lock);
            pthread_cond_timedwait(&olist->cond, &olist->lock, ts);
            pthread_mutex_unlock(&olist->lock);
         }
      }

      gettimeofday (&parser->ArgusRealTime, NULL);
      ArgusAdjustGlobalTime(parser, &parser->ArgusRealTime);

      if ((parser->ArgusRealTime.tv_sec  > timeoutValue.tv_sec) ||
         ((parser->ArgusRealTime.tv_sec == timeoutValue.tv_sec) &&
          (parser->ArgusRealTime.tv_usec > timeoutValue.tv_usec))) {

         ArgusClientTimeout ();

         if (parser->Tflag) {
            if ((parser->Tflag - 1) == 0) {
               ArgusShutDown(0);
            }
            parser->Tflag--;
         }

         timeoutValue.tv_sec  += parser->RaClientTimeout.tv_sec;
         timeoutValue.tv_usec += parser->RaClientTimeout.tv_usec;

         if (timeoutValue.tv_usec >= 1000000) {
            timeoutValue.tv_sec  += 1;
            timeoutValue.tv_usec -= 1000000;
         }
         if (timeoutValue.tv_usec < 0) {
            timeoutValue.tv_usec = 0;
         }
      }

      if ((olist != NULL) && (ilist != NULL)) {
         ArgusLoadList(olist, ilist);

         while ((argus = (struct ArgusRecordStruct *) ArgusPopFrontList(ilist, ARGUS_LOCK)) != NULL) {
            ArgusInputRecords++;
            RaProcessRecord(parser, argus);
            ArgusDeleteRecordStruct(ArgusParser, argus);
         }
      }
   }

   pthread_exit (retn);
}
#endif
*/


void *
ArgusConnectRemotes (void *arg)
{
#if defined(ARGUS_THREADS)
   struct ArgusQueueStruct *queue = arg;
   struct ArgusInput *addr = NULL;
   int status, retn, done = 0;
   pthread_attr_t attr;

   if ((status = pthread_attr_init(&attr)) != 0)
      ArgusLog (LOG_ERR, "pthreads init error");

   while (!done && !ArgusParser->RaParseDone) {
      if ((addr = (struct ArgusInput *) ArgusPopQueue(queue, ARGUS_LOCK)) != NULL) {
         if ((retn = pthread_create(&addr->tid, &attr, ArgusConnectRemote, addr)) != 0) {
            switch (retn) {
               case EAGAIN:
                  ArgusLog (LOG_ERR, "main: pthread_create ArgusProcessInputList: EAGAIN \n");
                  break;
               case EINVAL:
                  ArgusLog (LOG_ERR, "main: pthread_create ArgusProcessInputList, EINVAL\n");
                  break;
            }
         }
      }

      sleep(1);
   }
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusConnectRemotes() done!");
#endif

#if defined(ARGUS_THREADS)
   pthread_exit (NULL);
#else
   return (NULL);
#endif
}

/*
   This routine basically runs until it connects to the remote
   site and then it exists.  If it is run in a threaded environment,
   it will run forever until it connects, then it will exit.
*/

void *
ArgusConnectRemote (void *arg)
{
   struct ArgusInput *addr = (struct ArgusInput *)arg;
   int done = 0;

#if defined(ARGUS_THREADS)
   sigset_t blocked_signals;
   sigfillset(&blocked_signals);
   pthread_sigmask(SIG_BLOCK, &blocked_signals, NULL);
#endif

   if (ArgusParser->ArgusConnectTime == 0)
      ArgusParser->ArgusConnectTime = 10;

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusConnectRemote(0x%x) starting", arg);
#endif

   while (!done && !ArgusParser->RaParseDone) {
#if defined(ARGUS_THREADS)
      pthread_mutex_lock(&addr->lock);
#endif

      addr->status &= ~ARGUS_CLOSED;

      if (addr->fd < 0) {
         if ((addr->fd = ArgusGetServerSocket (addr, ArgusParser->ArgusConnectTime)) >= 0) {
            if ((ArgusReadConnection (ArgusParser, addr, ARGUS_SOCKET)) >= 0) {
               int flags;
               if ((flags = fcntl(addr->fd, F_GETFL, 0L)) < 0)
                  ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

               if (fcntl (addr->fd, F_SETFL, flags | O_NONBLOCK) < 0)
                  ArgusLog (LOG_ERR, "ArgusConnectRemote: fcntl error %s", strerror(errno));

#ifdef ARGUSDEBUG
               ArgusDebug (2, "ArgusConnectRemote(0x%x) connected to %s", arg, addr->hostname);
#endif
               if (addr->qhdr.queue != NULL) {
                  if (addr->qhdr.queue != ArgusParser->ArgusActiveHosts) {
                     ArgusRemoveFromQueue(addr->qhdr.queue, &addr->qhdr, ARGUS_LOCK);
                     ArgusAddToQueue(ArgusParser->ArgusActiveHosts, &addr->qhdr, ARGUS_LOCK);
                  }
               } else
                  ArgusAddToQueue(ArgusParser->ArgusActiveHosts, &addr->qhdr, ARGUS_LOCK);

               ArgusParser->ArgusTotalMarRecords++;
               ArgusParser->ArgusTotalRecords++;
               done++;
            } else {
#ifdef ARGUSDEBUG
               ArgusDebug (2, "ArgusConnectRemote() ArgusReadConnection(%s) failed", addr->hostname);
#endif
            }

         } else {
#ifdef ARGUSDEBUG
            ArgusDebug (2, "ArgusConnectRemote() ArgusGetServerSocket failed errno %d: %s", errno, strerror(errno));
#endif
            switch (errno) {
               case EHOSTDOWN:
               case EHOSTUNREACH:
               case ENETUNREACH: {
                  break;
               }
            }
         }
      }
#if defined(ARGUS_THREADS)
      pthread_mutex_unlock(&addr->lock);
#endif
      if (!done && ArgusParser->ArgusReliableConnection) {
         struct timespec tsbuf = {5, 0}, *ts = &tsbuf;
         struct timespec rtsbuf, *rts = &rtsbuf;
         int retn;

         while ((retn = nanosleep(ts, rts)) == EINTR)
            *rts = *ts;

         if (retn < 0) 
            ArgusLog (LOG_ERR, "ArgusConnectRemote %s: nanosleep error %s", addr->hostname, strerror(errno));
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusConnectRemote() done!");
#endif

#if defined(ARGUS_THREADS)
   pthread_exit (NULL);
#else
   return (NULL);
#endif
}


void
ArgusReadStream (struct ArgusParserStruct *parser, struct ArgusQueueStruct *queue)
{
   struct ArgusInput *input = NULL;
   struct timeval wait, timeoutValue;
   struct timespec tsbuf = {0, 3000000};
   struct timespec *ts = &tsbuf;
   int retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusReadStream(0x%x) starting", parser);
#endif
      
   timeoutValue.tv_sec = 0;

   while (!(parser->RaParseDone)) {
      int width = -1, i;
      fd_set readmask;

      FD_ZERO (&readmask);

      if ((input = (struct ArgusInput *) queue->start) != NULL) {
         for (i = 0; i < queue->count; i++) {
            if (input->fd >= 0) {
               FD_SET (input->fd, &readmask);
               width = (width < input->fd) ? input->fd : width;
            }
            input = (void *)input->qhdr.nxt;
         }

      } else {
         if (!(parser->ArgusRemotes > 0))
            parser->RaParseDone++;
      }

      if (width >= 0) {
         width++;
         wait.tv_sec = 0;
         wait.tv_usec = 150000;

         if ((retn = select (width, &readmask, NULL, NULL, &wait)) >= 0) {
            gettimeofday (&parser->ArgusRealTime, NULL);
            ArgusAdjustGlobalTime(ArgusParser, &ArgusParser->ArgusRealTime);

            for (input = (struct ArgusInput *) queue->start, i = 0; i < queue->count; i++) {
               if ((input->fd >= 0) && FD_ISSET (input->fd, &readmask)) {
                  if (input->ArgusStartTime.tv_sec == 0)
                     input->ArgusStartTime = parser->ArgusRealTime;

                  input->ArgusLastTime = parser->ArgusRealTime;

                  switch (input->mode) {
                     case ARGUS_DATA_SOURCE:
                     case ARGUS_V2_DATA_SOURCE:
#ifdef ARGUS_SASL
                        if (input->sasl_conn) {
                           if (ArgusReadSaslStreamSocket (parser, input))
                              ArgusCloseInput(parser, input);
                        } else
#endif /* ARGUS_SASL */
                           if (ArgusReadStreamSocket (parser, input))
                              ArgusCloseInput(parser, input);
                        break;

                     case ARGUS_CISCO_DATA_SOURCE:
                        if (parser->Sflag) {
                           if (ArgusReadCiscoDatagramSocket (parser, input))
                              ArgusCloseInput(parser, input);
                        } else {
                           if (ArgusReadCiscoStreamSocket (parser, input))
                              ArgusCloseInput(parser, input);
                        }
                        break;
                  }

               } else {
                  if (input->fd >= 0) {
                     gettimeofday (&ArgusParser->ArgusRealTime, NULL);
                     ArgusAdjustGlobalTime(parser, NULL);

                     if (input->hostname && input->ArgusMarInterval) {
                        if (input->ArgusLastTime.tv_sec) {
                           if ((parser->ArgusRealTime.tv_sec - input->ArgusLastTime.tv_sec) > (input->ArgusMarInterval * 2)) {
                              ArgusLog (LOG_WARNING, "ArgusReadStream %s: idle stream: closing", input->hostname);
                              ArgusCloseInput(parser, input);
                           }
                        }
                     }
                  }
               }
               input = (void *)input->qhdr.nxt;
            }
         }

      } else {
        gettimeofday (&parser->ArgusRealTime, NULL);
        ArgusAdjustGlobalTime(ArgusParser, &ArgusParser->ArgusRealTime);
      }

      if (timeoutValue.tv_sec == 0) {
         gettimeofday (&ArgusParser->ArgusRealTime, NULL);
         timeoutValue = parser->ArgusRealTime;
         timeoutValue.tv_sec  += parser->RaClientTimeout.tv_sec;
         timeoutValue.tv_usec += parser->RaClientTimeout.tv_usec;
         while (timeoutValue.tv_usec >= 1000000) {
            timeoutValue.tv_sec  += 1;
            timeoutValue.tv_usec -= 1000000;
         }
      }

      if ((parser->ArgusRealTime.tv_sec  > timeoutValue.tv_sec) ||
         ((parser->ArgusRealTime.tv_sec == timeoutValue.tv_sec) &&
          (parser->ArgusRealTime.tv_usec > timeoutValue.tv_usec))) {

         ArgusClientTimeout ();

         if (parser->Tflag) {
            if ((parser->Tflag - 1) == 0) {
               ArgusShutDown(0);
            }
            parser->Tflag--;
         }

         timeoutValue = parser->ArgusRealTime;
         timeoutValue.tv_sec  += parser->RaClientTimeout.tv_sec;
         timeoutValue.tv_usec += parser->RaClientTimeout.tv_usec;

         if (timeoutValue.tv_usec >= 1000000) {
            timeoutValue.tv_sec  += 1;
            timeoutValue.tv_usec -= 1000000;
         }
         if (timeoutValue.tv_usec < 0) {
            timeoutValue.tv_usec = 0;
         }
      }

      if ((!parser->RaParseDone && !retn) || (width < 0))
         if (parser->ArgusReliableConnection)
            nanosleep(ts, NULL);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusReadStream(0x%x) returning", parser);
#endif
}

int ArgusTotalRecords = 0;


#include <netdb.h>

extern void ArgusLog (int, char *, ...);

#define ARGUS_DEFAULTCISCOPORT		9995

char *ArgusRecordType = NULL;

extern int ArgusInitializeAuthentication(void);

#include <netinet/in.h>
#include <arpa/inet.h>

int
ArgusGetServerSocket (struct ArgusInput *input, int timeout)
{
#if defined(HAVE_GETADDRINFO)
   struct addrinfo *hp = input->host;
#else
   struct hostent *hp = input->host;
#endif
   struct servent *sp;
   int s, type = 0, retn = -1;
   u_short portnum = 0;

#if defined(HAVE_GETADDRINFO)
   char hbuf[NI_MAXHOST], sbuf[NI_MAXSERV];
   int optval = 1;
#endif

   switch (input->mode) {
      case ARGUS_DATA_SOURCE:
      case ARGUS_V2_DATA_SOURCE: {
         ArgusRecordType = "Argus";
         type = SOCK_STREAM;
         if (!input->portnum) {
            if (!ArgusParser->ArgusPortNum) {
               if ((sp = getservbyname ("monitor", "tcp")) != NULL)
                  portnum = sp->s_port;
               else
                  portnum = htons(ARGUS_DEFAULTPORT);
            } else
               portnum = htons(ArgusParser->ArgusPortNum);

            input->portnum = ntohs(portnum);

         } else
            portnum = htons(input->portnum);

         break;
      }

      case ARGUS_CISCO_DATA_SOURCE: {
         struct ArgusRecord argus;

         ArgusRecordType = "Netflow";
         type = SOCK_DGRAM;
         if (!input->portnum) {
            if (!ArgusParser->ArgusPortNum) {
               if ((sp = getservbyname ("monitor", "udp")) != NULL)
                  portnum = sp->s_port;
               else
                  portnum = htons(ARGUS_DEFAULTCISCOPORT);
            } else
               portnum = htons(ArgusParser->ArgusPortNum);
         } else
            portnum = htons(input->portnum);

         bzero ((char *)&argus, sizeof(argus));
         argus.hdr.type          = ARGUS_MAR | ARGUS_NETFLOW | ARGUS_VERSION;
         argus.hdr.cause         = ARGUS_START;
         argus.hdr.len           = sizeof (argus) / 4;
         argus.argus_mar.argusid = ARGUS_COOKIE;

         if (input->addr.s_addr != 0)
            argus.argus_mar.thisid  = htonl(input->addr.s_addr);

         argus.argus_mar.startime.tv_sec = ArgusParser->ArgusGlobalTime.tv_sec;
         argus.argus_mar.now.tv_sec      = ArgusParser->ArgusGlobalTime.tv_sec;
         argus.argus_mar.major_version   = VERSION_MAJOR;
         argus.argus_mar.minor_version   = VERSION_MINOR;
         argus.argus_mar.record_len      = -1;

         input->major_version = argus.argus_mar.major_version;
         input->minor_version = argus.argus_mar.minor_version;

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(&argus);
#endif
         bcopy ((char *) &argus, (char *)&input->ArgusInitCon, sizeof (argus));
         break;
      }
      
      default:
         ArgusLog (LOG_ERR, "ArgusGetServerSocket(0x%x) unknown type", input);
   }

   switch (input->mode) {
      case ARGUS_DATA_SOURCE:
      case ARGUS_V2_DATA_SOURCE: {

         if (hp != NULL) {
#if defined(HAVE_GETADDRINFO)
            do {
               if (getnameinfo(hp->ai_addr, hp->ai_addrlen, hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), NI_NUMERICHOST | NI_NUMERICSERV))
                  ArgusLog(LOG_ERR, "could not get numeric hostname");
               
               if (hp->ai_canonname) {
                  if (input->hostname)
                     free(input->hostname);
                  input->hostname = strdup(hp->ai_canonname);
               } else {
                  if (input->hostname)
                     free(input->hostname);
                  input->hostname = strdup(hbuf);
               }

               if ((s = socket (hp->ai_family, hp->ai_socktype, hp->ai_protocol)) >= 0) {
                  if (hp->ai_socktype == SOCK_DGRAM) {
#ifdef ARGUSDEBUG
                     ArgusLog (1, "Binding %s:%s Expecting %s records", input->hostname, sbuf, ArgusRecordType); 
#endif
                     if ((retn = bind (s, hp->ai_addr, hp->ai_addrlen)) < 0) {
                        ArgusLog(LOG_WARNING, "connect to %s:%s failed '%s'", input->hostname, sbuf, strerror(errno));
                        hp = hp->ai_next;
                     } else {
                        retn = s;
                        input->fd = s;
                     }

                  } else {
                     if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (char *)&optval, sizeof(int)) < 0) {
#ifdef ARGUSDEBUG
                        ArgusDebug (1, "setsockopt(%d, SOL_SOCKET, SO_KEEPALIVE, 0x%x, %d) failed:", s, optval, sizeof(int));
#endif
                     }
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "Trying %s port %s Expecting %s records\n", input->hostname, sbuf, ArgusRecordType); 
#endif
                     if ((retn = ArgusConnect (s, hp->ai_addr, hp->ai_addrlen, timeout)) < 0) {
                        ArgusLog(LOG_WARNING, "connect to %s:%s failed '%s'", input->hostname, sbuf, strerror(errno));
                        hp = hp->ai_next;
                     } else {
                        retn = s;
                        input->fd = s;
                     }
                  }

                  if (retn < 0)
                     close(s);
#ifdef ARGUSDEBUG
                  else {
                     if (hp->ai_socktype == SOCK_DGRAM)
                        ArgusDebug (1, "receiving\n");
                     else
                        ArgusDebug (1, "connected\n");
                  }
#endif
               } else
                  ArgusLog (LOG_ERR, "ArgusGetServerSocket: socket() failed. errno %d: %s", errno, strerror(errno));

            } while (hp && (retn < 0));
#endif
         } else {
#if !defined(HAVE_GETADDRINFO)
            struct sockaddr_in server;

            bzero ((char *) &server, sizeof (server));

            if ((s = socket (AF_INET, type, 0)) >= 0) {
               if (type == SOCK_DGRAM) {
                  if (input->addr.s_addr != 0)
                     server.sin_addr.s_addr = htonl(input->addr.s_addr);
                  else
                     server.sin_addr.s_addr = INADDR_ANY;

                  server.sin_family = AF_INET;
                  server.sin_port = portnum;

#ifdef ARGUSDEBUG
                  ArgusLog (1, "Binding %s:%d Expecting %s records", 
                       ArgusGetName(ArgusParser, (unsigned char *)&input->addr.s_addr), ntohs(portnum), ArgusRecordType); 
#endif
                  if ((bind (s, (struct sockaddr *)&server, sizeof(server))) < 0)
                     ArgusLog (LOG_ERR, "bind (%d, %s:%hu, %d) failed '%s'", s, inet_ntoa(server.sin_addr),
                                                    server.sin_port, sizeof(server), strerror(errno));
                  retn = s;
                  input->fd = s;

               } else {
                  int optval = 1;
                  if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (char *)&optval, sizeof(int)) < 0) {
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "setsockopt(%d, SOL_SOCKET, SO_KEEPALIVE, 0x%x, %d) failed:", s, optval, sizeof(int));
#endif
                  }

                  input->addr.s_addr = htonl(input->addr.s_addr);
                  if ((hp = gethostbyaddr ((char *)&input->addr.s_addr, sizeof (input->addr.s_addr), AF_INET)) != NULL) {
                     input->hostname = strdup(hp->h_name);
                     bcopy ((char *) hp->h_addr, (char *)&server.sin_addr, hp->h_length);
                     server.sin_family = hp->h_addrtype;
                     server.sin_port = portnum;
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "Trying %s port %d Expecting %s records\n", (hp->h_name) ?
                                 (hp->h_name) : intoa (input->addr.s_addr), ntohs(portnum), ArgusRecordType); 
#endif
                 } else {
                     server.sin_addr.s_addr = input->addr.s_addr;
                     server.sin_family = AF_INET;
                     server.sin_port = portnum;
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "Trying %s port %d Expecting %s records\n", 
                                   intoa (input->addr.s_addr), ntohs(portnum), ArgusRecordType); 
#endif
                  }

                  input->addr.s_addr = htonl(input->addr.s_addr);

                  if ((retn = ArgusConnect (s, (struct sockaddr *)&server, sizeof(server), timeout)) < 0) {
                     ArgusLog(LOG_WARNING, "connect to %s:%hu failed '%s'", inet_ntoa(server.sin_addr), 
                         ntohs(server.sin_port), strerror(errno));
                     close(s);

                  } else {
                     retn = s;
                     input->fd = s;

#ifdef ARGUSDEBUG
                     if (type == SOCK_DGRAM)
                        ArgusDebug (1, "receiving\n");
                     else
                        ArgusDebug (1, "connected\n");
#endif
                  }
               }

            } else {
               ArgusLog (LOG_ERR, "ArgusGetServerSocket: socket() failed. errno %d: %s", errno, strerror(errno));
            }
#endif
         }
         break;
      }

      case ARGUS_CISCO_DATA_SOURCE: {
#if defined(HAVE_GETADDRINFO)
         if (hp != NULL)
            s = socket (hp->ai_family, hp->ai_socktype, hp->ai_protocol);
         else
#endif
            s = socket (AF_INET, SOCK_DGRAM, 0);

         if (s >= 0) {
            if (hp != NULL) {
#if defined(HAVE_GETADDRINFO)
               if (getnameinfo(hp->ai_addr, hp->ai_addrlen, hbuf, sizeof(hbuf), sbuf, sizeof(sbuf), NI_NUMERICHOST | NI_NUMERICSERV))
                  ArgusLog(LOG_ERR, "could not get numeric hostname");

               if (hp->ai_canonname) {
                  if (input->hostname)
                     free(input->hostname);
                  input->hostname = strdup(hp->ai_canonname);
               } else {
                  if (input->hostname)
                     free(input->hostname);
                  input->hostname = strdup(hbuf);
               }

               if (hp->ai_socktype == SOCK_DGRAM) {
#ifdef ARGUSDEBUG
                  ArgusLog (1, "Binding %s:%s Expecting %s records", input->hostname, sbuf, ArgusRecordType);
#endif
                  if ((retn = bind (s, hp->ai_addr, hp->ai_addrlen)) < 0) {
                     ArgusLog(LOG_WARNING, "connect to %s:%s failed '%s'", input->hostname, sbuf, strerror(errno));
                     hp = hp->ai_next;
                  } else {
                     retn = s;
                     input->fd = s;
                  }

               } else {
                  if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (char *)&optval, sizeof(int)) < 0) {
#ifdef ARGUSDEBUG
                     ArgusDebug (1, "setsockopt(%d, SOL_SOCKET, SO_KEEPALIVE, 0x%x, %d) failed:", s, optval, sizeof(int));
#endif
                  }
#ifdef ARGUSDEBUG
                  ArgusDebug (1, "Trying %s port %s Expecting %s records\n", input->hostname, sbuf, ArgusRecordType);
#endif
                  if ((retn = ArgusConnect (s, hp->ai_addr, hp->ai_addrlen, timeout)) < 0) {
                     ArgusLog(LOG_WARNING, "connect to %s:%s failed '%s'", input->hostname, sbuf, strerror(errno));
                     hp = hp->ai_next;
                  } else {
                     retn = s;
                     input->fd = s;
                  }
               }

               if (retn < 0)
                  close(s);
#ifdef ARGUSDEBUG
               else {
                  if (hp->ai_socktype == SOCK_DGRAM)
                     ArgusDebug (1, "receiving\n");
                  else
                     ArgusDebug (1, "connected\n");
               }
#endif
#endif
            } else {
               struct sockaddr_in server;
               bzero(&server, sizeof(server));

               if (input->addr.s_addr != 0)
                  server.sin_addr.s_addr = htonl(input->addr.s_addr);
               else
                  server.sin_addr.s_addr = INADDR_ANY;

               server.sin_family = AF_INET;
               server.sin_port   = htons(input->portnum);
#ifdef ARGUSDEBUG
               if (server.sin_addr.s_addr == INADDR_ANY)
                  ArgusLog (1, "Binding %s:%d Expecting %s records", "AF_ANY", input->portnum, ArgusRecordType);
               else
                  ArgusLog (1, "Binding %s:%d Expecting %s records", (unsigned char *)&input->addr.s_addr, input->portnum, ArgusRecordType);
#endif
               if ((retn = bind (s, (struct sockaddr *)&server, sizeof(server))) < 0) {
                  ArgusLog(LOG_WARNING, "connect to %s:%d failed '%s'", inet_ntoa(server.sin_addr),
                                              ntohs(server.sin_port), sizeof(server), strerror(errno)); 
                  close(s);

               } else {
                  retn = s;
                  input->fd = s;
               }
            }
         } else
            ArgusLog (LOG_ERR, "ArgusGetServerSocket: socket() failed. errno %d: %s", errno, strerror(errno));
         break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusGetServerSocket (0x%x) returning %d", input, retn);
#endif

   return (retn);
}


int
ArgusConnect(int s, const struct sockaddr *name, socklen_t namelen, int timeout)
{
   int retn = 0, flags, width = (s + 1);
   struct timeval tbuf;
   fd_set rset, wset;

   if ((flags = fcntl(s, F_GETFL, 0)) < 0)
      ArgusLog (LOG_ERR, "ArgusConnect: fcntl error %s", strerror(errno));

   if ((fcntl(s, F_SETFL, flags | O_NONBLOCK)) < 0)
      ArgusLog (LOG_ERR, "ArgusConnect: fcntl error %s", strerror(errno));

   if ((retn = connect(s, name, namelen)) < 0)
      if (errno != EINPROGRESS)
         return(retn);

   if (retn) {
      FD_ZERO(&rset); FD_SET(s, &rset);
      FD_ZERO(&wset); FD_SET(s, &wset);
      tbuf.tv_sec  = timeout;
      tbuf.tv_usec = 0;

      if ((retn = select (width, &rset, &wset, NULL, &tbuf)) == 0) {
         errno = ETIMEDOUT;
         return(-1);

      } else {
         if (FD_ISSET(s, &rset) || FD_ISSET(s, &wset)) {
            int error;
            socklen_t len = sizeof(error);
            if (getsockopt(s, SOL_SOCKET, SO_ERROR, &error, &len) < 0) {
               return(-1);
            }
            if (error) {
               errno = error;
               return(-1);
            }
         }
      }
   }

   if ((fcntl(s, F_SETFL, flags)) < 0)
      ArgusLog (LOG_ERR, "ArgusConnect: fcntl error %s", strerror(errno));

   return(0);
}

struct ArgusRecordStruct ArgusGenerateRecordBuffer;

struct ArgusRecordStruct *
ArgusGenerateRecordStruct (struct ArgusParserStruct *parser, struct ArgusInput *input, struct ArgusRecord *argus)
{
   struct ArgusRecordStruct *retn = NULL;
   unsigned int ArgusReverse = 0, status = 0;

   if (argus == NULL) {
      if (input != NULL) {
         retn = &input->ArgusGenerateRecordStructBuf;
      } else {
         retn = &ArgusGenerateRecordBuffer;
      }

      bzero ((char *)retn, sizeof(*retn));

      retn->input = input;
      retn->hdr.type = ARGUS_FAR | ARGUS_VERSION;
      retn->hdr.cause = ARGUS_STATUS;
      retn->hdr.len = 8;

      bcopy((char *)&retn->hdr, (char *)&retn->canon.hdr, sizeof(retn->hdr));

      retn->dsrs[ARGUS_TIME_INDEX] = &retn->canon.time.hdr;
      retn->canon.time.hdr.type = ARGUS_TIME_DSR;
      retn->canon.time.hdr.subtype =  ARGUS_TIME_ABSOLUTE_RANGE;
      retn->canon.time.hdr.argus_dsrvl8.qual = ARGUS_TYPE_UTC_MICROSECONDS;
      retn->canon.time.hdr.argus_dsrvl8.len = 5;
      retn->dsrindex |= (0x01 << ARGUS_TIME_INDEX);

      retn->dsrs[ARGUS_METRIC_INDEX] = &retn->canon.metric.hdr;
      retn->canon.metric.hdr.type = ARGUS_METER_DSR;
      retn->canon.metric.hdr.argus_dsrvl8.qual = ARGUS_SRCDST_BYTE;
      retn->canon.metric.hdr.argus_dsrvl8.len = 2;
      retn->dsrindex |= (0x01 << ARGUS_METRIC_INDEX);
      retn->sload = 0.0;
      retn->dload = 0.0;
      retn->srate = 0.0;
      retn->drate = 0.0;
      retn->dur   = 0.0;

   } else {
      struct ArgusRecordHeader *hdr = &argus->hdr;
      struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) (hdr + 1);

      retn = &input->ArgusGenerateRecordStructBuf;
      bzero ((char *)retn, sizeof(*retn));

      retn->input = input;

      retn->sload = 0.0;
      retn->dload = 0.0;
      retn->srate = 0.0;
      retn->drate = 0.0;
      retn->dur   = 0.0;

      retn->bins = NULL;
      retn->htblhdr = NULL;
      retn->nsq = NULL;

      if (hdr->type & ARGUS_MAR) {
         if (argus->hdr.len > 1)
            bcopy ((char *)argus, (char *)&retn->canon, (argus->hdr.len * 4));
         bcopy((char *)hdr, (char *)&retn->hdr, sizeof(*hdr));

      } else {
         int dsrlen = hdr->len * 4;
         char *argusend = (char *)argus + dsrlen;
         double seconds = 0.0;
         int reverse = 0;

         bcopy((char *)hdr, (char *)&retn->canon.hdr, sizeof(*hdr));
         bcopy((char *)hdr, (char *)&retn->hdr, sizeof(*hdr));

         while (retn && ((char *) dsr < argusend)) {
            unsigned char type = dsr->type, subtype = dsr->subtype;
            int len;

            if ((len = (((type & ARGUS_IMMEDIATE_DATA) ? 1 :
                        ((subtype & ARGUS_LEN_16BITS)  ? dsr->argus_dsrvl16.len :
                                                         dsr->argus_dsrvl8.len))) * 4) > 0) {
               switch (type & 0x7F) {
                  case ARGUS_FLOW_DSR: {
                     struct ArgusFlow *flow = (struct ArgusFlow *) dsr;

                     switch (subtype & 0x3F) {
                        case ARGUS_FLOW_LAYER_3_MATRIX:
                        case ARGUS_FLOW_CLASSIC5TUPLE: {
                           status = flow->hdr.argus_dsrvl8.qual & ARGUS_DIRECTION;
                           reverse = flow->hdr.subtype & ARGUS_REVERSE;
                           switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                              case ARGUS_TYPE_IPV4: {
                                 if (flow->hdr.argus_dsrvl8.qual & ARGUS_FRAGMENT) {
                                    if ((flow->hdr.subtype & 0x3F) == ARGUS_FLOW_CLASSIC5TUPLE) {
                                       if (!(flow->hdr.subtype & ARGUS_REVERSE)) {
                                          retn->canon.flow.frag_flow.ip_src = flow->frag_flow.ip_src;
                                          retn->canon.flow.frag_flow.ip_dst = flow->frag_flow.ip_dst;
                                       } else {
                                          retn->canon.flow.frag_flow.ip_src = flow->frag_flow.ip_dst;
                                          retn->canon.flow.frag_flow.ip_dst = flow->frag_flow.ip_src;
                                       }

                                       retn->canon.flow.frag_flow.ip_p   = flow->frag_flow.ip_p;
                                       retn->canon.flow.frag_flow.ip_id  = flow->frag_flow.ip_id;
                                       retn->canon.flow.frag_flow.pad[0] = 0;
                                       retn->canon.flow.frag_flow.pad[1] = 0;
                                    }
                                 } else {
                                    if (!(flow->hdr.subtype & ARGUS_REVERSE)) {
                                       retn->canon.flow.ip_flow.ip_src = flow->ip_flow.ip_src;
                                       retn->canon.flow.ip_flow.ip_dst = flow->ip_flow.ip_dst;
                                    } else {
                                       retn->canon.flow.ip_flow.ip_src = flow->ip_flow.ip_dst;
                                       retn->canon.flow.ip_flow.ip_dst = flow->ip_flow.ip_src;
                                    }
                                    if ((flow->hdr.subtype & 0x3F) == ARGUS_FLOW_CLASSIC5TUPLE) {
                                       retn->canon.flow.ip_flow.ip_p   = flow->ip_flow.ip_p;
                                       switch (flow->ip_flow.ip_p) {
                                          case IPPROTO_UDP:
                                             if (flow->ip_flow.tp_p == ARGUS_V2_RTCP_FLOWTAG) {
                                                retn->dsrs[ARGUS_NETWORK_INDEX] = &retn->canon.net.hdr;
                                                retn->canon.net.hdr.type = ARGUS_NETWORK_DSR;
                                                retn->canon.net.hdr.subtype = ARGUS_RTCP_FLOW;
                                                retn->canon.net.hdr.argus_dsrvl8.qual = 0;
                                                retn->canon.net.hdr.argus_dsrvl8.len = 1;
                                                retn->dsrindex |= (0x01 << ARGUS_NETWORK_INDEX);
                                             }

                                          case IPPROTO_TCP:
                                             if (!(flow->hdr.subtype & ARGUS_REVERSE)) {
                                                retn->canon.flow.ip_flow.sport = flow->ip_flow.sport;
                                                retn->canon.flow.ip_flow.dport = flow->ip_flow.dport;
                                             } else {
                                                retn->canon.flow.ip_flow.sport = flow->ip_flow.dport;
                                                retn->canon.flow.ip_flow.dport = flow->ip_flow.sport;
                                             }
                                             break;

                                          case IPPROTO_ICMP: 
                                             retn->canon.flow.icmp_flow.type  = flow->icmp_flow.type;
                                             retn->canon.flow.icmp_flow.code  = flow->icmp_flow.code;
                                             retn->canon.flow.icmp_flow.id    = flow->icmp_flow.id;
                                             retn->canon.flow.icmp_flow.ip_id = flow->icmp_flow.ip_id;
                                             break;

                                          case IPPROTO_ESP:
                                             retn->canon.flow.esp_flow.spi = flow->esp_flow.spi;
                                             break;
                                       }
                                    }
                                 }
                              }
                              break; 

                              case ARGUS_TYPE_IPV6: {
                                 int i;
                                 for (i = 0; i < 4; i++) {
                                    if (!(flow->hdr.subtype & ARGUS_REVERSE)) {
                                       retn->canon.flow.ipv6_flow.ip_src[i] = flow->ipv6_flow.ip_src[i];
                                       retn->canon.flow.ipv6_flow.ip_dst[i] = flow->ipv6_flow.ip_dst[i];
                                    } else {
                                       retn->canon.flow.ipv6_flow.ip_dst[i] = flow->ipv6_flow.ip_src[i];
                                       retn->canon.flow.ipv6_flow.ip_src[i] = flow->ipv6_flow.ip_dst[i];
                                    }
                                 }
                                 if ((flow->hdr.subtype & 0x3F) == ARGUS_FLOW_CLASSIC5TUPLE) {
                                    retn->canon.flow.ipv6_flow.flow = flow->ipv6_flow.flow;
                                    switch (retn->canon.flow.ipv6_flow.ip_p = flow->ipv6_flow.ip_p) {
                                       case IPPROTO_TCP:
                                       case IPPROTO_UDP:
                                          if (!(flow->hdr.subtype & ARGUS_REVERSE)) {
                                             retn->canon.flow.ipv6_flow.sport = flow->ipv6_flow.sport;
                                             retn->canon.flow.ipv6_flow.dport = flow->ipv6_flow.dport;
                                          } else {
                                             retn->canon.flow.ipv6_flow.sport = flow->ipv6_flow.dport;
                                             retn->canon.flow.ipv6_flow.dport = flow->ipv6_flow.sport;
                                          }
                                          break;

                                       case IPPROTO_ICMPV6:
                                          retn->canon.flow.icmp6_flow.type = flow->icmp6_flow.type;
                                          retn->canon.flow.icmp6_flow.code = flow->icmp6_flow.code;
                                          retn->canon.flow.icmp6_flow.id   = ntohs(flow->icmp6_flow.id);
                                          break;

                                       case IPPROTO_ESP:
                                          retn->canon.flow.esp_flow.spi = flow->esp_flow.spi;
                                          break;

                                    }
                                 }
                              }
                              break; 

                              case ARGUS_TYPE_RARP: {
                                 struct ArgusLegacyRarpFlow *trarp = &flow->lrarp_flow;
                                 struct ArgusRarpFlow        *rarp = &retn->canon.flow.rarp_flow;

                                 flow->hdr.subtype           = ARGUS_FLOW_ARP;
                                 flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_RARP;
                                 flow->hdr.argus_dsrvl8.len  = 1 + sizeof(*rarp)/4;
                                 rarp->hrd     = 1;
                                 rarp->pro     = 2048;
                                 rarp->hln     = 6;
                                 rarp->pln     = 4;
                                 rarp->op      = 3;

                                 rarp->arp_tpa = trarp->arp_tpa;
                                 bcopy((char *)&trarp->srceaddr, (char *)&rarp->shaddr, 6);
                                 bcopy((char *)&trarp->tareaddr, (char *)&rarp->dhaddr, 6);
                                 break;
                              }
                              case ARGUS_TYPE_ARP: {
                                 struct ArgusLegacyArpFlow *tarp = &flow->larp_flow;
                                 struct ArgusArpFlow        *arp = &retn->canon.flow.arp_flow;

                                 flow->hdr.subtype           = ARGUS_FLOW_ARP;
                                 flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_ARP;
                                 flow->hdr.argus_dsrvl8.len  = 1 + sizeof(*arp)/4;
                                 arp->hrd     = 1;
                                 arp->pro     = 2048;
                                 arp->hln     = 6;
                                 arp->pln     = 4;
                                 arp->op      = 1;

                                 arp->arp_spa = (tarp->arp_spa);
                                 arp->arp_tpa = (tarp->arp_tpa);
                                 bcopy((char *)&tarp->etheraddr, (char *)&arp->haddr, 6);
                                 break;
                              }
                              case ARGUS_TYPE_ETHER: 
                                 bcopy (&flow->mac_flow, &retn->canon.flow.mac_flow, sizeof(retn->canon.flow.mac_flow));
                                 break;

                              case ARGUS_TYPE_MPLS:
                              case ARGUS_TYPE_VLAN:
                                 bcopy ((char *)flow, &retn->canon.flow, len);
                                 break; 

                              default:
#ifdef ARGUSDEBUG
                                 ArgusDebug (1, "ArgusGenerateRecordStruct: unknown flow type: %d", flow->hdr.argus_dsrvl8.qual);
#endif
                                 break; 
                           }
                           break; 
                        }

                        case ARGUS_FLOW_ARP: {
                           retn->canon.flow.hdr = flow->hdr;

                           switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                              case ARGUS_TYPE_RARP: {
                                 struct ArgusRarpFlow *trarp = &flow->flow_un.rarp;
                                 struct ArgusRarpFlow *rarp  = &retn->canon.flow.flow_un.rarp;
 
                                 retn->canon.flow.hdr = flow->hdr;
                                 rarp->hrd     = trarp->hrd;
                                 rarp->pro     = trarp->pro;
                                 rarp->hln     = trarp->hln;
                                 rarp->pln     = trarp->pln;
                                 rarp->op      = trarp->op;
                                 rarp->arp_tpa = trarp->arp_tpa;
                                 bcopy (&((char *)&trarp->shaddr)[0],         &rarp->shaddr, rarp->hln);
                                 bcopy (&((char *)&trarp->shaddr)[rarp->hln], &rarp->dhaddr, rarp->hln);
                                 break;
                              }

                              case ARGUS_TYPE_ARP: {
                                 struct ArgusArpFlow *tarp = &flow->flow_un.arp;
                                 struct ArgusArpFlow *arp  = &retn->canon.flow.flow_un.arp;

                                 flow->hdr.argus_dsrvl8.len = sizeof(*arp)/4 + 1;
                                 retn->canon.flow.hdr = flow->hdr;

                                 if ((arp->hrd = tarp->hrd) == 0)
                                    arp->hrd     = 1;
                                 if ((arp->pro = tarp->pro) == 0)
                                    arp->pro     = 2048;
                                 if ((arp->hln = tarp->hln) == 0)
                                    arp->hln     = 6;
                                 if ((arp->pln = tarp->pln) == 0)
                                    arp->pln     = 4;

                                 arp->op      = tarp->op;
                                 arp->arp_spa = tarp->arp_spa;
                                 arp->arp_tpa = tarp->arp_tpa;
                                 bcopy ((char *)&arp->haddr, &arp->haddr, arp->hln);
                                 break;
                              }

                              default: {
                                 struct ArgusInterimArpFlow *tarp = &flow->flow_un.iarp;
                                 struct ArgusArpFlow         *arp = &retn->canon.flow.flow_un.arp;
 
                                 flow->hdr.subtype = ARGUS_FLOW_ARP;
                                 flow->hdr.argus_dsrvl8.qual = ARGUS_TYPE_ARP;
                                 flow->hdr.argus_dsrvl8.len  = sizeof(*arp)/4 + 1;
                                 arp->hrd     = 1;
                                 arp->pro     = tarp->pro;
                                 arp->hln     = tarp->hln;
                                 arp->pln     = tarp->pln;
                                 arp->op      = 0;
                                 arp->arp_spa = tarp->arp_spa;
                                 arp->arp_tpa = tarp->arp_tpa;
                                 bcopy ((char *)&arp->haddr, &arp->haddr, arp->hln);
                              }
                           }
                           break; 
                        }

                        default:
                           break; 
                     }

                     if (subtype & ARGUS_REVERSE)
                        flow->hdr.subtype &= ~ARGUS_REVERSE;

                     bcopy((char *)&flow->hdr, (char *)&retn->canon.flow.hdr, sizeof(flow->hdr));

                     retn->dsrs[ARGUS_FLOW_INDEX] = (struct ArgusDSRHeader*) &retn->canon.flow;
                     retn->dsrindex |= (0x01 << ARGUS_FLOW_INDEX);
                     break;
                  }
  
                  case ARGUS_TRANSPORT_DSR: {
                     struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;
          
                     if (len >= 12)
                        trans->hdr.subtype |= (ARGUS_SRCID | ARGUS_SEQ);

                     bzero ((char *)&retn->canon.trans, sizeof(struct ArgusTransportStruct));
                     bcopy((char *)&trans->hdr, (char *)&retn->canon.trans.hdr, 4);

                     if (trans->hdr.subtype & ARGUS_SRCID) {
                        switch (trans->hdr.argus_dsrvl8.qual & 0x3F) {
                           case ARGUS_TYPE_INT:  {
                              retn->canon.trans.srcid.a_un.value = trans->srcid.a_un.value;
                              break;
                           }
                           case ARGUS_TYPE_IPV4: {
                              retn->canon.trans.srcid.a_un.ipv4 = trans->srcid.a_un.ipv4;
                              break;
                           }
                           case ARGUS_TYPE_IPV6: {
                              break;
                           }
                           case ARGUS_TYPE_ETHER: {
                              break;
                           }
                           case ARGUS_TYPE_STRING: {
                              break;
                           }
                        }
                     }

                     if (trans->hdr.subtype & ARGUS_SEQ)
                        retn->canon.trans.seqnum = trans->seqnum;

                     retn->dsrs[ARGUS_TRANSPORT_INDEX] = (struct ArgusDSRHeader*) &retn->canon.trans;
                     retn->dsrindex |= (0x01 << ARGUS_TRANSPORT_INDEX);
                     break;
                  }

                  case ARGUS_ENCAPS_DSR: {
                     struct ArgusEncapsStruct *encaps  = (struct ArgusEncapsStruct *) dsr;
                     struct ArgusEncapsStruct *cncaps = &retn->canon.encaps;

                     bcopy((char *) encaps, (char *) cncaps, sizeof(*encaps));

                     retn->dsrs[ARGUS_ENCAPS_INDEX] = &cncaps->hdr;
                     retn->dsrindex |= (0x01 << ARGUS_ENCAPS_INDEX);
                     break;
                  }

                  case ARGUS_TIME_DSR: {
                     struct ArgusTimeObject *time  = (struct ArgusTimeObject *) dsr;
                     struct ArgusTimeObject *ctime = (struct ArgusTimeObject *) &retn->canon.time;
                     int i, num = (time->hdr.argus_dsrvl8.len - 1)/2;

                     if (subtype & 0x78) {
                        unsigned int *tptr = (unsigned int *) (dsr + 1);
                        unsigned int *tval = NULL;
                        int tlen = 2, sindex = -1, tind = 0;

                        if (!(subtype & (ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START)))
                           ArgusLog (LOG_ERR, "ArgusGenerateRecord: time format incorrect:%d");

                        ctime->hdr = time->hdr;

                        for (i = 0; i < 4; i++) {
                           if (subtype & (ARGUS_TIME_SRC_START << i)) {
                              tind++;
                           }
                        }
                        if (tind != num) {
                           switch (num) {
                              case 1:
                                 subtype = ARGUS_TIME_SRC_START | (dsr->subtype & 0x07);
                                 break;
                              case 2:
                                 subtype = ARGUS_TIME_SRC_START |
                                           ARGUS_TIME_SRC_END   | (dsr->subtype & 0x07);
                                 break;
                              case 3:
                                 subtype = ARGUS_TIME_SRC_START |
                                           ARGUS_TIME_SRC_END   |
                                           ARGUS_TIME_DST_START | (dsr->subtype & 0x07);
                                 break;
                              case 4:
                                 subtype = ARGUS_TIME_SRC_START |
                                           ARGUS_TIME_SRC_END   |
                                           ARGUS_TIME_DST_START |
                                           ARGUS_TIME_DST_END   | (dsr->subtype & 0x07);
                                 break;
                           }
                        }

                        sindex = (subtype & ARGUS_TIME_SRC_START) ? 0 : 2;

                        for (i = 0; i < 4; i++) {
                           if (subtype & (ARGUS_TIME_SRC_START << i)) {
                              switch (ARGUS_TIME_SRC_START << i) {
                                 case ARGUS_TIME_SRC_START: {
                                    tval = (unsigned int *)&ctime->src.start;
                                    break;
                                 }
                                 case ARGUS_TIME_SRC_END: {
                                    tval = (unsigned int *)&ctime->src.end;
                                    break;
                                 }
                                 case ARGUS_TIME_DST_START: {
                                    tval = (unsigned int *)&ctime->dst.start;
                                    break;
                                 }
                                 case ARGUS_TIME_DST_END: {
                                    tval = (unsigned int *)&ctime->dst.end;
                                    break;
                                 }
                              }
                              if (tval && tlen) {
                                 switch (subtype & 0x07) {
                                    case ARGUS_TIME_ABSOLUTE_RANGE:
                                    case ARGUS_TIME_ABSOLUTE_TIMESTAMP: {
                                       int y;
                                       for (y = 0; y < tlen; y++)
                                          *tval++ = (*(unsigned int *)tptr++);
                                       break;
                                    }

                                    case ARGUS_TIME_RELATIVE_TIMESTAMP:
                                    case ARGUS_TIME_RELATIVE_RANGE:
                                    case ARGUS_TIME_ABSOLUTE_RELATIVE_RANGE: {
                                       if (i == sindex) {
                                          unsigned int rtime = *(unsigned int *)tptr++;
                                          ctime->src.end.tv_sec  = rtime / 1000000;
                                          ctime->src.end.tv_usec = rtime % 1000000;
                                          if (ctime->src.end.tv_usec > 1000000) {
                                             ctime->src.end.tv_sec++;
                                             ctime->src.end.tv_usec -= 1000000;
                                          }
                                       }
                                       break;
                                    }
                                 }
                              }
                           }
                        }

                        if (ctime->src.start.tv_sec == 0)
                           ctime->src.start = ctime->dst.start;

                        if (ctime->src.end.tv_sec == 0) 
                           ctime->src.end = ctime->src.start;
                        if (ctime->dst.end.tv_sec == 0) 
                           ctime->dst.end = ctime->dst.start;

                     } else {
                        bcopy((char *)dsr, (char *)&retn->canon.time, len);
                        retn->dsrs[ARGUS_TIME_INDEX] = (struct ArgusDSRHeader*) &retn->canon.time;
                        retn->dsrindex |= (0x01 << ARGUS_TIME_INDEX);

                        switch (subtype & 0x3F) {
                           case ARGUS_TIME_ABSOLUTE_TIMESTAMP:
                              ctime->src.end = ctime->src.start;
                              break;
                           case ARGUS_TIME_ABSOLUTE_RANGE:
                              break;
                           case ARGUS_TIME_ABSOLUTE_RELATIVE_RANGE: /* end.tv_sec is delta uSec */
                              ctime->src.end.tv_sec  = ctime->src.start.tv_sec  + (time->src.end.tv_sec / 1000000);
                              ctime->src.end.tv_usec = ctime->src.start.tv_usec + (time->src.end.tv_sec % 1000000);
                              if (ctime->src.end.tv_usec > 1000000) {
                                 ctime->src.end.tv_sec++;
                                 ctime->src.end.tv_usec -= 1000000;
                              }
                              break;
                           case ARGUS_TIME_RELATIVE_TIMESTAMP:
                              break;
                           case ARGUS_TIME_RELATIVE_RANGE:
                              break;
                        }
                        if (len < sizeof(struct ArgusTimeObject))
                           bzero (&((char *)&retn->canon.time)[len], sizeof(struct ArgusTimeObject) - len);

                        switch (subtype & 0x3F) {
                           case ARGUS_TIME_ABSOLUTE_TIMESTAMP:
                           case ARGUS_TIME_ABSOLUTE_RANGE:
                           case ARGUS_TIME_ABSOLUTE_RELATIVE_RANGE:
                              if (!(ctime->src.start.tv_sec))
                                 ctime->src.start = ctime->src.end;
                              if (!(ctime->src.end.tv_sec))
                                 ctime->src.end = ctime->src.start;
                              break;
                        }

                        if (ctime->src.start.tv_usec > 1000000)
                           ctime->src.start.tv_usec = 999999;
                        if (ctime->src.end.tv_usec > 1000000)
                           ctime->src.end.tv_usec = 999999;
                     }
                     
                     retn->dsrs[ARGUS_TIME_INDEX] = (struct ArgusDSRHeader*) &retn->canon.time;
                     retn->dsrindex |= (0x01 << ARGUS_TIME_INDEX);
                     break;
                  }

                  case ARGUS_METER_DSR: {
                     int offset = 1;
                     if (subtype & ARGUS_METER_PKTS_BYTES) {
                        retn->canon.metric.src.appbytes = 0;
                        retn->canon.metric.dst.appbytes = 0;

                        switch (dsr->argus_dsrvl8.qual & 0x0F) {
                           case ARGUS_SRCDST_BYTE:
                              retn->canon.metric.src.pkts  = ((unsigned char *)(dsr + 1))[0];
                              retn->canon.metric.src.bytes = ((unsigned char *)(dsr + 1))[1];
                              retn->canon.metric.dst.pkts  = ((unsigned char *)(dsr + 1))[2];
                              retn->canon.metric.dst.bytes = ((unsigned char *)(dsr + 1))[3];
                              offset++;
                              break;
                           case ARGUS_SRCDST_SHORT:
                              retn->canon.metric.src.pkts  = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes = (((unsigned short *)(dsr + 1))[1]);
                              retn->canon.metric.dst.pkts  = (((unsigned short *)(dsr + 1))[2]);
                              retn->canon.metric.dst.bytes = (((unsigned short *)(dsr + 1))[3]);
                              offset += 2;
                              break;
                           case ARGUS_SRCDST_INT:
                              retn->canon.metric.src.pkts  = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes = (((unsigned int *)(dsr + 1))[1]);
                              retn->canon.metric.dst.pkts  = (((unsigned int *)(dsr + 1))[2]);
                              retn->canon.metric.dst.bytes = (((unsigned int *)(dsr + 1))[3]);
                              offset += 4;
                              break;
                           case ARGUS_SRCDST_LONGLONG:
#if defined(LBL_ALIGN)
                              bcopy ((char *)(dsr + 1), (char *)&retn->canon.metric.src.pkts, 16);
#else
                              retn->canon.metric.src.pkts  = (((unsigned long long *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes = (((unsigned long long *)(dsr + 1))[1]);
#endif
#if defined(LBL_ALIGN)
                              bcopy ((char *)(dsr + 5), (char *)&retn->canon.metric.dst.pkts, 16);
#else
                              retn->canon.metric.dst.pkts  = (((unsigned long long *)(dsr + 1))[2]);
                              retn->canon.metric.dst.bytes = (((unsigned long long *)(dsr + 1))[3]);
#endif
                              offset += 8;
                              break;
                           case ARGUS_SRC_SHORT:
                              retn->canon.metric.src.pkts  = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes = (((unsigned short *)(dsr + 1))[1]);
                              retn->canon.metric.dst.pkts  = 0;
                              retn->canon.metric.dst.bytes = 0;
                              offset++;
                              break;
                           case ARGUS_SRC_INT:
                              retn->canon.metric.src.pkts  = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes = (((unsigned int *)(dsr + 1))[1]);
                              retn->canon.metric.dst.pkts  = 0;
                              retn->canon.metric.dst.bytes = 0;
                              offset += 2;
                              break;
                           case ARGUS_SRC_LONGLONG:
                              bcopy((char *)(dsr + 1), (char *)&retn->canon.metric.src, 16);
                              retn->canon.metric.dst.pkts  = 0;
                              retn->canon.metric.dst.bytes = 0;
                              offset += 4;
                              break;
                           case ARGUS_DST_SHORT:
                              retn->canon.metric.src.pkts  = 0;
                              retn->canon.metric.src.bytes = 0;
                              retn->canon.metric.dst.pkts  = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.dst.bytes = (((unsigned short *)(dsr + 1))[1]);
                              offset++;
                              break;
                           case ARGUS_DST_INT:
                              retn->canon.metric.src.pkts  = 0;
                              retn->canon.metric.src.bytes = 0;
                              retn->canon.metric.dst.pkts  = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.dst.bytes = (((unsigned int *)(dsr + 1))[1]);
                              offset += 2;
                              break;
                           case ARGUS_DST_LONGLONG:
                              retn->canon.metric.src.pkts  = 0;
                              retn->canon.metric.src.bytes = 0;
                              bcopy((char *)(dsr + 1), (char *)&retn->canon.metric.dst, 16);
                              offset += 4;
                              break;
                        }

                     } else
                     if (subtype & ARGUS_METER_PKTS_BYTES_APP) {
                        switch (dsr->argus_dsrvl8.qual & 0x0F) {
                           case ARGUS_SRCDST_BYTE:
                              retn->canon.metric.src.pkts     = ((unsigned char *)(dsr + 1))[0];
                              retn->canon.metric.src.bytes    = ((unsigned char *)(dsr + 1))[1];
                              retn->canon.metric.src.appbytes = ((unsigned char *)(dsr + 1))[2];
                              retn->canon.metric.dst.pkts     = ((unsigned char *)(dsr + 1))[3];
                              retn->canon.metric.dst.bytes    = ((unsigned char *)(dsr + 1))[4];
                              retn->canon.metric.dst.appbytes = ((unsigned char *)(dsr + 1))[5];
                              offset++;
                              break;
                           case ARGUS_SRCDST_SHORT:
                              retn->canon.metric.src.pkts     = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((unsigned short *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((unsigned short *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = (((unsigned short *)(dsr + 1))[3]);
                              retn->canon.metric.dst.bytes    = (((unsigned short *)(dsr + 1))[4]);
                              retn->canon.metric.dst.appbytes = (((unsigned short *)(dsr + 1))[5]);
                              offset += 2;
                              break;
                           case ARGUS_SRCDST_INT:
                              retn->canon.metric.src.pkts     = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((unsigned int *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((unsigned int *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = (((unsigned int *)(dsr + 1))[3]);
                              retn->canon.metric.dst.bytes    = (((unsigned int *)(dsr + 1))[4]);
                              retn->canon.metric.dst.appbytes = (((unsigned int *)(dsr + 1))[5]);
                              offset += 4;
                              break;
                           case ARGUS_SRCDST_LONGLONG:
#if defined(LBL_ALIGN)
                              bcopy ((char *)(dsr + 1), (char *)&retn->canon.metric.src.pkts, 48);
#else
                              retn->canon.metric.src.pkts     = (((long long *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((long long *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((long long *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = (((long long *)(dsr + 1))[3]);
                              retn->canon.metric.dst.bytes    = (((long long *)(dsr + 1))[4]);
                              retn->canon.metric.dst.appbytes = (((long long *)(dsr + 1))[5]);
#endif
                              offset += 8;
                              break;
                           case ARGUS_SRC_BYTE:
                              retn->canon.metric.src.pkts     = (((unsigned char *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((unsigned char *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((unsigned char *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = 0;
                              retn->canon.metric.dst.bytes    = 0;
                              retn->canon.metric.dst.appbytes = 0;
                              offset++;
                           case ARGUS_SRC_SHORT:
                              retn->canon.metric.src.pkts     = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((unsigned short *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((unsigned short *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = 0;
                              retn->canon.metric.dst.bytes    = 0;
                              retn->canon.metric.dst.appbytes = 0;
                              offset++;
                              break;
                           case ARGUS_SRC_INT:
                              retn->canon.metric.src.pkts     = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.src.bytes    = (((unsigned int *)(dsr + 1))[1]);
                              retn->canon.metric.src.appbytes = (((unsigned int *)(dsr + 1))[2]);
                              retn->canon.metric.dst.pkts     = 0;
                              retn->canon.metric.dst.bytes    = 0;
                              retn->canon.metric.dst.appbytes = 0;
                              offset += 2;
                              break;
                           case ARGUS_SRC_LONGLONG:
                              bcopy((char *)(dsr + 1), (char *)&retn->canon.metric.src, 24);
                              retn->canon.metric.dst.pkts     = 0;
                              retn->canon.metric.dst.bytes    = 0;
                              retn->canon.metric.dst.appbytes = 0;
                              offset += 4;
                              break;
                           case ARGUS_DST_BYTE:
                              retn->canon.metric.src.pkts     = 0;
                              retn->canon.metric.src.bytes    = 0;
                              retn->canon.metric.src.appbytes = 0;
                              retn->canon.metric.dst.pkts     = (((unsigned char *)(dsr + 1))[0]);
                              retn->canon.metric.dst.bytes    = (((unsigned char *)(dsr + 1))[1]);
                              retn->canon.metric.dst.appbytes = (((unsigned char *)(dsr + 1))[2]);
                              offset++;
                              break;
                           case ARGUS_DST_SHORT:
                              retn->canon.metric.src.pkts     = 0;
                              retn->canon.metric.src.bytes    = 0;
                              retn->canon.metric.src.appbytes = 0;
                              retn->canon.metric.dst.pkts     = (((unsigned short *)(dsr + 1))[0]);
                              retn->canon.metric.dst.bytes    = (((unsigned short *)(dsr + 1))[1]);
                              retn->canon.metric.dst.appbytes = (((unsigned short *)(dsr + 1))[2]);
                              offset++;
                              break;
                           case ARGUS_DST_INT:
                              retn->canon.metric.src.pkts     = 0;
                              retn->canon.metric.src.bytes    = 0;
                              retn->canon.metric.src.appbytes = 0;
                              retn->canon.metric.dst.pkts     = (((unsigned int *)(dsr + 1))[0]);
                              retn->canon.metric.dst.bytes    = (((unsigned int *)(dsr + 1))[1]);
                              retn->canon.metric.dst.appbytes = (((unsigned int *)(dsr + 1))[2]);
                              offset += 2;
                              break;
                           case ARGUS_DST_LONGLONG:
                              retn->canon.metric.src.pkts     = 0;
                              retn->canon.metric.src.bytes    = 0;
                              retn->canon.metric.src.appbytes = 0;
                              bcopy((char *)(dsr + 1), (char *)&retn->canon.metric.dst, 24);
                              offset += 4;
                              break;
                        }
                     }

                     bcopy((char *)dsr, (char *)&retn->canon.metric.hdr, sizeof(*dsr));
                     retn->dsrs[ARGUS_METRIC_INDEX] = (struct ArgusDSRHeader*) &retn->canon.metric;
                     retn->dsrindex |= (0x01 << ARGUS_METRIC_INDEX);
                     break;
                  }

                  case ARGUS_PSIZE_DSR: {
                     switch (dsr->argus_dsrvl8.qual & 0x0F) {
                        case ARGUS_SRCDST_SHORT:
                           retn->canon.psize.src.psizemin = (((unsigned short *)(dsr + 1))[0]);
                           retn->canon.psize.src.psizemax = (((unsigned short *)(dsr + 1))[1]);
                           retn->canon.psize.dst.psizemin = (((unsigned short *)(dsr + 1))[2]);
                           retn->canon.psize.dst.psizemax = (((unsigned short *)(dsr + 1))[3]);
                           break;

                        case ARGUS_SRC_SHORT:
                           retn->canon.psize.src.psizemin = (((unsigned short *)(dsr + 1))[0]);
                           retn->canon.psize.src.psizemax = (((unsigned short *)(dsr + 1))[1]);
                           retn->canon.psize.dst.psizemin = 0;
                           retn->canon.psize.dst.psizemax = 0;
                           break;

                        case ARGUS_DST_SHORT:
                           retn->canon.psize.src.psizemin = 0;
                           retn->canon.psize.src.psizemax = 0;
                           retn->canon.psize.dst.psizemin = (((unsigned short *)(dsr + 1))[0]);
                           retn->canon.psize.dst.psizemax = (((unsigned short *)(dsr + 1))[1]);
                           break;
                     }

                     bcopy((char *)dsr, (char *)&retn->canon.psize.hdr, sizeof(*dsr));
                     retn->dsrs[ARGUS_PSIZE_INDEX] = (struct ArgusDSRHeader*) &retn->canon.psize;
                     retn->dsrindex |= (0x01 << ARGUS_PSIZE_INDEX);
                     break;
                  }

                  case ARGUS_NETWORK_DSR: {
                     struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *) dsr;
                     retn->dsrs[ARGUS_NETWORK_INDEX] = (struct ArgusDSRHeader*) &retn->canon.net;
                     retn->dsrindex |= (0x01 << ARGUS_NETWORK_INDEX);

                     switch (subtype) {
                        case ARGUS_NETWORK_SUBTYPE_FRAG: {
                           bcopy((char *)net, (char *)&retn->canon.net, len);
                           break;
                        }
/*
struct ArgusTCPInitStatus {
   unsigned int status, seqbase;
   unsigned int options;
   unsigned short win;
   unsigned char flags, winshift;
};

struct ArgusTCPStatus {
   unsigned int status;
   unsigned char src, dst, pad[2];
};

struct ArgusTCPObjectMetrics {
   struct ArgusTime lasttime;
   unsigned int status, seqbase, seq, ack, winnum;
   unsigned int bytes, retrans, ackbytes;
   unsigned short state, win, winbytes;
   unsigned char flags, winshift;
};

struct ArgusTCPObject {
   unsigned int status, state, options;
   unsigned int synAckuSecs, ackDatauSecs;
   struct ArgusTCPObjectMetrics src, dst;
};

*/
                        case ARGUS_TCP_INIT: {
                           struct ArgusTCPInitStatus *tcpinit = (void *) &net->net_union.tcpinit;
                           struct ArgusTCPObject *tcp = (void *) &retn->canon.net.net_union.tcp;
                           bcopy((char *)&net->hdr, (char *)&retn->canon.net.hdr, sizeof(net->hdr));
                           memset(tcp, 0, sizeof(*tcp));
                           tcp->status = tcpinit->status;
                           tcp->options = tcpinit->options;
                           tcp->src.win = tcpinit->win;
                           tcp->src.flags = tcpinit->flags;
                           tcp->src.winshift = tcpinit->winshift;
                           break;
                        }
                        case ARGUS_TCP_STATUS: {
                           struct ArgusTCPStatus *tcpstatus = (void *) &net->net_union.tcpstatus;
                           struct ArgusTCPObject *tcp = (void *) &retn->canon.net.net_union.tcp;
                           bcopy((char *)&net->hdr, (char *)&retn->canon.net.hdr, sizeof(net->hdr));

                           memset(tcp, 0, sizeof(*tcp));
                           tcp->status = tcpstatus->status;
                           tcp->src.flags = tcpstatus->src;
                           tcp->dst.flags = tcpstatus->dst;

                           if (!retn->canon.metric.src.pkts) {
                              tcp->src.flags = 0;
                           }
                           if (!retn->canon.metric.dst.pkts) {
                              tcp->dst.flags = 0;
                           }
                           break;
                        }
                        case ARGUS_TCP_PERF: {
                           struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) &retn->canon.net.net_union.tcp;
                           bcopy((char *)net, (char *)&retn->canon.net, len);

                           if (!retn->canon.metric.src.pkts) {
                              tcp->src.win = 0;
                              tcp->src.flags = 0;
                           }
                           if (!retn->canon.metric.dst.pkts) {
                              tcp->dst.win = 0;
                              tcp->dst.flags = 0;
                           }
                           break;
                        }
                     }
                     break;
                  }

                  case ARGUS_MAC_DSR: {
                     struct ArgusMacStruct *mac = (struct ArgusMacStruct *) dsr;
           
                     bcopy((char *)mac, (char *)&retn->canon.mac, len);
                     retn->dsrs[ARGUS_MAC_INDEX] = (struct ArgusDSRHeader*) &retn->canon.mac;
                     retn->dsrindex |= (0x01 << ARGUS_MAC_INDEX);
                     break;
                  }

                  case ARGUS_VLAN_DSR: {
                     struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) dsr;
           
                     bcopy((char *)vlan, (char *)&retn->canon.vlan, len);
                     retn->dsrs[ARGUS_VLAN_INDEX] = (struct ArgusDSRHeader*) &retn->canon.vlan;
                     retn->dsrindex |= (0x01 << ARGUS_VLAN_INDEX);
                     break;
                  }

                  case ARGUS_MPLS_DSR: {
                     struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) dsr;
                     unsigned int *label = (unsigned int *)(dsr + 1);

                     bzero((char *)&retn->canon.mpls, sizeof(*mpls));
                     bcopy((char *)&mpls->hdr, (char *)&retn->canon.mpls.hdr, 4);
                     retn->canon.mpls.slabel = *label++;
                     retn->canon.mpls.dlabel = *label++;
                     retn->dsrs[ARGUS_MPLS_INDEX] = (struct ArgusDSRHeader*) &retn->canon.mpls;
                     retn->dsrindex |= (0x01 << ARGUS_MPLS_INDEX);
                     break;
                  }

                  case ARGUS_ICMP_DSR: {
                     struct ArgusIcmpStruct *icmp = (struct ArgusIcmpStruct *) dsr;
                     int icmpLen = sizeof(*icmp);
          
                     bcopy((char *)icmp, (char *)&retn->canon.icmp, (len > icmpLen) ? icmpLen : len);
                     retn->dsrs[ARGUS_ICMP_INDEX] = (struct ArgusDSRHeader*) &retn->canon.icmp;
                     retn->dsrindex |= (0x01 << ARGUS_ICMP_INDEX);
                     break;
                  }

                  case ARGUS_COR_DSR: {
                     struct ArgusCorrelateStruct *cor = (struct ArgusCorrelateStruct *) dsr;
                     int corLen = sizeof(*cor);

                     bcopy((char *)cor, (char *)&retn->canon.cor, (len > corLen) ? corLen : len);
                     retn->dsrs[ARGUS_COR_INDEX] = (struct ArgusDSRHeader*) &retn->canon.cor;
                     retn->dsrindex |= (0x01 << ARGUS_COR_INDEX);
                     break;
                  }

                  case ARGUS_AGR_DSR: {
                     struct ArgusAgrStruct *agr = (struct ArgusAgrStruct *) dsr;
                     int agrLen = sizeof(*agr);
          
                     if ((!(len > sizeof(*agr))) && (agr->count > 1)) {
                        bcopy((char *)agr, (char *)&retn->canon.agr, (len > agrLen) ? agrLen : len);
                        retn->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader*) &retn->canon.agr;
                        retn->dsrindex |= (0x01 << ARGUS_AGR_INDEX);
                     }

                     break;
                  }

                  case ARGUS_JITTER_DSR: {
                     struct ArgusStatObject *tjit = (struct ArgusStatObject *) (dsr + 1);
                     int tcount = 0;

                     retn->canon.jitter.hdr = *dsr;
                     if (dsr->argus_dsrvl8.qual & ARGUS_SRC_ACTIVE_JITTER)
                        tcount++;
                     if (dsr->argus_dsrvl8.qual & ARGUS_SRC_IDLE_JITTER)
                        tcount++;
                     if (dsr->argus_dsrvl8.qual & ARGUS_DST_ACTIVE_JITTER)
                        tcount++;
                     if (dsr->argus_dsrvl8.qual & ARGUS_DST_IDLE_JITTER)
                        tcount++;

                     if (((tcount * sizeof(*tjit)) + 4) == len) {
                        if (dsr->argus_dsrvl8.qual & ARGUS_SRC_ACTIVE_JITTER) {
                           bcopy((char *)tjit, (char *)&retn->canon.jitter.act.src, sizeof(struct ArgusStatObject));
                           tjit++;
                        } else
                           bzero((char *)&retn->canon.jitter.act.src, sizeof(struct ArgusStatObject));

                        if (dsr->argus_dsrvl8.qual & ARGUS_SRC_IDLE_JITTER) {
                           bcopy((char *)tjit, (char *)&retn->canon.jitter.idle.src, sizeof(struct ArgusStatObject));
                           tjit++;
                        } else
                           bzero((char *)&retn->canon.jitter.idle.src, sizeof(struct ArgusStatObject));

                        if (dsr->argus_dsrvl8.qual & ARGUS_DST_ACTIVE_JITTER) {
                           bcopy((char *)tjit, (char *)&retn->canon.jitter.act.dst, sizeof(struct ArgusStatObject));
                           tjit++;
                        } else
                           bzero((char *)&retn->canon.jitter.act.dst, sizeof(struct ArgusStatObject));

                        if (dsr->argus_dsrvl8.qual & ARGUS_DST_IDLE_JITTER) {
                           bcopy((char *)tjit, (char *)&retn->canon.jitter.idle.dst, sizeof(struct ArgusStatObject));
                           tjit++;
                        } else
                           bzero((char *)&retn->canon.jitter.idle.dst, sizeof(struct ArgusStatObject));

                        retn->dsrs[ARGUS_JITTER_INDEX] = (struct ArgusDSRHeader*) &retn->canon.jitter;
                        retn->dsrindex |= (0x01 << ARGUS_JITTER_INDEX);
                     }
                     break;
                  }

                  case ARGUS_IPATTR_DSR: {
                     struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
                     unsigned int *ptr = (unsigned int *)(dsr + 1);

                     bzero((char *)&retn->canon.attr, sizeof(*attr));
                     bcopy((char *)&attr->hdr, (char *)&retn->canon.attr.hdr, 4);

                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC)
                        *(unsigned int *)&retn->canon.attr.src = *ptr++;

                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS)
                        retn->canon.attr.src.options = *ptr++;

                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST)
                        *(unsigned int *)&retn->canon.attr.dst = *ptr++;

                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST_OPTIONS)
                        retn->canon.attr.dst.options = *ptr++;

                     retn->dsrs[ARGUS_IPATTR_INDEX] = (struct ArgusDSRHeader*) &retn->canon.attr;
                     retn->dsrindex |= (0x01 << ARGUS_IPATTR_INDEX);
                     break;
                  }

                  case ARGUS_COCODE_DSR: {
                     struct ArgusCountryCodeStruct *cocode = (struct ArgusCountryCodeStruct *) dsr;
                     bcopy((char *)cocode, (char *)&retn->canon.cocode, sizeof(*cocode));
                     retn->dsrs[ARGUS_COCODE_INDEX] = (struct ArgusDSRHeader*) &retn->canon.cocode;
                     retn->dsrindex |= (0x01 << ARGUS_COCODE_INDEX);
                     break;
                  }

                  case ARGUS_DATA_DSR: {
                     struct ArgusDataStruct *user = (struct ArgusDataStruct *) dsr;

                     if (subtype & ARGUS_LEN_16BITS) {
                        if (subtype & ARGUS_SRC_DATA) {
                           retn->dsrs[ARGUS_SRCUSERDATA_INDEX] = (struct ArgusDSRHeader *)input->ArgusSrcUserData;
                           bcopy (user, retn->dsrs[ARGUS_SRCUSERDATA_INDEX], len);
                           retn->dsrindex |= (0x01 << ARGUS_SRCUSERDATA_INDEX);
                           ((struct ArgusDataStruct *)input->ArgusSrcUserData)->size = len - 8;
                        } else {
                           retn->dsrs[ARGUS_DSTUSERDATA_INDEX] = (struct ArgusDSRHeader *)input->ArgusDstUserData;
                           bcopy (user, retn->dsrs[ARGUS_DSTUSERDATA_INDEX], len);
                           retn->dsrindex |= (0x01 << ARGUS_DSTUSERDATA_INDEX);
                           ((struct ArgusDataStruct *)input->ArgusDstUserData)->size = len - 8;
                        }

                     } else {
                        if (user->hdr.argus_dsrvl8.qual & ARGUS_SRC_DATA) {
                           retn->dsrs[ARGUS_SRCUSERDATA_INDEX] = (struct ArgusDSRHeader *)input->ArgusSrcUserData;
                           retn->dsrindex |= (0x01 << ARGUS_SRCUSERDATA_INDEX);
                           user->hdr.subtype  = ARGUS_LEN_16BITS;
                           user->hdr.subtype |= ARGUS_SRC_DATA;
                           user->hdr.argus_dsrvl16.len = len / 4;
                           bcopy (user, retn->dsrs[ARGUS_SRCUSERDATA_INDEX], len);
                        } else {
                           retn->dsrs[ARGUS_DSTUSERDATA_INDEX] = (struct ArgusDSRHeader *)input->ArgusDstUserData;
                           retn->dsrindex |= (0x01 << ARGUS_DSTUSERDATA_INDEX);
                           user->hdr.subtype  = ARGUS_LEN_16BITS;
                           user->hdr.subtype |= ARGUS_DST_DATA;
                           user->hdr.argus_dsrvl16.len = len / 4;
                           bcopy (user, retn->dsrs[ARGUS_DSTUSERDATA_INDEX], len);
                        }
                     }
                     break;
                  }


                  default:
                     break;
               }

               dsr = (struct ArgusDSRHeader *)((char *)dsr + len); 

            } else {
               retn = NULL;
               break;
            }
         }

         if (retn != NULL) {
            if (!retn->dsrs[ARGUS_AGR_INDEX]) {
               if ((retn->canon.metric.src.pkts + retn->canon.metric.dst.pkts) > 0) {
                  struct ArgusAgrStruct *agr = &retn->canon.agr;
                  double value;

                  if (parser->ArgusAggregator !=  NULL)
                     value = parser->ArgusAggregator->RaMetricFetchAlgorithm(retn);
                  else
                     value = ArgusFetchDuration(retn);

                  agr->hdr.type             = ARGUS_AGR_DSR;
                  agr->hdr.subtype          = 0x01;
                  agr->hdr.argus_dsrvl8.qual = 0x01;
                  agr->hdr.argus_dsrvl8.len  = (sizeof(*agr) + 3)/4;
                  agr->count                = 1;
                  agr->act.maxval           = value;
                  if (value > 0)
                     agr->act.minval        = value;
                  else
                     agr->act.minval        = 10000000000.0;

                  agr->act.meanval          = value;
                  agr->act.n                = 1;
                  bzero ((char *)&agr->idle, sizeof(agr->idle));
                  retn->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader *) agr;
                  retn->dsrindex |= (0x01 << ARGUS_AGR_INDEX);
               }
            }

            retn->dur   = RaGetFloatDuration(retn);

            if (!((seconds = RaGetFloatSrcDuration(retn)) > 0))
               seconds = retn->dur;
            if (seconds > 0) {
               retn->srate = (float)(retn->canon.metric.src.pkts * 1.0)/seconds;
               retn->sload = (float)(retn->canon.metric.src.bytes*8 * 1.0)/seconds;
            }

            if (!((seconds = RaGetFloatDstDuration(retn)) > 0)) 
               seconds = retn->dur;

            if (seconds > 0) {
               retn->drate = (float)(retn->canon.metric.dst.pkts * 1.0)/seconds;
               retn->dload = (float)(retn->canon.metric.dst.bytes*8 * 1.0)/seconds;
            }

            retn->offset   = input->offset;

            switch (retn->canon.flow.hdr.subtype & 0x3F) {
               case ARGUS_FLOW_LAYER_3_MATRIX:
               case ARGUS_FLOW_CLASSIC5TUPLE: {
                  struct ArgusFlow *flow = (struct ArgusFlow *) retn->dsrs[ARGUS_FLOW_INDEX];

                  switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                     case ARGUS_TYPE_IPV4: {
                        switch (retn->canon.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP: {
                              if (status & ARGUS_DIRECTION)
                                 ArgusReverse = 1;
                              break;
                           }

                           case IPPROTO_ICMP: {
                              if (status & ARGUS_DIRECTION) {
                                 if (retn->canon.metric.src.pkts && !retn->canon.metric.dst.pkts) {
                                    struct ArgusICMPObject *icmp = &retn->canon.net.net_union.icmp;
                                    retn->canon.flow.icmp_flow.type = icmp->icmp_type;
                                    retn->canon.flow.icmp_flow.code = icmp->icmp_code;
                                 }
                              }
                              break;
                           }
                           case IPPROTO_UDP: {
                              struct ArgusNetworkStruct *net;
                              if ((net = (struct ArgusNetworkStruct *) retn->dsrs[ARGUS_NETWORK_INDEX]) != NULL) {
                                 switch (net->hdr.subtype) {
                                    case ARGUS_RTP_FLOW: {
                                       break;
                                    }
                                    case ARGUS_RTCP_FLOW: 
                                       break;
                                 }
                              }
                              break;
                           }
                        }
                        break;
                     }

                     case ARGUS_TYPE_IPV6: {
                        switch (retn->canon.flow.ipv6_flow.ip_p) {
                           case IPPROTO_ICMP: {
                              struct ArgusICMPv6Flow *icmpv6 = (struct ArgusICMPv6Flow *) &retn->canon.flow;
                              if (!(icmpv6->type) && !(icmpv6->type)) {
                                 struct ArgusICMPObject *icmp = &retn->canon.net.net_union.icmp;
                                 icmpv6->type = icmp->icmp_type;
                                 icmpv6->code = icmp->icmp_code;
                              }
                              break;
                           }
                           case IPPROTO_UDP: {
                              struct ArgusNetworkStruct *net;
                              if ((net = (struct ArgusNetworkStruct *) retn->dsrs[ARGUS_NETWORK_INDEX]) != NULL) {
                                 switch (net->hdr.subtype) {
                                    case ARGUS_RTP_FLOW: {
                                       break;
                                    }

                                    case ARGUS_RTCP_FLOW: {
                                       if (flow->ipv6_flow.sport > flow->ipv6_flow.dport) {
                                          ArgusReverse = 1;
                                          status |= ARGUS_DIRECTION;
                                       }
                                       break;
                                    }
                                 }
                              }
                              break;
                           }
                        }
                        break;
                     }
                  }
               }
            }
         }
      }
   }
   
   if ((parser != NULL) && (retn != NULL)) {
      struct ArgusNetworkStruct *net = &retn->canon.net;

      switch (net->hdr.subtype) {
         case ARGUS_TCP_INIT:
            break;

         case ARGUS_TCP_STATUS: {
            struct ArgusTCPObject *tcp = &retn->canon.net.net_union.tcp;
            int status = tcp->status;

            if (status & ARGUS_SAW_SYN_SENT) {
               if (retn->canon.metric.src.pkts && retn->canon.metric.dst.pkts) {
                  if (retn->canon.metric.src.appbytes && tcp->dst.flags != TH_RST)
                     ArgusReverseRecord (retn);
               } else {
                  if ((retn->canon.metric.src.appbytes) && ((status & ARGUS_FIN) || (status & ARGUS_FIN_ACK)))
                     ArgusReverseRecord (retn);
               }
            }

            break;
         }

         case ARGUS_TCP_PERF: {
            struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) &retn->canon.net.net_union.tcp;

            if (tcp->src.bytes && (retn->dsrs[ARGUS_SRCUSERDATA_INDEX] == NULL))
               tcp->src.bytes = 0;
            if (tcp->dst.bytes && (retn->dsrs[ARGUS_DSTUSERDATA_INDEX] == NULL))
               tcp->dst.bytes = 0;
            break;
         }

         case ARGUS_RTP_FLOW: {
            if ((retn->canon.flow.ip_flow.ip_p != IPPROTO_UDP) ||
                (retn->canon.metric.src.pkts < 5))
               retn->dsrs[ARGUS_NETWORK_INDEX] = NULL;
            break;
         }
      }

      if (ArgusReverse && (status & ARGUS_DIRECTION)) {
         ArgusReverseRecord (retn);
      }

      if (retn->hdr.len < 1)
         retn = NULL;
   }

   return (retn);
}


struct ArgusRecordStruct *
ArgusCopyRecordStruct (struct ArgusRecordStruct *rec)
{
   struct ArgusRecordStruct *retn = NULL;
   struct ArgusDSRHeader *dsr;

   if ((retn = (struct ArgusRecordStruct *) ArgusCalloc (1, sizeof(*retn))) != NULL) {
      if (rec) {
         retn->status = rec->status;
         retn->input  = rec->input;
         bcopy ((char *)&rec->hdr, (char *)&retn->hdr, sizeof (rec->hdr));
         bcopy ((char *)&rec->canon, (char *)&retn->canon, sizeof (rec->canon));

         bzero ((char *)&retn->qhdr, sizeof (retn->qhdr));

         if ((retn->dsrindex = rec->dsrindex)) {
            int i;
            for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
               if ((dsr = rec->dsrs[i]) != NULL) {
                  switch (i) {
                     case ARGUS_TRANSPORT_INDEX: retn->dsrs[i] = &retn->canon.trans.hdr; break;
                     case ARGUS_TIME_INDEX:      retn->dsrs[i] = &retn->canon.time.hdr; break;
                     case ARGUS_FLOW_INDEX:      retn->dsrs[i] = (struct ArgusDSRHeader *)&retn->canon.flow.hdr; break;
                     case ARGUS_METRIC_INDEX:    retn->dsrs[i] = &retn->canon.metric.hdr; break;
                     case ARGUS_PSIZE_INDEX:     retn->dsrs[i] = &retn->canon.psize.hdr; break;
                     case ARGUS_NETWORK_INDEX:   retn->dsrs[i] = &retn->canon.net.hdr; break;
                     case ARGUS_IPATTR_INDEX:    retn->dsrs[i] = &retn->canon.attr.hdr; break;
                     case ARGUS_JITTER_INDEX:    retn->dsrs[i] = &retn->canon.jitter.hdr; break;
                     case ARGUS_ICMP_INDEX:      retn->dsrs[i] = &retn->canon.icmp.hdr; break;
                     case ARGUS_ENCAPS_INDEX:    retn->dsrs[i] = &retn->canon.encaps.hdr; break;
                     case ARGUS_MAC_INDEX:       retn->dsrs[i] = &retn->canon.mac.hdr; break;
                     case ARGUS_VLAN_INDEX:      retn->dsrs[i] = &retn->canon.vlan.hdr; break;
                     case ARGUS_MPLS_INDEX:      retn->dsrs[i] = &retn->canon.mpls.hdr; break;
                     case ARGUS_AGR_INDEX:       retn->dsrs[i] = &retn->canon.agr.hdr; break;
                     case ARGUS_COCODE_INDEX:    retn->dsrs[i] = &retn->canon.cocode.hdr; break;

                     case ARGUS_SRCUSERDATA_INDEX: 
                     case ARGUS_DSTUSERDATA_INDEX: {
                        struct ArgusDataStruct *user = (struct ArgusDataStruct *) dsr;

                        int len = (((dsr->type & ARGUS_IMMEDIATE_DATA) ? 1 :
                                   ((dsr->subtype & ARGUS_LEN_16BITS)  ? dsr->argus_dsrvl16.len :
                                                                        dsr->argus_dsrvl8.len)));
                        int size = user->size;

                        if (retn->dsrs[i] != NULL)
                           ArgusFree(retn->dsrs[i]);

                        if (!(size > 0))
                           size = (len - 2) * 4;

                        retn->dsrs[i] = (struct ArgusDSRHeader *) ArgusCalloc(1, size + 8);
                        bcopy (rec->dsrs[i], retn->dsrs[i], size + 8);
                        user = (struct ArgusDataStruct *) retn->dsrs[i];
                        user->size  = size;
                        break;
                     }
                  }

               } else {
                  switch (i) {
                     case ARGUS_SRCUSERDATA_INDEX: 
                     case ARGUS_DSTUSERDATA_INDEX:
                        if (retn->dsrs[i] != NULL)
                            ArgusFree(retn->dsrs[i]);

                     default:
                        retn->dsrs[i] = NULL;
                  }
               }
            }
         }

         retn->srate   = rec->srate;
         retn->drate   = rec->drate;
         retn->sload   = rec->sload;
         retn->dload   = rec->dload;
         retn->sploss  = ArgusFetchPercentSrcLoss(retn);
         retn->dploss  = ArgusFetchPercentDstLoss(retn);
         retn->bins    = NULL;
         retn->htblhdr = NULL;
         retn->nsq     = NULL;
      }

      if (retn->hdr.type & ARGUS_FAR) {
         retn->trans     = rec->trans;
      }
      retn->timeout   = rec->timeout;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusCopyRecordStruct (0x%x) seq %u retn 0x%x\n", rec, rec->canon.trans.seqnum, retn);
#endif 

   return (retn);
}

void
ArgusDeleteRecordStruct (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   struct ArgusRecordStruct *tsns = NULL;

   if (ns != NULL) {
      if (ns->qhdr.queue != NULL)
         ArgusRemoveFromQueue(ns->qhdr.queue, &ns->qhdr, ARGUS_LOCK);

      if (ns->nsq != NULL) {
         while ((tsns = (struct ArgusRecordStruct *) ArgusPopQueue(ns->nsq, ARGUS_LOCK)) != NULL)
            ArgusDeleteRecordStruct (parser, tsns);
         ArgusDeleteQueue (ns->nsq);
         ns->nsq = NULL;
      }

      if (ns->bins) {
         int i;
         if (ns->bins->array != NULL) {
            for (i = 0; i < ns->bins->len; i++)
               if (ns->bins->array[i] != NULL) {
                  RaDeleteBin (parser, ns->bins->array[i]);
                  ns->bins->array[i] = NULL;
               }

            ArgusFree (ns->bins->array);
            ns->bins->array = NULL;
         }

         ArgusFree (ns->bins);
         ns->bins = NULL;
      }

      if (ns->correlates) {
         int i;
         for (i = 0; i < ns->correlates->count; i++)
            ArgusDeleteRecordStruct(parser, ns->correlates->array[i]);
         ArgusFree(ns->correlates->array);
         ns->correlates->array = NULL;
      }

      if (ns->dsrs[ARGUS_SRCUSERDATA_INDEX] != NULL) {
         ArgusFree (ns->dsrs[ARGUS_SRCUSERDATA_INDEX]);
         ns->dsrs[ARGUS_SRCUSERDATA_INDEX] = NULL;
      }
      if (ns->dsrs[ARGUS_DSTUSERDATA_INDEX] != NULL) {
         ArgusFree (ns->dsrs[ARGUS_DSTUSERDATA_INDEX]);
         ns->dsrs[ARGUS_DSTUSERDATA_INDEX] = NULL;
      }

      ArgusFree(ns);
   }
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusDeleteRecordStruct (0x%x, 0x%0x)", parser, ns);
#endif 
}


struct ArgusRecord *
ArgusGenerateRecord (struct ArgusRecordStruct *rec, unsigned char state, char *buf)
{
   struct ArgusRecord *retn = (struct ArgusRecord *) buf;
   unsigned int ind, dsrlen = 1, dsrindex = 0;
   unsigned int *dsrptr = (unsigned int *)retn + 1;
   int i, x, len = 0, type = 0;
   struct ArgusDSRHeader *dsr;

   if (rec) {
      retn->hdr.type  = rec->hdr.type;
      retn->hdr.type  |= ARGUS_VERSION;
      retn->hdr.cause = state & 0xF0;
      
      if (rec->hdr.type & ARGUS_FAR) {
         dsrindex = rec->dsrindex;
         for (i = 0, ind = 1; (dsrindex && (i < ARGUSMAXDSRTYPE)); i++, ind <<= 1) {
            if ((dsr = rec->dsrs[i]) != NULL) {
               len = ((dsr->type & ARGUS_IMMEDIATE_DATA) ? 1 :
                     ((dsr->subtype & ARGUS_LEN_16BITS)  ? dsr->argus_dsrvl16.len :
                                                           dsr->argus_dsrvl8.len));
               switch (i) {
                  case ARGUS_NETWORK_INDEX: {
                     struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *) dsr;
                     switch (net->hdr.subtype) {
                        case ARGUS_TCP_INIT: {
                           struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)dsr;
                           struct ArgusTCPObject *tcp = &net->net_union.tcp;
                           struct ArgusTCPInitStatus tcpinit;
                           tcpinit.status  = tcp->status;
                           tcpinit.options = tcp->options;
                           tcpinit.win = tcp->src.win;
                           tcpinit.flags = tcp->src.flags;
                           tcpinit.winshift = tcp->src.winshift;

                           *dsrptr++ = *(unsigned int *)&net->hdr;
                           *dsrptr++ = ((unsigned int *)&tcpinit)[0];
                           *dsrptr++ = ((unsigned int *)&tcpinit)[1];
                           *dsrptr++ = ((unsigned int *)&tcpinit)[2];
                           *dsrptr++ = ((unsigned int *)&tcpinit)[3];
                           break;
                        }
                        case ARGUS_TCP_STATUS: {
                           struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)dsr;
                           struct ArgusTCPObject *tcp = &net->net_union.tcp;
                           struct ArgusTCPStatus tcpstatus;
                           tcpstatus.status = tcp->status;
                           tcpstatus.src = tcp->src.flags;
                           tcpstatus.dst = tcp->dst.flags;
                           tcpstatus.pad[0] = 0;
                           tcpstatus.pad[1] = 0;
                           *dsrptr++ = *(unsigned int *)&net->hdr;
                           *dsrptr++ = ((unsigned int *)&tcpstatus)[0];
                           *dsrptr++ = ((unsigned int *)&tcpstatus)[1];
                           break;
                        }

                        case ARGUS_TCP_PERF:
                        default: {
                           for (x = 0; x < len; x++)
                              *dsrptr++ = ((unsigned int *)rec->dsrs[i])[x];
                           break;
                        }
                     }
                     break;
                  }

                  case ARGUS_AGR_INDEX: {
                     struct ArgusAgrStruct *agr = (struct ArgusAgrStruct *) dsr;
                     if (agr->count == 1) {
                        len = 0;
                        break;
                     }
                  }

                  default:
                     for (x = 0; x < len; x++)
                        *dsrptr++ = ((unsigned int *)rec->dsrs[i])[x];
                     break;


                  case ARGUS_TIME_INDEX: {
                     struct ArgusTimeObject *dtime = (struct ArgusTimeObject *) dsr;
                     unsigned char subtype = 0;
                     unsigned char tlen = 1;

                        if (dtime->src.start.tv_sec) {
                           subtype |= ARGUS_TIME_SRC_START;
                           tlen += 2;
                        }
                        if (dtime->src.end.tv_sec) {
                           subtype |= ARGUS_TIME_SRC_END;
                           tlen += 2;
                        }
                        if (dtime->dst.start.tv_sec) {
                           subtype |= ARGUS_TIME_DST_START;
                           tlen += 2;
                        }
                        if (dtime->dst.end.tv_sec) {
                           subtype |= ARGUS_TIME_DST_END;
                           tlen += 2;
                        }

                        dtime->hdr.argus_dsrvl8.len = tlen;
                        dtime->hdr.subtype |= subtype;

                        *dsrptr++ = ((unsigned int *)rec->dsrs[i])[0];

                        for (x = 0; x < 4; x++) {
                           if (subtype & (ARGUS_TIME_SRC_START << x)) {
                              switch (ARGUS_TIME_SRC_START << x) {
                                 case ARGUS_TIME_SRC_START:
                                    *dsrptr++ = dtime->src.start.tv_sec;
                                    *dsrptr++ = dtime->src.start.tv_usec;
                                    break;
                                 case ARGUS_TIME_SRC_END:
                                    *dsrptr++ = dtime->src.end.tv_sec;
                                    *dsrptr++ = dtime->src.end.tv_usec;
                                    break;
                                 case ARGUS_TIME_DST_START:
                                    *dsrptr++ = dtime->dst.start.tv_sec;
                                    *dsrptr++ = dtime->dst.start.tv_usec;
                                    break;
                                 case ARGUS_TIME_DST_END:
                                    *dsrptr++ = dtime->dst.end.tv_sec;
                                    *dsrptr++ = dtime->dst.end.tv_usec;
                                    break;
                              }
                           }
                        }
                        len = tlen;
                     break;
                  }

                  case ARGUS_METRIC_INDEX: {
                     struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *) dsr;
                     if ((metric->src.pkts + metric->dst.pkts) > 0) {
                        if ((metric->src.pkts) && (metric->dst.pkts)) {
                           if ((0xFF >= metric->src.pkts)  && (0xFF >= metric->dst.pkts) &&
                               (0xFF >= metric->src.bytes) && (0xFF >= metric->dst.bytes))
                              type = ARGUS_SRCDST_BYTE;
                           else
                           if ((0xFFFF >= metric->src.bytes) && (0xFFFF >= metric->dst.bytes))
                              type = ARGUS_SRCDST_SHORT;
                           else
                           if ((0xFFFFFFFF >= metric->src.bytes) && (0xFFFFFFFF >= metric->dst.bytes))
                              type = ARGUS_SRCDST_INT;
                           else
                              type = ARGUS_SRCDST_LONGLONG;
                        } else {
                           if (metric->src.pkts) {
                              if (0xFFFF >= metric->src.bytes)
                                 type = ARGUS_SRC_SHORT;
                              else
                              if (0xFFFFFFFF >= metric->src.bytes)
                                 type = ARGUS_SRC_INT;
                              else
                                 type = ARGUS_SRC_LONGLONG;
                           } else {
                              if (0xFFFF >= metric->dst.bytes)
                                 type = ARGUS_DST_SHORT;
                              else
                              if (0xFFFFFFFF >= metric->dst.bytes)
                                 type = ARGUS_DST_INT;
                              else
                                 type = ARGUS_DST_LONGLONG;
                           }
                        }
                     } else {
                        type = ARGUS_SRCDST_BYTE;
                     }

                     dsr = (struct ArgusDSRHeader *)dsrptr;
                     dsr->type    = ARGUS_METER_DSR;

                     if (metric->src.appbytes || metric->dst.appbytes) {
                        dsr->subtype = ARGUS_METER_PKTS_BYTES_APP;
                        switch (type) {
                           case ARGUS_SRCDST_BYTE:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned char *)(dsr + 1))[0] = (unsigned char) metric->src.pkts;
                              ((unsigned char *)(dsr + 1))[1] = (unsigned char) metric->src.bytes;
                              ((unsigned char *)(dsr + 1))[2] = (unsigned char) metric->src.appbytes;
                              ((unsigned char *)(dsr + 1))[3] = (unsigned char) metric->dst.pkts;
                              ((unsigned char *)(dsr + 1))[4] = (unsigned char) metric->dst.bytes;
                              ((unsigned char *)(dsr + 1))[5] = (unsigned char) metric->dst.appbytes;
                              break;
                           case ARGUS_SRCDST_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 4;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->src.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->src.bytes);
                              ((unsigned short *)(dsr + 1))[2] = ((unsigned short) metric->src.appbytes);
                              ((unsigned short *)(dsr + 1))[3] = ((unsigned short) metric->dst.pkts);
                              ((unsigned short *)(dsr + 1))[4] = ((unsigned short) metric->dst.bytes);
                              ((unsigned short *)(dsr + 1))[5] = ((unsigned short) metric->dst.appbytes);
                              break;
                           case ARGUS_SRCDST_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 7;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->src.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->src.bytes);
                              ((unsigned int *)(dsr + 1))[2] = ((unsigned int) metric->src.appbytes);
                              ((unsigned int *)(dsr + 1))[3] = ((unsigned int) metric->dst.pkts);
                              ((unsigned int *)(dsr + 1))[4] = ((unsigned int) metric->dst.bytes);
                              ((unsigned int *)(dsr + 1))[5] = ((unsigned int) metric->dst.appbytes);
                              break;
                           case ARGUS_SRCDST_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 13;
                              ((unsigned int *)(dsr + 1))[0]  = (((unsigned int *)&metric->src.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1]  = (((unsigned int *)&metric->src.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2]  = (((unsigned int *)&metric->src.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3]  = (((unsigned int *)&metric->src.bytes)[1]);
                              ((unsigned int *)(dsr + 1))[4]  = (((unsigned int *)&metric->src.appbytes)[0]);
                              ((unsigned int *)(dsr + 1))[5]  = (((unsigned int *)&metric->src.appbytes)[1]);
                              ((unsigned int *)(dsr + 1))[6]  = (((unsigned int *)&metric->dst.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[7]  = (((unsigned int *)&metric->dst.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[8]  = (((unsigned int *)&metric->dst.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[9]  = (((unsigned int *)&metric->dst.bytes)[1]);
                              ((unsigned int *)(dsr + 1))[10] = (((unsigned int *)&metric->dst.appbytes)[0]);
                              ((unsigned int *)(dsr + 1))[11] = (((unsigned int *)&metric->dst.appbytes)[1]);
                              break;

                           case ARGUS_SRC_BYTE:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 2;
                              ((unsigned char *)(dsr + 1))[0] = ((unsigned char) metric->src.pkts);
                              ((unsigned char *)(dsr + 1))[1] = ((unsigned char) metric->src.bytes);
                              ((unsigned char *)(dsr + 1))[2] = ((unsigned char) metric->src.appbytes);
                              break;
                           case ARGUS_SRC_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->src.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->src.bytes);
                              ((unsigned short *)(dsr + 1))[2] = ((unsigned short) metric->src.appbytes);
                              break;
                           case ARGUS_SRC_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 4;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->src.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->src.bytes);
                              ((unsigned int *)(dsr + 1))[2] = ((unsigned int) metric->src.appbytes);
                              break;
                           case ARGUS_SRC_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 7;
                              ((unsigned int *)(dsr + 1))[0] = (((unsigned int *)&metric->src.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1] = (((unsigned int *)&metric->src.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2] = (((unsigned int *)&metric->src.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3] = (((unsigned int *)&metric->src.bytes)[1]);
                              ((unsigned int *)(dsr + 1))[4] = (((unsigned int *)&metric->src.appbytes)[0]);
                              ((unsigned int *)(dsr + 1))[5] = (((unsigned int *)&metric->src.appbytes)[1]);
                              break;

                           case ARGUS_DST_BYTE:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 2;
                              ((unsigned char *)(dsr + 1))[0] = ((unsigned char) metric->dst.pkts);
                              ((unsigned char *)(dsr + 1))[1] = ((unsigned char) metric->dst.bytes);
                              ((unsigned char *)(dsr + 1))[2] = ((unsigned char) metric->dst.appbytes);
                              break;
                           case ARGUS_DST_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->dst.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->dst.bytes);
                              ((unsigned short *)(dsr + 1))[2] = ((unsigned short) metric->dst.appbytes);
                              break;
                           case ARGUS_DST_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 4;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->dst.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->dst.bytes);
                              ((unsigned int *)(dsr + 1))[2] = ((unsigned int) metric->dst.appbytes);
                              break;
                           case ARGUS_DST_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 7;
                              ((unsigned int *)(dsr + 1))[0] = (((unsigned int *)&metric->dst.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1] = (((unsigned int *)&metric->dst.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2] = (((unsigned int *)&metric->dst.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3] = (((unsigned int *)&metric->dst.bytes)[1]);
                              ((unsigned int *)(dsr + 1))[4] = (((unsigned int *)&metric->dst.appbytes)[0]);
                              ((unsigned int *)(dsr + 1))[5] = (((unsigned int *)&metric->dst.appbytes)[1]);
                              break;
                        }
                     } else {
                        dsr->subtype = ARGUS_METER_PKTS_BYTES;
                        switch (type) {
                           case ARGUS_SRCDST_BYTE:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 2;
                              ((unsigned char *)(dsr + 1))[0] = (unsigned char) metric->src.pkts;
                              ((unsigned char *)(dsr + 1))[1] = (unsigned char) metric->src.bytes;
                              ((unsigned char *)(dsr + 1))[2] = (unsigned char) metric->dst.pkts;
                              ((unsigned char *)(dsr + 1))[3] = (unsigned char) metric->dst.bytes;
                              break;
                           case ARGUS_SRCDST_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->src.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->src.bytes);
                              ((unsigned short *)(dsr + 1))[2] = ((unsigned short) metric->dst.pkts);
                              ((unsigned short *)(dsr + 1))[3] = ((unsigned short) metric->dst.bytes);
                              break;
                           case ARGUS_SRCDST_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 5;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->src.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->src.bytes);
                              ((unsigned int *)(dsr + 1))[2] = ((unsigned int) metric->dst.pkts);
                              ((unsigned int *)(dsr + 1))[3] = ((unsigned int) metric->dst.bytes);
                              break;
                           case ARGUS_SRCDST_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 9;
                              ((unsigned int *)(dsr + 1))[0] = (((unsigned int *)&metric->src.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1] = (((unsigned int *)&metric->src.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2] = (((unsigned int *)&metric->src.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3] = (((unsigned int *)&metric->src.bytes)[1]);
                              ((unsigned int *)(dsr + 1))[4] = (((unsigned int *)&metric->dst.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[5] = (((unsigned int *)&metric->dst.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[6] = (((unsigned int *)&metric->dst.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[7] = (((unsigned int *)&metric->dst.bytes)[1]);
                              break;

                           case ARGUS_SRC_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 2;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->src.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->src.bytes);
                              break;
                           case ARGUS_SRC_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->src.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->src.bytes);
                              break;
                           case ARGUS_SRC_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 5;
                              ((unsigned int *)(dsr + 1))[0] = (((unsigned int *)&metric->src.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1] = (((unsigned int *)&metric->src.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2] = (((unsigned int *)&metric->src.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3] = (((unsigned int *)&metric->src.bytes)[1]);
                              break;

                           case ARGUS_DST_SHORT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 2;
                              ((unsigned short *)(dsr + 1))[0] = ((unsigned short) metric->dst.pkts);
                              ((unsigned short *)(dsr + 1))[1] = ((unsigned short) metric->dst.bytes);
                              break;
                           case ARGUS_DST_INT:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 3;
                              ((unsigned int *)(dsr + 1))[0] = ((unsigned int) metric->dst.pkts);
                              ((unsigned int *)(dsr + 1))[1] = ((unsigned int) metric->dst.bytes);
                              break;
                           case ARGUS_DST_LONGLONG:
                              dsr->argus_dsrvl8.qual = type;
                              dsr->argus_dsrvl8.len = 5;
                              ((unsigned int *)(dsr + 1))[0] = (((unsigned int *)&metric->dst.pkts)[0]);
                              ((unsigned int *)(dsr + 1))[1] = (((unsigned int *)&metric->dst.pkts)[1]);
                              ((unsigned int *)(dsr + 1))[2] = (((unsigned int *)&metric->dst.bytes)[0]);
                              ((unsigned int *)(dsr + 1))[3] = (((unsigned int *)&metric->dst.bytes)[1]);
                              break;
                        }
                     }
                     len     = dsr->argus_dsrvl8.len;
                     dsrptr += len;
                     break;
                  }

                  case ARGUS_MPLS_INDEX: {
                     struct ArgusMplsStruct *mpls  = (struct ArgusMplsStruct *) dsr;
                     struct ArgusMplsStruct *tmpls = (struct ArgusMplsStruct *) dsrptr;
                     unsigned char subtype = tmpls->hdr.subtype & ~(ARGUS_MPLS_SRC_LABEL | ARGUS_MPLS_DST_LABEL);

                     *dsrptr++ = *(unsigned int *)dsr;
                     tmpls->hdr.argus_dsrvl8.len = 1;

                     if (((mpls->hdr.argus_dsrvl8.qual & 0xF0) >> 4) > 0) {
                        subtype |= ARGUS_MPLS_SRC_LABEL;
                        *dsrptr++ = mpls->slabel;
                        tmpls->hdr.argus_dsrvl8.len++;
                     }
                     if (((mpls->hdr.argus_dsrvl8.qual & 0x0F)) > 0) {
                        subtype |= ARGUS_MPLS_DST_LABEL;
                        *dsrptr++ = mpls->dlabel;
                        tmpls->hdr.argus_dsrvl8.len++;
                     }
                     tmpls->hdr.subtype = subtype;
                     len = tmpls->hdr.argus_dsrvl8.len;
                     break;
                  }

                  case ARGUS_JITTER_INDEX: {
                     struct ArgusJitterStruct *jitter = (struct ArgusJitterStruct *) dsr;
                     struct ArgusJitterStruct *tjit   = (struct ArgusJitterStruct *) dsrptr;
                     int size = (sizeof(jitter->act.src) + 3) / 4;

                     *dsrptr++ = *(unsigned int *)dsr;
                     tjit->hdr.argus_dsrvl8.len = 1;
                     
                     if (jitter->hdr.argus_dsrvl8.qual & ARGUS_SRC_ACTIVE_JITTER) {
                        unsigned int *tptr = (unsigned int *)&jitter->act.src;
                        for (x = 0; x < size; x++)
                           *dsrptr++ = *tptr++;
                        tjit->hdr.argus_dsrvl8.len += size;
                     }
                     if (jitter->hdr.argus_dsrvl8.qual & ARGUS_SRC_IDLE_JITTER) {
                        unsigned int *tptr = (unsigned int *)&jitter->idle.src;
                        for (x = 0; x < size; x++)
                           *dsrptr++ = *tptr++;
                        tjit->hdr.argus_dsrvl8.len += size;
                     }
                     if (jitter->hdr.argus_dsrvl8.qual & ARGUS_DST_ACTIVE_JITTER) {
                        unsigned int *tptr = (unsigned int *)&jitter->act.dst;
                        for (x = 0; x < size; x++)
                           *dsrptr++ = *tptr++;
                        tjit->hdr.argus_dsrvl8.len += size;
                     }
                     if (jitter->hdr.argus_dsrvl8.qual & ARGUS_DST_IDLE_JITTER) {
                        unsigned int *tptr = (unsigned int *)&jitter->idle.dst;
                        for (x = 0; x < size; x++)
                           *dsrptr++ = *tptr++;
                        tjit->hdr.argus_dsrvl8.len += size;
                     }

                     len = tjit->hdr.argus_dsrvl8.len;
                     break;
                  }

                  case ARGUS_IPATTR_INDEX: {
                     struct ArgusIPAttrStruct *attr  = (struct ArgusIPAttrStruct *) dsr;
                     struct ArgusIPAttrStruct *tattr = (struct ArgusIPAttrStruct *) dsrptr;

                     *dsrptr++ = *(unsigned int *)dsr;
                     tattr->hdr.argus_dsrvl8.len = 1;

                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) {
                        *dsrptr++ = *(unsigned int *)&attr->src;
                        tattr->hdr.argus_dsrvl8.len++;
                     }
                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS) {
                        *dsrptr++ = attr->src.options;
                        tattr->hdr.argus_dsrvl8.len++;
                     }
                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) {
                        *dsrptr++ = *(unsigned int *)&attr->dst;
                        tattr->hdr.argus_dsrvl8.len++;
                     }
                     if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST_OPTIONS) {
                        *dsrptr++ = attr->dst.options;
                        tattr->hdr.argus_dsrvl8.len++;
                     }

                     len = tattr->hdr.argus_dsrvl8.len;
                     break;
                  }
               }

               dsrlen += len;
               dsrindex &= ~ind;
            }
         }

         retn->hdr.len = dsrlen;

         if (((char *)dsrptr - (char *)retn) != (dsrlen * 4))
            ArgusLog (LOG_ERR, "ArgusGenerateRecord: parse length error %d:%d", dsrlen);

      } else  {
         bcopy ((char *)&rec->canon, (char *) retn, rec->hdr.len * 4);
         retn->hdr = rec->hdr;
      }
         
   } else {
      retn->hdr.type = ARGUS_MAR;
      retn->hdr.type  |= ARGUS_VERSION;
      retn->hdr.cause = state & 0xF0;
      retn->hdr.len = dsrlen;
   }
 

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusGenerateRecord (0x%x, %d) done\n", rec, state);
#endif 

   return (retn);
}


struct ArgusHashTableHdr *ArgusFindHashEntry (struct ArgusHashTable *, struct ArgusHashStruct *);
struct ArgusHashTable *ArgusNewHashTable (size_t);
void ArgusDeleteHashTable (struct ArgusHashTable *);

struct RaBinStruct *
RaNewBin (struct ArgusParserStruct *parser, struct RaBinProcessStruct *rbps, struct ArgusRecordStruct *ns, int startpt, int sindex)
{
   struct RaBinStruct *retn = NULL;
   struct ArgusRecordStruct *tns = NULL;

   if ((retn = (struct RaBinStruct *) ArgusCalloc (1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "RaNewBin: ArgusCalloc error %s\n", strerror(errno));

   if ((retn->queue = ArgusNewQueue()) == NULL)
      ArgusLog (LOG_ERR, "RaNewBin: ArgusNewQueue error %s\n", strerror(errno));

   if ((retn->htable = ArgusNewHashTable(RABINS_HASHTABLESIZE)) == NULL)
      ArgusLog (LOG_ERR, "RaNewBin: ArgusNewQueue error %s\n", strerror(errno));

   switch (rbps->nadp.mode) {
      default:
      case ARGUSSPLITTIME:
         retn->value         = startpt;
         retn->stime.tv_sec  = startpt;
         retn->stime.tv_usec = 0;
         retn->etime.tv_sec  = startpt + rbps->size;
         retn->etime.tv_usec = 0;
         break;

      case ARGUSSPLITSIZE:
      case ARGUSSPLITCOUNT:
         retn->value = rbps->start + (rbps->size * (rbps->index - sindex));
         break;
   }

   retn->size  = rbps->nadp.value;

   if (ns != NULL) {
      if ((tns =  ArgusCopyRecordStruct(ns)) != NULL) {
         if (parser->ArgusAggregator != NULL) {
            struct ArgusHashStruct *hstruct = NULL;

            if ((hstruct = ArgusGenerateHashStruct(parser->ArgusAggregator, tns, NULL)) == NULL)
               ArgusLog (LOG_ERR, "RaNewBin: ArgusGenerateHashStruct error %s", strerror(errno));

            ArgusAddHashEntry (retn->htable, tns, hstruct);
            ArgusAddToQueue (retn->queue, &tns->qhdr, ARGUS_LOCK);
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewBin(0x%x, 0x%x) returns 0x%x\n", rbps, ns, retn);
#endif
   return (retn);
}

void
RaDeleteBin (struct ArgusParserStruct *parser, struct RaBinStruct *bin)
{
   struct ArgusRecordStruct *ns;

   if (bin != NULL) {
      if (bin->queue != NULL) {
         while ((ns = (struct ArgusRecordStruct *) ArgusPopQueue(bin->queue, ARGUS_LOCK))) {
            RaSendArgusRecord(ns);
            ArgusDeleteRecordStruct (parser, ns);
         }
        
         ArgusDeleteQueue (bin->queue);
         bin->queue = NULL;
      }

      if (bin->htable != NULL)
         ArgusDeleteHashTable(bin->htable);

      ArgusFree(bin);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaDeleteBin(0x%x)\n", bin);
#endif
   return;
}

struct ArgusMaskStruct *
ArgusSelectMaskDefs(struct ArgusRecordStruct *ns)
{
   struct ArgusNetworkStruct *net;
   struct ArgusMaskStruct *mask;
   struct ArgusFlow *flow;

   if ((flow = (struct ArgusFlow *)ns->dsrs[ARGUS_FLOW_INDEX]) != NULL) {
      net  = &ns->canon.net;

      if (flow->hdr.subtype == ARGUS_FLOW_ARP) {
         mask = ArgusArpMaskDefs;
      } else {
         switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
            case ARGUS_TYPE_IPV6: {
               switch (net->hdr.argus_dsrvl8.qual) {
                  default:           mask = ArgusIpV6MaskDefs;     break;
               }

               break;
            }

            case ARGUS_TYPE_IPV4: {
               switch (net->hdr.argus_dsrvl8.qual) {
                  default:           mask = ArgusIpV4MaskDefs;   break;
               }

               break;
            }

            case ARGUS_TYPE_RARP:
            case ARGUS_TYPE_ARP:   mask = ArgusArpMaskDefs;  break;

            default:
            case ARGUS_TYPE_ETHER: mask = ArgusEtherMaskDefs; break;
         }
      }
   }

   return mask;
}


struct ArgusMaskStruct *
ArgusSelectRevMaskDefs(struct ArgusRecordStruct *ns)
{
   struct ArgusNetworkStruct *net;
   struct ArgusMaskStruct *mask;
   struct ArgusFlow *flow;

   flow = (struct ArgusFlow *)ns->dsrs[ARGUS_FLOW_INDEX];
   net  = &ns->canon.net;

   switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
      case ARGUS_TYPE_IPV6: {
         switch (net->hdr.argus_dsrvl8.qual) {
            default:           mask = ArgusIpV6RevMaskDefs;     break;
         }
         break;
      }

       case ARGUS_TYPE_IPV4: {
          switch (net->hdr.argus_dsrvl8.qual) {
             default:           mask = ArgusIpV4RevMaskDefs;   break;
          }

          break;
       }

      case ARGUS_TYPE_RARP:
      case ARGUS_TYPE_ARP:   mask = ArgusArpRevMaskDefs;  break;

      default:
      case ARGUS_TYPE_ETHER: mask = ArgusEtherRevMaskDefs; break;
   }

   return mask;
}


struct ArgusHashStruct *
ArgusGenerateHashStruct (struct ArgusAggregatorStruct *na,  struct ArgusRecordStruct *ns, struct ArgusFlow *flow)
{
   struct ArgusHashStruct *retn = &na->hstruct;
   struct ArgusCanonRecord *canon = &ns->canon;
   struct ArgusFlow *tflow = (struct ArgusFlow *) ns->dsrs[ARGUS_FLOW_INDEX];

   char *ptr = (char *) na->hstruct.buf; 
   int i, len, tlen = 0, s = sizeof(unsigned short);
   unsigned short *sptr;

   if (ptr == NULL) {
      if ((na->hstruct.buf = (unsigned int *) ArgusCalloc(1, RA_HASHSIZE)) == NULL)
         ArgusLog (LOG_ERR, "ArgusGenerateHashStruct(0x%x, 0x%x, 0x%x) ArgusCalloc returned error %s\n", na, ns, flow, strerror(errno));

      ptr = (char *) na->hstruct.buf;

   } else
      bzero ((char *)ptr, retn->len);

   retn->hash = 0; 
   retn->len  = 0;

   if (na->mask) {
      if (flow == NULL)
         flow = tflow;

      if ((flow != NULL)) {
         if (na->ArgusMaskDefs == NULL)
            na->ArgusMaskDefs = ArgusSelectMaskDefs(ns);

         if (na->mask & ((1 << ARGUS_MASK_SADDR) | (1 << ARGUS_MASK_DADDR))) {
            bcopy ((char *)&flow->hdr, ptr, sizeof(flow->hdr));
            ((struct ArgusFlow *)ptr)->hdr.subtype           &= 0x3F;
            ((struct ArgusFlow *)ptr)->hdr.argus_dsrvl8.qual &= 0x07;
            ptr += sizeof(flow->hdr);
            tlen += sizeof(flow->hdr);
         }
          
         for (i = 0; ((i < ARGUS_MAX_MASK_LIST) && (tlen < RA_HASHSIZE)); i++) {
            int offset = 0, slen = 0;
            if (na->mask & (0x01 << i)) {
               switch (i) {
                  case ARGUS_MASK_SMPLS:
                  case ARGUS_MASK_DMPLS: {
                     unsigned int label  = (*(unsigned int *)&((char *) canon)[na->ArgusMaskDefs[i].offset]) >> 12;
                     bcopy ((char *)&label, ptr, na->ArgusMaskDefs[i].len);
                     ptr  += na->ArgusMaskDefs[i].len;
                     tlen += na->ArgusMaskDefs[i].len;
                     break;
                  }

                  case ARGUS_MASK_SPORT:
                  case ARGUS_MASK_DPORT:
                     switch (flow->hdr.subtype & 0x3F) {
                        case ARGUS_FLOW_CLASSIC5TUPLE: {
                           switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                              case ARGUS_TYPE_IPV4:
                                 switch (flow->ip_flow.ip_p) {
                                    case IPPROTO_ESP: { 
                                       slen   = (i == ARGUS_MASK_SPORT) ? -1 : 4;
                                       offset = (i == ARGUS_MASK_SPORT) ? -1 : na->ArgusMaskDefs[i].offset;
                                       break;
                                    }

                                    case IPPROTO_ICMP: { 
                                       slen = 1;
                                       offset = (i == ARGUS_MASK_SPORT) ? na->ArgusMaskDefs[i].offset :
                                                                          na->ArgusMaskDefs[i].offset - 1;
                                       break;
                                    }
                                 }
                                 break;  
                           }
                        }
                     }

                  default: {
                     if (slen < 0) {
                        slen = 0;
                        break;
                     }

                     if (na->ArgusMaskDefs[i].len > 0) {
                        if (!slen) {
                           slen = na->ArgusMaskDefs[i].len;
                           offset = na->ArgusMaskDefs[i].offset;
                        }

                        if (offset == NLI) {
                           unsigned char *cptr = NULL, cbuf;
                           unsigned short sbuf;
                           switch (slen) {
                              case 0: cbuf = na->ArgusMaskDefs[i].index; cptr = &cbuf; break;
                              case 1: cbuf = na->ArgusMaskDefs[i].index; cptr = &cbuf; break;
                              case 2: sbuf = na->ArgusMaskDefs[i].index; cptr = (unsigned char *)&sbuf; break;
                              case 4:
                              default:
                                 cptr = (unsigned char *)&na->ArgusMaskDefs[i].index;
                                 break;
                           }

                           bcopy (cptr, ptr, slen);

                        } else
                           bcopy (&((char *) canon)[offset], ptr, slen);
                     }
                     break;
                  }
               }

               ptr  += slen;
               tlen += slen;
            }
         }

         retn->len = s * ((tlen + (s - 1))/ s); 
         if (retn->len > RA_HASHSIZE)
            retn->len = RA_HASHSIZE;
         sptr = (unsigned short *)&retn->buf[0];

         for (i = 0, len = retn->len / s; i < len; i++)
            retn->hash += *sptr++;

         na->ArgusMaskDefs = NULL;
      }
   }

   return (retn);
}


struct ArgusHashStruct *
ArgusGenerateReverseHashStruct (struct ArgusAggregatorStruct *na,  struct ArgusRecordStruct *ns, struct ArgusFlow *flow)
{
   struct ArgusHashStruct *retn = &na->hstruct;
   struct ArgusCanonRecord *canon = &ns->canon;
   struct ArgusFlow *tflow = (struct ArgusFlow *) ns->dsrs[ARGUS_FLOW_INDEX];

   char *ptr = (char *) na->hstruct.buf; 
   int i, len, tlen = 0, s = sizeof(unsigned short);
   unsigned short *sptr;

   if (ptr == NULL) {
      if ((na->hstruct.buf = (unsigned int *) ArgusCalloc(1, RA_HASHSIZE)) == NULL)
         ArgusLog (LOG_ERR, "ArgusGenerateHashStruct(0x%x, 0x%x, 0x%x) ArgusCalloc returned error %s\n", na, ns, flow, strerror(errno));

      ptr = (char *) na->hstruct.buf;

   } else
      bzero ((char *)ptr, retn->len);

   retn->hash = 0; 
   retn->len  = 0;

   if (na->mask) {
      if (flow == NULL)
         flow = tflow;

      if ((flow != NULL)) {
         if (na->ArgusMaskDefs == NULL)
            na->ArgusMaskDefs = ArgusSelectRevMaskDefs(ns);

         if (na->mask & ((1 << ARGUS_MASK_SADDR) | (1 << ARGUS_MASK_DADDR))) {
            bcopy ((char *)&flow->hdr, ptr, sizeof(flow->hdr));
            ((struct ArgusFlow *)ptr)->hdr.argus_dsrvl8.qual &= 0x07;
            ptr += sizeof(flow->hdr);
            tlen += sizeof(flow->hdr);
         }
          
         for (i = 0; ((i < ARGUS_MAX_MASK_LIST) && (tlen < RA_HASHSIZE)); i++) {
            int offset = 0, slen = 0;
            if (na->mask & (0x01 << i)) {
               switch (i) {
                  case ARGUS_MASK_SMPLS:
                  case ARGUS_MASK_DMPLS: {
                     unsigned int label  = (*(unsigned int *)&((char *) canon)[na->ArgusMaskDefs[i].offset]) >> 12;
                     bcopy ((char *)&label, ptr, na->ArgusMaskDefs[i].len);
                     ptr  += na->ArgusMaskDefs[i].len;
                     tlen += na->ArgusMaskDefs[i].len;
                     break;
                  }

                  case ARGUS_MASK_SPORT:
                  case ARGUS_MASK_DPORT:
                     switch (flow->hdr.subtype & 0x3F) {
                        case ARGUS_FLOW_CLASSIC5TUPLE: {
                           switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                              case ARGUS_TYPE_IPV4:
                                 switch (flow->ip_flow.ip_p) {
                                    case IPPROTO_ESP: { 
                                       slen   = (i == ARGUS_MASK_SPORT) ? -1 : 4;
                                       offset = (i == ARGUS_MASK_SPORT) ? -1 : na->ArgusMaskDefs[i].offset;
                                       break;
                                    }

                                    case IPPROTO_ICMP: { 
                                       slen = 1;
                                       offset = (i == ARGUS_MASK_SPORT) ? na->ArgusMaskDefs[i].offset - 2:
                                                                          na->ArgusMaskDefs[i].offset + 1;
                                       break;
                                    }
                                 }
                                 break;  
                           }
                        }
                     }

                  default:
                     if (slen < 0) {
                        slen = 0;
                        break;
                     }

                     if (na->ArgusMaskDefs[i].len > 0) {
                        if (!slen) {
                           slen = na->ArgusMaskDefs[i].len;
                           offset = na->ArgusMaskDefs[i].offset;
                        }

                        if (offset == NLI) {
                           unsigned char *cptr = NULL, cbuf;
                           unsigned short sbuf;
                           switch (slen) {
                              case 0: cbuf = na->ArgusMaskDefs[i].index; cptr = &cbuf; break;
                              case 1: cbuf = na->ArgusMaskDefs[i].index; cptr = &cbuf; break;
                              case 2: sbuf = na->ArgusMaskDefs[i].index; cptr = (unsigned char *)&sbuf; break;
                              case 4:
                              default:
                                 cptr = (unsigned char *)&na->ArgusMaskDefs[i].index;
                                 break;
                           }

                           bcopy (cptr, ptr, slen);

                        } else
                           bcopy (&((char *) canon)[offset], ptr, slen);
                     }
                     break;
               }

               ptr  += slen;
               tlen += slen;
            }
         }

         retn->len = s * ((tlen + (s - 1))/ s); 
         if (retn->len > RA_HASHSIZE)
            retn->len = RA_HASHSIZE;
         sptr = (unsigned short *)&retn->buf[0];

         for (i = 0, len = retn->len / s; i < len; i++)
            retn->hash += *sptr++;
      }
   }

   na->ArgusMaskDefs = NULL;

   return (retn);
}


struct ArgusRecordStruct *
ArgusFindRecord (struct ArgusHashTable *htable, struct ArgusHashStruct *hstruct)
{
   struct ArgusRecordStruct *retn = NULL;
   struct ArgusHashTableHdr *hashEntry = NULL, *target, *head;
   unsigned int ind = (hstruct->hash % htable->size), i, len;

#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&htable->lock);
#endif
   if ((target = htable->array[ind]) != NULL) {
      head = target;
      do {
         unsigned short *ptr1 = (unsigned short *) hstruct->buf;
         unsigned short *ptr2 = (unsigned short *) target->hstruct.buf;

         if (ptr1 && ptr2) {
            for (i = 0, len = hstruct->len/sizeof(unsigned short); i < len; i++)
               if (*ptr1++ != *ptr2++)
                  break;
            if (i == len) {
               hashEntry = target;
               break;
            }

         } else
           if (!(ptr1 || ptr2)) {
               hashEntry = target;
               break;
           }

         target = target->nxt;
      } while (target != head);
 
      if (hashEntry != NULL) {
         if (hashEntry != head)
            htable->array[ind] = hashEntry;
         retn = hashEntry->object;
      }
   }
 
#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&htable->lock);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusFindFlow () returning 0x%x\n", retn);
#endif
  
   return (retn);
}


void ArgusEmptyHashTable (struct ArgusHashTable *htbl);

struct ArgusHashTable *
ArgusNewHashTable (size_t size)
{
   struct ArgusHashTable *retn = NULL;

   if ((retn = (struct ArgusHashTable *) ArgusCalloc (1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewHashTable: ArgusCalloc(1, %d) error %s\n", size, strerror(errno));

   if ((retn->array = (struct ArgusHashTableHdr **) ArgusCalloc (size, 
                                      sizeof (struct ArgusHashTableHdr *))) == NULL)
      ArgusLog (LOG_ERR, "RaMergeQueue: ArgusCalloc error %s\n", strerror(errno));

   retn->size = size;

#if defined(ARGUS_THREADS)
   pthread_mutex_init(&retn->lock, NULL);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusNewHashTable (%d) returning 0x%x\n", size, retn);
#endif

   return (retn);
}

void
ArgusDeleteHashTable (struct ArgusHashTable *htbl)
{

   if (htbl != NULL) {
      ArgusEmptyHashTable (htbl);

      if (htbl->array != NULL)
         ArgusFree(htbl->array);

      ArgusFree(htbl);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusDeleteHashTable (0x%x)\n", htbl);
#endif
}

void
ArgusEmptyHashTable (struct ArgusHashTable *htbl)
{
   struct ArgusHashTableHdr *htblhdr = NULL, *tmp;
   int i;
 
#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&htbl->lock);
#endif
   for (i = 0; i < htbl->size; i++) {
      if ((htblhdr = htbl->array[i]) != NULL) {
         htblhdr->prv->nxt = NULL;
         while ((tmp = htblhdr) != NULL) {
            htblhdr = htblhdr->nxt;
            if (tmp->hstruct.buf != NULL)
               ArgusFree (tmp->hstruct.buf);
            ArgusFree (tmp);
         }
         htbl->array[i] = NULL;
      }
   }
#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&htbl->lock);
#endif
 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusEmptyHashTable (0x%x) returning\n", htbl);
#endif
}


struct ArgusHashTableHdr *
ArgusFindHashEntry (struct ArgusHashTable *htable, struct ArgusHashStruct *hstruct)
{
   struct ArgusHashTableHdr *retn = NULL, *target, *head;
   unsigned int ind = (hstruct->hash % htable->size), i, len;

#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&htable->lock);
#endif 

   if ((target = htable->array[ind]) != NULL) {
      head = target;
      do {
         unsigned short *ptr1 = (unsigned short *) hstruct->buf;
         unsigned short *ptr2 = (unsigned short *) target->hstruct.buf;

         for (i = 0, len = hstruct->len/sizeof(unsigned short); i < len; i++)
            if (*ptr1++ != *ptr2++)
               break;
         if (i == len) {
            retn = target;
            break;
         }
         target = target->nxt;
      } while (target != head);
 
      if (retn != NULL) {
         if (retn != head)
            htable->array[ind] = retn;
      }
   }

#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&htable->lock);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusFindHashEntry () returning 0x%x\n", retn);
#endif
  
   return (retn);
}


struct ArgusHashTableHdr *
ArgusAddHashEntry (struct ArgusHashTable *table, struct ArgusRecordStruct *ns, struct ArgusHashStruct *hstruct)
{
   struct ArgusHashTableHdr *retn = NULL, *start = NULL;
   int ind;

   if (hstruct->len > 0) {
      if ((retn = (struct ArgusHashTableHdr *) ArgusCalloc (1, sizeof (struct ArgusHashTableHdr))) == NULL)
         ArgusLog (LOG_ERR, "ArgusAddHashEntry(0x%x, 0x%x, %d) ArgusCalloc returned error %s\n", table, ns, hstruct, strerror(errno));

      retn->object = ns;

      if (hstruct->len > 0) {
         retn->hstruct = *hstruct;
         if ((retn->hstruct.buf = (unsigned int *) ArgusCalloc (1, hstruct->len)) == NULL)
            ArgusLog (LOG_ERR, "ArgusAddHashEntry(0x%x, 0x%x, %d) ArgusCalloc returned error %s\n", table, ns, hstruct, strerror(errno));

         bcopy (hstruct->buf, retn->hstruct.buf, hstruct->len);
      }

      ind = (hstruct->hash % table->size);
      
#if defined(ARGUS_THREADS)
      pthread_mutex_lock(&table->lock);
#endif

      if ((start = table->array[ind]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      table->array[ind] = retn;

#if defined(ARGUS_THREADS)
      pthread_mutex_unlock(&table->lock);
#endif
      retn->htbl = table;

   } else {
#ifdef ARGUSDEBUG
      ArgusDebug (3, "ArgusAddHashEntry (0x%x) no hash struct: returning 0x%x\n", ns, retn);
#endif
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusAddHashEntry (0x%x) returning 0x%x\n", ns, retn);
#endif

   return (retn);
}

void
ArgusRemoveHashEntry (struct ArgusHashTableHdr *htblhdr)
{
   unsigned int hash = htblhdr->hstruct.hash;
   struct ArgusHashTable *table = htblhdr->htbl;
   int ind = hash % table->size;

#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&table->lock);
#endif
   htblhdr->prv->nxt = htblhdr->nxt;
   htblhdr->nxt->prv = htblhdr->prv;

   if (htblhdr == table->array[ind]) {
      if (htblhdr == htblhdr->nxt)
         table->array[ind] = NULL;
      else
         table->array[ind] = htblhdr->nxt;
   }

   if (htblhdr->hstruct.buf != NULL)
      ArgusFree (htblhdr->hstruct.buf);
   ArgusFree (htblhdr);

#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&table->lock);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusRemoveHashEntry (0x%x) returning\n", htblhdr);
#endif
}


int
ArgusCheckTimeout (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns1, struct ArgusRecordStruct *ns2)
{
   int retn = 0, lapseTime;
   struct timeval *tvp = &parser->ArgusGlobalTime;

   int lapseTimeSecs, lapseTimeuSecs;
   int starTimeSecs, starTimeuSecs;
   int lastTimeSecs, lastTimeuSecs;

   if (ns1->timeout > 0) {
      lapseTime = ns1->qhdr.lasttime.tv_sec + ns1->timeout;

      if ((tvp->tv_sec > lapseTime) || ((tvp->tv_sec == lapseTime) &&
              (tvp->tv_usec > ns1->qhdr.lasttime.tv_usec))) {
         ns1->status |= ARGUS_STATUS;
         RaSendArgusRecord(ns1);

      } else {
         starTimeSecs  = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.start.tv_sec;
         starTimeuSecs = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.start.tv_usec;
         
         if ((((struct ArgusTimeObject *)ns2->dsrs[ARGUS_TIME_DSR])->src.end.tv_sec  >
              ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.end.tv_sec) ||
            ((((struct ArgusTimeObject *)ns2->dsrs[ARGUS_TIME_DSR])->src.end.tv_sec  ==
              ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.end.tv_sec) &&
             (((struct ArgusTimeObject *)ns2->dsrs[ARGUS_TIME_DSR])->src.end.tv_usec >
              ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.end.tv_usec))) {
            lastTimeSecs  = ((struct ArgusTimeObject *)ns2->dsrs[ARGUS_TIME_DSR])->src.start.tv_sec;
            lastTimeuSecs = ((struct ArgusTimeObject *)ns2->dsrs[ARGUS_TIME_DSR])->src.start.tv_usec;

         } else {
            lastTimeSecs  = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.start.tv_sec;
            lastTimeuSecs = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.start.tv_usec;
         }

         lapseTimeSecs  = (lastTimeSecs  - starTimeSecs);
         lapseTimeuSecs = (lastTimeuSecs - starTimeuSecs);

         if (lapseTimeuSecs < 0) {
            lapseTimeSecs--;
            lapseTimeuSecs += 1000000;
         }

         if (lapseTimeSecs >= ns1->timeout) {
            ns1->status |= ARGUS_STATUS;
            RaSendArgusRecord(ns1);
         }
      }
   }

   if (ns1->idle > 0) {
      lastTimeSecs  = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.end.tv_sec;
      lastTimeuSecs = ((struct ArgusTimeObject *)ns1->dsrs[ARGUS_TIME_DSR])->src.end.tv_usec;

      if (lastTimeSecs > 0) {
         lapseTimeSecs  = (tvp->tv_sec  - lastTimeSecs);
         lapseTimeuSecs = (tvp->tv_usec - lastTimeuSecs);

         if (lapseTimeSecs >= ns1->idle) {
            ns1->status |= ARGUS_TIMEOUT;
            retn++;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "RaCheckTimeout: returning %d \n", retn);
#endif

   return (retn);
}

int ArgusProcessServiceAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);
int ArgusProcessTCPAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);
int ArgusProcessUDPAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);
int ArgusProcessICMPAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);
int ArgusProcessIPAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);
int ArgusProcessARPAvailability (struct ArgusParserStruct *, struct ArgusRecordStruct *);

int
ArgusProcessServiceAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusFlow *flow = (struct ArgusFlow *)argus->dsrs[ARGUS_FLOW_INDEX];
   int retn = 1;

   argus->status &= ~(RA_SVCPASSED | RA_SVCFAILED);

   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
            case ARGUS_TYPE_IPV4: {
               switch (flow->ip_flow.ip_p) {
                  case IPPROTO_TCP: 
                     argus->status |= ArgusProcessTCPAvailability (parser, argus);
                     break;
                  case IPPROTO_ICMP: 
                     argus->status |= ArgusProcessICMPAvailability (parser, argus);
                     break;
                  case IPPROTO_UDP:
                     argus->status |= ArgusProcessUDPAvailability (parser, argus);
                     break;
                  default:
                     argus->status |= ArgusProcessIPAvailability (parser, argus);
                     break;
               }
               break;
            }
            case ARGUS_TYPE_IPV6: {
               switch (flow->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                     argus->status |= ArgusProcessTCPAvailability (parser, argus);
                     break;
                  case IPPROTO_ICMP: 
                     argus->status |= ArgusProcessICMPAvailability (parser, argus);
                     break;
                  case IPPROTO_UDP:
                     argus->status |= ArgusProcessUDPAvailability (parser, argus);
                     break;
                  default:
                     argus->status |= ArgusProcessIPAvailability (parser, argus);
                     break;
               }
               break;
            }
            case ARGUS_TYPE_RARP:
            case ARGUS_TYPE_ARP:
               argus->status |= ArgusProcessARPAvailability (parser, argus);
               break;

            case ARGUS_TYPE_ETHER:
               break;
         }
         break;
      }
                                                                                                                                  
      default:
         break;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessServiceAvailability: returning %d \n", retn);
#endif

   return (retn);
}


int
ArgusProcessTCPAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *)argus->dsrs[ARGUS_METRIC_INDEX];
   struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)argus->dsrs[ARGUS_NETWORK_INDEX];
   int retn = RA_SVCPASSED, status = 0;

   switch (net->hdr.subtype) {
      case ARGUS_TCP_INIT: 
      case ARGUS_TCP_STATUS: 
      case ARGUS_TCP_PERF: {
         struct ArgusTCPObject *tcp = (void *)&net->net_union.tcp;
         status = tcp->status;
         break;
      }

      default:
         return(retn);
   }

   if (status & ARGUS_SAW_SYN) {
      if (status & ARGUS_RESET) {
         if (status & ARGUS_DST_RESET) {
            if (!(status & ARGUS_SAW_SYN_SENT))
               retn = RA_SVCFAILED;
         } else {
            if (metric->src.pkts && !(metric->dst.pkts))
               retn = RA_SVCFAILED;
         }
      } else {
         if (!(status & (ARGUS_FIN | ARGUS_FIN_ACK | ARGUS_NORMAL_CLOSE)))
            retn = RA_SVCINCOMPLETE;
      }

   } else {
      if (status & (ARGUS_SAW_SYN | ARGUS_SAW_SYN_SENT)) {
         if (metric->src.pkts && !(metric->dst.pkts))
            retn = RA_SVCFAILED;
      }
   }

   if (ARGUS_TIMEOUT)
      if (metric->src.pkts && !(metric->dst.pkts))
         retn = RA_SVCFAILED;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessTCPAvailability: returning %d \n", retn);
#endif

   return (retn);
}

int
ArgusProcessUDPAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *)argus->dsrs[ARGUS_METRIC_INDEX];
   int retn = RA_SVCPASSED;

   if (!metric->src.pkts || !metric->dst.pkts)
      retn = RA_SVCFAILED;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessUDPAvailability: returning %d \n", retn);
#endif

   return (retn);
}

int
ArgusProcessICMPAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *)argus->dsrs[ARGUS_METRIC_INDEX];
   struct ArgusFlow *flow = (struct ArgusFlow *)argus->dsrs[ARGUS_FLOW_INDEX];
   int retn = RA_SVCPASSED;
                                                                                                                                                                           
   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
            case ARGUS_TYPE_IPV4: {
               struct ArgusICMPFlow *icmp = (struct ArgusICMPFlow *) flow;
               switch (icmp->type) {
                  case ICMP_UNREACH:
                  case ICMP_REDIRECT:
                  case ICMP_ROUTERADVERT:
                  case ICMP_TIMXCEED:
                  case ICMP_PARAMETERPROB:
                     break;

                  default:
                     if (!metric->src.pkts || !metric->dst.pkts)
                        retn = RA_SVCFAILED;

                     if (parser->vflag && (metric->src.pkts != metric->dst.pkts))
                        retn = RA_SVCFAILED;
                     break;
               }
               break;
            }

            case ARGUS_TYPE_IPV6: {
/*
               struct ArgusICMPv6Flow *icmpv6 = (struct ArgusICMPv6Flow *) flow;
*/
               break;
            }
        }
     }
  }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessICMPAvailability: returning %d \n", retn);
#endif

   return (retn);
}

int
ArgusProcessIPAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *)argus->dsrs[ARGUS_METRIC_INDEX];
   int retn = RA_SVCPASSED;

   if (metric->src.pkts && !metric->dst.pkts)
      retn = RA_SVCFAILED;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessUDPAvailability: returning %d \n", retn);
#endif

   return (retn);
}

int
ArgusProcessARPAvailability (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   struct ArgusMetricStruct *metric = (struct ArgusMetricStruct *)argus->dsrs[ARGUS_METRIC_INDEX];
   int retn = RA_SVCPASSED;

   if (!metric->src.pkts || !metric->dst.pkts)
      retn = RA_SVCFAILED;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusProcessARPAvailability: returning %d \n", retn);
#endif

   return (retn);
}


long long RaGetuSecAvgDuration (struct ArgusRecordStruct *);
long long RaGetuSecDeltaDuration (struct ArgusRecordStruct *);

long long
RaGetActiveDuration (struct ArgusRecordStruct *argus)
{
   long long retn = 0;

   return (retn);
}


long long
RaGetuSecAvgDuration (struct ArgusRecordStruct *argus)
{
   long long retn = 0;

   return (retn);
}


long long
RaGetuSecDeltaDuration (struct ArgusRecordStruct *argus)
{
   long long retn = 0;

   return (retn);
}

long long
RaGetuSecDuration (struct ArgusRecordStruct *argus)
{
   long long retn = 0;
   int sec, usec;
                                                                                
   sec  = argus->canon.time.src.end.tv_sec  - argus->canon.time.src.start.tv_sec;
   usec = argus->canon.time.src.end.tv_usec - argus->canon.time.src.start.tv_usec;
                                                                                
   if (usec < 0) {
      sec--; usec += 1000000;
   }
   retn  = (sec * 1000000) + usec;
   return (retn);
}

char RaUserDataStr[MAXSTRLEN];

char *
RaGetUserDataString (struct ArgusRecordStruct *argus)
{
   char *retn = RaUserDataStr;
   return (retn);
}


/*
   Format is "metric bins[L]:range" or "metric bins:size" where
   range is value-value and size if just a single number.  Value is 
   %f[umsMHD] or %f[umKMG] depending on the type of metric used.

   Appropriate metrics are any metrics support for sorting.

*/

int ArgusHistoMetricParse (struct ArgusParserStruct *, struct ArgusAggregatorStruct *);

int
ArgusHistoMetricParse (struct ArgusParserStruct *parser, struct ArgusAggregatorStruct *agr)
{
   char *ptr, *vptr, tmpbuf[128], *tmp = tmpbuf; 
   char *str = parser->Hstr, *endptr = NULL;
   char *metric = NULL;
   int retn = 0, keyword;

   bzero (tmpbuf, 128);
   snprintf (tmpbuf, 128, "%s", str);

   if ((ptr = strchr (tmp, ' ')) != NULL) {
      int x, found = 0;
      metric = tmp;
      *ptr++ = '\0';
      tmp = ptr;

      for (x = 0; x < MAX_METRIC_ALG_TYPES; x++) {
         if (!strncmp (ArgusMetricKeyWords[x], metric, strlen(metric))) {
            agr->RaMetricFetchAlgorithm = ArgusFetchAlgorithmTable[x];
            keyword = x;
            found++;
            break;
         }
      }
      if (!found)
         usage();
   }

   if ((ptr = strchr (tmp, ':')) != NULL) {
      *ptr++ = '\0';
      vptr = ptr;

      if (strchr (tmp, 'L'))
         parser->RaHistoMetricLog++;

      if (isdigit((int)*tmp))
         if ((parser->RaHistoBins = atoi(tmp)) < 0)
            return (retn);

      if ((ptr = strchr (vptr, '-')) != NULL) {
         *ptr++ = '\0';
         parser->RaHistoStart = strtod(vptr, &endptr);
         if (endptr == vptr)
            return (retn);
         parser->RaHistoEnd = strtod(ptr, &endptr);
         if (endptr == ptr)
            return (retn);
      } else {
         parser->RaHistoStart = 0.0;
         parser->RaHistoBinSize = strtod(vptr, &endptr);
         if (endptr == vptr)
            return (retn);
         parser->RaHistoEnd = parser->RaHistoBinSize * (parser->RaHistoBins * 1.0);
      }

      switch (*endptr) {
         case 'u': parser->RaHistoStart *= 0.000001;
                   parser->RaHistoEnd   *= 0.000001; break;
         case 'm': parser->RaHistoStart *= 0.001;
                   parser->RaHistoEnd   *= 0.001;    break;
         case 's': parser->RaHistoStart *= 1.0;
                   parser->RaHistoEnd   *= 1.0;      break;
         case 'M': {
            switch (keyword) {
               case ARGUSMETRICSTARTTIME:
               case ARGUSMETRICLASTTIME:
               case ARGUSMETRICDURATION:
               case ARGUSMETRICAVGDURATION:
               case ARGUSMETRICMINDURATION:
               case ARGUSMETRICMAXDURATION:
                  parser->RaHistoStart *= 60.0;
                  parser->RaHistoEnd   *= 60.0;
                  break;

               default:
                  parser->RaHistoStart *= 1000000.0;
                  parser->RaHistoEnd   *= 1000000.0;
                  break;
            }
            break;
         }
         case 'H': parser->RaHistoStart *= 3600.0;
                   parser->RaHistoEnd   *= 3600.0;   break;
         case 'D': parser->RaHistoStart *= 86400.0;
                   parser->RaHistoEnd   *= 86400.0;  break;
         case 'K': parser->RaHistoStart *= 1000.0;
                   parser->RaHistoEnd   *= 1000.0;  break;
         case 'G': parser->RaHistoStart *= 1000000000.0;
                   parser->RaHistoEnd   *= 1000000000.0;  break;
         case  ' ':
         case '\0': break;

         default:
            return (retn);
      }

      retn++;

      if ((parser->RaHistoRecords = (struct ArgusRecordStruct **) ArgusCalloc (parser->RaHistoBins + 2, sizeof(struct ArgusRecordStruct *))) != NULL) {
         if (parser->RaHistoMetricLog) {
            parser->RaHistoEndLog      = log10(parser->RaHistoEnd);

            if (parser->RaHistoStart > 0) {
               parser->RaHistoStartLog = log10(parser->RaHistoStart);
            } else {
               parser->RaHistoLogInterval = (parser->RaHistoEndLog/(parser->RaHistoBins * 1.0));
               parser->RaHistoStartLog = parser->RaHistoEndLog - (parser->RaHistoLogInterval * parser->RaHistoBins);
            }

            parser->RaHistoBinSize = (parser->RaHistoEndLog - parser->RaHistoStartLog) / parser->RaHistoBins * 1.0;

         } else
            parser->RaHistoBinSize = ((parser->RaHistoEnd - parser->RaHistoStart) * 1.0) / parser->RaHistoBins * 1.0;
      } else
         ArgusLog (LOG_ERR, "RaHistoMetricParse: ArgusCalloc %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusHistoMetricParse(0x%x): returning %d \n", parser, retn);
#endif
   return (retn);
}

int
ArgusHistoTallyMetric (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   int retn = 0, i;
   double start, bsize;
   double value, iptr;

   if (parser && parser->ArgusAggregator && (ns != NULL)) {
      value = parser->ArgusAggregator->RaMetricFetchAlgorithm(ns);

      if (parser->RaHistoMetricLog) {
         value = log10(value);
         start = parser->RaHistoStartLog;
         bsize = parser->RaHistoBinSize;
      } else {
         start = parser->RaHistoStart;
         bsize = parser->RaHistoBinSize;
      }

      if (value < start) {
         i = 0;
      } else {
         modf((value - start)/bsize, &iptr);

         if ((i = iptr) > parser->RaHistoBins)
            i = parser->RaHistoBins;
         i++;
      }
   }

   if (parser->RaHistoRecords[i] != NULL) {
      ArgusMergeRecords (parser->ArgusAggregator, parser->RaHistoRecords[i], ns);
   } else
      parser->RaHistoRecords[i] = ArgusCopyRecordStruct(ns);

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusHistoTallyMetric(0x%x, 0x%x): returning %d\n", parser, ns, retn);
#endif
   return (retn);
}


struct RaPolicyStruct *
RaFlowModelOverRides(struct ArgusAggregatorStruct *na, struct ArgusRecordStruct *ns)
{
   struct RaPolicyStruct *retn = NULL;

   return (retn);
}


void
ArgusGenerateNewFlow(struct ArgusAggregatorStruct *na, struct ArgusRecordStruct *ns)
{
   struct ArgusFlow tflow, *flow = (struct ArgusFlow *)ns->dsrs[ARGUS_FLOW_INDEX];
   int i = 0, x = 0, tlen = 0;

   bzero ((char *)&tflow, sizeof(tflow));

   na->ArgusMaskDefs = ArgusSelectMaskDefs(ns);

   if (na->mask && (flow != NULL)) {
      bcopy ((char *)&flow->hdr, (char *)&tflow.hdr, sizeof(tflow.hdr));
      if (!(na->mask & (ARGUS_MASK_SADDR_INDEX | ARGUS_MASK_DADDR_INDEX | ARGUS_MASK_PROTO_INDEX | ARGUS_MASK_SPORT_INDEX | ARGUS_MASK_DPORT_INDEX))) {
         tflow.hdr.subtype &= 0x3F;
         tflow.hdr.argus_dsrvl8.qual &= 0x1F;
      }
      for (i = 0; ((i < ARGUS_MAX_MASK_LIST) && (tlen < sizeof(tflow))); i++) {
         if (na->mask & (0x01 << i)) {
            switch(i) {
               case ARGUS_MASK_SADDR:
                  switch(flow->hdr.subtype & 0x3F) {
                     case ARGUS_FLOW_LAYER_3_MATRIX:
                     case ARGUS_FLOW_CLASSIC5TUPLE:  {
                        switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_IPV4:
                              tflow.ip_flow.ip_src = flow->ip_flow.ip_src;
                              if (na->saddrlen > 0)
                                 tflow.ip_flow.ip_src &= na->smask.addr_un.ipv4;
                              break;

                           case ARGUS_TYPE_IPV6:  {
                              for (x = 0; x < 4; x++)
                                 tflow.ipv6_flow.ip_src[x] = flow->ipv6_flow.ip_src[x];
                              
                              if (na->saddrlen > 0)
                                 for (x = 0; x < 4; x++)
                                    tflow.ipv6_flow.ip_src[x] &= na->smask.addr_un.ipv6[x];
                              break;
                           }

                           case ARGUS_TYPE_RARP:
                              bcopy (&flow->rarp_flow.shaddr, &tflow.rarp_flow.shaddr, tflow.rarp_flow.hln);
                              break;

                           case ARGUS_TYPE_ETHER: {
                              bcopy (&flow->mac_flow.ehdr.ether_shost,
                                     &tflow.mac_flow.ehdr.ether_shost,
                                     sizeof(tflow.mac_flow.ehdr.ether_shost));
                              break;
                           }
                        }
                        break;
                     }
                     case ARGUS_FLOW_ARP:  {
                        tflow.arp_flow.pln = flow->arp_flow.pln;
                        tflow.arp_flow.arp_spa = flow->arp_flow.arp_spa;
                        break;
                     }
                  }
                  break;

               case ARGUS_MASK_DADDR:
                  switch(flow->hdr.subtype & 0x3F) {
                     case ARGUS_FLOW_LAYER_3_MATRIX:
                     case ARGUS_FLOW_CLASSIC5TUPLE: {
                        switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_IPV4:
                              tflow.ip_flow.ip_dst = flow->ip_flow.ip_dst;
                              if (na->daddrlen) 
                                 tflow.ip_flow.ip_dst &= na->dmask.addr_un.ipv4;
                              break;
                           case ARGUS_TYPE_IPV6:
                              for (x = 0; x < 4; x++)
                                 tflow.ipv6_flow.ip_dst[x] = flow->ipv6_flow.ip_dst[x];
                              
                              if (na->daddrlen > 0)
                                 for (x = 0; x < 4; x++)
                                    tflow.ipv6_flow.ip_dst[x] &= na->dmask.addr_un.ipv6[x];
                              break;

                           case ARGUS_TYPE_RARP:
                              bcopy (&flow->rarp_flow.dhaddr, &tflow.rarp_flow.dhaddr, tflow.rarp_flow.hln);
                              break;
                           case ARGUS_TYPE_ARP: 
                              tflow.arp_flow.arp_tpa = flow->arp_flow.arp_tpa;
                              break;

                           case ARGUS_TYPE_ETHER:
                              bcopy (&flow->mac_flow.ehdr.ether_dhost,
                                     &tflow.mac_flow.ehdr.ether_dhost,
                                     sizeof(tflow.mac_flow.ehdr.ether_dhost));
                              break;

                        }
                        break;
                     }

                     case ARGUS_FLOW_ARP:  {
                        tflow.arp_flow.arp_tpa = flow->arp_flow.arp_tpa;
                        break;
                     }
                  }
                  break;

               case ARGUS_MASK_PROTO:
                  switch(flow->hdr.subtype & 0x3F) {
                     case ARGUS_FLOW_CLASSIC5TUPLE: {
                        switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_IPV4: tflow.ip_flow.ip_p   = flow->ip_flow.ip_p; break;
                           case ARGUS_TYPE_IPV6: {
                              tflow.ipv6_flow.ip_p = flow->ipv6_flow.ip_p;
                              break;
                           }
                           case ARGUS_TYPE_ETHER:
                              tflow.mac_flow.ehdr.ether_type = flow->mac_flow.ehdr.ether_type;
                              break;
                        }
                        break;
                     }
                     case ARGUS_FLOW_ARP:  {
                        tflow.arp_flow.pro = flow->arp_flow.pro;
                        break;
                     }
                     break;
                  }
                  break;

               case ARGUS_MASK_SPORT:
                  switch(flow->hdr.subtype & 0x3F) {
                     case ARGUS_FLOW_CLASSIC5TUPLE: {
                        switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_IPV4: 
                              switch (flow->ip_flow.ip_p) {
                                 case IPPROTO_ESP: {
                                    break;
                                 }
                                 default:
                                    tflow.ip_flow.sport = flow->ip_flow.sport;
                                    break;
                              }
                              break;
                           case ARGUS_TYPE_IPV6: tflow.ipv6_flow.sport = flow->ipv6_flow.sport; break;
                           case ARGUS_TYPE_ETHER:
                              tflow.mac_flow.ssap = flow->mac_flow.ssap;
                              break;
                        }
                        break;
                     }
                     case ARGUS_FLOW_ARP:  {
                        break;
                     }
                     break;
                  }
                  break;

               case ARGUS_MASK_DPORT:
                  switch(flow->hdr.subtype & 0x3F) {
                     case ARGUS_FLOW_CLASSIC5TUPLE: {
                        switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_IPV4: 
                              switch (flow->ip_flow.ip_p) {
                                 case IPPROTO_ESP: {
                                    tflow.esp_flow.spi = flow->esp_flow.spi;
                                    break;
                                 }
                                 default:
                                    tflow.ip_flow.dport = flow->ip_flow.dport;
                                    break;
                              }
                              break;

                           case ARGUS_TYPE_IPV6: tflow.ipv6_flow.dport = flow->ipv6_flow.dport; break;
                           case ARGUS_TYPE_ETHER:
                              tflow.mac_flow.dsap = flow->mac_flow.dsap;
                              break;
                        }
                        break;
                     }
                     case ARGUS_FLOW_ARP:  {
                        int hln = flow->arp_flow.hln;
                        
                        if (hln > sizeof(flow->arp_flow.haddr))
                           hln = sizeof(flow->arp_flow.haddr);
                        tflow.arp_flow.hln = hln;
                        bcopy((char *)&flow->arp_flow.haddr, (char *)&tflow.arp_flow.haddr, hln);
                        break;
                     }
                     break;
                  }
                  break;

               case ARGUS_MASK_SRCID:
                  break;

               default:
                  break;
            }
         }
      }
      bcopy ((char *)flow, (char *)&na->fstruct, sizeof(*flow));
      bcopy ((char *)&tflow, (char *)flow, sizeof(tflow));
   }
}

/*
    ArgusMergeRecords
       This routine takes 2 ArgusRecordStructs and merges each DSR,
       leaving the resultant in ns1.
*/

/*
    ArgusMergeAddress
       Given two non-equal addresses, ArgusMergeAddress() will do a MSB
       run length compare, leaving the result in the first address passed.
       If either the addrs are the broadcast address (0xFFFFFFFF) the
       the result will be the other address.
*/

unsigned int
ArgusMergeAddress(unsigned int *a1, unsigned int *a2, int type, int dir)
{
   unsigned int retn = 0;

   switch (type) {
      case ARGUS_TYPE_IPV4: {
         if (*a1 != *a2) {
            if (*a1 == 0xFFFFFFFF) {
               *a1 = *a2;
            } else
            if (*a2 == 0xFFFFFFFF) {
            } else {
               unsigned int value = 0, ind = 0x80000000;
               while (ind && ((*a1 & ind) == (*a2 & ind))) {
                  value |= (*a1 & ind);
                  ind >>= 1;
               }
               *a1 = value;
            }
         }
         break;
      }

      case ARGUS_TYPE_IPV6: {
         int i, z;

         for (i = 0, z = 0; i < 4; i++) {
            if (z) {
               a1[i] = 0;
            } else {
               if (a1[i] != a2[i]) {
                  if (a1[i] == 0xFFFFFFFF) {
                     a1[i] = a2[i];
                  } else
                  if (a2[i] == 0xFFFFFFFF) {
                  } else {
                     unsigned int value = 0, ind = 0x80000000;
                     while (ind && ((a1[i] & ind) == (a2[i] & ind))) {
                        value |= (a1[i] & ind);
                        ind >>= 1;
                     }
                     if ((a1[i] & ind) != (a2[i] & ind))
                        z++;
                     a1[i] = value;
                  }
               }
            }
         }
         break;
      }

      case ARGUS_TYPE_ETHER: {
         break;
      }

      case ARGUS_TYPE_RARP:
      case ARGUS_TYPE_ARP: {
         break;
      }
   }

   switch (dir) {
      case ARGUS_SRC:
      case ARGUS_DST:
         break;
   }

   return (retn);
}


void
ArgusMergeRecords (struct ArgusAggregatorStruct *na, struct ArgusRecordStruct *ns1, struct ArgusRecordStruct *ns2)
{
   struct ArgusAgrStruct *agr = NULL;
   double seconds;
   int i;

   if ((ns1 && ns2) && ((ns1->hdr.type & ARGUS_FAR) && (ns2->hdr.type & ARGUS_FAR))) {
      if ((agr = (struct ArgusAgrStruct *) ns1->dsrs[ARGUS_AGR_INDEX]) == NULL) {
         if ((ns1->canon.metric.src.pkts + ns1->canon.metric.dst.pkts) > 0) {
            double value = na->RaMetricFetchAlgorithm(ns1);
            if ((agr = &ns1->canon.agr) == NULL)
               ArgusLog (LOG_ERR, "ArgusMergeRecords: agr record missing");
            agr->hdr.type            = ARGUS_AGR_DSR;
            agr->hdr.subtype         = 0x01;
            agr->hdr.argus_dsrvl8.qual = 0x01;
            agr->hdr.argus_dsrvl8.len  = (sizeof(*agr) + 3)/4;
            agr->count               = 1;
            agr->act.maxval          = value;
            if ((agr->act.minval = value) == 0)
               agr->act.minval = 1000000000.0;
            agr->act.meanval         = value;
            agr->act.n               = 1;
            bzero ((char *)&agr->idle, sizeof(agr->idle));
            ns1->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader *) agr;
            ns1->dsrindex |= (0x01 << ARGUS_AGR_INDEX);
         }
      }

      for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
         switch (i) {
/*
   Merging Flow records is a matter of testing each field and
   transforming values that are not equal to either run length
   and'ing or zeroing out the value.  When a value is zero'ed
   we need to indicate it in the status field of the flow
   descriptor so that values resulting from merging are not
   confused with values actually off the wire.
 
   run length and'ing is an attempt to preserve CIDR addresses.
   any other value should be either preserved or invalidated.

*/
            case ARGUS_FLOW_INDEX: {
               struct ArgusFlow *f1 = (struct ArgusFlow *) ns1->dsrs[ARGUS_FLOW_INDEX];
               struct ArgusFlow *f2 = (struct ArgusFlow *) ns2->dsrs[ARGUS_FLOW_INDEX];

               if (f1 && f2) {
                  if ((f1->hdr.subtype == f2->hdr.subtype)) {
                     char f1qual = f1->hdr.argus_dsrvl8.qual & 0x1F;
                     char f2qual = f2->hdr.argus_dsrvl8.qual & 0x1F;

                     switch (f1->hdr.subtype & 0x3F) {
                        case ARGUS_FLOW_LAYER_3_MATRIX: {
                           if (f1qual == f2qual) {
                              switch (f1qual) {
                                 case ARGUS_TYPE_IPV4:
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ip_flow.ip_src, &f2->ip_flow.ip_src, ARGUS_TYPE_IPV4, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ip_flow.ip_dst, &f2->ip_flow.ip_dst, ARGUS_TYPE_IPV4, ARGUS_DST);
                                    break;

                                 case ARGUS_TYPE_IPV6:  
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_src[0], &f2->ipv6_flow.ip_src[0], ARGUS_TYPE_IPV6, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_dst[0], &f2->ipv6_flow.ip_dst[0], ARGUS_TYPE_IPV6, ARGUS_DST);
                                    break;

                              }
                           }
                           break;
                        }

                        case ARGUS_FLOW_CLASSIC5TUPLE: {
                              switch (f1qual) {
                                 case ARGUS_TYPE_IPV4:
                                    if (f1qual == f2qual) {
                                       ArgusMergeAddress(&f1->ip_flow.ip_src, &f2->ip_flow.ip_src, ARGUS_TYPE_IPV4, ARGUS_SRC);
                                       ArgusMergeAddress(&f1->ip_flow.ip_dst, &f2->ip_flow.ip_dst, ARGUS_TYPE_IPV4, ARGUS_DST);
                                       if (f1->ip_flow.ip_p  != f2->ip_flow.ip_p)
                                          f1->ip_flow.ip_p = 0;
                                       else {
                                          switch (f1->ip_flow.ip_p) {
                                             case IPPROTO_ESP: {
                                                if (f1->esp_flow.spi != f2->esp_flow.spi)
                                                   f1->esp_flow.spi = 0;
                                                break;
                                             }

                                             default: {
                                                if (f1->ip_flow.sport != f2->ip_flow.sport)
                                                   f1->ip_flow.sport = 0;
                                                if (f1->ip_flow.dport != f2->ip_flow.dport)
                                                   f1->ip_flow.dport = 0;
                                                break;
                                             }
                                          }
                                       }

                                    } else {
                                       f1->ip_flow.ip_src = 0;
                                       f1->ip_flow.ip_dst = 0;
                                    
                                       switch (f2qual) {
                                          case ARGUS_TYPE_IPV6:
                                             if (f1->ip_flow.ip_p  != f2->ipv6_flow.ip_p)
                                                f1->ip_flow.ip_p = 0;
                                             if (f1->ip_flow.sport != f2->ipv6_flow.sport)
                                                f1->ip_flow.sport = 0;
                                             if (f1->ip_flow.dport != f2->ipv6_flow.dport)
                                                f1->ip_flow.dport = 0;
                                             break;
              
                                          default:
                                             f1->ip_flow.ip_p = 0;
                                             f1->ip_flow.sport = 0;
                                             f1->ip_flow.dport = 0;
                                             break;
                                       }
                                    }
                                    break;

                                 case ARGUS_TYPE_IPV6:  
                                    if (f1qual == f2qual) {
                                       f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_src[0],
                                               &f2->ipv6_flow.ip_src[0], ARGUS_TYPE_IPV6, ARGUS_SRC);
                                       f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_dst[0],
                                               &f2->ipv6_flow.ip_dst[0], ARGUS_TYPE_IPV6, ARGUS_DST);

                                       if (f1->ipv6_flow.ip_p  != f2->ipv6_flow.ip_p)  f1->ipv6_flow.ip_p = 0;
                                       if (f1->ipv6_flow.sport != f2->ipv6_flow.sport) f1->ipv6_flow.sport = 0;
                                       if (f1->ipv6_flow.dport != f2->ipv6_flow.dport) f1->ipv6_flow.dport = 0;

                                    } else {
                                       bzero ((char *)&f1->ipv6_flow.ip_src[0], sizeof(f1->ipv6_flow.ip_src));
                                       bzero ((char *)&f1->ipv6_flow.ip_dst[0], sizeof(f1->ipv6_flow.ip_dst));
                                       if (f1->ipv6_flow.ip_p  != f2->ip_flow.ip_p)  f1->ipv6_flow.ip_p = 0;
                                       if (f1->ipv6_flow.sport != f2->ip_flow.sport) f1->ipv6_flow.sport = 0;
                                       if (f1->ipv6_flow.dport != f2->ip_flow.dport) f1->ipv6_flow.dport = 0;
                                    }
                                    break;

                               case ARGUS_TYPE_RARP:
                                  if (bcmp(&f1->rarp_flow.shaddr, &f2->rarp_flow.shaddr, 6))
                                     bzero(&f1->rarp_flow.shaddr, 6);
                                  if (bcmp(&f1->rarp_flow.dhaddr, &f2->rarp_flow.dhaddr, 6))
                                     bzero(&f1->rarp_flow.dhaddr, 6);
                                  break;
                               case ARGUS_TYPE_ARP:
                                  f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_spa, &f2->arp_flow.arp_spa, ARGUS_TYPE_ARP, ARGUS_SRC);
                                  f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_tpa, &f2->arp_flow.arp_tpa, ARGUS_TYPE_ARP, ARGUS_DST);
                                  break;
                           }
                           break;
                        }

                        case ARGUS_FLOW_ARP: {
                           switch (f1qual) {
                               case ARGUS_TYPE_RARP: {
                                  if (bcmp(&f1->rarp_flow.shaddr, &f2->rarp_flow.shaddr, 6))
                                          bzero(&f1->rarp_flow.shaddr, 6);
                                  if (bcmp(&f1->rarp_flow.dhaddr, &f2->rarp_flow.dhaddr, 6))
                                          bzero(&f1->rarp_flow.dhaddr, 6);

                                  if (f1->arp_flow.pln == 4) {
                                     ArgusMergeAddress(&f1->arp_flow.arp_tpa, &f2->arp_flow.arp_tpa, ARGUS_TYPE_IPV4, ARGUS_DST);
                                  }
                                  break;
                              }

                               case ARGUS_TYPE_ARP: {
                                  if (bcmp(&f1->arp_flow.haddr, &f2->arp_flow.haddr, 6))
                                     bzero(&f1->arp_flow.haddr, 6);

                                  if (f1->arp_flow.pln == 4) {
                                     ArgusMergeAddress(&f1->arp_flow.arp_spa, &f2->arp_flow.arp_spa, ARGUS_TYPE_IPV4, ARGUS_SRC);
                                     ArgusMergeAddress(&f1->arp_flow.arp_tpa, &f2->arp_flow.arp_tpa, ARGUS_TYPE_IPV4, ARGUS_DST);
                                  }
                                  break;
                              }
                           }
                           break;
                        }
                     }
                  }
               }
            }
            break;
/*
   Merging Transport objects involves simply checking that the source
   id and seqnum are the same, and if not, removing the fields until
   we're actually removing the struct.

struct ArgusTransportStruct {
   struct ArgusDSRHeader hdr;
   struct ArgusAddrStruct srcid;
   unsigned int seqnum;
};

*/
            case ARGUS_TRANSPORT_INDEX: {
               struct ArgusTransportStruct *t1 = (struct ArgusTransportStruct *) ns1->dsrs[ARGUS_TRANSPORT_INDEX];
               struct ArgusTransportStruct *t2 = (struct ArgusTransportStruct *) ns2->dsrs[ARGUS_TRANSPORT_INDEX];

               if ((t1 && t2) && (t1->hdr.argus_dsrvl8.qual == t2->hdr.argus_dsrvl8.qual)) {
                  if (t1->hdr.argus_dsrvl8.qual == t2->hdr.argus_dsrvl8.qual) {
                     if ((t1->hdr.subtype & ARGUS_SRCID) && (t2->hdr.subtype & ARGUS_SRCID)) {
                        switch (t1->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_INT:
                           case ARGUS_TYPE_IPV4:
                              if (t1->srcid.a_un.ipv4 != t2->srcid.a_un.ipv4) {
                                 ns1->dsrs[ARGUS_TRANSPORT_INDEX] = NULL;
                                 ns1->dsrindex &= ~(0x1 << ARGUS_TRANSPORT_INDEX);
                              }
                              break;

                           case ARGUS_TYPE_IPV6:
                           case ARGUS_TYPE_ETHER:
                           case ARGUS_TYPE_STRING:
                              break;
                        }
                     }

                  } else {
                     ns1->dsrs[ARGUS_TRANSPORT_INDEX] = NULL;
                     ns1->dsrindex &= ~(0x1 << ARGUS_TRANSPORT_INDEX);
                  }
               }

               break;
            }
/*
   Merging Time objects may result in a change in the storage
   type of the time structure, from an ABSOLUTE_TIMESTAMP
   to an ABSOLUTE_RANGE, to hold the new ending time.
*/
            case ARGUS_TIME_INDEX: {
               struct ArgusTimeObject *t1 = (struct ArgusTimeObject *) ns1->dsrs[ARGUS_TIME_INDEX];
               struct ArgusTimeObject *t2 = (struct ArgusTimeObject *) ns2->dsrs[ARGUS_TIME_INDEX];

               if (t1 && t2) {
                  unsigned int st1, st2;
                  st1 = t1->hdr.subtype & (ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START |
                                           ARGUS_TIME_SRC_END   | ARGUS_TIME_DST_END);

                  st2 = t2->hdr.subtype & (ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START |
                                           ARGUS_TIME_SRC_END   | ARGUS_TIME_DST_END);

                  if (st2) {
                     if (st2 & ARGUS_TIME_SRC_START) {
                        if (st1 & ARGUS_TIME_SRC_START) {
                           if ((t1->src.start.tv_sec  >  t2->src.start.tv_sec) ||
                              ((t1->src.start.tv_sec  == t2->src.start.tv_sec) &&
                               (t1->src.start.tv_usec >  t2->src.start.tv_usec))) {
                              t1->src.start = t2->src.start;
                              t1->hdr.subtype |= ARGUS_TIME_SRC_START;
                           } else {
                              if ((t1->src.end.tv_sec  <  t2->src.start.tv_sec) ||
                                 ((t1->src.end.tv_sec  == t2->src.start.tv_sec) &&
                                  (t1->src.end.tv_usec >  t2->src.start.tv_usec))) {
                                 t1->src.end = t2->src.start;
                                 t1->hdr.subtype |= ARGUS_TIME_SRC_END;
                              }
                           }
                        } else {
                           t1->src = t2->src;
                           t1->hdr.subtype |= st2 & (ARGUS_TIME_SRC_START |
                                                     ARGUS_TIME_SRC_END);
                        }
                     }
                     if (st2 & (ARGUS_TIME_SRC_START | ARGUS_TIME_SRC_END)) {
                        if (st1 & (ARGUS_TIME_SRC_START | ARGUS_TIME_SRC_END)) {
                           if (t2->src.end.tv_sec) {
                              if ((t1->src.end.tv_sec  <  t2->src.end.tv_sec) ||
                                 ((t1->src.end.tv_sec  == t2->src.end.tv_sec) &&
                                  (t1->src.end.tv_usec <  t2->src.end.tv_usec))) {
                                 t1->src.end = t2->src.end;
                                 t1->hdr.subtype |= ARGUS_TIME_SRC_END;
                              }
                           } else {
                              if ((t1->src.end.tv_sec  <  t2->src.start.tv_sec) ||
                                 ((t1->src.end.tv_sec  == t2->src.start.tv_sec) &&
                                  (t1->src.end.tv_usec <  t2->src.start.tv_usec))) {
                                 t1->src.end = t2->src.end;
                                 t1->hdr.subtype |= ARGUS_TIME_SRC_END;
                              }
                           }

                        } else {
                           t1->src.end = t2->src.end;
                           t1->hdr.subtype |= st2 & (ARGUS_TIME_SRC_START |
                                                     ARGUS_TIME_SRC_END);
                        }
                     }
                     if (st2 & ARGUS_TIME_DST_START) {
                        if (st1 & ARGUS_TIME_DST_START) {
                           if ((t1->dst.start.tv_sec  >  t2->dst.start.tv_sec) ||
                              ((t1->dst.start.tv_sec  == t2->dst.start.tv_sec) &&
                               (t1->dst.start.tv_usec >  t2->dst.start.tv_usec))) {
                              t1->dst.start = t2->dst.start;
                              t1->hdr.subtype |= ARGUS_TIME_DST_START;
                           }
                        } else {
                           t1->dst = t2->dst;
                           t1->hdr.subtype |= st2 & (ARGUS_TIME_DST_START |
                                                     ARGUS_TIME_DST_END);
                        }
                     }
                     if (st2 & (ARGUS_TIME_DST_START | ARGUS_TIME_DST_END)) {
                        if (st1 & (ARGUS_TIME_DST_END | ARGUS_TIME_DST_END)) {
                           if ((t1->dst.end.tv_sec  <  t2->dst.end.tv_sec) ||
                              ((t1->dst.end.tv_sec  == t2->dst.end.tv_sec) &&
                               (t1->dst.end.tv_usec <  t2->dst.end.tv_usec))) {
                              t1->dst.end = t2->dst.end;
                              t1->hdr.subtype |= ARGUS_TIME_DST_END;
                           }
                        } else {
                           t1->dst = t2->dst;
                           t1->hdr.subtype |= st2 & (ARGUS_TIME_DST_START |
                                                     ARGUS_TIME_DST_END);
                        }
                     }

                  } else {

                     if (t1->src.start.tv_sec == 0) {
                        bcopy ((char *)t2, (char *)t1, sizeof (*t1));
                     } else {
                        if ((t1->src.start.tv_sec  >  t2->src.start.tv_sec) ||
                           ((t1->src.start.tv_sec  == t2->src.start.tv_sec) &&
                            (t1->src.start.tv_usec >  t2->src.start.tv_usec)))
                           t1->src.start = t2->src.start;

                        if ((t1->src.end.tv_sec == 0) || (t1->hdr.subtype == ARGUS_TIME_ABSOLUTE_TIMESTAMP)) {
                           t1->src.end = t1->src.start;
                           t1->hdr.subtype         = ARGUS_TIME_ABSOLUTE_RANGE;
                           t1->hdr.argus_dsrvl8.len = 5;
                        }
                        if ((t2->src.end.tv_sec == 0) || (t2->hdr.subtype == ARGUS_TIME_ABSOLUTE_TIMESTAMP)) {
                           t2->src.end = t2->src.start;
                           t2->hdr.subtype         = ARGUS_TIME_ABSOLUTE_RANGE;
                           t2->hdr.argus_dsrvl8.len = 5;
                        }
                        if ((t1->src.end.tv_sec  <  t2->src.end.tv_sec) ||
                           ((t1->src.end.tv_sec  == t2->src.end.tv_sec) &&
                            (t1->src.end.tv_usec <  t2->src.end.tv_usec)))
                           t1->src.end = t2->src.end;
                     }
                  }
               }
               break;
            }

            case ARGUS_TIME_ADJ_INDEX: {
               break;
            }

/*
   Merging networks objects involve copying and masking
   various protocol specific network structs together.
   First test for the protocols, and if they are the same,
   then merge, if not, just remove the dsrs[] pointers;
*/

            case ARGUS_NETWORK_INDEX: {
               struct ArgusNetworkStruct *n1 = &ns1->canon.net;
               struct ArgusNetworkStruct *n2 = &ns2->canon.net;
               if (ns1->dsrs[ARGUS_NETWORK_INDEX]) {
                  switch (n1->hdr.subtype) {
                     case ARGUS_TCP_INIT: 
                     case ARGUS_TCP_STATUS:
                     case ARGUS_TCP_PERF: {
                        struct ArgusTCPObject *t1 = (struct ArgusTCPObject *)&n1->net_union.tcp;
                        struct ArgusTCPObject *t2 = (struct ArgusTCPObject *)&n2->net_union.tcp;
           
                        t1->status  |= t2->status;
                        t1->state   |= t2->state;
                        t1->options |= t2->options;

                        t1->src.status   |= t2->src.status;
                        t1->src.ack       = t2->src.ack;
                        t1->src.seq       = t2->src.seq;

                        t1->src.winnum   += t2->src.winnum;
                        t1->src.bytes    += t2->src.bytes;
                        t1->src.retrans  += t2->src.retrans;

                        t1->src.ackbytes += t2->src.ackbytes;
                        t1->src.state    |= t2->src.state;
                        t1->src.win       = t2->src.win;
                        t1->src.winbytes  = t2->src.winbytes;
                        t1->src.flags    |= t2->src.flags;

                        t1->dst.status   |= t2->dst.status;
                        t1->dst.winnum   += t2->dst.winnum;
                        t1->dst.bytes    += t2->dst.bytes;
                        t1->dst.retrans  += t2->dst.retrans;
                        t1->dst.ackbytes += t2->dst.ackbytes;
                        t1->dst.state    |= t2->dst.state;
                        t1->dst.win       = t2->dst.win;
                        t1->dst.winbytes  = t2->dst.winbytes;
                        t1->dst.flags    |= t2->dst.flags;

                        if (n1->hdr.subtype != n2->hdr.subtype) {
                           n1->hdr.subtype = ARGUS_TCP_PERF;
                           n1->hdr.argus_dsrvl8.len = (sizeof(*t1) + 3) / 4;
                        }
                        break;
                     }

                     case ARGUS_RTP_FLOW: {
                        struct ArgusRTPObject *r1 = &n1->net_union.rtp;
                        struct ArgusRTPObject *r2 = &n2->net_union.rtp;
                        r1->sdrop += r2->sdrop;
                        r1->ddrop += r2->ddrop;
                        r1->src = r2->src;
                        r1->dst = r2->dst;
                        break;
                     }

                     case ARGUS_ESP_DSR: {
                        struct ArgusESPObject *e1 = &n1->net_union.esp;
                        struct ArgusESPObject *e2 = &n2->net_union.esp;

                        n1->hdr.argus_dsrvl8.qual |= n2->hdr.argus_dsrvl8.qual;

                        e1->lastseq = e2->lastseq;
                        e1->lostseq += e2->lostseq;
                        if (e1->spi != e2->spi) {
                           e1->spi = 0;
                        }
                     }
                  }
               }

               break;
            }

/*
   Merging IP Attribute objects involves
   of rollover any time soon, as we're working with 64 bit
   ints with the canonical DSR.  We should test for rollover
   but lets do that later - cb
*/
            case ARGUS_IPATTR_INDEX: {
               struct ArgusIPAttrStruct *attr1 = (struct ArgusIPAttrStruct *)ns1->dsrs[ARGUS_IPATTR_INDEX];
               struct ArgusIPAttrStruct *attr2 = (struct ArgusIPAttrStruct *)ns2->dsrs[ARGUS_IPATTR_INDEX]; 

               if (attr1 && attr2) {
                  if ((attr1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) &&
                      (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC)) {
                     if (attr1->src.tos != attr2->src.tos)
                        attr1->src.tos = 0;

                     if (attr1->src.ttl != attr2->src.ttl)
                        attr1->src.ttl = 0;

                     attr1->src.options ^= attr2->src.options;

                  } else 
                  if (!(attr1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) &&
                       (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC)) {
                     bcopy ((char *)&attr2->src, (char *)&attr1->src, sizeof(attr1->src));
                     attr1->hdr.argus_dsrvl8.qual |= ARGUS_IPATTR_SRC;
                     if (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS)
                        attr1->hdr.argus_dsrvl8.qual |= ARGUS_IPATTR_SRC_OPTIONS;
                  }

                  if ((attr1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) &&
                      (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST)) {
                     if (attr1->dst.tos != attr2->dst.tos)
                        attr1->dst.tos = 0;

                     if (attr1->dst.ttl != attr2->dst.ttl)
                        attr1->dst.ttl = 0;

                     attr1->dst.options ^= attr2->dst.options;

                  } else
                  if (!(attr1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) &&
                       (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST)) {
                     bcopy ((char *)&attr2->dst, (char *)&attr1->dst, sizeof(attr1->dst));
                     attr1->hdr.argus_dsrvl8.qual |= ARGUS_IPATTR_DST;
                     if (attr2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS)
                        attr1->hdr.argus_dsrvl8.qual |= ARGUS_IPATTR_SRC_OPTIONS;
                  }
               }

               break;
            }
/*
   Merging metrics data  involves accumulating counters.
*/
            case ARGUS_METRIC_INDEX: {
               struct ArgusMetricStruct *m1 = (struct ArgusMetricStruct *) ns1->dsrs[ARGUS_METRIC_INDEX];
               struct ArgusMetricStruct *m2 = (struct ArgusMetricStruct *) ns2->dsrs[ARGUS_METRIC_INDEX];

               if (m1 && m2) {
                  if (m1->hdr.type == 0) {
                     bcopy ((char *) m2, (char *) m1, sizeof (*m1));
                     break;
                  }

                  m1->src.pkts     += m2->src.pkts;
                  m1->src.bytes    += m2->src.bytes;
                  m1->src.appbytes += m2->src.appbytes;
                  m1->dst.pkts     += m2->dst.pkts;
                  m1->dst.bytes    += m2->dst.bytes;
                  m1->dst.appbytes += m2->dst.appbytes;
               }
               break;
            }

/*
   Merging packet size data involves max min comparisons.
*/
            case ARGUS_PSIZE_INDEX: {
               struct ArgusPacketSizeStruct *p1 = (struct ArgusPacketSizeStruct *) ns1->dsrs[ARGUS_PSIZE_INDEX];
               struct ArgusPacketSizeStruct *p2 = (struct ArgusPacketSizeStruct *) ns2->dsrs[ARGUS_PSIZE_INDEX];

               if (p1 && p2) {
                  if (p1->src.psizemax < p2->src.psizemax)
                     p1->src.psizemax = p2->src.psizemax;

                  if (p1->src.psizemin > p2->src.psizemin)
                     p1->src.psizemin = p2->src.psizemin;
               }
               break;
            }

/*
   Merging the aggregation object results in ns1 having
   a valid aggregation object, with updates to the various
   aggregation metrics.  So, ns1 should have a valid agr, 
   as prior merging of the metrics fields will have
   generated it if ns1 did not have an agr.  If ns2 does
   not have an agr, then merge into ns1's agr the values
   for ns2's metrics.  If ns2 does exist, then just merge
   the two agr's.
*/

            case ARGUS_AGR_INDEX: {
               struct ArgusAgrStruct *a1 = (struct ArgusAgrStruct *) ns1->dsrs[ARGUS_AGR_INDEX];
               struct ArgusAgrStruct *a2 = (struct ArgusAgrStruct *) ns2->dsrs[ARGUS_AGR_INDEX];
               struct ArgusAgrStruct databuf, *data = &databuf;
               double ss1 = 0, ss2 = 0, sum1, sum2;
               float scratch;

               if (a1 && a2) {
                  double tvalstd  = 0;
                  double tvalmean = 0;

                  bcopy ((char *)&a1->hdr, (char *)&data->hdr, sizeof(data->hdr));

                  data->count        = a1->count + a2->count;
                  data->act.maxval   = (a1->act.maxval > a2->act.maxval) ? a1->act.maxval : a2->act.maxval;
                  data->act.minval   = (a1->act.minval < a2->act.minval) ? a1->act.minval : a2->act.minval;
                  data->act.n        = a1->act.n + a2->act.n;
                  sum1               = (a1->act.n > 1) ? (a1->act.meanval * a1->act.n) : a1->act.meanval;
                  sum2               = (a2->act.n > 1) ? (a2->act.meanval * a2->act.n) : a2->act.meanval;

                  if (a1->act.n) {
                     tvalstd  = pow(a1->act.stdev, 2.0);
                     tvalmean = pow(a1->act.meanval, 2.0);

                     ss1 = ((a1->act.n - 1) * tvalstd) + (a1->act.n * tvalmean);
                  }
                  if (a2->act.n) {
                     tvalstd  = pow(a2->act.stdev, 2.0);
                     tvalmean = pow(a2->act.meanval, 2.0);
                     ss2 = ((a2->act.n - 1) * tvalstd) + (a2->act.n * tvalmean);
                  }

                  data->act.meanval  = (sum1 + sum2) / data->act.n;
                  tvalmean = pow(data->act.meanval, 2.0);

                  data->act.stdev    = sqrt(fabs((ss1 + ss2) - (data->act.n * tvalmean))/(data->act.n - 1));

                  data->idle.maxval  = (a1->idle.maxval > a2->idle.maxval) ? a1->idle.maxval : a2->idle.maxval;
                  data->idle.minval  = (a1->idle.minval < a2->idle.minval) ? a1->idle.minval : a2->idle.minval;
                  data->idle.n       = a1->idle.n + a2->idle.n;
                  sum1               = (a1->idle.n > 1) ? (a1->idle.meanval * a1->idle.n) : a1->idle.meanval;
                  sum2               = (a2->idle.n > 1) ? (a2->idle.meanval * a2->idle.n) : a2->idle.meanval;
                  if (a1->idle.n) {
                     ss1 = ((a1->idle.n - 1) * pow(a1->idle.stdev, 2.0)) + (a1->idle.n * pow(a1->idle.meanval, 2.0));
                  }
                  if (a2->idle.n) {
                     ss2 = ((a2->idle.n - 1) * pow(a2->idle.stdev, 2.0)) + (a2->idle.n * pow(a2->idle.meanval, 2.0));
                  }
                  data->idle.meanval  = (sum1 + sum2) / data->idle.n;
                  data->idle.stdev    = sqrt (fabs((ss1 + ss2) - (data->idle.n * pow(data->idle.meanval, 2.0)))/(data->idle.n - 1));


                  data->laststartime = ((a1->laststartime.tv_sec  > a2->laststartime.tv_sec) ||
                                       ((a1->laststartime.tv_sec == a2->laststartime.tv_sec) &&
                                        (a1->laststartime.tv_usec > a2->laststartime.tv_usec))) ?
                                         a1->laststartime : a2->laststartime;

                  data->lasttime     = ((a1->lasttime.tv_sec  > a2->lasttime.tv_sec) ||
                                       ((a1->lasttime.tv_sec == a2->lasttime.tv_sec) &&
                                        (a1->lasttime.tv_usec > a2->lasttime.tv_usec))) ?
                                         a1->lasttime : a2->lasttime;
                  bcopy ((char *)data, (char *) a1, sizeof (databuf));
               } else {
                  if (a1 && !(a2)) {
                     double value = na->RaMetricFetchAlgorithm(ns2);
                     a1->count++;

                     if (a1->act.maxval  < value) a1->act.maxval = value;
                     if (a1->act.minval  > value) a1->act.minval = value;
                     scratch             = a1->act.meanval * a1->act.n;
                     scratch            += value;
                     a1->act.n++;
                     a1->act.meanval     = scratch / a1->act.n;
                  } else
                  if (!(a1) && a2) {
                     a1 = &ns1->canon.agr;
                     bcopy ((char *)a2, (char *)a1, sizeof(*a2));
                     ns1->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader *) a1;
                     ns1->dsrindex |= (0x01 << ARGUS_AGR_INDEX);
                  }
               }
               break;
            }
/*
   Merging the jitter object involves both records having
   a valid jitter object.  If they don't just drop the dsr;
*/

            case ARGUS_JITTER_INDEX: {
               struct ArgusJitterStruct *j1 = (struct ArgusJitterStruct *) ns1->dsrs[ARGUS_JITTER_INDEX];
               struct ArgusJitterStruct *j2 = (struct ArgusJitterStruct *) ns2->dsrs[ARGUS_JITTER_INDEX];

               if (j1 && j2) {
                  if (j2->act.src.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->act.src.n + j2->act.src.n);
                     meanval  = (((double)j1->act.src.meanval * (double)j1->act.src.n) +
                                 ((double)j2->act.src.meanval * (double)j2->act.src.n)) / n;

                     if (j1->act.src.n) {
                        double sum  =  (double)j1->act.src.meanval * (double)j1->act.src.n;
                        sumsqrd += (j1->act.src.n * ((double)j1->act.src.stdev * (double)j1->act.src.stdev)) +
                                   (sum * sum)/j1->act.src.n;
                     }

                     if (j2->act.src.n) {
                        double sum  =  (double)j2->act.src.meanval * (double)j2->act.src.n;
                        sumsqrd += (j2->act.src.n * ((double)j2->act.src.stdev * (double)j2->act.src.stdev)) +
                                   (sum * sum)/j2->act.src.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->act.src.n       = n;
                     j1->act.src.meanval = (unsigned int) meanval;
                     j1->act.src.stdev   = stdev;
                     if (j1->act.src.minval > j2->act.src.minval)
                        j1->act.src.minval = j2->act.src.minval;
                     if (j1->act.src.maxval < j2->act.src.maxval)
                        j1->act.src.maxval = j2->act.src.maxval;
                  }

                  if (j2->idle.src.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->idle.src.n + j2->idle.src.n);
                     meanval  = (((double) j1->idle.src.meanval * (double) j1->idle.src.n) +
                                 ((double) j2->idle.src.meanval * (double) j2->idle.src.n)) / n;

                     if (j1->idle.src.n) {
                        double sum  =  (double) j1->idle.src.meanval * (double) j1->idle.src.n;
                        sumsqrd += (j1->idle.src.n * ((double)j1->idle.src.stdev * (double)j1->idle.src.stdev)) +
                                   ((double)sum *(double)sum)/j1->idle.src.n;
                     }

                     if (j2->idle.src.n) {
                        double sum  =  (double) j2->idle.src.meanval * (double) j2->idle.src.n;
                        sumsqrd += (j2->idle.src.n * ((double)j2->idle.src.stdev * (double)j2->idle.src.stdev)) +
                                   ((double)sum *(double)sum)/j2->idle.src.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->idle.src.n       = n;
                     j1->idle.src.meanval = (unsigned int) meanval;
                     j1->idle.src.stdev   = stdev;
                     if (j1->idle.src.minval > j2->idle.src.minval)
                        j1->idle.src.minval = j2->idle.src.minval;
                     if (j1->idle.src.maxval < j2->idle.src.maxval)
                        j1->idle.src.maxval = j2->idle.src.maxval;
                  }

                  if (j2->act.dst.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->act.dst.n + j2->act.dst.n);
                     meanval  = (((double) j1->act.dst.meanval * (double) j1->act.dst.n) +
                                 ((double) j2->act.dst.meanval * (double) j2->act.dst.n)) / n;

                     if (j1->act.dst.n) {
                        double sum  =  j1->act.dst.meanval * j1->act.dst.n;
                        sumsqrd += (j1->act.dst.n * ((double)j1->act.dst.stdev * (double)j1->act.dst.stdev)) +
                                   (sum * sum)/j1->act.dst.n;
                     }

                     if (j2->act.dst.n) {
                        double sum  =  (double) j2->act.dst.meanval * (double) j2->act.dst.n;
                        sumsqrd += (j2->act.dst.n * ((double)j2->act.dst.stdev * (double)j2->act.dst.stdev)) +
                                   ((double)sum *(double)sum)/j2->act.dst.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->act.dst.n       = n;
                     j1->act.dst.meanval = (unsigned int) meanval;
                     j1->act.dst.stdev   = stdev;
                     if (j1->act.dst.minval > j2->act.dst.minval)
                        j1->act.dst.minval = j2->act.dst.minval;
                     if (j1->act.dst.maxval < j2->act.dst.maxval)
                        j1->act.dst.maxval = j2->act.dst.maxval;
                  }

                  if (j2->idle.dst.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->idle.dst.n + j2->idle.dst.n);
                     meanval  = (((double) j1->idle.dst.meanval * (double) j1->idle.dst.n) +
                                 ((double) j2->idle.dst.meanval * (double) j2->idle.dst.n)) / n;

                     if (j1->idle.dst.n) {
                        int sum  =  (double) j1->idle.dst.meanval * (double) j1->idle.dst.n;
                        sumsqrd += (j1->idle.dst.n * ((double)j1->idle.dst.stdev * (double)j1->idle.dst.stdev)) +
                                   ((double)sum *(double)sum)/j1->idle.dst.n;
                     }

                     if (j2->idle.dst.n) {
                        double sum  =  (double) j2->idle.dst.meanval * (double) j2->idle.dst.n;
                        sumsqrd += (j2->idle.dst.n * ((double)j2->idle.dst.stdev * (double)j2->idle.dst.stdev)) +
                                   ((double)sum *(double)sum)/j2->idle.dst.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->idle.dst.n       = n;
                     j1->idle.dst.meanval = (unsigned int) meanval;
                     j1->idle.dst.stdev   = stdev;
                     if (j1->idle.dst.minval > j2->idle.dst.minval)
                        j1->idle.dst.minval = j2->idle.dst.minval;
                     if (j1->idle.dst.maxval < j2->idle.dst.maxval)
                        j1->idle.dst.maxval = j2->idle.dst.maxval;
                  }

               }
               break;
            }

/*
   Merging the user data object involves leaving the ns1 buffer,
   or making the ns2 buffer, ns1's.  Since these are allocated
   objects, make sure you deal with them as such.
*/

            case ARGUS_SRCUSERDATA_INDEX: 
            case ARGUS_DSTUSERDATA_INDEX: {
               struct ArgusDataStruct *d1 = (struct ArgusDataStruct *) ns1->dsrs[i];
               struct ArgusDataStruct *d2 = (struct ArgusDataStruct *) ns2->dsrs[i];

               if (d1 && d2) {
                  int tlen = d1->size - d1->count;
                  tlen = (tlen > d2->count) ? d2->count : tlen;
                  if (tlen > 0) {
                     bcopy(d2->array, &d1->array[d1->count], tlen);
                     d1->count += tlen;
                  }
               } else
               if (!d1 && d2) {
                  struct ArgusDataStruct *t2;
                  int len = (((d2->hdr.type & ARGUS_IMMEDIATE_DATA) ? 1 :
                             ((d2->hdr.subtype & ARGUS_LEN_16BITS)  ? d2->hdr.argus_dsrvl16.len :
                                                                     d2->hdr.argus_dsrvl8.len)));
                  if ((t2 = (struct ArgusDataStruct *) ArgusCalloc((2 + len), 4)) == NULL)
                     ArgusLog (LOG_ERR, "ArgusMergeRecords: ArgusCalloc error %s", strerror(errno));

                  bcopy ((char *)d2, (char *)t2, len * 4);
                  t2->size  = len;
                  ns1->dsrs[i] = (struct ArgusDSRHeader *) t2;
                  ns1->dsrindex |= (0x01 << i);
               }
               break;
            }

/*
   Merging the MAC data object involves comparing the ns1 buffer,
   leaving them if they are equal and blowing away the value if they
   are different.
*/
            case ARGUS_ENCAPS_INDEX: {
               struct ArgusEncapsStruct *e1  = (struct ArgusEncapsStruct *) ns1->dsrs[ARGUS_ENCAPS_INDEX];
               struct ArgusEncapsStruct *e2  = (struct ArgusEncapsStruct *) ns2->dsrs[ARGUS_ENCAPS_INDEX];

               if (e1 && e2) {
                  if (e1->src != e2->src) {
                     e1->hdr.argus_dsrvl8.qual |= ARGUS_SRC_CHANGED;
                     e1->src |= e2->src;
                  }
                  if (e1->dst != e2->dst) {
                     e1->hdr.argus_dsrvl8.qual |= ARGUS_DST_CHANGED;
                     e1->dst |= e2->dst;
                  }
               }
               break;
            }

            case ARGUS_MAC_INDEX: {
               struct ArgusMacStruct *m1 = (struct ArgusMacStruct *) ns1->dsrs[ARGUS_MAC_INDEX];
               struct ArgusMacStruct *m2 = (struct ArgusMacStruct *) ns2->dsrs[ARGUS_MAC_INDEX];

               if (m1 && m2) {
                  struct ether_header *e1 = &m1->mac_union.ether.ehdr;
                  struct ether_header *e2 = &m2->mac_union.ether.ehdr;

                  if (bcmp(&e1->ether_shost, &e2->ether_shost, sizeof(e1->ether_shost)))
                     bzero ((char *)&e1->ether_shost, sizeof(e1->ether_shost));
                 
                  if (bcmp(&e1->ether_dhost, &e2->ether_dhost, sizeof(e1->ether_dhost)))
                     bzero ((char *)&e1->ether_dhost, sizeof(e1->ether_dhost));

               } else {
                  ns1->dsrs[ARGUS_MAC_INDEX] = NULL;
               }
               break;
            }
/*
   Merging the MAC data object involves comparing the ns1 buffer,
   leaving them if they are equal and blowing away the value if they
   are different.
*/
            case ARGUS_VLAN_INDEX:
            case ARGUS_MPLS_INDEX: {
               break;
            }

            case ARGUS_ICMP_INDEX: {
               struct ArgusIcmpStruct *i1 = (struct ArgusIcmpStruct *) ns1->dsrs[ARGUS_ICMP_INDEX];
               struct ArgusIcmpStruct *i2 = (struct ArgusIcmpStruct *) ns2->dsrs[ARGUS_ICMP_INDEX];

               if (!i1 && i2) {
                  i1 = &ns1->canon.icmp;
                  bcopy ((char *)i2, (char *)i1, sizeof(*i1));

                  ns1->dsrs[ARGUS_ICMP_INDEX] = (struct ArgusDSRHeader *) i1;
                  ns1->dsrindex |= (0x01 << ARGUS_ICMP_INDEX);
               }
               break;
            }
         }
      }

      if ((seconds = RaGetFloatDuration(ns1)) > 0) {
         ns1->srate = (float) (ns1->canon.metric.src.pkts * 1.0)/seconds;
         ns1->drate = (float) (ns1->canon.metric.dst.pkts * 1.0)/seconds;
         ns1->sload = (float) (ns1->canon.metric.src.bytes*8 * 1.0)/seconds;
         ns1->dload = (float) (ns1->canon.metric.dst.bytes*8 * 1.0)/seconds;
         ns1->dur   = seconds;
      }
   }

   return;
}


void
ArgusIntersectRecords (struct ArgusAggregatorStruct *na, struct ArgusRecordStruct *ns1, struct ArgusRecordStruct *ns2)
{
   struct ArgusAgrStruct *agr = NULL;
   int i;

   if ((ns1 && ns2) && ((ns1->hdr.type & ARGUS_FAR) && (ns2->hdr.type & ARGUS_FAR))) {
      if ((agr = (struct ArgusAgrStruct *) ns1->dsrs[ARGUS_AGR_INDEX]) == NULL) {
         float dur = RaGetFloatDuration(ns1);
         if ((agr = &ns1->canon.agr) == NULL)
            ArgusLog (LOG_ERR, "ArgusMergeRecords: agr record missing");
         agr->hdr.type            = ARGUS_AGR_DSR;
         agr->hdr.subtype         = 0x01;
         agr->hdr.argus_dsrvl8.qual = 0x01;
         agr->hdr.argus_dsrvl8.len  = (sizeof(*agr) + 3)/4;
         agr->count               = 1;
         agr->act.maxval          = dur;
         agr->act.minval          = dur;
         agr->act.meanval         = dur;
         agr->act.n               = 1;
         ns1->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader *) agr;
         ns1->dsrindex |= (0x01 << ARGUS_AGR_INDEX);
      }

      for (i = 0; i < ARGUSMAXDSRTYPE; i++) {
         switch (i) {
            case ARGUS_FLOW_INDEX: {
/*
   Intersecting Flow records is a matter of testing each field and
   transforming values that are not equal to either run length
   and'ing or zeroing out the value.  When a value is zero'ed
   we need to indicate it in the status field of the flow
   descriptor so that values resulting from merging are not
   confused with values actually off the wire.
 
   run length and'ing is an attempt to preserve CIDR addresses.
   any other value should be either preserved or invalidated.

*/
               struct ArgusFlow *f1 = (struct ArgusFlow *) ns1->dsrs[ARGUS_FLOW_INDEX];
               struct ArgusFlow *f2 = (struct ArgusFlow *) ns2->dsrs[ARGUS_FLOW_INDEX];

               if (f1 && f2) {
                  if ((f1->hdr.subtype == f2->hdr.subtype)) {
                     switch (f1->hdr.subtype & 0x3F) {
                        case ARGUS_FLOW_LAYER_3_MATRIX: {
                           if (f1->hdr.argus_dsrvl8.qual == f2->hdr.argus_dsrvl8.qual) {
                              switch (f1->hdr.argus_dsrvl8.qual & 0x1F) {
                                 case ARGUS_TYPE_IPV4:
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ip_flow.ip_src, &f2->ip_flow.ip_src, ARGUS_TYPE_IPV4, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ip_flow.ip_dst, &f2->ip_flow.ip_dst, ARGUS_TYPE_IPV4, ARGUS_DST);
                                    break;

                                 case ARGUS_TYPE_IPV6:  
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_src[0], &f2->ipv6_flow.ip_src[0], ARGUS_TYPE_IPV6, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_dst[0], &f2->ipv6_flow.ip_dst[0], ARGUS_TYPE_IPV6, ARGUS_DST);
                                    break;
                                 case ARGUS_TYPE_RARP:
                                    if (bcmp(&f1->rarp_flow.shaddr, &f2->rarp_flow.shaddr, 6))
                                       bzero(&f1->rarp_flow.shaddr, 6);
                                    if (bcmp(&f1->rarp_flow.dhaddr, &f2->rarp_flow.dhaddr, 6))
                                       bzero(&f1->rarp_flow.dhaddr, 6);
                                    break;
                                 case ARGUS_TYPE_ARP:
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_spa, &f2->arp_flow.arp_spa, ARGUS_TYPE_ARP, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_tpa, &f2->arp_flow.arp_tpa, ARGUS_TYPE_ARP, ARGUS_DST);
                                    break;
                              }
                           }
                           break;
                        }

                        case ARGUS_FLOW_CLASSIC5TUPLE: {
                              switch (f1->hdr.argus_dsrvl8.qual & 0x1F) {
                                 case ARGUS_TYPE_IPV4:
                                    if (f1->hdr.argus_dsrvl8.qual == f2->hdr.argus_dsrvl8.qual) {
                                       ArgusMergeAddress(&f1->ip_flow.ip_src, &f2->ip_flow.ip_src, ARGUS_TYPE_IPV4, ARGUS_SRC);
                                       ArgusMergeAddress(&f1->ip_flow.ip_dst, &f2->ip_flow.ip_dst, ARGUS_TYPE_IPV4, ARGUS_DST);
                                       if (f1->ip_flow.ip_p  != f2->ip_flow.ip_p)
                                          f1->ip_flow.ip_p = 0;
                                       if (f1->ip_flow.sport != f2->ip_flow.sport)
                                          f1->ip_flow.sport = 0;
                                       if (f1->ip_flow.dport != f2->ip_flow.dport)
                                          f1->ip_flow.dport = 0;

                                    } else {
                                       f1->ip_flow.ip_src = 0;
                                       f1->ip_flow.ip_dst = 0;
                                    
                                       switch (f2->hdr.argus_dsrvl8.qual & 0x1F) {
                                          case ARGUS_TYPE_IPV6:
                                             if (f1->ip_flow.ip_p  != f2->ipv6_flow.ip_p)
                                                f1->ip_flow.ip_p = 0;
                                             if (f1->ip_flow.sport != f2->ipv6_flow.sport)
                                                f1->ip_flow.sport = 0;
                                             if (f1->ip_flow.dport != f2->ipv6_flow.dport)
                                                f1->ip_flow.dport = 0;
                                             break;
              
                                          default:
                                             f1->ip_flow.ip_p = 0;
                                             f1->ip_flow.sport = 0;
                                             f1->ip_flow.dport = 0;
                                             break;
                                       }
                                    }
                                    break;

                                 case ARGUS_TYPE_IPV6:  
                                    if (f1->hdr.argus_dsrvl8.qual == f2->hdr.argus_dsrvl8.qual) {
                                       f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_src[0],
                                               &f2->ipv6_flow.ip_src[0], ARGUS_TYPE_IPV6, ARGUS_SRC);
                                       f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->ipv6_flow.ip_dst[0],
                                               &f2->ipv6_flow.ip_dst[0], ARGUS_TYPE_IPV6, ARGUS_DST);

                                       if (f1->ipv6_flow.ip_p  != f2->ipv6_flow.ip_p)  f1->ipv6_flow.ip_p = 0;
                                       if (f1->ipv6_flow.sport != f2->ipv6_flow.sport) f1->ipv6_flow.sport = 0;
                                       if (f1->ipv6_flow.dport != f2->ipv6_flow.dport) f1->ipv6_flow.dport = 0;

                                    } else {
                                       bzero ((char *)&f1->ipv6_flow.ip_src[0], sizeof(f1->ipv6_flow.ip_src));
                                       bzero ((char *)&f1->ipv6_flow.ip_dst[0], sizeof(f1->ipv6_flow.ip_dst));
                                       if (f1->ipv6_flow.ip_p  != f2->ip_flow.ip_p)  f1->ipv6_flow.ip_p = 0;
                                       if (f1->ipv6_flow.sport != f2->ip_flow.sport) f1->ipv6_flow.sport = 0;
                                       if (f1->ipv6_flow.dport != f2->ip_flow.dport) f1->ipv6_flow.dport = 0;
                                    }
                                    break;

                                 case ARGUS_TYPE_RARP:
                                    if (bcmp(&f1->rarp_flow.shaddr, &f2->rarp_flow.shaddr, 6))
                                       bzero(&f1->rarp_flow.shaddr, 6);
                                    if (bcmp(&f1->rarp_flow.dhaddr, &f2->rarp_flow.dhaddr, 6))
                                       bzero(&f1->rarp_flow.dhaddr, 6);
                                 case ARGUS_TYPE_ARP:
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_spa, &f2->arp_flow.arp_spa, ARGUS_TYPE_ARP, ARGUS_SRC);
                                    f1->hdr.argus_dsrvl8.qual |= ArgusMergeAddress(&f1->arp_flow.arp_tpa, &f2->arp_flow.arp_tpa, ARGUS_TYPE_ARP, ARGUS_DST);
                                    break;
                              }
                              break;
                           }

                           case ARGUS_FLOW_ARP: {
                              break;
                           }
                        }
                     }
                  }
               }
               break;
/*
   Intersecting Transport objects involves simply checking that the source
   id and seqnum are the same, and if not, removing the fields until
   we're actually removing the struct.

struct ArgusTransportStruct {
   struct ArgusDSRHeader hdr;
   struct ArgusAddrStruct srcid;
   unsigned int seqnum;
};

*/
            case ARGUS_TRANSPORT_INDEX: {
               struct ArgusTransportStruct *t1 = (struct ArgusTransportStruct *) ns1->dsrs[ARGUS_TRANSPORT_INDEX];
               struct ArgusTransportStruct *t2 = (struct ArgusTransportStruct *) ns2->dsrs[ARGUS_TRANSPORT_INDEX];

               if ((t1 && t2) && (t1->hdr.argus_dsrvl8.qual == t2->hdr.argus_dsrvl8.qual)) {
                  if (t1->hdr.argus_dsrvl8.qual == t2->hdr.argus_dsrvl8.qual) {
                     if ((t1->hdr.subtype & ARGUS_SRCID) && (t2->hdr.subtype & ARGUS_SRCID)) {
                        switch (t1->hdr.argus_dsrvl8.qual & 0x1F) {
                           case ARGUS_TYPE_INT:
                           case ARGUS_TYPE_IPV4:
                              if (t1->srcid.a_un.ipv4 != t2->srcid.a_un.ipv4) {
                                 ns1->dsrs[ARGUS_TRANSPORT_INDEX] = NULL;
                                 ns1->dsrindex &= ~(0x1 << ARGUS_TRANSPORT_INDEX);
                              }
                              break;

                           case ARGUS_TYPE_IPV6:
                           case ARGUS_TYPE_ETHER:
                           case ARGUS_TYPE_STRING:
                              break;
                        }
                     }

                  } else {
                     ns1->dsrs[ARGUS_TRANSPORT_INDEX] = NULL;
                     ns1->dsrindex &= ~(0x1 << ARGUS_TRANSPORT_INDEX);
                  }
               }

               break;
            }
/*
   Intersecting Time objects may result in a change in the storage
   type of the time structure, from an ABSOLUTE_TIMESTAMP
   to an ABSOLUTE_RANGE, to hold the new ending time.
*/
            case ARGUS_TIME_INDEX: {
               struct ArgusTimeObject *t1 = (struct ArgusTimeObject *) ns1->dsrs[ARGUS_TIME_INDEX];
               struct ArgusTimeObject *t2 = (struct ArgusTimeObject *) ns2->dsrs[ARGUS_TIME_INDEX];

               if (t1 && t2) {
                  if (t1->src.start.tv_sec == 0) {
                     bcopy ((char *)t2, (char *)t1, sizeof (*t1));
                  } else {
                     if ((t1->src.start.tv_sec  >  t2->src.start.tv_sec) ||
                        ((t1->src.start.tv_sec  == t2->src.start.tv_sec) &&
                         (t1->src.start.tv_usec >  t2->src.start.tv_usec)))
                        t1->src.start = t2->src.start;

                     if ((t1->src.end.tv_sec == 0) || (t1->hdr.subtype == ARGUS_TIME_ABSOLUTE_TIMESTAMP)) {
                        t1->src.end = t1->src.start;
                        t1->hdr.subtype         = ARGUS_TIME_ABSOLUTE_RANGE;
                        t1->hdr.argus_dsrvl8.len  = 5;
                     }
                     if ((t2->src.end.tv_sec == 0) || (t2->hdr.subtype == ARGUS_TIME_ABSOLUTE_TIMESTAMP)) {
                        t2->src.end = t2->src.start;
                        t2->hdr.subtype         = ARGUS_TIME_ABSOLUTE_RANGE;
                        t2->hdr.argus_dsrvl8.len  = 5;
                     }
                     if ((t1->src.end.tv_sec  <  t2->src.end.tv_sec) ||
                        ((t1->src.end.tv_sec  == t2->src.end.tv_sec) &&
                         (t1->src.end.tv_usec <  t2->src.end.tv_usec)))
                        t1->src.end = t2->src.end;
                  }
               }
               break;
            }
/*
   Intersecting metric objects should not result in any type
   of rollover any time soon, as we're working with 64 bit
   ints with the canonical DSR.  We should test for rollover
   but lets do that later - cb
*/
            case ARGUS_METRIC_INDEX: {
               struct ArgusMetricStruct *m1 = (struct ArgusMetricStruct *) ns1->dsrs[ARGUS_METRIC_INDEX];
               struct ArgusMetricStruct *m2 = (struct ArgusMetricStruct *) ns2->dsrs[ARGUS_METRIC_INDEX];

               if (m1 && m2) {
                  if (m1->hdr.type == 0) {
                     bcopy ((char *) m2, (char *) m1, sizeof (*m1));
                     break;
                  }

                  m1->src.pkts     -= m2->src.pkts;
                  m1->src.bytes    -= m2->src.bytes;
                  m1->src.appbytes -= m2->src.appbytes;
                  m1->dst.pkts     -= m2->dst.pkts;
                  m1->dst.bytes    -= m2->dst.bytes;
                  m1->dst.appbytes -= m2->dst.appbytes;
               }
               break;
            }

/*
   Intersecting packet size object choses the smaller of the
   two values for psizemax and psizemin. opposite of merge - cb
*/
            case ARGUS_PSIZE_INDEX: {
               struct ArgusPacketSizeStruct *p1 = (struct ArgusPacketSizeStruct *) ns1->dsrs[ARGUS_METRIC_INDEX];
               struct ArgusPacketSizeStruct *p2 = (struct ArgusPacketSizeStruct *) ns2->dsrs[ARGUS_METRIC_INDEX];

               if (p1 && p2) {
                  if (p1->src.psizemax > p2->src.psizemax)
                     p1->src.psizemax = p2->src.psizemax;

                  if (p1->src.psizemin < p2->src.psizemin)
                     p1->src.psizemin = p2->src.psizemin;
               }
               break;
            }
/*
   Intersecting the aggregation object results in ns1 having
   a valid aggregation object, but without the updates
   from this merger.  So, if ns1 and ns2 have valid
   agr's, then just update ns1 fields.  If ns2 has an
   agr, but ns1 does not, then move ns2's agr to ns1.
   Do this by updating the canonical agr struct and
   then putting the pointer into the dsrs[].
   
   if neither, then add one to ns1 with a count of 1.
*/
            case ARGUS_AGR_INDEX: {
               struct ArgusAgrStruct *a1 = (struct ArgusAgrStruct *) ns1->dsrs[ARGUS_AGR_INDEX];
               struct ArgusAgrStruct *a2 = (struct ArgusAgrStruct *) ns2->dsrs[ARGUS_AGR_INDEX];

               if (a1 && a2) {
                  a1->count += a2->count;
               } else {
                  if (!(a1 || a2)) {
                     a1 = &ns1->canon.agr;
                     a1->hdr.type            = ARGUS_AGR_DSR;
                     a1->hdr.subtype         = 0x01;
                     a1->hdr.argus_dsrvl8.qual = 0x01;
                     a1->hdr.argus_dsrvl8.len  = (sizeof(*agr) + 3)/4;
                     a1->count = 1;
                     ns1->dsrs[ARGUS_AGR_INDEX] = (struct ArgusDSRHeader *) a1;
                  } else
                  if (a2) {
                     a1 = &ns1->canon.agr;
                     bcopy((char *)a2, (char *)a1, sizeof(*a1));
                     a1->count++;
                  }
               }
               break;
            }

            case ARGUS_JITTER_INDEX: {
               struct ArgusJitterStruct *j1 = (struct ArgusJitterStruct *) ns1->dsrs[ARGUS_JITTER_INDEX];
               struct ArgusJitterStruct *j2 = (struct ArgusJitterStruct *) ns2->dsrs[ARGUS_JITTER_INDEX];

               if (j1 && j2) {
                  if (j2->act.src.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->act.src.n + j2->act.src.n);
                     meanval  = (((double)j1->act.src.meanval * (double)j1->act.src.n) +
                                 ((double)j2->act.src.meanval * (double)j2->act.src.n)) / n;

                     if (j1->act.src.n) {
                        double sum  =  (double)j1->act.src.meanval * (double)j1->act.src.n;
                        sumsqrd += (j1->act.src.n * ((double)j1->act.src.stdev * (double)j1->act.src.stdev)) +
                                   (sum * sum)/j1->act.src.n;
                     }

                     if (j2->act.src.n) {
                        double sum  =  (double)j2->act.src.meanval * (double)j2->act.src.n;
                        sumsqrd += (j2->act.src.n * ((double)j2->act.src.stdev * (double)j2->act.src.stdev)) +
                                   (sum * sum)/j2->act.src.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->act.src.n       = n;
                     j1->act.src.meanval = (unsigned int) meanval;
                     j1->act.src.stdev   = stdev;
                     if (j1->act.src.minval > j2->act.src.minval)
                        j1->act.src.minval = j2->act.src.minval;
                     if (j1->act.src.maxval < j2->act.src.maxval)
                        j1->act.src.maxval = j2->act.src.maxval;
                  }

                  if (j2->idle.src.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->idle.src.n + j2->idle.src.n);
                     meanval  = (((double) j1->idle.src.meanval * (double) j1->idle.src.n) +
                                 ((double) j2->idle.src.meanval * (double) j2->idle.src.n)) / n;

                     if (j1->idle.src.n) {
                        double sum  =  (double) j1->idle.src.meanval * (double) j1->idle.src.n;
                        sumsqrd += (j1->idle.src.n * ((double)j1->idle.src.stdev * (double)j1->idle.src.stdev)) +
                                   ((double)sum *(double)sum)/j1->idle.src.n;
                     }

                     if (j2->idle.src.n) {
                        double sum  =  (double) j2->idle.src.meanval * (double) j2->idle.src.n;
                        sumsqrd += (j2->idle.src.n * ((double)j2->idle.src.stdev * (double)j2->idle.src.stdev)) +
                                   ((double)sum *(double)sum)/j2->idle.src.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->idle.src.n       = n;
                     j1->idle.src.meanval = (unsigned int) meanval;
                     j1->idle.src.stdev   = stdev;
                     if (j1->idle.src.minval > j2->idle.src.minval)
                        j1->idle.src.minval = j2->idle.src.minval;
                     if (j1->idle.src.maxval < j2->idle.src.maxval)
                        j1->idle.src.maxval = j2->idle.src.maxval;
                  }

                  if (j2->act.dst.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->act.dst.n + j2->act.dst.n);
                     meanval  = (((double) j1->act.dst.meanval * (double) j1->act.dst.n) +
                                 ((double) j2->act.dst.meanval * (double) j2->act.dst.n)) / n;

                     if (j1->act.dst.n) {
                        double sum  =  j1->act.dst.meanval * j1->act.dst.n;
                        sumsqrd += (j1->act.dst.n * ((double)j1->act.dst.stdev * (double)j1->act.dst.stdev)) +
                                   (sum * sum)/j1->act.dst.n;
                     }

                     if (j2->act.dst.n) {
                        double sum  =  (double) j2->act.dst.meanval * (double) j2->act.dst.n;
                        sumsqrd += (j2->act.dst.n * ((double)j2->act.dst.stdev * (double)j2->act.dst.stdev)) +
                                   ((double)sum *(double)sum)/j2->act.dst.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->act.dst.n       = n;
                     j1->act.dst.meanval = (unsigned int) meanval;
                     j1->act.dst.stdev   = stdev;
                     if (j1->act.dst.minval > j2->act.dst.minval)
                        j1->act.dst.minval = j2->act.dst.minval;
                     if (j1->act.dst.maxval < j2->act.dst.maxval)
                        j1->act.dst.maxval = j2->act.dst.maxval;
                  }

                  if (j2->idle.dst.n > 0) {
                     unsigned int n, stdev = 0;
                     double meanval, sumsqrd = 0.0;
                     
                     n = (j1->idle.dst.n + j2->idle.dst.n);
                     meanval  = (((double) j1->idle.dst.meanval * (double) j1->idle.dst.n) +
                                 ((double) j2->idle.dst.meanval * (double) j2->idle.dst.n)) / n;

                     if (j1->idle.dst.n) {
                        int sum  =  (double) j1->idle.dst.meanval * (double) j1->idle.dst.n;
                        sumsqrd += (j1->idle.dst.n * ((double)j1->idle.dst.stdev * (double)j1->idle.dst.stdev)) +
                                   ((double)sum *(double)sum)/j1->idle.dst.n;
                     }

                     if (j2->idle.dst.n) {
                        double sum  =  (double) j2->idle.dst.meanval * (double) j2->idle.dst.n;
                        sumsqrd += (j2->idle.dst.n * ((double)j2->idle.dst.stdev * (double)j2->idle.dst.stdev)) +
                                   ((double)sum *(double)sum)/j2->idle.dst.n;
                     }
                     stdev = (int) sqrt (fabs((sumsqrd/n) - ((double)meanval * (double)meanval)));

                     j1->idle.dst.n       = n;
                     j1->idle.dst.meanval = (unsigned int) meanval;
                     j1->idle.dst.stdev   = stdev;
                     if (j1->idle.dst.minval > j2->idle.dst.minval)
                        j1->idle.dst.minval = j2->idle.dst.minval;
                     if (j1->idle.dst.maxval < j2->idle.dst.maxval)
                        j1->idle.dst.maxval = j2->idle.dst.maxval;
                  }

               } else
                  ns1->dsrs[ARGUS_JITTER_INDEX] = NULL;

               break;
            }

            case ARGUS_MAC_INDEX: {
               struct ArgusMacStruct *m1 = (struct ArgusMacStruct *) ns1->dsrs[ARGUS_MAC_INDEX];
               struct ArgusMacStruct *m2 = (struct ArgusMacStruct *) ns2->dsrs[ARGUS_MAC_INDEX];

               if (m1 && m2) {
                  struct ether_header *e1 = &m1->mac_union.ether.ehdr;
                  struct ether_header *e2 = &m2->mac_union.ether.ehdr;

                  if (bcmp(&e1->ether_shost, &e2->ether_shost, sizeof(e1->ether_shost)))
                     bzero ((char *)&e1->ether_shost, sizeof(e1->ether_shost));
                 
                  if (bcmp(&e1->ether_dhost, &e2->ether_dhost, sizeof(e1->ether_dhost)))
                     bzero ((char *)&e1->ether_dhost, sizeof(e1->ether_dhost));

               } else {
                  ns1->dsrs[ARGUS_MAC_INDEX] = NULL;
               }
               break;
            }
         }
      }
   }

   return;
}


#include <math.h>

#define RATOPSTARTINGINDEX     0

void
ArgusCalculatePeriod (struct ArgusRecordStruct *ns, struct ArgusAdjustStruct *nadp, float duration)
{
   if (nadp->tperiod == 0.0) {
      nadp->tperiod = duration / (nadp->size * 1.0);
      /* simple linear model */
      if (nadp->tperiod < 1.0) nadp->tperiod = 1.0;
      nadp->spkts     = (ns->canon.metric.src.pkts     / nadp->tperiod);
      nadp->sbytes    = (ns->canon.metric.src.bytes    / nadp->tperiod);
      nadp->sappbytes = (ns->canon.metric.src.appbytes / nadp->tperiod);

      nadp->dpkts     = (ns->canon.metric.dst.pkts     / nadp->tperiod);
      nadp->dbytes    = (ns->canon.metric.dst.bytes    / nadp->tperiod);
      nadp->dappbytes = (ns->canon.metric.dst.appbytes / nadp->tperiod);
   }

/*
   if (!(nadp->spkts || nadp->sbytes || nadp->sappbytes)) {
      if ((tmp = ns->canon.metric.src.pkts) > 0) {
         nadp->spkts = duration / (float) tmp * 1.0;
         nadp->speriod = duration / (float) tmp * 1.0;
      } else
         nadp->speriod = 0.0;
   }

   if (!(nadp->dperiod)) {
      if ((tmp = ns->canon.metric.dst.pkts) > 0) {
         nadp->dperiod = duration / (float) tmp * 1.0;
      } else
         nadp->dperiod = 0.0;
   }

   if ((tperiod = (nadp->speriod > nadp->dperiod) ? nadp->speriod : nadp->dperiod) > 0.0) {
      fusecs = modf (tperiod, &secs);
      timevalbuf.tv_sec  = secs;
      timevalbuf.tv_usec = fusecs * 1000000;

   } else {
      timevalbuf.tv_sec  = 0;
      timevalbuf.tv_usec = 0;
   }

   return(&timevalbuf);
*/
}


struct ArgusRecordStruct *ArgusAlignRecord(struct ArgusParserStruct *parser, struct ArgusRecordStruct *, struct ArgusAdjustStruct *);

/*
   ArgusAlignRecord is designed to snip records on hard time boundaries.
   What those time boundary's are, are specified in the ArgusAdjustStruct.
   The idea is that one specifies a number of seconds, and we try to find
   a time boundary to snip the records to.  If the bin size is 60s, we
   will clip on 1 minute boundaries.  

   A problem arises, when the boundary does not coincide with minute boundaries,
   as we have to decide where the starting boundary is, and then clip accordingly.
   So, as an example, if we are aligning on 90s boundaries (1.5m).  What is
   the correct starting point for the clipping?  Can't be on the hour, as 90s
   boundaries don't conincide with hourly boundaries.

   As a convention, if the boundary does not align well on minutes, we shall
   start on the yearly boundary, and adjust accordingly, so we find our
   current time, and adjust it so that the alignment bin is aligned with the
   beginning of the year.

*/


struct ArgusRecordStruct *
ArgusAlignRecord(struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns, struct ArgusAdjustStruct *nadp)
{
   struct ArgusRecordStruct *retn = NULL;

   if ((ns->canon.metric.src.pkts + ns->canon.metric.dst.pkts) > 0) {
      time_t tsec = ns->canon.time.src.start.tv_sec;
      localtime_r(&tsec, &nadp->RaStartTmStruct);

      if (!(nadp->modify)) {
         retn =  ArgusCopyRecordStruct(ns);
         ns->canon.metric.src.pkts = 0;
         ns->canon.metric.dst.pkts = 0;

      } else {
         struct timeval *start = &nadp->start;
         struct timeval *end = &nadp->end;
         float startsec = 0.0, duration = 0.0, thisduration = 0.0;
         time_t startSecs, endSecs, dsecs;
         int count = 0, bytes = 0;
         time_t tsec;

         nadp->turns++;

         duration = RaGetFloatDuration(ns);

         if ((parser->startime_t != 0) && (parser->startime_t != 0x7FFFFFFF)) {
            if (ns->canon.time.src.start.tv_sec <  parser->startime_t) {
               float ratio;
               thisduration  = ((parser->startime_t - ns->canon.time.src.start.tv_sec) * 1.0) +
                                (1.0 - (ns->canon.time.src.start.tv_usec / 1000000.0));
               ratio = thisduration/duration;
               ns->canon.time.src.start.tv_sec  = parser->startime_t;
               ns->canon.time.src.start.tv_usec = 0;
               ns->canon.metric.src.pkts  *= ratio;
               ns->canon.metric.dst.pkts  *= ratio;
               ns->canon.metric.src.bytes *= ratio;
               ns->canon.metric.dst.bytes *= ratio;
               ns->canon.metric.src.appbytes *= ratio;
               ns->canon.metric.dst.appbytes *= ratio;
               duration = RaGetFloatDuration(ns);
            }
            if (ns->canon.time.src.end.tv_sec >  parser->lasttime_t) {
               float ratio;
               ns->canon.time.src.end.tv_sec  = parser->lasttime_t;
               ns->canon.time.src.end.tv_usec = 0;
               thisduration = RaGetFloatDuration(ns);
               ratio = thisduration/duration;
               ns->canon.metric.src.pkts  *= ratio;
               ns->canon.metric.dst.pkts  *= ratio;
               ns->canon.metric.src.bytes *= ratio;
               ns->canon.metric.dst.bytes *= ratio;
               ns->canon.metric.src.appbytes *= ratio;
               ns->canon.metric.dst.appbytes *= ratio;
               duration = RaGetFloatDuration(ns);
            }
         }

         start->tv_sec  = ns->canon.time.src.start.tv_sec;
         start->tv_usec = ns->canon.time.src.start.tv_usec;

         end->tv_sec    = ns->canon.time.src.end.tv_sec;
         end->tv_usec   = ns->canon.time.src.end.tv_usec;

         startsec = (float)(start->tv_sec/1.0) + (float)(start->tv_usec/1000000.0);

         tsec = start->tv_sec;
         localtime_r(&tsec, &nadp->RaStartTmStruct);

         nadp->RaStartTmStruct.tm_sec = 0;
         nadp->RaStartTmStruct.tm_min = 0;
         nadp->RaStartTmStruct.tm_hour = 0;
         nadp->RaStartTmStruct.tm_mday = 1;
         nadp->RaStartTmStruct.tm_mon = 0;

         if (nadp->startSecs == 0) {
            nadp->startSecs = mktime(&nadp->RaStartTmStruct);

            switch (nadp->qual) {
               case ARGUSSPLITSECOND: nadp->size = nadp->value; break;
               case ARGUSSPLITMINUTE: nadp->size = nadp->value*60; break;
               case ARGUSSPLITHOUR:   nadp->size = nadp->value*3600; break;
               case ARGUSSPLITDAY:    nadp->size = nadp->value*3600*24; break;   
               case ARGUSSPLITWEEK:   nadp->size = nadp->value*3600*24*7; break;
               case ARGUSSPLITMONTH:  nadp->size = nadp->value*3600*24*7*4; break;
               case ARGUSSPLITYEAR:   nadp->size = nadp->value*3600*24*7*52; break;
            }
         }

         if (nadp->size) {
            dsecs = ((start->tv_sec - nadp->startSecs) / nadp->size) * nadp->size;
            startSecs = nadp->startSecs + dsecs;
            localtime_r(&startSecs, &nadp->RaStartTmStruct);
            endSecs = startSecs + nadp->size;
            localtime_r(&endSecs, &nadp->RaEndTmStruct);

            if ((retn = ArgusCopyRecordStruct (ns)) == NULL)
               return(retn);

            if ((duration > 0.0) && ((end->tv_sec > endSecs) ||
                                    ((end->tv_sec == endSecs) && end->tv_usec > 0))) {
               if ((count = (retn->canon.metric.src.pkts + retn->canon.metric.dst.pkts)) == 2) {
                  startsec = endSecs;            

                  retn->canon.time.src.end.tv_sec  = endSecs;
                  retn->canon.time.src.end.tv_usec = 0;

                  ns->canon.time.src.start = retn->canon.time.src.end;

                  if (retn->canon.metric.src.pkts) {
                     if (retn->canon.metric.src.pkts > 1) {
                        retn->canon.metric.src.pkts = 1;
                        bytes = retn->canon.metric.src.bytes;
                        retn->canon.metric.src.bytes /= 2;
                        if (bytes & 0x01)
                           retn->canon.metric.src.bytes += 1;
                        bytes = retn->canon.metric.src.appbytes;
                        retn->canon.metric.src.appbytes /= 2;
                        if (bytes & 0x01)
                           retn->canon.metric.src.appbytes += 1;
                     } else {
                        retn->canon.metric.dst.pkts  = 0;
                        retn->canon.metric.dst.bytes = 0;
                        retn->canon.metric.dst.appbytes = 0;
                     }

                  } else {
                     retn->canon.metric.dst.pkts = 1;
                     bytes = retn->canon.metric.dst.bytes;
                     retn->canon.metric.dst.bytes /= 2;
                     if (bytes & 0x01)
                        retn->canon.metric.dst.bytes += 1;
                     bytes = retn->canon.metric.dst.appbytes;
                     retn->canon.metric.dst.appbytes /= 2;
                     if (bytes & 0x01)
                        retn->canon.metric.dst.appbytes += 1;
                  }

                  ns->canon.metric.src.pkts     -= retn->canon.metric.src.pkts;
                  ns->canon.metric.src.bytes    -= retn->canon.metric.src.bytes;
                  ns->canon.metric.src.appbytes -= retn->canon.metric.src.appbytes;
                  ns->canon.metric.dst.pkts     -= retn->canon.metric.dst.pkts;
                  ns->canon.metric.dst.bytes    -= retn->canon.metric.dst.bytes;
                  ns->canon.metric.dst.appbytes -= retn->canon.metric.dst.appbytes;

               } else {
                  startsec = endSecs;

                  retn->canon.time.src.end.tv_sec    = endSecs;
                  retn->canon.time.src.end.tv_usec   = 0;

                  thisduration = RaGetFloatDuration(retn);

                  if (nadp->tperiod == 0.0) {
                     ArgusCalculatePeriod (ns, nadp, duration);
                     nadp->scpkts = 0.0;
                     nadp->dcpkts = 0.0;
                  }

                  ns->canon.time.src.start = retn->canon.time.src.end;

                  if (retn->canon.metric.src.pkts) {
                     int thisBytes = 0, thisAppBytes = 0, thisCount = 0;
                     float iptr, fptr, ratio;
                     float pkts = ((nadp->spkts + nadp->scpkts) * thisduration)/nadp->size;
                     /* add carry from last accumluation */
                     fptr = modff(pkts, &iptr);
                     
                     thisCount = iptr;

                     if (thisCount > retn->canon.metric.src.pkts)
                        thisCount = retn->canon.metric.src.pkts;

                     if ((nadp->turns == 1) && (thisCount == 0))
                        thisCount = 1;

                     if (thisCount == 0) {
                        if (nadp->turns == 1) {
                           thisCount = 1;
                           nadp->scpkts += (thisCount * 1.0) - nadp->spkts;
                        } else {
                           nadp->scpkts += nadp->spkts;
                        }

                     } else 
                        nadp->scpkts += ((nadp->spkts * thisduration)/nadp->size) - (thisCount * 1.0);

                     ratio        = ((thisCount * 1.0)/nadp->spkts);
                     thisBytes    = nadp->sbytes    * ratio;
                     thisAppBytes = nadp->sappbytes * ratio;

                     retn->canon.metric.src.pkts     = thisCount;
                     retn->canon.metric.src.bytes    = thisBytes;
                     retn->canon.metric.src.appbytes = thisAppBytes;
                  }

                  if (retn->canon.metric.dst.pkts) {
                     int thisBytes = 0, thisAppBytes = 0, thisCount = 0;
                     float iptr, fptr, ratio;
                     float pkts = ((nadp->dpkts + nadp->dcpkts) * thisduration)/nadp->size;
                     /* add carry from last accumluation */
                     fptr = modff(pkts, &iptr);

                     thisCount = iptr;

                     if (thisCount > retn->canon.metric.dst.pkts)
                        thisCount = retn->canon.metric.dst.pkts;

                     if ((nadp->turns == 1) && (thisCount == 0))
                        thisCount = 1;

                     if (thisCount == 0) {
                        if (nadp->turns == 1) {
                           thisCount = 1;
                           nadp->dcpkts += (thisCount * 1.0) - nadp->dpkts;
                        } else {
                           nadp->dcpkts += nadp->dpkts;
                        }
                     } else
                        nadp->dcpkts += ((nadp->dpkts * thisduration)/nadp->size) - (thisCount * 1.0);

                     ratio        = ((thisCount * 1.0)/nadp->dpkts);
                     thisBytes    = nadp->dbytes * ratio;
                     thisAppBytes = nadp->dappbytes * ratio;

                     retn->canon.metric.dst.pkts     = thisCount;
                     retn->canon.metric.dst.bytes    = thisBytes;
                     retn->canon.metric.dst.appbytes = thisAppBytes;
                  }

                  ns->canon.metric.src.pkts  -= retn->canon.metric.src.pkts;
                  ns->canon.metric.dst.pkts  -= retn->canon.metric.dst.pkts;
                  ns->canon.metric.src.bytes -= retn->canon.metric.src.bytes;
                  ns->canon.metric.dst.bytes -= retn->canon.metric.dst.bytes;
                  ns->canon.metric.src.appbytes -= retn->canon.metric.src.appbytes;
                  ns->canon.metric.dst.appbytes -= retn->canon.metric.dst.appbytes;

                  if ((ns->canon.metric.src.pkts == 0) && (ns->canon.metric.src.bytes > 0)) {
                     if (retn->canon.metric.src.pkts > 1) {
                        retn->canon.metric.src.pkts--;
                        ns->canon.metric.src.pkts++;
                     } else {
                        retn->canon.metric.src.bytes += ns->canon.metric.src.bytes;
                        retn->canon.metric.src.appbytes += ns->canon.metric.src.appbytes;
                     }
                  }

                  if ((ns->canon.metric.dst.pkts == 0) && (ns->canon.metric.dst.bytes > 0)) {
                     if (retn->canon.metric.dst.pkts > 1) {
                        retn->canon.metric.dst.pkts--;
                        ns->canon.metric.dst.pkts++;
                     } else {
                        retn->canon.metric.dst.bytes += ns->canon.metric.dst.bytes;
                        retn->canon.metric.dst.appbytes += ns->canon.metric.dst.appbytes;
                     }
                  }
               }

               end->tv_sec    = retn->canon.time.src.end.tv_sec;
               end->tv_usec   = retn->canon.time.src.end.tv_usec;


            } else {
               ns->canon.metric.src.pkts = 0;
               ns->canon.metric.dst.pkts = 0;
            }

            if (nadp->soft) {
               retn->canon.time.src.start.tv_sec  = startSecs;
               retn->canon.time.src.start.tv_usec = 0;
               retn->canon.time.src.end.tv_sec    = endSecs;
               retn->canon.time.src.end.tv_usec   = 0;
            }

            retn->canon.time.hdr.subtype          = ARGUS_TIME_ABSOLUTE_RANGE;
            retn->canon.time.hdr.argus_dsrvl8.len = 5; 
         }
      }

   } else
      nadp->turns = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusAlignRecord () returning 0x%x\n", retn); 
#endif
   return(retn);
}


/*
   ArgusInsertRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
      This routine takes an ArgusRecordStruct and inserts it into the current
      array structure that is held in the RaBinProcessStruct *RaBinProcess.

      If the structure has not been initialized, this routines initializes
      the RaBinProcess by allocating an array of sufficient size in order
      to accomodate a number of insertions, ARGUSMINARRAYSIZE.  If the mode
      is ARGUSSPLITTIME, then the start value will be the start time in seconds
      of the first record seen.  If any adjustments need to be made, they
      need to be done prior to calling ArgusInsertRecord().

      The array strategy is to assume that records will not come in order,
      so we'll allocate a number of slots for a negative index insertion.
      If the array is not large enough, we'll allocate more space as we go.

struct RaBinStruct {
   int status, timeout; 
   int value, size;
   struct ArgusQueueStruct *queue;
   struct ArgusHashTable hashtable;
};
 
struct RaBinProcessStruct {
   unsigned int status, start, end, size;
   int arraylen, len, count, index;  
   struct RaBinStruct **array;
 
   struct ArgusAdjustStruct *nadp;
};

*/


/*
   The concept here is to insert a record into an array, based on an aggregation
   strategy.  There are two strategies, one where a bin is defined by mode
   specifiers on the command line, and the other is where there is no
   definition, and you want to insert the record into a one second bin
   specified by the starting seconds in the record.

   So the issues are to create the array using the rbps to provide hints.
   Indicate the starting and ending points for the array, and then insert
   the record so that it is in the 'bin' it belongs.
   
   Depending on the mode of operation, use either the rbps->nadp to tell
   us which time bin were in (nadp.RaStartTmStruct and nadp.RaEndTmStruct),
   or use the ns->canon.time.src.start.tv_sec to determine the bin.

   Usually the TmStructs do the right thing, as the program has called
   routines like ArgusAlignRecord() which sets up the TmStructs to
   be the bounding regions for the bin the record should go in.

*/


#define ARGUSMINARRAYSIZE		0x10000
void ArgusShiftArray (struct ArgusParserStruct *, struct RaBinProcessStruct *);

void
ArgusShiftArray (struct ArgusParserStruct *parser, struct RaBinProcessStruct *rbps)
{
   struct RaBinStruct *bin = rbps->array[0];
   int i /*, size */;

   if (bin != NULL) {
/*
      if (rbps->array[rbps->arraylen] == NULL) {
         struct ArgusRecordStruct *ns = NULL;
         if (bin->queue != NULL) {
            if ((ns = (struct ArgusRecordStruct *) bin->queue->start) != NULL) {
               unsigned int stime = rbps->start + ((rbps->arraylen) * rbps->size);
               ArgusZeroRecord(ns);
               ns->canon.time.src.start.tv_sec = stime;
               ns->canon.time.src.end.tv_sec   = stime + rbps->size;
               if ((rbps->array[rbps->arraylen] = RaNewBin(rbps, ns, stime, rbps->index)) == NULL)
                  ArgusLog (LOG_ERR, "ArgusInsertRecord: RaNewBin error %s", strerror(errno));
            }
         }
      }
*/
      RaDeleteBin(parser, (struct RaBinStruct *) bin);
   }
 
   for (i = 0; i < rbps->max; i++)
      rbps->array[i] = rbps->array[i + 1];

   rbps->array[rbps->max] = NULL;

   if (rbps->array[rbps->index] != NULL)
      rbps->startpt.tv_sec = rbps->array[rbps->index]->stime.tv_sec;
   else
      rbps->startpt.tv_sec +=  rbps->size;

   rbps->start  += rbps->size;
   rbps->end    += rbps->size;

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusShiftArray (0x%x, 0x%x) shifted array 1 slot\n", parser, rbps); 
#endif
}

int
ArgusInsertRecord (struct ArgusParserStruct *parser, struct RaBinProcessStruct *rbps, struct ArgusRecordStruct *ns, int offset)
{
   struct RaBinStruct *bin = NULL;
   int val = 0, ind = 0;
   int retn = -1;

   if (rbps && ns) {
      if (rbps->array == NULL) {
         rbps->arraylen = rbps->nadp.count + offset;
         rbps->len      = ARGUSMINARRAYSIZE;
         rbps->index    = offset;
         rbps->max      = 0;

         if ((rbps->array = (struct RaBinStruct **) ArgusCalloc(sizeof(void *), rbps->len)) == NULL)
            ArgusLog (LOG_ERR, "ArgusInsertRecord: ArgusCalloc error %s", strerror(errno));
      }

      if (rbps->startpt.tv_sec == 0) {
         rbps->startpt.tv_sec = mktime(&rbps->nadp.RaStartTmStruct);
         rbps->endpt.tv_sec = rbps->startpt.tv_sec + (rbps->nadp.count * rbps->nadp.size);

#ifdef ARGUSDEBUG
         ArgusDebug (3, "ArgusInsertRecord (0x%x, 0x%x) initializing array\n", rbps, ns); 
#endif
      }

      if (rbps->start == 0) {
         rbps->start = rbps->startpt.tv_sec;
         rbps->end   = rbps->endpt.tv_sec;
      }

/*  
    set the current records value and index for insertion into array.
*/
   
      switch (rbps->nadp.mode) {
         default:
            val = ns->canon.time.src.start.tv_sec;
            break;

         case ARGUSSPLITRATE:
         case ARGUSSPLITTIME:
            val = ns->canon.time.src.start.tv_sec;
            break;

         case ARGUSSPLITSIZE:
         case ARGUSSPLITCOUNT:
            val = rbps->start + (rbps->size * (rbps->index - RATOPSTARTINGINDEX));
            break;
      }

/* using val, calculate the index offset for this record */

      switch (rbps->nadp.mode) {
         default:
         case ARGUSSPLITRATE:
         case ARGUSSPLITTIME:
         case ARGUSSPLITSIZE:
            ind = rbps->index + ((val - rbps->start) / rbps->size);
            break;
         case ARGUSSPLITCOUNT:
            if (!((rbps->count++ + 1) % rbps->size))
               rbps->index++;
            ind = rbps->index + ((val - rbps->start) / rbps->size);
            break;
      }

      if (ind < 0) {
#ifdef ARGUSDEBUG
         ArgusDebug (2, "ArgusInsertRecord (0x%x, 0x%x) array too short ind %d index %d", rbps, ns, ind, rbps->index); 
#endif
         return (retn);
      }

/*
   here is where we do a lot of array and queue management.  we want
   to get the rbps->array set up right, as it is our time series
   buffer, so we want to shift the record so that it represents
   now, and however many seconds back is needed to cover the period
   of the "-M rate %d:%d[smhdwMy]" needed.

   So in ArgusProcessQueue(), we will manage the arrays but that is
   in a periodic fashion.  Here, we have data, and so it is also
   a driver for correcting the array series.  We need to shift array
   members, and delete bins to get the current time correct.

   We could use this as a hint to correct the array to current time,
   but because ArgusGlobalTime and ArgusLastTime are not necessarily
   in sync, go ahead and let the actual data drive the array corrections,
   and let ArgusProcessQueue, do its thing to get the array aligned with
   ArgusGlobalTime.
*/

/*
   at this point we're ready to add the record to the array.
   test if the array has a bin struct, if not add one, then
   add the record to the bin struct, based on the split mode.
*/
      if (ind >= rbps->len) {
#ifdef ARGUSDEBUG
         ArgusDebug (2, "ArgusInsertRecord (0x%x, 0x%x, 0x%x) ind %d greater than arraylen %d\n", parser, rbps, ns, ind, rbps->arraylen); 
#endif
         return(retn);
      }

      if ((bin = rbps->array[ind]) == NULL) {
         if (ind > rbps->max)
            rbps->max = ind;

         if ((rbps->array[ind] = RaNewBin(parser, rbps, ns, (rbps->start + (ind * rbps->size)), RATOPSTARTINGINDEX)) == NULL) 
            ArgusLog (LOG_ERR, "ArgusInsertRecord: RaNewBin error %s", strerror(errno));

         if (rbps->end < rbps->array[ind]->stime.tv_sec)
            rbps->end = rbps->array[ind]->stime.tv_sec;

      } else {
         if (parser->ArgusAggregator != NULL) {
            struct ArgusHashStruct *hstruct = NULL;
            struct ArgusRecordStruct *tns = NULL;

            if ((hstruct = ArgusGenerateHashStruct(parser->ArgusAggregator, ns, NULL)) == NULL)
               ArgusLog (LOG_ERR, "ArgusInsertRecord: ArgusGenerateHashStruct error %s", strerror(errno));

            if (((tns = ArgusFindRecord(bin->htable, hstruct)) != NULL) && (ArgusParser->RaCumulativeMerge)) {
               ArgusMergeRecords (parser->ArgusAggregator, tns, ns);

            } else {
               ArgusAddHashEntry (bin->htable, ns, hstruct);
               ArgusAddToQueue (bin->queue, &ns->qhdr, ARGUS_LOCK);
               retn = 1;
            }

         } else {
            ArgusAddToQueue (bin->queue, &ns->qhdr, ARGUS_LOCK);
            retn = 1;
         }
      }

      rbps->scalesecs = rbps->endpt.tv_sec - rbps->startpt.tv_sec;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusInsertRecord (0x%x, 0x%x, 0x%x, %d) ind %d start %d end %d", parser, rbps, ns, offset,
                    ind, rbps->start, rbps->end); 
#endif
   return (retn);
}


void ArgusInitAggregatorStructs(struct ArgusAggregatorStruct *);

void
ArgusInitAggregatorStructs(struct ArgusAggregatorStruct *nag)
{
   struct ArgusCanonRecord argusbuf, *argus = &argusbuf;
   int i;

   for (i = 0; i < ARGUS_MAX_MASK_LIST; i++) {
      switch (i) {
         case ARGUS_MASK_SRCID:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->trans.srcid - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->trans.srcid - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->trans.srcid - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->trans.srcid - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->trans.srcid - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->trans.srcid - (char *)argus);
            break;

         case ARGUS_MASK_SMPLS:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->mpls.slabel - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->mpls.dlabel - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->mpls.slabel - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->mpls.dlabel - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->mpls.slabel - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->mpls.dlabel - (char *)argus);
            break;
         case ARGUS_MASK_DMPLS:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->mpls.dlabel - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->mpls.slabel - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->mpls.dlabel - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->mpls.slabel - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->mpls.dlabel - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->mpls.slabel - (char *)argus);
            break;

         case ARGUS_MASK_SVLAN:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->vlan.sid - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->vlan.did - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->vlan.sid - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->vlan.did - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->vlan.sid - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->vlan.did - (char *)argus);
            break;
         case ARGUS_MASK_DVLAN:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->vlan.did - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->vlan.sid - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->vlan.did - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->vlan.sid - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->vlan.did - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->vlan.sid - (char *)argus);
            break;

         case ARGUS_MASK_PROTO:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->flow.ip_flow.ip_p - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->flow.ip_flow.ip_p - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = 43;
            ArgusIpV6RevMaskDefs[i].offset  = 43;
            ArgusEtherMaskDefs[i].len       = 2;
            ArgusEtherRevMaskDefs[i].len    = 2;
            break;

         case ARGUS_MASK_SNET:
         case ARGUS_MASK_SADDR:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->flow.ip_flow.ip_src - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->flow.ip_flow.ip_dst - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->flow.ipv6_flow.ip_src - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->flow.ipv6_flow.ip_dst - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->flow.mac_flow.ehdr.ether_shost - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->flow.mac_flow.ehdr.ether_dhost - (char *)argus);
            break;

         case ARGUS_MASK_DNET:
         case ARGUS_MASK_DADDR:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->flow.ip_flow.ip_dst - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->flow.ip_flow.ip_src - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->flow.ipv6_flow.ip_dst - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->flow.ipv6_flow.ip_src - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->flow.mac_flow.ehdr.ether_dhost - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->flow.mac_flow.ehdr.ether_shost - (char *)argus);
            break;

         case ARGUS_MASK_SPORT:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->flow.ip_flow.sport - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->flow.ip_flow.dport - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->flow.ipv6_flow.sport - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->flow.ipv6_flow.dport - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->flow.mac_flow.ssap - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->flow.mac_flow.dsap - (char *)argus);
            break;

         case ARGUS_MASK_DPORT:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->flow.ip_flow.dport - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->flow.ip_flow.sport - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->flow.ipv6_flow.dport - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->flow.ipv6_flow.sport - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->flow.mac_flow.dsap - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->flow.mac_flow.ssap - (char *)argus);
            break;

         case ARGUS_MASK_STOS:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->attr.src.tos - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->attr.dst.tos - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->attr.src.tos - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->attr.dst.tos - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 1;
            ArgusEtherRevMaskDefs[i].len    = 1;
            break;

         case ARGUS_MASK_DTOS:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->attr.dst.tos - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->attr.src.tos - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->attr.dst.tos - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->attr.src.tos - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 1;
            ArgusEtherRevMaskDefs[i].len    = 1;
            break;
         case ARGUS_MASK_STTL:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->attr.src.ttl - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->attr.dst.ttl - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->attr.src.ttl - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->attr.dst.ttl - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 1;
            ArgusEtherRevMaskDefs[i].len    = 1;
            break;
         case ARGUS_MASK_DTTL:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->attr.dst.ttl - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->attr.src.ttl - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->attr.dst.ttl - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->attr.src.ttl - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 1;
            ArgusEtherRevMaskDefs[i].len    = 1;
            break;
         case ARGUS_MASK_STCPB:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->net.net_union.tcp.src.seqbase - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->net.net_union.tcp.dst.seqbase - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->net.net_union.tcp.src.seqbase - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->net.net_union.tcp.dst.seqbase - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 4;
            ArgusEtherRevMaskDefs[i].len    = 4;
            break;
         case ARGUS_MASK_DTCPB:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->net.net_union.tcp.dst.seqbase - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->net.net_union.tcp.src.seqbase - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->net.net_union.tcp.dst.seqbase - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->net.net_union.tcp.src.seqbase - (char *)argus);
            ArgusEtherMaskDefs[i].len       = 4;
            ArgusEtherRevMaskDefs[i].len    = 4;
            break;
         case ARGUS_MASK_SMAC:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            break;
         case ARGUS_MASK_DMAC:
            ArgusIpV4MaskDefs[i].offset     = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            ArgusIpV4RevMaskDefs[i].offset  = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            ArgusIpV6MaskDefs[i].offset     = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            ArgusIpV6RevMaskDefs[i].offset  = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            ArgusEtherMaskDefs[i].offset    = ((char *)&argus->mac.mac_union.ether.ehdr.ether_dhost - (char *)argus);
            ArgusEtherRevMaskDefs[i].offset = ((char *)&argus->mac.mac_union.ether.ehdr.ether_shost - (char *)argus);
            break;

         case ARGUS_MASK_ICMP: {
            int offset = ((char *)&argus->icmp.osrcaddr - (char *)argus);
            ArgusIpV4MaskDefs[i].offset     = offset;
            ArgusIpV4RevMaskDefs[i].offset  = offset;
            ArgusIpV6MaskDefs[i].offset     = offset;
            ArgusIpV6RevMaskDefs[i].offset  = offset;
            ArgusEtherMaskDefs[i].offset    = offset;
            ArgusEtherRevMaskDefs[i].offset = offset;
            break;
         }
      }
   }
}


/*
*/

struct ArgusAggregatorStruct *
ArgusNewAggregator (struct ArgusParserStruct *parser, char *masklist)
{
   struct ArgusAggregatorStruct *retn = NULL;
   struct ArgusModeStruct *mode = NULL, *modelist = NULL, *list;
   char *ptr, *tok;
   int i;

   if ((retn = (struct ArgusAggregatorStruct *) ArgusCalloc (1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewAggregator: ArgusCalloc error %s", strerror(errno));

   ArgusInitAggregatorStructs(retn);

   if ((ptr = masklist) != NULL) {
      while ((tok = strtok (ptr, " \t")) != NULL) {
         if ((mode = (struct ArgusModeStruct *) ArgusCalloc (1, sizeof(struct ArgusModeStruct))) != NULL) {
            if ((list = modelist) != NULL) {
               while (list->nxt)
                  list = list->nxt;
               list->nxt = mode;
            } else
               modelist = mode;

            mode->mode = strdup(tok);
         }
         ptr = NULL;
      }

   } else {
      if ((modelist = parser->ArgusMaskList) == NULL)
         retn->mask  = ( ARGUS_MASK_SRCID_INDEX | ARGUS_MASK_PROTO_INDEX |
                         ARGUS_MASK_SADDR_INDEX | ARGUS_MASK_SPORT_INDEX |
                         ARGUS_MASK_DADDR_INDEX | ARGUS_MASK_DPORT_INDEX );
   }

   retn->correct = strdup("yes");

   if ((mode = modelist) != NULL) {
      while (mode) {
         char *ptr = NULL, *endptr = NULL;
         struct ArgusIPAddrStruct mask;
         int len = 0, x = 0;

         bzero((char *)&mask, sizeof(mask));

         if ((ptr = strchr(mode->mode, '/')) != NULL) {
            *ptr++ = '\0';
            if (strchr(ptr, ':')) {
               if (!(inet_pton(AF_INET6, (const char *) ptr, &mask.addr_un.ipv6) > 0))
                  ArgusLog (LOG_ERR, "syntax error: %s %s", ptr, strerror(errno));
#if defined(_LITTLE_ENDIAN)
               for (x = 0 ; x < 4 ; x++)
                  mask.addr_un.ipv6[x] = htonl(mask.addr_un.ipv6[x]);
#endif
               len = 128;
            } else
            if (strchr(ptr, '.')) {
               if (!(inet_pton(AF_INET, (const char *) ptr, &mask.addr_un.ipv4) > 0))
                  ArgusLog (LOG_ERR, "syntax error: %s %s", ptr, strerror(errno));
#if defined(_LITTLE_ENDIAN)
               mask.addr_un.ipv4 = htonl(mask.addr_un.ipv4);
#endif
               len = 32;
            } else {
               if ((len = strtol(ptr, &endptr, 10)) == 0)
                  if (endptr == ptr)
                     ArgusLog (LOG_ERR, "syntax error: %s %s", ptr, strerror(errno));

               if (len <= 32)
                  mask.addr_un.ipv4 = (0xFFFFFFFF << (32 - len));
               else {
                  int tlen = len;
                  for (x = 0; x < 4; x++) {
                     int masklen = (tlen > 32) ? 32 : tlen;
                     if (masklen)
                        mask.addr_un.ipv6[x] = (0xFFFFFFFF << (32 - masklen));
                     else
                        mask.addr_un.ipv6[x] = 0;
                     if (tlen)
                        tlen -= masklen;
                  }
               }
            }
         }
         
         if (!(strncasecmp (mode->mode, "none", 4)))
            retn->mask  = 0;
         else
         if (!(strncasecmp (mode->mode, "mac", 3))) {
            parser->RaMonMode++;
            retn->mask |= (0x01 << ARGUS_MASK_SMAC);
            if (len > 0) {
               retn->saddrlen = len;
               retn->daddrlen = len;
            }
         } else
         if (!(strncasecmp (mode->mode, "addr", 4))) {
            parser->RaMonMode++;
            retn->mask |= (0x01 << ARGUS_MASK_SADDR);
            retn->mask |= (0x01 << ARGUS_MASK_DADDR);
            if (len > 0) {
               retn->saddrlen = len;
               retn->daddrlen = len;
               bcopy((char *)&mask, (char *)&retn->smask, sizeof(mask));
               bcopy((char *)&mask, (char *)&retn->dmask, sizeof(mask));
            }
         } else
         if (!(strncasecmp (mode->mode, "matrix", 6))) {
            retn->ArgusMatrixMode++;
            retn->mask |= (0x01 << ARGUS_MASK_SADDR);
            retn->mask |= (0x01 << ARGUS_MASK_DADDR);
            if (len > 0) {
               retn->saddrlen = len;
               retn->daddrlen = len;
               bcopy((char *)&mask, (char *)&retn->smask, sizeof(mask));
               bcopy((char *)&mask, (char *)&retn->dmask, sizeof(mask));
            }

         } else {
            struct ArgusMaskStruct *ArgusMaskDefs = ArgusIpV4MaskDefs;

            for (i = 0; i < ARGUS_MAX_MASK_LIST; i++) {
               if (!(strncasecmp (mode->mode, ArgusMaskDefs[i].name, ArgusMaskDefs[i].slen))) {
                  retn->mask |= (0x01 << i);
                  switch (i) {
                     case ARGUS_MASK_SADDR:
                        if (len > 0) {
                           retn->saddrlen = len;
                           bcopy((char *)&mask, (char *)&retn->smask, sizeof(mask));
                        }
                        break;
                     case ARGUS_MASK_DADDR:
                        if (len > 0) {
                           retn->daddrlen = len;
                           bcopy((char *)&mask, (char *)&retn->dmask, sizeof(mask));
                        }
                        break;

                     case ARGUS_MASK_SMPLS:
                     case ARGUS_MASK_DMPLS: {
                        int x, RaNewIndex = 0;
                        char *ptr;

                        if ((ptr = strchr(mode->mode, '[')) != NULL) {
                           char *cptr = NULL;
                           int sind = -1, dind = -1;
                           *ptr++ = '\0';
                           while (*ptr != ']') {
                              if (isdigit((int)*ptr)) {
                                 dind = strtol(ptr, (char **)&cptr, 10);
                                 if (cptr == ptr)
                                    usage ();
               
                                 if (sind < 0)
                                    sind = dind;

                                 for (x = sind; x <= dind; x++)
                                    RaNewIndex |= 0x01 << x;

                                 ptr = cptr;
                                 if (*ptr != ']')
                                    ptr++;
                                 if (*cptr != '-')
                                    sind = -1;
                              } else
                                 usage ();
                           }
                           ArgusIpV4MaskDefs[i].index = RaNewIndex;
                           ArgusIpV6MaskDefs[i].index = RaNewIndex;
                           ArgusEtherMaskDefs[i].index = RaNewIndex;
                        }
                        break;
                     }
                  }
                  break;
               }
            }
         }

         mode = mode->nxt;
      }
   }

   if ((retn->drap = (struct RaPolicyStruct *) ArgusCalloc(1, sizeof(*retn->drap))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewAggregator: ArgusCalloc error %s", strerror(errno));

   if ((retn->queue = ArgusNewQueue()) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewAggregator: ArgusNewQueue error %s", strerror(errno));

   if ((retn->htable = ArgusNewHashTable (RA_HASHTABLESIZE)) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewAggregator: ArgusNewHashTable error %s", strerror(errno));

   retn->RaMetricFetchAlgorithm = ArgusFetchDuration;

   return (retn);
}


/*
struct ArgusAggregatorStruct {
   struct ArgusAggregatorStruct *nxt;
   int status, mask, saddrlen, daddrlen;

   struct RaPolicyStruct *drap, *rap;
   struct RaFlowModelStruct *fmodel;
   struct ArgusModeStruct *ArgusModeList, *ArgusMaskList;
   struct ArgusQueueStruct *queue;
   struct ArgusHashTable htable;
   struct ArgusHashStruct hstruct;

   char *filterstr;
   struct nff_program filter;
   double (*RaMetricFetchAlgorithm)(struct ArgusRecordStruct *);

   char ArgusMatrixMode, ArgusRmonMode, ArgusAgMode;
};
*/

void
ArgusDeleteAggregator (struct ArgusParserStruct *parser, struct ArgusAggregatorStruct *agg)
{
   struct ArgusModeStruct *mode = NULL, *prv;

   if (agg->nxt != NULL) {
      ArgusDeleteAggregator (parser, agg->nxt);
      agg->nxt = NULL;
   }

   if (agg->drap != NULL)
      ArgusFree(agg->drap);

   if ((mode = agg->ArgusModeList) != NULL) {
      while ((prv = mode) != NULL) {
         if (mode->mode != NULL)
            free (mode->mode);
         mode = mode->nxt;
         ArgusFree(prv);
      }
   }

   if ((mode = agg->ArgusMaskList) != NULL) {
      while ((prv = mode) != NULL) {
         if (mode->mode != NULL)
            free (mode->mode);
         mode = mode->nxt;
         ArgusFree(prv);
      }
   }

   if (agg->queue != NULL)
      ArgusDeleteQueue(agg->queue);

   if (agg->htable != NULL)
      ArgusDeleteHashTable(agg->htable);

   if (agg->filterstr) {
      free(agg->filterstr);
      if (agg->filter.bf_insns != NULL)
         free(agg->filter.bf_insns);
   }

   if (parser->ArgusAggregator == agg)
      parser->ArgusAggregator = NULL;

   ArgusFree(agg);
}


#define ARGUS_RCITEMS    5

#define ARGUS_RC_FILTER  0
#define ARGUS_RC_MODEL   1
#define ARGUS_RC_STATUS  2
#define ARGUS_RC_IDLE    3
#define ARGUS_RC_CONT    4

char *ArgusAggregatorFields[ARGUS_RCITEMS] = {
   "filter", "model", "status", "idle", "cont",
};

struct ArgusAggregatorStruct *
ArgusParseAggregator (struct ArgusParserStruct *parser, char *buf[])
{
   struct ArgusAggregatorStruct *retn = NULL, *agg;
   char strbuf[MAXSTRLEN], *str = strbuf;
   char *file, *ptr, *end, tmp, *value;
   char *name = NULL, *pres = NULL;
   char *report = NULL, *correct = NULL;
   char *histo = NULL;
   int i, tlines = 0;
   FILE *fd = NULL;

   if (buf == NULL) {
      if ((file = parser->ArgusFlowModelFile) != NULL) {
         if ((fd = fopen (file, "r")) == NULL)
            ArgusLog (LOG_ERR, "%s: %s", file, strerror(errno));
      }
   }

   while ((fd ? (fgets(str, MAXSTRLEN, fd)) : (str = (*buf) ? strdup(*buf++) : NULL)) != NULL)  {
      int done = 0, defined = 0;
      tlines++;

      while (*str && isspace((int)*str)) str++;
      ptr = str; 

      if (*str && (*str != '#') && (*str != '\n') && (*str != '!')) {
         char *filter = NULL, *model = NULL, *status = NULL, *idle = NULL, *cptr = NULL;
         int cont = 0;
         while (!done) {
            if (!(strncmp(str, RA_MODELNAMETAGSTR, strlen(RA_MODELNAMETAGSTR)))) {
               name = strdup(&str[strlen(RA_MODELNAMETAGSTR)]);
               done++;
            } else
            if (!(strncmp(str, RA_PRESERVETAGSTR, strlen(RA_PRESERVETAGSTR)))) {
               pres = strdup(&str[strlen(RA_PRESERVETAGSTR)]);
               done++;
            } else
            if (!(strncmp(str, RA_REPORTTAGSTR, strlen(RA_REPORTTAGSTR)))) {
               report = strdup(&str[strlen(RA_REPORTTAGSTR)]);
               done++;
            } else
            if (!(strncmp(str, RA_AUTOCORRECTSTR, strlen(RA_AUTOCORRECTSTR)))) {
               correct = strdup(&str[strlen(RA_AUTOCORRECTSTR)]);
               if (!(strstr(correct, "yes")))
                  correct = NULL;
               done++;
            } else
            if (!(strncmp(str, RA_HISTOGRAM, strlen(RA_HISTOGRAM)))) {
               histo = strdup(&str[strlen(RA_HISTOGRAM)]);
               done++;
            } else
            for (i = 0; i < ARGUS_RCITEMS; i++) {
               if (!(strncmp(str, ArgusAggregatorFields[i], strlen(ArgusAggregatorFields[i])))) {
                  ptr = str + strlen(ArgusAggregatorFields[i]); 
                  while (*ptr && isspace((int)*ptr)) ptr++;

                  if (!(*ptr == '=') && (i != ARGUS_RC_CONT))
                     ArgusLog (LOG_ERR, "ArgusParseAggregator: syntax error line %d %s", tlines, str);

                  ptr++;
                  while (*ptr && isspace((int)*ptr)) ptr++;

                  switch (i) {
                     case ARGUS_RC_FILTER:
                     case ARGUS_RC_MODEL: {
                        if (*ptr == '\"') {
                          ptr++;
                          end = ptr;
                          while (*end != '\"') end++;
                          *end++ = '\0';
                  
                           value = strdup(ptr);

                           ptr = end;
                        }
                        break;
                     }

                     case ARGUS_RC_STATUS:
                     case ARGUS_RC_IDLE: {
                        strtol(ptr, (char **)&end, 10);
                        if (end == ptr)
                           ArgusLog (LOG_ERR, "ArgusParseAggregator: syntax error line %d %s", tlines, str);

                        switch (*end) {
                           case 's': 
                           case 'm': 
                           case 'h': 
                           case 'd':
                              end++; break;
                        }
                        tmp = *end;
                        *end = '\0';
                        value = strdup(ptr);
                        ptr = end;
                        *ptr = tmp;
                        break;
                     }
                  }

                  switch (i) {
                     case ARGUS_RC_FILTER: filter = value; break;
                     case ARGUS_RC_MODEL:  model  = value; break;
                     case ARGUS_RC_STATUS: status = value; break;
                     case ARGUS_RC_IDLE:   idle   = value; break;
                     case ARGUS_RC_CONT:   cont++; break;
                  }

                  while (*ptr && isspace((int)*ptr)) ptr++;
                  str = ptr;
                  defined++;
               }
            }

            if (!(done || defined))
               ArgusLog (LOG_ERR, "ArgusParseAggregator: syntax error line %d: %s", tlines, str);

            if (ptr && ((*ptr == '\n') || (*ptr == '\0')))
               done++;
         }

         if (defined) {
            if ((agg = ArgusNewAggregator(parser, model)) == NULL)
               ArgusLog (LOG_ERR, "ArgusParseAggregator: ArgusNewAggregator returned NULL");

            if (cont)
               agg->cont++;

            if (name)
               agg->name = strdup(name);

            if (pres)
               agg->pres = strdup(pres);

            if (report)
               agg->report = strdup(report);

            if (correct)
               agg->correct = strdup(correct);

            agg->filterstr = filter;
            if (status) {
               agg->statusint = strtol(status, (char **)&cptr, 10);
               switch(*cptr) {
                  case 'm': agg->statusint *= 60; break;
                  case 'h': agg->statusint *= 3600; break;
                  case 'd': agg->statusint *= 86400; break;
               }
               free(status);
            }
            if (idle) {
               agg->idleint = strtol(idle, (char **)&cptr, 10);
               switch(*cptr) {
                  case 'm': agg->idleint *= 60; break;
                  case 'h': agg->idleint *= 3600; break;
                  case 'd': agg->idleint *= 86400; break;
               }
               free(idle);
            }

            if (ArgusFilterCompile (&agg->filter, agg->filterstr, ArgusParser->Oflag) < 0)
               ArgusLog (LOG_ERR, "ArgusNewAggregator ArgusFilterCompile returned error");

            if (retn != NULL) {
               struct ArgusAggregatorStruct *tagg = retn;
                while (tagg->nxt != NULL)
                   tagg = tagg->nxt;
                tagg->nxt = agg;
            } else
               retn = agg;

         } else
            if (!done)
               ArgusLog (LOG_ERR, "ArgusNewAggregator: line %d, syntax error %s", tlines, str);
      }
   }

   if ((agg = ArgusNewAggregator(parser, NULL)) == NULL)
      ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");

   if (retn != NULL) {
      struct ArgusAggregatorStruct *tagg = retn;
      while (tagg->nxt != NULL)
         tagg = tagg->nxt;
      tagg->nxt = agg;
   } else
      retn = agg;

   return (retn);
}


int
RaParseType (char *str)
{
   return(argus_nametoeproto(str));
}


struct RaAddressStruct *RaFindAddress (struct ArgusParserStruct *, struct RaAddressStruct *, struct RaAddressStruct *, int);
struct RaAddressStruct *RaInsertAddress (struct ArgusParserStruct *, struct ArgusLabelerStruct *, struct RaAddressStruct *, struct RaAddressStruct *, int);

void RaInsertRIRTree (struct ArgusParserStruct *, struct ArgusLabelerStruct *labeler, char *);
void RaInsertAddressTree (struct ArgusParserStruct *, struct ArgusLabelerStruct *labeler, char *);

struct RaAddressStruct *
RaFindAddress (struct ArgusParserStruct *parser, struct RaAddressStruct *tree, struct RaAddressStruct *node, int mode)
{
   struct RaAddressStruct *retn = NULL;
   int done = 0;

   while (tree && !done) {
     unsigned int mask, taddr, naddr;

      switch (tree->addr.type) {
         case AF_INET: {
            if (tree->addr.masklen > 0)
               mask = 0xFFFFFFFF << (32 - tree->addr.masklen);
            else
               mask = 0;

            taddr = tree->addr.addr[0] & mask;
            naddr = node->addr.addr[0] & mask;
/*
struct RaAddressStruct {
   struct ArgusQueueHeader qhdr;
   struct RaAddressStruct *l, *r;
   struct ArgusCIDRAddr addr;
   int offset, count, status;
   float x, y;

   char *label;
};
struct ArgusCIDRAddr {
   u_char type, len, masklen, pad;
   u_int addr[4], mask[4];
   char *str;
};
*/
            if (taddr == naddr) {
               switch (mode) {
                  case ARGUS_EXACT_MATCH: 
                     if (node->addr.masklen == tree->addr.masklen)
                        retn = tree;
                     else
                     if ((tree->l == NULL) && (tree->r == NULL))
                        retn = NULL;
                     else {
                        if ((node->addr.addr[0] >> (32 - (tree->addr.masklen + 1))) & 0x01)
                          retn = RaFindAddress (parser, tree->l, node, mode);
                        else 
                          retn = RaFindAddress (parser, tree->r, node, mode);
                     }
                     done++;
                     break;

                  case ARGUS_LONGEST_MATCH: 
                     if ((node->addr.addr[0] >> (32 - (tree->addr.masklen + 1))) & 0x01) {
                        if ((retn = RaFindAddress (parser, tree->l, node, mode)) == NULL)
                           retn = tree;
                     } else {
                        if ((retn = RaFindAddress (parser, tree->r, node, mode)) == NULL)
                           retn = tree;
                     }
                     done++;
                     break;
                  
                  case ARGUS_ANY_MATCH:
                     retn = tree;
                     done++;
                     break;
               }

            } else 
               done++;
         }
         break;

         case AF_INET6: {
         }
         break;
      }
   }

   return (retn);
}

struct RaAddressStruct *
RaInsertAddress (struct ArgusParserStruct *parser, struct ArgusLabelerStruct *labeler, struct RaAddressStruct *tree, struct RaAddressStruct *node, int status)
{
   struct RaAddressStruct *retn  = NULL;
   struct RaAddressStruct *ptree = NULL;

   if (labeler != NULL) {
      struct RaAddressStruct **ArgusAddrTree = labeler->ArgusAddrTree;

      if ((tree == NULL) && (ArgusAddrTree[node->addr.type] == NULL)) {
         ArgusAddrTree[node->addr.type] = node;

         if (node->addr.masklen)
            node->addr.mask[0] = 0xFFFFFFFF << (32 - node->addr.masklen);
         else
            node->addr.mask[0] = 0;

         node->status |= status;
         retn = node;

      } else {
         unsigned int taddr, naddr;
         unsigned int tmask, nmask;

         while (retn == NULL) {
            switch (tree->addr.type) {
               case AF_INET: {
                  if (tree->addr.masklen > 0)
                     tmask = 0xFFFFFFFF << (32 - tree->addr.masklen);
                  else
                     tmask = 0;

                  taddr = tree->addr.addr[0] & tmask;
                  naddr = node->addr.addr[0] & tmask;

                  if (naddr == taddr) {
                     tree->status |= status;
                     if (node->addr.masklen == tree->addr.masklen)
                        return (NULL);

                     if (node->addr.masklen > tree->addr.masklen) {
                        unsigned int naddr = 0;
                        if (32 - (tree->addr.masklen + 1))
                           naddr = node->addr.addr[0] >> (32 - (tree->addr.masklen + 1));

                        if (naddr & 0x01) {
                           if (tree->l == NULL) {
                              if (node->addr.masklen > 0) {
                                 nmask = (0xFFFFFFFF << (32 - node->addr.masklen));
                                 nmask &= (0xFFFFFFFF >> tree->addr.masklen);
                              } else
                                 nmask = 0;

                              node->offset = tree->addr.masklen;
                              node->addr.mask[0] = nmask;
                              tree->l = node;
                              retn = node;
                           } else {
                              ptree = tree;
                              tree = tree->l;
                           }

                        } else {
                           if (tree->r == NULL) {
                              if (node->addr.masklen > 0) {
                                 nmask  = (0xFFFFFFFF << (32 - node->addr.masklen));
                                 nmask &= (0xFFFFFFFF >> tree->addr.masklen);
                              } else
                                 nmask = 0;

                              node->offset = tree->addr.masklen;
                              node->addr.mask[0] = nmask;
                              tree->r = node;
                              retn = node;

                           } else {
                              ptree = tree;
                              tree = tree->r;
                           }
                        }

                     } else {
                        if ((32 - (node->addr.masklen + 1)) > 0)
                           naddr = tree->addr.addr[0] >> (32 - (node->addr.masklen + 1));
                        if (node->addr.masklen > 0)
                           nmask = (0xFFFFFFFF << (32 - node->addr.masklen));
                        else
                           nmask = 0;

                        if (naddr & 0x01) {
                           node->l = tree;
                        } else
                           node->r = tree;
                        
                        if (ptree != NULL) {
                           if (ptree->l == tree)
                              ptree->l = node;
                           else
                              ptree->r = node;

                           if (ptree->addr.masklen < 32)
                              nmask &= nmask & (0xFFFFFFFF >> ptree->addr.masklen);
                           else
                              nmask = 0;
                           node->offset = ptree->addr.masklen;

                        } else {
                           ArgusAddrTree[node->addr.type] = node;
                           node->offset = 0;
                        }

                        node->addr.mask[0] = nmask;
                        tree->offset = node->addr.masklen;

                        if (32 - tree->addr.masklen)
                           tmask = (0xFFFFFFFF << (32 - tree->addr.masklen));
                        else
                           tmask = 0;
                        tmask &= tmask & (0xFFFFFFFF >> node->addr.masklen);
                        tree->addr.mask[0] = tmask;
                        retn = node;
                     }

                  } else {
                     struct RaAddressStruct *addr = NULL;
                     unsigned int value;
                     int i, len = (node->addr.masklen - tree->offset);
                     int masklen = 0;

                     if (tree->addr.masklen > 0)
                        tmask = (0xFFFFFFFF << (32 - tree->addr.masklen));
                     else
                        tmask = 0;
 
                     value = ~(taddr ^ naddr) & tmask;
                     value = value << tree->offset;

                     for (i = 0; i < len; i++) {
                        if (value & 0x80000000) {
                           masklen++;
                           value = value << 1;
                        } else
                           break;
                     }

                     if ((addr = (struct RaAddressStruct *) ArgusCalloc (1, sizeof(*addr))) != NULL) {
                        bcopy ((char *) node, (char *) addr, sizeof(*addr));
                        addr->addr.str = NULL;
                        addr->label = NULL;
                        addr->offset = tree->offset;
                        addr->status = status;

                        if (ptree) {
                           addr->addr.masklen = ptree->addr.masklen + masklen;
                           if (ptree->l == tree) {
                              ptree->l = addr;
                           } else {
                              ptree->r = addr;
                           }
                        } else {
                           addr->addr.masklen = masklen;
                           ArgusAddrTree[tree->addr.type] = addr;
                        }

                        if (addr->addr.masklen)
                           addr->addr.mask[0] = (0xFFFFFFFF << (32 - addr->addr.masklen));
                        else
                           addr->addr.mask[0] = 0;

                        addr->addr.mask[0] &= addr->addr.mask[0] & (0xFFFFFFFF >> addr->offset);

                        tree->offset += masklen;
                        node->offset  = tree->offset;

                        value = tree->addr.addr[0] >> (32 - (tree->offset + 1));

                        if (value & 0x01) {
                           addr->l = tree;
                           addr->r = node;
                        } else {
                           addr->l = node;
                           addr->r = tree;
                        }

                        if (tree->addr.masklen)
                           tree->addr.mask[0] = (0xFFFFFFFF << (32 - tree->addr.masklen));
                        else
                           tree->addr.mask[0] = 0;

                        tree->addr.mask[0] &= tree->addr.mask[0] & (0xFFFFFFFF >> tree->offset);

                        if (node->addr.masklen > 0)
                           node->addr.mask[0] = (0xFFFFFFFF << (32 - node->addr.masklen));

                        node->addr.mask[0] &= node->addr.mask[0] & (0xFFFFFFFF >> node->offset);
                     }

                     retn = node;
                  }
               }
               break;

               case AF_INET6: {
               }
               break;
            }
         }
      }

      if (retn != NULL)
         retn->status |= status;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaInsertAddress (0x%x, 0x%x, 0x%x, 0x%x) returning 0x%x\n", parser, tree, node, status, retn);
#endif

   return (retn);
}

void RaDeleteAddressTree(struct RaAddressStruct *);
char *RaPruneAddressTree (struct ArgusLabelerStruct *, struct RaAddressStruct *);

void
RaDeleteAddressTree(struct RaAddressStruct *node)
{
   if (node->l) RaDeleteAddressTree(node->l);
   if (node->r) RaDeleteAddressTree(node->r);

   free(node->addr.str);
   ArgusFree(node);
}

char *
RaPruneAddressTree (struct ArgusLabelerStruct *labeler, struct RaAddressStruct *node)
{
   char *retn = NULL, *lstr, *rstr;

   if ((node == NULL) || (node->addr.masklen == 0))
      return(retn);

   lstr = RaPruneAddressTree(labeler, node->l);
   rstr = RaPruneAddressTree(labeler, node->r);

   if (node->l && node->r) {
      if (lstr && rstr)
         if (!(strcmp(lstr, rstr)))
            retn = lstr;
   } else {
      if (node->l) {
         if (lstr) 
            retn = lstr;
      } else 
         if (node->r) {
            if (rstr) 
               retn = rstr;
         } else
            retn = node->label;
   }

   if (retn != NULL) {
      if (node->label) {
        if (node->label != retn)
           if ((strcmp(node->label, retn)))
              retn = NULL;
      } else
         node->label = retn;
   }

   if (retn != NULL) {
      if (node->l) {
         RaDeleteAddressTree(node->l);
         node->l = NULL;
      }
      if (node->r) {
         RaDeleteAddressTree(node->r);
         node->r = NULL;
      }
   }

   return (retn);
}


void
RaInsertRIRTree (struct ArgusParserStruct *parser, struct ArgusLabelerStruct *labeler, char *str)
{
   struct RaAddressStruct *saddr = NULL, *node;
   char tstrbuf[MAXSTRLEN], *sptr = tstrbuf, *tptr;
   char *rir = NULL, *co = NULL, *type = NULL;
   char *addr, *endptr = NULL;
   int tok = 0, elem = -1;

   if (labeler != NULL) {
      struct RaAddressStruct **ArgusAddrTree = labeler->ArgusAddrTree;

      snprintf (tstrbuf, MAXSTRLEN, "%s", str);

      while ((tptr = strtok(sptr, "|\n")) != NULL) {
         switch (tok++) {
            case 0:  rir  = tptr; break;
            case 1:  co   = tptr; break;
            case 2:  type = tptr; break;
            case 3:  addr = tptr; break;
            case 4:
               if (isdigit((int)*tptr)) {
                  if ((elem = strtol(tptr, &endptr, 10)) == 0)
                     if (endptr == tptr)
                        usage();
               }

            case 5:  break;
            case 6:  break;
         }

         sptr = NULL;
      }

      if (!(strcmp ("ipv4", type)) && (strcmp ("*", co))) {
         if ((co != NULL) && (addr != NULL)) {
            char sbuf[128];
            double base, value;
            int masklen;
            
            base = elem;
            if ((value = log(base)/log(2)) > 0) {
               struct ArgusCIDRAddr *cidr = NULL;

               masklen = (32 - value);
               snprintf (sbuf, 128, "%s/%d", addr, masklen);

               if ((cidr = RaParseCIDRAddr (parser, sbuf)) != NULL) {
                  if ((saddr = (struct RaAddressStruct *) ArgusCalloc (1, sizeof(*saddr))) != NULL) {
                     bcopy ((char *)cidr, (char *)&saddr->addr, sizeof (*cidr));
                     if ((node = RaFindAddress (parser, ArgusAddrTree[saddr->addr.type], saddr, ARGUS_EXACT_MATCH)) == NULL) {
                        saddr->addr.str = strdup(sbuf);
                        saddr->label = strdup(co);
                        RaInsertAddress (parser, labeler, ArgusAddrTree[saddr->addr.type], saddr, ARGUS_VISITED);
                     } else {
                        ArgusFree(saddr);
                        saddr = NULL;
                     }
                  }
               }
            }
         }
      }
   }
}

void
RaInsertAddressTree (struct ArgusParserStruct *parser, struct ArgusLabelerStruct *labeler, char *str)
{
   struct RaAddressStruct *saddr = NULL, *node;
   struct ArgusCIDRAddr *cidr, scidr, dcidr;
   char tstrbuf[MAXSTRLEN], *tptr;
   char *sptr = NULL, *eptr = NULL, *label;
   unsigned int i;

   if (labeler != NULL) {
      struct RaAddressStruct **ArgusAddrTree = labeler->ArgusAddrTree;

      snprintf (tstrbuf, MAXSTRLEN, "%s", str);
      label = tstrbuf;

      if ((sptr = strtok(label, "\"")) != NULL) {
         eptr = sptr;
         sptr = strtok(eptr, "-");
         
         if (sptr) {
            while ((tptr = strtok(sptr, " \t\n")) != NULL) {
               if ((cidr = RaParseCIDRAddr (parser, tptr)) != NULL)
                  bcopy ((char *)cidr, (char *)&scidr, sizeof (*cidr));

               sptr = NULL;
            }
         }

         if (eptr) {
            while ((tptr = strtok(eptr, " \t\n")) != NULL) {
               if ((cidr = RaParseCIDRAddr (parser, tptr)) != NULL)
                  bcopy ((char *)cidr, (char *)&dcidr, sizeof (*cidr));

               eptr = NULL;
            }

         } else
            bcopy ((char *)&scidr, (char *)&dcidr, sizeof (*cidr));
            

         for (i = scidr.addr[0]; i <= dcidr.addr[0]; i++) {
            if ((saddr = (struct RaAddressStruct *) ArgusCalloc (1, sizeof(*saddr))) != NULL) {
               bcopy ((char *)&scidr, (char *)&saddr->addr, sizeof (*cidr));
               saddr->addr.addr[0] = i;

               if ((node = RaFindAddress (parser, ArgusAddrTree[saddr->addr.type], saddr, ARGUS_EXACT_MATCH)) == NULL) {
                  if (tptr)
                     saddr->addr.str = strdup(tptr);
                  RaInsertAddress (parser, labeler, ArgusAddrTree[saddr->addr.type], saddr, ARGUS_VISITED);
                  if (label) {
                     char sbuf[128], *tptr;
                     snprintf (sbuf, 128, "\"%s", label);
                     while ((tptr = strchr (sbuf, '\n')) != NULL) *tptr = '\0';
                     saddr->label = strdup (sbuf);
                  }
               } else {
                  ArgusFree(saddr);
                  saddr = NULL;
               }
            }
         }
      }
   }
}

int RaReadAddressConfig (struct ArgusParserStruct *, struct ArgusLabelerStruct *, char *);

int
RaReadAddressConfig (struct ArgusParserStruct *parser, struct ArgusLabelerStruct *labeler, char *file)
{
   char strbuf[MAXSTRLEN], *str = strbuf, *ptr;
   int retn = 1;
   FILE *fd =  NULL;

   if (labeler != NULL) {
      if ((labeler->ArgusAddrTree = ArgusCalloc(128, sizeof(void *))) == NULL)
         ArgusLog (LOG_ERR, "RaReadAddressConfig: ArgusCalloc error %s\n", strerror(errno));

      if ((fd = fopen (file, "r")) != NULL) {
         while ((ptr = fgets (str, MAXSTRLEN, fd)) != NULL) {
            while (isspace((int)*ptr)) ptr++;
            switch (*ptr) {
               case '#': break;
               default:
                  if (isdigit((int)*ptr))
                     RaInsertAddressTree (parser, labeler, ptr);
                  else {
                     if (strchr(ptr, '|')) {
                        RaInsertRIRTree (parser, labeler, ptr);
                     }
                  }
            }
         }

         fclose(fd);

      } else
         ArgusLog (LOG_ERR, "%s: %s", file, strerror(errno));

      RaPruneAddressTree(labeler, labeler->ArgusAddrTree[AF_INET]);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaReadAddressConfig (0x%x, 0x%x, %s) returning %d\n", parser, labeler, file, retn);
#endif

   return (retn);
}


struct ArgusLabelerStruct *ArgusNewLabeler (struct ArgusParserStruct *);

struct ArgusLabelerStruct *
ArgusNewLabeler (struct ArgusParserStruct *parser)
{
   struct ArgusLabelerStruct *retn = NULL;

   if ((retn = (struct ArgusLabelerStruct *) ArgusCalloc (1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewLabeler: ArgusCalloc error %s", strerror(errno));

   if ((retn->drap = (struct RaPolicyStruct *) ArgusCalloc(1, sizeof(*retn->drap))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewLabeler: ArgusCalloc error %s", strerror(errno));

   if ((retn->queue = ArgusNewQueue()) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewLabeler: ArgusNewQueue error %s", strerror(errno));

   if ((retn->htable.array = (struct ArgusHashTableHdr **) ArgusCalloc (RA_HASHTABLESIZE, sizeof(void *))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewLabeler: ArgusCalloc error %s", strerror(errno));

   retn->htable.size = RA_HASHTABLESIZE;

   if (parser->ArgusDelegatedIPFile) {
      if (!(RaReadAddressConfig (parser, retn, parser->ArgusDelegatedIPFile) > 0))
         ArgusLog (LOG_ERR, "ArgusNewLabeler: RaReadAddressConfig error");
   }

   return (retn);
}


double
ArgusFetchSrcId (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   struct ArgusTransportStruct *t = (struct ArgusTransportStruct *)ns->dsrs[ARGUS_TRANSPORT_INDEX];
   int len;

   if (t) {
      if (t->hdr.subtype & ARGUS_SRCID) {
         retn = t->srcid.a_un.value;
         switch (t->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_INT:    len = 1; break;
            case ARGUS_TYPE_IPV4:   len = 1; break;
            case ARGUS_TYPE_IPV6:   len = 4; break;
            case ARGUS_TYPE_ETHER:  len = 6; break;
            case ARGUS_TYPE_STRING: len = t->hdr.argus_dsrvl8.len - 2;
         }
      }
   }

   return (retn);
}


double
ArgusFetchStartTime (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   int sec = 0, usec = 0;

   if (ns->hdr.type & ARGUS_MAR) {
      struct ArgusRecord *rec = (struct ArgusRecord *) &ns->canon;
      sec  = rec->argus_mar.startime.tv_sec;
      usec = rec->argus_mar.startime.tv_usec;

   } else {
      struct ArgusTimeObject *dtime = &ns->canon.time;
      unsigned int subtype = dtime->hdr.subtype & (ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START);
      struct timeval stimebuf, *st = &stimebuf;
      struct timeval etimebuf, *et = &etimebuf;
      struct timeval *stime = NULL;

      if (subtype) {
         switch (subtype) {
            case ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START: {
               st->tv_sec  = dtime->src.start.tv_sec;
               st->tv_usec = dtime->src.start.tv_usec;
               et->tv_sec  = dtime->dst.start.tv_sec;
               et->tv_usec = dtime->dst.start.tv_usec;

               stime = RaMinTime(st, et);
               break;
            }

            case ARGUS_TIME_SRC_START: {
               st->tv_sec  = dtime->src.start.tv_sec;
               st->tv_usec = dtime->src.start.tv_usec;
               stime = st;
               break;
            }

            case ARGUS_TIME_DST_START: {
               st->tv_sec  = dtime->dst.start.tv_sec;
               st->tv_usec = dtime->dst.start.tv_usec;
               stime = st;
               break;
            }
         }

      } else {
         st->tv_sec  = dtime->src.start.tv_sec;
         st->tv_usec = dtime->src.start.tv_usec;
         stime = st;
      }

      sec  = stime->tv_sec;
      usec = stime->tv_usec;
   }

   retn = (sec * 1.0) + usec/1000000.0;
   return(retn);
}


double
ArgusFetchLastTime (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   int sec = 0, usec = 0;

   if (ns->hdr.type & ARGUS_MAR) {
      struct ArgusRecord *rec = (struct ArgusRecord *) &ns->canon;
      sec  = rec->argus_mar.now.tv_sec;
      usec = rec->argus_mar.now.tv_usec;

   } else {
      struct ArgusTimeObject *dtime = &ns->canon.time;
      unsigned int subtype = dtime->hdr.subtype & (ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START);
      struct timeval stimebuf, *st = &stimebuf;
      struct timeval etimebuf, *et = &etimebuf;
      struct timeval *stime = NULL;

      if (subtype) {
         switch (subtype) {
            case ARGUS_TIME_SRC_START | ARGUS_TIME_DST_START: {
               st->tv_sec  = dtime->src.end.tv_sec;
               st->tv_usec = dtime->src.end.tv_usec;
               et->tv_sec  = dtime->dst.end.tv_sec;
               et->tv_usec = dtime->dst.end.tv_usec;

               stime = RaMaxTime(st, et);
               break;
            }

            case ARGUS_TIME_SRC_START: {
               st->tv_sec  = dtime->src.end.tv_sec;
               st->tv_usec = dtime->src.end.tv_usec;
               stime = st;
               break;
            }

            case ARGUS_TIME_DST_START: {
               st->tv_sec  = dtime->dst.end.tv_sec;
               st->tv_usec = dtime->dst.end.tv_usec;
               stime = st;
               break;
            }
         }

      } else {
         st->tv_sec  = dtime->src.end.tv_sec;
         st->tv_usec = dtime->src.end.tv_usec;
         stime = st;
      }

      sec  = stime->tv_sec;
      usec = stime->tv_usec;
   }

   retn = (sec * 1.0) + usec/1000000.0;
   return(retn);
}

double
ArgusFetchAvgDuration (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      struct ArgusAgrStruct *agr;

      if ((agr = (struct ArgusAgrStruct *) ns->dsrs[ARGUS_AGR_INDEX]) != NULL)
         retn = agr->act.meanval;
      else
         retn = RaGetFloatDuration (ns);
   }
   return retn;
}

double
ArgusFetchMinDuration (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      struct ArgusAgrStruct *agr;

      if ((agr = (struct ArgusAgrStruct *) ns->dsrs[ARGUS_AGR_INDEX]) != NULL)
         retn = agr->act.minval;
      else
         retn = RaGetFloatDuration (ns);
   }
   return retn;
}

double
ArgusFetchMaxDuration (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      struct ArgusAgrStruct *agr;

      if ((agr = (struct ArgusAgrStruct *) ns->dsrs[ARGUS_AGR_INDEX]) != NULL)
         retn = agr->act.maxval;
      else
         retn = RaGetFloatDuration (ns);
   }
   return (retn);
}

double
ArgusFetchDuration (struct ArgusRecordStruct *ns)
{
   float dur = RaGetFloatDuration(ns);
   double retn = dur;
   return (retn);
}

double
ArgusFetchSrcDuration (struct ArgusRecordStruct *ns)
{
   float dur = RaGetFloatDuration(ns);
   double retn = dur;
   return (retn);
}

double
ArgusFetchDstDuration (struct ArgusRecordStruct *ns)
{
   float dur = RaGetFloatDuration(ns);
   double retn = dur;
   return (retn);
}

double
ArgusFetchSrcMac (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   return(retn);
}

double
ArgusFetchDstMac (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   return(retn);
}

double
ArgusFetchSrcAddr (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   return(retn);
}

double
ArgusFetchDstAddr (struct ArgusRecordStruct *ns)
{
    return(0);
}

double
ArgusFetchProtocol (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   struct ArgusFlow *f1 = (struct ArgusFlow *) ns->dsrs[ARGUS_FLOW_INDEX];

   if (f1) {
      switch (f1->hdr.subtype & 0x3F) {
         case ARGUS_FLOW_CLASSIC5TUPLE: {
            switch (f1->hdr.argus_dsrvl8.qual) {
               case ARGUS_TYPE_IPV4:
                  retn = f1->ip_flow.ip_p;
                  break;
               case ARGUS_TYPE_IPV6:
                  retn = f1->ipv6_flow.ip_p;
                  break;
            }
            break;
         }
 
         default:
            break;
      }
   }
 
   return(retn);
}

double
ArgusFetchSrcPort (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   struct ArgusFlow *flow = (struct ArgusFlow *) ns->dsrs[ARGUS_FLOW_INDEX];

   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
            case ARGUS_TYPE_IPV4:
               if ((flow->ip_flow.ip_p == IPPROTO_TCP) || (flow->ip_flow.ip_p == IPPROTO_UDP))
                  retn = (flow->hdr.subtype & ARGUS_REVERSE) ? flow->ip_flow.dport : flow->ip_flow.sport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (flow->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     retn = (flow->hdr.subtype & ARGUS_REVERSE) ? flow->ipv6_flow.dport : flow->ipv6_flow.sport;
                     break;
                  }
               }

               break;
         }
         break;
      }

      default:
         break;
   }

   return(retn);
}

double
ArgusFetchDstPort (struct ArgusRecordStruct *ns)
{
   double retn = 0;

   struct ArgusFlow *flow = (struct ArgusFlow *) ns->dsrs[ARGUS_FLOW_INDEX];
 
   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
            case ARGUS_TYPE_IPV4:
               if ((flow->ip_flow.ip_p == IPPROTO_TCP) || (flow->ip_flow.ip_p == IPPROTO_UDP))
                  retn = (flow->hdr.subtype & ARGUS_REVERSE) ? flow->ip_flow.sport : flow->ip_flow.dport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (flow->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     retn = (flow->hdr.subtype & ARGUS_REVERSE) ? flow->ipv6_flow.sport : flow->ipv6_flow.dport;
                     break;
                  }
               }

               break;
         }
         break;
      }
 
      default:
         break;
   }

   return(retn);
}


double
ArgusFetchSrcMpls (struct ArgusRecordStruct *ns)
{
   struct ArgusMplsStruct *m1 = (struct ArgusMplsStruct *)ns->dsrs[ARGUS_MPLS_INDEX];
   double retn = 0;

   if (m1 && (m1->hdr.subtype & ARGUS_MPLS_SRC_LABEL)) {
      unsigned char *p1 = (unsigned char *)&m1->slabel;

#if defined(_LITTLE_ENDIAN)
      retn = (p1[0] << 12) | (p1[1] << 4) | ((p1[2] >> 4) & 0xff);
#else
      retn = (p1[3] << 12) | (p1[2] << 4) | ((p1[1] >> 4) & 0xff);
#endif
   }

   return (retn);
}

double
ArgusFetchDstMpls (struct ArgusRecordStruct *ns)
{
   struct ArgusMplsStruct *m1 = (struct ArgusMplsStruct *)ns->dsrs[ARGUS_MPLS_INDEX];
   double retn = 0;

   if (m1 && (m1->hdr.subtype & ARGUS_MPLS_DST_LABEL)) {
      unsigned char *p1 = (unsigned char *)&m1->dlabel;

#if defined(_LITTLE_ENDIAN)
      retn = (p1[0] << 12) | (p1[1] << 4) | ((p1[2] >> 4) & 0xff);
#else
      retn = (p1[3] << 12) | (p1[2] << 4) | ((p1[1] >> 4) & 0xff);
#endif
   }

   return (retn);
}

double
ArgusFetchSrcVlan (struct ArgusRecordStruct *ns)
{
   struct ArgusVlanStruct *v1 = (struct ArgusVlanStruct *)ns->dsrs[ARGUS_VLAN_INDEX];
   double retn = 0;

   if (v1 && (v1->hdr.argus_dsrvl8.qual & ARGUS_SRC_VLAN))
      retn = v1->sid & 0x0FFF;

   return (retn);
}

double
ArgusFetchDstVlan (struct ArgusRecordStruct *ns)
{
   struct ArgusVlanStruct *v1 = (struct ArgusVlanStruct *)ns->dsrs[ARGUS_VLAN_INDEX];
   double retn = 0;

   if (v1 && (v1->hdr.argus_dsrvl8.qual & ARGUS_DST_VLAN))
      retn = (v1->did & 0x0FFF);

   return (retn);
}

double
ArgusFetchSrcIpId (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;
 
   if (ip1)
      retn = ip1->src.ip_id;
 
   return (retn);
}


double
ArgusFetchDstIpId (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;
 
   if (ip1)  
      retn = ip1->dst.ip_id;
 
   return (retn);
}

double
ArgusFetchSrcTos (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;

   if (ip1)
      retn = ip1->src.tos;

   return (retn);
}

double
ArgusFetchDstTos (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;

   if (ip1)
      retn = ip1->dst.tos;

   return (retn);
}

double
ArgusFetchSrcTtl (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;
 
   if (ip1 && (ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC))
      retn = (ip1->src.ttl * 1.0);

   return (retn);
}

double
ArgusFetchDstTtl (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   double retn = 0;
 
   if (ip1  && (ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST))
      retn = (ip1->dst.ttl * 1.0);

   return (retn);
}

double
ArgusFetchSrcHopCount (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   int esthops = 1;
   double retn = 0;

   if (ip1 && (ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC)) {
      while (esthops <= ip1->dst.ttl)
         esthops = esthops * 2;
      
      retn = ((esthops - ip1->src.ttl) * 1.0);
   }

   return (retn);
}

double
ArgusFetchDstHopCount (struct ArgusRecordStruct *ns)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)ns->dsrs[ARGUS_IPATTR_INDEX];
   int esthops = 1;
   double retn = 0;
 
   if (ip1  && (ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST)) {
      while (esthops <= ip1->dst.ttl)
         esthops = esthops * 2;
      
      retn = ((esthops - ip1->dst.ttl) * 1.0);
   }

   return (retn);
}

double
ArgusFetchTransactions (struct ArgusRecordStruct *ns)
{
   struct ArgusAgrStruct *a1 = (struct ArgusAgrStruct *)ns->dsrs[ARGUS_AGR_INDEX];
   double retn = 1.0;

   if (a1)
      retn = (a1->count * 1.0);

   return (retn);
}

double
ArgusFetchSrcLoad (struct ArgusRecordStruct *ns)
{
   double retn = ns->sload;
   return (retn);
}

double
ArgusFetchDstLoad (struct ArgusRecordStruct *ns)
{
   double retn = ns->dload;
   return (retn);
}


double
ArgusFetchLoad (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   float d1 = RaGetFloatDuration(ns);
   long long cnt1 = 0;
   double retn = 0.0;

   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = (m1->src.bytes + m1->dst.bytes); 

   if ((cnt1 > 0) && (d1 > 0.0))
      retn = (cnt1 * 8.0)/d1;
                    
   return (retn);
}


double
ArgusFetchLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;

   if (ns) {
      if (ns->hdr.type & ARGUS_MAR) {
          retn = ((struct ArgusRecord *)&ns->canon)->argus_mar.dropped * 1.0;
      } else {
         struct ArgusFlow *flow = (struct ArgusFlow *)&ns->canon.flow;
         switch (flow->hdr.subtype & 0x3F) {
            case ARGUS_FLOW_CLASSIC5TUPLE: {
               switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
                  case ARGUS_TYPE_IPV4: {
                     switch (ns->canon.flow.ip_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = (rtp->sdrop + rtp->ddrop) * 1.0;
                           }
                           break;
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }
                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;

                           if ((tcp != NULL) && (tcp->state != 0)) {
                              if (ns->canon.metric.src.pkts)
                                 retn = (tcp->src.retrans + tcp->dst.retrans) * 1.0;
                           }
                           break;
                        }
                        case IPPROTO_ESP: {
                           struct ArgusESPObject *esp = (void *)&ns->canon.net.net_union.esp;
                           if (esp != NULL) {
                              if (ns->canon.metric.src.pkts)
                                 retn = esp->lostseq * 1.0;
                           }
                           break;
                        }
                     }
                     break;
                  }

                  case ARGUS_TYPE_IPV6: {
                     switch (flow->ipv6_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = (rtp->sdrop + rtp->ddrop) * 1.0;
                           }
                           break;
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }

                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;

                           if ((tcp != NULL) && (tcp->state != 0)) {
                              if (ns->canon.metric.src.pkts)
                                 retn = (tcp->src.retrans + tcp->dst.retrans) * 1.0;
                           }
                           break;
                        }
                     }
                  }
               }
               break;
            }
         }
      }
   }

   return (retn);
}

double
ArgusFetchSrcLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;

   if (ns) {
      if (ns->hdr.type & ARGUS_MAR) {
      } else {
         struct ArgusFlow *flow = (struct ArgusFlow *)&ns->canon.flow;
         switch (flow->hdr.subtype & 0x3F) {
            case ARGUS_FLOW_CLASSIC5TUPLE: {
               switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
                  case ARGUS_TYPE_IPV4: {
                     switch (ns->canon.flow.ip_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = rtp->sdrop * 1.0;
                           }
                           break;
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }
                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;

                           if ((tcp != NULL) && (tcp->state != 0)) {
                              if (ns->canon.metric.src.pkts)
                                 retn = tcp->src.retrans * 1.0;
                           }
                           break;
                        }
                        case IPPROTO_ESP: {
                           struct ArgusESPObject *esp = (void *)&ns->canon.net.net_union.esp;
                           if (esp != NULL) {
                              if (ns->canon.metric.src.pkts)
                                 retn = esp->lostseq * 1.0;
                           }
                           break;
                        }
                     }
                     break;
                  }

                  case ARGUS_TYPE_IPV6: {
                     switch (flow->ipv6_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = rtp->sdrop * 1.0;
                           }
                           break;
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }

                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;

                           if ((tcp != NULL) && (tcp->state != 0)) {
                              if (ns->canon.metric.src.pkts)
                                 retn = tcp->src.retrans * 1.0;
                           }
                           break;
                        }
                     }
                  }
               }
               break;
            }
         }
      }
   }

   return (retn);
}

double
ArgusFetchDstLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;

   if (ns) {
      if (ns->hdr.type & ARGUS_MAR) {
      } else {
         struct ArgusFlow *flow = (struct ArgusFlow *)&ns->canon.flow;
         switch (flow->hdr.subtype & 0x3F) {
            case ARGUS_FLOW_CLASSIC5TUPLE: {
               switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
                  case ARGUS_TYPE_IPV4: {
                     switch (ns->canon.flow.ip_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = rtp->ddrop * 1.0;
                           }
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }
                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;
                           unsigned int status;

                           if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                              if (ns->canon.metric.dst.pkts)
                                 retn = tcp->dst.retrans * 1.0;
                           }
                           break;
                        }
                        case IPPROTO_ESP: {
                           struct ArgusESPObject *esp = (void *)&ns->canon.net.net_union.esp;
                           if (esp != NULL) {
                              if (ns->canon.metric.dst.pkts)
                                 retn = esp->lostseq * 1.0;
                           }
                        }
                     }
                     break;
                  }

                  case ARGUS_TYPE_IPV6: {
                     switch (flow->ipv6_flow.ip_p) {
                        case IPPROTO_UDP: {
                           if (ns->canon.net.hdr.subtype == ARGUS_RTP_FLOW) {
                              struct ArgusRTPObject *rtp = (void *)&ns->canon.net.net_union.rtp;
                              retn = rtp->ddrop * 1.0;
                           }
                           break;
                        }

                        case IPPROTO_ICMP: {
                           break;
                        }

                        case IPPROTO_TCP: {
                           struct ArgusTCPObject *tcp = (void *)&ns->canon.net.net_union.tcp;
                           unsigned int status;

                           if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                              if (ns->canon.metric.dst.pkts)
                                 retn = tcp->dst.retrans * 1.0;
                           }
                           break;
                        }
                     }
                  }
               }
               break;
            }
         }
      }
   }

   return (retn);
}

double
ArgusFetchPercentLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   int pkts = 0;

   if (ns) {
      retn = ArgusFetchLoss(ns);
      pkts = ns->canon.metric.src.pkts + ns->canon.metric.dst.pkts;
      if (pkts > 0) {
         retn = (retn * 100.0)/((pkts * 1.0 )+ retn);
      } else
         retn = 0.0;
   }

   return (retn);
}

double
ArgusFetchPercentSrcLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   int pkts = 0;

   if (ns) {
      retn = ArgusFetchSrcLoss(ns);
      pkts = ns->canon.metric.src.pkts;
      if (pkts > 0) {
         retn = (retn * 100.0)/((pkts * 1.0) + retn);
      } else
         retn = 0.0;
   }

   return (retn);
}

double
ArgusFetchPercentDstLoss (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   int pkts = 0;

   if (ns) {
      retn = ArgusFetchDstLoss(ns);
      pkts = ns->canon.metric.dst.pkts;
      if (pkts > 0) {
         retn = (retn * 100.0)/((pkts * 1.0) + retn);
      } else
         retn = 0.0;
   }

   return (retn);
}

double
ArgusFetchSrcRate (struct ArgusRecordStruct *ns)
{
   double retn = ns->srate;
   return (retn);
}

double
ArgusFetchDstRate (struct ArgusRecordStruct *ns)
{
   double retn = ns->drate;
   return (retn);
}

double
ArgusFetchRate (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   float d1 = RaGetFloatDuration(ns);
   long long cnt1 = 0;
   double retn = 0.0;

   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = (m1->src.pkts + m1->dst.pkts);

   if ((cnt1 > 0) && (d1 > 0.0))
      retn = (cnt1 * 1.0)/d1;

   return (retn);

}

double
ArgusFetchTranRef (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

double
ArgusFetchSeq (struct ArgusRecordStruct *ns)
{
   struct ArgusTransportStruct *t1 = (struct ArgusTransportStruct *)ns->dsrs[ARGUS_TRANSPORT_INDEX];
   double retn = 0;

   if (t1->hdr.subtype & ARGUS_SEQ)
      retn = t1->seqnum;
   return (retn);
}


double
ArgusFetchByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
     
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.bytes + m1->dst.bytes;
   return (retn); 
}

double
ArgusFetchSrcByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.bytes;
   return (retn);
}

double
ArgusFetchDstByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->dst.bytes;
 
   return (retn);
}


double
ArgusFetchPktsCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.pkts + m1->dst.pkts;
   return (retn);
}

double
ArgusFetchSrcPktsCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.pkts;
   return (retn);
}

double
ArgusFetchDstPktsCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->dst.pkts;
   return (retn);
}

double
ArgusFetchAppByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
   
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.appbytes + m1->dst.appbytes;
   return (retn); 
}
 
double
ArgusFetchSrcAppByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->src.appbytes;
   return (retn);
}

double
ArgusFetchDstAppByteCount (struct ArgusRecordStruct *ns)
{
   struct ArgusMetricStruct *m1 = NULL;
   double retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) ns->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      retn = m1->dst.appbytes;
 
   return (retn);
}

double
ArgusFetchSrcTcpBase (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp1 = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &ns->canon.flow;

   unsigned int seq1 = 0;
   double retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->src.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->src.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = seq1;
   return (retn);
}


double
ArgusFetchDstTcpBase (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp1 = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &ns->canon.flow;
   unsigned int seq1 = 0;
   double retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->dst.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->dst.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = seq1;
   return (retn);
}

double
ArgusFetchTcpRtt (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp1 = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &ns->canon.flow;
   unsigned int rtt1 = 0;
   double retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt1 = tcp1->synAckuSecs + tcp1->ackDatauSecs;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt1 = tcp1->synAckuSecs + tcp1->ackDatauSecs;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = (rtt1 * 1.0)/1000000.0;
   return (retn);
}


double
ArgusFetchTcpSynAck (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &ns->canon.flow;
   unsigned int value = 0;
   double retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     value = tcp->synAckuSecs;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     value = tcp->synAckuSecs;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = (value * 1.0)/1000000.0;
   return (retn);
}

double
ArgusFetchTcpAckDat (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &ns->canon.flow;
   unsigned int value = 0;
   double retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     value = tcp->ackDatauSecs;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     value = tcp->ackDatauSecs;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = (value * 1.0)/1000000.0;
   return (retn);
}

double
ArgusFetchSrcTcpMax (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow = &ns->canon.flow;
   double rtt = 0.0, retn = 0.0;

   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
            case ARGUS_TYPE_IPV4:
               switch (flow->ip_flow.ip_p) {
                  case  IPPROTO_TCP: {
                     if ((rtt = ArgusFetchTcpRtt(ns)) > 0.0) {
                        double synack, ackdat;
                        synack = ArgusFetchTcpSynAck(ns);
                        ackdat = ArgusFetchTcpAckDat(ns);
                        if (ackdat < 0.000100) {
                           rtt = synack;
                        } else
                        if (synack < 0.000100) {
                           rtt = ackdat;
                        }

                        if (ns->canon.metric.dst.pkts > 0) {
                           unsigned int win = tcp->dst.win << tcp->dst.winshift;
                           retn = (win * 8.0) / (rtt * 1000);
                        }
                        break;
                     }
                  }
                  default:
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP: {
                     if ((rtt = ArgusFetchTcpRtt(ns)) > 0.0) {
                        double synack, ackdat;
                        synack = ArgusFetchTcpSynAck(ns);
                        ackdat = ArgusFetchTcpAckDat(ns);
                        if (ackdat < 0.000100) {
                           rtt = synack;
                        } else 
                        if (synack < 0.000100) {
                           rtt = ackdat;
                        }
                        if (ns->canon.metric.dst.pkts > 0) {
                           unsigned int win = tcp->dst.win << tcp->dst.winshift;
                           retn = (win * 8.0) / (rtt * 1000);
                        }
                        break;
                     }
                  }
                  default:
                     break;
               }
               break;
         }
         break;
      }

      default:
         break;
   }
   return (retn);
}


double
ArgusFetchDstTcpMax (struct ArgusRecordStruct *ns)
{
   struct ArgusTCPObject *tcp = &ns->canon.net.net_union.tcp;
   struct ArgusFlow *flow = &ns->canon.flow;
   double rtt = 0.0, retn = 0.0;

   switch (flow->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
            case ARGUS_TYPE_IPV4:
               switch (flow->ip_flow.ip_p) {
                  case  IPPROTO_TCP: {
                     if ((rtt = ArgusFetchTcpRtt(ns)) > 0.0) {
                        double synack, ackdat;
                        synack = ArgusFetchTcpSynAck(ns);
                        ackdat = ArgusFetchTcpAckDat(ns);
                        if (ackdat < 0.000100) {
                           rtt = synack;
                        } else
                        if (synack < 0.000100) {
                           rtt = ackdat;
                        }

                        if (ns->canon.metric.src.pkts > 0) {
                           unsigned int win = tcp->src.win << tcp->src.winshift;
                           retn = (win * 8.0) / rtt;
                        }
                     }
                     break;
                  }
                  default:
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP: {
                     if ((rtt = ArgusFetchTcpRtt(ns)) > 0.0) {
                        double synack, ackdat;
                        synack = ArgusFetchTcpSynAck(ns);
                        ackdat = ArgusFetchTcpAckDat(ns);
                        if (ackdat < 0.000100) {
                           rtt = synack;
                        } else
                        if (synack < 0.000100) {
                           rtt = ackdat;
                        }

                        if (ns->canon.metric.src.pkts > 0) {
                           unsigned int win = tcp->src.win << tcp->src.winshift;
                           retn = (win * 8.0) / rtt;
                        }
                     }
                     break;
                  }
                  default:
                     break;
               }
               break;
         }
         break;
      }

      default:
         break;
   }
   return (retn);
}


double
ArgusFetchSrcIntPkt (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {

   } else {
      unsigned int n;

      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL) {
         if ((n = (jitter->act.src.n + jitter->idle.src.n)) > 0) {
            if (jitter->act.src.n && jitter->idle.src.n) {
               retn  = ((jitter->act.src.meanval * jitter->act.src.n) +
                        (jitter->idle.src.meanval * jitter->idle.src.n)) / n;
            } else {
               retn = (jitter->act.src.n) ? jitter->act.src.meanval : jitter->idle.src.meanval;
            }
         }
      }
   }

   return (retn/1000.0);
}

double
ArgusFetchSrcIntPktAct (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {

   } else {
      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL)
         if (jitter->act.src.n > 0)
            retn = jitter->act.src.meanval;
   }

   return (retn/1000.0);
}

double
ArgusFetchSrcIntPktIdl (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {
   
   } else {
      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL) 
         if (jitter->idle.src.n > 0)
            retn = jitter->idle.src.meanval;
   }        
         
   return (retn/1000.0);
} 

double
ArgusFetchDstIntPkt (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {

   } else {
      unsigned int n;

      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL) {
         if ((n = (jitter->act.src.n + jitter->idle.src.n)) > 0) {
            if (jitter->act.src.n && jitter->idle.src.n) {
               retn  = ((jitter->act.src.meanval * jitter->act.src.n) +
                        (jitter->idle.src.meanval * jitter->idle.src.n)) / n;
            } else {
               retn = (jitter->act.src.n) ? jitter->act.src.meanval : jitter->idle.src.meanval;
            }
         }
      }
   }

   return (retn/1000.0);
}

double
ArgusFetchDstIntPktAct (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {
   
   } else {
      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL) 
         if (jitter->act.dst.n > 0)
            retn = jitter->act.dst.meanval;
   }        
         
   return (retn/1000.0);
}  

double
ArgusFetchDstIntPktIdl (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
   struct ArgusJitterStruct *jitter;

   if (ns->hdr.type & ARGUS_MAR) {

   } else {
      if ((jitter = (struct ArgusJitterStruct *)ns->dsrs[ARGUS_JITTER_INDEX]) != NULL)
         if (jitter->idle.dst.n > 0)
            retn = jitter->idle.dst.meanval;
   }

   return (retn/1000.0);
}
   
double
ArgusFetchSrcWindow (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;

   if (ns->hdr.type & ARGUS_MAR) {
  
   } else {
      struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)&ns->canon.net;
      struct ArgusTCPObject *tcp = (struct ArgusTCPObject *)&net->net_union.tcp;
      struct ArgusFlow *flow = &ns->canon.flow;
      unsigned int win = 0;

      if (ns->canon.metric.src.pkts > 0) {
         switch (flow->hdr.subtype & 0x3F) {
            case ARGUS_FLOW_CLASSIC5TUPLE: {
               switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
                  case ARGUS_TYPE_IPV4:
                     switch (flow->ip_flow.ip_p) {
                        case  IPPROTO_TCP: {
                           win = tcp->src.win << tcp->src.winshift;
                           break;
                        }
                        default:
                           break;
                     }
                     break;
   
                  case ARGUS_TYPE_IPV6:
                     switch (flow->ipv6_flow.ip_p) {
                        case  IPPROTO_TCP: {
                           win = tcp->src.win << tcp->src.winshift;
                           break;
                        }
                        default:
                           break;
                     }
                     break;
               }
               break;
            }
   
            default:
               break;
         }

         retn = win;
      }
   }

   return (retn);
}


double
ArgusFetchDstWindow (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;

   if (ns->hdr.type & ARGUS_MAR) {
 
   } else {
      struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)&ns->canon.net;
      struct ArgusTCPObject *tcp = (struct ArgusTCPObject *)&net->net_union.tcp;
      struct ArgusFlow *flow = &ns->canon.flow;
      unsigned int win = 0;

      if (ns->canon.metric.dst.pkts > 0) {
         switch (flow->hdr.subtype & 0x3F) {
            case ARGUS_FLOW_CLASSIC5TUPLE: {
               switch ((flow->hdr.argus_dsrvl8.qual & 0x1F)) {
                  case ARGUS_TYPE_IPV4:
                     switch (flow->ip_flow.ip_p) {
                        case  IPPROTO_TCP: {
                           win = tcp->dst.win << tcp->dst.winshift;
                           break;
                        }
                        default:
                           break;
                     }
                     break;
  
                  case ARGUS_TYPE_IPV6:
                     switch (flow->ipv6_flow.ip_p) {
                        case  IPPROTO_TCP: {
                           win = tcp->dst.win << tcp->dst.winshift;
                           break;
                        }
                        default:
                           break;
                     }
                     break;
               }
               break;
            }
  
            default:
               break;
         }

         retn = win;
      }
   }
   return (retn);
}

/*
struct ArgusCorrelateStruct {
   struct ArgusDSRHeader hdr;
   struct ArgusAddrStruct srcid;
   int deltaDur, deltaStart, deltaLast;
};
*/

double
ArgusFetchDeltaDuration (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
 
   if (ns->hdr.type & ARGUS_MAR) {

   } else {
      struct ArgusCorrelateStruct *cor = (void *)ns->dsrs[ARGUS_COR_INDEX];
      if (cor != NULL)
         retn = cor->deltaDur / 1000.0;
   }
   return (retn);
}

double
ArgusFetchDeltaStartTime (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
 
   if (ns->hdr.type & ARGUS_MAR) {
   
   } else {
      struct ArgusCorrelateStruct *cor = (void *)ns->dsrs[ARGUS_COR_INDEX];
      if (cor != NULL) 
         retn = cor->deltaStart / 1000.0;
   }
   return (retn);
}

double
ArgusFetchDeltaLastTime (struct ArgusRecordStruct *ns)
{
   double retn = 0.0;
 
   if (ns->hdr.type & ARGUS_MAR) {
   
   } else {
      struct ArgusCorrelateStruct *cor = (void *)ns->dsrs[ARGUS_COR_INDEX];
      if (cor != NULL) 
         retn = cor->deltaLast / 1000.0;
   }
   return (retn);
}


double
ArgusFetchIpId (struct ArgusRecordStruct *ns)
{
   double retn = 0;
   return (retn);
}

struct ArgusSorterStruct *ArgusNewSorter (void);

struct ArgusSorterStruct *
ArgusNewSorter ()
{
   struct ArgusSorterStruct *retn = NULL;
   
   if ((retn = (struct ArgusSorterStruct *) ArgusCalloc (1, sizeof (*retn))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewSorter ArgusCalloc error %s", strerror(errno));
  
   if ((retn->ArgusRecordQueue = ArgusNewQueue()) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewSorter ArgusNewQueue error %s", strerror(errno));

   retn->ArgusSortAlgorithms[0] = ArgusSortAlgorithmTable[0];
   ArgusSorter = retn;

   return (retn);
}


void
ArgusSortQueue (struct ArgusSorterStruct *sorter, struct ArgusQueueStruct *queue)
{
   int i = 0, cnt = queue->count;

#if defined(ARGUS_THREADS)
   pthread_mutex_lock(&queue->lock);
#endif

   if (queue->array != NULL) {
      ArgusFree(queue->array);
      queue->array = NULL;
   }

   if ((queue->array = (struct ArgusQueueHeader **)
               ArgusMalloc(sizeof(struct ArgusQueueHeader *) * (cnt + 1))) != NULL) {

      for (i = 0; i < cnt; i++)
         queue->array[i] = ArgusPopQueue(queue, ARGUS_NOLOCK);

      queue->array[i] = NULL;

      if (cnt > 1)
         qsort ((char *) queue->array, cnt, sizeof (struct ArgusQueueHeader *), ArgusSortRoutine);

      for (i = 0; i < cnt; i++)
         ArgusAddToQueue(queue, queue->array[i], ARGUS_NOLOCK);

   } else
      ArgusLog (LOG_ERR, "ArgusSortQueue: ArgusMalloc %s\n", strerror(errno));

#if defined(ARGUS_THREADS)
   pthread_mutex_unlock(&queue->lock);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusSortQueue(0x%x) returned\n", queue);
#endif
}



int
ArgusSortRoutine (const void *void1, const void *void2)
{
   int retn = 0, i = 0;
   struct ArgusRecordStruct *ns1 = *(struct ArgusRecordStruct **)void1;
   struct ArgusRecordStruct *ns2 = *(struct ArgusRecordStruct **)void2;

   if (ns1 && ns2) {
      for (i = 0; i < ARGUS_MAX_SORT_ALG; i++)
         if (ArgusSorter->ArgusSortAlgorithms[i] != NULL) {
            if ((retn = ArgusSorter->ArgusSortAlgorithms[i](ns2, ns1)) != 0)
               break;
         } else
            break;
   }

   return (retn);
}


int
ArgusSortSrcId (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusTransportStruct *t1 = (struct ArgusTransportStruct *)n1->dsrs[ARGUS_TRANSPORT_INDEX];
   struct ArgusTransportStruct *t2 = (struct ArgusTransportStruct *)n2->dsrs[ARGUS_TRANSPORT_INDEX];
   unsigned int *sid1 = NULL, *sid2 = NULL;
   int retn = 0, len, i;

   if (t1 && t2) {
      if (t1->hdr.subtype & ARGUS_SRCID) {
         sid1 = &t1->srcid.a_un.ipv4;
         switch (t1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4: len = 1; break;
            case ARGUS_TYPE_IPV6: len = 4; break;
         }
      }
      if (t2->hdr.subtype & ARGUS_SRCID) {
         sid2 = &t2->srcid.a_un.ipv4;
         switch (t2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4: len = 1; break;
            case ARGUS_TYPE_IPV6: len = (len < 4) ? len : 4; break;
         }
      }

      if (sid1 && sid2) {
         for (i = 0; i < len; i++, sid1++, sid2++) {
            if (*sid2 != *sid1) {
               retn = sid2[i] - sid1[i];
               break;
            }
         }
      }
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortStartTime (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double t1 = 0.0, t2 = 0.0;
   int retn = 0;

   if (n1)
      t1 = ArgusFetchStartTime(n1);

   if (n2)
      t2 = ArgusFetchStartTime(n2);

   retn = (t2 > t1) ? 1 : ((t1 == t2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortLastTime (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double t1 = 0.0, t2 = 0.0;
   int retn = 0;

   if (n1)
      t1 = ArgusFetchLastTime(n1);

   if (n2)
      t2 = ArgusFetchLastTime(n2);

   retn = (t2 > t1) ? 1 : ((t1 == t2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortAvgDuration (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   float ad1 = 0.0, ad2 = 0.0;
   int retn = 0;
 
   if (n1 && n2) {
      ad1 = RaGetFloatAvgDuration(n1);
      ad2 = RaGetFloatAvgDuration(n2);
      retn = (ad1 > ad2) ? 1 : ((ad1 == ad2) ? 0 : -1);
   }
 
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortMinDuration (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   float ad1 = 0.0, ad2 = 0.0;
   int retn = 0;
 
   if (n1 && n2) {
      ad1 = RaGetFloatMinDuration(n1);
      ad2 = RaGetFloatMinDuration(n2);
      retn = (ad1 > ad2) ? 1 : ((ad1 == ad2) ? 0 : -1);
   }
 
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortMaxDuration (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   float ad1 = 0.0, ad2 = 0.0;
   int retn = 0;
 
   if (n1 && n2) {
      ad1 = RaGetFloatMaxDuration(n1);
      ad2 = RaGetFloatMaxDuration(n2);
      retn = (ad1 > ad2) ? 1 : ((ad1 == ad2) ? 0 : -1);
   }
 
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortDuration (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   double d1 = 0.0, d2 = 0.0;
   int retn = 0;

   if (n1)
      d1 = ArgusFetchDuration(n1);

   if (n2)
      d2 = ArgusFetchDuration(n2);

   retn = (d1 > d2) ? 1 : ((d1 == d2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortSrcMac (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   struct ArgusMacStruct *m1 = (struct ArgusMacStruct *) n1->dsrs[ARGUS_MAC_INDEX];
   struct ArgusMacStruct *m2 = (struct ArgusMacStruct *) n2->dsrs[ARGUS_MAC_INDEX];
   int retn = 0;

   if (m1 && m2) {
      retn = bcmp ((unsigned char *)&m1->mac_union.ether.ehdr.ether_shost,
                   (unsigned char *)&m2->mac_union.ether.ehdr.ether_shost, 6);
   }
 
 
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortDstMac (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   struct ArgusMacStruct *m1 = (struct ArgusMacStruct *) n1->dsrs[ARGUS_MAC_INDEX];
   struct ArgusMacStruct *m2 = (struct ArgusMacStruct *) n2->dsrs[ARGUS_MAC_INDEX];
   int retn = 0;

   if (m1 && m2) {
      retn = bcmp ((unsigned char *)&m1->mac_union.ether.ehdr.ether_dhost,
                   (unsigned char *)&m2->mac_union.ether.ehdr.ether_dhost, 6);
   }
 
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortSrcAddr (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   struct ArgusFlow *f1 = (struct ArgusFlow *) n1->dsrs[ARGUS_FLOW_INDEX];
   struct ArgusFlow *f2 = (struct ArgusFlow *) n2->dsrs[ARGUS_FLOW_INDEX];
   int retn = 0, len = 0, i = 0;

   if ((f1->hdr.subtype & 0x3F) != (f2->hdr.subtype & 0x3F))
      return((f1->hdr.subtype & 0x3F) - (f2->hdr.subtype & 0x3F));

   if (f1->hdr.argus_dsrvl8.qual != f2->hdr.argus_dsrvl8.qual)
      return (f1->hdr.argus_dsrvl8.qual - f2->hdr.argus_dsrvl8.qual);

   switch (f1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f1->hdr.argus_dsrvl8.qual & 0x07) {
            case ARGUS_TYPE_IPV4: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->ip_flow.ip_src;
               a2 = (unsigned int *)&f2->ip_flow.ip_src;
               retn = (*a1 > *a2) ? 1 : ((*a1 < *a2) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_IPV6: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->ipv6_flow.ip_src;
               a2 = (unsigned int *)&f2->ipv6_flow.ip_src;
               len = 4;
               for (i = 0; i < len; i++)
                  if (a1[i] != a2[i])
                     break;
               if (i != len)
                  retn = (a1[i] > a2[i]) ? 1 : ((a1[i] < a2[i]) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_RARP: 
               retn = bcmp (&f1->rarp_flow.shaddr, &f2->rarp_flow.shaddr, 6);
               break;
            case ARGUS_TYPE_ARP: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->arp_flow.arp_spa;
               a2 = (unsigned int *)&f2->arp_flow.arp_spa;
               retn = (*a1 > *a2) ? 1 : ((*a1 < *a2) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_ETHER: {
               unsigned char *a1, *a2;
               a1 = (unsigned char *)&f1->mac_flow.ehdr.ether_shost;
               a2 = (unsigned char *)&f2->mac_flow.ehdr.ether_shost;
               len = 6;
               for (i = 0; i < len; i++)
                  if (a1[i] != a2[i])
                     break;
               if (i != len)
                  retn = (a1[i] > a2[i]) ? 1 : ((a1[i] < a2[i]) ? -1 : 0);
               break;
            }
         }
         break;
      }
      default:
         break;
   }
 
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstAddr (struct ArgusRecordStruct *n2, struct ArgusRecordStruct *n1)
{
   struct ArgusFlow *f1 = (struct ArgusFlow *) n1->dsrs[ARGUS_FLOW_INDEX];
   struct ArgusFlow *f2 = (struct ArgusFlow *) n2->dsrs[ARGUS_FLOW_INDEX];
   int retn = 0, len = 0, i = 0;

   if ((f1->hdr.subtype & 0x3F) != (f2->hdr.subtype & 0x3F))
      return((f1->hdr.subtype & 0x3F) - (f2->hdr.subtype & 0x3F));

   if (f1->hdr.argus_dsrvl8.qual != f2->hdr.argus_dsrvl8.qual)
      return (f1->hdr.argus_dsrvl8.qual - f2->hdr.argus_dsrvl8.qual);

   switch (f1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f1->hdr.argus_dsrvl8.qual & 0x07) {
            case ARGUS_TYPE_IPV4: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->ip_flow.ip_dst;
               a2 = (unsigned int *)&f2->ip_flow.ip_dst;
               retn = (*a1 > *a2) ? 1 : ((*a1 < *a2) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_IPV6: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->ipv6_flow.ip_dst;
               a2 = (unsigned int *)&f2->ipv6_flow.ip_dst;
               len = 4;
               for (i = 0; i < len; i++)
                  if (a1[i] != a2[i])
                     break;
               if (i != len)
                  retn = (a1[i] > a2[i]) ? 1 : ((a1[i] < a2[i]) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_RARP: 
               retn = bcmp (&f1->rarp_flow.dhaddr, &f2->rarp_flow.dhaddr, 6);
               break;
            case ARGUS_TYPE_ARP: {
               unsigned int *a1, *a2;
               a1 = (unsigned int *)&f1->arp_flow.arp_tpa;
               a2 = (unsigned int *)&f2->arp_flow.arp_tpa;
               retn = (*a1 > *a2) ? 1 : ((*a1 < *a2) ? -1 : 0);
               break;
            }
            case ARGUS_TYPE_ETHER: {
               unsigned char *a1, *a2;
               a1 = (unsigned char *)&f1->mac_flow.ehdr.ether_dhost;
               a2 = (unsigned char *)&f2->mac_flow.ehdr.ether_dhost;
               len = 6;
               for (i = 0; i < len; i++)
                  if (a1[i] != a2[i])
                     break;
               if (i != len)
                  retn = (a1[i] > a2[i]) ? 1 : ((a1[i] < a2[i]) ? -1 : 0);
               break;
            }
         }
         break;
      }
      default:
         break;
   }
 
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortProtocol (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusFlow *f1 = (struct ArgusFlow *) n1->dsrs[ARGUS_FLOW_INDEX];
   struct ArgusFlow *f2 = (struct ArgusFlow *) n2->dsrs[ARGUS_FLOW_INDEX];
   unsigned char p1 = 0, p2 = 0;
   int retn = 0;

   if (f1 && f2) {
      switch (f1->hdr.subtype & 0x3F) {
         case ARGUS_FLOW_CLASSIC5TUPLE: {
            switch (f1->hdr.argus_dsrvl8.qual) {
               case ARGUS_TYPE_IPV4:
                  p1 = f1->ip_flow.ip_p;
                  break;
               case ARGUS_TYPE_IPV6:
                  p1 = f1->ipv6_flow.ip_p;
                  break;
            }
            break;
         }
 
         default:
            break;
      }
      switch (f2->hdr.subtype & 0x3F) {
         case ARGUS_FLOW_CLASSIC5TUPLE: {
            switch (f2->hdr.argus_dsrvl8.qual) {
               case ARGUS_TYPE_IPV4:
                  p2 = f2->ip_flow.ip_p;
                  break;
               case ARGUS_TYPE_IPV6:
                  p2 = f2->ipv6_flow.ip_p;
                  break;
            }
            break;
         }
 
         default:
            break;
      }
   }
 
   retn = p1 - p2;
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcPort (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusFlow *f1 = (struct ArgusFlow *) n1->dsrs[ARGUS_FLOW_INDEX];
   struct ArgusFlow *f2 = (struct ArgusFlow *) n2->dsrs[ARGUS_FLOW_INDEX];
   unsigned short p1 = 0, p2 = 0;
   int retn = 0;
 
   switch (f1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               if ((f1->ip_flow.ip_p == IPPROTO_TCP) || (f1->ip_flow.ip_p == IPPROTO_UDP))
                  p1 = (f1->hdr.subtype & ARGUS_REVERSE) ? f1->ip_flow.dport : f1->ip_flow.sport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (f1->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     p1 = (f1->hdr.subtype & ARGUS_REVERSE) ? f1->ipv6_flow.dport : f1->ipv6_flow.sport;
                     break;
                  }
               }
               break;
         }
         break;
      }
 
      default:
         break;
   }
   switch (f2->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               if ((f2->ip_flow.ip_p == IPPROTO_TCP) || (f2->ip_flow.ip_p == IPPROTO_UDP))
                  p2 = (f2->hdr.subtype & ARGUS_REVERSE) ? f2->ip_flow.dport : f2->ip_flow.sport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (f2->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     p2 = (f2->hdr.subtype & ARGUS_REVERSE) ? f2->ipv6_flow.dport : f2->ipv6_flow.sport;
                     break;
                  }
               }
         }
         break;
      }
      default:
         break;
   }
 
   retn = p1 - p2;
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstPort (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusFlow *f1 = (struct ArgusFlow *) n1->dsrs[ARGUS_FLOW_INDEX];
   struct ArgusFlow *f2 = (struct ArgusFlow *) n2->dsrs[ARGUS_FLOW_INDEX];
   unsigned short p1 = 0, p2 = 0;
   int retn = 0;
 
   switch (f1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               if ((f1->ip_flow.ip_p == IPPROTO_TCP) || (f1->ip_flow.ip_p == IPPROTO_UDP))
                  p1 = (f1->hdr.subtype & ARGUS_REVERSE) ? f1->ip_flow.sport : f1->ip_flow.dport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (f1->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     p1 = (f1->hdr.subtype & ARGUS_REVERSE) ? f1->ipv6_flow.sport : f1->ipv6_flow.dport;
                     break;
                  }
               }

               break;
         }
         break;
      }
 
      default:
         break;
   }
   switch (f2->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (f2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               if ((f2->ip_flow.ip_p == IPPROTO_TCP) || (f2->ip_flow.ip_p == IPPROTO_UDP))
                  p2 = (f2->hdr.subtype & ARGUS_REVERSE) ? f2->ip_flow.sport : f2->ip_flow.dport;
               break;
            case ARGUS_TYPE_IPV6:
               switch (f1->ipv6_flow.ip_p) {
                  case IPPROTO_TCP:
                  case IPPROTO_UDP: {
                     p2 = (f2->hdr.subtype & ARGUS_REVERSE) ? f2->ipv6_flow.sport : f2->ipv6_flow.dport;
                     break;
                  }
               }
               break;
         }
         break;
      }
 
      default:
         break;
   }
 
   retn = p2 - p1;
   return(ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcMpls (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMplsStruct *m1 = (struct ArgusMplsStruct *)n1->dsrs[ARGUS_MPLS_INDEX];
   struct ArgusMplsStruct *m2 = (struct ArgusMplsStruct *)n2->dsrs[ARGUS_MPLS_INDEX];
   unsigned int l1, l2;
   int retn = 0;

   if (m1 && m2) {
      if ((m1->hdr.subtype & ARGUS_MPLS_SRC_LABEL) && (m2->hdr.subtype & ARGUS_MPLS_SRC_LABEL)) {
         unsigned char *p1 = (unsigned char *)&m1->slabel;
         unsigned char *p2 = (unsigned char *)&m2->slabel;

#if defined(_LITTLE_ENDIAN)
         l1 = (p1[0] << 12) | (p1[1] << 4) | ((p1[2] >> 4) & 0xff);
         l2 = (p2[0] << 12) | (p2[1] << 4) | ((p2[2] >> 4) & 0xff);
#else
         l1 = (p1[3] << 12) | (p1[2] << 4) | ((p1[1] >> 4) & 0xff);
         l2 = (p2[3] << 12) | (p2[2] << 4) | ((p2[1] >> 4) & 0xff);
#endif
         retn = l1 - l2;

      } else
         retn = (m1->hdr.subtype & ARGUS_MPLS_SRC_LABEL) ? 1 :
               ((m2->hdr.subtype & ARGUS_MPLS_SRC_LABEL) ? -1 : 0);
   } else
      retn = (m1) ? 1 : ((m2) ? -1 : 0);

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstMpls (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMplsStruct *m1 = (struct ArgusMplsStruct *)n1->dsrs[ARGUS_MPLS_INDEX];
   struct ArgusMplsStruct *m2 = (struct ArgusMplsStruct *)n2->dsrs[ARGUS_MPLS_INDEX];
   unsigned int l1, l2;
   int retn = 0;

   if (m1 && m2) {
      if ((m1->hdr.subtype & ARGUS_MPLS_DST_LABEL) && (m2->hdr.subtype & ARGUS_MPLS_DST_LABEL)) {
         unsigned char *p1 = (unsigned char *)&m1->dlabel;
         unsigned char *p2 = (unsigned char *)&m2->dlabel;

#if defined(_LITTLE_ENDIAN)
         l1 = (p1[0] << 12) | (p1[1] << 4) | ((p1[2] >> 4) & 0xff);
         l2 = (p2[0] << 12) | (p2[1] << 4) | ((p2[2] >> 4) & 0xff);
#else
         l1 = (p1[3] << 12) | (p1[2] << 4) | ((p1[1] >> 4) & 0xff);
         l2 = (p2[3] << 12) | (p2[2] << 4) | ((p2[1] >> 4) & 0xff);
#endif
         retn = l1 - l2;

      } else
         retn = (m1->hdr.subtype & ARGUS_MPLS_DST_LABEL) ? 1 :
               ((m2->hdr.subtype & ARGUS_MPLS_DST_LABEL) ? -1 : 0);
   } else
      retn = (m1) ? 1 : ((m2) ? -1 : 0);

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcVlan (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusVlanStruct *v1 = (struct ArgusVlanStruct *)n1->dsrs[ARGUS_VLAN_INDEX];
   struct ArgusVlanStruct *v2 = (struct ArgusVlanStruct *)n2->dsrs[ARGUS_VLAN_INDEX];
   int retn = 0;

   if (v1 && v2) {
      if ((v1->hdr.argus_dsrvl8.qual & ARGUS_SRC_VLAN) && (v2->hdr.argus_dsrvl8.qual & ARGUS_SRC_VLAN)) {
         retn = (v1->sid & 0x0FFF) - (v2->sid & 0x0FFF);
      } else {
         retn = (v1->hdr.argus_dsrvl8.qual & ARGUS_SRC_VLAN) ? 1 :
               ((v2->hdr.argus_dsrvl8.qual & ARGUS_SRC_VLAN) ? -1 : 0);
      }
   } else
      retn = (v1) ? 1 : ((v2) ? -1 : 0);

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstVlan (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusVlanStruct *v1 = (struct ArgusVlanStruct *)n1->dsrs[ARGUS_VLAN_INDEX];
   struct ArgusVlanStruct *v2 = (struct ArgusVlanStruct *)n2->dsrs[ARGUS_VLAN_INDEX];
   int retn = 0;

   if (v1 && v2) {
      if ((v1->hdr.argus_dsrvl8.qual & ARGUS_DST_VLAN) && (v2->hdr.argus_dsrvl8.qual & ARGUS_DST_VLAN)) {
         retn = (v1->did & 0x0FFF) - (v2->did & 0x0FFF);
      } else {
         retn = (v1->hdr.argus_dsrvl8.qual & ARGUS_DST_VLAN) ? 1 :
               ((v2->hdr.argus_dsrvl8.qual & ARGUS_DST_VLAN) ? -1 : 0);
      }

   } else
      retn = (v1) ? 1 : ((v2) ? -1 : 0);

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcIpId (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned short ipid1, ipid2;
   int retn = 0;

   if (ip1 && ip2) {
      ipid1 = ip1->src.ip_id;
      ipid2 = ip2->src.ip_id;
      retn = ipid1 - ipid2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstIpId (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char ipid1, ipid2;
   int retn = 0;

   if (ip1 && ip2) {
      ipid1 = ip1->src.ip_id;
      ipid2 = ip2->src.ip_id;
      retn = ipid1 - ipid2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcTos (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char tos1, tos2;
   int retn = 0;

   if (ip1 && ip2) {
      tos1 = ip1->src.tos;
      tos2 = ip2->src.tos;
      retn = tos1 - tos2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstTos (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char tos1, tos2;
   int retn = 0;

   if (ip1 && ip2) {
      tos1 = ip1->dst.tos;
      tos2 = ip2->dst.tos;
      retn = tos1 - tos2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcDSByte (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char dsb1, dsb2;
   int retn = 0;

   if (ip1 && ip2) {
      dsb1 = (ip1->src.tos >> 2);
      dsb2 = (ip2->src.tos >> 2);
      retn = dsb1 - dsb2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstDSByte (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char dsb1, dsb2;
   int retn = 0;

   if (ip1 && ip2) {
      dsb1 = (ip1->dst.tos >> 2);
      dsb2 = (ip2->dst.tos >> 2);
      retn =  dsb1 - dsb2;
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortSrcTtl (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char ttl1, ttl2;
   int retn = 0;
 
   if (ip1 && ip2) {
      if ((ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) &&
          (ip2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC)) {
         ttl1 = ip1->src.ttl;
         ttl2 = ip2->src.ttl;
         retn = (ttl1 < ttl2) ? 1 : ((ttl1 == ttl2) ? 0 : -1);
      }
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstTtl (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusIPAttrStruct *ip1 = (struct ArgusIPAttrStruct *)n1->dsrs[ARGUS_IPATTR_INDEX];
   struct ArgusIPAttrStruct *ip2 = (struct ArgusIPAttrStruct *)n2->dsrs[ARGUS_IPATTR_INDEX];
   unsigned char ttl1, ttl2;
   int retn = 0;
 
   if (ip1 && ip2) {
      if ((ip1->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) &&
          (ip2->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST)) {
         ttl1 = ip1->dst.ttl;
         ttl2 = ip2->dst.ttl;
         retn = (ttl1 < ttl2) ? 1 : ((ttl1 == ttl2) ? 0 : -1);
      }
   }

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortTransactions (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusAgrStruct *a1 = (struct ArgusAgrStruct *)n1->dsrs[ARGUS_AGR_INDEX];
   struct ArgusAgrStruct *a2 = (struct ArgusAgrStruct *)n2->dsrs[ARGUS_AGR_INDEX];
   int retn = 0;

   if (a1 && a2)
      retn = (a1->count > a2->count) ? 1 : ((a1->count < a2->count) ? -1 : 0);

   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortSrcLoad (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1)
      r1 = ArgusFetchSrcLoad(n1);
   if (n2)
      r2 = ArgusFetchSrcLoad(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortDstLoad (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1)
      r1 = ArgusFetchDstLoad(n1);
   if (n2)
      r2 = ArgusFetchDstLoad(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortLoad (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1)
      r1 = ArgusFetchLoad(n1);
   if (n2)
      r2 = ArgusFetchLoad(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1) 
      l1 = ArgusFetchLoss(n1);

   if (n2) 
      l2 = ArgusFetchLoss(n2);
   
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortSrcLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1)
      l1 = ArgusFetchSrcLoss(n1);

   if (n2)
      l2 = ArgusFetchSrcLoss(n2);
   
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortDstLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1) 
      l1 = ArgusFetchDstLoss(n1);
   if (n2) 
      l2 = ArgusFetchDstLoss(n2);
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortPercentLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1) 
      l1 = ArgusFetchPercentLoss(n1);

   if (n2) 
      l2 = ArgusFetchPercentLoss(n2);
   
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortPercentSrcLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1)
      l1 = ArgusFetchPercentSrcLoss(n1);

   if (n2)
      l2 = ArgusFetchPercentSrcLoss(n2);
   
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int ArgusSortPercentDstLoss (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double l1 = 0.0, l2 = 0.0;
   int retn = 0;

   if (n1) 
      l1 = ArgusFetchPercentDstLoss(n1);
   if (n2) 
      l2 = ArgusFetchPercentDstLoss(n2);
   retn = (l1 > l2) ? 1 : ((l1 == l2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortSrcRate (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1) 
      r1 = ArgusFetchSrcRate(n1);
   if (n2) 
      r2 = ArgusFetchSrcRate(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortDstRate (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1)
      r1 = ArgusFetchSrcRate(n1);
   if (n2)
      r2 = ArgusFetchSrcRate(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortRate (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   double r1 = 0.0, r2 = 0.0;
   int retn = 0;

   if (n1)
      r1 = ArgusFetchRate(n1);
   if (n2)
      r2 = ArgusFetchRate(n2);
   retn = (r1 > r2) ? 1 : ((r1 == r2) ? 0 : -1);
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortTranRef (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   int retn = 0;
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}

int
ArgusSortSeq (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusTransportStruct *t1 = (struct ArgusTransportStruct *)n1->dsrs[ARGUS_TRANSPORT_INDEX];
   struct ArgusTransportStruct *t2 = (struct ArgusTransportStruct *)n2->dsrs[ARGUS_TRANSPORT_INDEX];
   unsigned int *seq1, *seq2;
   int retn = 0;

   if (t1->hdr.subtype & ARGUS_SEQ)
      seq1 = &t1->seqnum;

   if (t2->hdr.subtype & ARGUS_SEQ)
      seq2 = &t2->seqnum;

   retn = seq2 - seq1; 
   return (ArgusReverseSortDir ? ((retn > 0) ? -1 : ((retn == 0) ? 0 : 1)) : retn);
}


int
ArgusSortByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0; 
   int retn = 0;
     
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.bytes + m1->dst.bytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.bytes + m2->dst.bytes;
    
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                               ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ; 
   return (retn); 
}

int
ArgusSortSrcByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.bytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.bytes;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                               ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortDstByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->dst.bytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->dst.bytes;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                               ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}


int
ArgusSortPktsCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.pkts + m1->dst.pkts;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.pkts + m2->dst.pkts;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortSrcPktsCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.pkts;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.pkts;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortDstPktsCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->dst.pkts;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->dst.pkts;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortAppByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0; 
   int retn = 0;
     
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.appbytes + m1->dst.appbytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.appbytes + m2->dst.appbytes;
    
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ; 
   return (retn); 
}

int
ArgusSortSrcAppByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->src.appbytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->src.appbytes;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortDstAppByteCount (struct ArgusRecordStruct *n1, struct ArgusRecordStruct *n2)
{
   struct ArgusMetricStruct *m1 = NULL, *m2 = NULL;
   long long cnt1 = 0, cnt2 = 0;
   int retn = 0;
 
   if ((m1 = (struct ArgusMetricStruct *) n1->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt1 = m1->dst.appbytes;
 
   if ((m2 = (struct ArgusMetricStruct *) n2->dsrs[ARGUS_METRIC_INDEX]) != NULL)
      cnt2 = m2->dst.appbytes;
 
   retn = ArgusReverseSortDir ? ((cnt1 < cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) :
                                ((cnt1 > cnt2) ? 1 : ((cnt1 == cnt2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortSrcTcpBase (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   struct ArgusTCPObject *tcp1 = &argus1->canon.net.net_union.tcp;
   struct ArgusTCPObject *tcp2 = &argus2->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &argus1->canon.flow;
   struct ArgusFlow *flow2 = &argus2->canon.flow;

   unsigned int seq1 = 0;
   unsigned int seq2 = 0;
   int retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->src.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->src.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   switch (flow2->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow2->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq2 = tcp2->src.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow2->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq2 = tcp2->src.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = ArgusReverseSortDir ? ((seq1 < seq2) ? 1 : ((seq1 == seq2) ? 0 : -1)) :
                               ((seq1 > seq2) ? 1 : ((seq1 == seq2) ? 0 : -1)) ;
   return (retn);
}


int
ArgusSortDstTcpBase (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   struct ArgusTCPObject *tcp1 = &argus1->canon.net.net_union.tcp;
   struct ArgusTCPObject *tcp2 = &argus2->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &argus1->canon.flow;
   struct ArgusFlow *flow2 = &argus2->canon.flow;

   unsigned int seq1 = 0;
   unsigned int seq2 = 0;
   int retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->dst.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq1 = tcp1->dst.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   switch (flow2->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow2->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq2 = tcp2->dst.seqbase;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow2->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     seq2 = tcp2->dst.seqbase;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = ArgusReverseSortDir ? ((seq1 < seq2) ? 1 : ((seq1 == seq2) ? 0 : -1)) :
                               ((seq1 > seq2) ? 1 : ((seq1 == seq2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortTcpRtt (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   struct ArgusTCPObject *tcp1 = &argus1->canon.net.net_union.tcp;
   struct ArgusTCPObject *tcp2 = &argus2->canon.net.net_union.tcp;
   struct ArgusFlow *flow1 = &argus1->canon.flow;
   struct ArgusFlow *flow2 = &argus2->canon.flow;

   unsigned int rtt1 = 0;
   unsigned int rtt2 = 0;
   int retn = 0;

   switch (flow1->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow1->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow1->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt1 = tcp1->synAckuSecs + tcp1->ackDatauSecs;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow1->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt1 = tcp1->synAckuSecs + tcp1->ackDatauSecs;
                     break;
               }
               break;
         }
         break;
      }
   }

   switch (flow2->hdr.subtype & 0x3F) {
      case ARGUS_FLOW_CLASSIC5TUPLE: {
         switch (flow2->hdr.argus_dsrvl8.qual) {
            case ARGUS_TYPE_IPV4:
               switch (flow2->ip_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt2 = tcp2->synAckuSecs + tcp2->ackDatauSecs;
                     break;
               }
               break;

            case ARGUS_TYPE_IPV6:
               switch (flow2->ipv6_flow.ip_p) {
                  case  IPPROTO_TCP:
                     rtt2 = tcp2->synAckuSecs + tcp2->ackDatauSecs;
                     break;
               }
               break;
         }
         break;
      }
   }

   retn = ArgusReverseSortDir ? ((rtt1 < rtt2) ? 1 : ((rtt1 == rtt2) ? 0 : -1)) :
                                ((rtt1 > rtt2) ? 1 : ((rtt1 == rtt2) ? 0 : -1)) ;
   return (retn);
}


int
ArgusSortSrcMaxPktSize (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   unsigned short smaxsz1 = argus1->canon.psize.src.psizemax;
   unsigned short smaxsz2 = argus2->canon.psize.src.psizemax;

   retn = ArgusReverseSortDir ? ((smaxsz1 < smaxsz2) ? 1 : ((smaxsz1 == smaxsz2) ? 0 : -1)) :
                                ((smaxsz1 > smaxsz2) ? 1 : ((smaxsz1 == smaxsz2) ? 0 : -1)) ;
   return (retn);
}


int
ArgusSortSrcMinPktSize (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   unsigned short smiargusz1 = argus1->canon.psize.src.psizemin;
   unsigned short smiargusz2 = argus2->canon.psize.src.psizemin;

   retn = ArgusReverseSortDir ? ((smiargusz1 > smiargusz2) ? 1 : ((smiargusz1 == smiargusz2) ? 0 : -1)) :
                                ((smiargusz1 < smiargusz2) ? 1 : ((smiargusz1 == smiargusz2) ? 0 : -1)) ;
   return (retn);
}


int
ArgusSortDstMaxPktSize (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   unsigned short dmaxsz1 = argus1->canon.psize.dst.psizemax;
   unsigned short dmaxsz2 = argus2->canon.psize.dst.psizemax;

   retn = ArgusReverseSortDir ? ((dmaxsz1 < dmaxsz2) ? 1 : ((dmaxsz1 == dmaxsz2) ? 0 : -1)) :
                                ((dmaxsz1 > dmaxsz2) ? 1 : ((dmaxsz1 == dmaxsz2) ? 0 : -1)) ;
   return (retn);
}

int
ArgusSortDstMinPktSize (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   unsigned short dmiargusz1 = argus1->canon.psize.dst.psizemin;
   unsigned short dmiargusz2 = argus2->canon.psize.dst.psizemin;

   retn = ArgusReverseSortDir ? ((dmiargusz1 > dmiargusz2) ? 1 : ((dmiargusz1 == dmiargusz2) ? 0 : -1)) :
                                ((dmiargusz1 < dmiargusz2) ? 1 : ((dmiargusz1 == dmiargusz2) ? 0 : -1)) ;
   return (retn);
}

/*
struct ArgusCountryCodeStruct {
   struct ArgusDSRHeader hdr;
   char src[2], dst[2];
};
*/

int
ArgusSortSrcCountryCode (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   char *sco1 = argus1->canon.cocode.src;
   char *sco2 = argus2->canon.cocode.src;

   retn = strcmp(sco1, sco2);
                  
   retn = ArgusReverseSortDir ? ((retn < 0) ? 1 : ((retn == 0) ? 0 : -1)) : retn;
   return (retn);
}  

int
ArgusSortDstCountryCode (struct ArgusRecordStruct *argus1, struct ArgusRecordStruct *argus2)
{
   int retn = 0;
   char *sco1 = argus1->canon.cocode.dst;
   char *sco2 = argus2->canon.cocode.dst;

   retn = strcmp(sco1, sco2);
                  
   retn = ArgusReverseSortDir ? ((retn < 0) ? 1 : ((retn == 0) ? 0 : -1)) : retn;
   return (retn);
}
