/*
    Copyright (C) 2006 Fons Adriaensen
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <string.h>
#include <math.h>
#include "ambisonic2.h"


#define DEG2RAD 0.0174533f



void Ladspa_Monopan21::setport (unsigned long port, LADSPA_Data *data)
{
    _port [port] = data;
}


void Ladspa_Monopan21::active (bool act)
{
    if (act) calcpar ();
}


void Ladspa_Monopan21::calcpar (void)
{
    float e, a, ce; 

    e = *(_port [CTL_ELEV]) * DEG2RAD;
    _zz = sinf (e);
    ce = cosf (e);
    a = *(_port [CTL_AZIM]) * DEG2RAD;
    _xx = ce * cosf (-a);
    _yy = ce * sinf (-a);
    _uu = ce * cosf (-2 * a);
    _vv = ce * sinf (-2 * a);
}


void Ladspa_Monopan21::runproc (unsigned long len, bool add)
{
    float t, xx, yy, zz, uu, vv, dxx, dyy, dzz, duu, dvv; 
    float *in, *out_w, *out_x, *out_y, *out_z, *out_u, *out_v;

    xx = _xx;
    yy = _yy;
    zz = _zz;
    uu = _uu;
    vv = _vv;
    calcpar ();
    dxx = (_xx - xx) / len;
    dyy = (_yy - yy) / len;
    dzz = (_zz - zz) / len;
    duu = (_uu - uu) / len;
    dvv = (_vv - vv) / len;

    in = _port [INP];
    out_w = _port [OUT_W];
    out_x = _port [OUT_X];
    out_y = _port [OUT_Y];
    out_z = _port [OUT_Z];
    out_u = _port [OUT_U];
    out_v = _port [OUT_V];

    while (len--)
    {
	xx += dxx;
	yy += dyy;
	zz += dzz;
	uu += duu;
	vv += dvv;
        t = *in++;        
        *out_w++ = 0.7071f * t;
        *out_x++ = xx * t;
        *out_y++ = yy * t;
        *out_z++ = zz * t;
        *out_u++ = uu * t;
        *out_v++ = vv * t;
    }
}



void Ladspa_Rotator21::setport (unsigned long port, LADSPA_Data *data)
{
    _port [port] = data;
}


void Ladspa_Rotator21::active (bool act)
{
    if (act) calcpar ();
}


void Ladspa_Rotator21::calcpar (void)
{
    float a; 

    a = _port [CTL_AZIM][0] * DEG2RAD;
    _c1 = cosf (a);
    _s1 = sinf (a);
    _c2 = cosf (2 * a);
    _s2 = sinf (2 * a);
}


void Ladspa_Rotator21::runproc (unsigned long len, bool add)
{
    unsigned long k;
    float x, y;
    float c1, s1;
    float c2, s2;
    float dc, ds;
    float *in_a, *in_b; 
    float *out_a, *out_b;

    memcpy (_port [OUT_W], _port [INP_W], len * sizeof (float));
    memcpy (_port [OUT_Z], _port [INP_Z], len * sizeof (float));

    c1 = _c1;
    s1 = _s1;
    c2 = _c2;
    s2 = _s2;
    calcpar ();

    in_a  = _port [INP_X];
    in_b  = _port [INP_Y];
    out_a = _port [OUT_X];
    out_b = _port [OUT_Y];
    k = len;
    dc = (_c1 - c1) / len;
    ds = (_s1 - s1) / len;
    while (k--)
    {
	c1 += dc;
	s1 += ds;
        x = *in_a++;
        y = *in_b++; 
        *out_a++ = c1 * x + s1 * y;
        *out_b++ = c1 * y - s1 * x;
    }

    in_a  = _port [INP_U];
    in_b  = _port [INP_V];
    out_a = _port [OUT_U];
    out_b = _port [OUT_V];
    k = len;
    dc = (_c2 - c2) / len;
    ds = (_s2 - s2) / len;
    while (k--)
    {
	c2 += dc;
	s2 += ds;
        x = *in_a++;
        y = *in_b++; 
        *out_a++ = c2 * x + s2 * y;
        *out_b++ = c2 * y - s2 * x;
    }
}



