// -*- C++ -*- (c) 2007, 2008 Petr Rockai <me@mornfall.net>

#include <ept/progresscallback.h>
#include <adept/util.h>
#include <adept/processevents.h>

#include <apt-pkg/acquire-worker.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/acquire.h>
#include <apt-pkg/strutl.h> // for TimeToStr

#include <QCoreApplication>
#include <QDebug>
#include <QStandardItemModel>
#include <QItemDelegate>
#include <QTextDocument>
#include <QPainter>
#include <QLayout>
#include <KLocale>
#include <KIcon>

#include <QPushButton>
#include <QListView>
#include <QProgressBar>
#include <QLabel>
#include <KVBox>
#include <KHBox>

#ifndef ADEPT_DOWNLOADPROGRESS_H
#define ADEPT_DOWNLOADPROGRESS_H

inline bool operator<(
    const pkgAcquire::ItemDesc &a,
    const pkgAcquire::ItemDesc &b )
{
    return a.URI < b.URI;
}

namespace adept {

class DownloadProgressItem : public KHBox {
    Q_OBJECT
    QLabel *m_label;
    QProgressBar *m_bar;

public:
    DownloadProgressItem( QWidget *p = 0 )
        : KHBox( p )
    {
        m_label = new QLabel( this );
        m_bar = new QProgressBar( this );
        m_bar->setMaximum( 0 );
        layout()->setSpacing( 3 );
    }

    void setText( QString txt ) {
        m_label->setText( txt );
    }

    void setProgress( int prog = 0 ) {
        m_bar->setMaximum( 100 );
        m_bar->setValue( prog );
    }

    void spin() {
        m_bar->setValue( m_bar->value() + 1 );
    }
};

class RichTextDelegate : public QItemDelegate {
    Q_OBJECT
    QTextDocument *doc;
public:
    RichTextDelegate( QObject *p ) : QItemDelegate( p ) {
        doc = new QTextDocument( this );
    }
    virtual void paint( QPainter *painter,
                        const QStyleOptionViewItem &opt,
                        const QModelIndex &index ) const
    {
        painter->save();
        painter->translate( opt.rect.left(), opt.rect.top() );
        painter->setClipRect( 0, 0, opt.rect.width(), opt.rect.height() );
        doc->setHtml( index.data( Qt::DisplayRole ).toString() );
        doc->drawContents( painter );
        painter->restore();
    }
};

// TODO stack a media-change-needed screen below the progress view
// TODO apparently the map believes there is only one item at all
// times -- probably there's a problem with operator< somewhere we
// should also wait a little with removing progressbars, since it
// hoses scrolling in the list, and the whole thing is jumpy and
// blinky
class DownloadProgress : public KVBox, public ept::ProgressCallback {
    Q_OBJECT
protected:
    QListView *m_doneView;
    QStandardItemModel *m_doneModel;
    KVBox *m_current;
    QLabel *m_info;
    QProgressBar *m_total;
    QPushButton *m_cancel;
    std::map< pkgAcquire::ItemDesc, DownloadProgressItem * > m_currentMap;
    bool m_cancelled;

public:
    enum Status { StIdle, StWaiting, StDownloading, StDone };
    Status m_status;

    DownloadProgress( QWidget *p ) : KVBox( p ) {
        m_status = StIdle;
        m_doneView = new QListView( this );
        m_current = new KVBox( this );
        QWidget *totalBox = new KHBox( this );
        m_total = new QProgressBar( totalBox );
        m_cancel = new QPushButton( totalBox );
        m_cancel->setText( i18n( "Cancel" ) );
        m_cancel->setIcon( KIcon( "dialog-cancel" ) );
        connect( m_cancel, SIGNAL( clicked() ),
                 this, SLOT( requestCancel() ) );

        totalBox->layout()->setSpacing( 3 );
        layout()->setSpacing( 3 );
        setMargin( 10 );

        m_doneModel = new QStandardItemModel();
        m_doneView->setModel( m_doneModel );
        m_doneView->setItemDelegate( new RichTextDelegate( this ) );

        m_total->setMaximum( 100 );
        m_total->setValue( 0 );
    }

    void status( Status s ) {
        QCoreApplication::processEvents();
        if ( m_status == s )
            return;
        m_status = s;
        statusChanged( s );
        QCoreApplication::processEvents();
    }

    void doneItem( pkgAcquire::ItemDesc &i, QString status ) {
        removeFromCurrent( i );
        QStandardItem *n = new QStandardItem();
        n->setText( u8( i.Description ) + i18n( ": " ) + status );
        m_doneModel->appendRow( n );
        m_doneView->scrollTo( m_doneModel->indexFromItem( n ) );
        QCoreApplication::processEvents();
    }

    void removeFromCurrent( const pkgAcquire::ItemDesc &i ) {
        delete m_currentMap[ i ];
        m_currentMap.erase( i );
    }

    void fetchItem( pkgAcquire::ItemDesc &i ) {
        qDebug() << "fetching item: " << u8( i.URI );
        DownloadProgressItem *item;
        if ( !m_currentMap[ i ] ) {
            item = new DownloadProgressItem( m_current );
            m_currentMap[ i ] = item;
        } else
            item = m_currentMap[ i ];
        item->setText( u8( i.ShortDesc ) );
        status( StDownloading );
    }

    void clear() {
        m_doneModel->clear();
        while ( !m_currentMap.empty() ) {
            removeFromCurrent( m_currentMap.begin()->first );
        }
    }

    void updateProgress( pkgAcquire::Worker *w ) {
        DownloadProgressItem *item = m_currentMap[ *w->CurrentItem ];
        if ( !item )
            return; // a bug?
        if ( w->TotalSize )
            item->setProgress(
                long( double( w->CurrentSize * 100.0 )
                      / double( w->TotalSize ) ) );
        else
            item->spin();
    }

    virtual void IMSHit(pkgAcquire::ItemDesc &i) {
        doneItem( i, i18n(
                      "<span style=\"color: green;\">cached</span>" ) );
    }

    virtual void Fetch(pkgAcquire::ItemDesc &i) {
        fetchItem( i );
    }

    virtual void Done(pkgAcquire::ItemDesc &i) {
        doneItem( i, i18n( "<b style=\"color: green;\">OK</b>" ) );
    }

    virtual void Fail(pkgAcquire::ItemDesc &i) {
        // mysterious ignore due to bad code, comes from previous generation
        if (i.Owner->Status == pkgAcquire::Item::StatIdle)
            return;

        doneItem( i, i18n( "<b style=\"color: red;\">FAILED</b>" ) );
    }

    virtual void Start() {
        clear();
        m_cancelled = false;
        pkgAcquireStatus::Start();
        _config->Set( "APT::Fetcher::Select-Timeout-Usec", 100000 );
        status( StWaiting );
    }

    virtual void Stop() {
        pkgAcquireStatus::Stop();
        m_total->setFormat(
            i18n( "%p% [%1B downloaded]",
                  u8( SizeToStr( CurrentBytes ) ) ) );
        status( StDone );
    }

    bool Pulse( pkgAcquire *o ) {
        pkgAcquireStatus::Pulse( o );
        for ( pkgAcquire::Worker *w = o->WorkersBegin(); w != 0;
              w = o->WorkerStep( w ) ) {
            if ( w->CurrentItem ) {
                updateProgress( w );
            }
        }
        m_total->setMaximum( TotalBytes + TotalItems );
        m_total->setValue( CurrentBytes + CurrentItems );
        long eta = 0;
        if ( CurrentCPS )
            eta = long( (TotalBytes - CurrentBytes) / CurrentCPS );
        m_total->setFormat(
            i18n( "%p% [%1/%2B done, %3B/s] ETA: %4",
                  u8( SizeToStr( CurrentBytes ) ),
                  u8( SizeToStr( TotalBytes ) ),
                  u8( SizeToStr( CurrentCPS ) ),
                  eta ? u8( TimeToStr( eta ) ) : i18n( "no estimate" ) ) );
        ProcessEvents::process();
        return !m_cancelled;
    }

public Q_SLOTS:
    void requestCancel() {
        m_cancelled = true;
    }

Q_SIGNALS:
    void statusChanged( DownloadProgress::Status st );
};

}

#endif
