/* OrthogonalProjection.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.transformations;


/**
 * Orthogonal projection from a three-dimensional space
 * onto a 2-dimensional space.
 */
public class OrthogonalProjection extends AbstractTransformation {
    
    /**
     * Orthogonal matrix used by this transformation.
     */
    private double[][] matrix;
    
    /**
     * Create a projection along the Z-axis.
     */
    public OrthogonalProjection() {
        this.matrix = new double[3][3];
        for (int i=0; i < 3; i++)
            this.matrix[i][i] =1.0;
    }
    
    
    // implements Transformation
    @Override public double[] transform(double[] src, double[] dest) {
        if (src.length != 3)
            throw new IllegalArgumentException("Incorrect source dimension");
        dest[0] = src[0]*matrix[0][0] + src[1]*matrix[1][0]
                + src[2]*matrix[2][0];
        dest[1] = src[0]*matrix[0][1] + src[1]*matrix[1][1]
                + src[2]*matrix[2][1];
        return dest;
    }
    
    /**
     * Perform a rotation around the X-axis over the given angle.
     */
    public void rotateX(double angle) {
        if (angle == 0.0)
            return;
        else {
            rotateXImpl(angle);
            fireTransformationChanged();
        }
    }
    
    // performs the computations for rotateX, but does not signal observers
    private void rotateXImpl(double angle) {
        double[][] result = new double[3][3];
        
        for (int i=0; i < 3; i++) {
            result[i][0] = matrix[i][0];
            result[i][1] = matrix[i][1]*Math.cos(angle)
            - matrix[i][2]*Math.sin(angle);
            result[i][2] = matrix[i][1]*Math.sin(angle)
            + matrix[i][2]*Math.cos(angle);
        }
        matrix = result;
    }
    
    
    
    /**
     * Perform a rotation around the Y-axis over the given angle.
     */
    public void rotateY(double angle) {
        if (angle == 0.0)
            return;
        rotateYImpl(angle);
        fireTransformationChanged();
    }

    // performs the computations for rotateY, but does not signal observers
    private void rotateYImpl(final double angle) {
        double[][] result = new double[3][3];
        
        for (int i=0; i < 3; i++) {
            result[i][2] = matrix[i][2]*Math.cos(angle)
            - matrix[i][0]*Math.sin(angle);
            result[i][1] = matrix[i][1];
            result[i][0] = matrix[i][2]*Math.sin(angle)
            + matrix[i][0]*Math.cos(angle);
        }
        matrix = result;
    }
    
    /**
     * Perform a rotation around the Z-axis over the given angle.
     */
    public void rotateZ(double angle) {
        if (angle == 0.0)
            return;
        rotateZImpl(angle);
        fireTransformationChanged();
    }

    // performs the computations for rotateZ, but does not signal observers
    private void rotateZImpl(final double angle) {
        double[][] result = new double[3][3];
        
        for (int i=0; i < 3; i++) {
            result[i][0] = matrix[i][0]*Math.cos(angle)
            - matrix[i][1]*Math.sin(angle);
            result[i][1] = matrix[i][0]*Math.sin(angle)
            + matrix[i][1]*Math.cos(angle);
            result[i][2] = matrix[i][2];
        }
        matrix = result;
    }
    
    /**
     * Combines the effect of a rotation along the X-axis followed
     * by a rotation along the Y-axis.
     */
    public void rotateXY (double angleX, double angleY) {
        if (angleX == 0.0 && angleY == 0.0)
            return;
        else {
            rotateXImpl (angleX);
            rotateYImpl (angleY);
            fireTransformationChanged();
        }
    }
    
    /**
     * Zoom by the given factor in destination space.
     */
    public void zoom (double factor) {
        if (factor != 1.0) {
            for (int r=0; r < 3; r++)
                for (int c=0; c < 3; c++)
                    matrix[r][c] *= factor;
            fireTransformationChanged();
        }
    }
    
    // implements Transformation
    public int getDimension() {
        return 3;
    }
    
}
