// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_
#define CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_

#include <string>

#include "base/memory/scoped_refptr.h"
#include "base/memory/weak_ptr.h"
#include "base/strings/string16.h"
#include "build/build_config.h"
#include "content/common/content_export.h"
#include "mojo/public/cpp/bindings/binding.h"
#include "third_party/blink/public/mojom/loader/code_cache.mojom.h"
#include "third_party/blink/public/platform/modules/cache_storage/cache_storage.mojom.h"

class GURL;

namespace net {
class IOBuffer;
}

namespace url {
class Origin;
}

namespace content {

class CacheStorageContextImpl;
class CacheStorageCacheHandle;
class GeneratedCodeCache;
class GeneratedCodeCacheContext;

// The implementation of a CodeCacheHost, which stores and retrieves resource
// metadata, either bytecode or native code, generated by a renderer process.
// Instances of this class are owned by by the Mojo pipe that's passed to the
// renderer process via StrongBinding.
// Instances of this class must be created and used on the IO thread.
class CONTENT_EXPORT CodeCacheHostImpl : public blink::mojom::CodeCacheHost {
 public:
  CodeCacheHostImpl(
      int render_process_id,
      scoped_refptr<CacheStorageContextImpl> cache_storage_context,
      scoped_refptr<GeneratedCodeCacheContext> generated_code_cache_context);
  ~CodeCacheHostImpl() override;

  static void Create(
      int render_process_id,
      scoped_refptr<CacheStorageContextImpl> cache_storage_context,
      scoped_refptr<GeneratedCodeCacheContext> generated_code_cache_context,
      blink::mojom::CodeCacheHostRequest request);

 private:
  // blink::mojom::CodeCacheHost implementation.
  void DidGenerateCacheableMetadata(blink::mojom::CodeCacheType cache_type,
                                    const GURL& url,
                                    base::Time expected_response_time,
                                    const std::vector<uint8_t>& data) override;
  void FetchCachedCode(blink::mojom::CodeCacheType cache_type,
                       const GURL& url,
                       FetchCachedCodeCallback) override;
  void ClearCodeCacheEntry(blink::mojom::CodeCacheType cache_type,
                           const GURL& url) override;
  void DidGenerateCacheableMetadataInCacheStorage(
      const GURL& url,
      base::Time expected_response_time,
      const std::vector<uint8_t>& data,
      const url::Origin& cache_storage_origin,
      const std::string& cache_storage_cache_name) override;

  // Helpers.
  GeneratedCodeCache* GetCodeCache(blink::mojom::CodeCacheType cache_type);
  void OnReceiveCachedCode(FetchCachedCodeCallback callback,
                           const base::Time& response_time,
                           const std::vector<uint8_t>& data);
  void OnCacheStorageOpenCallback(const GURL& url,
                                  base::Time expected_response_time,
                                  scoped_refptr<net::IOBuffer> buf,
                                  int buf_len,
                                  CacheStorageCacheHandle cache_handle,
                                  blink::mojom::CacheStorageError error);
  static void DidGenerateCacheableMetadataOnUI(
      int render_process_id,
      const GURL& url,
      base::Time expected_response_time,
      const std::vector<uint8_t>& data);

  // Our render process host ID, used to bind to the correct render process.
  const int render_process_id_;

  scoped_refptr<CacheStorageContextImpl> cache_storage_context_;

  scoped_refptr<GeneratedCodeCacheContext> generated_code_cache_context_;

  base::WeakPtrFactory<CodeCacheHostImpl> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(CodeCacheHostImpl);
};

}  // namespace content

#endif  // CONTENT_BROWSER_RENDERER_HOST_CODE_CACHE_HOST_IMPL_H_
