/*****************************************************************************
 *** sujoin.h
 ***
 *** Copyright (c) 1998 Susanne Cech <scech@edu.uni-klu.ac.at>
 ***
 *** Requires the Qt widget libraries, available at no cost at
 *** http://www.troll.no/
 ***
 ***  This program is free software; you can redistribute it and/or modify
 ***  it under the terms of the GNU General Public License as published by
 ***  the Free Software Foundation; either version 2 of the License, or
 ***  (at your option) any later version.
 ***
 ***  This program is distributed in the hope that it will be useful,
 ***  but WITHOUT ANY WARRANTY; without even the implied warranty of
 ***  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ***  GNU General Public License for more details.
 ***
 ***  You should have received a copy of the GNU General Public License
 ***  along with this program; if not, write to the Free Software
 ***  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ***
 *****************************************************************************/

#ifndef __SUJOIN_H__
#define __SUJOIN_H__

#include "serialize.h"
#include "kpvm.h"
#include <qstring.h>
#include <qlist.h>
#include <qfile.h>
#include <qtstream.h>
#include <stdlib.h>

#include <qpushbt.h>
#include <qfont.h>
#include <qlcdnum.h>
#include <qgrpbox.h>
#include <qdatetm.h>
                         
#define NRPROCS  4
#define PA_FILE  "dat/pa.dat"
#define BT_FILE  "dat/me_pa.dat"
#define JOIN_RES "dat/res.dat"


/*****************************************************************************/
/********************   CLASS JoinAttrib   ***********************************/
/*****************************************************************************/

/**
  * JoinAttrib is the Superclass of my "parallel database". 
  * It is a sendable structured OO-Data-Type, holding the join-attribute
  * for joining the two subclasses.
  * It inherits from Serialize to be a pvm_sendable object.
  *
  * @short is the Superclass of my "parallel database". 
  * @author Susanne Cech <scech@edu.uni-klu.ac.at
 **/
class JoinAttrib : public Serialize {
Q_OBJECT

 private: 
   int _codePA;

 public:
/** 
  * to enable SuKPvm to call JoinAttrib's  private methods
 **/
   friend class SuKPvm;
   JoinAttrib( Serialize* parent=0, const char* name=0);   // Constructor
   ~JoinAttrib() {};
   int codePA() { return _codePA; };

 public slots:
   void setCodePA( int codePA );

 signals:
   void codePAChanged();

 protected:
   /**
     * Overwrites the virtual method of @ref Serialize,
     * saves the object-data (codePA) into QList of VarTuples.
    **/
   QList<VarTuple> objState() const;

   /**
     * Overwrites the virtual method of @ref Serialize.
     * sets object's state with given QList of VarTuples,
     * has to be in same order as given by @ref objState.
    **/
   void setFixedOrderObjState( QList<VarTuple> state);

   /**
     * Overwrites the virtual method of @ref Serialize,
     * sets object's state with given QList of VarTuples, 
     * attribute may miss or can be in another order.
    **/
   void setObjState( QList<VarTuple> state);
};


/*****************************************************************************/
/********************   CLASS PrincipeActif   *******************************/
/*****************************************************************************/

/** 
  * PrincipeActif is the Subclass of @ref JoinAttrib.
  * Further attribute: name 
  *
  * @short is the Subclass of @ref JoinAttrib.
  * @author Susanne Cech <scech@edu.uni-klu.ac.at>
 **/
class PrincipeActif : virtual public JoinAttrib {
Q_OBJECT

 private:
   QString _name;

 public: 
/** 
  * to enable SuKPvm to call PrincipeActif's  private methods
 **/
   friend class SuKPvm;
   PrincipeActif( JoinAttrib* parent=0, const char* name=0);
   ~PrincipeActif() {};
   QString name() { return _name; };

 public slots:
   void setName( QString name );

 signals:
   void nameChanged();

 protected:
   /**
     * Overwrites the virtual method of @ref JoinAttrib.
     * Saves the object-data (codePA,name) into QList of VarTuples.
    **/
   QList<VarTuple> objState() const;

   /**
     * Overwrites the virtual method of @ref JoinAttrib.
     * Sets object's state with given QList of VarTuples,
     * has to be in same order as given by @ref objState.
    **/
   void setFixedOrderObjState( QList<VarTuple> state);

   /**
     * Overwrites the virtual method of @ref JoinAttrib,
     * sets object's state with given QList of VarTuples, 
     * attributes may miss or can be in another order.
    **/
   void setObjState( QList<VarTuple> state);
};

 
/*****************************************************************************/
/********************   CLASS BelongsTo   ***********************************/
/*****************************************************************************/

/** 
  * BelongsTo is the Subclass of @ref JoinAttrib.
  * Further attribute: codeMed 
  *
  * @short is the Subclass of @ref JoinAttrib.
  * @author Susanne Cech <scech@edu.uni-klu.ac.at>
 **/
class BelongsTo : virtual public JoinAttrib {
Q_OBJECT

 private:
   int _codeMed;

 public:  
  /** 
    * to enable SuKPvm to call Belong_to's  private methods
   **/
  friend class SuKPvm;
   BelongsTo( JoinAttrib* parent=0, const char* name=0);
   ~BelongsTo() {};
   int codeMed() { return _codeMed; };

 public slots:
   void setCodeMed( int codeMed );

 signals:
   void codeMedChanged();

 protected:
   /**
     * Overwrites the virtual method of @ref JoinAttrib.
     * Saves the object-data (codePA,codeMed) into QList of VarTuples.
    **/
   QList<VarTuple> objState() const;

   /**
     * Overwrites the virtual method of @ref JoinAttrib.
     * Sets object's state with given QList of VarTuples,
     * has to be in same order as given by @ref objState.
    **/
   void setFixedOrderObjState( QList<VarTuple> state);

   /**
     * Overwrites the virtual method of @ref JoinAttrib,
     * sets object's state with given QList of VarTuples, 
     * attributes may miss or can be in another order.
    **/
   void setObjState( QList<VarTuple> state);
};


/*****************************************************************************/
/********************   CLASS JoinRes   *************************************/
/*****************************************************************************/

/** 
  * JoinRes is the Subclass of @ref PrincipeActif and @ref BelongsTo
  * , merging all resulting attributes together.
  * 
  * @short is the result of the join of @ref PrincipeActif and @ref BelongsTo
  * @author Susanne Cech <scech@edu.uni-klu.ac.at>
 **/
class JoinRes : virtual public PrincipeActif, virtual public BelongsTo {
Q_OBJECT

 public: 
  /** 
    * to enable SuKPvm to call JoinRes's private methods
   **/
   friend class SuKPvm;
   JoinRes( JoinAttrib* parent=0, const char* name=0);
   ~JoinRes() {};

 protected:
   /**
     * Overwrites the virtual methods of @ref PrincipeActif and @ref 
     * BelongsTo. Saves the object-data (codePA, name, codeMed) into 
     * QList of VarTuples.
    **/
   QList<VarTuple> objState() const;

   /**
     * Overwrites the virtual methods of @ref PrincipeActif and @ref 
     * BelongsTo. 
     * Sets object's state with given QList of VarTuples,
     * has to be in same order as given by @ref objState.
    **/
   void setFixedOrderObjState( QList<VarTuple> state);

   /**
     * Overwrites the virtual methods of @ref PrincipeActif and @ref 
     * BelongsTo.
     * Sets object's state with given QList of VarTuples, 
     * attributes may miss or can be in another order.
    **/
   void setObjState( QList<VarTuple> state);
};


/*****************************************************************************/
/********************   CLASS TransferComplete   ****************************/
/*****************************************************************************/

/** 
  * TransferComplete is only needed to indicate the end of a pvm-session.
  * It inherits from Serialize to be a pvm_sendable object, but
  * implements useless virtuell methods with empty bodies.
  *
  * @short is only needed to indicate the end of a pvm-session.
  * @author Susanne Cech <scech@edu.uni-klu.ac.at>
 **/
class TransferComplete : public Serialize { 
Q_OBJECT

  protected:
    QList<VarTuple> objState() const { QList<VarTuple> ql; return ql; };
    void setFixedOrderObjState( QList<VarTuple> ) {};
    void setObjState( QList<VarTuple> ) {};
};


/*****************************************************************************/
/********************   CLASS SuKPvm   ***************************************/
/*****************************************************************************/

/**
  * @short is the main pvm-class.
  * SuKPvm is the main pvm-class. It inherits from KPvm.
  * @author Susanne Cech <scech@edu.uni-klu.ac.at>
 **/ 
class SuKPvm : public KPvm {
 Q_OBJECT
  
  private:
    /**
      * computes hash-function from given hash-value, is needed to
      * distribute objects.
     **/ 
    int hash( int);
    /**
      * creates the lcd-counters and button
     **/
    void guiInit();
    /** 
      * sends all @ref PrincipeActif-tuples to the children, distributed
      * to the several children with @ref hash-method.
     **/
    void sendPrincipeActif();
    /** 
      * sends all @ref BelongsTo-tuples to the children, distributed
      * to the several children with @ref hash-method.
     **/
    void sendBelongsTo();
    /**
      * contains all spawned @ref KPvmChild-objects, QList of NRPROCS 
      * elements.
     **/
    QList<KPvmChild> children;
    /**
      * Dictionary is the local hash-table predefined for 997 elements.
     **/ 
    QDict<PrincipeActif> *_pa;
    /**
      * Filehandles for result of join.
     **/
    QFile fout;
    QFile fin;
    /**
      * TextStream.for for reading and writing on filehandle.
     **/
    QTextStream tstream;
    /**
      * needed to end all children correctly.
     **/    
    int runProcs, i;
    QString s; 
    QPushButton *quit;
    QLCDNumber  *paLCD, *btLCD, *jrLCD, *timeLCD;
    QGroupBox   *paBOX, *btBOX, *jrBOX;
    QTime       *time;
   
  public:
   /**
     * It is distincted between parent- and child part. The parent sends
     * tuples of @ref PrincipeActif and @ref BelongsTo to his spawned 
     * children, which the children receive and process.
    **/ 
    SuKPvm( KPvm* parent=0, const char* name=0);
    ~SuKPvm() {};

  private slots:
   /**
     * is a CHILD-SLOT for the dataReceived-SIGNAL. It receives the Pricipe_
     * Actif-tuples and builds the local hash-table.
     *
     * When finished - got object of class @ref TransferComplete -, it
     * disconnects the dataReceived-SIGNAL to connect it with @ref 
     * childHandleRecvBt to be able to receive @ref BelongsTo-tuples.
    **/  
    void childHandleRecvPA( QString className, QList<VarTuple> objState,
        int msgId, KPvmEntity* entity);

   /**
     * is a CHILD-SLOT for the dataReceived-SIGNAL. It receives the
     * @ref BelongsTo tuples and probes them against the local hash-table.
     *
     * In case of a match it sends the the correspondig @ref JoinRes-tuple
     * back to parent.
    **/
    void childHandleRecvBt( QString className, QList<VarTuple> objState,
        int msgId, KPvmEntity* entity);

   /**
     * is the PARENT-SLOT for the dataReceived-SIGNAL. Is is connected with 
     * the signal after the @ref PrincipeActif-tuples have been sent but 
     * before the parent starts to send the @ref BelongsTo-tuples.
     *
     * It receives the @ref JoinRes-tuples and stores them on harddisk.
    **/
    void parentHandleReceivedData( QString className, QList<VarTuple> objState,
        int msgId, KPvmEntity* entity); 

  protected:
    void closeEvent( QCloseEvent *);
};

#endif
