/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "klradiobutton.h"

static QWidget *dummy = 0; // QRadioButton needs a parent
static ulong dummycount = 0;

KLRadioButton::KLRadioButton(const char *text,KRadioManager *rm, long id) : QRadioButton(dummy,0)
{
  setKLWidget(this); 
  setText(text);
  setWeight(1);
  if(rm)
    rm->insert(this,id);
}

bool KLRadioButton::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  if( !KLWidgetBase::klAskMinMax(minMaxSizes) )
    return false;
  QSize size = sizeHint();
  minMaxSizes->minX += size.height();
  minMaxSizes->minY += size.width();
  minMaxSizes->defX += size.height();
  minMaxSizes->defY += size.width();
  minMaxSizes->maxX += size.height();
  minMaxSizes->maxY += size.width() + KLMaxSize;
  return true;
}

void KLRadioButton::setText(const char *text)
{
  QRadioButton::setText(text);
  doRelayout(false);
}

///////////////////////////////////////////////////////////////
// RadioManager
///////////////////////////////////////////////////////////////
void KRadioManager::insert(QRadioButton *button,long id )
{
  if( id == -1 ) 
    id = buttons.count();
  KRMButton *butinfo = new KRMButton();
  butinfo->button = button;
  butinfo->id = id;
  buttons.append(butinfo);
  connect(button,SIGNAL(pressed()),this,SLOT(buttonPressed()));
  connect(button,SIGNAL(released()),this,SLOT(buttonReleased()));
  connect(button,SIGNAL(clicked()),this,SLOT(buttonClicked()));
}


void KRadioManager::remove(QRadioButton *button)
{
  for( buttons.first(); buttons.current(); buttons.next()){
    if( buttons.current()->button == button ) {
      buttons.current()->button->disconnect(this);
      KRMButton *butinfo = buttons.current();
      buttons.remove();
      delete butinfo;
      return;
    }
  } // for
}


QRadioButton *KRadioManager::find(long id)
{
  for( buttons.first(); buttons.current(); buttons.next()){
    if( buttons.current()->id == id) 
      return buttons.current()->button;
  }
  return 0;
}


KRadioManager::KRMButton *KRadioManager::find(QRadioButton *but)
{
  for( buttons.first(); buttons.current(); buttons.next()){
    if( buttons.current()->button == but ) 
      return buttons.current();
  }
  return 0;
}


void KRadioManager::buttonPressed()
{
  KRMButton *butinfo = find((QRadioButton *)sender());
  if( !butinfo )
    return;
  emit pressed(butinfo->id);
}

void KRadioManager::buttonReleased()
{
  KRMButton *butinfo = find((QRadioButton *)sender());
  if( !butinfo )
    return;
  emit released(butinfo->id);
}

void KRadioManager::buttonClicked()
{
  KRMButton *butinfo = find((QRadioButton *)sender());
  if( !butinfo )
    return;
  setChecked(butinfo->button);
  emit clicked(butinfo->id);
}


KRadioManager &KRadioManager::operator<<(QRadioButton* but)
{
  insert(but);
  return *this;
}

KRadioManager::KRadioManager()
{
  if( !dummy )
    dummy = new QWidget(0,0);
  dummycount++;
  a_checked = 0;
}

KRadioManager::~KRadioManager()
{
  if( dummy ) {
    if( (dummycount--) == 0 ) {
      delete dummy;
      dummy = 0;
    }
  }
}


void KRadioManager::setChecked(long id)
{
  QRadioButton *button = find(id);
  if( !button )
    return;
  setChecked(button);
}

void KRadioManager::setChecked(QRadioButton *button)
{
  for( buttons.first(); buttons.current(); buttons.next()){
    if( buttons.current()->button != button ) 
      buttons.current()->button->setChecked(false);
    else
      button->setChecked(true);
  }
  a_checked = find(button)->id;
}

ulong KRadioManager::checked()
{
  return a_checked;
}

#include "klradiobutton.moc"
