/* valareport.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valareport.h>
#include <stdio.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefile.h>
#include <vala/valacodecontext.h>




struct _ValaReportPrivate {
	gint warnings;
	gint errors;
	gboolean verbose_errors;
};

#define VALA_REPORT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_REPORT, ValaReportPrivate))
enum  {
	VALA_REPORT_DUMMY_PROPERTY
};
static void vala_report_report_source (ValaSourceReference* source);
static void vala_report_real_warn (ValaReport* self, ValaSourceReference* source, const char* message);
static void vala_report_real_err (ValaReport* self, ValaSourceReference* source, const char* message);
static gpointer vala_report_parent_class = NULL;
static void vala_report_finalize (GObject* obj);



/**
 * Set the error verbosity.
 */
void vala_report_set_verbose_errors (ValaReport* self, gboolean verbose) {
	g_return_if_fail (self != NULL);
	self->priv->verbose_errors = verbose;
}


/**
 * Returns the total number of warnings reported.
 */
gint vala_report_get_warnings (ValaReport* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->warnings;
}


/**
 * Returns the total number of errors reported.
 */
gint vala_report_get_errors (ValaReport* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->errors;
}


/**
 * Pretty-print the actual line of offending code if possible.
 */
static void vala_report_report_source (ValaSourceReference* source) {
	char* offending_line;
	g_return_if_fail (source != NULL);
	if (vala_source_reference_get_first_line (source) != vala_source_reference_get_last_line (source)) {
		/* FIXME Cannot report multi-line issues currently*/
		return;
	}
	offending_line = vala_source_file_get_source_line (vala_source_reference_get_file (source), vala_source_reference_get_first_line (source));
	if (offending_line != NULL) {
		gint idx;
		fprintf (stderr, "%s\n", offending_line);
		idx = 0;
		/* We loop in this manner so that we don't fall over on differing
		 * tab widths. This means we get the ^s in the right places.
		 */
		for (idx = 1; idx < vala_source_reference_get_first_column (source); idx = idx + 1) {
			if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
				fprintf (stderr, "\t");
			} else {
				fprintf (stderr, " ");
			}
		}
		for (idx = vala_source_reference_get_first_column (source); idx <= vala_source_reference_get_last_column (source); idx = idx + 1) {
			if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
				fprintf (stderr, "\t");
			} else {
				fprintf (stderr, "^");
			}
		}
		fprintf (stderr, "\n");
	}
	offending_line = (g_free (offending_line), NULL);
}


/**
 * Reports the specified message as warning.
 *
 * @param source  reference to source code
 * @param message warning message
 */
static void vala_report_real_warn (ValaReport* self, ValaSourceReference* source, const char* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (message != NULL);
	self->priv->warnings++;
	if (source == NULL) {
		fprintf (stderr, "warning: %s\n", message);
	} else {
		char* _tmp0;
		_tmp0 = NULL;
		fprintf (stderr, "%s: warning: %s\n", _tmp0 = vala_source_reference_to_string (source), message);
		_tmp0 = (g_free (_tmp0), NULL);
		if (self->priv->verbose_errors) {
			vala_report_report_source (source);
		}
	}
}


void vala_report_warn (ValaReport* self, ValaSourceReference* source, const char* message) {
	VALA_REPORT_GET_CLASS (self)->warn (self, source, message);
}


/**
 * Reports the specified message as error.
 *
 * @param source  reference to source code
 * @param message error message
 */
static void vala_report_real_err (ValaReport* self, ValaSourceReference* source, const char* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (message != NULL);
	self->priv->errors++;
	if (source == NULL) {
		fprintf (stderr, "error: %s\n", message);
	} else {
		char* _tmp0;
		_tmp0 = NULL;
		fprintf (stderr, "%s: error: %s\n", _tmp0 = vala_source_reference_to_string (source), message);
		_tmp0 = (g_free (_tmp0), NULL);
		if (self->priv->verbose_errors) {
			vala_report_report_source (source);
		}
	}
}


void vala_report_err (ValaReport* self, ValaSourceReference* source, const char* message) {
	VALA_REPORT_GET_CLASS (self)->err (self, source, message);
}


/* Convenience methods calling warn and err on correct instance */
void vala_report_warning (ValaSourceReference* source, const char* message) {
	ValaCodeContext* _tmp0;
	g_return_if_fail (message != NULL);
	_tmp0 = NULL;
	vala_report_warn (vala_code_context_get_report (_tmp0 = vala_code_context_get ()), source, message);
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_context_unref (_tmp0), NULL));
}


void vala_report_error (ValaSourceReference* source, const char* message) {
	ValaCodeContext* _tmp0;
	g_return_if_fail (message != NULL);
	_tmp0 = NULL;
	vala_report_err (vala_code_context_get_report (_tmp0 = vala_code_context_get ()), source, message);
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_context_unref (_tmp0), NULL));
}


/**
 * Namespace to centralize reporting warnings and errors.
 */
ValaReport* vala_report_construct (GType object_type) {
	ValaReport * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


ValaReport* vala_report_new (void) {
	return vala_report_construct (VALA_TYPE_REPORT);
}


static void vala_report_class_init (ValaReportClass * klass) {
	vala_report_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaReportPrivate));
	G_OBJECT_CLASS (klass)->finalize = vala_report_finalize;
	VALA_REPORT_CLASS (klass)->warn = vala_report_real_warn;
	VALA_REPORT_CLASS (klass)->err = vala_report_real_err;
}


static void vala_report_instance_init (ValaReport * self) {
	self->priv = VALA_REPORT_GET_PRIVATE (self);
}


static void vala_report_finalize (GObject* obj) {
	ValaReport * self;
	self = VALA_REPORT (obj);
	G_OBJECT_CLASS (vala_report_parent_class)->finalize (obj);
}


GType vala_report_get_type (void) {
	static GType vala_report_type_id = 0;
	if (vala_report_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaReportClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_report_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaReport), 0, (GInstanceInitFunc) vala_report_instance_init, NULL };
		vala_report_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaReport", &g_define_type_info, 0);
	}
	return vala_report_type_id;
}




