/* valablock.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valablock.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valalocalvariable.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valastatementlist.h>
#include <vala/valamethod.h>
#include <vala/valascope.h>
#include <vala/valareport.h>




struct _ValaBlockPrivate {
	gboolean _contains_jump_statement;
	GeeList* statement_list;
	GeeList* local_variables;
};

#define VALA_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BLOCK, ValaBlockPrivate))
enum  {
	VALA_BLOCK_DUMMY_PROPERTY
};
static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_block_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_block_parent_class = NULL;
static ValaStatementIface* vala_block_vala_statement_parent_iface = NULL;
static void vala_block_finalize (ValaCodeNode* obj);



/**
 * Creates a new block.
 *
 * @param source reference to source code
 */
ValaBlock* vala_block_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaBlock* self;
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaBlock*) vala_symbol_construct (object_type, NULL, source_reference);
	return self;
}


ValaBlock* vala_block_new (ValaSourceReference* source_reference) {
	return vala_block_construct (VALA_TYPE_BLOCK, source_reference);
}


/**
 * Append a statement to this block.
 *
 * @param stmt a statement
 */
void vala_block_add_statement (ValaBlock* self, ValaStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
	gee_collection_add ((GeeCollection*) self->priv->statement_list, stmt);
}


void vala_block_insert_statement (ValaBlock* self, gint index, ValaStatement* stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
	gee_list_insert (self->priv->statement_list, index, stmt);
}


/**
 * Returns a copy of the list of statements.
 *
 * @return statement list
 */
GeeList* vala_block_get_statements (ValaBlock* self) {
	GeeArrayList* list;
	g_return_val_if_fail (self != NULL, NULL);
	list = gee_array_list_new (VALA_TYPE_STATEMENT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	{
		GeeIterator* _stmt_it;
		_stmt_it = gee_iterable_iterator ((GeeIterable*) self->priv->statement_list);
		while (gee_iterator_next (_stmt_it)) {
			ValaStatement* stmt;
			ValaStatementList* _tmp1;
			ValaStatement* _tmp0;
			ValaStatementList* stmt_list;
			stmt = (ValaStatement*) gee_iterator_get (_stmt_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			stmt_list = (_tmp1 = (_tmp0 = stmt, VALA_IS_STATEMENT_LIST (_tmp0) ? ((ValaStatementList*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			if (stmt_list != NULL) {
				{
					gint i;
					i = 0;
					for (; i < vala_statement_list_get_length (stmt_list); i++) {
						ValaStatement* _tmp2;
						_tmp2 = NULL;
						gee_collection_add ((GeeCollection*) list, _tmp2 = vala_statement_list_get (stmt_list, i));
						(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
					}
				}
			} else {
				gee_collection_add ((GeeCollection*) list, stmt);
			}
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
			(stmt_list == NULL) ? NULL : (stmt_list = (vala_code_node_unref (stmt_list), NULL));
		}
		(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
	}
	return (GeeList*) list;
}


/**
 * Add a local variable to this block.
 *
 * @param decl a variable declarator
 */
void vala_block_add_local_variable (ValaBlock* self, ValaLocalVariable* local) {
	ValaSymbol* _tmp0;
	ValaSymbol* parent_block;
	g_return_if_fail (self != NULL);
	g_return_if_fail (local != NULL);
	_tmp0 = NULL;
	parent_block = (_tmp0 = vala_symbol_get_parent_symbol ((ValaSymbol*) self), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
	while (TRUE) {
		gboolean _tmp1;
		ValaSymbol* _tmp2;
		gboolean _tmp3;
		ValaSymbol* _tmp6;
		ValaSymbol* _tmp5;
		_tmp1 = FALSE;
		if (VALA_IS_BLOCK (parent_block)) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = VALA_IS_METHOD (parent_block);
		}
		if (!_tmp1) {
			break;
		}
		_tmp2 = NULL;
		if ((_tmp3 = (_tmp2 = vala_scope_lookup (vala_symbol_get_scope (parent_block), vala_symbol_get_name ((ValaSymbol*) local))) != NULL, (_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL)), _tmp3)) {
			char* _tmp4;
			_tmp4 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) local), _tmp4 = g_strdup_printf ("Local variable `%s' conflicts with another local variable declared in a parent scope", vala_symbol_get_name ((ValaSymbol*) local)));
			_tmp4 = (g_free (_tmp4), NULL);
			break;
		}
		_tmp6 = NULL;
		_tmp5 = NULL;
		parent_block = (_tmp6 = (_tmp5 = vala_symbol_get_parent_symbol (parent_block), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5)), (parent_block == NULL) ? NULL : (parent_block = (vala_code_node_unref (parent_block), NULL)), _tmp6);
	}
	gee_collection_add ((GeeCollection*) self->priv->local_variables, local);
	(parent_block == NULL) ? NULL : (parent_block = (vala_code_node_unref (parent_block), NULL));
}


void vala_block_remove_local_variable (ValaBlock* self, ValaLocalVariable* local) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (local != NULL);
	gee_collection_remove ((GeeCollection*) self->priv->local_variables, local);
}


/**
 * Returns a copy of the list of local variables.
 *
 * @return variable declarator list
 */
GeeList* vala_block_get_local_variables (ValaBlock* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->local_variables);
}


static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = (ValaBlock*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_block (visitor, self);
}


static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = (ValaBlock*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _stmt_it;
		_stmt_it = gee_iterable_iterator ((GeeIterable*) self->priv->statement_list);
		while (gee_iterator_next (_stmt_it)) {
			ValaStatement* stmt;
			stmt = (ValaStatement*) gee_iterator_get (_stmt_it);
			vala_code_node_accept ((ValaCodeNode*) stmt, visitor);
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
		}
		(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
	}
}


static gboolean vala_block_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaBlock * self;
	ValaSymbol* _tmp1;
	ValaSymbol* old_symbol;
	ValaBlock* _tmp2;
	ValaBlock* old_insert_block;
	ValaBlock* _tmp4;
	ValaBlock* _tmp3;
	ValaBlock* _tmp10;
	ValaBlock* _tmp9;
	gboolean _tmp11;
	self = (ValaBlock*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_symbol_set_owner ((ValaSymbol*) self, vala_symbol_get_scope (vala_semantic_analyzer_get_current_symbol (analyzer)));
	_tmp1 = NULL;
	old_symbol = (_tmp1 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp2 = NULL;
	old_insert_block = (_tmp2 = analyzer->insert_block, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	_tmp4 = NULL;
	_tmp3 = NULL;
	analyzer->insert_block = (_tmp4 = (_tmp3 = self, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp4);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->statement_list); i++) {
			ValaStatement* _tmp5;
			_tmp5 = NULL;
			vala_code_node_check ((ValaCodeNode*) (_tmp5 = (ValaStatement*) gee_list_get (self->priv->statement_list, i)), analyzer);
			(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		}
	}
	{
		GeeList* _tmp6;
		GeeIterator* _tmp7;
		GeeIterator* _local_it;
		_tmp6 = NULL;
		_tmp7 = NULL;
		_local_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_block_get_local_variables (self))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			vala_symbol_set_active ((ValaSymbol*) local, FALSE);
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	{
		GeeIterator* _stmt_it;
		_stmt_it = gee_iterable_iterator ((GeeIterable*) self->priv->statement_list);
		while (gee_iterator_next (_stmt_it)) {
			ValaStatement* stmt;
			GeeList* _tmp8;
			stmt = (ValaStatement*) gee_iterator_get (_stmt_it);
			_tmp8 = NULL;
			vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp8 = vala_code_node_get_error_types ((ValaCodeNode*) stmt));
			(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
			(stmt == NULL) ? NULL : (stmt = (vala_code_node_unref (stmt), NULL));
		}
		(_stmt_it == NULL) ? NULL : (_stmt_it = (gee_collection_object_unref (_stmt_it), NULL));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	_tmp10 = NULL;
	_tmp9 = NULL;
	analyzer->insert_block = (_tmp10 = (_tmp9 = old_insert_block, (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9)), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp10);
	return (_tmp11 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_insert_block == NULL) ? NULL : (old_insert_block = (vala_code_node_unref (old_insert_block), NULL)), _tmp11);
}


void vala_block_insert_before (ValaBlock* self, ValaStatement* stmt, ValaStatement* new_stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (stmt != NULL);
	g_return_if_fail (new_stmt != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->statement_list); i++) {
			ValaStatement* _tmp0;
			ValaStatementList* stmt_list;
			_tmp0 = NULL;
			stmt_list = (_tmp0 = (ValaStatement*) gee_list_get (self->priv->statement_list, i), VALA_IS_STATEMENT_LIST (_tmp0) ? ((ValaStatementList*) _tmp0) : NULL);
			if (stmt_list != NULL) {
				{
					gint j;
					j = 0;
					for (; j < vala_statement_list_get_length (stmt_list); j++) {
						ValaStatement* _tmp1;
						gboolean _tmp2;
						_tmp1 = NULL;
						if ((_tmp2 = (_tmp1 = vala_statement_list_get (stmt_list, j)) == stmt, (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)), _tmp2)) {
							vala_statement_list_insert (stmt_list, j, new_stmt);
							break;
						}
					}
				}
			} else {
				ValaStatement* _tmp3;
				gboolean _tmp4;
				_tmp3 = NULL;
				if ((_tmp4 = (_tmp3 = (ValaStatement*) gee_list_get (self->priv->statement_list, i)) == stmt, (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)), _tmp4)) {
					ValaStatementList* _tmp5;
					_tmp5 = NULL;
					stmt_list = (_tmp5 = vala_statement_list_new (vala_code_node_get_source_reference ((ValaCodeNode*) self)), (stmt_list == NULL) ? NULL : (stmt_list = (vala_code_node_unref (stmt_list), NULL)), _tmp5);
					vala_statement_list_add (stmt_list, new_stmt);
					vala_statement_list_add (stmt_list, stmt);
					gee_list_set (self->priv->statement_list, i, (ValaStatement*) stmt_list);
				}
			}
			(stmt_list == NULL) ? NULL : (stmt_list = (vala_code_node_unref (stmt_list), NULL));
		}
	}
}


void vala_block_replace_statement (ValaBlock* self, ValaStatement* old_stmt, ValaStatement* new_stmt) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (old_stmt != NULL);
	g_return_if_fail (new_stmt != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->statement_list); i++) {
			ValaStatement* _tmp0;
			ValaStatementList* stmt_list;
			_tmp0 = NULL;
			stmt_list = (_tmp0 = (ValaStatement*) gee_list_get (self->priv->statement_list, i), VALA_IS_STATEMENT_LIST (_tmp0) ? ((ValaStatementList*) _tmp0) : NULL);
			if (stmt_list != NULL) {
				{
					gint j;
					j = 0;
					for (; j < vala_statement_list_get_length (stmt_list); j++) {
						ValaStatement* _tmp1;
						gboolean _tmp2;
						_tmp1 = NULL;
						if ((_tmp2 = (_tmp1 = vala_statement_list_get (stmt_list, j)) == old_stmt, (_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL)), _tmp2)) {
							vala_statement_list_set (stmt_list, j, new_stmt);
							break;
						}
					}
				}
			} else {
				ValaStatement* _tmp3;
				gboolean _tmp4;
				_tmp3 = NULL;
				if ((_tmp4 = (_tmp3 = (ValaStatement*) gee_list_get (self->priv->statement_list, i)) == old_stmt, (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)), _tmp4)) {
					gee_list_set (self->priv->statement_list, i, new_stmt);
					(stmt_list == NULL) ? NULL : (stmt_list = (vala_code_node_unref (stmt_list), NULL));
					break;
				}
			}
			(stmt_list == NULL) ? NULL : (stmt_list = (vala_code_node_unref (stmt_list), NULL));
		}
	}
}


gboolean vala_block_get_contains_jump_statement (ValaBlock* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_contains_jump_statement;
}


void vala_block_set_contains_jump_statement (ValaBlock* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_contains_jump_statement = value;
}


static void vala_block_class_init (ValaBlockClass * klass) {
	vala_block_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_block_finalize;
	g_type_class_add_private (klass, sizeof (ValaBlockPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_block_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_block_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_block_real_check;
}


static void vala_block_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_block_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_block_instance_init (ValaBlock * self) {
	self->priv = VALA_BLOCK_GET_PRIVATE (self);
	self->priv->statement_list = (GeeList*) gee_array_list_new (VALA_TYPE_STATEMENT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->local_variables = (GeeList*) gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_block_finalize (ValaCodeNode* obj) {
	ValaBlock * self;
	self = VALA_BLOCK (obj);
	(self->priv->statement_list == NULL) ? NULL : (self->priv->statement_list = (gee_collection_object_unref (self->priv->statement_list), NULL));
	(self->priv->local_variables == NULL) ? NULL : (self->priv->local_variables = (gee_collection_object_unref (self->priv->local_variables), NULL));
	VALA_CODE_NODE_CLASS (vala_block_parent_class)->finalize (obj);
}


GType vala_block_get_type (void) {
	static GType vala_block_type_id = 0;
	if (vala_block_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBlock), 0, (GInstanceInitFunc) vala_block_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_block_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_block_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaBlock", &g_define_type_info, 0);
		g_type_add_interface_static (vala_block_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_block_type_id;
}




