/* valaccodefunction.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <ccode/valaccodefunction.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <ccode/valaccodelinedirective.h>




struct _ValaCCodeFunctionPrivate {
	char* _name;
	ValaCCodeModifiers _modifiers;
	char* _return_type;
	ValaCCodeBlock* _block;
	GeeList* parameters;
};

#define VALA_CCODE_FUNCTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunctionPrivate))
enum  {
	VALA_CCODE_FUNCTION_DUMMY_PROPERTY
};
static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_function_parent_class = NULL;
static void vala_ccode_function_finalize (ValaCCodeNode* obj);



ValaCCodeFunction* vala_ccode_function_construct (GType object_type, const char* name, const char* return_type) {
	ValaCCodeFunction* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaCCodeFunction*) g_type_create_instance (object_type);
	vala_ccode_function_set_name (self, name);
	vala_ccode_function_set_return_type (self, return_type);
	return self;
}


ValaCCodeFunction* vala_ccode_function_new (const char* name, const char* return_type) {
	return vala_ccode_function_construct (VALA_TYPE_CCODE_FUNCTION, name, return_type);
}


/**
 * Appends the specified parameter to the list of function parameters.
 *
 * @param param a formal parameter
 */
void vala_ccode_function_add_parameter (ValaCCodeFunction* self, ValaCCodeFormalParameter* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
}


/**
 * Returns a copy of this function.
 *
 * @return copied function
 */
ValaCCodeFunction* vala_ccode_function_copy (ValaCCodeFunction* self) {
	ValaCCodeFunction* func;
	g_return_val_if_fail (self != NULL, NULL);
	func = vala_ccode_function_new (self->priv->_name, self->priv->_return_type);
	vala_ccode_function_set_modifiers (func, self->priv->_modifiers);
	/* no deep copy for lists available yet
	 * func.parameters = parameters.copy ();
	 */
	{
		GeeIterator* _param_it;
		/* no deep copy for lists available yet
		 * func.parameters = parameters.copy ();
		 */
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		/* no deep copy for lists available yet
		 * func.parameters = parameters.copy ();
		 */
		while (gee_iterator_next (_param_it)) {
			ValaCCodeFormalParameter* param;
			/* no deep copy for lists available yet
			 * func.parameters = parameters.copy ();
			 */
			param = (ValaCCodeFormalParameter*) gee_iterator_get (_param_it);
			gee_collection_add ((GeeCollection*) func->priv->parameters, param);
			(param == NULL) ? NULL : (param = (vala_ccode_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	vala_ccode_function_set_block (func, self->priv->_block);
	return func;
}


static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeFunction * self;
	gboolean first;
	self = (ValaCCodeFunction*) base;
	g_return_if_fail (writer != NULL);
	vala_ccode_writer_write_indent (writer, vala_ccode_node_get_line ((ValaCCodeNode*) self));
	if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_STATIC) == VALA_CCODE_MODIFIERS_STATIC) {
		vala_ccode_writer_write_string (writer, "static ");
	}
	if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_INLINE) == VALA_CCODE_MODIFIERS_INLINE) {
		vala_ccode_writer_write_string (writer, "inline ");
	}
	vala_ccode_writer_write_string (writer, self->priv->_return_type);
	vala_ccode_writer_write_string (writer, " ");
	vala_ccode_writer_write_string (writer, self->priv->_name);
	vala_ccode_writer_write_string (writer, " (");
	first = TRUE;
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaCCodeFormalParameter* param;
			param = (ValaCCodeFormalParameter*) gee_iterator_get (_param_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			vala_ccode_node_write ((ValaCCodeNode*) param, writer);
			(param == NULL) ? NULL : (param = (vala_ccode_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	if (first) {
		vala_ccode_writer_write_string (writer, "void");
	}
	vala_ccode_writer_write_string (writer, ")");
	if (self->priv->_block == NULL) {
		vala_ccode_writer_write_string (writer, ";");
	} else {
		vala_ccode_node_write ((ValaCCodeNode*) self->priv->_block, writer);
		vala_ccode_writer_write_newline (writer);
	}
	vala_ccode_writer_write_newline (writer);
}


const char* vala_ccode_function_get_name (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_ccode_function_set_name (ValaCCodeFunction* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2);
}


ValaCCodeModifiers vala_ccode_function_get_modifiers (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_modifiers;
}


void vala_ccode_function_set_modifiers (ValaCCodeFunction* self, ValaCCodeModifiers value) {
	g_return_if_fail (self != NULL);
	self->priv->_modifiers = value;
}


const char* vala_ccode_function_get_return_type (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_type;
}


void vala_ccode_function_set_return_type (ValaCCodeFunction* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_return_type = (g_free (self->priv->_return_type), NULL), _tmp2);
}


ValaCCodeBlock* vala_ccode_function_get_block (ValaCCodeFunction* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_block;
}


void vala_ccode_function_set_block (ValaCCodeFunction* self, ValaCCodeBlock* value) {
	ValaCCodeBlock* _tmp2;
	ValaCCodeBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_ccode_node_ref (_tmp1)), (self->priv->_block == NULL) ? NULL : (self->priv->_block = (vala_ccode_node_unref (self->priv->_block), NULL)), _tmp2);
}


static void vala_ccode_function_class_init (ValaCCodeFunctionClass * klass) {
	vala_ccode_function_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_function_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeFunctionPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_function_real_write;
}


static void vala_ccode_function_instance_init (ValaCCodeFunction * self) {
	self->priv = VALA_CCODE_FUNCTION_GET_PRIVATE (self);
	self->priv->parameters = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_function_finalize (ValaCCodeNode* obj) {
	ValaCCodeFunction * self;
	self = VALA_CCODE_FUNCTION (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	self->priv->_return_type = (g_free (self->priv->_return_type), NULL);
	(self->priv->_block == NULL) ? NULL : (self->priv->_block = (vala_ccode_node_unref (self->priv->_block), NULL));
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_function_parent_class)->finalize (obj);
}


GType vala_ccode_function_get_type (void) {
	static GType vala_ccode_function_type_id = 0;
	if (vala_ccode_function_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeFunctionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_function_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeFunction), 0, (GInstanceInitFunc) vala_ccode_function_instance_init, NULL };
		vala_ccode_function_type_id = g_type_register_static (VALA_TYPE_CCODE_NODE, "ValaCCodeFunction", &g_define_type_info, 0);
	}
	return vala_ccode_function_type_id;
}




