/* valapointerindirection.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valapointerindirection.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valapointertype.h>
#include <vala/valareferencetype.h>




struct _ValaPointerIndirectionPrivate {
	ValaExpression* _inner;
};

#define VALA_POINTER_INDIRECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_POINTER_INDIRECTION, ValaPointerIndirectionPrivate))
enum  {
	VALA_POINTER_INDIRECTION_DUMMY_PROPERTY
};
static void vala_pointer_indirection_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_pointer_indirection_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_pointer_indirection_real_is_pure (ValaExpression* base);
static gboolean vala_pointer_indirection_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_pointer_indirection_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_pointer_indirection_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_pointer_indirection_parent_class = NULL;
static void vala_pointer_indirection_finalize (ValaCodeNode* obj);



/**
 * Creates a new pointer indirection.
 *
 * @param inner pointer to be dereferenced
 * @return      newly created pointer indirection
 */
ValaPointerIndirection* vala_pointer_indirection_construct (GType object_type, ValaExpression* inner, ValaSourceReference* source_reference) {
	ValaPointerIndirection* self;
	g_return_val_if_fail (inner != NULL, NULL);
	self = (ValaPointerIndirection*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_pointer_indirection_set_inner (self, inner);
	return self;
}


ValaPointerIndirection* vala_pointer_indirection_new (ValaExpression* inner, ValaSourceReference* source_reference) {
	return vala_pointer_indirection_construct (VALA_TYPE_POINTER_INDIRECTION, inner, source_reference);
}


static void vala_pointer_indirection_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_pointer_indirection_get_inner (self), visitor);
	vala_code_visitor_visit_pointer_indirection (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_pointer_indirection_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_pointer_indirection_get_inner (self) == old_node) {
		vala_pointer_indirection_set_inner (self, new_node);
	}
}


static gboolean vala_pointer_indirection_real_is_pure (ValaExpression* base) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	return vala_expression_is_pure (vala_pointer_indirection_get_inner (self));
}


static gboolean vala_pointer_indirection_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_pointer_indirection_get_inner (self), analyzer)) {
		return FALSE;
	}
	if (vala_expression_get_value_type (vala_pointer_indirection_get_inner (self)) == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "internal error: unknown type of inner expression");
		return FALSE;
	}
	if (VALA_IS_POINTER_TYPE (vala_expression_get_value_type (vala_pointer_indirection_get_inner (self)))) {
		ValaPointerType* _tmp3;
		ValaPointerType* pointer_type;
		_tmp3 = NULL;
		pointer_type = (_tmp3 = VALA_POINTER_TYPE (vala_expression_get_value_type (vala_pointer_indirection_get_inner (self))), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
		if (VALA_IS_REFERENCE_TYPE (vala_pointer_type_get_base_type (pointer_type))) {
			gboolean _tmp4;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Pointer indirection not supported for this expression");
			return (_tmp4 = FALSE, (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp4);
		}
		vala_expression_set_value_type ((ValaExpression*) self, vala_pointer_type_get_base_type (pointer_type));
		(pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL));
	} else {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Pointer indirection not supported for this expression");
		return FALSE;
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_pointer_indirection_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_pointer_indirection_get_inner (self), collection);
}


static void vala_pointer_indirection_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaPointerIndirection * self;
	self = (ValaPointerIndirection*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_pointer_indirection_get_inner (self), collection);
}


ValaExpression* vala_pointer_indirection_get_inner (ValaPointerIndirection* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_inner;
}


void vala_pointer_indirection_set_inner (ValaPointerIndirection* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_inner = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_inner, (ValaCodeNode*) self);
}


static void vala_pointer_indirection_class_init (ValaPointerIndirectionClass * klass) {
	vala_pointer_indirection_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_pointer_indirection_finalize;
	g_type_class_add_private (klass, sizeof (ValaPointerIndirectionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_pointer_indirection_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_pointer_indirection_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_pointer_indirection_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_pointer_indirection_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_pointer_indirection_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_pointer_indirection_real_get_used_variables;
}


static void vala_pointer_indirection_instance_init (ValaPointerIndirection * self) {
	self->priv = VALA_POINTER_INDIRECTION_GET_PRIVATE (self);
}


static void vala_pointer_indirection_finalize (ValaCodeNode* obj) {
	ValaPointerIndirection * self;
	self = VALA_POINTER_INDIRECTION (obj);
	(self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL));
	VALA_CODE_NODE_CLASS (vala_pointer_indirection_parent_class)->finalize (obj);
}


GType vala_pointer_indirection_get_type (void) {
	static GType vala_pointer_indirection_type_id = 0;
	if (vala_pointer_indirection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPointerIndirectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_pointer_indirection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaPointerIndirection), 0, (GInstanceInitFunc) vala_pointer_indirection_instance_init, NULL };
		vala_pointer_indirection_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaPointerIndirection", &g_define_type_info, 0);
	}
	return vala_pointer_indirection_type_id;
}




