/* valaunaryexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaunaryexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valadatatype.h>
#include <vala/valastruct.h>
#include <vala/valamemberaccess.h>
#include <vala/valaparenthesizedexpression.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valafieldprototype.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valabinaryexpression.h>
#include <vala/valaintegerliteral.h>
#include <vala/valaassignment.h>
#include <vala/valafield.h>
#include <vala/valaformalparameter.h>
#include <vala/valalocalvariable.h>




struct _ValaUnaryExpressionPrivate {
	ValaUnaryOperator _operator;
	ValaExpression* _inner;
};

#define VALA_UNARY_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_UNARY_EXPRESSION, ValaUnaryExpressionPrivate))
enum  {
	VALA_UNARY_EXPRESSION_DUMMY_PROPERTY
};
static void vala_unary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_unary_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_unary_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static char* vala_unary_expression_get_operator_string (ValaUnaryExpression* self);
static char* vala_unary_expression_real_to_string (ValaCodeNode* base);
static gboolean vala_unary_expression_real_is_pure (ValaExpression* base);
static gboolean vala_unary_expression_is_numeric_type (ValaUnaryExpression* self, ValaDataType* type);
static gboolean vala_unary_expression_is_integer_type (ValaUnaryExpression* self, ValaDataType* type);
static ValaMemberAccess* vala_unary_expression_find_member_access (ValaUnaryExpression* self, ValaExpression* expr);
static gboolean vala_unary_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_unary_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_unary_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_unary_expression_parent_class = NULL;
static void vala_unary_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new unary expression.
 *
 * @param op     unary operator
 * @param inner  operand
 * @param source reference to source code
 * @return       newly created binary expression
 */
ValaUnaryExpression* vala_unary_expression_construct (GType object_type, ValaUnaryOperator op, ValaExpression* _inner, ValaSourceReference* source) {
	ValaUnaryExpression* self;
	g_return_val_if_fail (_inner != NULL, NULL);
	g_return_val_if_fail (source != NULL, NULL);
	self = (ValaUnaryExpression*) g_type_create_instance (object_type);
	vala_unary_expression_set_operator (self, op);
	vala_unary_expression_set_inner (self, _inner);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source);
	return self;
}


ValaUnaryExpression* vala_unary_expression_new (ValaUnaryOperator op, ValaExpression* _inner, ValaSourceReference* source) {
	return vala_unary_expression_construct (VALA_TYPE_UNARY_EXPRESSION, op, _inner, source);
}


static void vala_unary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaUnaryExpression * self;
	self = (ValaUnaryExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_unary_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_unary_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaUnaryExpression * self;
	self = (ValaUnaryExpression*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_unary_expression_get_inner (self), visitor);
}


static void vala_unary_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaUnaryExpression * self;
	self = (ValaUnaryExpression*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_unary_expression_get_inner (self) == old_node) {
		vala_unary_expression_set_inner (self, new_node);
	}
}


static char* vala_unary_expression_get_operator_string (ValaUnaryExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	switch (self->priv->_operator) {
		case VALA_UNARY_OPERATOR_PLUS:
		{
			return g_strdup ("+");
		}
		case VALA_UNARY_OPERATOR_MINUS:
		{
			return g_strdup ("-");
		}
		case VALA_UNARY_OPERATOR_LOGICAL_NEGATION:
		{
			return g_strdup ("!");
		}
		case VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT:
		{
			return g_strdup ("~");
		}
		case VALA_UNARY_OPERATOR_INCREMENT:
		{
			return g_strdup ("++");
		}
		case VALA_UNARY_OPERATOR_DECREMENT:
		{
			return g_strdup ("--");
		}
		case VALA_UNARY_OPERATOR_REF:
		{
			return g_strdup ("ref ");
		}
		case VALA_UNARY_OPERATOR_OUT:
		{
			return g_strdup ("out ");
		}
	}
	g_assert_not_reached ();
}


static char* vala_unary_expression_real_to_string (ValaCodeNode* base) {
	ValaUnaryExpression * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = (ValaUnaryExpression*) base;
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strconcat (_tmp0 = vala_unary_expression_get_operator_string (self), _tmp1 = vala_code_node_to_string ((ValaCodeNode*) self->priv->_inner), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static gboolean vala_unary_expression_real_is_pure (ValaExpression* base) {
	ValaUnaryExpression * self;
	gboolean _tmp0;
	self = (ValaUnaryExpression*) base;
	_tmp0 = FALSE;
	if (self->priv->_operator == VALA_UNARY_OPERATOR_INCREMENT) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = self->priv->_operator == VALA_UNARY_OPERATOR_DECREMENT;
	}
	if (_tmp0) {
		return FALSE;
	}
	return vala_expression_is_pure (vala_unary_expression_get_inner (self));
}


static gboolean vala_unary_expression_is_numeric_type (ValaUnaryExpression* self, ValaDataType* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_data_type_get_data_type (type)), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	_tmp2 = FALSE;
	if (vala_struct_is_integer_type (st)) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = vala_struct_is_floating_type (st);
	}
	return (_tmp3 = _tmp2, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp3);
}


static gboolean vala_unary_expression_is_integer_type (ValaUnaryExpression* self, ValaDataType* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type != NULL, FALSE);
	if (!(VALA_IS_STRUCT (vala_data_type_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_data_type_get_data_type (type)), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
	return (_tmp2 = vala_struct_is_integer_type (st), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp2);
}


static ValaMemberAccess* vala_unary_expression_find_member_access (ValaUnaryExpression* self, ValaExpression* expr) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (expr != NULL, NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* _tmp0;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* _tmp1;
		_tmp0 = NULL;
		pe = (_tmp0 = VALA_PARENTHESIZED_EXPRESSION (expr), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		_tmp1 = NULL;
		return (_tmp1 = vala_unary_expression_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL) ? NULL : (pe = (vala_code_node_unref (pe), NULL)), _tmp1);
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* _tmp2;
		_tmp2 = NULL;
		return (_tmp2 = VALA_MEMBER_ACCESS (expr), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	}
	return NULL;
}


static gboolean vala_unary_expression_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaUnaryExpression * self;
	gboolean _tmp1;
	gboolean _tmp6;
	self = (ValaUnaryExpression*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	_tmp1 = FALSE;
	if (self->priv->_operator == VALA_UNARY_OPERATOR_REF) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = self->priv->_operator == VALA_UNARY_OPERATOR_OUT;
	}
	if (_tmp1) {
		vala_expression_set_lvalue (vala_unary_expression_get_inner (self), TRUE);
		vala_expression_set_target_type (vala_unary_expression_get_inner (self), vala_expression_get_target_type ((ValaExpression*) self));
	}
	if (!vala_code_node_check ((ValaCodeNode*) vala_unary_expression_get_inner (self), analyzer)) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	if (VALA_IS_FIELD_PROTOTYPE (vala_expression_get_value_type (vala_unary_expression_get_inner (self)))) {
		char* _tmp4;
		char* _tmp3;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_unary_expression_get_inner (self)), _tmp4 = g_strdup_printf ("Access to instance member `%s' denied", _tmp3 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (self)))));
		_tmp4 = (g_free (_tmp4), NULL);
		_tmp3 = (g_free (_tmp3), NULL);
		return FALSE;
	}
	_tmp6 = FALSE;
	if (self->priv->_operator == VALA_UNARY_OPERATOR_PLUS) {
		_tmp6 = TRUE;
	} else {
		_tmp6 = self->priv->_operator == VALA_UNARY_OPERATOR_MINUS;
	}
	if (_tmp6) {
		/* integer or floating point type*/
		if (!vala_unary_expression_is_numeric_type (self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)))) {
			char* _tmp8;
			char* _tmp7;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp8 = NULL;
			_tmp7 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp8 = g_strdup_printf ("Operator not supported for `%s'", _tmp7 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_unary_expression_get_inner (self)))));
			_tmp8 = (g_free (_tmp8), NULL);
			_tmp7 = (g_free (_tmp7), NULL);
			return FALSE;
		}
		vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)));
	} else {
		if (self->priv->_operator == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (!vala_data_type_compatible (vala_expression_get_value_type (vala_unary_expression_get_inner (self)), analyzer->bool_type)) {
				char* _tmp11;
				char* _tmp10;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp11 = NULL;
				_tmp10 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp11 = g_strdup_printf ("Operator not supported for `%s'", _tmp10 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_unary_expression_get_inner (self)))));
				_tmp11 = (g_free (_tmp11), NULL);
				_tmp10 = (g_free (_tmp10), NULL);
				return FALSE;
			}
			vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)));
		} else {
			if (self->priv->_operator == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_unary_expression_is_integer_type (self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)))) {
					char* _tmp14;
					char* _tmp13;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp14 = NULL;
					_tmp13 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp14 = g_strdup_printf ("Operator not supported for `%s'", _tmp13 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_unary_expression_get_inner (self)))));
					_tmp14 = (g_free (_tmp14), NULL);
					_tmp13 = (g_free (_tmp13), NULL);
					return FALSE;
				}
				vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)));
			} else {
				gboolean _tmp16;
				_tmp16 = FALSE;
				if (self->priv->_operator == VALA_UNARY_OPERATOR_INCREMENT) {
					_tmp16 = TRUE;
				} else {
					_tmp16 = self->priv->_operator == VALA_UNARY_OPERATOR_DECREMENT;
				}
				if (_tmp16) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaBinaryOperator _tmp21;
					ValaIntegerLiteral* _tmp22;
					ValaBinaryExpression* _tmp23;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					ValaParenthesizedExpression* parenthexp;
					gboolean _tmp24;
					/* integer type*/
					if (!vala_unary_expression_is_integer_type (self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)))) {
						char* _tmp18;
						char* _tmp17;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						_tmp18 = NULL;
						_tmp17 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp18 = g_strdup_printf ("Operator not supported for `%s'", _tmp17 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_unary_expression_get_inner (self)))));
						_tmp18 = (g_free (_tmp18), NULL);
						_tmp17 = (g_free (_tmp17), NULL);
						return FALSE;
					}
					ma = vala_unary_expression_find_member_access (self, vala_unary_expression_get_inner (self));
					if (ma == NULL) {
						gboolean _tmp20;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Prefix operators not supported for this expression");
						return (_tmp20 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), _tmp20);
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), vala_code_node_get_source_reference ((ValaCodeNode*) vala_unary_expression_get_inner (self)));
					_tmp21 = 0;
					if (self->priv->_operator == VALA_UNARY_OPERATOR_INCREMENT) {
						_tmp21 = VALA_BINARY_OPERATOR_PLUS;
					} else {
						_tmp21 = VALA_BINARY_OPERATOR_MINUS;
					}
					_tmp22 = NULL;
					_tmp23 = NULL;
					bin = (_tmp23 = vala_binary_expression_new (_tmp21, (ValaExpression*) old_value, (ValaExpression*) (_tmp22 = vala_integer_literal_new ("1", NULL)), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL)), _tmp23);
					assignment = vala_assignment_new ((ValaExpression*) ma, (ValaExpression*) bin, VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) self));
					parenthexp = vala_parenthesized_expression_new ((ValaExpression*) assignment, vala_code_node_get_source_reference ((ValaCodeNode*) self));
					vala_expression_set_target_type ((ValaExpression*) parenthexp, vala_expression_get_target_type ((ValaExpression*) self));
					gee_collection_add ((GeeCollection*) analyzer->replaced_nodes, (ValaCodeNode*) self);
					vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) parenthexp);
					vala_code_node_check ((ValaCodeNode*) parenthexp, analyzer);
					return (_tmp24 = TRUE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), (old_value == NULL) ? NULL : (old_value = (vala_code_node_unref (old_value), NULL)), (bin == NULL) ? NULL : (bin = (vala_code_node_unref (bin), NULL)), (assignment == NULL) ? NULL : (assignment = (vala_code_node_unref (assignment), NULL)), (parenthexp == NULL) ? NULL : (parenthexp = (vala_code_node_unref (parenthexp), NULL)), _tmp24);
				} else {
					gboolean _tmp25;
					_tmp25 = FALSE;
					if (self->priv->_operator == VALA_UNARY_OPERATOR_REF) {
						_tmp25 = TRUE;
					} else {
						_tmp25 = self->priv->_operator == VALA_UNARY_OPERATOR_OUT;
					}
					if (_tmp25) {
						gboolean _tmp26;
						gboolean _tmp27;
						_tmp26 = FALSE;
						_tmp27 = FALSE;
						if (VALA_IS_FIELD (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (self)))) {
							_tmp27 = TRUE;
						} else {
							_tmp27 = VALA_IS_FORMAL_PARAMETER (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (self)));
						}
						if (_tmp27) {
							_tmp26 = TRUE;
						} else {
							_tmp26 = VALA_IS_LOCAL_VARIABLE (vala_expression_get_symbol_reference (vala_unary_expression_get_inner (self)));
						}
						if (_tmp26) {
							/* ref and out can only be used with fields, parameters, and local variables*/
							vala_expression_set_lvalue ((ValaExpression*) self, TRUE);
							vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_unary_expression_get_inner (self)));
						} else {
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "ref and out method arguments can only be used with fields, parameters, and local variables");
							return FALSE;
						}
					} else {
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "internal error: unsupported unary operator");
						return FALSE;
					}
				}
			}
		}
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_unary_expression_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaUnaryExpression * self;
	gboolean _tmp0;
	self = (ValaUnaryExpression*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_unary_expression_get_inner (self), collection);
	_tmp0 = FALSE;
	if (self->priv->_operator == VALA_UNARY_OPERATOR_OUT) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = self->priv->_operator == VALA_UNARY_OPERATOR_REF;
	}
	if (_tmp0) {
		ValaLocalVariable* _tmp2;
		ValaSymbol* _tmp1;
		ValaLocalVariable* local;
		_tmp2 = NULL;
		_tmp1 = NULL;
		local = (_tmp2 = (_tmp1 = vala_expression_get_symbol_reference (vala_unary_expression_get_inner (self)), VALA_IS_LOCAL_VARIABLE (_tmp1) ? ((ValaLocalVariable*) _tmp1) : NULL), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		if (local != NULL) {
			gee_collection_add (collection, local);
		}
		(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
	}
}


static void vala_unary_expression_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaUnaryExpression * self;
	self = (ValaUnaryExpression*) base;
	g_return_if_fail (collection != NULL);
	if (self->priv->_operator != VALA_UNARY_OPERATOR_OUT) {
		vala_code_node_get_used_variables ((ValaCodeNode*) vala_unary_expression_get_inner (self), collection);
	}
}


ValaUnaryOperator vala_unary_expression_get_operator (ValaUnaryExpression* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_operator;
}


void vala_unary_expression_set_operator (ValaUnaryExpression* self, ValaUnaryOperator value) {
	g_return_if_fail (self != NULL);
	self->priv->_operator = value;
}


ValaExpression* vala_unary_expression_get_inner (ValaUnaryExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_inner;
}


void vala_unary_expression_set_inner (ValaUnaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_inner = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_inner, (ValaCodeNode*) self);
}


static void vala_unary_expression_class_init (ValaUnaryExpressionClass * klass) {
	vala_unary_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_unary_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaUnaryExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_unary_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_unary_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_unary_expression_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_unary_expression_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_unary_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_unary_expression_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_unary_expression_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_unary_expression_real_get_used_variables;
}


static void vala_unary_expression_instance_init (ValaUnaryExpression * self) {
	self->priv = VALA_UNARY_EXPRESSION_GET_PRIVATE (self);
}


static void vala_unary_expression_finalize (ValaCodeNode* obj) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (obj);
	(self->priv->_inner == NULL) ? NULL : (self->priv->_inner = (vala_code_node_unref (self->priv->_inner), NULL));
	VALA_CODE_NODE_CLASS (vala_unary_expression_parent_class)->finalize (obj);
}


GType vala_unary_expression_get_type (void) {
	static GType vala_unary_expression_type_id = 0;
	if (vala_unary_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaUnaryExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_unary_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaUnaryExpression), 0, (GInstanceInitFunc) vala_unary_expression_instance_init, NULL };
		vala_unary_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaUnaryExpression", &g_define_type_info, 0);
	}
	return vala_unary_expression_type_id;
}



GType vala_unary_operator_get_type (void) {
	static GType vala_unary_operator_type_id = 0;
	if (G_UNLIKELY (vala_unary_operator_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_UNARY_OPERATOR_NONE, "VALA_UNARY_OPERATOR_NONE", "none"}, {VALA_UNARY_OPERATOR_PLUS, "VALA_UNARY_OPERATOR_PLUS", "plus"}, {VALA_UNARY_OPERATOR_MINUS, "VALA_UNARY_OPERATOR_MINUS", "minus"}, {VALA_UNARY_OPERATOR_LOGICAL_NEGATION, "VALA_UNARY_OPERATOR_LOGICAL_NEGATION", "logical-negation"}, {VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT, "VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT", "bitwise-complement"}, {VALA_UNARY_OPERATOR_INCREMENT, "VALA_UNARY_OPERATOR_INCREMENT", "increment"}, {VALA_UNARY_OPERATOR_DECREMENT, "VALA_UNARY_OPERATOR_DECREMENT", "decrement"}, {VALA_UNARY_OPERATOR_REF, "VALA_UNARY_OPERATOR_REF", "ref"}, {VALA_UNARY_OPERATOR_OUT, "VALA_UNARY_OPERATOR_OUT", "out"}, {0, NULL, NULL}};
		vala_unary_operator_type_id = g_enum_register_static ("ValaUnaryOperator", values);
	}
	return vala_unary_operator_type_id;
}




