/* valapropertyaccessor.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valapropertyaccessor.h>
#include <vala/valaproperty.h>
#include <vala/valablock.h>
#include <vala/valabasicblock.h>
#include <vala/valaformalparameter.h>
#include <vala/valadatatype.h>
#include <vala/valatypesymbol.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valavoidtype.h>
#include <vala/valainterface.h>
#include <vala/valareport.h>
#include <vala/valamemberaccess.h>
#include <vala/valareturnstatement.h>
#include <vala/valaexpression.h>
#include <vala/valastatement.h>
#include <vala/valaassignment.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valascope.h>




struct _ValaPropertyAccessorPrivate {
	ValaProperty* _prop;
	gboolean _readable;
	gboolean _writable;
	gboolean _construction;
	ValaSymbolAccessibility _access;
	ValaBlock* _body;
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	gboolean _automatic_body;
	ValaFormalParameter* _value_parameter;
	ValaDataType* _value_type;
	char* _cname;
};

#define VALA_PROPERTY_ACCESSOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorPrivate))
enum  {
	VALA_PROPERTY_ACCESSOR_DUMMY_PROPERTY
};
static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_property_accessor_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_property_accessor_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_property_accessor_parent_class = NULL;
static void vala_property_accessor_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * The publicly accessible name of the function that performs the
 * access in C code.
 */
char* vala_property_accessor_get_cname (ValaPropertyAccessor* self) {
	ValaTypeSymbol* _tmp2;
	ValaTypeSymbol* t;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_cname != NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->_cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	}
	_tmp2 = NULL;
	t = (_tmp2 = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) self->priv->_prop)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (self->priv->_readable) {
		char* _tmp3;
		char* _tmp4;
		char* _tmp5;
		_tmp3 = NULL;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = g_strdup_printf ("%s_get_%s", _tmp3 = vala_symbol_get_lower_case_cname ((ValaSymbol*) t, NULL), vala_symbol_get_name ((ValaSymbol*) self->priv->_prop)), _tmp3 = (g_free (_tmp3), NULL), _tmp4), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp5);
	} else {
		char* _tmp6;
		char* _tmp7;
		char* _tmp8;
		_tmp6 = NULL;
		_tmp7 = NULL;
		_tmp8 = NULL;
		return (_tmp8 = (_tmp7 = g_strdup_printf ("%s_set_%s", _tmp6 = vala_symbol_get_lower_case_cname ((ValaSymbol*) t, NULL), vala_symbol_get_name ((ValaSymbol*) self->priv->_prop)), _tmp6 = (g_free (_tmp6), NULL), _tmp7), (t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL)), _tmp8);
	}
	(t == NULL) ? NULL : (t = (vala_code_node_unref (t), NULL));
}


/**
 * Creates a new property accessor.
 *
 * @param readable     true if get accessor, false otherwise
 * @param writable     true if set accessor, false otherwise
 * @param construction true if construct accessor, false otherwise
 * @param body         accessor body
 * @param source       reference to source code
 * @return             newly created property accessor
 */
ValaPropertyAccessor* vala_property_accessor_construct (GType object_type, gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaPropertyAccessor* self;
	self = (ValaPropertyAccessor*) g_type_create_instance (object_type);
	vala_property_accessor_set_readable (self, readable);
	vala_property_accessor_set_writable (self, writable);
	vala_property_accessor_set_construction (self, construction);
	vala_property_accessor_set_value_type (self, value_type);
	vala_property_accessor_set_body (self, body);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaPropertyAccessor* vala_property_accessor_new (gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_property_accessor_construct (VALA_TYPE_PROPERTY_ACCESSOR, readable, writable, construction, value_type, body, source_reference);
}


static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_property_accessor (visitor, self);
}


static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_property_accessor_get_value_type (self), visitor);
	if (self->priv->_body != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_body, visitor);
	}
}


/**
 * Process all associated attributes.
 */
void vala_property_accessor_process_attributes (ValaPropertyAccessor* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp1;
			ValaAttribute* a;
			_tmp1 = NULL;
			a = (_tmp1 = (ValaAttribute*) a_it->data, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					if (vala_attribute_has_argument (a, "cname")) {
						char* _tmp0;
						_tmp0 = NULL;
						self->priv->_cname = (_tmp0 = vala_attribute_get_string (a, "cname"), self->priv->_cname = (g_free (self->priv->_cname), NULL), _tmp0);
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_property_accessor_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaPropertyAccessor * self;
	ValaDataType* _tmp2;
	ValaDataType* old_return_type;
	ValaDataType* _tmp20;
	ValaDataType* _tmp19;
	gboolean _tmp21;
	self = (ValaPropertyAccessor*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_property_accessor_process_attributes (self);
	if (!vala_code_node_check ((ValaCodeNode*) vala_property_accessor_get_value_type (self), analyzer)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp2 = NULL;
	old_return_type = (_tmp2 = analyzer->current_return_type, (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (self->priv->_readable) {
		ValaDataType* _tmp4;
		ValaDataType* _tmp3;
		_tmp4 = NULL;
		_tmp3 = NULL;
		analyzer->current_return_type = (_tmp4 = (_tmp3 = vala_property_accessor_get_value_type (self), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp4);
	} else {
		ValaDataType* _tmp5;
		/* void*/
		_tmp5 = NULL;
		analyzer->current_return_type = (_tmp5 = (ValaDataType*) vala_void_type_new (NULL), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp5);
	}
	if (!vala_symbol_get_external_package ((ValaSymbol*) self->priv->_prop)) {
		gboolean _tmp6;
		gboolean _tmp7;
		gboolean _tmp16;
		_tmp6 = FALSE;
		_tmp7 = FALSE;
		if (self->priv->_body == NULL) {
			_tmp7 = !vala_property_get_interface_only (self->priv->_prop);
		} else {
			_tmp7 = FALSE;
		}
		if (_tmp7) {
			_tmp6 = !vala_property_get_is_abstract (self->priv->_prop);
		} else {
			_tmp6 = FALSE;
		}
		if (_tmp6) {
			ValaBlock* _tmp9;
			char* _tmp10;
			ValaMemberAccess* _tmp11;
			ValaMemberAccess* ma;
			/* no accessor body specified, insert default body */
			if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self->priv->_prop))) {
				gboolean _tmp8;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Automatic properties can't be used in interfaces");
				return (_tmp8 = FALSE, (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), _tmp8);
			}
			vala_property_accessor_set_automatic_body (self, TRUE);
			_tmp9 = NULL;
			vala_property_accessor_set_body (self, _tmp9 = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
			(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
			_tmp10 = NULL;
			_tmp11 = NULL;
			ma = (_tmp11 = vala_member_access_new_simple (_tmp10 = g_strdup_printf ("_%s", vala_symbol_get_name ((ValaSymbol*) self->priv->_prop)), vala_code_node_get_source_reference ((ValaCodeNode*) self)), _tmp10 = (g_free (_tmp10), NULL), _tmp11);
			if (self->priv->_readable) {
				ValaReturnStatement* _tmp12;
				_tmp12 = NULL;
				vala_block_add_statement (self->priv->_body, (ValaStatement*) (_tmp12 = vala_return_statement_new ((ValaExpression*) ma, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
				(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
			} else {
				ValaMemberAccess* _tmp13;
				ValaAssignment* _tmp14;
				ValaAssignment* assignment;
				ValaExpressionStatement* _tmp15;
				_tmp13 = NULL;
				_tmp14 = NULL;
				assignment = (_tmp14 = vala_assignment_new ((ValaExpression*) ma, (ValaExpression*) (_tmp13 = vala_member_access_new_simple ("value", vala_code_node_get_source_reference ((ValaCodeNode*) self))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL)), _tmp14);
				_tmp15 = NULL;
				vala_block_add_statement (self->priv->_body, (ValaStatement*) (_tmp15 = vala_expression_statement_new ((ValaExpression*) assignment, NULL)));
				(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL));
				(assignment == NULL) ? NULL : (assignment = (vala_code_node_unref (assignment), NULL));
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
		_tmp16 = FALSE;
		if (self->priv->_body != NULL) {
			gboolean _tmp17;
			_tmp17 = FALSE;
			if (self->priv->_writable) {
				_tmp17 = TRUE;
			} else {
				_tmp17 = self->priv->_construction;
			}
			_tmp16 = (_tmp17);
		} else {
			_tmp16 = FALSE;
		}
		if (_tmp16) {
			ValaFormalParameter* _tmp18;
			_tmp18 = NULL;
			vala_property_accessor_set_value_parameter (self, _tmp18 = vala_formal_parameter_new ("value", vala_property_accessor_get_value_type (self), vala_code_node_get_source_reference ((ValaCodeNode*) self)));
			(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
			vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self->priv->_body), vala_symbol_get_name ((ValaSymbol*) self->priv->_value_parameter), (ValaSymbol*) self->priv->_value_parameter);
		}
	}
	if (self->priv->_body != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_body, analyzer);
	}
	_tmp20 = NULL;
	_tmp19 = NULL;
	analyzer->current_return_type = (_tmp20 = (_tmp19 = old_return_type, (_tmp19 == NULL) ? NULL : vala_code_node_ref (_tmp19)), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp20);
	return (_tmp21 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), _tmp21);
}


static void vala_property_accessor_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_property_accessor_get_value_type (self) == old_type) {
		vala_property_accessor_set_value_type (self, new_type);
	}
}


ValaProperty* vala_property_accessor_get_prop (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_prop;
}


void vala_property_accessor_set_prop (ValaPropertyAccessor* self, ValaProperty* value) {
	g_return_if_fail (self != NULL);
	self->priv->_prop = value;
}


ValaDataType* vala_property_accessor_get_value_type (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value_type;
}


void vala_property_accessor_set_value_type (ValaPropertyAccessor* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_value_type == NULL) ? NULL : (self->priv->_value_type = (vala_code_node_unref (self->priv->_value_type), NULL)), _tmp2);
	if (value != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_value_type, (ValaCodeNode*) self);
	}
}


gboolean vala_property_accessor_get_readable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_readable;
}


void vala_property_accessor_set_readable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_readable = value;
}


gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_writable;
}


void vala_property_accessor_set_writable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_writable = value;
}


gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_construction;
}


void vala_property_accessor_set_construction (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_construction = value;
}


ValaSymbolAccessibility vala_property_accessor_get_access (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_access;
}


void vala_property_accessor_set_access (ValaPropertyAccessor* self, ValaSymbolAccessibility value) {
	g_return_if_fail (self != NULL);
	self->priv->_access = value;
}


ValaBlock* vala_property_accessor_get_body (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_property_accessor_set_body (ValaPropertyAccessor* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_entry_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_entry_block;
}


void vala_property_accessor_set_entry_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_entry_block == NULL) ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL)), _tmp2);
}


ValaBasicBlock* vala_property_accessor_get_exit_block (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_exit_block;
}


void vala_property_accessor_set_exit_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_exit_block == NULL) ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL)), _tmp2);
}


gboolean vala_property_accessor_get_automatic_body (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_automatic_body;
}


void vala_property_accessor_set_automatic_body (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_automatic_body = value;
}


ValaFormalParameter* vala_property_accessor_get_value_parameter (ValaPropertyAccessor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value_parameter;
}


void vala_property_accessor_set_value_parameter (ValaPropertyAccessor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_value_parameter == NULL) ? NULL : (self->priv->_value_parameter = (vala_code_node_unref (self->priv->_value_parameter), NULL)), _tmp2);
}


static void vala_property_accessor_class_init (ValaPropertyAccessorClass * klass) {
	vala_property_accessor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_property_accessor_finalize;
	g_type_class_add_private (klass, sizeof (ValaPropertyAccessorPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_accessor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_accessor_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_property_accessor_real_check;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_property_accessor_real_replace_type;
}


static void vala_property_accessor_instance_init (ValaPropertyAccessor * self) {
	self->priv = VALA_PROPERTY_ACCESSOR_GET_PRIVATE (self);
}


static void vala_property_accessor_finalize (ValaCodeNode* obj) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (obj);
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	(self->priv->_entry_block == NULL) ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL));
	(self->priv->_exit_block == NULL) ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL));
	(self->priv->_value_parameter == NULL) ? NULL : (self->priv->_value_parameter = (vala_code_node_unref (self->priv->_value_parameter), NULL));
	(self->priv->_value_type == NULL) ? NULL : (self->priv->_value_type = (vala_code_node_unref (self->priv->_value_type), NULL));
	self->priv->_cname = (g_free (self->priv->_cname), NULL);
	VALA_CODE_NODE_CLASS (vala_property_accessor_parent_class)->finalize (obj);
}


GType vala_property_accessor_get_type (void) {
	static GType vala_property_accessor_type_id = 0;
	if (vala_property_accessor_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyAccessorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_accessor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaPropertyAccessor), 0, (GInstanceInitFunc) vala_property_accessor_instance_init, NULL };
		vala_property_accessor_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaPropertyAccessor", &g_define_type_info, 0);
	}
	return vala_property_accessor_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




