/* valaccodedeclaration.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <ccode/valaccodedeclaration.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <ccode/valaccodenode.h>
#include <ccode/valaccodelinedirective.h>
#include <ccode/valaccodevariabledeclarator.h>




struct _ValaCCodeDeclarationPrivate {
	char* _type_name;
	ValaCCodeModifiers _modifiers;
	GeeList* declarators;
};

#define VALA_CCODE_DECLARATION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_DECLARATION, ValaCCodeDeclarationPrivate))
enum  {
	VALA_CCODE_DECLARATION_DUMMY_PROPERTY
};
static void vala_ccode_declaration_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gboolean vala_ccode_declaration_has_initializer (ValaCCodeDeclaration* self);
static void vala_ccode_declaration_real_write_declaration (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_declaration_parent_class = NULL;
static void vala_ccode_declaration_finalize (ValaCCodeNode* obj);



ValaCCodeDeclaration* vala_ccode_declaration_construct (GType object_type, const char* type_name) {
	ValaCCodeDeclaration* self;
	g_return_val_if_fail (type_name != NULL, NULL);
	self = (ValaCCodeDeclaration*) g_type_create_instance (object_type);
	vala_ccode_declaration_set_type_name (self, type_name);
	return self;
}


ValaCCodeDeclaration* vala_ccode_declaration_new (const char* type_name) {
	return vala_ccode_declaration_construct (VALA_TYPE_CCODE_DECLARATION, type_name);
}


/**
 * Adds the specified declarator to this declaration.
 *
 * @param decl a declarator
 */
void vala_ccode_declaration_add_declarator (ValaCCodeDeclaration* self, ValaCCodeDeclarator* decl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (decl != NULL);
	gee_collection_add ((GeeCollection*) self->priv->declarators, decl);
}


static void vala_ccode_declaration_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeDeclaration * self;
	self = (ValaCCodeDeclaration*) base;
	g_return_if_fail (writer != NULL);
	if ((self->priv->_modifiers & (VALA_CCODE_MODIFIERS_STATIC | VALA_CCODE_MODIFIERS_EXTERN)) != 0) {
		gboolean _tmp0;
		gboolean first;
		/* combined declaration and initialization for static and extern variables*/
		vala_ccode_writer_write_indent (writer, vala_ccode_node_get_line ((ValaCCodeNode*) self));
		if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_STATIC) != 0) {
			vala_ccode_writer_write_string (writer, "static ");
		}
		if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_VOLATILE) != 0) {
			vala_ccode_writer_write_string (writer, "volatile ");
		}
		_tmp0 = FALSE;
		if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_EXTERN) != 0) {
			_tmp0 = !vala_ccode_declaration_has_initializer (self);
		} else {
			_tmp0 = FALSE;
		}
		if (_tmp0) {
			vala_ccode_writer_write_string (writer, "extern ");
		}
		vala_ccode_writer_write_string (writer, self->priv->_type_name);
		vala_ccode_writer_write_string (writer, " ");
		first = TRUE;
		{
			GeeIterator* _decl_it;
			_decl_it = gee_iterable_iterator ((GeeIterable*) self->priv->declarators);
			while (gee_iterator_next (_decl_it)) {
				ValaCCodeDeclarator* decl;
				decl = (ValaCCodeDeclarator*) gee_iterator_get (_decl_it);
				if (!first) {
					vala_ccode_writer_write_string (writer, ", ");
				} else {
					first = FALSE;
				}
				vala_ccode_node_write ((ValaCCodeNode*) decl, writer);
				(decl == NULL) ? NULL : (decl = (vala_ccode_node_unref (decl), NULL));
			}
			(_decl_it == NULL) ? NULL : (_decl_it = (gee_collection_object_unref (_decl_it), NULL));
		}
		vala_ccode_writer_write_string (writer, ";");
		vala_ccode_writer_write_newline (writer);
	} else {
		{
			GeeIterator* _decl_it;
			_decl_it = gee_iterable_iterator ((GeeIterable*) self->priv->declarators);
			while (gee_iterator_next (_decl_it)) {
				ValaCCodeDeclarator* decl;
				decl = (ValaCCodeDeclarator*) gee_iterator_get (_decl_it);
				vala_ccode_declarator_write_initialization (decl, writer);
				(decl == NULL) ? NULL : (decl = (vala_ccode_node_unref (decl), NULL));
			}
			(_decl_it == NULL) ? NULL : (_decl_it = (gee_collection_object_unref (_decl_it), NULL));
		}
	}
}


static gboolean vala_ccode_declaration_has_initializer (ValaCCodeDeclaration* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GeeIterator* _decl_it;
		_decl_it = gee_iterable_iterator ((GeeIterable*) self->priv->declarators);
		while (gee_iterator_next (_decl_it)) {
			ValaCCodeDeclarator* decl;
			ValaCCodeVariableDeclarator* _tmp1;
			ValaCCodeDeclarator* _tmp0;
			ValaCCodeVariableDeclarator* var_decl;
			gboolean _tmp2;
			decl = (ValaCCodeDeclarator*) gee_iterator_get (_decl_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			var_decl = (_tmp1 = (_tmp0 = decl, VALA_IS_CCODE_VARIABLE_DECLARATOR (_tmp0) ? ((ValaCCodeVariableDeclarator*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_ccode_node_ref (_tmp1));
			_tmp2 = FALSE;
			if (var_decl != NULL) {
				_tmp2 = vala_ccode_variable_declarator_get_initializer (var_decl) == NULL;
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				gboolean _tmp3;
				return (_tmp3 = FALSE, (decl == NULL) ? NULL : (decl = (vala_ccode_node_unref (decl), NULL)), (var_decl == NULL) ? NULL : (var_decl = (vala_ccode_node_unref (var_decl), NULL)), (_decl_it == NULL) ? NULL : (_decl_it = (gee_collection_object_unref (_decl_it), NULL)), _tmp3);
			}
			(decl == NULL) ? NULL : (decl = (vala_ccode_node_unref (decl), NULL));
			(var_decl == NULL) ? NULL : (var_decl = (vala_ccode_node_unref (var_decl), NULL));
		}
		(_decl_it == NULL) ? NULL : (_decl_it = (gee_collection_object_unref (_decl_it), NULL));
	}
	return TRUE;
}


static void vala_ccode_declaration_real_write_declaration (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeDeclaration * self;
	gboolean first;
	self = (ValaCCodeDeclaration*) base;
	g_return_if_fail (writer != NULL);
	if ((self->priv->_modifiers & (VALA_CCODE_MODIFIERS_STATIC | VALA_CCODE_MODIFIERS_EXTERN)) != 0) {
		/* no separate declaration for static variables*/
		return;
	}
	vala_ccode_writer_write_indent (writer, NULL);
	if ((self->priv->_modifiers & VALA_CCODE_MODIFIERS_REGISTER) == VALA_CCODE_MODIFIERS_REGISTER) {
		vala_ccode_writer_write_string (writer, "register ");
	}
	vala_ccode_writer_write_string (writer, self->priv->_type_name);
	vala_ccode_writer_write_string (writer, " ");
	first = TRUE;
	{
		GeeIterator* _decl_it;
		_decl_it = gee_iterable_iterator ((GeeIterable*) self->priv->declarators);
		while (gee_iterator_next (_decl_it)) {
			ValaCCodeDeclarator* decl;
			decl = (ValaCCodeDeclarator*) gee_iterator_get (_decl_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			vala_ccode_node_write_declaration ((ValaCCodeNode*) decl, writer);
			(decl == NULL) ? NULL : (decl = (vala_ccode_node_unref (decl), NULL));
		}
		(_decl_it == NULL) ? NULL : (_decl_it = (gee_collection_object_unref (_decl_it), NULL));
	}
	vala_ccode_writer_write_string (writer, ";");
	vala_ccode_writer_write_newline (writer);
}


const char* vala_ccode_declaration_get_type_name (ValaCCodeDeclaration* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_type_name;
}


void vala_ccode_declaration_set_type_name (ValaCCodeDeclaration* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_type_name = (g_free (self->priv->_type_name), NULL), _tmp2);
}


ValaCCodeModifiers vala_ccode_declaration_get_modifiers (ValaCCodeDeclaration* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_modifiers;
}


void vala_ccode_declaration_set_modifiers (ValaCCodeDeclaration* self, ValaCCodeModifiers value) {
	g_return_if_fail (self != NULL);
	self->priv->_modifiers = value;
}


static void vala_ccode_declaration_class_init (ValaCCodeDeclarationClass * klass) {
	vala_ccode_declaration_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_declaration_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeDeclarationPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_declaration_real_write;
	VALA_CCODE_NODE_CLASS (klass)->write_declaration = vala_ccode_declaration_real_write_declaration;
}


static void vala_ccode_declaration_instance_init (ValaCCodeDeclaration * self) {
	self->priv = VALA_CCODE_DECLARATION_GET_PRIVATE (self);
	self->priv->declarators = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_DECLARATOR, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_declaration_finalize (ValaCCodeNode* obj) {
	ValaCCodeDeclaration * self;
	self = VALA_CCODE_DECLARATION (obj);
	self->priv->_type_name = (g_free (self->priv->_type_name), NULL);
	(self->priv->declarators == NULL) ? NULL : (self->priv->declarators = (gee_collection_object_unref (self->priv->declarators), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_declaration_parent_class)->finalize (obj);
}


GType vala_ccode_declaration_get_type (void) {
	static GType vala_ccode_declaration_type_id = 0;
	if (vala_ccode_declaration_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeDeclarationClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_declaration_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeDeclaration), 0, (GInstanceInitFunc) vala_ccode_declaration_instance_init, NULL };
		vala_ccode_declaration_type_id = g_type_register_static (VALA_TYPE_CCODE_STATEMENT, "ValaCCodeDeclaration", &g_define_type_info, 0);
	}
	return vala_ccode_declaration_type_id;
}




