/* valamethod.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valamethod.h>
#include <gee/readonlylist.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valaformalparameter.h>
#include <vala/valablock.h>
#include <vala/valasymbol.h>
#include <vala/valaattribute.h>
#include <vala/valascope.h>
#include <vala/valaexpression.h>
#include <vala/valalocalvariable.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaclass.h>
#include <vala/valareport.h>
#include <vala/valainterface.h>
#include <vala/valasourcefile.h>
#include <vala/valastruct.h>
#include <vala/valacodecontext.h>
#include <vala/valavaluetype.h>
#include <vala/valacreationmethod.h>
#include <vala/valaobjecttype.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valagenerictype.h>
#include <vala/valasignal.h>
#include <vala/valavoidtype.h>
#include <vala/valaarraytype.h>
#include <vala/valabasicblock.h>




struct _ValaMethodPrivate {
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	MemberBinding _binding;
	gboolean _is_abstract;
	gboolean _is_virtual;
	gboolean _overrides;
	gboolean _is_inline;
	gboolean _returns_modified_pointer;
	gboolean _entry_point;
	ValaFormalParameter* _this_parameter;
	ValaLocalVariable* _result_var;
	double _cinstance_parameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	gboolean _no_array_length;
	gboolean _array_null_terminated;
	gboolean _printf_format;
	gboolean _has_construct_function;
	gboolean _coroutine;
	GeeList* parameters;
	char* cname;
	char* _vfunc_name;
	char* _sentinel;
	GeeList* preconditions;
	GeeList* postconditions;
	ValaDataType* _return_type;
	ValaBlock* _body;
	ValaMethod* _base_method;
	ValaMethod* _base_interface_method;
	gboolean base_methods_valid;
};

#define VALA_METHOD_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD, ValaMethodPrivate))
enum  {
	VALA_METHOD_DUMMY_PROPERTY
};
static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_method_real_get_default_cname (ValaMethod* self);
static char* vala_method_real_get_real_cname (ValaMethod* self);
static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a);
static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_method_find_base_methods (ValaMethod* self);
static void vala_method_find_base_class_method (ValaMethod* self, ValaClass* cl);
static void vala_method_find_base_interface_method (ValaMethod* self, ValaClass* cl);
static gboolean vala_method_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gboolean vala_method_is_possible_entry_point (ValaMethod* self, ValaSemanticAnalyzer* analyzer);
static void vala_method_set_entry_point (ValaMethod* self, gboolean value);
static gpointer vala_method_parent_class = NULL;
static void vala_method_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new method.
 *
 * @param name        method name
 * @param return_type method return type
 * @param source      reference to source code
 * @return            newly created method
 */
ValaMethod* vala_method_construct (GType object_type, const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaMethod* self;
	g_return_val_if_fail (return_type != NULL, NULL);
	self = (ValaMethod*) vala_member_construct (object_type, name, source_reference);
	vala_method_set_return_type (self, return_type);
	vala_method_set_carray_length_parameter_position (self, (double) (-3));
	vala_method_set_cdelegate_target_parameter_position (self, (double) (-3));
	return self;
}


ValaMethod* vala_method_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	return vala_method_construct (VALA_TYPE_METHOD, name, return_type, source_reference);
}


/**
 * Appends parameter to this method.
 *
 * @param param a formal parameter
 */
void vala_method_add_parameter (ValaMethod* self, ValaFormalParameter* param) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (param != NULL);
	/* default C parameter position*/
	vala_formal_parameter_set_cparameter_position (param, (double) (gee_collection_get_size ((GeeCollection*) self->priv->parameters) + 1));
	vala_formal_parameter_set_carray_length_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	vala_formal_parameter_set_cdelegate_target_parameter_position (param, vala_formal_parameter_get_cparameter_position (param) + 0.1);
	gee_collection_add ((GeeCollection*) self->priv->parameters, param);
	if (!vala_formal_parameter_get_ellipsis (param)) {
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) param), (ValaSymbol*) param);
	}
}


GeeList* vala_method_get_parameters (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->parameters);
}


/**
 * Remove all parameters from this method.
 */
void vala_method_clear_parameters (ValaMethod* self) {
	g_return_if_fail (self != NULL);
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			if (!vala_formal_parameter_get_ellipsis (param)) {
				vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) param));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	gee_collection_clear ((GeeCollection*) self->priv->parameters);
}


static void vala_method_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = (ValaMethod*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_method (visitor, self);
}


static void vala_method_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethod * self;
	self = (ValaMethod*) base;
	g_return_if_fail (visitor != NULL);
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_method_get_return_type (self), visitor);
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_accept ((ValaCodeNode*) param, visitor);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _error_type_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_error_type_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_error_type_it)) {
			ValaDataType* error_type;
			error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
			vala_code_node_accept ((ValaCodeNode*) error_type, visitor);
			(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
		}
		(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
	}
	if (self->priv->_result_var != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_local_variable_get_variable_type (self->priv->_result_var), visitor);
	}
	{
		GeeIterator* _precondition_it;
		_precondition_it = gee_iterable_iterator ((GeeIterable*) self->priv->preconditions);
		while (gee_iterator_next (_precondition_it)) {
			ValaExpression* precondition;
			precondition = (ValaExpression*) gee_iterator_get (_precondition_it);
			vala_code_node_accept ((ValaCodeNode*) precondition, visitor);
			(precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL));
		}
		(_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL));
	}
	{
		GeeIterator* _postcondition_it;
		_postcondition_it = gee_iterable_iterator ((GeeIterable*) self->priv->postconditions);
		while (gee_iterator_next (_postcondition_it)) {
			ValaExpression* postcondition;
			postcondition = (ValaExpression*) gee_iterator_get (_postcondition_it);
			vala_code_node_accept ((ValaCodeNode*) postcondition, visitor);
			(postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL));
		}
		(_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL));
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_method_get_body (self), visitor);
	}
}


/**
 * Returns the interface name of this method as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_method_get_cname (ValaMethod* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->cname = (_tmp0 = vala_method_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cname, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


/**
 * Returns the default interface name of this method as it is used in C
 * code.
 *
 * @return the name to be used in C code by default
 */
static char* vala_method_real_get_default_cname (ValaMethod* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = FALSE;
	if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) self), "main") == 0) {
		_tmp0 = vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) self)) == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		/* avoid conflict with generated main function*/
		return g_strdup ("_main");
	} else {
		if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) self), "_")) {
			char* _tmp2;
			char* _tmp3;
			_tmp2 = NULL;
			_tmp3 = NULL;
			return (_tmp3 = g_strdup_printf ("_%s%s", _tmp2 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), g_utf8_offset_to_pointer (vala_symbol_get_name ((ValaSymbol*) self), (glong) 1)), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
		} else {
			char* _tmp5;
			char* _tmp6;
			_tmp5 = NULL;
			_tmp6 = NULL;
			return (_tmp6 = g_strdup_printf ("%s%s", _tmp5 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), _tmp5 = (g_free (_tmp5), NULL), _tmp6);
		}
	}
}


char* vala_method_get_default_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_default_cname (self);
}


/**
 * Returns the implementation name of this data type as it is used in C
 * code.
 *
 * @return the name to be used in C code
 */
static char* vala_method_real_get_real_cname (ValaMethod* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = FALSE;
	if (vala_method_get_base_method (self) != NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = vala_method_get_base_interface_method (self) != NULL;
	}
	if (_tmp0) {
		char* _tmp1;
		char* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = g_strdup_printf ("%sreal_%s", _tmp1 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
	} else {
		return vala_method_get_cname (self);
	}
}


char* vala_method_get_real_cname (ValaMethod* self) {
	return VALA_METHOD_GET_CLASS (self)->get_real_cname (self);
}


/**
 * Sets the name of this method as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_method_set_cname (ValaMethod* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
}


static void vala_method_process_ccode_attribute (ValaMethod* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_method_set_cname (self, _tmp0 = vala_attribute_get_string (a, "cname"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp1;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp1 = NULL;
			filename_collection = _tmp1 = g_strsplit (val, ",", 0);
			filename_collection_length1 = g_strv_length (_tmp1);
			for (filename_it = 0; filename_it < g_strv_length (_tmp1); filename_it = filename_it + 1) {
				const char* _tmp2;
				char* filename;
				_tmp2 = NULL;
				filename = (_tmp2 = filename_collection[filename_it], (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
				{
					vala_member_add_cheader_filename ((ValaMember*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "vfunc_name")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_method_set_vfunc_name (self, _tmp3 = vala_attribute_get_string (a, "vfunc_name"));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "sentinel")) {
		char* _tmp4;
		_tmp4 = NULL;
		vala_method_set_sentinel (self, _tmp4 = vala_attribute_get_string (a, "sentinel"));
		_tmp4 = (g_free (_tmp4), NULL);
	}
	if (vala_attribute_has_argument (a, "instance_pos")) {
		vala_method_set_cinstance_parameter_position (self, vala_attribute_get_double (a, "instance_pos"));
	}
	if (vala_attribute_has_argument (a, "array_length")) {
		vala_method_set_no_array_length (self, !vala_attribute_get_bool (a, "array_length"));
	}
	if (vala_attribute_has_argument (a, "array_null_terminated")) {
		vala_method_set_array_null_terminated (self, vala_attribute_get_bool (a, "array_null_terminated"));
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_method_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_method_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
	if (vala_attribute_has_argument (a, "has_construct_function")) {
		vala_method_set_has_construct_function (self, vala_attribute_get_bool (a, "has_construct_function"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_method_process_attributes (ValaMethod* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_method_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "ReturnsModifiedPointer") == 0) {
						vala_method_set_returns_modified_pointer (self, TRUE);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "FloatingReference") == 0) {
							vala_data_type_set_floating_reference (vala_method_get_return_type (self), TRUE);
						} else {
							if (_vala_strcmp0 (vala_attribute_get_name (a), "PrintfFormat") == 0) {
								vala_method_set_printf_format (self, TRUE);
							}
						}
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


/**
 * Checks whether the parameters and return type of this method are
 * compatible with the specified method
 *
 * @param base_method a method
 * @param invalid_match error string about which check failed
 * @return true if the specified method is compatible to this method
 */
gboolean vala_method_compatible (ValaMethod* self, ValaMethod* base_method, char** invalid_match) {
	ValaObjectType* object_type;
	ValaDataType* actual_base_type;
	GeeIterator* method_params_it;
	gint param_index;
	gboolean _tmp21;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (base_method != NULL, FALSE);
	*invalid_match = NULL;
	object_type = NULL;
	if (VALA_IS_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		ValaObjectType* _tmp0;
		_tmp0 = NULL;
		object_type = (_tmp0 = vala_object_type_new (VALA_OBJECT_TYPE_SYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) self))), (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), _tmp0);
		{
			GeeList* _tmp1;
			GeeIterator* _tmp2;
			GeeIterator* _type_parameter_it;
			_tmp1 = NULL;
			_tmp2 = NULL;
			_type_parameter_it = (_tmp2 = gee_iterable_iterator ((GeeIterable*) (_tmp1 = vala_object_type_symbol_get_type_parameters (vala_object_type_get_type_symbol (object_type)))), (_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL)), _tmp2);
			while (gee_iterator_next (_type_parameter_it)) {
				ValaTypeParameter* type_parameter;
				ValaGenericType* type_arg;
				type_parameter = (ValaTypeParameter*) gee_iterator_get (_type_parameter_it);
				type_arg = vala_generic_type_new (type_parameter);
				vala_data_type_set_value_owned ((ValaDataType*) type_arg, TRUE);
				vala_data_type_add_type_argument ((ValaDataType*) object_type, (ValaDataType*) type_arg);
				(type_parameter == NULL) ? NULL : (type_parameter = (vala_code_node_unref (type_parameter), NULL));
				(type_arg == NULL) ? NULL : (type_arg = (vala_code_node_unref (type_arg), NULL));
			}
			(_type_parameter_it == NULL) ? NULL : (_type_parameter_it = (gee_collection_object_unref (_type_parameter_it), NULL));
		}
	}
	actual_base_type = vala_data_type_get_actual_type (vala_method_get_return_type (base_method), (ValaDataType*) object_type, (ValaCodeNode*) self);
	if (!vala_data_type_equals (vala_method_get_return_type (self), actual_base_type)) {
		char* _tmp3;
		gboolean _tmp4;
		_tmp3 = NULL;
		(*invalid_match) = (_tmp3 = g_strdup ("incompatible return type"), (*invalid_match) = (g_free ((*invalid_match)), NULL), _tmp3);
		return (_tmp4 = FALSE, (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), _tmp4);
	}
	method_params_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
	param_index = 1;
	{
		GeeIterator* _base_param_it;
		_base_param_it = gee_iterable_iterator ((GeeIterable*) base_method->priv->parameters);
		while (gee_iterator_next (_base_param_it)) {
			ValaFormalParameter* base_param;
			ValaDataType* _tmp7;
			ValaFormalParameter* _tmp8;
			gboolean _tmp9;
			base_param = (ValaFormalParameter*) gee_iterator_get (_base_param_it);
			/* this method may not expect less arguments */
			if (!gee_iterator_next (method_params_it)) {
				char* _tmp5;
				gboolean _tmp6;
				_tmp5 = NULL;
				(*invalid_match) = (_tmp5 = g_strdup ("too few parameters"), (*invalid_match) = (g_free ((*invalid_match)), NULL), _tmp5);
				return (_tmp6 = FALSE, (base_param == NULL) ? NULL : (base_param = (vala_code_node_unref (base_param), NULL)), (_base_param_it == NULL) ? NULL : (_base_param_it = (gee_collection_object_unref (_base_param_it), NULL)), (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp6);
			}
			_tmp7 = NULL;
			actual_base_type = (_tmp7 = vala_data_type_get_actual_type (vala_formal_parameter_get_parameter_type (base_param), (ValaDataType*) object_type, (ValaCodeNode*) self), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), _tmp7);
			_tmp8 = NULL;
			if ((_tmp9 = !vala_data_type_equals (actual_base_type, vala_formal_parameter_get_parameter_type (_tmp8 = (ValaFormalParameter*) gee_iterator_get (method_params_it))), (_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL)), _tmp9)) {
				char* _tmp10;
				gboolean _tmp11;
				_tmp10 = NULL;
				(*invalid_match) = (_tmp10 = g_strdup_printf ("incompatible type of parameter %d", param_index), (*invalid_match) = (g_free ((*invalid_match)), NULL), _tmp10);
				return (_tmp11 = FALSE, (base_param == NULL) ? NULL : (base_param = (vala_code_node_unref (base_param), NULL)), (_base_param_it == NULL) ? NULL : (_base_param_it = (gee_collection_object_unref (_base_param_it), NULL)), (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp11);
			}
			param_index++;
			(base_param == NULL) ? NULL : (base_param = (vala_code_node_unref (base_param), NULL));
		}
		(_base_param_it == NULL) ? NULL : (_base_param_it = (gee_collection_object_unref (_base_param_it), NULL));
	}
	/* this method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		char* _tmp12;
		gboolean _tmp13;
		_tmp12 = NULL;
		(*invalid_match) = (_tmp12 = g_strdup ("too many parameters"), (*invalid_match) = (g_free ((*invalid_match)), NULL), _tmp12);
		return (_tmp13 = FALSE, (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp13);
	}
	/* this method may throw more but not less errors than the base method */
	{
		GeeList* _tmp14;
		GeeIterator* _tmp15;
		GeeIterator* _method_error_type_it;
		/* this method may throw more but not less errors than the base method */
		_tmp14 = NULL;
		_tmp15 = NULL;
		_method_error_type_it = (_tmp15 = gee_iterable_iterator ((GeeIterable*) (_tmp14 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp14 == NULL) ? NULL : (_tmp14 = (gee_collection_object_unref (_tmp14), NULL)), _tmp15);
		/* this method may throw more but not less errors than the base method */
		while (gee_iterator_next (_method_error_type_it)) {
			ValaDataType* method_error_type;
			gboolean match;
			/* this method may throw more but not less errors than the base method */
			method_error_type = (ValaDataType*) gee_iterator_get (_method_error_type_it);
			match = FALSE;
			{
				GeeList* _tmp16;
				GeeIterator* _tmp17;
				GeeIterator* _base_method_error_type_it;
				_tmp16 = NULL;
				_tmp17 = NULL;
				_base_method_error_type_it = (_tmp17 = gee_iterable_iterator ((GeeIterable*) (_tmp16 = vala_code_node_get_error_types ((ValaCodeNode*) base_method))), (_tmp16 == NULL) ? NULL : (_tmp16 = (gee_collection_object_unref (_tmp16), NULL)), _tmp17);
				while (gee_iterator_next (_base_method_error_type_it)) {
					ValaDataType* base_method_error_type;
					base_method_error_type = (ValaDataType*) gee_iterator_get (_base_method_error_type_it);
					if (vala_data_type_compatible (method_error_type, base_method_error_type)) {
						match = TRUE;
						(base_method_error_type == NULL) ? NULL : (base_method_error_type = (vala_code_node_unref (base_method_error_type), NULL));
						break;
					}
					(base_method_error_type == NULL) ? NULL : (base_method_error_type = (vala_code_node_unref (base_method_error_type), NULL));
				}
				(_base_method_error_type_it == NULL) ? NULL : (_base_method_error_type_it = (gee_collection_object_unref (_base_method_error_type_it), NULL));
			}
			if (!match) {
				char* _tmp19;
				char* _tmp18;
				gboolean _tmp20;
				_tmp19 = NULL;
				_tmp18 = NULL;
				(*invalid_match) = (_tmp19 = g_strdup_printf ("incompatible error type `%s'", _tmp18 = vala_code_node_to_string ((ValaCodeNode*) method_error_type)), (*invalid_match) = (g_free ((*invalid_match)), NULL), _tmp19);
				_tmp18 = (g_free (_tmp18), NULL);
				return (_tmp20 = FALSE, (method_error_type == NULL) ? NULL : (method_error_type = (vala_code_node_unref (method_error_type), NULL)), (_method_error_type_it == NULL) ? NULL : (_method_error_type_it = (gee_collection_object_unref (_method_error_type_it), NULL)), (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp20);
			}
			(method_error_type == NULL) ? NULL : (method_error_type = (vala_code_node_unref (method_error_type), NULL));
		}
		(_method_error_type_it == NULL) ? NULL : (_method_error_type_it = (gee_collection_object_unref (_method_error_type_it), NULL));
	}
	return (_tmp21 = TRUE, (object_type == NULL) ? NULL : (object_type = (vala_code_node_unref (object_type), NULL)), (actual_base_type == NULL) ? NULL : (actual_base_type = (vala_code_node_unref (actual_base_type), NULL)), (method_params_it == NULL) ? NULL : (method_params_it = (gee_collection_object_unref (method_params_it), NULL)), _tmp21);
}


/**
 * Adds a precondition to this method.
 *
 * @param precondition a boolean precondition expression
 */
void vala_method_add_precondition (ValaMethod* self, ValaExpression* precondition) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (precondition != NULL);
	gee_collection_add ((GeeCollection*) self->priv->preconditions, precondition);
	vala_code_node_set_parent_node ((ValaCodeNode*) precondition, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the list of preconditions of this method.
 *
 * @return list of preconditions
 */
GeeList* vala_method_get_preconditions (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->preconditions);
}


/**
 * Adds a postcondition to this method.
 *
 * @param postcondition a boolean postcondition expression
 */
void vala_method_add_postcondition (ValaMethod* self, ValaExpression* postcondition) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (postcondition != NULL);
	gee_collection_add ((GeeCollection*) self->priv->postconditions, postcondition);
	vala_code_node_set_parent_node ((ValaCodeNode*) postcondition, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the list of postconditions of this method.
 *
 * @return list of postconditions
 */
GeeList* vala_method_get_postconditions (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->postconditions);
}


static void vala_method_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaMethod * self;
	GeeList* error_types;
	self = (ValaMethod*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_method_get_return_type (self) == old_type) {
		vala_method_set_return_type (self, new_type);
		return;
	}
	error_types = vala_code_node_get_error_types ((ValaCodeNode*) self);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) error_types); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get ((GeeList*) error_types, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set ((GeeList*) error_types, i, new_type);
				(error_types == NULL) ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL));
				return;
			}
		}
	}
	(error_types == NULL) ? NULL : (error_types = (gee_collection_object_unref (error_types), NULL));
}


static void vala_method_find_base_methods (ValaMethod* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->base_methods_valid) {
		return;
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		/* VAPI classes don't specify overridden methods */
		if (!vala_symbol_get_external_package (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			if (!(VALA_IS_CREATION_METHOD (self))) {
				gboolean _tmp0;
				gboolean _tmp1;
				vala_method_find_base_interface_method (self, VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
				_tmp0 = FALSE;
				_tmp1 = FALSE;
				if (self->priv->_is_virtual) {
					_tmp1 = TRUE;
				} else {
					_tmp1 = self->priv->_is_abstract;
				}
				if (_tmp1) {
					_tmp0 = TRUE;
				} else {
					_tmp0 = self->priv->_overrides;
				}
				if (_tmp0) {
					vala_method_find_base_class_method (self, VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
				}
			}
		} else {
			gboolean _tmp2;
			_tmp2 = FALSE;
			if (self->priv->_is_virtual) {
				_tmp2 = TRUE;
			} else {
				_tmp2 = self->priv->_is_abstract;
			}
			if (_tmp2) {
				self->priv->_base_method = self;
			}
		}
	} else {
		if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			gboolean _tmp3;
			_tmp3 = FALSE;
			if (self->priv->_is_virtual) {
				_tmp3 = TRUE;
			} else {
				_tmp3 = self->priv->_is_abstract;
			}
			if (_tmp3) {
				ValaMethod* _tmp5;
				ValaMethod* _tmp4;
				_tmp5 = NULL;
				_tmp4 = NULL;
				self->priv->_base_interface_method = (_tmp5 = (_tmp4 = self, (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4)), (self->priv->_base_interface_method == NULL) ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL)), _tmp5);
			}
		}
	}
	self->priv->base_methods_valid = TRUE;
}


static void vala_method_find_base_class_method (ValaMethod* self, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) cl), vala_symbol_get_name ((ValaSymbol*) self));
	if (VALA_IS_METHOD (sym)) {
		ValaMethod* _tmp0;
		ValaMethod* base_method;
		gboolean _tmp1;
		_tmp0 = NULL;
		base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		_tmp1 = FALSE;
		if (base_method->priv->_is_abstract) {
			_tmp1 = TRUE;
		} else {
			_tmp1 = base_method->priv->_is_virtual;
		}
		if (_tmp1) {
			char* invalid_match;
			char* _tmp4;
			gboolean _tmp3;
			char* _tmp2;
			invalid_match = NULL;
			_tmp4 = NULL;
			_tmp2 = NULL;
			if (!(_tmp3 = vala_method_compatible (self, base_method, &_tmp2), invalid_match = (_tmp4 = _tmp2, invalid_match = (g_free (invalid_match), NULL), _tmp4), _tmp3)) {
				char* _tmp7;
				char* _tmp6;
				char* _tmp5;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", _tmp5 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) base_method), invalid_match));
				_tmp7 = (g_free (_tmp7), NULL);
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				invalid_match = (g_free (invalid_match), NULL);
				(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				return;
			}
			self->priv->_base_method = base_method;
			invalid_match = (g_free (invalid_match), NULL);
			(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
			(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			return;
		}
		(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
	} else {
		if (VALA_IS_SIGNAL (sym)) {
			ValaSignal* _tmp8;
			ValaSignal* sig;
			_tmp8 = NULL;
			sig = (_tmp8 = VALA_SIGNAL (sym), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
			if (vala_signal_get_is_virtual (sig)) {
				ValaMethod* base_method;
				char* invalid_match;
				char* _tmp11;
				gboolean _tmp10;
				char* _tmp9;
				base_method = vala_signal_get_method_handler (sig);
				invalid_match = NULL;
				_tmp11 = NULL;
				_tmp9 = NULL;
				if (!(_tmp10 = vala_method_compatible (self, base_method, &_tmp9), invalid_match = (_tmp11 = _tmp9, invalid_match = (g_free (invalid_match), NULL), _tmp11), _tmp10)) {
					char* _tmp14;
					char* _tmp13;
					char* _tmp12;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp14 = NULL;
					_tmp13 = NULL;
					_tmp12 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp14 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", _tmp12 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp13 = vala_symbol_get_full_name ((ValaSymbol*) base_method), invalid_match));
					_tmp14 = (g_free (_tmp14), NULL);
					_tmp13 = (g_free (_tmp13), NULL);
					_tmp12 = (g_free (_tmp12), NULL);
					(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
					invalid_match = (g_free (invalid_match), NULL);
					(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
					(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
					return;
				}
				self->priv->_base_method = base_method;
				(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
				invalid_match = (g_free (invalid_match), NULL);
				(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
				return;
			}
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_method_find_base_class_method (self, vala_class_get_base_class (cl));
	}
	(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
}


static void vala_method_find_base_interface_method (ValaMethod* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	/* FIXME report error if multiple possible base methods are found*/
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _type_it;
		/* FIXME report error if multiple possible base methods are found*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		_type_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_class_get_base_types (cl))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		/* FIXME report error if multiple possible base methods are found*/
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			/* FIXME report error if multiple possible base methods are found*/
			type = (ValaDataType*) gee_iterator_get (_type_it);
			if (VALA_IS_INTERFACE (vala_data_type_get_data_type (type))) {
				ValaSymbol* sym;
				sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_data_type_get_data_type (type)), vala_symbol_get_name ((ValaSymbol*) self));
				if (VALA_IS_METHOD (sym)) {
					ValaMethod* _tmp2;
					ValaMethod* base_method;
					gboolean _tmp3;
					_tmp2 = NULL;
					base_method = (_tmp2 = VALA_METHOD (sym), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
					_tmp3 = FALSE;
					if (base_method->priv->_is_abstract) {
						_tmp3 = TRUE;
					} else {
						_tmp3 = base_method->priv->_is_virtual;
					}
					if (_tmp3) {
						char* invalid_match;
						char* _tmp6;
						gboolean _tmp5;
						char* _tmp4;
						ValaMethod* _tmp11;
						ValaMethod* _tmp10;
						invalid_match = NULL;
						_tmp6 = NULL;
						_tmp4 = NULL;
						if (!(_tmp5 = vala_method_compatible (self, base_method, &_tmp4), invalid_match = (_tmp6 = _tmp4, invalid_match = (g_free (invalid_match), NULL), _tmp6), _tmp5)) {
							char* _tmp9;
							char* _tmp8;
							char* _tmp7;
							vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
							_tmp9 = NULL;
							_tmp8 = NULL;
							_tmp7 = NULL;
							vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp9 = g_strdup_printf ("overriding method `%s' is incompatible with base method `%s': %s.", _tmp7 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp8 = vala_symbol_get_full_name ((ValaSymbol*) base_method), invalid_match));
							_tmp9 = (g_free (_tmp9), NULL);
							_tmp8 = (g_free (_tmp8), NULL);
							_tmp7 = (g_free (_tmp7), NULL);
							invalid_match = (g_free (invalid_match), NULL);
							(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
							(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
							(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
							(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
							return;
						}
						_tmp11 = NULL;
						_tmp10 = NULL;
						self->priv->_base_interface_method = (_tmp11 = (_tmp10 = base_method, (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (self->priv->_base_interface_method == NULL) ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL)), _tmp11);
						invalid_match = (g_free (invalid_match), NULL);
						(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
						(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
						(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
						(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
						return;
					}
					(base_method == NULL) ? NULL : (base_method = (vala_code_node_unref (base_method), NULL));
				}
				(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
}


static gboolean vala_method_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaMethod * self;
	gboolean _tmp9;
	ValaSourceFile* _tmp14;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp15;
	ValaSymbol* old_symbol;
	ValaClass* _tmp16;
	ValaClass* old_class;
	ValaStruct* _tmp17;
	ValaStruct* old_struct;
	ValaDataType* _tmp18;
	ValaDataType* old_return_type;
	ValaDataType* _tmp24;
	ValaDataType* _tmp23;
	ValaAttribute* init_attr;
	ValaClass* _tmp28;
	ValaClass* _tmp27;
	ValaStruct* _tmp30;
	ValaStruct* _tmp29;
	ValaDataType* _tmp32;
	ValaDataType* _tmp31;
	gboolean _tmp56;
	gboolean _tmp63;
	self = (ValaMethod*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_method_process_attributes (self);
	if (self->priv->_is_abstract) {
		if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			ValaClass* _tmp1;
			ValaClass* cl;
			_tmp1 = NULL;
			cl = (_tmp1 = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			if (!vala_class_get_is_abstract (cl)) {
				gboolean _tmp2;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Abstract methods may not be declared in non-abstract classes");
				return (_tmp2 = FALSE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp2);
			}
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		} else {
			if (!(VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self)))) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Abstract methods may not be declared outside of classes and interfaces");
				return FALSE;
			}
		}
	} else {
		if (self->priv->_is_virtual) {
			gboolean _tmp4;
			_tmp4 = FALSE;
			if (!(VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)))) {
				_tmp4 = !(VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) self)));
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Virtual methods may not be declared outside of classes and interfaces");
				return FALSE;
			}
			if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
				ValaClass* _tmp6;
				ValaClass* cl;
				_tmp6 = NULL;
				cl = (_tmp6 = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6));
				if (vala_class_get_is_compact (cl)) {
					gboolean _tmp7;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Virtual methods may not be declared in compact classes");
					return (_tmp7 = FALSE, (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp7);
				}
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			}
		} else {
			if (self->priv->_overrides) {
				if (!(VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)))) {
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Methods may not be overridden outside of classes");
					return FALSE;
				}
			}
		}
	}
	_tmp9 = FALSE;
	if (self->priv->_is_abstract) {
		_tmp9 = vala_method_get_body (self) != NULL;
	} else {
		_tmp9 = FALSE;
	}
	if (_tmp9) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Abstract methods cannot have bodies");
	} else {
		gboolean _tmp10;
		_tmp10 = FALSE;
		if (vala_symbol_get_external ((ValaSymbol*) self)) {
			_tmp10 = vala_method_get_body (self) != NULL;
		} else {
			_tmp10 = FALSE;
		}
		if (_tmp10) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Extern methods cannot have bodies");
		} else {
			gboolean _tmp11;
			gboolean _tmp12;
			gboolean _tmp13;
			_tmp11 = FALSE;
			_tmp12 = FALSE;
			_tmp13 = FALSE;
			if (!self->priv->_is_abstract) {
				_tmp13 = !vala_symbol_get_external ((ValaSymbol*) self);
			} else {
				_tmp13 = FALSE;
			}
			if (_tmp13) {
				_tmp12 = !vala_symbol_get_external_package ((ValaSymbol*) self);
			} else {
				_tmp12 = FALSE;
			}
			if (_tmp12) {
				_tmp11 = vala_method_get_body (self) == NULL;
			} else {
				_tmp11 = FALSE;
			}
			if (_tmp11) {
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Non-abstract, non-extern methods must have bodies");
			}
		}
	}
	_tmp14 = NULL;
	old_source_file = (_tmp14 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp14 == NULL) ? NULL : vala_source_file_ref (_tmp14));
	_tmp15 = NULL;
	old_symbol = (_tmp15 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp15 == NULL) ? NULL : vala_code_node_ref (_tmp15));
	_tmp16 = NULL;
	old_class = (_tmp16 = analyzer->current_class, (_tmp16 == NULL) ? NULL : vala_code_node_ref (_tmp16));
	_tmp17 = NULL;
	old_struct = (_tmp17 = analyzer->current_struct, (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17));
	_tmp18 = NULL;
	old_return_type = (_tmp18 = analyzer->current_return_type, (_tmp18 == NULL) ? NULL : vala_code_node_ref (_tmp18));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
		ValaClass* _tmp20;
		ValaClass* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		analyzer->current_class = (_tmp20 = (_tmp19 = VALA_CLASS (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp19 == NULL) ? NULL : vala_code_node_ref (_tmp19)), (analyzer->current_class == NULL) ? NULL : (analyzer->current_class = (vala_code_node_unref (analyzer->current_class), NULL)), _tmp20);
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) self))) {
			ValaStruct* _tmp22;
			ValaStruct* _tmp21;
			_tmp22 = NULL;
			_tmp21 = NULL;
			analyzer->current_struct = (_tmp22 = (_tmp21 = VALA_STRUCT (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp21 == NULL) ? NULL : vala_code_node_ref (_tmp21)), (analyzer->current_struct == NULL) ? NULL : (analyzer->current_struct = (vala_code_node_unref (analyzer->current_struct), NULL)), _tmp22);
		}
	}
	_tmp24 = NULL;
	_tmp23 = NULL;
	analyzer->current_return_type = (_tmp24 = (_tmp23 = vala_method_get_return_type (self), (_tmp23 == NULL) ? NULL : vala_code_node_ref (_tmp23)), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp24);
	vala_code_node_check ((ValaCodeNode*) vala_method_get_return_type (self), analyzer);
	init_attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "ModuleInit");
	if (init_attr != NULL) {
		vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self))), self);
	}
	if (!vala_symbol_is_internal_symbol ((ValaSymbol*) self)) {
		if (VALA_IS_VALUE_TYPE (vala_method_get_return_type (self))) {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_method_get_return_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
		} else {
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_method_get_return_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	}
	vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_method_get_return_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	if (vala_method_get_return_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_method_get_return_type (self), analyzer);
	}
	{
		GeeIterator* _param_it;
		_param_it = gee_iterable_iterator ((GeeIterable*) self->priv->parameters);
		while (gee_iterator_next (_param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (_param_it);
			vala_code_node_check ((ValaCodeNode*) param, analyzer);
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(_param_it == NULL) ? NULL : (_param_it = (gee_collection_object_unref (_param_it), NULL));
	}
	{
		GeeList* _tmp25;
		GeeIterator* _tmp26;
		GeeIterator* _error_type_it;
		_tmp25 = NULL;
		_tmp26 = NULL;
		_error_type_it = (_tmp26 = gee_iterable_iterator ((GeeIterable*) (_tmp25 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp25 == NULL) ? NULL : (_tmp25 = (gee_collection_object_unref (_tmp25), NULL)), _tmp26);
		while (gee_iterator_next (_error_type_it)) {
			ValaDataType* error_type;
			error_type = (ValaDataType*) gee_iterator_get (_error_type_it);
			vala_code_node_check ((ValaCodeNode*) error_type, analyzer);
			(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
		}
		(_error_type_it == NULL) ? NULL : (_error_type_it = (gee_collection_object_unref (_error_type_it), NULL));
	}
	if (self->priv->_result_var != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_local_variable_get_variable_type (self->priv->_result_var), analyzer);
	}
	{
		GeeIterator* _precondition_it;
		_precondition_it = gee_iterable_iterator ((GeeIterable*) self->priv->preconditions);
		while (gee_iterator_next (_precondition_it)) {
			ValaExpression* precondition;
			precondition = (ValaExpression*) gee_iterator_get (_precondition_it);
			vala_code_node_check ((ValaCodeNode*) precondition, analyzer);
			(precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL));
		}
		(_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL));
	}
	{
		GeeIterator* _postcondition_it;
		_postcondition_it = gee_iterable_iterator ((GeeIterable*) self->priv->postconditions);
		while (gee_iterator_next (_postcondition_it)) {
			ValaExpression* postcondition;
			postcondition = (ValaExpression*) gee_iterator_get (_postcondition_it);
			vala_code_node_check ((ValaCodeNode*) postcondition, analyzer);
			(postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL));
		}
		(_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL));
	}
	if (vala_method_get_body (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_method_get_body (self), analyzer);
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	_tmp28 = NULL;
	_tmp27 = NULL;
	analyzer->current_class = (_tmp28 = (_tmp27 = old_class, (_tmp27 == NULL) ? NULL : vala_code_node_ref (_tmp27)), (analyzer->current_class == NULL) ? NULL : (analyzer->current_class = (vala_code_node_unref (analyzer->current_class), NULL)), _tmp28);
	_tmp30 = NULL;
	_tmp29 = NULL;
	analyzer->current_struct = (_tmp30 = (_tmp29 = old_struct, (_tmp29 == NULL) ? NULL : vala_code_node_ref (_tmp29)), (analyzer->current_struct == NULL) ? NULL : (analyzer->current_struct = (vala_code_node_unref (analyzer->current_struct), NULL)), _tmp30);
	_tmp32 = NULL;
	_tmp31 = NULL;
	analyzer->current_return_type = (_tmp32 = (_tmp31 = old_return_type, (_tmp31 == NULL) ? NULL : vala_code_node_ref (_tmp31)), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp32);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (vala_semantic_analyzer_get_current_symbol (analyzer)))) {
		ValaMethod* _tmp33;
		ValaMethod* up_method;
		ValaDataType* _tmp35;
		ValaDataType* _tmp34;
		/* lambda expressions produce nested methods */
		_tmp33 = NULL;
		up_method = (_tmp33 = VALA_METHOD (vala_symbol_get_parent_symbol (vala_semantic_analyzer_get_current_symbol (analyzer))), (_tmp33 == NULL) ? NULL : vala_code_node_ref (_tmp33));
		_tmp35 = NULL;
		_tmp34 = NULL;
		analyzer->current_return_type = (_tmp35 = (_tmp34 = vala_method_get_return_type (up_method), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34)), (analyzer->current_return_type == NULL) ? NULL : (analyzer->current_return_type = (vala_code_node_unref (analyzer->current_return_type), NULL)), _tmp35);
		(up_method == NULL) ? NULL : (up_method = (vala_code_node_unref (up_method), NULL));
	}
	if (VALA_IS_STRUCT (vala_semantic_analyzer_get_current_symbol (analyzer))) {
		gboolean _tmp36;
		gboolean _tmp37;
		_tmp36 = FALSE;
		_tmp37 = FALSE;
		if (self->priv->_is_abstract) {
			_tmp37 = TRUE;
		} else {
			_tmp37 = self->priv->_is_virtual;
		}
		if (_tmp37) {
			_tmp36 = TRUE;
		} else {
			_tmp36 = self->priv->_overrides;
		}
		if (_tmp36) {
			char* _tmp39;
			char* _tmp38;
			gboolean _tmp40;
			_tmp39 = NULL;
			_tmp38 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp39 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", _tmp38 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp39 = (g_free (_tmp39), NULL);
			_tmp38 = (g_free (_tmp38), NULL);
			return (_tmp40 = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp40);
		}
	} else {
		gboolean _tmp41;
		_tmp41 = FALSE;
		if (self->priv->_overrides) {
			_tmp41 = vala_method_get_base_method (self) == NULL;
		} else {
			_tmp41 = FALSE;
		}
		if (_tmp41) {
			char* _tmp43;
			char* _tmp42;
			_tmp43 = NULL;
			_tmp42 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp43 = g_strdup_printf ("%s: no suitable method found to override", _tmp42 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp43 = (g_free (_tmp43), NULL);
			_tmp42 = (g_free (_tmp42), NULL);
		}
	}
	/* check whether return type is at least as accessible as the method*/
	if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, vala_method_get_return_type (self))) {
		char* _tmp46;
		char* _tmp45;
		char* _tmp44;
		gboolean _tmp47;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		_tmp46 = NULL;
		_tmp45 = NULL;
		_tmp44 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp46 = g_strdup_printf ("return type `%s` is less accessible than method `%s`", _tmp44 = vala_code_node_to_string ((ValaCodeNode*) vala_method_get_return_type (self)), _tmp45 = vala_symbol_get_full_name ((ValaSymbol*) self)));
		_tmp46 = (g_free (_tmp46), NULL);
		_tmp45 = (g_free (_tmp45), NULL);
		_tmp44 = (g_free (_tmp44), NULL);
		return (_tmp47 = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp47);
	}
	{
		GeeList* _tmp48;
		GeeIterator* _tmp49;
		GeeIterator* _precondition_it;
		_tmp48 = NULL;
		_tmp49 = NULL;
		_precondition_it = (_tmp49 = gee_iterable_iterator ((GeeIterable*) (_tmp48 = vala_method_get_preconditions (self))), (_tmp48 == NULL) ? NULL : (_tmp48 = (gee_collection_object_unref (_tmp48), NULL)), _tmp49);
		while (gee_iterator_next (_precondition_it)) {
			ValaExpression* precondition;
			precondition = (ValaExpression*) gee_iterator_get (_precondition_it);
			if (vala_code_node_get_error ((ValaCodeNode*) precondition)) {
				gboolean _tmp50;
				/* if there was an error in the precondition, skip this check*/
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				return (_tmp50 = FALSE, (precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL)), (_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp50);
			}
			if (!vala_data_type_compatible (vala_expression_get_value_type (precondition), analyzer->bool_type)) {
				gboolean _tmp51;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) precondition), "Precondition must be boolean");
				return (_tmp51 = FALSE, (precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL)), (_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp51);
			}
			(precondition == NULL) ? NULL : (precondition = (vala_code_node_unref (precondition), NULL));
		}
		(_precondition_it == NULL) ? NULL : (_precondition_it = (gee_collection_object_unref (_precondition_it), NULL));
	}
	{
		GeeList* _tmp52;
		GeeIterator* _tmp53;
		GeeIterator* _postcondition_it;
		_tmp52 = NULL;
		_tmp53 = NULL;
		_postcondition_it = (_tmp53 = gee_iterable_iterator ((GeeIterable*) (_tmp52 = vala_method_get_postconditions (self))), (_tmp52 == NULL) ? NULL : (_tmp52 = (gee_collection_object_unref (_tmp52), NULL)), _tmp53);
		while (gee_iterator_next (_postcondition_it)) {
			ValaExpression* postcondition;
			postcondition = (ValaExpression*) gee_iterator_get (_postcondition_it);
			if (vala_code_node_get_error ((ValaCodeNode*) postcondition)) {
				gboolean _tmp54;
				/* if there was an error in the postcondition, skip this check*/
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				return (_tmp54 = FALSE, (postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL)), (_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp54);
			}
			if (!vala_data_type_compatible (vala_expression_get_value_type (postcondition), analyzer->bool_type)) {
				gboolean _tmp55;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) postcondition), "Postcondition must be boolean");
				return (_tmp55 = FALSE, (postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL)), (_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp55);
			}
			(postcondition == NULL) ? NULL : (postcondition = (vala_code_node_unref (postcondition), NULL));
		}
		(_postcondition_it == NULL) ? NULL : (_postcondition_it = (gee_collection_object_unref (_postcondition_it), NULL));
	}
	_tmp56 = FALSE;
	if (vala_code_node_get_tree_can_fail ((ValaCodeNode*) self)) {
		_tmp56 = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) self), "main") == 0;
	} else {
		_tmp56 = FALSE;
	}
	if (_tmp56) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "\"main\" method cannot throw errors");
	}
	/* check that all errors that can be thrown in the method body are declared*/
	if (vala_method_get_body (self) != NULL) {
		{
			GeeList* _tmp57;
			GeeIterator* _tmp58;
			GeeIterator* _body_error_type_it;
			_tmp57 = NULL;
			_tmp58 = NULL;
			_body_error_type_it = (_tmp58 = gee_iterable_iterator ((GeeIterable*) (_tmp57 = vala_code_node_get_error_types ((ValaCodeNode*) vala_method_get_body (self)))), (_tmp57 == NULL) ? NULL : (_tmp57 = (gee_collection_object_unref (_tmp57), NULL)), _tmp58);
			while (gee_iterator_next (_body_error_type_it)) {
				ValaDataType* body_error_type;
				gboolean can_propagate_error;
				body_error_type = (ValaDataType*) gee_iterator_get (_body_error_type_it);
				can_propagate_error = FALSE;
				{
					GeeList* _tmp59;
					GeeIterator* _tmp60;
					GeeIterator* _method_error_type_it;
					_tmp59 = NULL;
					_tmp60 = NULL;
					_method_error_type_it = (_tmp60 = gee_iterable_iterator ((GeeIterable*) (_tmp59 = vala_code_node_get_error_types ((ValaCodeNode*) self))), (_tmp59 == NULL) ? NULL : (_tmp59 = (gee_collection_object_unref (_tmp59), NULL)), _tmp60);
					while (gee_iterator_next (_method_error_type_it)) {
						ValaDataType* method_error_type;
						method_error_type = (ValaDataType*) gee_iterator_get (_method_error_type_it);
						if (vala_data_type_compatible (body_error_type, method_error_type)) {
							can_propagate_error = TRUE;
						}
						(method_error_type == NULL) ? NULL : (method_error_type = (vala_code_node_unref (method_error_type), NULL));
					}
					(_method_error_type_it == NULL) ? NULL : (_method_error_type_it = (gee_collection_object_unref (_method_error_type_it), NULL));
				}
				if (!can_propagate_error) {
					char* _tmp62;
					char* _tmp61;
					_tmp62 = NULL;
					_tmp61 = NULL;
					vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) body_error_type), _tmp62 = g_strdup_printf ("unhandled error `%s'", _tmp61 = vala_code_node_to_string ((ValaCodeNode*) body_error_type)));
					_tmp62 = (g_free (_tmp62), NULL);
					_tmp61 = (g_free (_tmp61), NULL);
				}
				(body_error_type == NULL) ? NULL : (body_error_type = (vala_code_node_unref (body_error_type), NULL));
			}
			(_body_error_type_it == NULL) ? NULL : (_body_error_type_it = (gee_collection_object_unref (_body_error_type_it), NULL));
		}
	}
	if (vala_method_is_possible_entry_point (self, analyzer)) {
		vala_method_set_entry_point (self, TRUE);
	}
	return (_tmp63 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_class == NULL) ? NULL : (old_class = (vala_code_node_unref (old_class), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), (old_return_type == NULL) ? NULL : (old_return_type = (vala_code_node_unref (old_return_type), NULL)), (init_attr == NULL) ? NULL : (init_attr = (vala_code_node_unref (init_attr), NULL)), _tmp63);
}


static gboolean vala_method_is_possible_entry_point (ValaMethod* self, ValaSemanticAnalyzer* analyzer) {
	gboolean _tmp0;
	GeeList* params;
	GeeIterator* params_it;
	ValaFormalParameter* param;
	ValaArrayType* _tmp8;
	ValaArrayType* array_type;
	gboolean _tmp10;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (analyzer != NULL, FALSE);
	_tmp0 = FALSE;
	if (vala_symbol_get_name ((ValaSymbol*) self) == NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) self), "main") != 0;
	}
	if (_tmp0) {
		/* method must be called "main"*/
		return FALSE;
	}
	if (self->priv->_binding == MEMBER_BINDING_INSTANCE) {
		/* method must be static*/
		return FALSE;
	}
	if (VALA_IS_VOID_TYPE (vala_method_get_return_type (self))) {
	} else {
		if (vala_data_type_get_data_type (vala_method_get_return_type (self)) == vala_data_type_get_data_type (analyzer->int_type)) {
		} else {
			/* return type must be void or int*/
			return FALSE;
		}
	}
	params = vala_method_get_parameters (self);
	if (gee_collection_get_size ((GeeCollection*) params) == 0) {
		gboolean _tmp4;
		/* method may have no parameters*/
		return (_tmp4 = TRUE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp4);
	}
	if (gee_collection_get_size ((GeeCollection*) params) > 1) {
		gboolean _tmp5;
		/* method must not have more than one parameter*/
		return (_tmp5 = FALSE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp5);
	}
	params_it = gee_iterable_iterator ((GeeIterable*) params);
	gee_iterator_next (params_it);
	param = (ValaFormalParameter*) gee_iterator_get (params_it);
	if (vala_formal_parameter_get_direction (param) == VALA_PARAMETER_DIRECTION_OUT) {
		gboolean _tmp6;
		/* parameter must not be an out parameter*/
		return (_tmp6 = FALSE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp6);
	}
	if (!(VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)))) {
		gboolean _tmp7;
		/* parameter must be an array*/
		return (_tmp7 = FALSE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), _tmp7);
	}
	_tmp8 = NULL;
	array_type = (_tmp8 = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
	if (vala_data_type_get_data_type (vala_array_type_get_element_type (array_type)) != vala_data_type_get_data_type (analyzer->string_type)) {
		gboolean _tmp9;
		/* parameter must be an array of strings*/
		return (_tmp9 = FALSE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp9);
	}
	return (_tmp10 = TRUE, (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (params_it == NULL) ? NULL : (params_it = (gee_collection_object_unref (params_it), NULL)), (param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp10);
}


ValaDataType* vala_method_get_return_type (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_return_type;
}


void vala_method_set_return_type (ValaMethod* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_return_type, (ValaCodeNode*) self);
}


ValaBlock* vala_method_get_body (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_method_set_body (ValaMethod* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
	if (self->priv->_body != NULL) {
		vala_symbol_set_owner ((ValaSymbol*) self->priv->_body, vala_symbol_get_scope ((ValaSymbol*) self));
	}
}


ValaBasicBlock* vala_method_get_entry_block (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_entry_block;
}


void vala_method_set_entry_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_entry_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_entry_block == NULL) ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL)), _tmp2);
}


ValaBasicBlock* vala_method_get_exit_block (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_exit_block;
}


void vala_method_set_exit_block (ValaMethod* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp2;
	ValaBasicBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_exit_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_basic_block_ref (_tmp1)), (self->priv->_exit_block == NULL) ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL)), _tmp2);
}


MemberBinding vala_method_get_binding (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_binding;
}


void vala_method_set_binding (ValaMethod* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


const char* vala_method_get_vfunc_name (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_vfunc_name == NULL) {
		char* _tmp1;
		const char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->_vfunc_name = (_tmp1 = (_tmp0 = vala_symbol_get_name ((ValaSymbol*) self), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL), _tmp1);
	}
	return self->priv->_vfunc_name;
}


void vala_method_set_vfunc_name (ValaMethod* self, const char* value) {
	char* _tmp4;
	const char* _tmp3;
	g_return_if_fail (self != NULL);
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->_vfunc_name = (_tmp4 = (_tmp3 = value, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL), _tmp4);
}


const char* vala_method_get_sentinel (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_sentinel == NULL) {
		return VALA_METHOD_DEFAULT_SENTINEL;
	}
	return self->priv->_sentinel;
}


void vala_method_set_sentinel (ValaMethod* self, const char* value) {
	char* _tmp3;
	const char* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_sentinel = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL), _tmp3);
}


gboolean vala_method_get_is_abstract (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_abstract;
}


void vala_method_set_is_abstract (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_abstract = value;
}


gboolean vala_method_get_is_virtual (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_virtual;
}


void vala_method_set_is_virtual (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_virtual = value;
}


gboolean vala_method_get_overrides (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_overrides;
}


void vala_method_set_overrides (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_overrides = value;
}


gboolean vala_method_get_is_inline (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_inline;
}


void vala_method_set_is_inline (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_inline = value;
}


gboolean vala_method_get_returns_modified_pointer (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_returns_modified_pointer;
}


void vala_method_set_returns_modified_pointer (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_returns_modified_pointer = value;
}


ValaMethod* vala_method_get_base_method (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_method_find_base_methods (self);
	return self->priv->_base_method;
}


ValaMethod* vala_method_get_base_interface_method (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	vala_method_find_base_methods (self);
	return self->priv->_base_interface_method;
}


gboolean vala_method_get_entry_point (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_entry_point;
}


static void vala_method_set_entry_point (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_entry_point = value;
}


ValaFormalParameter* vala_method_get_this_parameter (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_this_parameter;
}


void vala_method_set_this_parameter (ValaMethod* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_this_parameter == NULL) ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)), _tmp2);
}


ValaLocalVariable* vala_method_get_result_var (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_result_var;
}


void vala_method_set_result_var (ValaMethod* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_result_var = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_result_var == NULL) ? NULL : (self->priv->_result_var = (vala_code_node_unref (self->priv->_result_var), NULL)), _tmp2);
}


double vala_method_get_cinstance_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cinstance_parameter_position;
}


void vala_method_set_cinstance_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cinstance_parameter_position = value;
}


double vala_method_get_carray_length_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_method_set_carray_length_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_carray_length_parameter_position = value;
}


double vala_method_get_cdelegate_target_parameter_position (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_method_set_cdelegate_target_parameter_position (ValaMethod* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cdelegate_target_parameter_position = value;
}


gboolean vala_method_get_no_array_length (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_array_length;
}


void vala_method_set_no_array_length (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


gboolean vala_method_get_array_null_terminated (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_array_null_terminated;
}


void vala_method_set_array_null_terminated (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_array_null_terminated = value;
}


gboolean vala_method_get_printf_format (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_printf_format;
}


void vala_method_set_printf_format (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_printf_format = value;
}


gboolean vala_method_get_has_construct_function (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_has_construct_function;
}


void vala_method_set_has_construct_function (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_has_construct_function = value;
}


gboolean vala_method_get_coroutine (ValaMethod* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_coroutine;
}


void vala_method_set_coroutine (ValaMethod* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_coroutine = value;
}


static void vala_method_class_init (ValaMethodClass * klass) {
	vala_method_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_method_finalize;
	g_type_class_add_private (klass, sizeof (ValaMethodPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_real_accept_children;
	VALA_METHOD_CLASS (klass)->get_default_cname = vala_method_real_get_default_cname;
	VALA_METHOD_CLASS (klass)->get_real_cname = vala_method_real_get_real_cname;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_method_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_method_real_check;
}


static void vala_method_instance_init (ValaMethod * self) {
	self->priv = VALA_METHOD_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
	self->priv->_has_construct_function = TRUE;
	self->priv->parameters = (GeeList*) gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->preconditions = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->postconditions = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_method_finalize (ValaCodeNode* obj) {
	ValaMethod * self;
	self = VALA_METHOD (obj);
	(self->priv->_entry_block == NULL) ? NULL : (self->priv->_entry_block = (vala_basic_block_unref (self->priv->_entry_block), NULL));
	(self->priv->_exit_block == NULL) ? NULL : (self->priv->_exit_block = (vala_basic_block_unref (self->priv->_exit_block), NULL));
	(self->priv->_this_parameter == NULL) ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL));
	(self->priv->_result_var == NULL) ? NULL : (self->priv->_result_var = (vala_code_node_unref (self->priv->_result_var), NULL));
	(self->priv->parameters == NULL) ? NULL : (self->priv->parameters = (gee_collection_object_unref (self->priv->parameters), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->_vfunc_name = (g_free (self->priv->_vfunc_name), NULL);
	self->priv->_sentinel = (g_free (self->priv->_sentinel), NULL);
	(self->priv->preconditions == NULL) ? NULL : (self->priv->preconditions = (gee_collection_object_unref (self->priv->preconditions), NULL));
	(self->priv->postconditions == NULL) ? NULL : (self->priv->postconditions = (gee_collection_object_unref (self->priv->postconditions), NULL));
	(self->priv->_return_type == NULL) ? NULL : (self->priv->_return_type = (vala_code_node_unref (self->priv->_return_type), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	(self->priv->_base_interface_method == NULL) ? NULL : (self->priv->_base_interface_method = (vala_code_node_unref (self->priv->_base_interface_method), NULL));
	VALA_CODE_NODE_CLASS (vala_method_parent_class)->finalize (obj);
}


GType vala_method_get_type (void) {
	static GType vala_method_type_id = 0;
	if (vala_method_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethod), 0, (GInstanceInitFunc) vala_method_instance_init, NULL };
		vala_method_type_id = g_type_register_static (VALA_TYPE_MEMBER, "ValaMethod", &g_define_type_info, 0);
	}
	return vala_method_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




