/* valagidlparser.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vapigen/valagidlparser.h>
#include <vala/valasourcereference.h>
#include <vala/valatypesymbol.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/hashmap.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valacodenode.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gidlparser.h>
#include <gidlmodule.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gidlnode.h>
#include <vala/valadatatype.h>
#include <vala/valaformalparameter.h>
#include <vala/valaunresolvedtype.h>
#include <gmetadata.h>
#include <vala/valapointertype.h>
#include <vala/valavoidtype.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valaarraytype.h>
#include <stdio.h>
#include <vala/valamethod.h>
#include <vala/valacreationmethod.h>
#include <gee/collection.h>
#include <vala/valamember.h>
#include <float.h>
#include <math.h>
#include <vala/valafield.h>
#include <vala/valaexpression.h>
#include <gee/arraylist.h>
#include <vala/valaenumvalue.h>
#include <vala/valaerrorcode.h>
#include <gee/hashset.h>
#include <vala/valaattribute.h>
#include <vala/valaproperty.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valasignal.h>
#include <vala/valaobjecttypesymbol.h>
#include <vala/valaconstant.h>




static char* string_substring (const char* self, glong offset, glong len);
struct _ValaGIdlParserPrivate {
	ValaCodeContext* context;
	ValaSourceFile* current_source_file;
	ValaSourceReference* current_source_reference;
	ValaNamespace* current_namespace;
	ValaTypeSymbol* current_data_type;
	GeeMap* codenode_attributes_map;
	GeeMap* codenode_attributes_patterns;
	GeeSet* current_type_symbol_set;
	GeeMap* cname_type_map;
};

#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gidl_parser_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed);
static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d);
static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypeSymbol* t);
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void _g_list_free_g_idl_module_free (GList* self);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns);
static void _g_list_free_vala_code_node_unref (GList* self);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname);
static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module);
static ValaTypeSymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static ValaDataType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node, ValaParameterDirection* direction);
static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name);
static ValaDataType* vala_gidl_parser_parse_type_string (ValaGIdlParser* self, const char* n);
static ValaDataType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param, ValaParameterDirection* direction);
static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface);
static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface);
static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode, int* result_length1);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_finalize (ValaCodeVisitor* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context) {
	GeeMap* _tmp0;
	ValaCodeContext* _tmp2;
	ValaCodeContext* _tmp1;
	GeeMap* _tmp3;
	g_return_if_fail (self != NULL);
	g_return_if_fail (context != NULL);
	_tmp0 = NULL;
	self->priv->cname_type_map = (_tmp0 = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, VALA_TYPE_TYPESYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL)), _tmp0);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->context = (_tmp2 = (_tmp1 = context, (_tmp1 == NULL) ? NULL : vala_code_context_ref (_tmp1)), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp2);
	vala_code_context_accept (context, (ValaCodeVisitor*) self);
	_tmp3 = NULL;
	self->priv->cname_type_map = (_tmp3 = NULL, (self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL)), _tmp3);
}


static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (ns != NULL);
	vala_code_node_accept_children ((ValaCodeNode*) ns, (ValaCodeVisitor*) self);
}


static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (cl != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) cl);
}


static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (st != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) st);
}


static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (iface != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) iface);
}


static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (en != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) en);
}


static void vala_gidl_parser_real_visit_error_domain (ValaCodeVisitor* base, ValaErrorDomain* ed) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (ed != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) ed);
}


static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (d != NULL);
	vala_gidl_parser_visit_type (self, (ValaTypeSymbol*) d);
}


static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypeSymbol* t) {
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	g_return_if_fail (t != NULL);
	_tmp0 = NULL;
	if ((_tmp1 = !gee_map_contains (self->priv->cname_type_map, _tmp0 = vala_typesymbol_get_cname (t, FALSE)), _tmp0 = (g_free (_tmp0), NULL), _tmp1)) {
		char* _tmp2;
		_tmp2 = NULL;
		gee_map_set ((GeeMap*) self->priv->cname_type_map, _tmp2 = vala_typesymbol_get_cname (t, FALSE), t);
		_tmp2 = (g_free (_tmp2), NULL);
	}
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGIdlParser * self;
	self = (ValaGIdlParser*) base;
	g_return_if_fail (source_file != NULL);
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gi")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void _g_list_free_g_idl_module_free (GList* self) {
	g_list_foreach (self, (GFunc) g_idl_module_free, NULL);
	g_list_free (self);
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file) {
	GError * inner_error;
	char* _tmp0;
	char* _tmp1;
	char* metadata_filename;
	ValaSourceFile* _tmp3;
	ValaSourceFile* _tmp2;
	GeeMap* _tmp4;
	GeeMap* _tmp5;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_file != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	_tmp1 = NULL;
	metadata_filename = (_tmp1 = g_strdup_printf ("%s.metadata", _tmp0 = g_strndup (vala_source_file_get_filename (source_file), (gsize) (strlen (vala_source_file_get_filename (source_file)) - strlen (".gi")))), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_source_file = (_tmp3 = (_tmp2 = source_file, (_tmp2 == NULL) ? NULL : vala_source_file_ref (_tmp2)), (self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL)), _tmp3);
	_tmp4 = NULL;
	self->priv->codenode_attributes_map = (_tmp4 = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal), (self->priv->codenode_attributes_map == NULL) ? NULL : (self->priv->codenode_attributes_map = (gee_collection_object_unref (self->priv->codenode_attributes_map), NULL)), _tmp4);
	_tmp5 = NULL;
	self->priv->codenode_attributes_patterns = (_tmp5 = (GeeMap*) gee_hash_map_new (G_TYPE_POINTER, NULL, NULL, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_hash, (GEqualFunc) g_pattern_spec_equal, g_direct_equal), (self->priv->codenode_attributes_patterns == NULL) ? NULL : (self->priv->codenode_attributes_patterns = (gee_collection_object_unref (self->priv->codenode_attributes_patterns), NULL)), _tmp5);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			char* _tmp8;
			gboolean _tmp7;
			char* _tmp6;
			metadata = NULL;
			_tmp8 = NULL;
			_tmp6 = NULL;
			_tmp7 = g_file_get_contents (metadata_filename, &_tmp6, NULL, &inner_error);
			metadata = (_tmp8 = _tmp6, metadata = (g_free (metadata), NULL), _tmp8);
			_tmp7;
			if (inner_error != NULL) {
				metadata = (g_free (metadata), NULL);
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			{
				char** _tmp9;
				char** line_collection;
				int line_collection_length1;
				int line_it;
				_tmp9 = NULL;
				line_collection = _tmp9 = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = g_strv_length (_tmp9);
				for (line_it = 0; line_it < g_strv_length (_tmp9); line_it = line_it + 1) {
					const char* _tmp12;
					char* line;
					_tmp12 = NULL;
					line = (_tmp12 = line_collection[line_it], (_tmp12 == NULL) ? NULL : g_strdup (_tmp12));
					{
						char** _tmp11;
						gint tokens_size;
						gint tokens_length1;
						char** _tmp10;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							/* ignore comment lines*/
							line = (g_free (line), NULL);
							continue;
						}
						_tmp11 = NULL;
						_tmp10 = NULL;
						tokens = (_tmp11 = _tmp10 = g_strsplit (line, " ", 2), tokens_length1 = g_strv_length (_tmp10), tokens_size = tokens_length1, _tmp11);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
							continue;
						}
						if (NULL != g_utf8_strchr (tokens[0], (glong) (-1), (gunichar) '*')) {
							GPatternSpec* pattern;
							pattern = g_pattern_spec_new (tokens[0]);
							gee_map_set ((GeeMap*) self->priv->codenode_attributes_patterns, pattern, tokens[0]);
						}
						gee_map_set ((GeeMap*) self->priv->codenode_attributes_map, tokens[0], tokens[1]);
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp13;
				_tmp13 = NULL;
				vala_report_error (NULL, _tmp13 = g_strdup_printf ("Unable to read metadata file: %s", e->message));
				_tmp13 = (g_free (_tmp13), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (inner_error != NULL) {
			metadata_filename = (g_free (metadata_filename), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	{
		GList* modules;
		ValaSourceReference* _tmp14;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			goto __finally1;
		}
		_tmp14 = NULL;
		self->priv->current_source_reference = (_tmp14 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL) ? NULL : (self->priv->current_source_reference = (vala_source_reference_unref (self->priv->current_source_reference), NULL)), _tmp14);
		{
			GList* module_collection;
			GList* module_it;
			module_collection = modules;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = (GIdlModule*) module_it->data;
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					if (ns != NULL) {
						vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
					}
					(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
				}
			}
		}
		(modules == NULL) ? NULL : (modules = (_g_list_free_g_idl_module_free (modules), NULL));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally1:
	if (inner_error != NULL) {
		metadata_filename = (g_free (metadata_filename), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	metadata_filename = (g_free (metadata_filename), NULL);
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns) {
	char** _tmp1;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp0;
	char** attributes;
	const char* _tmp18;
	char* _tmp19;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (ns != NULL, NULL);
	_tmp1 = NULL;
	attributes = (_tmp1 = vala_gidl_parser_get_attributes (self, type_name, &_tmp0), attributes_length1 = _tmp0, attributes_size = attributes_length1, _tmp1);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp5;
				char* attr;
				_tmp5 = NULL;
				attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
				{
					char** _tmp3;
					gint nv_size;
					gint nv_length1;
					char** _tmp2;
					char** nv;
					_tmp3 = NULL;
					_tmp2 = NULL;
					nv = (_tmp3 = _tmp2 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp2), nv_size = nv_length1, _tmp3);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp4;
						_tmp4 = NULL;
						return (_tmp4 = vala_gidl_parser_eval (self, nv[1]), attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp4);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (g_str_has_prefix (type_name, vala_symbol_get_name ((ValaSymbol*) ns))) {
		const char* _tmp6;
		char* _tmp7;
		_tmp6 = NULL;
		_tmp7 = NULL;
		return (_tmp7 = (_tmp6 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) ns), -1)), (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp7);
	} else {
		gboolean _tmp8;
		_tmp8 = FALSE;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) ns), "GLib") == 0) {
			_tmp8 = g_str_has_prefix (type_name, "G");
		} else {
			_tmp8 = FALSE;
		}
		if (_tmp8) {
			const char* _tmp9;
			char* _tmp10;
			_tmp9 = NULL;
			_tmp10 = NULL;
			return (_tmp10 = (_tmp9 = g_utf8_offset_to_pointer (type_name, (glong) 1), (_tmp9 == NULL) ? NULL : g_strdup (_tmp9)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10);
		} else {
			char* best_match;
			best_match = NULL;
			{
				GeeList* _tmp11;
				GeeIterator* _tmp12;
				GeeIterator* _cprefix_it;
				_tmp11 = NULL;
				_tmp12 = NULL;
				_cprefix_it = (_tmp12 = gee_iterable_iterator ((GeeIterable*) (_tmp11 = vala_namespace_get_cprefixes (ns))), (_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL)), _tmp12);
				while (gee_iterator_next (_cprefix_it)) {
					char* cprefix;
					cprefix = (char*) gee_iterator_get (_cprefix_it);
					if (g_str_has_prefix (type_name, cprefix)) {
						gboolean _tmp13;
						_tmp13 = FALSE;
						if (best_match == NULL) {
							_tmp13 = TRUE;
						} else {
							_tmp13 = g_utf8_strlen (cprefix, -1) > g_utf8_strlen (best_match, -1);
						}
						if (_tmp13) {
							char* _tmp15;
							const char* _tmp14;
							_tmp15 = NULL;
							_tmp14 = NULL;
							best_match = (_tmp15 = (_tmp14 = cprefix, (_tmp14 == NULL) ? NULL : g_strdup (_tmp14)), best_match = (g_free (best_match), NULL), _tmp15);
						}
					}
					cprefix = (g_free (cprefix), NULL);
				}
				(_cprefix_it == NULL) ? NULL : (_cprefix_it = (gee_collection_object_unref (_cprefix_it), NULL));
			}
			if (best_match != NULL) {
				const char* _tmp16;
				char* _tmp17;
				_tmp16 = NULL;
				_tmp17 = NULL;
				return (_tmp17 = (_tmp16 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (best_match, -1)), (_tmp16 == NULL) ? NULL : g_strdup (_tmp16)), best_match = (g_free (best_match), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp17);
			}
			best_match = (g_free (best_match), NULL);
		}
	}
	_tmp18 = NULL;
	_tmp19 = NULL;
	return (_tmp19 = (_tmp18 = type_name, (_tmp18 == NULL) ? NULL : g_strdup (_tmp18)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp19);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns) {
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	const char* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (ns != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = g_str_has_prefix (const_name, _tmp1 = g_strconcat (_tmp0 = g_utf8_strup (vala_symbol_get_name ((ValaSymbol*) ns), -1), "_", NULL)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2)) {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) ns), -1) + 1), (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
	} else {
		gboolean _tmp5;
		_tmp5 = FALSE;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) ns), "GLib") == 0) {
			_tmp5 = g_str_has_prefix (const_name, "G_");
		} else {
			_tmp5 = FALSE;
		}
		if (_tmp5) {
			const char* _tmp6;
			_tmp6 = NULL;
			return (_tmp6 = g_utf8_offset_to_pointer (const_name, (glong) 2), (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
		}
	}
	_tmp8 = NULL;
	return (_tmp8 = const_name, (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
}


static void _g_list_free_vala_code_node_unref (GList* self) {
	g_list_foreach (self, (GFunc) vala_code_node_unref, NULL);
	g_list_free (self);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module) {
	ValaSymbol* sym;
	ValaNamespace* ns;
	ValaNamespace* _tmp5;
	ValaNamespace* _tmp4;
	char** _tmp7;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp6;
	char** attributes;
	ValaNamespace* _tmp25;
	ValaNamespace* _tmp27;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) vala_code_context_get_root (self->priv->context)), module->name);
	ns = NULL;
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp1;
		ValaNamespace* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		ns = (_tmp1 = (_tmp0 = VALA_NAMESPACE (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp1);
		if (vala_symbol_get_external_package ((ValaSymbol*) ns)) {
			GList* _tmp2;
			_tmp2 = NULL;
			((ValaCodeNode*) ns)->attributes = (_tmp2 = NULL, (((ValaCodeNode*) ns)->attributes == NULL) ? NULL : (((ValaCodeNode*) ns)->attributes = (_g_list_free_vala_code_node_unref (((ValaCodeNode*) ns)->attributes), NULL)), _tmp2);
			vala_code_node_set_source_reference ((ValaCodeNode*) ns, self->priv->current_source_reference);
		}
	} else {
		ValaNamespace* _tmp3;
		_tmp3 = NULL;
		ns = (_tmp3 = vala_namespace_new (module->name, self->priv->current_source_reference), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), _tmp3);
	}
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_namespace = (_tmp5 = (_tmp4 = ns, (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4)), (self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL)), _tmp5);
	_tmp7 = NULL;
	attributes = (_tmp7 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name ((ValaSymbol*) ns), &_tmp6), attributes_length1 = _tmp6, attributes_size = attributes_length1, _tmp7);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp17;
				char* attr;
				_tmp17 = NULL;
				attr = (_tmp17 = attr_collection[attr_it], (_tmp17 == NULL) ? NULL : g_strdup (_tmp17));
				{
					char** _tmp9;
					gint nv_size;
					gint nv_length1;
					char** _tmp8;
					char** nv;
					_tmp9 = NULL;
					_tmp8 = NULL;
					nv = (_tmp9 = _tmp8 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp8), nv_size = nv_length1, _tmp9);
					if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
						char* _tmp10;
						_tmp10 = NULL;
						vala_namespace_set_cheader_filename (ns, _tmp10 = vala_gidl_parser_eval (self, nv[1]));
						_tmp10 = (g_free (_tmp10), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
							char** _tmp14;
							gint cprefixes_size;
							gint cprefixes_length1;
							char** _tmp12;
							char* _tmp11;
							char** _tmp13;
							char** cprefixes;
							_tmp14 = NULL;
							_tmp12 = NULL;
							_tmp11 = NULL;
							_tmp13 = NULL;
							cprefixes = (_tmp14 = (_tmp13 = _tmp12 = g_strsplit (_tmp11 = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp11 = (g_free (_tmp11), NULL), _tmp13), cprefixes_length1 = g_strv_length (_tmp12), cprefixes_size = cprefixes_length1, _tmp14);
							{
								char** name_collection;
								int name_collection_length1;
								int name_it;
								name_collection = cprefixes;
								name_collection_length1 = cprefixes_length1;
								for (name_it = 0; name_it < cprefixes_length1; name_it = name_it + 1) {
									const char* _tmp15;
									char* name;
									_tmp15 = NULL;
									name = (_tmp15 = name_collection[name_it], (_tmp15 == NULL) ? NULL : g_strdup (_tmp15));
									{
										vala_namespace_add_cprefix (ns, name);
										name = (g_free (name), NULL);
									}
								}
							}
							cprefixes = (_vala_array_free (cprefixes, cprefixes_length1, (GDestroyNotify) g_free), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "lower_case_cprefix") == 0) {
								char* _tmp16;
								_tmp16 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, _tmp16 = vala_gidl_parser_eval (self, nv[1]));
								_tmp16 = (g_free (_tmp16), NULL);
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	{
		GList* node_collection;
		GList* node_it;
		node_collection = module->entries;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = (GIdlNode*) node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaDelegate* cb;
					char* _tmp18;
					cb = vala_gidl_parser_parse_delegate (self, (GIdlNodeFunction*) node);
					if (cb == NULL) {
						(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
						continue;
					}
					_tmp18 = NULL;
					vala_symbol_set_name ((ValaSymbol*) cb, _tmp18 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) cb), ns));
					_tmp18 = (g_free (_tmp18), NULL);
					vala_namespace_add_delegate (ns, cb);
					vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cb);
					(cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						vala_gidl_parser_parse_struct (self, (GIdlNodeStruct*) node, ns, module);
					} else {
						if (node->type == G_IDL_NODE_UNION) {
							vala_gidl_parser_parse_union (self, (GIdlNodeUnion*) node, ns, module);
						} else {
							if (node->type == G_IDL_NODE_BOXED) {
								vala_gidl_parser_parse_boxed (self, (GIdlNodeBoxed*) node, ns, module);
							} else {
								if (node->type == G_IDL_NODE_ENUM) {
									ValaTypeSymbol* en;
									char* _tmp19;
									en = vala_gidl_parser_parse_enum (self, (GIdlNodeEnum*) node);
									if (en == NULL) {
										(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
										continue;
									}
									_tmp19 = NULL;
									vala_symbol_set_name ((ValaSymbol*) en, _tmp19 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) en), ns));
									_tmp19 = (g_free (_tmp19), NULL);
									if (VALA_IS_ERROR_DOMAIN (en)) {
										ValaTypeSymbol* _tmp20;
										_tmp20 = NULL;
										vala_namespace_add_error_domain (ns, (_tmp20 = en, VALA_IS_ERROR_DOMAIN (_tmp20) ? ((ValaErrorDomain*) _tmp20) : NULL));
									} else {
										ValaTypeSymbol* _tmp21;
										_tmp21 = NULL;
										vala_namespace_add_enum (ns, (_tmp21 = en, VALA_IS_ENUM (_tmp21) ? ((ValaEnum*) _tmp21) : NULL));
									}
									vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) en);
									(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
								} else {
									if (node->type == G_IDL_NODE_FLAGS) {
										ValaTypeSymbol* _tmp22;
										ValaEnum* en;
										char* _tmp23;
										_tmp22 = NULL;
										en = (_tmp22 = vala_gidl_parser_parse_enum (self, (GIdlNodeEnum*) node), VALA_IS_ENUM (_tmp22) ? ((ValaEnum*) _tmp22) : NULL);
										if (en == NULL) {
											(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
											continue;
										}
										_tmp23 = NULL;
										vala_symbol_set_name ((ValaSymbol*) en, _tmp23 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name ((ValaSymbol*) en), ns));
										_tmp23 = (g_free (_tmp23), NULL);
										vala_enum_set_is_flags (en, TRUE);
										vala_namespace_add_enum (ns, en);
										vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) en);
										(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
									} else {
										if (node->type == G_IDL_NODE_OBJECT) {
											vala_gidl_parser_parse_object (self, (GIdlNodeInterface*) node, ns, module);
										} else {
											if (node->type == G_IDL_NODE_INTERFACE) {
												vala_gidl_parser_parse_interface (self, (GIdlNodeInterface*) node, ns, module);
											} else {
												if (node->type == G_IDL_NODE_CONSTANT) {
													ValaConstant* c;
													c = vala_gidl_parser_parse_constant (self, (GIdlNodeConstant*) node);
													if (c != NULL) {
														char* _tmp24;
														_tmp24 = NULL;
														vala_symbol_set_name ((ValaSymbol*) c, _tmp24 = vala_gidl_parser_fix_const_name (self, vala_symbol_get_name ((ValaSymbol*) c), ns));
														_tmp24 = (g_free (_tmp24), NULL);
														vala_namespace_add_constant (ns, c);
														vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) c);
													}
													(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
												} else {
													if (node->type == G_IDL_NODE_FUNCTION) {
														ValaMethod* m;
														m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) node, FALSE);
														if (m != NULL) {
															vala_method_set_binding (m, MEMBER_BINDING_STATIC);
															vala_namespace_add_method (ns, m);
															vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) m);
														}
														(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp25 = NULL;
	self->priv->current_namespace = (_tmp25 = NULL, (self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL)), _tmp25);
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp26;
		_tmp26 = NULL;
		return (_tmp26 = NULL, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp26);
	}
	_tmp27 = NULL;
	return (_tmp27 = ns, (sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27);
}


static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node) {
	GIdlNode* node;
	ValaDataType* _tmp0;
	ValaDelegate* _tmp1;
	ValaDelegate* cb;
	gboolean check_has_target;
	char** _tmp3;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp2;
	char** attributes;
	guint remaining_params;
	ValaDelegate* _tmp34;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = (GIdlNode*) f_node;
	_tmp0 = NULL;
	_tmp1 = NULL;
	cb = (_tmp1 = vala_delegate_new (node->name, _tmp0 = vala_gidl_parser_parse_param (self, f_node->result, NULL), self->priv->current_source_reference), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1);
	vala_symbol_set_access ((ValaSymbol*) cb, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	check_has_target = TRUE;
	_tmp3 = NULL;
	attributes = (_tmp3 = vala_gidl_parser_get_attributes (self, node->name, &_tmp2), attributes_length1 = _tmp2, attributes_size = attributes_length1, _tmp3);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp14;
				char* attr;
				_tmp14 = NULL;
				attr = (_tmp14 = attr_collection[attr_it], (_tmp14 == NULL) ? NULL : g_strdup (_tmp14));
				{
					char** _tmp5;
					gint nv_size;
					gint nv_length1;
					char** _tmp4;
					char** nv;
					_tmp5 = NULL;
					_tmp4 = NULL;
					nv = (_tmp5 = _tmp4 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp4), nv_size = nv_length1, _tmp5);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp6;
						gboolean _tmp7;
						_tmp6 = NULL;
						if ((_tmp7 = _vala_strcmp0 (_tmp6 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp6 = (g_free (_tmp6), NULL), _tmp7)) {
							ValaDelegate* _tmp8;
							_tmp8 = NULL;
							return (_tmp8 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (cb == NULL) ? NULL : (cb = (vala_code_node_unref (cb), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp8);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp9;
							_tmp9 = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cb, _tmp9 = vala_gidl_parser_eval (self, nv[1]));
							_tmp9 = (g_free (_tmp9), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "has_target") == 0) {
								char* _tmp10;
								gboolean _tmp11;
								_tmp10 = NULL;
								if ((_tmp11 = _vala_strcmp0 (_tmp10 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp10 = (g_free (_tmp10), NULL), _tmp11)) {
									check_has_target = FALSE;
								} else {
									char* _tmp12;
									gboolean _tmp13;
									_tmp12 = NULL;
									if ((_tmp13 = _vala_strcmp0 (_tmp12 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp12 = (g_free (_tmp12), NULL), _tmp13)) {
										vala_delegate_set_has_target (cb, TRUE);
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	remaining_params = g_list_length (f_node->parameters);
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				gboolean _tmp15;
				gboolean _tmp16;
				param_node = (GIdlNode*) param;
				_tmp15 = FALSE;
				_tmp16 = FALSE;
				if (check_has_target) {
					_tmp16 = remaining_params == 1;
				} else {
					_tmp16 = FALSE;
				}
				if (_tmp16) {
					gboolean _tmp17;
					_tmp17 = FALSE;
					if (_vala_strcmp0 (param_node->name, "user_data") == 0) {
						_tmp17 = TRUE;
					} else {
						_tmp17 = _vala_strcmp0 (param_node->name, "data") == 0;
					}
					_tmp15 = (_tmp17);
				} else {
					_tmp15 = FALSE;
				}
				if (_tmp15) {
					/* hide user_data parameter for instance delegates*/
					vala_delegate_set_has_target (cb, TRUE);
				} else {
					const char* _tmp18;
					char* param_name;
					ValaParameterDirection direction;
					ValaDataType* _tmp21;
					ValaFormalParameter* _tmp22;
					ValaFormalParameter* p;
					gboolean hide_param;
					gboolean show_param;
					char** _tmp25;
					gint _tmp24;
					char* _tmp23;
					gboolean _tmp33;
					_tmp18 = NULL;
					param_name = (_tmp18 = param_node->name, (_tmp18 == NULL) ? NULL : g_strdup (_tmp18));
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp19;
						/* avoid conflict with string type*/
						_tmp19 = NULL;
						param_name = (_tmp19 = g_strdup ("str"), param_name = (g_free (param_name), NULL), _tmp19);
					} else {
						if (_vala_strcmp0 (param_name, "self") == 0) {
							char* _tmp20;
							/* avoid conflict with delegate target*/
							_tmp20 = NULL;
							param_name = (_tmp20 = g_strdup ("_self"), param_name = (g_free (param_name), NULL), _tmp20);
						}
					}
					direction = 0;
					_tmp21 = NULL;
					_tmp22 = NULL;
					p = (_tmp22 = vala_formal_parameter_new (param_name, _tmp21 = vala_gidl_parser_parse_param (self, param, &direction), NULL), (_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL)), _tmp22);
					vala_formal_parameter_set_direction (p, direction);
					hide_param = FALSE;
					show_param = FALSE;
					_tmp25 = NULL;
					_tmp23 = NULL;
					attributes = (_tmp25 = vala_gidl_parser_get_attributes (self, _tmp23 = g_strdup_printf ("%s.%s", node->name, param_node->name), &_tmp24), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp24, attributes_size = attributes_length1, _tmp25);
					_tmp23 = (g_free (_tmp23), NULL);
					if (attributes != NULL) {
						{
							char** attr_collection;
							int attr_collection_length1;
							int attr_it;
							attr_collection = attributes;
							attr_collection_length1 = attributes_length1;
							for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
								const char* _tmp32;
								char* attr;
								_tmp32 = NULL;
								attr = (_tmp32 = attr_collection[attr_it], (_tmp32 == NULL) ? NULL : g_strdup (_tmp32));
								{
									char** _tmp27;
									gint nv_size;
									gint nv_length1;
									char** _tmp26;
									char** nv;
									_tmp27 = NULL;
									_tmp26 = NULL;
									nv = (_tmp27 = _tmp26 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp26), nv_size = nv_length1, _tmp27);
									if (_vala_strcmp0 (nv[0], "hidden") == 0) {
										char* _tmp28;
										gboolean _tmp29;
										_tmp28 = NULL;
										if ((_tmp29 = _vala_strcmp0 (_tmp28 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp28 = (g_free (_tmp28), NULL), _tmp29)) {
											hide_param = TRUE;
										} else {
											char* _tmp30;
											gboolean _tmp31;
											_tmp30 = NULL;
											if ((_tmp31 = _vala_strcmp0 (_tmp30 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp30 = (g_free (_tmp30), NULL), _tmp31)) {
												show_param = TRUE;
											}
										}
									}
									attr = (g_free (attr), NULL);
									nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
								}
							}
						}
					}
					_tmp33 = FALSE;
					if (show_param) {
						_tmp33 = TRUE;
					} else {
						_tmp33 = !hide_param;
					}
					if (_tmp33) {
						vala_delegate_add_parameter (cb, p);
					}
					param_name = (g_free (param_name), NULL);
					(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
				}
				remaining_params--;
			}
		}
	}
	_tmp34 = NULL;
	return (_tmp34 = cb, attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp34);
}


static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname) {
	char** _tmp1;
	gint st_attributes_size;
	gint st_attributes_length1;
	gint _tmp0;
	char** st_attributes;
	gboolean _tmp8;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (cname != NULL, FALSE);
	_tmp1 = NULL;
	st_attributes = (_tmp1 = vala_gidl_parser_get_attributes (self, cname, &_tmp0), st_attributes_length1 = _tmp0, st_attributes_size = st_attributes_length1, _tmp1);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = st_attributes;
			attr_collection_length1 = st_attributes_length1;
			for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp7;
				char* attr;
				_tmp7 = NULL;
				attr = (_tmp7 = attr_collection[attr_it], (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
				{
					char** _tmp3;
					gint nv_size;
					gint nv_length1;
					char** _tmp2;
					char** nv;
					gboolean _tmp4;
					_tmp3 = NULL;
					_tmp2 = NULL;
					nv = (_tmp3 = _tmp2 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp2), nv_size = nv_length1, _tmp3);
					_tmp4 = FALSE;
					if (_vala_strcmp0 (nv[0], "is_value_type") == 0) {
						char* _tmp5;
						_tmp5 = NULL;
						_tmp4 = _vala_strcmp0 (_tmp5 = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
						_tmp5 = (g_free (_tmp5), NULL);
					} else {
						_tmp4 = FALSE;
					}
					if (_tmp4) {
						gboolean _tmp6;
						return (_tmp6 = FALSE, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp6);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	return (_tmp8 = TRUE, st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp8);
}


static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (self != NULL);
	g_return_if_fail (st_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) st_node;
	if (st_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaTypeSymbol* _tmp13;
		ValaTypeSymbol* _tmp12;
		ValaTypeSymbol* _tmp14;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp3;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp2;
			char** st_attributes;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp1);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp3 = NULL;
			st_attributes = (_tmp3 = vala_gidl_parser_get_attributes (self, node->name, &_tmp2), st_attributes_length1 = _tmp2, st_attributes_size = st_attributes_length1, _tmp3);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp11;
						char* attr;
						_tmp11 = NULL;
						attr = (_tmp11 = attr_collection[attr_it], (_tmp11 == NULL) ? NULL : g_strdup (_tmp11));
						{
							char** _tmp5;
							gint nv_size;
							gint nv_length1;
							char** _tmp4;
							char** nv;
							_tmp5 = NULL;
							_tmp4 = NULL;
							nv = (_tmp5 = _tmp4 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp4), nv_size = nv_length1, _tmp5);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp6;
								_tmp6 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp6 = vala_gidl_parser_eval (self, nv[1]));
								_tmp6 = (g_free (_tmp6), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp7;
									gboolean _tmp8;
									_tmp7 = NULL;
									if ((_tmp8 = _vala_strcmp0 (_tmp7 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp7 = (g_free (_tmp7), NULL), _tmp8)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
										(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
										name = (g_free (name), NULL);
										return;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "simple_type") == 0) {
										char* _tmp9;
										gboolean _tmp10;
										_tmp9 = NULL;
										if ((_tmp10 = _vala_strcmp0 (_tmp9 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp9 = (g_free (_tmp9), NULL), _tmp10)) {
											vala_struct_set_simple_type (st, TRUE);
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp13 = NULL;
		_tmp12 = NULL;
		self->priv->current_data_type = (_tmp13 = (_tmp12 = (ValaTypeSymbol*) st, (_tmp12 == NULL) ? NULL : vala_code_node_ref (_tmp12)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp13);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp14 = NULL;
		self->priv->current_data_type = (_tmp14 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp14);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp15;
		ValaClass* cl;
		ValaTypeSymbol* _tmp34;
		ValaTypeSymbol* _tmp33;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp44;
		_tmp15 = NULL;
		cl = (_tmp15 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp15) ? ((ValaClass*) _tmp15) : NULL);
		if (cl == NULL) {
			char* base_class;
			ValaClass* _tmp16;
			char** _tmp18;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp17;
			char** cl_attributes;
			base_class = NULL;
			_tmp16 = NULL;
			cl = (_tmp16 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp16);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp18 = NULL;
			cl_attributes = (_tmp18 = vala_gidl_parser_get_attributes (self, node->name, &_tmp17), cl_attributes_length1 = _tmp17, cl_attributes_size = cl_attributes_length1, _tmp18);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp32;
						char* attr;
						_tmp32 = NULL;
						attr = (_tmp32 = attr_collection[attr_it], (_tmp32 == NULL) ? NULL : g_strdup (_tmp32));
						{
							char** _tmp20;
							gint nv_size;
							gint nv_length1;
							char** _tmp19;
							char** nv;
							_tmp20 = NULL;
							_tmp19 = NULL;
							nv = (_tmp20 = _tmp19 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp19), nv_size = nv_length1, _tmp20);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp21;
								_tmp21 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp21 = vala_gidl_parser_eval (self, nv[1]));
								_tmp21 = (g_free (_tmp21), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "base_class") == 0) {
									char* _tmp22;
									_tmp22 = NULL;
									base_class = (_tmp22 = vala_gidl_parser_eval (self, nv[1]), base_class = (g_free (base_class), NULL), _tmp22);
								} else {
									if (_vala_strcmp0 (nv[0], "hidden") == 0) {
										char* _tmp23;
										gboolean _tmp24;
										_tmp23 = NULL;
										if ((_tmp24 = _vala_strcmp0 (_tmp23 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp23 = (g_free (_tmp23), NULL), _tmp24)) {
											attr = (g_free (attr), NULL);
											nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
											base_class = (g_free (base_class), NULL);
											cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
											(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
											name = (g_free (name), NULL);
											return;
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_immutable") == 0) {
											char* _tmp25;
											gboolean _tmp26;
											_tmp25 = NULL;
											if ((_tmp26 = _vala_strcmp0 (_tmp25 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25 = (g_free (_tmp25), NULL), _tmp26)) {
												vala_class_set_is_immutable (cl, TRUE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "is_fundamental") == 0) {
												char* _tmp27;
												gboolean _tmp28;
												_tmp27 = NULL;
												if ((_tmp28 = _vala_strcmp0 (_tmp27 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp27 = (g_free (_tmp27), NULL), _tmp28)) {
													vala_class_set_is_compact (cl, FALSE);
												}
											} else {
												gboolean _tmp29;
												_tmp29 = FALSE;
												if (_vala_strcmp0 (nv[0], "abstract") == 0) {
													_tmp29 = base_class != NULL;
												} else {
													_tmp29 = FALSE;
												}
												if (_tmp29) {
													char* _tmp30;
													gboolean _tmp31;
													_tmp30 = NULL;
													if ((_tmp31 = _vala_strcmp0 (_tmp30 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30 = (g_free (_tmp30), NULL), _tmp31)) {
														vala_class_set_is_abstract (cl, TRUE);
													}
												}
											}
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			if (base_class != NULL) {
				ValaDataType* parent;
				parent = vala_gidl_parser_parse_type_string (self, base_class);
				vala_class_add_base_type (cl, parent);
				(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
			}
			base_class = (g_free (base_class), NULL);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp34 = NULL;
		_tmp33 = NULL;
		self->priv->current_data_type = (_tmp34 = (_tmp33 = (ValaTypeSymbol*) cl, (_tmp33 == NULL) ? NULL : vala_code_node_ref (_tmp33)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp34);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp36;
							const char* _tmp35;
							_tmp36 = NULL;
							_tmp35 = NULL;
							ref_function = (_tmp36 = (_tmp35 = ((GIdlNodeFunction*) member)->symbol, (_tmp35 == NULL) ? NULL : g_strdup (_tmp35)), ref_function = (g_free (ref_function), NULL), _tmp36);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp38;
								const char* _tmp37;
								_tmp38 = NULL;
								_tmp37 = NULL;
								unref_function = (_tmp38 = (_tmp37 = ((GIdlNodeFunction*) member)->symbol, (_tmp37 == NULL) ? NULL : g_strdup (_tmp37)), unref_function = (g_free (unref_function), NULL), _tmp38);
							} else {
								gboolean _tmp39;
								_tmp39 = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp39 = TRUE;
								} else {
									_tmp39 = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp39) {
									char* _tmp41;
									const char* _tmp40;
									_tmp41 = NULL;
									_tmp40 = NULL;
									free_function = (_tmp41 = (_tmp40 = ((GIdlNodeFunction*) member)->symbol, (_tmp40 == NULL) ? NULL : g_strdup (_tmp40)), free_function = (g_free (free_function), NULL), _tmp41);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp43;
										const char* _tmp42;
										_tmp43 = NULL;
										_tmp42 = NULL;
										copy_function = (_tmp43 = (_tmp42 = ((GIdlNodeFunction*) member)->symbol, (_tmp42 == NULL) ? NULL : g_strdup (_tmp42)), copy_function = (g_free (copy_function), NULL), _tmp43);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp44 = NULL;
		self->priv->current_data_type = (_tmp44 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp44);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (self != NULL);
	g_return_if_fail (un_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) un_node;
	if (un_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaTypeSymbol* _tmp11;
		ValaTypeSymbol* _tmp10;
		ValaTypeSymbol* _tmp12;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp3;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp2;
			char** st_attributes;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp1);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp3 = NULL;
			st_attributes = (_tmp3 = vala_gidl_parser_get_attributes (self, node->name, &_tmp2), st_attributes_length1 = _tmp2, st_attributes_size = st_attributes_length1, _tmp3);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp9;
						char* attr;
						_tmp9 = NULL;
						attr = (_tmp9 = attr_collection[attr_it], (_tmp9 == NULL) ? NULL : g_strdup (_tmp9));
						{
							char** _tmp5;
							gint nv_size;
							gint nv_length1;
							char** _tmp4;
							char** nv;
							_tmp5 = NULL;
							_tmp4 = NULL;
							nv = (_tmp5 = _tmp4 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp4), nv_size = nv_length1, _tmp5);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp6;
								_tmp6 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp6 = vala_gidl_parser_eval (self, nv[1]));
								_tmp6 = (g_free (_tmp6), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp7;
									gboolean _tmp8;
									_tmp7 = NULL;
									if ((_tmp8 = _vala_strcmp0 (_tmp7 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp7 = (g_free (_tmp7), NULL), _tmp8)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
										(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp11 = NULL;
		_tmp10 = NULL;
		self->priv->current_data_type = (_tmp11 = (_tmp10 = (ValaTypeSymbol*) st, (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp11);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp12 = NULL;
		self->priv->current_data_type = (_tmp12 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp12);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp13;
		ValaClass* cl;
		ValaTypeSymbol* _tmp24;
		ValaTypeSymbol* _tmp23;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp34;
		_tmp13 = NULL;
		cl = (_tmp13 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp13) ? ((ValaClass*) _tmp13) : NULL);
		if (cl == NULL) {
			ValaClass* _tmp14;
			char** _tmp16;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp15;
			char** cl_attributes;
			_tmp14 = NULL;
			cl = (_tmp14 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp14);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp16 = NULL;
			cl_attributes = (_tmp16 = vala_gidl_parser_get_attributes (self, node->name, &_tmp15), cl_attributes_length1 = _tmp15, cl_attributes_size = cl_attributes_length1, _tmp16);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp22;
						char* attr;
						_tmp22 = NULL;
						attr = (_tmp22 = attr_collection[attr_it], (_tmp22 == NULL) ? NULL : g_strdup (_tmp22));
						{
							char** _tmp18;
							gint nv_size;
							gint nv_length1;
							char** _tmp17;
							char** nv;
							_tmp18 = NULL;
							_tmp17 = NULL;
							nv = (_tmp18 = _tmp17 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp17), nv_size = nv_length1, _tmp18);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp19;
								_tmp19 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp19 = vala_gidl_parser_eval (self, nv[1]));
								_tmp19 = (g_free (_tmp19), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp20;
									gboolean _tmp21;
									_tmp20 = NULL;
									if ((_tmp21 = _vala_strcmp0 (_tmp20 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp20 = (g_free (_tmp20), NULL), _tmp21)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
										(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp24 = NULL;
		_tmp23 = NULL;
		self->priv->current_data_type = (_tmp24 = (_tmp23 = (ValaTypeSymbol*) cl, (_tmp23 == NULL) ? NULL : vala_code_node_ref (_tmp23)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp24);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp26;
							const char* _tmp25;
							_tmp26 = NULL;
							_tmp25 = NULL;
							ref_function = (_tmp26 = (_tmp25 = ((GIdlNodeFunction*) member)->symbol, (_tmp25 == NULL) ? NULL : g_strdup (_tmp25)), ref_function = (g_free (ref_function), NULL), _tmp26);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp28;
								const char* _tmp27;
								_tmp28 = NULL;
								_tmp27 = NULL;
								unref_function = (_tmp28 = (_tmp27 = ((GIdlNodeFunction*) member)->symbol, (_tmp27 == NULL) ? NULL : g_strdup (_tmp27)), unref_function = (g_free (unref_function), NULL), _tmp28);
							} else {
								gboolean _tmp29;
								_tmp29 = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp29 = TRUE;
								} else {
									_tmp29 = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp29) {
									char* _tmp31;
									const char* _tmp30;
									_tmp31 = NULL;
									_tmp30 = NULL;
									free_function = (_tmp31 = (_tmp30 = ((GIdlNodeFunction*) member)->symbol, (_tmp30 == NULL) ? NULL : g_strdup (_tmp30)), free_function = (g_free (free_function), NULL), _tmp31);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp33;
										const char* _tmp32;
										_tmp33 = NULL;
										_tmp32 = NULL;
										copy_function = (_tmp33 = (_tmp32 = ((GIdlNodeFunction*) member)->symbol, (_tmp32 == NULL) ? NULL : g_strdup (_tmp32)), copy_function = (g_free (copy_function), NULL), _tmp33);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp34 = NULL;
		self->priv->current_data_type = (_tmp34 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp34);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	char** _tmp1;
	gint node_attributes_size;
	gint node_attributes_length1;
	gint _tmp0;
	char** node_attributes;
	g_return_if_fail (self != NULL);
	g_return_if_fail (boxed_node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	node = (GIdlNode*) boxed_node;
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	_tmp1 = NULL;
	node_attributes = (_tmp1 = vala_gidl_parser_get_attributes (self, node->name, &_tmp0), node_attributes_length1 = _tmp0, node_attributes_size = node_attributes_length1, _tmp1);
	if (node_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = node_attributes;
			attr_collection_length1 = node_attributes_length1;
			for (attr_it = 0; attr_it < node_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp4;
				char* attr;
				_tmp4 = NULL;
				attr = (_tmp4 = attr_collection[attr_it], (_tmp4 == NULL) ? NULL : g_strdup (_tmp4));
				{
					char** _tmp3;
					gint nv_size;
					gint nv_length1;
					char** _tmp2;
					char** nv;
					_tmp3 = NULL;
					_tmp2 = NULL;
					nv = (_tmp3 = _tmp2 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp2), nv_size = nv_length1, _tmp3);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						name = (g_free (name), NULL);
						node_attributes = (_vala_array_free (node_attributes, node_attributes_length1, (GDestroyNotify) g_free), NULL);
						return;
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp5;
		ValaStruct* st;
		ValaTypeSymbol* _tmp15;
		ValaTypeSymbol* _tmp14;
		ValaTypeSymbol* _tmp16;
		_tmp5 = NULL;
		st = (_tmp5 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_STRUCT (_tmp5) ? ((ValaStruct*) _tmp5) : NULL);
		if (st == NULL) {
			ValaStruct* _tmp6;
			char** _tmp8;
			gint st_attributes_size;
			gint st_attributes_length1;
			gint _tmp7;
			char** st_attributes;
			char* _tmp13;
			_tmp6 = NULL;
			st = (_tmp6 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp6);
			vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp8 = NULL;
			st_attributes = (_tmp8 = vala_gidl_parser_get_attributes (self, node->name, &_tmp7), st_attributes_length1 = _tmp7, st_attributes_size = st_attributes_length1, _tmp8);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = st_attributes;
					attr_collection_length1 = st_attributes_length1;
					for (attr_it = 0; attr_it < st_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp12;
						char* attr;
						_tmp12 = NULL;
						attr = (_tmp12 = attr_collection[attr_it], (_tmp12 == NULL) ? NULL : g_strdup (_tmp12));
						{
							char** _tmp10;
							gint nv_size;
							gint nv_length1;
							char** _tmp9;
							char** nv;
							_tmp10 = NULL;
							_tmp9 = NULL;
							nv = (_tmp10 = _tmp9 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp9), nv_size = nv_length1, _tmp10);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp11;
								_tmp11 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) st, _tmp11 = vala_gidl_parser_eval (self, nv[1]));
								_tmp11 = (g_free (_tmp11), NULL);
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			_tmp13 = NULL;
			vala_struct_set_type_id (st, _tmp13 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) st, "TYPE_"));
			_tmp13 = (g_free (_tmp13), NULL);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) st);
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp15 = NULL;
		_tmp14 = NULL;
		self->priv->current_data_type = (_tmp15 = (_tmp14 = (ValaTypeSymbol*) st, (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp15);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		_tmp16 = NULL;
		self->priv->current_data_type = (_tmp16 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp16);
		(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
	} else {
		ValaSymbol* _tmp17;
		ValaClass* cl;
		ValaTypeSymbol* _tmp29;
		ValaTypeSymbol* _tmp28;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypeSymbol* _tmp39;
		_tmp17 = NULL;
		cl = (_tmp17 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp17) ? ((ValaClass*) _tmp17) : NULL);
		if (cl == NULL) {
			ValaClass* _tmp18;
			char** _tmp20;
			gint cl_attributes_size;
			gint cl_attributes_length1;
			gint _tmp19;
			char** cl_attributes;
			char* _tmp27;
			_tmp18 = NULL;
			cl = (_tmp18 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp18);
			vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			vala_class_set_is_compact (cl, TRUE);
			_tmp20 = NULL;
			cl_attributes = (_tmp20 = vala_gidl_parser_get_attributes (self, node->name, &_tmp19), cl_attributes_length1 = _tmp19, cl_attributes_size = cl_attributes_length1, _tmp20);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_collection_length1;
					int attr_it;
					attr_collection = cl_attributes;
					attr_collection_length1 = cl_attributes_length1;
					for (attr_it = 0; attr_it < cl_attributes_length1; attr_it = attr_it + 1) {
						const char* _tmp26;
						char* attr;
						_tmp26 = NULL;
						attr = (_tmp26 = attr_collection[attr_it], (_tmp26 == NULL) ? NULL : g_strdup (_tmp26));
						{
							char** _tmp22;
							gint nv_size;
							gint nv_length1;
							char** _tmp21;
							char** nv;
							_tmp22 = NULL;
							_tmp21 = NULL;
							nv = (_tmp22 = _tmp21 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp21), nv_size = nv_length1, _tmp22);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp23;
								_tmp23 = NULL;
								vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp23 = vala_gidl_parser_eval (self, nv[1]));
								_tmp23 = (g_free (_tmp23), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "is_immutable") == 0) {
									char* _tmp24;
									gboolean _tmp25;
									_tmp24 = NULL;
									if ((_tmp25 = _vala_strcmp0 (_tmp24 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp24 = (g_free (_tmp24), NULL), _tmp25)) {
										vala_class_set_is_immutable (cl, TRUE);
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			_tmp27 = NULL;
			vala_class_set_type_id (cl, _tmp27 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) cl, "TYPE_"));
			_tmp27 = (g_free (_tmp27), NULL);
			vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, (GDestroyNotify) g_free), NULL);
		}
		_tmp29 = NULL;
		_tmp28 = NULL;
		self->priv->current_data_type = (_tmp29 = (_tmp28 = (ValaTypeSymbol*) cl, (_tmp28 == NULL) ? NULL : vala_code_node_ref (_tmp28)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp29);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = (GIdlNode*) member_it->data;
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp31;
							const char* _tmp30;
							_tmp31 = NULL;
							_tmp30 = NULL;
							ref_function = (_tmp31 = (_tmp30 = ((GIdlNodeFunction*) member)->symbol, (_tmp30 == NULL) ? NULL : g_strdup (_tmp30)), ref_function = (g_free (ref_function), NULL), _tmp31);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp33;
								const char* _tmp32;
								_tmp33 = NULL;
								_tmp32 = NULL;
								unref_function = (_tmp33 = (_tmp32 = ((GIdlNodeFunction*) member)->symbol, (_tmp32 == NULL) ? NULL : g_strdup (_tmp32)), unref_function = (g_free (unref_function), NULL), _tmp33);
							} else {
								gboolean _tmp34;
								_tmp34 = FALSE;
								if (_vala_strcmp0 (member->name, "free") == 0) {
									_tmp34 = TRUE;
								} else {
									_tmp34 = _vala_strcmp0 (member->name, "destroy") == 0;
								}
								if (_tmp34) {
									char* _tmp36;
									const char* _tmp35;
									_tmp36 = NULL;
									_tmp35 = NULL;
									free_function = (_tmp36 = (_tmp35 = ((GIdlNodeFunction*) member)->symbol, (_tmp35 == NULL) ? NULL : g_strdup (_tmp35)), free_function = (g_free (free_function), NULL), _tmp36);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp38;
										const char* _tmp37;
										_tmp38 = NULL;
										_tmp37 = NULL;
										copy_function = (_tmp38 = (_tmp37 = ((GIdlNodeFunction*) member)->symbol, (_tmp37 == NULL) ? NULL : g_strdup (_tmp37)), copy_function = (g_free (copy_function), NULL), _tmp38);
									}
									m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp39 = NULL;
		self->priv->current_data_type = (_tmp39 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp39);
		(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
	node_attributes = (_vala_array_free (node_attributes, node_attributes_length1, (GDestroyNotify) g_free), NULL);
}


static ValaTypeSymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node) {
	GIdlNode* node;
	ValaEnum* en;
	gboolean _tmp0;
	char* common_prefix;
	gboolean is_errordomain;
	GeeArrayList* cheader_filenames;
	char** _tmp11;
	gint en_attributes_size;
	gint en_attributes_length1;
	gint _tmp10;
	char** en_attributes;
	ValaTypeSymbol* _tmp28;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = (GIdlNode*) en_node;
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp0 = FALSE;
	if (en_node->gtype_name != NULL) {
		_tmp0 = _vala_strcmp0 (en_node->gtype_name, "") != 0;
	} else {
		_tmp0 = FALSE;
	}
	vala_enum_set_has_type_id (en, (_tmp0));
	common_prefix = NULL;
	{
		GList* value_collection;
		GList* value_it;
		value_collection = en_node->values;
		for (value_it = value_collection; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = (GIdlNode*) value_it->data;
			{
				if (common_prefix == NULL) {
					char* _tmp2;
					const char* _tmp1;
					_tmp2 = NULL;
					_tmp1 = NULL;
					common_prefix = (_tmp2 = (_tmp1 = value->name, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), common_prefix = (g_free (common_prefix), NULL), _tmp2);
					while (TRUE) {
						gboolean _tmp3;
						char* _tmp4;
						_tmp3 = FALSE;
						if (g_utf8_strlen (common_prefix, -1) > 0) {
							_tmp3 = !g_str_has_suffix (common_prefix, "_");
						} else {
							_tmp3 = FALSE;
						}
						if (!_tmp3) {
							break;
						}
						/* FIXME: could easily be made faster*/
						_tmp4 = NULL;
						common_prefix = (_tmp4 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp4);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* _tmp5;
						_tmp5 = NULL;
						common_prefix = (_tmp5 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp5);
					}
				}
				while (TRUE) {
					gboolean _tmp6;
					char* _tmp9;
					_tmp6 = FALSE;
					if (g_utf8_strlen (common_prefix, -1) > 0) {
						gboolean _tmp7;
						_tmp7 = FALSE;
						if (!g_str_has_suffix (common_prefix, "_")) {
							_tmp7 = TRUE;
						} else {
							gboolean _tmp8;
							_tmp8 = FALSE;
							if ((g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (value->name, strlen (common_prefix)))))) {
								_tmp8 = (g_utf8_strlen (value->name, -1) - g_utf8_strlen (common_prefix, -1)) <= 1;
							} else {
								_tmp8 = FALSE;
							}
							_tmp7 = _tmp8;
						}
						_tmp6 = (_tmp7);
					} else {
						_tmp6 = FALSE;
					}
					if (!_tmp6) {
						break;
					}
					/* enum values may not consist solely of digits*/
					_tmp9 = NULL;
					common_prefix = (_tmp9 = g_strndup (common_prefix, (gsize) (strlen (common_prefix) - 1)), common_prefix = (g_free (common_prefix), NULL), _tmp9);
				}
			}
		}
	}
	is_errordomain = FALSE;
	cheader_filenames = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	_tmp11 = NULL;
	en_attributes = (_tmp11 = vala_gidl_parser_get_attributes (self, node->name, &_tmp10), en_attributes_length1 = _tmp10, en_attributes_size = en_attributes_length1, _tmp11);
	if (en_attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = en_attributes;
			attr_collection_length1 = en_attributes_length1;
			for (attr_it = 0; attr_it < en_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp23;
				char* attr;
				_tmp23 = NULL;
				attr = (_tmp23 = attr_collection[attr_it], (_tmp23 == NULL) ? NULL : g_strdup (_tmp23));
				{
					char** _tmp13;
					gint nv_size;
					gint nv_length1;
					char** _tmp12;
					char** nv;
					_tmp13 = NULL;
					_tmp12 = NULL;
					nv = (_tmp13 = _tmp12 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp12), nv_size = nv_length1, _tmp13);
					if (_vala_strcmp0 (nv[0], "common_prefix") == 0) {
						char* _tmp14;
						_tmp14 = NULL;
						common_prefix = (_tmp14 = vala_gidl_parser_eval (self, nv[1]), common_prefix = (g_free (common_prefix), NULL), _tmp14);
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp15;
							char* _tmp16;
							_tmp15 = NULL;
							gee_collection_add ((GeeCollection*) cheader_filenames, _tmp15 = vala_gidl_parser_eval (self, nv[1]));
							_tmp15 = (g_free (_tmp15), NULL);
							_tmp16 = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) en, _tmp16 = vala_gidl_parser_eval (self, nv[1]));
							_tmp16 = (g_free (_tmp16), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp17;
								gboolean _tmp18;
								_tmp17 = NULL;
								if ((_tmp18 = _vala_strcmp0 (_tmp17 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp17 = (g_free (_tmp17), NULL), _tmp18)) {
									ValaTypeSymbol* _tmp19;
									_tmp19 = NULL;
									return (_tmp19 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp19);
								}
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									char* _tmp20;
									_tmp20 = NULL;
									vala_symbol_set_name ((ValaSymbol*) en, _tmp20 = vala_gidl_parser_eval (self, nv[1]));
									_tmp20 = (g_free (_tmp20), NULL);
								} else {
									if (_vala_strcmp0 (nv[0], "errordomain") == 0) {
										char* _tmp21;
										gboolean _tmp22;
										_tmp21 = NULL;
										if ((_tmp22 = _vala_strcmp0 (_tmp21 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp21 = (g_free (_tmp21), NULL), _tmp22)) {
											is_errordomain = TRUE;
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* value2_collection;
		GList* value2_it;
		value2_collection = en_node->values;
		for (value2_it = value2_collection; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = (GIdlNode*) value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)), NULL);
				vala_enum_add_value (en, ev);
				(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			}
		}
	}
	if (is_errordomain) {
		ValaErrorDomain* ed;
		ValaTypeSymbol* _tmp27;
		ed = vala_error_domain_new (vala_symbol_get_name ((ValaSymbol*) en), self->priv->current_source_reference);
		vala_symbol_set_access ((ValaSymbol*) ed, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		vala_error_domain_set_cprefix (ed, common_prefix);
		{
			GeeIterator* _filename_it;
			_filename_it = gee_iterable_iterator ((GeeIterable*) cheader_filenames);
			while (gee_iterator_next (_filename_it)) {
				char* filename;
				filename = (char*) gee_iterator_get (_filename_it);
				vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) ed, filename);
				filename = (g_free (filename), NULL);
			}
			(_filename_it == NULL) ? NULL : (_filename_it = (gee_collection_object_unref (_filename_it), NULL));
		}
		{
			GeeList* _tmp24;
			GeeIterator* _tmp25;
			GeeIterator* _ev_it;
			_tmp24 = NULL;
			_tmp25 = NULL;
			_ev_it = (_tmp25 = gee_iterable_iterator ((GeeIterable*) (_tmp24 = vala_enum_get_values (en))), (_tmp24 == NULL) ? NULL : (_tmp24 = (gee_collection_object_unref (_tmp24), NULL)), _tmp25);
			while (gee_iterator_next (_ev_it)) {
				ValaEnumValue* ev;
				ValaErrorCode* _tmp26;
				ev = (ValaEnumValue*) gee_iterator_get (_ev_it);
				_tmp26 = NULL;
				vala_error_domain_add_code (ed, _tmp26 = vala_error_code_new (vala_symbol_get_name ((ValaSymbol*) ev), NULL));
				(_tmp26 == NULL) ? NULL : (_tmp26 = (vala_code_node_unref (_tmp26), NULL));
				(ev == NULL) ? NULL : (ev = (vala_code_node_unref (ev), NULL));
			}
			(_ev_it == NULL) ? NULL : (_ev_it = (gee_collection_object_unref (_ev_it), NULL));
		}
		_tmp27 = NULL;
		return (_tmp27 = (ValaTypeSymbol*) ed, (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27);
	}
	_tmp28 = NULL;
	return (_tmp28 = (ValaTypeSymbol*) en, common_prefix = (g_free (common_prefix), NULL), (cheader_filenames == NULL) ? NULL : (cheader_filenames = (gee_collection_object_unref (cheader_filenames), NULL)), en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, (GDestroyNotify) g_free), NULL), _tmp28);
}


static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	char* base_class;
	ValaSymbol* _tmp0;
	ValaClass* cl;
	ValaTypeSymbol* _tmp19;
	ValaTypeSymbol* _tmp18;
	GeeSet* _tmp20;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypeSymbol* _tmp25;
	GeeSet* _tmp26;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, ((GIdlNode*) node)->name, ns);
	base_class = NULL;
	_tmp0 = NULL;
	cl = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_CLASS (_tmp0) ? ((ValaClass*) _tmp0) : NULL);
	if (cl == NULL) {
		ValaClass* _tmp1;
		char** _tmp3;
		gint attributes_size;
		gint attributes_length1;
		gint _tmp2;
		char** attributes;
		_tmp1 = NULL;
		cl = (_tmp1 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), _tmp1);
		vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp3 = NULL;
		attributes = (_tmp3 = vala_gidl_parser_get_attributes (self, node->gtype_name, &_tmp2), attributes_length1 = _tmp2, attributes_size = attributes_length1, _tmp3);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = attributes;
				attr_collection_length1 = attributes_length1;
				for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp13;
					char* attr;
					_tmp13 = NULL;
					attr = (_tmp13 = attr_collection[attr_it], (_tmp13 == NULL) ? NULL : g_strdup (_tmp13));
					{
						char** _tmp5;
						gint nv_size;
						gint nv_length1;
						char** _tmp4;
						char** nv;
						_tmp5 = NULL;
						_tmp4 = NULL;
						nv = (_tmp5 = _tmp4 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp4), nv_size = nv_length1, _tmp5);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp6;
							_tmp6 = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) cl, _tmp6 = vala_gidl_parser_eval (self, nv[1]));
							_tmp6 = (g_free (_tmp6), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "base_class") == 0) {
								char* _tmp7;
								_tmp7 = NULL;
								base_class = (_tmp7 = vala_gidl_parser_eval (self, nv[1]), base_class = (g_free (base_class), NULL), _tmp7);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp8;
									gboolean _tmp9;
									_tmp8 = NULL;
									if ((_tmp9 = _vala_strcmp0 (_tmp8 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp8 = (g_free (_tmp8), NULL), _tmp9)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
										attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
										name = (g_free (name), NULL);
										base_class = (g_free (base_class), NULL);
										(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
										return;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "type_check_function") == 0) {
										char* _tmp10;
										_tmp10 = NULL;
										vala_class_set_type_check_function (cl, _tmp10 = vala_gidl_parser_eval (self, nv[1]));
										_tmp10 = (g_free (_tmp10), NULL);
									} else {
										if (_vala_strcmp0 (nv[0], "abstract") == 0) {
											char* _tmp11;
											gboolean _tmp12;
											_tmp11 = NULL;
											if ((_tmp12 = _vala_strcmp0 (_tmp11 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp11 = (g_free (_tmp11), NULL), _tmp12)) {
												vala_class_set_is_abstract (cl, TRUE);
											}
										}
									}
								}
							}
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		vala_namespace_add_class (ns, cl);
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) cl);
		attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	if (base_class != NULL) {
		ValaDataType* parent;
		parent = vala_gidl_parser_parse_type_string (self, base_class);
		vala_class_add_base_type (cl, parent);
		(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
	} else {
		if (node->parent != NULL) {
			ValaDataType* parent;
			parent = vala_gidl_parser_parse_type_string (self, node->parent);
			vala_class_add_base_type (cl, parent);
			(parent == NULL) ? NULL : (parent = (vala_code_node_unref (parent), NULL));
		} else {
			ValaUnresolvedSymbol* _tmp14;
			ValaUnresolvedSymbol* _tmp15;
			ValaUnresolvedSymbol* gobject_symbol;
			ValaUnresolvedType* _tmp16;
			_tmp14 = NULL;
			_tmp15 = NULL;
			gobject_symbol = (_tmp15 = vala_unresolved_symbol_new (_tmp14 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Object", NULL), (_tmp14 == NULL) ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL)), _tmp15);
			_tmp16 = NULL;
			vala_class_add_base_type (cl, (ValaDataType*) (_tmp16 = vala_unresolved_type_new_from_symbol (gobject_symbol, NULL)));
			(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_code_node_unref (_tmp16), NULL));
			(gobject_symbol == NULL) ? NULL : (gobject_symbol = (vala_code_node_unref (gobject_symbol), NULL));
		}
	}
	{
		GList* iface_name_collection;
		GList* iface_name_it;
		iface_name_collection = node->interfaces;
		for (iface_name_it = iface_name_collection; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			const char* _tmp17;
			char* iface_name;
			_tmp17 = NULL;
			iface_name = (_tmp17 = (const char*) iface_name_it->data, (_tmp17 == NULL) ? NULL : g_strdup (_tmp17));
			{
				ValaDataType* iface;
				iface = vala_gidl_parser_parse_type_string (self, iface_name);
				vala_class_add_base_type (cl, iface);
				iface_name = (g_free (iface_name), NULL);
				(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			}
		}
	}
	_tmp19 = NULL;
	_tmp18 = NULL;
	self->priv->current_data_type = (_tmp19 = (_tmp18 = (ValaTypeSymbol*) cl, (_tmp18 == NULL) ? NULL : vala_code_node_ref (_tmp18)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp19);
	_tmp20 = NULL;
	self->priv->current_type_symbol_set = (_tmp20 = (GeeSet*) gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal), (self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL)), _tmp20);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set ((GeeMap*) current_type_func_map, member->name, (GIdlNodeFunction*) member);
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set ((GeeMap*) current_type_vfunc_map, member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains ((GeeMap*) current_type_vfunc_map, member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, (GIdlNodeVFunc*) member, (GIdlNodeFunction*) gee_map_get ((GeeMap*) current_type_func_map, member->name), FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, (GIdlNodeProperty*) member);
							if (prop != NULL) {
								vala_class_add_property (cl, prop);
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, (GIdlNodeSignal*) member);
								if (sig != NULL) {
									vala_class_add_signal (cl, sig);
								}
								(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
							}
						}
					}
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (!gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, member->name)) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, (GIdlNodeField*) member);
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
					}
				}
			}
		}
	}
	{
		GeeList* _tmp21;
		GeeIterator* _tmp22;
		GeeIterator* _prop_it;
		_tmp21 = NULL;
		_tmp22 = NULL;
		_prop_it = (_tmp22 = gee_iterable_iterator ((GeeIterable*) (_tmp21 = vala_object_type_symbol_get_properties ((ValaObjectTypeSymbol*) cl))), (_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL)), _tmp22);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			char* getter;
			gboolean _tmp23;
			char* setter;
			gboolean _tmp24;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			getter = g_strdup_printf ("get_%s", vala_symbol_get_name ((ValaSymbol*) prop));
			_tmp23 = FALSE;
			if (vala_property_get_get_accessor (prop) != NULL) {
				_tmp23 = !gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, getter);
			} else {
				_tmp23 = FALSE;
			}
			if (_tmp23) {
				vala_property_set_no_accessor_method (prop, TRUE);
			}
			setter = g_strdup_printf ("set_%s", vala_symbol_get_name ((ValaSymbol*) prop));
			_tmp24 = FALSE;
			if (vala_property_get_set_accessor (prop) != NULL) {
				_tmp24 = !gee_collection_contains ((GeeCollection*) self->priv->current_type_symbol_set, setter);
			} else {
				_tmp24 = FALSE;
			}
			if (_tmp24) {
				vala_property_set_no_accessor_method (prop, TRUE);
			}
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
			getter = (g_free (getter), NULL);
			setter = (g_free (setter), NULL);
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	_tmp25 = NULL;
	self->priv->current_data_type = (_tmp25 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp25);
	_tmp26 = NULL;
	self->priv->current_type_symbol_set = (_tmp26 = NULL, (self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL)), _tmp26);
	name = (g_free (name), NULL);
	base_class = (g_free (base_class), NULL);
	(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
	(current_type_func_map == NULL) ? NULL : (current_type_func_map = (gee_collection_object_unref (current_type_func_map), NULL));
	(current_type_vfunc_map == NULL) ? NULL : (current_type_vfunc_map = (gee_collection_object_unref (current_type_vfunc_map), NULL));
}


static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	ValaSymbol* _tmp0;
	ValaInterface* iface;
	ValaTypeSymbol* _tmp11;
	ValaTypeSymbol* _tmp10;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypeSymbol* _tmp12;
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	g_return_if_fail (ns != NULL);
	g_return_if_fail (module != NULL);
	name = vala_gidl_parser_fix_type_name (self, node->gtype_name, ns);
	_tmp0 = NULL;
	iface = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name), VALA_IS_INTERFACE (_tmp0) ? ((ValaInterface*) _tmp0) : NULL);
	if (iface == NULL) {
		ValaInterface* _tmp1;
		char** _tmp3;
		gint attributes_size;
		gint attributes_length1;
		gint _tmp2;
		char** attributes;
		_tmp1 = NULL;
		iface = (_tmp1 = vala_interface_new (name, self->priv->current_source_reference), (iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL)), _tmp1);
		vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp3 = NULL;
		attributes = (_tmp3 = vala_gidl_parser_get_attributes (self, node->gtype_name, &_tmp2), attributes_length1 = _tmp2, attributes_size = attributes_length1, _tmp3);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = attributes;
				attr_collection_length1 = attributes_length1;
				for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp8;
					char* attr;
					_tmp8 = NULL;
					attr = (_tmp8 = attr_collection[attr_it], (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
					{
						char** _tmp5;
						gint nv_size;
						gint nv_length1;
						char** _tmp4;
						char** nv;
						_tmp5 = NULL;
						_tmp4 = NULL;
						nv = (_tmp5 = _tmp4 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp4), nv_size = nv_length1, _tmp5);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp6;
							_tmp6 = NULL;
							vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) iface, _tmp6 = vala_gidl_parser_eval (self, nv[1]));
							_tmp6 = (g_free (_tmp6), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "lower_case_csuffix") == 0) {
								char* _tmp7;
								_tmp7 = NULL;
								vala_interface_set_lower_case_csuffix (iface, _tmp7 = vala_gidl_parser_eval (self, nv[1]));
								_tmp7 = (g_free (_tmp7), NULL);
							}
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		{
			GList* prereq_name_collection;
			GList* prereq_name_it;
			prereq_name_collection = node->prerequisites;
			for (prereq_name_it = prereq_name_collection; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
				const char* _tmp9;
				char* prereq_name;
				_tmp9 = NULL;
				prereq_name = (_tmp9 = (const char*) prereq_name_it->data, (_tmp9 == NULL) ? NULL : g_strdup (_tmp9));
				{
					ValaDataType* prereq;
					prereq = vala_gidl_parser_parse_type_string (self, prereq_name);
					vala_interface_add_prerequisite (iface, prereq);
					prereq_name = (g_free (prereq_name), NULL);
					(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				}
			}
		}
		vala_namespace_add_interface (ns, iface);
		vala_source_file_add_node (self->priv->current_source_file, (ValaCodeNode*) iface);
		attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	_tmp11 = NULL;
	_tmp10 = NULL;
	self->priv->current_data_type = (_tmp11 = (_tmp10 = (ValaTypeSymbol*) iface, (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp11);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set ((GeeMap*) current_type_func_map, member->name, (GIdlNodeFunction*) member);
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set ((GeeMap*) current_type_vfunc_map, member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = (GIdlNode*) member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains ((GeeMap*) current_type_vfunc_map, member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, (GIdlNodeFunction*) member, TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, (GIdlNodeVFunc*) member, (GIdlNodeFunction*) gee_map_get ((GeeMap*) current_type_func_map, member->name), TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, (GIdlNodeProperty*) member);
							if (prop != NULL) {
								vala_interface_add_property (iface, prop);
							}
							(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, (GIdlNodeSignal*) member);
								if (sig != NULL) {
									vala_interface_add_signal (iface, sig);
								}
								(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
							}
						}
					}
				}
			}
		}
	}
	_tmp12 = NULL;
	self->priv->current_data_type = (_tmp12 = NULL, (self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL)), _tmp12);
	name = (g_free (name), NULL);
	(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
	(current_type_func_map == NULL) ? NULL : (current_type_func_map = (gee_collection_object_unref (current_type_func_map), NULL));
	(current_type_vfunc_map == NULL) ? NULL : (current_type_vfunc_map = (gee_collection_object_unref (current_type_vfunc_map), NULL));
}


static ValaDataType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node, ValaParameterDirection* direction) {
	ValaParameterDirection dir;
	ValaUnresolvedType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	dir = VALA_PARAMETER_DIRECTION_IN;
	type = vala_unresolved_type_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			ValaVoidType* _tmp0;
			ValaDataType* _tmp1;
			ValaDataType* _tmp2;
			_tmp0 = NULL;
			_tmp1 = NULL;
			_tmp2 = NULL;
			return (_tmp2 = (_tmp1 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp0 = vala_void_type_new (NULL)), NULL), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp2);
		} else {
			ValaDataType* _tmp3;
			_tmp3 = NULL;
			return (_tmp3 = (ValaDataType*) vala_void_type_new (NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp3);
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			ValaUnresolvedSymbol* _tmp4;
			_tmp4 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, _tmp4 = vala_unresolved_symbol_new (NULL, "bool", NULL));
			(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				ValaUnresolvedSymbol* _tmp5;
				_tmp5 = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp5 = vala_unresolved_symbol_new (NULL, "char", NULL));
				(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					ValaUnresolvedSymbol* _tmp6;
					_tmp6 = NULL;
					vala_unresolved_type_set_unresolved_symbol (type, _tmp6 = vala_unresolved_symbol_new (NULL, "uchar", NULL));
					(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						ValaUnresolvedSymbol* _tmp7;
						_tmp7 = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, _tmp7 = vala_unresolved_symbol_new (NULL, "int16", NULL));
						(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							ValaUnresolvedSymbol* _tmp8;
							_tmp8 = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, _tmp8 = vala_unresolved_symbol_new (NULL, "uint16", NULL));
							(_tmp8 == NULL) ? NULL : (_tmp8 = (vala_code_node_unref (_tmp8), NULL));
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								ValaUnresolvedSymbol* _tmp9;
								_tmp9 = NULL;
								vala_unresolved_type_set_unresolved_symbol (type, _tmp9 = vala_unresolved_symbol_new (NULL, "int32", NULL));
								(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									ValaUnresolvedSymbol* _tmp10;
									_tmp10 = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, _tmp10 = vala_unresolved_symbol_new (NULL, "uint32", NULL));
									(_tmp10 == NULL) ? NULL : (_tmp10 = (vala_code_node_unref (_tmp10), NULL));
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										ValaUnresolvedSymbol* _tmp11;
										_tmp11 = NULL;
										vala_unresolved_type_set_unresolved_symbol (type, _tmp11 = vala_unresolved_symbol_new (NULL, "int64", NULL));
										(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											ValaUnresolvedSymbol* _tmp12;
											_tmp12 = NULL;
											vala_unresolved_type_set_unresolved_symbol (type, _tmp12 = vala_unresolved_symbol_new (NULL, "uint64", NULL));
											(_tmp12 == NULL) ? NULL : (_tmp12 = (vala_code_node_unref (_tmp12), NULL));
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												ValaUnresolvedSymbol* _tmp13;
												_tmp13 = NULL;
												vala_unresolved_type_set_unresolved_symbol (type, _tmp13 = vala_unresolved_symbol_new (NULL, "int", NULL));
												(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													ValaUnresolvedSymbol* _tmp14;
													_tmp14 = NULL;
													vala_unresolved_type_set_unresolved_symbol (type, _tmp14 = vala_unresolved_symbol_new (NULL, "uint", NULL));
													(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_code_node_unref (_tmp14), NULL));
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														ValaUnresolvedSymbol* _tmp15;
														_tmp15 = NULL;
														vala_unresolved_type_set_unresolved_symbol (type, _tmp15 = vala_unresolved_symbol_new (NULL, "long", NULL));
														(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL));
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															ValaUnresolvedSymbol* _tmp16;
															_tmp16 = NULL;
															vala_unresolved_type_set_unresolved_symbol (type, _tmp16 = vala_unresolved_symbol_new (NULL, "ulong", NULL));
															(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_code_node_unref (_tmp16), NULL));
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																ValaUnresolvedSymbol* _tmp17;
																_tmp17 = NULL;
																vala_unresolved_type_set_unresolved_symbol (type, _tmp17 = vala_unresolved_symbol_new (NULL, "ssize_t", NULL));
																(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_code_node_unref (_tmp17), NULL));
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	ValaUnresolvedSymbol* _tmp18;
																	_tmp18 = NULL;
																	vala_unresolved_type_set_unresolved_symbol (type, _tmp18 = vala_unresolved_symbol_new (NULL, "size_t", NULL));
																	(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		ValaUnresolvedSymbol* _tmp19;
																		_tmp19 = NULL;
																		vala_unresolved_type_set_unresolved_symbol (type, _tmp19 = vala_unresolved_symbol_new (NULL, "float", NULL));
																		(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_code_node_unref (_tmp19), NULL));
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			ValaUnresolvedSymbol* _tmp20;
																			_tmp20 = NULL;
																			vala_unresolved_type_set_unresolved_symbol (type, _tmp20 = vala_unresolved_symbol_new (NULL, "double", NULL));
																			(_tmp20 == NULL) ? NULL : (_tmp20 = (vala_code_node_unref (_tmp20), NULL));
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				ValaUnresolvedSymbol* _tmp21;
																				_tmp21 = NULL;
																				vala_unresolved_type_set_unresolved_symbol (type, _tmp21 = vala_unresolved_symbol_new (NULL, "string", NULL));
																				(_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL));
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					ValaUnresolvedSymbol* _tmp22;
																					_tmp22 = NULL;
																					vala_unresolved_type_set_unresolved_symbol (type, _tmp22 = vala_unresolved_symbol_new (NULL, "string", NULL));
																					(_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL));
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaDataType* element_type;
																						ValaUnresolvedType* _tmp25;
																						ValaUnresolvedType* _tmp24;
																						ValaDataType* _tmp23;
																						ValaDataType* _tmp27;
																						element_type = vala_gidl_parser_parse_type (self, type_node->parameter_type1, NULL);
																						_tmp25 = NULL;
																						_tmp24 = NULL;
																						_tmp23 = NULL;
																						type = (_tmp25 = (_tmp24 = (_tmp23 = element_type, VALA_IS_UNRESOLVED_TYPE (_tmp23) ? ((ValaUnresolvedType*) _tmp23) : NULL), (_tmp24 == NULL) ? NULL : vala_code_node_ref (_tmp24)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp25);
																						if (type == NULL) {
																							ValaDataType* _tmp26;
																							_tmp26 = NULL;
																							return (_tmp26 = element_type, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp26);
																						}
																						_tmp27 = NULL;
																						return (_tmp27 = (ValaDataType*) vala_array_type_new (element_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) element_type)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp27);
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							ValaUnresolvedSymbol* _tmp29;
																							ValaUnresolvedSymbol* _tmp28;
																							_tmp29 = NULL;
																							_tmp28 = NULL;
																							vala_unresolved_type_set_unresolved_symbol (type, _tmp29 = vala_unresolved_symbol_new (_tmp28 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "List", NULL));
																							(_tmp29 == NULL) ? NULL : (_tmp29 = (vala_code_node_unref (_tmp29), NULL));
																							(_tmp28 == NULL) ? NULL : (_tmp28 = (vala_code_node_unref (_tmp28), NULL));
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								ValaUnresolvedSymbol* _tmp31;
																								ValaUnresolvedSymbol* _tmp30;
																								_tmp31 = NULL;
																								_tmp30 = NULL;
																								vala_unresolved_type_set_unresolved_symbol (type, _tmp31 = vala_unresolved_symbol_new (_tmp30 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "SList", NULL));
																								(_tmp31 == NULL) ? NULL : (_tmp31 = (vala_code_node_unref (_tmp31), NULL));
																								(_tmp30 == NULL) ? NULL : (_tmp30 = (vala_code_node_unref (_tmp30), NULL));
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									ValaUnresolvedSymbol* _tmp33;
																									ValaUnresolvedSymbol* _tmp32;
																									_tmp33 = NULL;
																									_tmp32 = NULL;
																									vala_unresolved_type_set_unresolved_symbol (type, _tmp33 = vala_unresolved_symbol_new (_tmp32 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "HashTable", NULL));
																									(_tmp33 == NULL) ? NULL : (_tmp33 = (vala_code_node_unref (_tmp33), NULL));
																									(_tmp32 == NULL) ? NULL : (_tmp32 = (vala_code_node_unref (_tmp32), NULL));
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										ValaUnresolvedSymbol* _tmp35;
																										ValaUnresolvedSymbol* _tmp34;
																										_tmp35 = NULL;
																										_tmp34 = NULL;
																										vala_unresolved_type_set_unresolved_symbol (type, _tmp35 = vala_unresolved_symbol_new (_tmp34 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Error", NULL));
																										(_tmp35 == NULL) ? NULL : (_tmp35 = (vala_code_node_unref (_tmp35), NULL));
																										(_tmp34 == NULL) ? NULL : (_tmp34 = (vala_code_node_unref (_tmp34), NULL));
																									} else {
																										if (type_node->is_interface) {
																											const char* _tmp36;
																											char* n;
																											gboolean _tmp40;
																											_tmp36 = NULL;
																											n = (_tmp36 = type_node->interface, (_tmp36 == NULL) ? NULL : g_strdup (_tmp36));
																											if (_vala_strcmp0 (n, "") == 0) {
																												ValaDataType* _tmp37;
																												_tmp37 = NULL;
																												return (_tmp37 = NULL, n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp37);
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* _tmp39;
																												const char* _tmp38;
																												_tmp39 = NULL;
																												_tmp38 = NULL;
																												n = (_tmp39 = (_tmp38 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (_tmp38 == NULL) ? NULL : g_strdup (_tmp38)), n = (g_free (n), NULL), _tmp39);
																											}
																											_tmp40 = FALSE;
																											if (type_node->is_pointer) {
																												gboolean _tmp41;
																												_tmp41 = FALSE;
																												if (_vala_strcmp0 (n, "gchar") == 0) {
																													_tmp41 = TRUE;
																												} else {
																													_tmp41 = _vala_strcmp0 (n, "char") == 0;
																												}
																												_tmp40 = (_tmp41);
																											} else {
																												_tmp40 = FALSE;
																											}
																											if (_tmp40) {
																												ValaUnresolvedSymbol* _tmp42;
																												_tmp42 = NULL;
																												vala_unresolved_type_set_unresolved_symbol (type, _tmp42 = vala_unresolved_symbol_new (NULL, "string", NULL));
																												(_tmp42 == NULL) ? NULL : (_tmp42 = (vala_code_node_unref (_tmp42), NULL));
																												if (g_str_has_suffix (type_node->unparsed, "**")) {
																													dir = VALA_PARAMETER_DIRECTION_OUT;
																												}
																											} else {
																												if (_vala_strcmp0 (n, "gunichar") == 0) {
																													ValaUnresolvedSymbol* _tmp43;
																													_tmp43 = NULL;
																													vala_unresolved_type_set_unresolved_symbol (type, _tmp43 = vala_unresolved_symbol_new (NULL, "unichar", NULL));
																													(_tmp43 == NULL) ? NULL : (_tmp43 = (vala_code_node_unref (_tmp43), NULL));
																												} else {
																													if (_vala_strcmp0 (n, "gchar") == 0) {
																														ValaUnresolvedSymbol* _tmp44;
																														_tmp44 = NULL;
																														vala_unresolved_type_set_unresolved_symbol (type, _tmp44 = vala_unresolved_symbol_new (NULL, "char", NULL));
																														(_tmp44 == NULL) ? NULL : (_tmp44 = (vala_code_node_unref (_tmp44), NULL));
																													} else {
																														gboolean _tmp45;
																														_tmp45 = FALSE;
																														if (_vala_strcmp0 (n, "guchar") == 0) {
																															_tmp45 = TRUE;
																														} else {
																															_tmp45 = _vala_strcmp0 (n, "guint8") == 0;
																														}
																														if (_tmp45) {
																															ValaUnresolvedSymbol* _tmp46;
																															_tmp46 = NULL;
																															vala_unresolved_type_set_unresolved_symbol (type, _tmp46 = vala_unresolved_symbol_new (NULL, "uchar", NULL));
																															(_tmp46 == NULL) ? NULL : (_tmp46 = (vala_code_node_unref (_tmp46), NULL));
																															if (type_node->is_pointer) {
																																ValaDataType* _tmp47;
																																_tmp47 = NULL;
																																return (_tmp47 = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp47);
																															}
																														} else {
																															if (_vala_strcmp0 (n, "gushort") == 0) {
																																ValaUnresolvedSymbol* _tmp48;
																																_tmp48 = NULL;
																																vala_unresolved_type_set_unresolved_symbol (type, _tmp48 = vala_unresolved_symbol_new (NULL, "ushort", NULL));
																																(_tmp48 == NULL) ? NULL : (_tmp48 = (vala_code_node_unref (_tmp48), NULL));
																															} else {
																																if (_vala_strcmp0 (n, "gshort") == 0) {
																																	ValaUnresolvedSymbol* _tmp49;
																																	_tmp49 = NULL;
																																	vala_unresolved_type_set_unresolved_symbol (type, _tmp49 = vala_unresolved_symbol_new (NULL, "short", NULL));
																																	(_tmp49 == NULL) ? NULL : (_tmp49 = (vala_code_node_unref (_tmp49), NULL));
																																} else {
																																	gboolean _tmp50;
																																	_tmp50 = FALSE;
																																	if (_vala_strcmp0 (n, "gconstpointer") == 0) {
																																		_tmp50 = TRUE;
																																	} else {
																																		_tmp50 = _vala_strcmp0 (n, "void") == 0;
																																	}
																																	if (_tmp50) {
																																		ValaVoidType* _tmp51;
																																		ValaDataType* _tmp52;
																																		ValaDataType* _tmp53;
																																		_tmp51 = NULL;
																																		_tmp52 = NULL;
																																		_tmp53 = NULL;
																																		return (_tmp53 = (_tmp52 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp51 = vala_void_type_new (NULL)), NULL), (_tmp51 == NULL) ? NULL : (_tmp51 = (vala_code_node_unref (_tmp51), NULL)), _tmp52), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp53);
																																	} else {
																																		gboolean _tmp54;
																																		_tmp54 = FALSE;
																																		if (_vala_strcmp0 (n, "goffset") == 0) {
																																			_tmp54 = TRUE;
																																		} else {
																																			_tmp54 = _vala_strcmp0 (n, "off_t") == 0;
																																		}
																																		if (_tmp54) {
																																			ValaUnresolvedSymbol* _tmp55;
																																			_tmp55 = NULL;
																																			vala_unresolved_type_set_unresolved_symbol (type, _tmp55 = vala_unresolved_symbol_new (NULL, "int64", NULL));
																																			(_tmp55 == NULL) ? NULL : (_tmp55 = (vala_code_node_unref (_tmp55), NULL));
																																		} else {
																																			if (_vala_strcmp0 (n, "value_array") == 0) {
																																				ValaUnresolvedSymbol* _tmp57;
																																				ValaUnresolvedSymbol* _tmp56;
																																				_tmp57 = NULL;
																																				_tmp56 = NULL;
																																				vala_unresolved_type_set_unresolved_symbol (type, _tmp57 = vala_unresolved_symbol_new (_tmp56 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "ValueArray", NULL));
																																				(_tmp57 == NULL) ? NULL : (_tmp57 = (vala_code_node_unref (_tmp57), NULL));
																																				(_tmp56 == NULL) ? NULL : (_tmp56 = (vala_code_node_unref (_tmp56), NULL));
																																			} else {
																																				if (_vala_strcmp0 (n, "time_t") == 0) {
																																					ValaUnresolvedSymbol* _tmp58;
																																					_tmp58 = NULL;
																																					vala_unresolved_type_set_unresolved_symbol (type, _tmp58 = vala_unresolved_symbol_new (NULL, "ulong", NULL));
																																					(_tmp58 == NULL) ? NULL : (_tmp58 = (vala_code_node_unref (_tmp58), NULL));
																																				} else {
																																					if (_vala_strcmp0 (n, "socklen_t") == 0) {
																																						ValaUnresolvedSymbol* _tmp59;
																																						_tmp59 = NULL;
																																						vala_unresolved_type_set_unresolved_symbol (type, _tmp59 = vala_unresolved_symbol_new (NULL, "uint32", NULL));
																																						(_tmp59 == NULL) ? NULL : (_tmp59 = (vala_code_node_unref (_tmp59), NULL));
																																					} else {
																																						if (_vala_strcmp0 (n, "mode_t") == 0) {
																																							ValaUnresolvedSymbol* _tmp60;
																																							_tmp60 = NULL;
																																							vala_unresolved_type_set_unresolved_symbol (type, _tmp60 = vala_unresolved_symbol_new (NULL, "uint", NULL));
																																							(_tmp60 == NULL) ? NULL : (_tmp60 = (vala_code_node_unref (_tmp60), NULL));
																																						} else {
																																							gboolean _tmp61;
																																							_tmp61 = FALSE;
																																							if (_vala_strcmp0 (n, "gint") == 0) {
																																								_tmp61 = TRUE;
																																							} else {
																																								_tmp61 = _vala_strcmp0 (n, "pid_t") == 0;
																																							}
																																							if (_tmp61) {
																																								ValaUnresolvedSymbol* _tmp62;
																																								_tmp62 = NULL;
																																								vala_unresolved_type_set_unresolved_symbol (type, _tmp62 = vala_unresolved_symbol_new (NULL, "int", NULL));
																																								(_tmp62 == NULL) ? NULL : (_tmp62 = (vala_code_node_unref (_tmp62), NULL));
																																							} else {
																																								gboolean _tmp63;
																																								_tmp63 = FALSE;
																																								if (_vala_strcmp0 (n, "unsigned") == 0) {
																																									_tmp63 = TRUE;
																																								} else {
																																									_tmp63 = _vala_strcmp0 (n, "unsigned-int") == 0;
																																								}
																																								if (_tmp63) {
																																									ValaUnresolvedSymbol* _tmp64;
																																									_tmp64 = NULL;
																																									vala_unresolved_type_set_unresolved_symbol (type, _tmp64 = vala_unresolved_symbol_new (NULL, "uint", NULL));
																																									(_tmp64 == NULL) ? NULL : (_tmp64 = (vala_code_node_unref (_tmp64), NULL));
																																								} else {
																																									if (_vala_strcmp0 (n, "FILE") == 0) {
																																										ValaUnresolvedSymbol* _tmp66;
																																										ValaUnresolvedSymbol* _tmp65;
																																										_tmp66 = NULL;
																																										_tmp65 = NULL;
																																										vala_unresolved_type_set_unresolved_symbol (type, _tmp66 = vala_unresolved_symbol_new (_tmp65 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "FileStream", NULL));
																																										(_tmp66 == NULL) ? NULL : (_tmp66 = (vala_code_node_unref (_tmp66), NULL));
																																										(_tmp65 == NULL) ? NULL : (_tmp65 = (vala_code_node_unref (_tmp65), NULL));
																																									} else {
																																										if (_vala_strcmp0 (n, "struct") == 0) {
																																											ValaVoidType* _tmp67;
																																											ValaDataType* _tmp68;
																																											ValaDataType* _tmp69;
																																											_tmp67 = NULL;
																																											_tmp68 = NULL;
																																											_tmp69 = NULL;
																																											return (_tmp69 = (_tmp68 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp67 = vala_void_type_new (NULL)), NULL), (_tmp67 == NULL) ? NULL : (_tmp67 = (vala_code_node_unref (_tmp67), NULL)), _tmp68), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp69);
																																										} else {
																																											if (_vala_strcmp0 (n, "iconv_t") == 0) {
																																												ValaVoidType* _tmp70;
																																												ValaDataType* _tmp71;
																																												ValaDataType* _tmp72;
																																												_tmp70 = NULL;
																																												_tmp71 = NULL;
																																												_tmp72 = NULL;
																																												return (_tmp72 = (_tmp71 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp70 = vala_void_type_new (NULL)), NULL), (_tmp70 == NULL) ? NULL : (_tmp70 = (vala_code_node_unref (_tmp70), NULL)), _tmp71), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp72);
																																											} else {
																																												if (_vala_strcmp0 (n, "GType") == 0) {
																																													ValaUnresolvedSymbol* _tmp74;
																																													ValaUnresolvedSymbol* _tmp73;
																																													_tmp74 = NULL;
																																													_tmp73 = NULL;
																																													vala_unresolved_type_set_unresolved_symbol (type, _tmp74 = vala_unresolved_symbol_new (_tmp73 = vala_unresolved_symbol_new (NULL, "GLib", NULL), "Type", NULL));
																																													(_tmp74 == NULL) ? NULL : (_tmp74 = (vala_code_node_unref (_tmp74), NULL));
																																													(_tmp73 == NULL) ? NULL : (_tmp73 = (vala_code_node_unref (_tmp73), NULL));
																																													if (type_node->is_pointer) {
																																														ValaDataType* _tmp75;
																																														_tmp75 = NULL;
																																														return (_tmp75 = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp75);
																																													}
																																												} else {
																																													if (_vala_strcmp0 (n, "GStrv") == 0) {
																																														ValaUnresolvedSymbol* _tmp76;
																																														ValaDataType* _tmp77;
																																														_tmp76 = NULL;
																																														vala_unresolved_type_set_unresolved_symbol (type, _tmp76 = vala_unresolved_symbol_new (NULL, "string", NULL));
																																														(_tmp76 == NULL) ? NULL : (_tmp76 = (vala_code_node_unref (_tmp76), NULL));
																																														_tmp77 = NULL;
																																														return (_tmp77 = (ValaDataType*) vala_array_type_new ((ValaDataType*) type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp77);
																																													} else {
																																														ValaDataType* named_type;
																																														ValaUnresolvedType* _tmp80;
																																														ValaUnresolvedType* _tmp79;
																																														ValaDataType* _tmp78;
																																														named_type = vala_gidl_parser_parse_type_string (self, n);
																																														_tmp80 = NULL;
																																														_tmp79 = NULL;
																																														_tmp78 = NULL;
																																														type = (_tmp80 = (_tmp79 = (_tmp78 = named_type, VALA_IS_UNRESOLVED_TYPE (_tmp78) ? ((ValaUnresolvedType*) _tmp78) : NULL), (_tmp79 == NULL) ? NULL : vala_code_node_ref (_tmp79)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp80);
																																														if (type == NULL) {
																																															ValaDataType* _tmp81;
																																															_tmp81 = NULL;
																																															return (_tmp81 = named_type, n = (g_free (n), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp81);
																																														}
																																														if (vala_gidl_parser_is_simple_type (self, n)) {
																																															if (type_node->is_pointer) {
																																																dir = VALA_PARAMETER_DIRECTION_OUT;
																																															}
																																														} else {
																																															if (g_str_has_suffix (type_node->unparsed, "**")) {
																																																dir = VALA_PARAMETER_DIRECTION_OUT;
																																															}
																																														}
																																														(named_type == NULL) ? NULL : (named_type = (vala_code_node_unref (named_type), NULL));
																																													}
																																												}
																																											}
																																										}
																																									}
																																								}
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											n = (g_free (n), NULL);
																										} else {
																											fprintf (stdout, "%d\n", (gint) type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	if ((&(*direction)) != NULL) {
		(*direction) = dir;
	}
	return (ValaDataType*) type;
}


static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name) {
	ValaTypeSymbol* _tmp0;
	ValaStruct* st;
	gboolean _tmp1;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	_tmp0 = NULL;
	st = (_tmp0 = (ValaTypeSymbol*) gee_map_get ((GeeMap*) self->priv->cname_type_map, type_name), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL);
	_tmp1 = FALSE;
	if (st != NULL) {
		_tmp1 = vala_struct_is_simple_type (st);
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		gboolean _tmp2;
		return (_tmp2 = TRUE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp2);
	}
	return (_tmp3 = FALSE, (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), _tmp3);
}


static ValaDataType* vala_gidl_parser_parse_type_string (ValaGIdlParser* self, const char* n) {
	ValaUnresolvedType* type;
	ValaTypeSymbol* dt;
	char** _tmp7;
	gint type_attributes_size;
	gint type_attributes_length1;
	gint _tmp6;
	char** type_attributes;
	char* ns_name;
	ValaDataType* _tmp32;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (n != NULL, NULL);
	if (_vala_strcmp0 (n, "va_list") == 0) {
		ValaVoidType* _tmp0;
		ValaDataType* _tmp1;
		/* unsupported*/
		_tmp0 = NULL;
		_tmp1 = NULL;
		return (_tmp1 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp0 = vala_void_type_new (NULL)), NULL), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1);
	}
	type = vala_unresolved_type_new ();
	dt = (ValaTypeSymbol*) gee_map_get ((GeeMap*) self->priv->cname_type_map, n);
	if (dt != NULL) {
		ValaUnresolvedSymbol* parent_symbol;
		ValaUnresolvedSymbol* _tmp4;
		ValaDataType* _tmp5;
		parent_symbol = NULL;
		if (vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) dt)) != NULL) {
			ValaUnresolvedSymbol* _tmp3;
			_tmp3 = NULL;
			parent_symbol = (_tmp3 = vala_unresolved_symbol_new (NULL, vala_symbol_get_name (vala_symbol_get_parent_symbol ((ValaSymbol*) dt)), NULL), (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), _tmp3);
		}
		_tmp4 = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, _tmp4 = vala_unresolved_symbol_new (parent_symbol, vala_symbol_get_name ((ValaSymbol*) dt), NULL));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		_tmp5 = NULL;
		return (_tmp5 = (ValaDataType*) type, (parent_symbol == NULL) ? NULL : (parent_symbol = (vala_code_node_unref (parent_symbol), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), _tmp5);
	}
	_tmp7 = NULL;
	type_attributes = (_tmp7 = vala_gidl_parser_get_attributes (self, n, &_tmp6), type_attributes_length1 = _tmp6, type_attributes_size = type_attributes_length1, _tmp7);
	ns_name = NULL;
	if (NULL != type_attributes) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = type_attributes;
			attr_collection_length1 = type_attributes_length1;
			for (attr_it = 0; attr_it < type_attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp17;
				char* attr;
				_tmp17 = NULL;
				attr = (_tmp17 = attr_collection[attr_it], (_tmp17 == NULL) ? NULL : g_strdup (_tmp17));
				{
					char** _tmp9;
					gint nv_size;
					gint nv_length1;
					char** _tmp8;
					char** nv;
					_tmp9 = NULL;
					_tmp8 = NULL;
					nv = (_tmp9 = _tmp8 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp8), nv_size = nv_length1, _tmp9);
					if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
						ValaUnresolvedSymbol* _tmp11;
						char* _tmp10;
						_tmp11 = NULL;
						_tmp10 = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, _tmp11 = vala_unresolved_symbol_new (NULL, g_utf8_offset_to_pointer (n, g_utf8_strlen (_tmp10 = vala_gidl_parser_eval (self, nv[1]), -1)), NULL));
						(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL));
						_tmp10 = (g_free (_tmp10), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "name") == 0) {
							ValaUnresolvedSymbol* _tmp13;
							char* _tmp12;
							_tmp13 = NULL;
							_tmp12 = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, _tmp13 = vala_unresolved_symbol_new (NULL, _tmp12 = vala_gidl_parser_eval (self, nv[1]), NULL));
							(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_code_node_unref (_tmp13), NULL));
							_tmp12 = (g_free (_tmp12), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "namespace") == 0) {
								char* _tmp14;
								_tmp14 = NULL;
								ns_name = (_tmp14 = vala_gidl_parser_eval (self, nv[1]), ns_name = (g_free (ns_name), NULL), _tmp14);
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									ValaUnresolvedSymbol* _tmp16;
									char* _tmp15;
									_tmp16 = NULL;
									_tmp15 = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, _tmp16 = vala_unresolved_symbol_new (NULL, _tmp15 = vala_gidl_parser_eval (self, nv[1]), NULL));
									(_tmp16 == NULL) ? NULL : (_tmp16 = (vala_code_node_unref (_tmp16), NULL));
									_tmp15 = (g_free (_tmp15), NULL);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (vala_unresolved_type_get_unresolved_symbol (type) != NULL) {
		ValaDataType* _tmp22;
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) vala_unresolved_type_get_unresolved_symbol (type)), "pointer") == 0) {
			ValaVoidType* _tmp18;
			ValaDataType* _tmp19;
			ValaDataType* _tmp20;
			_tmp18 = NULL;
			_tmp19 = NULL;
			_tmp20 = NULL;
			return (_tmp20 = (_tmp19 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp18 = vala_void_type_new (NULL)), NULL), (_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL)), _tmp19), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp20);
		}
		if (ns_name != NULL) {
			ValaUnresolvedSymbol* _tmp21;
			_tmp21 = NULL;
			vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol (type), _tmp21 = vala_unresolved_symbol_new (NULL, ns_name, NULL));
			(_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL));
		}
		_tmp22 = NULL;
		return (_tmp22 = (ValaDataType*) type, (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp22);
	}
	if (g_str_has_prefix (n, vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace))) {
		ValaUnresolvedSymbol* _tmp24;
		ValaUnresolvedSymbol* _tmp23;
		_tmp24 = NULL;
		_tmp23 = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, _tmp24 = vala_unresolved_symbol_new (_tmp23 = vala_unresolved_symbol_new (NULL, vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace), NULL), g_utf8_offset_to_pointer (n, g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) self->priv->current_namespace), -1)), NULL));
		(_tmp24 == NULL) ? NULL : (_tmp24 = (vala_code_node_unref (_tmp24), NULL));
		(_tmp23 == NULL) ? NULL : (_tmp23 = (vala_code_node_unref (_tmp23), NULL));
	} else {
		if (g_str_has_prefix (n, "G")) {
			ValaUnresolvedSymbol* _tmp26;
			ValaUnresolvedSymbol* _tmp25;
			_tmp26 = NULL;
			_tmp25 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, _tmp26 = vala_unresolved_symbol_new (_tmp25 = vala_unresolved_symbol_new (NULL, "GLib", NULL), g_utf8_offset_to_pointer (n, (glong) 1), NULL));
			(_tmp26 == NULL) ? NULL : (_tmp26 = (vala_code_node_unref (_tmp26), NULL));
			(_tmp25 == NULL) ? NULL : (_tmp25 = (vala_code_node_unref (_tmp25), NULL));
		} else {
			char** _tmp28;
			gint name_parts_size;
			gint name_parts_length1;
			char** _tmp27;
			char** name_parts;
			_tmp28 = NULL;
			_tmp27 = NULL;
			name_parts = (_tmp28 = _tmp27 = g_strsplit (n, ".", 2), name_parts_length1 = g_strv_length (_tmp27), name_parts_size = name_parts_length1, _tmp28);
			if (name_parts[1] == NULL) {
				ValaUnresolvedSymbol* _tmp29;
				_tmp29 = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp29 = vala_unresolved_symbol_new (NULL, name_parts[0], NULL));
				(_tmp29 == NULL) ? NULL : (_tmp29 = (vala_code_node_unref (_tmp29), NULL));
			} else {
				ValaUnresolvedSymbol* _tmp31;
				ValaUnresolvedSymbol* _tmp30;
				_tmp31 = NULL;
				_tmp30 = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, _tmp31 = vala_unresolved_symbol_new (_tmp30 = vala_unresolved_symbol_new (NULL, name_parts[0], NULL), name_parts[1], NULL));
				(_tmp31 == NULL) ? NULL : (_tmp31 = (vala_code_node_unref (_tmp31), NULL));
				(_tmp30 == NULL) ? NULL : (_tmp30 = (vala_code_node_unref (_tmp30), NULL));
			}
			name_parts = (_vala_array_free (name_parts, name_parts_length1, (GDestroyNotify) g_free), NULL);
		}
	}
	_tmp32 = NULL;
	return (_tmp32 = (ValaDataType*) type, (dt == NULL) ? NULL : (dt = (vala_code_node_unref (dt), NULL)), type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, (GDestroyNotify) g_free), NULL), ns_name = (g_free (ns_name), NULL), _tmp32);
}


static ValaDataType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param, ValaParameterDirection* direction) {
	ValaDataType* type;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type, &(*direction));
	/* disable for now as null_ok not yet correctly set
	 type.non_null = !param.null_ok;*/
	return type;
}


static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface) {
	ValaDataType* return_type;
	ValaMethod* m;
	gboolean _tmp1;
	gboolean add_ellipsis;
	gboolean suppress_throws;
	char** _tmp21;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp20;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaDataType* last_param_type;
	gboolean _tmp133;
	ValaMethod* _tmp135;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (symbol != NULL, NULL);
	return_type = NULL;
	if (res != NULL) {
		ValaDataType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gidl_parser_parse_param (self, res, NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp0);
	}
	m = NULL;
	_tmp1 = FALSE;
	if (!is_interface) {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if (is_constructor) {
			_tmp2 = TRUE;
		} else {
			_tmp2 = g_str_has_prefix (name, "new");
		}
		_tmp1 = (_tmp2);
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		ValaMethod* _tmp3;
		gboolean _tmp4;
		_tmp3 = NULL;
		m = (_tmp3 = (ValaMethod*) vala_creation_method_new (NULL, name, self->priv->current_source_reference), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp3);
		vala_method_set_has_construct_function (m, FALSE);
		if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) m), "new") == 0) {
			vala_symbol_set_name ((ValaSymbol*) m, NULL);
		} else {
			if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "new_")) {
				vala_symbol_set_name ((ValaSymbol*) m, g_utf8_offset_to_pointer (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("new_", -1)));
			}
		}
		_tmp4 = FALSE;
		if (VALA_IS_CLASS (self->priv->current_data_type)) {
			_tmp4 = res != NULL;
		} else {
			_tmp4 = FALSE;
		}
		/* For classes, check whether a creation method return type equals to the
		 type of the class created. If the types do not match (e.g. in most
		 gtk widgets) add an attribute to the creation method indicating the used
		 return type.*/
		if (_tmp4) {
			char* _tmp6;
			char* _tmp5;
			gboolean _tmp7;
			_tmp6 = NULL;
			_tmp5 = NULL;
			if ((_tmp7 = _vala_strcmp0 (_tmp6 = g_strdup_printf ("%s*", _tmp5 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), res->type->unparsed) != 0, _tmp6 = (g_free (_tmp6), NULL), _tmp5 = (g_free (_tmp5), NULL), _tmp7)) {
				vala_creation_method_set_custom_return_type_cname ((VALA_CREATION_METHOD (m)), res->type->unparsed);
			}
		}
	} else {
		ValaMethod* _tmp8;
		_tmp8 = NULL;
		m = (_tmp8 = vala_method_new (name, return_type, self->priv->current_source_reference), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp8);
	}
	vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, name);
	}
	if (self->priv->current_data_type != NULL) {
		char** _tmp13;
		gint sig_attributes_size;
		gint sig_attributes_length1;
		gint _tmp11;
		char* _tmp10;
		char* _tmp9;
		char** _tmp12;
		char** sig_attributes;
		_tmp13 = NULL;
		_tmp10 = NULL;
		_tmp9 = NULL;
		_tmp12 = NULL;
		sig_attributes = (_tmp13 = (_tmp12 = vala_gidl_parser_get_attributes (self, _tmp10 = g_strdup_printf ("%s::%s", _tmp9 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), name), &_tmp11), _tmp10 = (g_free (_tmp10), NULL), _tmp9 = (g_free (_tmp9), NULL), _tmp12), sig_attributes_length1 = _tmp11, sig_attributes_size = sig_attributes_length1, _tmp13);
		if (sig_attributes != NULL) {
			{
				char** attr_collection;
				int attr_collection_length1;
				int attr_it;
				attr_collection = sig_attributes;
				attr_collection_length1 = sig_attributes_length1;
				for (attr_it = 0; attr_it < sig_attributes_length1; attr_it = attr_it + 1) {
					const char* _tmp19;
					char* attr;
					_tmp19 = NULL;
					attr = (_tmp19 = attr_collection[attr_it], (_tmp19 == NULL) ? NULL : g_strdup (_tmp19));
					{
						char** _tmp15;
						gint nv_size;
						gint nv_length1;
						char** _tmp14;
						char** nv;
						gboolean _tmp16;
						_tmp15 = NULL;
						_tmp14 = NULL;
						nv = (_tmp15 = _tmp14 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp14), nv_size = nv_length1, _tmp15);
						_tmp16 = FALSE;
						if (_vala_strcmp0 (nv[0], "has_emitter") == 0) {
							char* _tmp17;
							_tmp17 = NULL;
							_tmp16 = _vala_strcmp0 (_tmp17 = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
							_tmp17 = (g_free (_tmp17), NULL);
						} else {
							_tmp16 = FALSE;
						}
						if (_tmp16) {
							ValaMethod* _tmp18;
							_tmp18 = NULL;
							return (_tmp18 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, (GDestroyNotify) g_free), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), _tmp18);
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
					}
				}
			}
		}
		sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, (GDestroyNotify) g_free), NULL);
	}
	add_ellipsis = FALSE;
	suppress_throws = FALSE;
	_tmp21 = NULL;
	attributes = (_tmp21 = vala_gidl_parser_get_attributes (self, symbol, &_tmp20), attributes_length1 = _tmp20, attributes_size = attributes_length1, _tmp21);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp57;
				char* attr;
				_tmp57 = NULL;
				attr = (_tmp57 = attr_collection[attr_it], (_tmp57 == NULL) ? NULL : g_strdup (_tmp57));
				{
					char** _tmp23;
					gint nv_size;
					gint nv_length1;
					char** _tmp22;
					char** nv;
					_tmp23 = NULL;
					_tmp22 = NULL;
					nv = (_tmp23 = _tmp22 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp22), nv_size = nv_length1, _tmp23);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp24;
						vala_method_set_cname (m, vala_symbol_get_name ((ValaSymbol*) m));
						_tmp24 = NULL;
						vala_symbol_set_name ((ValaSymbol*) m, _tmp24 = vala_gidl_parser_eval (self, nv[1]));
						_tmp24 = (g_free (_tmp24), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "hidden") == 0) {
							char* _tmp25;
							gboolean _tmp26;
							_tmp25 = NULL;
							if ((_tmp26 = _vala_strcmp0 (_tmp25 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25 = (g_free (_tmp25), NULL), _tmp26)) {
								ValaMethod* _tmp27;
								_tmp27 = NULL;
								return (_tmp27 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), (m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp27);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "ellipsis") == 0) {
								char* _tmp28;
								gboolean _tmp29;
								_tmp28 = NULL;
								if ((_tmp29 = _vala_strcmp0 (_tmp28 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp28 = (g_free (_tmp28), NULL), _tmp29)) {
									add_ellipsis = TRUE;
								}
							} else {
								if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
									char* _tmp30;
									gboolean _tmp31;
									_tmp30 = NULL;
									if ((_tmp31 = _vala_strcmp0 (_tmp30 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30 = (g_free (_tmp30), NULL), _tmp31)) {
										vala_data_type_set_value_owned (return_type, TRUE);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "nullable") == 0) {
										char* _tmp32;
										gboolean _tmp33;
										_tmp32 = NULL;
										if ((_tmp33 = _vala_strcmp0 (_tmp32 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32 = (g_free (_tmp32), NULL), _tmp33)) {
											vala_data_type_set_nullable (return_type, TRUE);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "sentinel") == 0) {
											char* _tmp34;
											_tmp34 = NULL;
											vala_method_set_sentinel (m, _tmp34 = vala_gidl_parser_eval (self, nv[1]));
											_tmp34 = (g_free (_tmp34), NULL);
										} else {
											if (_vala_strcmp0 (nv[0], "is_array") == 0) {
												char* _tmp35;
												gboolean _tmp36;
												_tmp35 = NULL;
												if ((_tmp36 = _vala_strcmp0 (_tmp35 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp35 = (g_free (_tmp35), NULL), _tmp36)) {
													ValaDataType* _tmp37;
													_tmp37 = NULL;
													return_type = (_tmp37 = (ValaDataType*) vala_array_type_new (return_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) return_type)), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp37);
													vala_method_set_return_type (m, return_type);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "throws") == 0) {
													char* _tmp38;
													gboolean _tmp39;
													_tmp38 = NULL;
													if ((_tmp39 = _vala_strcmp0 (_tmp38 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp38 = (g_free (_tmp38), NULL), _tmp39)) {
														suppress_throws = TRUE;
													}
												} else {
													if (_vala_strcmp0 (nv[0], "no_array_length") == 0) {
														char* _tmp40;
														gboolean _tmp41;
														_tmp40 = NULL;
														if ((_tmp41 = _vala_strcmp0 (_tmp40 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp40 = (g_free (_tmp40), NULL), _tmp41)) {
															vala_method_set_no_array_length (m, TRUE);
														}
													} else {
														if (_vala_strcmp0 (nv[0], "type_name") == 0) {
															char* _tmp42;
															ValaUnresolvedSymbol* _tmp43;
															ValaUnresolvedSymbol* sym;
															_tmp42 = NULL;
															_tmp43 = NULL;
															sym = (_tmp43 = vala_unresolved_symbol_new (NULL, _tmp42 = vala_gidl_parser_eval (self, nv[1]), NULL), _tmp42 = (g_free (_tmp42), NULL), _tmp43);
															if (VALA_IS_UNRESOLVED_TYPE (return_type)) {
																vala_unresolved_type_set_unresolved_symbol ((VALA_UNRESOLVED_TYPE (return_type)), sym);
															} else {
																ValaDataType* _tmp44;
																/* Overwrite old return_type, so "type_name" must be before any
																 other return type modifying metadata*/
																_tmp44 = NULL;
																vala_method_set_return_type (m, return_type = (_tmp44 = (ValaDataType*) vala_unresolved_type_new_from_symbol (sym, vala_code_node_get_source_reference ((ValaCodeNode*) return_type)), (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), _tmp44));
															}
															(sym == NULL) ? NULL : (sym = (vala_code_node_unref (sym), NULL));
														} else {
															if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
																char** _tmp48;
																gint type_args_size;
																gint type_args_length1;
																char** _tmp46;
																char* _tmp45;
																char** _tmp47;
																char** type_args;
																_tmp48 = NULL;
																_tmp46 = NULL;
																_tmp45 = NULL;
																_tmp47 = NULL;
																type_args = (_tmp48 = (_tmp47 = _tmp46 = g_strsplit (_tmp45 = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp45 = (g_free (_tmp45), NULL), _tmp47), type_args_length1 = g_strv_length (_tmp46), type_args_size = type_args_length1, _tmp48);
																{
																	char** type_arg_collection;
																	int type_arg_collection_length1;
																	int type_arg_it;
																	type_arg_collection = type_args;
																	type_arg_collection_length1 = type_args_length1;
																	for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
																		const char* _tmp51;
																		char* type_arg;
																		_tmp51 = NULL;
																		type_arg = (_tmp51 = type_arg_collection[type_arg_it], (_tmp51 == NULL) ? NULL : g_strdup (_tmp51));
																		{
																			ValaUnresolvedSymbol* _tmp49;
																			ValaUnresolvedType* _tmp50;
																			ValaUnresolvedType* arg_type;
																			_tmp49 = NULL;
																			_tmp50 = NULL;
																			arg_type = (_tmp50 = vala_unresolved_type_new_from_symbol (_tmp49 = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp49 == NULL) ? NULL : (_tmp49 = (vala_code_node_unref (_tmp49), NULL)), _tmp50);
																			vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
																			vala_data_type_add_type_argument (return_type, (ValaDataType*) arg_type);
																			type_arg = (g_free (type_arg), NULL);
																			(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
																		}
																	}
																}
																type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
															} else {
																if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
																	char* _tmp52;
																	_tmp52 = NULL;
																	vala_member_add_cheader_filename ((ValaMember*) m, _tmp52 = vala_gidl_parser_eval (self, nv[1]));
																	_tmp52 = (g_free (_tmp52), NULL);
																} else {
																	if (_vala_strcmp0 (nv[0], "abstract") == 0) {
																		char* _tmp53;
																		gboolean _tmp54;
																		_tmp53 = NULL;
																		if ((_tmp54 = _vala_strcmp0 (_tmp53 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp53 = (g_free (_tmp53), NULL), _tmp54)) {
																			vala_method_set_is_abstract (m, TRUE);
																		}
																	} else {
																		if (_vala_strcmp0 (nv[0], "virtual") == 0) {
																			char* _tmp55;
																			gboolean _tmp56;
																			_tmp55 = NULL;
																			if ((_tmp56 = _vala_strcmp0 (_tmp55 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp55 = (g_free (_tmp55), NULL), _tmp56)) {
																				vala_method_set_is_virtual (m, TRUE);
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_method_set_cname (m, symbol);
	first = TRUE;
	last_param = NULL;
	last_param_type = NULL;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				gboolean _tmp70;
				const char* _tmp72;
				char* param_name;
				ValaParameterDirection direction;
				ValaDataType* param_type;
				ValaFormalParameter* p;
				gboolean hide_param;
				gboolean show_param;
				gboolean set_array_length_pos;
				double array_length_pos;
				gboolean set_delegate_target_pos;
				double delegate_target_pos;
				gboolean array_requested;
				char** _tmp77;
				gint _tmp76;
				char* _tmp75;
				gboolean _tmp124;
				gboolean _tmp128;
				ValaFormalParameter* _tmp130;
				ValaFormalParameter* _tmp129;
				ValaDataType* _tmp132;
				ValaDataType* _tmp131;
				param_node = (GIdlNode*) param;
				if (first) {
					gboolean _tmp58;
					gboolean _tmp59;
					gboolean _tmp60;
					first = FALSE;
					_tmp58 = FALSE;
					_tmp59 = FALSE;
					_tmp60 = FALSE;
					if (!(VALA_IS_CREATION_METHOD (m))) {
						_tmp60 = self->priv->current_data_type != NULL;
					} else {
						_tmp60 = FALSE;
					}
					if (_tmp60) {
						_tmp59 = param->type->is_interface;
					} else {
						_tmp59 = FALSE;
					}
					if (_tmp59) {
						gboolean _tmp61;
						_tmp61 = FALSE;
						if (_vala_strcmp0 (param_node->name, "self") == 0) {
							_tmp61 = TRUE;
						} else {
							char* _tmp62;
							_tmp62 = NULL;
							_tmp61 = g_str_has_suffix (param->type->interface, _tmp62 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE));
							_tmp62 = (g_free (_tmp62), NULL);
						}
						_tmp58 = (_tmp61);
					} else {
						_tmp58 = FALSE;
					}
					if (_tmp58) {
						/* instance method*/
						continue;
					} else {
						gboolean _tmp63;
						gboolean _tmp64;
						gboolean _tmp65;
						_tmp63 = FALSE;
						_tmp64 = FALSE;
						_tmp65 = FALSE;
						if (!(VALA_IS_CREATION_METHOD (m))) {
							_tmp65 = self->priv->current_data_type != NULL;
						} else {
							_tmp65 = FALSE;
						}
						if (_tmp65) {
							_tmp64 = param->type->is_interface;
						} else {
							_tmp64 = FALSE;
						}
						if (_tmp64) {
							gboolean _tmp66;
							_tmp66 = FALSE;
							if (_vala_strcmp0 (param_node->name, "klass") == 0) {
								_tmp66 = TRUE;
							} else {
								char* _tmp68;
								char* _tmp67;
								_tmp68 = NULL;
								_tmp67 = NULL;
								_tmp66 = g_str_has_suffix (param->type->interface, _tmp68 = g_strdup_printf ("%sClass", _tmp67 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)));
								_tmp68 = (g_free (_tmp68), NULL);
								_tmp67 = (g_free (_tmp67), NULL);
							}
							_tmp63 = (_tmp66);
						} else {
							_tmp63 = FALSE;
						}
						if (_tmp63) {
							/* class method*/
							vala_method_set_binding (m, MEMBER_BINDING_CLASS);
							if (g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) m), "class_")) {
								char* _tmp69;
								_tmp69 = NULL;
								vala_symbol_set_name ((ValaSymbol*) m, _tmp69 = string_substring (vala_symbol_get_name ((ValaSymbol*) m), g_utf8_strlen ("class_", -1), g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) m), -1) - g_utf8_strlen ("class_", -1)));
								_tmp69 = (g_free (_tmp69), NULL);
							}
							continue;
						} else {
							/* static method*/
							vala_method_set_binding (m, MEMBER_BINDING_STATIC);
						}
					}
				}
				_tmp70 = FALSE;
				if (suppress_throws == FALSE) {
					_tmp70 = vala_gidl_parser_param_is_exception (self, param);
				} else {
					_tmp70 = FALSE;
				}
				if (_tmp70) {
					ValaDataType* _tmp71;
					_tmp71 = NULL;
					vala_code_node_add_error_type ((ValaCodeNode*) m, _tmp71 = vala_gidl_parser_parse_type (self, param->type, NULL));
					(_tmp71 == NULL) ? NULL : (_tmp71 = (vala_code_node_unref (_tmp71), NULL));
					continue;
				}
				_tmp72 = NULL;
				param_name = (_tmp72 = param_node->name, (_tmp72 == NULL) ? NULL : g_strdup (_tmp72));
				if (_vala_strcmp0 (param_name, "result") == 0) {
					char* _tmp73;
					/* avoid conflict with generated result variable*/
					_tmp73 = NULL;
					param_name = (_tmp73 = g_strdup ("_result"), param_name = (g_free (param_name), NULL), _tmp73);
				} else {
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp74;
						/* avoid conflict with string type*/
						_tmp74 = NULL;
						param_name = (_tmp74 = g_strdup ("str"), param_name = (g_free (param_name), NULL), _tmp74);
					}
				}
				direction = 0;
				param_type = vala_gidl_parser_parse_param (self, param, &direction);
				p = vala_formal_parameter_new (param_name, param_type, NULL);
				vala_formal_parameter_set_direction (p, direction);
				hide_param = FALSE;
				show_param = FALSE;
				set_array_length_pos = FALSE;
				array_length_pos = (double) 0;
				set_delegate_target_pos = FALSE;
				delegate_target_pos = (double) 0;
				array_requested = FALSE;
				_tmp77 = NULL;
				_tmp75 = NULL;
				attributes = (_tmp77 = vala_gidl_parser_get_attributes (self, _tmp75 = g_strdup_printf ("%s.%s", symbol, param_node->name), &_tmp76), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp76, attributes_size = attributes_length1, _tmp77);
				_tmp75 = (g_free (_tmp75), NULL);
				if (attributes != NULL) {
					{
						char** attr_collection;
						int attr_collection_length1;
						int attr_it;
						attr_collection = attributes;
						attr_collection_length1 = attributes_length1;
						for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
							const char* _tmp123;
							char* attr;
							_tmp123 = NULL;
							attr = (_tmp123 = attr_collection[attr_it], (_tmp123 == NULL) ? NULL : g_strdup (_tmp123));
							{
								char** _tmp79;
								gint nv_size;
								gint nv_length1;
								char** _tmp78;
								char** nv;
								_tmp79 = NULL;
								_tmp78 = NULL;
								nv = (_tmp79 = _tmp78 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp78), nv_size = nv_length1, _tmp79);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp80;
									gboolean _tmp81;
									_tmp80 = NULL;
									if ((_tmp81 = _vala_strcmp0 (_tmp80 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp80 = (g_free (_tmp80), NULL), _tmp81)) {
										ValaDataType* _tmp82;
										_tmp82 = NULL;
										param_type = (_tmp82 = (ValaDataType*) vala_array_type_new (param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) param_type)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp82);
										vala_formal_parameter_set_parameter_type (p, param_type);
										vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_IN);
										array_requested = TRUE;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp83;
										gboolean _tmp84;
										_tmp83 = NULL;
										if ((_tmp84 = _vala_strcmp0 (_tmp83 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp83 = (g_free (_tmp83), NULL), _tmp84)) {
											gboolean _tmp85;
											vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_OUT);
											_tmp85 = FALSE;
											if (!array_requested) {
												_tmp85 = VALA_IS_ARRAY_TYPE (param_type);
											} else {
												_tmp85 = FALSE;
											}
											if (_tmp85) {
												ValaArrayType* _tmp86;
												ValaArrayType* array_type;
												ValaDataType* _tmp88;
												ValaDataType* _tmp87;
												_tmp86 = NULL;
												array_type = (_tmp86 = VALA_ARRAY_TYPE (param_type), (_tmp86 == NULL) ? NULL : vala_code_node_ref (_tmp86));
												_tmp88 = NULL;
												_tmp87 = NULL;
												param_type = (_tmp88 = (_tmp87 = vala_array_type_get_element_type (array_type), (_tmp87 == NULL) ? NULL : vala_code_node_ref (_tmp87)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp88);
												vala_formal_parameter_set_parameter_type (p, param_type);
												(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
											}
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp89;
											gboolean _tmp90;
											_tmp89 = NULL;
											if ((_tmp90 = _vala_strcmp0 (_tmp89 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp89 = (g_free (_tmp89), NULL), _tmp90)) {
												gboolean _tmp91;
												vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_REF);
												_tmp91 = FALSE;
												if (!array_requested) {
													_tmp91 = VALA_IS_ARRAY_TYPE (param_type);
												} else {
													_tmp91 = FALSE;
												}
												if (_tmp91) {
													ValaArrayType* _tmp92;
													ValaArrayType* array_type;
													ValaDataType* _tmp94;
													ValaDataType* _tmp93;
													_tmp92 = NULL;
													array_type = (_tmp92 = VALA_ARRAY_TYPE (param_type), (_tmp92 == NULL) ? NULL : vala_code_node_ref (_tmp92));
													_tmp94 = NULL;
													_tmp93 = NULL;
													param_type = (_tmp94 = (_tmp93 = vala_array_type_get_element_type (array_type), (_tmp93 == NULL) ? NULL : vala_code_node_ref (_tmp93)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp94);
													vala_formal_parameter_set_parameter_type (p, param_type);
													(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
												}
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp95;
												gboolean _tmp96;
												_tmp95 = NULL;
												if ((_tmp96 = _vala_strcmp0 (_tmp95 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp95 = (g_free (_tmp95), NULL), _tmp96)) {
													vala_data_type_set_nullable (param_type, TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
													char* _tmp97;
													gboolean _tmp98;
													_tmp97 = NULL;
													if ((_tmp98 = _vala_strcmp0 (_tmp97 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp97 = (g_free (_tmp97), NULL), _tmp98)) {
														vala_data_type_set_value_owned (param_type, TRUE);
													}
												} else {
													if (_vala_strcmp0 (nv[0], "takes_ownership") == 0) {
														char* _tmp99;
														gboolean _tmp100;
														_tmp99 = NULL;
														if ((_tmp100 = _vala_strcmp0 (_tmp99 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp99 = (g_free (_tmp99), NULL), _tmp100)) {
															vala_data_type_set_value_owned (param_type, TRUE);
														}
													} else {
														if (_vala_strcmp0 (nv[0], "value_owned") == 0) {
															char* _tmp101;
															gboolean _tmp102;
															_tmp101 = NULL;
															if ((_tmp102 = _vala_strcmp0 (_tmp101 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp101 = (g_free (_tmp101), NULL), _tmp102)) {
																vala_data_type_set_value_owned (param_type, FALSE);
															} else {
																char* _tmp103;
																gboolean _tmp104;
																_tmp103 = NULL;
																if ((_tmp104 = _vala_strcmp0 (_tmp103 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp103 = (g_free (_tmp103), NULL), _tmp104)) {
																	vala_data_type_set_value_owned (param_type, TRUE);
																}
															}
														} else {
															if (_vala_strcmp0 (nv[0], "hidden") == 0) {
																char* _tmp105;
																gboolean _tmp106;
																_tmp105 = NULL;
																if ((_tmp106 = _vala_strcmp0 (_tmp105 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp105 = (g_free (_tmp105), NULL), _tmp106)) {
																	hide_param = TRUE;
																} else {
																	char* _tmp107;
																	gboolean _tmp108;
																	_tmp107 = NULL;
																	if ((_tmp108 = _vala_strcmp0 (_tmp107 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp107 = (g_free (_tmp107), NULL), _tmp108)) {
																		show_param = TRUE;
																	}
																}
															} else {
																if (_vala_strcmp0 (nv[0], "no_array_length") == 0) {
																	char* _tmp109;
																	gboolean _tmp110;
																	_tmp109 = NULL;
																	if ((_tmp110 = _vala_strcmp0 (_tmp109 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp109 = (g_free (_tmp109), NULL), _tmp110)) {
																		vala_formal_parameter_set_no_array_length (p, TRUE);
																	}
																} else {
																	if (_vala_strcmp0 (nv[0], "array_length_pos") == 0) {
																		char* _tmp111;
																		set_array_length_pos = TRUE;
																		_tmp111 = NULL;
																		array_length_pos = strtod (_tmp111 = vala_gidl_parser_eval (self, nv[1]), NULL);
																		_tmp111 = (g_free (_tmp111), NULL);
																	} else {
																		if (_vala_strcmp0 (nv[0], "delegate_target_pos") == 0) {
																			char* _tmp112;
																			set_delegate_target_pos = TRUE;
																			_tmp112 = NULL;
																			delegate_target_pos = strtod (_tmp112 = vala_gidl_parser_eval (self, nv[1]), NULL);
																			_tmp112 = (g_free (_tmp112), NULL);
																		} else {
																			if (_vala_strcmp0 (nv[0], "type_name") == 0) {
																				ValaUnresolvedSymbol* _tmp114;
																				char* _tmp113;
																				_tmp114 = NULL;
																				_tmp113 = NULL;
																				vala_unresolved_type_set_unresolved_symbol ((VALA_UNRESOLVED_TYPE (param_type)), _tmp114 = vala_unresolved_symbol_new (NULL, _tmp113 = vala_gidl_parser_eval (self, nv[1]), NULL));
																				(_tmp114 == NULL) ? NULL : (_tmp114 = (vala_code_node_unref (_tmp114), NULL));
																				_tmp113 = (g_free (_tmp113), NULL);
																			} else {
																				if (_vala_strcmp0 (nv[0], "ctype") == 0) {
																					char* _tmp115;
																					_tmp115 = NULL;
																					vala_formal_parameter_set_ctype (p, _tmp115 = vala_gidl_parser_eval (self, nv[1]));
																					_tmp115 = (g_free (_tmp115), NULL);
																				} else {
																					if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
																						char** _tmp119;
																						gint type_args_size;
																						gint type_args_length1;
																						char** _tmp117;
																						char* _tmp116;
																						char** _tmp118;
																						char** type_args;
																						_tmp119 = NULL;
																						_tmp117 = NULL;
																						_tmp116 = NULL;
																						_tmp118 = NULL;
																						type_args = (_tmp119 = (_tmp118 = _tmp117 = g_strsplit (_tmp116 = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp116 = (g_free (_tmp116), NULL), _tmp118), type_args_length1 = g_strv_length (_tmp117), type_args_size = type_args_length1, _tmp119);
																						{
																							char** type_arg_collection;
																							int type_arg_collection_length1;
																							int type_arg_it;
																							type_arg_collection = type_args;
																							type_arg_collection_length1 = type_args_length1;
																							for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
																								const char* _tmp122;
																								char* type_arg;
																								_tmp122 = NULL;
																								type_arg = (_tmp122 = type_arg_collection[type_arg_it], (_tmp122 == NULL) ? NULL : g_strdup (_tmp122));
																								{
																									ValaUnresolvedSymbol* _tmp120;
																									ValaUnresolvedType* _tmp121;
																									ValaUnresolvedType* arg_type;
																									_tmp120 = NULL;
																									_tmp121 = NULL;
																									arg_type = (_tmp121 = vala_unresolved_type_new_from_symbol (_tmp120 = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp120 == NULL) ? NULL : (_tmp120 = (vala_code_node_unref (_tmp120), NULL)), _tmp121);
																									vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
																									vala_data_type_add_type_argument (param_type, (ValaDataType*) arg_type);
																									type_arg = (g_free (type_arg), NULL);
																									(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
																								}
																							}
																						}
																						type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
							}
						}
					}
				}
				_tmp124 = FALSE;
				if (last_param != NULL) {
					char* _tmp125;
					_tmp125 = NULL;
					_tmp124 = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) p), _tmp125 = g_strconcat ("n_", vala_symbol_get_name ((ValaSymbol*) last_param), NULL)) == 0;
					_tmp125 = (g_free (_tmp125), NULL);
				} else {
					_tmp124 = FALSE;
				}
				if (_tmp124) {
					if (!(VALA_IS_ARRAY_TYPE (last_param_type))) {
						ValaDataType* _tmp126;
						/* last_param is array, p is array length*/
						_tmp126 = NULL;
						last_param_type = (_tmp126 = (ValaDataType*) vala_array_type_new (last_param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) last_param_type)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp126);
						vala_formal_parameter_set_parameter_type (last_param, last_param_type);
						vala_formal_parameter_set_direction (last_param, VALA_PARAMETER_DIRECTION_IN);
					}
					/* hide array length param*/
					hide_param = TRUE;
				} else {
					gboolean _tmp127;
					_tmp127 = FALSE;
					if (last_param != NULL) {
						_tmp127 = _vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) p), "user_data") == 0;
					} else {
						_tmp127 = FALSE;
					}
					if (_tmp127) {
						/* last_param is delegate
						 hide deleate target param*/
						hide_param = TRUE;
					}
				}
				_tmp128 = FALSE;
				if (show_param) {
					_tmp128 = TRUE;
				} else {
					_tmp128 = !hide_param;
				}
				if (_tmp128) {
					vala_method_add_parameter (m, p);
					if (set_array_length_pos) {
						vala_formal_parameter_set_carray_length_parameter_position (p, array_length_pos);
					}
					if (set_delegate_target_pos) {
						vala_formal_parameter_set_cdelegate_target_parameter_position (p, delegate_target_pos);
					}
				}
				_tmp130 = NULL;
				_tmp129 = NULL;
				last_param = (_tmp130 = (_tmp129 = p, (_tmp129 == NULL) ? NULL : vala_code_node_ref (_tmp129)), (last_param == NULL) ? NULL : (last_param = (vala_code_node_unref (last_param), NULL)), _tmp130);
				_tmp132 = NULL;
				_tmp131 = NULL;
				last_param_type = (_tmp132 = (_tmp131 = param_type, (_tmp131 == NULL) ? NULL : vala_code_node_ref (_tmp131)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp132);
				param_name = (g_free (param_name), NULL);
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
				(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		vala_method_set_binding (m, MEMBER_BINDING_STATIC);
	}
	_tmp133 = FALSE;
	if (last_param != NULL) {
		_tmp133 = g_str_has_prefix (vala_symbol_get_name ((ValaSymbol*) last_param), "first_");
	} else {
		_tmp133 = FALSE;
	}
	if (_tmp133) {
		vala_formal_parameter_set_ellipsis (last_param, TRUE);
	} else {
		if (add_ellipsis) {
			ValaFormalParameter* _tmp134;
			_tmp134 = NULL;
			vala_method_add_parameter (m, _tmp134 = vala_formal_parameter_new_with_ellipsis (NULL));
			(_tmp134 == NULL) ? NULL : (_tmp134 = (vala_code_node_unref (_tmp134), NULL));
		}
	}
	_tmp135 = NULL;
	return (_tmp135 = m, (return_type == NULL) ? NULL : (return_type = (vala_code_node_unref (return_type), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), (last_param == NULL) ? NULL : (last_param = (vala_code_node_unref (last_param), NULL)), (last_param_type == NULL) ? NULL : (last_param_type = (vala_code_node_unref (last_param_type), NULL)), _tmp135);
}


static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param) {
	const char* _tmp1;
	char* s;
	gboolean _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (param != NULL, FALSE);
	if (!param->type->is_error) {
		return FALSE;
	}
	_tmp1 = NULL;
	s = (_tmp1 = g_strchomp (param->type->unparsed), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
	if (g_str_has_suffix (s, "**")) {
		gboolean _tmp2;
		return (_tmp2 = TRUE, s = (g_free (s), NULL), _tmp2);
	}
	return (_tmp3 = FALSE, s = (g_free (s), NULL), _tmp3);
}


static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface) {
	GIdlNode* node;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = (GIdlNode*) f;
	if (f->deprecated) {
		return NULL;
	}
	return vala_gidl_parser_create_method (self, node->name, f->symbol, f->result, f->parameters, f->is_constructor, is_interface);
}


static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface) {
	GIdlNode* node;
	char* _tmp0;
	char* _tmp1;
	char* symbol;
	GList* _tmp4;
	ValaMethod* m;
	ValaMethod* _tmp7;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (v != NULL, NULL);
	node = (GIdlNode*) v;
	_tmp0 = NULL;
	_tmp1 = NULL;
	symbol = (_tmp1 = g_strdup_printf ("%s%s", _tmp0 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self->priv->current_data_type), node->name), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
	if (func != NULL) {
		char* _tmp3;
		const char* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		symbol = (_tmp3 = (_tmp2 = func->symbol, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), symbol = (g_free (symbol), NULL), _tmp3);
	}
	_tmp4 = NULL;
	if (func != NULL) {
		_tmp4 = func->parameters;
	} else {
		_tmp4 = v->parameters;
	}
	m = vala_gidl_parser_create_method (self, node->name, symbol, v->result, _tmp4, FALSE, is_interface);
	if (m != NULL) {
		gboolean _tmp5;
		gboolean _tmp6;
		vala_method_set_binding (m, MEMBER_BINDING_INSTANCE);
		_tmp5 = FALSE;
		if (vala_method_get_is_abstract (m)) {
			_tmp5 = TRUE;
		} else {
			_tmp5 = is_interface;
		}
		vala_method_set_is_virtual (m, !(_tmp5));
		_tmp6 = FALSE;
		if (vala_method_get_is_abstract (m)) {
			_tmp6 = TRUE;
		} else {
			_tmp6 = is_interface;
		}
		vala_method_set_is_abstract (m, _tmp6);
		if (func == NULL) {
			((ValaCodeNode*) m)->attributes = g_list_append (((ValaCodeNode*) m)->attributes, vala_attribute_new ("NoWrapper", NULL));
		}
	}
	_tmp7 = NULL;
	return (_tmp7 = m, symbol = (g_free (symbol), NULL), _tmp7);
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), i = (g_free (i), NULL), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3)), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), i = (g_free (i), NULL), _tmp4);
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node) {
	GIdlNode* node;
	gboolean _tmp1;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaDataType* _tmp5;
	char* _tmp4;
	ValaProperty* _tmp6;
	ValaProperty* prop;
	char** _tmp11;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp9;
	char* _tmp8;
	char* _tmp7;
	char** _tmp10;
	char** attributes;
	ValaProperty* _tmp25;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = (GIdlNode*) prop_node;
	if (prop_node->deprecated) {
		return NULL;
	}
	_tmp1 = FALSE;
	if (!prop_node->readable) {
		_tmp1 = !prop_node->writable;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* _tmp2;
		_tmp2 = NULL;
		get_acc = (_tmp2 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL) ? NULL : (get_acc = (vala_code_node_unref (get_acc), NULL)), _tmp2);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* _tmp3;
		_tmp3 = NULL;
		set_acc = (_tmp3 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL) ? NULL : (set_acc = (vala_code_node_unref (set_acc), NULL)), _tmp3);
		if (prop_node->construct_only) {
			vala_property_accessor_set_construction (set_acc, TRUE);
		} else {
			vala_property_accessor_set_writable (set_acc, TRUE);
			vala_property_accessor_set_construction (set_acc, prop_node->construct);
		}
	}
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp6 = NULL;
	prop = (_tmp6 = vala_property_new (_tmp4 = vala_gidl_parser_fix_prop_name (self, node->name), _tmp5 = vala_gidl_parser_parse_type (self, prop_node->type, NULL), get_acc, set_acc, self->priv->current_source_reference), (_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL)), _tmp4 = (g_free (_tmp4), NULL), _tmp6);
	vala_symbol_set_access ((ValaSymbol*) prop, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_property_set_interface_only (prop, TRUE);
	_tmp11 = NULL;
	_tmp8 = NULL;
	_tmp7 = NULL;
	_tmp10 = NULL;
	attributes = (_tmp11 = (_tmp10 = vala_gidl_parser_get_attributes (self, _tmp8 = g_strdup_printf ("%s:%s", _tmp7 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), node->name), &_tmp9), _tmp8 = (g_free (_tmp8), NULL), _tmp7 = (g_free (_tmp7), NULL), _tmp10), attributes_length1 = _tmp9, attributes_size = attributes_length1, _tmp11);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp24;
				char* attr;
				_tmp24 = NULL;
				attr = (_tmp24 = attr_collection[attr_it], (_tmp24 == NULL) ? NULL : g_strdup (_tmp24));
				{
					char** _tmp13;
					gint nv_size;
					gint nv_length1;
					char** _tmp12;
					char** nv;
					_tmp13 = NULL;
					_tmp12 = NULL;
					nv = (_tmp13 = _tmp12 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp12), nv_size = nv_length1, _tmp13);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp14;
						gboolean _tmp15;
						_tmp14 = NULL;
						if ((_tmp15 = _vala_strcmp0 (_tmp14 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp14 = (g_free (_tmp14), NULL), _tmp15)) {
							ValaProperty* _tmp16;
							_tmp16 = NULL;
							return (_tmp16 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (get_acc == NULL) ? NULL : (get_acc = (vala_code_node_unref (get_acc), NULL)), (set_acc == NULL) ? NULL : (set_acc = (vala_code_node_unref (set_acc), NULL)), (prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp16);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
							char** _tmp20;
							gint type_args_size;
							gint type_args_length1;
							char** _tmp18;
							char* _tmp17;
							char** _tmp19;
							char** type_args;
							_tmp20 = NULL;
							_tmp18 = NULL;
							_tmp17 = NULL;
							_tmp19 = NULL;
							type_args = (_tmp20 = (_tmp19 = _tmp18 = g_strsplit (_tmp17 = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp17 = (g_free (_tmp17), NULL), _tmp19), type_args_length1 = g_strv_length (_tmp18), type_args_size = type_args_length1, _tmp20);
							{
								char** type_arg_collection;
								int type_arg_collection_length1;
								int type_arg_it;
								type_arg_collection = type_args;
								type_arg_collection_length1 = type_args_length1;
								for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
									const char* _tmp23;
									char* type_arg;
									_tmp23 = NULL;
									type_arg = (_tmp23 = type_arg_collection[type_arg_it], (_tmp23 == NULL) ? NULL : g_strdup (_tmp23));
									{
										ValaUnresolvedSymbol* _tmp21;
										ValaUnresolvedType* _tmp22;
										ValaUnresolvedType* arg_type;
										_tmp21 = NULL;
										_tmp22 = NULL;
										arg_type = (_tmp22 = vala_unresolved_type_new_from_symbol (_tmp21 = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp21 == NULL) ? NULL : (_tmp21 = (vala_code_node_unref (_tmp21), NULL)), _tmp22);
										vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
										vala_data_type_add_type_argument (vala_property_get_property_type (prop), (ValaDataType*) arg_type);
										type_arg = (g_free (type_arg), NULL);
										(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
									}
								}
							}
							type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, vala_symbol_get_name ((ValaSymbol*) prop));
	}
	_tmp25 = NULL;
	return (_tmp25 = prop, (get_acc == NULL) ? NULL : (get_acc = (vala_code_node_unref (get_acc), NULL)), (set_acc == NULL) ? NULL : (set_acc = (vala_code_node_unref (set_acc), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp25);
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node) {
	GIdlNode* node;
	ValaDataType* type;
	ValaConstant* c;
	char** _tmp2;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp1;
	char** attributes;
	ValaConstant* _tmp10;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = (GIdlNode*) const_node;
	type = vala_gidl_parser_parse_type (self, const_node->type, NULL);
	if (type == NULL) {
		ValaConstant* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0);
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	_tmp2 = NULL;
	attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name, &_tmp1), attributes_length1 = _tmp1, attributes_size = attributes_length1, _tmp2);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp9;
				char* attr;
				_tmp9 = NULL;
				attr = (_tmp9 = attr_collection[attr_it], (_tmp9 == NULL) ? NULL : g_strdup (_tmp9));
				{
					char** _tmp4;
					gint nv_size;
					gint nv_length1;
					char** _tmp3;
					char** nv;
					_tmp4 = NULL;
					_tmp3 = NULL;
					nv = (_tmp4 = _tmp3 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp3), nv_size = nv_length1, _tmp4);
					if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
						char* _tmp5;
						_tmp5 = NULL;
						vala_member_add_cheader_filename ((ValaMember*) c, _tmp5 = vala_gidl_parser_eval (self, nv[1]));
						_tmp5 = (g_free (_tmp5), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "hidden") == 0) {
							char* _tmp6;
							gboolean _tmp7;
							_tmp6 = NULL;
							if ((_tmp7 = _vala_strcmp0 (_tmp6 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp6 = (g_free (_tmp6), NULL), _tmp7)) {
								ValaConstant* _tmp8;
								_tmp8 = NULL;
								return (_tmp8 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp8);
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_symbol_set_access ((ValaSymbol*) c, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp10 = NULL;
	return (_tmp10 = c, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10);
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node) {
	GIdlNode* node;
	gboolean unhidden;
	ValaDataType* type;
	char* cheader_filename;
	char* ctype;
	char** _tmp5;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp3;
	char* _tmp2;
	char* _tmp1;
	char** _tmp4;
	char** attributes;
	gboolean _tmp28;
	const char* _tmp30;
	char* field_name;
	ValaField* field;
	ValaField* _tmp32;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = (GIdlNode*) field_node;
	unhidden = FALSE;
	type = vala_gidl_parser_parse_type (self, field_node->type, NULL);
	if (type == NULL) {
		ValaField* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp0);
	}
	cheader_filename = NULL;
	ctype = NULL;
	_tmp5 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp4 = NULL;
	attributes = (_tmp5 = (_tmp4 = vala_gidl_parser_get_attributes (self, _tmp2 = g_strdup_printf ("%s.%s", _tmp1 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), node->name), &_tmp3), _tmp2 = (g_free (_tmp2), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp4), attributes_length1 = _tmp3, attributes_size = attributes_length1, _tmp5);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp27;
				char* attr;
				_tmp27 = NULL;
				attr = (_tmp27 = attr_collection[attr_it], (_tmp27 == NULL) ? NULL : g_strdup (_tmp27));
				{
					char** _tmp7;
					gint nv_size;
					gint nv_length1;
					char** _tmp6;
					char** nv;
					_tmp7 = NULL;
					_tmp6 = NULL;
					nv = (_tmp7 = _tmp6 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp6), nv_size = nv_length1, _tmp7);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp8;
						gboolean _tmp9;
						_tmp8 = NULL;
						if ((_tmp9 = _vala_strcmp0 (_tmp8 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp8 = (g_free (_tmp8), NULL), _tmp9)) {
							ValaField* _tmp10;
							_tmp10 = NULL;
							return (_tmp10 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp10);
						} else {
							unhidden = TRUE;
						}
					} else {
						if (_vala_strcmp0 (nv[0], "is_array") == 0) {
							char* _tmp11;
							gboolean _tmp12;
							_tmp11 = NULL;
							if ((_tmp12 = _vala_strcmp0 (_tmp11 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp11 = (g_free (_tmp11), NULL), _tmp12)) {
								ValaDataType* _tmp13;
								_tmp13 = NULL;
								type = (_tmp13 = (ValaDataType*) vala_array_type_new (type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) type)), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), _tmp13);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "weak") == 0) {
								char* _tmp14;
								gboolean _tmp15;
								_tmp14 = NULL;
								if ((_tmp15 = _vala_strcmp0 (_tmp14 = vala_gidl_parser_eval (self, nv[1]), "0") == 0, _tmp14 = (g_free (_tmp14), NULL), _tmp15)) {
									vala_data_type_set_value_owned (type, TRUE);
								}
							} else {
								if (_vala_strcmp0 (nv[0], "type_name") == 0) {
									ValaUnresolvedSymbol* _tmp17;
									char* _tmp16;
									_tmp17 = NULL;
									_tmp16 = NULL;
									vala_unresolved_type_set_unresolved_symbol ((VALA_UNRESOLVED_TYPE (type)), _tmp17 = vala_unresolved_symbol_new (NULL, _tmp16 = vala_gidl_parser_eval (self, nv[1]), NULL));
									(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_code_node_unref (_tmp17), NULL));
									_tmp16 = (g_free (_tmp16), NULL);
								} else {
									if (_vala_strcmp0 (nv[0], "type_arguments") == 0) {
										char** _tmp21;
										gint type_args_size;
										gint type_args_length1;
										char** _tmp19;
										char* _tmp18;
										char** _tmp20;
										char** type_args;
										_tmp21 = NULL;
										_tmp19 = NULL;
										_tmp18 = NULL;
										_tmp20 = NULL;
										type_args = (_tmp21 = (_tmp20 = _tmp19 = g_strsplit (_tmp18 = vala_gidl_parser_eval (self, nv[1]), ",", 0), _tmp18 = (g_free (_tmp18), NULL), _tmp20), type_args_length1 = g_strv_length (_tmp19), type_args_size = type_args_length1, _tmp21);
										{
											char** type_arg_collection;
											int type_arg_collection_length1;
											int type_arg_it;
											type_arg_collection = type_args;
											type_arg_collection_length1 = type_args_length1;
											for (type_arg_it = 0; type_arg_it < type_args_length1; type_arg_it = type_arg_it + 1) {
												const char* _tmp24;
												char* type_arg;
												_tmp24 = NULL;
												type_arg = (_tmp24 = type_arg_collection[type_arg_it], (_tmp24 == NULL) ? NULL : g_strdup (_tmp24));
												{
													ValaUnresolvedSymbol* _tmp22;
													ValaUnresolvedType* _tmp23;
													ValaUnresolvedType* arg_type;
													_tmp22 = NULL;
													_tmp23 = NULL;
													arg_type = (_tmp23 = vala_unresolved_type_new_from_symbol (_tmp22 = vala_unresolved_symbol_new (NULL, type_arg, NULL), NULL), (_tmp22 == NULL) ? NULL : (_tmp22 = (vala_code_node_unref (_tmp22), NULL)), _tmp23);
													vala_data_type_set_value_owned ((ValaDataType*) arg_type, TRUE);
													vala_data_type_add_type_argument (type, (ValaDataType*) arg_type);
													type_arg = (g_free (type_arg), NULL);
													(arg_type == NULL) ? NULL : (arg_type = (vala_code_node_unref (arg_type), NULL));
												}
											}
										}
										type_args = (_vala_array_free (type_args, type_args_length1, (GDestroyNotify) g_free), NULL);
									} else {
										if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
											char* _tmp25;
											_tmp25 = NULL;
											cheader_filename = (_tmp25 = vala_gidl_parser_eval (self, nv[1]), cheader_filename = (g_free (cheader_filename), NULL), _tmp25);
										} else {
											if (_vala_strcmp0 (nv[0], "ctype") == 0) {
												char* _tmp26;
												_tmp26 = NULL;
												ctype = (_tmp26 = vala_gidl_parser_eval (self, nv[1]), ctype = (g_free (ctype), NULL), _tmp26);
											}
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	_tmp28 = FALSE;
	if (g_str_has_prefix (node->name, "_")) {
		_tmp28 = !unhidden;
	} else {
		_tmp28 = FALSE;
	}
	if (_tmp28) {
		ValaField* _tmp29;
		_tmp29 = NULL;
		return (_tmp29 = NULL, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp29);
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add ((GeeCollection*) self->priv->current_type_symbol_set, node->name);
	}
	_tmp30 = NULL;
	field_name = (_tmp30 = node->name, (_tmp30 == NULL) ? NULL : g_strdup (_tmp30));
	if (_vala_strcmp0 (field_name, "string") == 0) {
		char* _tmp31;
		/* avoid conflict with string type*/
		_tmp31 = NULL;
		field_name = (_tmp31 = g_strdup ("str"), field_name = (g_free (field_name), NULL), _tmp31);
	}
	field = vala_field_new (field_name, type, NULL, self->priv->current_source_reference);
	vala_symbol_set_access ((ValaSymbol*) field, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (_vala_strcmp0 (field_name, node->name) != 0) {
		vala_field_set_cname (field, node->name);
	}
	if (ctype != NULL) {
		vala_field_set_ctype (field, ctype);
	}
	if (cheader_filename != NULL) {
		vala_member_add_cheader_filename ((ValaMember*) field, cheader_filename);
	}
	vala_field_set_no_array_length (field, TRUE);
	_tmp32 = NULL;
	return (_tmp32 = field, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), cheader_filename = (g_free (cheader_filename), NULL), ctype = (g_free (ctype), NULL), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), field_name = (g_free (field_name), NULL), _tmp32);
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode, int* result_length1) {
	char* attributes;
	char** _tmp8;
	char** _tmp9;
	char** _tmp10;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	attributes = (char*) gee_map_get (self->priv->codenode_attributes_map, codenode);
	if (attributes == NULL) {
		gboolean dot_required;
		gboolean colon_required;
		GeeSet* pattern_specs;
		dot_required = (NULL != g_utf8_strchr (codenode, (glong) (-1), (gunichar) '.'));
		colon_required = (NULL != g_utf8_strchr (codenode, (glong) (-1), (gunichar) ':'));
		pattern_specs = gee_map_get_keys (self->priv->codenode_attributes_patterns);
		{
			GeeIterator* _pattern_it;
			_pattern_it = gee_iterable_iterator ((GeeIterable*) pattern_specs);
			while (gee_iterator_next (_pattern_it)) {
				GPatternSpec* pattern;
				char* pspec;
				gboolean _tmp0;
				gboolean _tmp1;
				pattern = gee_iterator_get (_pattern_it);
				pspec = (char*) gee_map_get ((GeeMap*) self->priv->codenode_attributes_patterns, pattern);
				_tmp0 = FALSE;
				_tmp1 = FALSE;
				if (dot_required) {
					_tmp1 = NULL == g_utf8_strchr (pspec, (glong) (-1), (gunichar) '.');
				} else {
					_tmp1 = FALSE;
				}
				if ((_tmp1)) {
					_tmp0 = TRUE;
				} else {
					gboolean _tmp2;
					_tmp2 = FALSE;
					if (colon_required) {
						_tmp2 = NULL == g_utf8_strchr (pspec, (glong) (-1), (gunichar) ':');
					} else {
						_tmp2 = FALSE;
					}
					_tmp0 = (_tmp2);
				}
				if (_tmp0) {
					pspec = (g_free (pspec), NULL);
					continue;
				}
				if (g_pattern_match_string (pattern, codenode)) {
					gint _tmp3;
					char** _tmp4;
					char** _tmp5;
					_tmp4 = NULL;
					_tmp5 = NULL;
					return (_tmp5 = (_tmp4 = vala_gidl_parser_get_attributes (self, pspec, &_tmp3), *result_length1 = _tmp3, _tmp4), pspec = (g_free (pspec), NULL), (_pattern_it == NULL) ? NULL : (_pattern_it = (gee_collection_object_unref (_pattern_it), NULL)), (pattern_specs == NULL) ? NULL : (pattern_specs = (gee_collection_object_unref (pattern_specs), NULL)), attributes = (g_free (attributes), NULL), _tmp5);
				}
				pspec = (g_free (pspec), NULL);
			}
			(_pattern_it == NULL) ? NULL : (_pattern_it = (gee_collection_object_unref (_pattern_it), NULL));
		}
		(pattern_specs == NULL) ? NULL : (pattern_specs = (gee_collection_object_unref (pattern_specs), NULL));
	}
	if (attributes == NULL) {
		gpointer _tmp6;
		char** _tmp7;
		_tmp7 = NULL;
		return (_tmp7 = (_tmp6 = NULL, *result_length1 = 0, _tmp6), attributes = (g_free (attributes), NULL), _tmp7);
	}
	_tmp8 = NULL;
	_tmp9 = NULL;
	_tmp10 = NULL;
	return (_tmp10 = (_tmp9 = _tmp8 = g_strsplit (attributes, " ", 0), *result_length1 = g_strv_length (_tmp8), _tmp9), attributes = (g_free (attributes), NULL), _tmp10);
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, (glong) 1), (gsize) (strlen (s) - 2));
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node) {
	GIdlNode* node;
	gboolean _tmp0;
	ValaDataType* _tmp3;
	char* _tmp2;
	ValaSignal* _tmp4;
	ValaSignal* sig;
	char** _tmp9;
	gint attributes_size;
	gint attributes_length1;
	gint _tmp7;
	char* _tmp6;
	char* _tmp5;
	char** _tmp8;
	char** attributes;
	gboolean first;
	ValaSignal* _tmp39;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = (GIdlNode*) sig_node;
	_tmp0 = FALSE;
	if (sig_node->deprecated) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = sig_node->result == NULL;
	}
	if (_tmp0) {
		return NULL;
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	_tmp4 = NULL;
	sig = (_tmp4 = vala_signal_new (_tmp2 = vala_gidl_parser_fix_prop_name (self, node->name), _tmp3 = vala_gidl_parser_parse_param (self, sig_node->result, NULL), self->priv->current_source_reference), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)), _tmp2 = (g_free (_tmp2), NULL), _tmp4);
	vala_symbol_set_access ((ValaSymbol*) sig, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp9 = NULL;
	_tmp6 = NULL;
	_tmp5 = NULL;
	_tmp8 = NULL;
	attributes = (_tmp9 = (_tmp8 = vala_gidl_parser_get_attributes (self, _tmp6 = g_strdup_printf ("%s::%s", _tmp5 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), vala_symbol_get_name ((ValaSymbol*) sig)), &_tmp7), _tmp6 = (g_free (_tmp6), NULL), _tmp5 = (g_free (_tmp5), NULL), _tmp8), attributes_length1 = _tmp7, attributes_size = attributes_length1, _tmp9);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_collection_length1;
			int attr_it;
			attr_collection = attributes;
			attr_collection_length1 = attributes_length1;
			for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
				const char* _tmp18;
				char* attr;
				_tmp18 = NULL;
				attr = (_tmp18 = attr_collection[attr_it], (_tmp18 == NULL) ? NULL : g_strdup (_tmp18));
				{
					char** _tmp11;
					gint nv_size;
					gint nv_length1;
					char** _tmp10;
					char** nv;
					_tmp11 = NULL;
					_tmp10 = NULL;
					nv = (_tmp11 = _tmp10 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp10), nv_size = nv_length1, _tmp11);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp12;
						vala_signal_set_cname (sig, vala_symbol_get_name ((ValaSymbol*) sig));
						_tmp12 = NULL;
						vala_symbol_set_name ((ValaSymbol*) sig, _tmp12 = vala_gidl_parser_eval (self, nv[1]));
						_tmp12 = (g_free (_tmp12), NULL);
					} else {
						gboolean _tmp13;
						_tmp13 = FALSE;
						if (_vala_strcmp0 (nv[0], "has_emitter") == 0) {
							char* _tmp14;
							_tmp14 = NULL;
							_tmp13 = _vala_strcmp0 (_tmp14 = vala_gidl_parser_eval (self, nv[1]), "1") == 0;
							_tmp14 = (g_free (_tmp14), NULL);
						} else {
							_tmp13 = FALSE;
						}
						if (_tmp13) {
							vala_signal_set_has_emitter (sig, TRUE);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp15;
								gboolean _tmp16;
								_tmp15 = NULL;
								if ((_tmp16 = _vala_strcmp0 (_tmp15 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp15 = (g_free (_tmp15), NULL), _tmp16)) {
									ValaSignal* _tmp17;
									_tmp17 = NULL;
									return (_tmp17 = NULL, attr = (g_free (attr), NULL), nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL), (sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL)), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp17);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
	}
	vala_signal_set_is_virtual (sig, TRUE);
	first = TRUE;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = sig_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = (GIdlNodeParam*) param_it->data;
			{
				GIdlNode* param_node;
				ValaParameterDirection direction;
				ValaDataType* param_type;
				ValaFormalParameter* p;
				char** _tmp22;
				gint _tmp21;
				char* _tmp20;
				char* _tmp19;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = (GIdlNode*) param;
				direction = 0;
				param_type = vala_gidl_parser_parse_param (self, param, &direction);
				p = vala_formal_parameter_new (param_node->name, param_type, NULL);
				vala_formal_parameter_set_direction (p, direction);
				vala_signal_add_parameter (sig, p);
				_tmp22 = NULL;
				_tmp20 = NULL;
				_tmp19 = NULL;
				attributes = (_tmp22 = vala_gidl_parser_get_attributes (self, _tmp20 = g_strdup_printf ("%s::%s.%s", _tmp19 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE), vala_symbol_get_name ((ValaSymbol*) sig), param_node->name), &_tmp21), attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), attributes_length1 = _tmp21, attributes_size = attributes_length1, _tmp22);
				_tmp20 = (g_free (_tmp20), NULL);
				_tmp19 = (g_free (_tmp19), NULL);
				if (attributes != NULL) {
					char* ns_name;
					ns_name = NULL;
					{
						char** attr_collection;
						int attr_collection_length1;
						int attr_it;
						attr_collection = attributes;
						attr_collection_length1 = attributes_length1;
						for (attr_it = 0; attr_it < attributes_length1; attr_it = attr_it + 1) {
							const char* _tmp37;
							char* attr;
							_tmp37 = NULL;
							attr = (_tmp37 = attr_collection[attr_it], (_tmp37 == NULL) ? NULL : g_strdup (_tmp37));
							{
								char** _tmp24;
								gint nv_size;
								gint nv_length1;
								char** _tmp23;
								char** nv;
								_tmp24 = NULL;
								_tmp23 = NULL;
								nv = (_tmp24 = _tmp23 = g_strsplit (attr, "=", 2), nv_length1 = g_strv_length (_tmp23), nv_size = nv_length1, _tmp24);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp25;
									gboolean _tmp26;
									_tmp25 = NULL;
									if ((_tmp26 = _vala_strcmp0 (_tmp25 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp25 = (g_free (_tmp25), NULL), _tmp26)) {
										ValaDataType* _tmp27;
										_tmp27 = NULL;
										param_type = (_tmp27 = (ValaDataType*) vala_array_type_new (param_type, 1, vala_code_node_get_source_reference ((ValaCodeNode*) param_type)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp27);
										vala_formal_parameter_set_parameter_type (p, param_type);
										vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_IN);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp28;
										gboolean _tmp29;
										_tmp28 = NULL;
										if ((_tmp29 = _vala_strcmp0 (_tmp28 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp28 = (g_free (_tmp28), NULL), _tmp29)) {
											vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_OUT);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp30;
											gboolean _tmp31;
											_tmp30 = NULL;
											if ((_tmp31 = _vala_strcmp0 (_tmp30 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp30 = (g_free (_tmp30), NULL), _tmp31)) {
												vala_formal_parameter_set_direction (p, VALA_PARAMETER_DIRECTION_REF);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp32;
												gboolean _tmp33;
												_tmp32 = NULL;
												if ((_tmp33 = _vala_strcmp0 (_tmp32 = vala_gidl_parser_eval (self, nv[1]), "1") == 0, _tmp32 = (g_free (_tmp32), NULL), _tmp33)) {
													vala_data_type_set_nullable (param_type, TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "type_name") == 0) {
													ValaUnresolvedSymbol* _tmp35;
													char* _tmp34;
													_tmp35 = NULL;
													_tmp34 = NULL;
													vala_unresolved_type_set_unresolved_symbol ((VALA_UNRESOLVED_TYPE (param_type)), _tmp35 = vala_unresolved_symbol_new (NULL, _tmp34 = vala_gidl_parser_eval (self, nv[1]), NULL));
													(_tmp35 == NULL) ? NULL : (_tmp35 = (vala_code_node_unref (_tmp35), NULL));
													_tmp34 = (g_free (_tmp34), NULL);
												} else {
													if (_vala_strcmp0 (nv[0], "namespace_name") == 0) {
														char* _tmp36;
														_tmp36 = NULL;
														ns_name = (_tmp36 = vala_gidl_parser_eval (self, nv[1]), ns_name = (g_free (ns_name), NULL), _tmp36);
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, (GDestroyNotify) g_free), NULL);
							}
						}
					}
					if (ns_name != NULL) {
						ValaUnresolvedSymbol* _tmp38;
						_tmp38 = NULL;
						vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol ((VALA_UNRESOLVED_TYPE (param_type))), _tmp38 = vala_unresolved_symbol_new (NULL, ns_name, NULL));
						(_tmp38 == NULL) ? NULL : (_tmp38 = (vala_code_node_unref (_tmp38), NULL));
					}
					ns_name = (g_free (ns_name), NULL);
				}
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
				(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
			}
		}
	}
	_tmp39 = NULL;
	return (_tmp39 = sig, attributes = (_vala_array_free (attributes, attributes_length1, (GDestroyNotify) g_free), NULL), _tmp39);
}


/**
 * Code visitor parsing all GIDL files.
 */
ValaGIdlParser* vala_gidl_parser_construct (GType object_type) {
	ValaGIdlParser* self;
	self = (ValaGIdlParser*) g_type_create_instance (object_type);
	return self;
}


ValaGIdlParser* vala_gidl_parser_new (void) {
	return vala_gidl_parser_construct (VALA_TYPE_GIDL_PARSER);
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass) {
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_VISITOR_CLASS (klass)->finalize = vala_gidl_parser_finalize;
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gidl_parser_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gidl_parser_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gidl_parser_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gidl_parser_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gidl_parser_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_error_domain = vala_gidl_parser_real_visit_error_domain;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gidl_parser_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_instance_init (ValaGIdlParser * self) {
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_finalize (ValaCodeVisitor* obj) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
	(self->priv->current_source_file == NULL) ? NULL : (self->priv->current_source_file = (vala_source_file_unref (self->priv->current_source_file), NULL));
	(self->priv->current_source_reference == NULL) ? NULL : (self->priv->current_source_reference = (vala_source_reference_unref (self->priv->current_source_reference), NULL));
	(self->priv->current_namespace == NULL) ? NULL : (self->priv->current_namespace = (vala_code_node_unref (self->priv->current_namespace), NULL));
	(self->priv->current_data_type == NULL) ? NULL : (self->priv->current_data_type = (vala_code_node_unref (self->priv->current_data_type), NULL));
	(self->priv->codenode_attributes_map == NULL) ? NULL : (self->priv->codenode_attributes_map = (gee_collection_object_unref (self->priv->codenode_attributes_map), NULL));
	(self->priv->codenode_attributes_patterns == NULL) ? NULL : (self->priv->codenode_attributes_patterns = (gee_collection_object_unref (self->priv->codenode_attributes_patterns), NULL));
	(self->priv->current_type_symbol_set == NULL) ? NULL : (self->priv->current_type_symbol_set = (gee_collection_object_unref (self->priv->current_type_symbol_set), NULL));
	(self->priv->cname_type_map == NULL) ? NULL : (self->priv->cname_type_map = (gee_collection_object_unref (self->priv->cname_type_map), NULL));
	VALA_CODE_VISITOR_CLASS (vala_gidl_parser_parent_class)->finalize (obj);
}


GType vala_gidl_parser_get_type (void) {
	static GType vala_gidl_parser_type_id = 0;
	if (vala_gidl_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_instance_init, NULL };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




