/* valainterface.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valainterface.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/collection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valadatatype.h>
#include <vala/valamethod.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <vala/valaclass.h>
#include <vala/valastruct.h>
#include <vala/valaenum.h>
#include <vala/valadelegate.h>
#include <vala/valasourcereference.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valamember.h>
#include <vala/valaformalparameter.h>
#include <vala/valaobjecttype.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valavoidtype.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valacodevisitor.h>
#include <vala/valatypeparameter.h>
#include <vala/valatypesymbol.h>
#include <vala/valaattribute.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>




static char* string_substring (const char* self, glong offset, glong len);
struct _ValaInterfacePrivate {
	GeeList* prerequisites;
	GeeList* methods;
	GeeList* fields;
	GeeList* properties;
	GeeList* signals;
	GeeList* classes;
	GeeList* structs;
	GeeList* enums;
	GeeList* delegates;
	char* cname;
	char* lower_case_csuffix;
	char* type_cname;
	char* type_id;
};

#define VALA_INTERFACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE, ValaInterfacePrivate))
enum  {
	VALA_INTERFACE_DUMMY_PROPERTY
};
static GeeList* vala_interface_real_get_methods (ValaObjectTypeSymbol* base);
static GeeList* vala_interface_real_get_properties (ValaObjectTypeSymbol* base);
static GeeList* vala_interface_real_get_signals (ValaObjectTypeSymbol* base);
static char* vala_interface_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_interface_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_interface_real_is_reference_type (ValaTypeSymbol* base);
static gboolean vala_interface_real_is_reference_counting (ValaTypeSymbol* base);
static char* vala_interface_real_get_ref_function (ValaTypeSymbol* base);
static char* vala_interface_real_get_unref_function (ValaTypeSymbol* base);
static gboolean vala_interface_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t);
static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a);
static char* vala_interface_real_get_marshaller_type_name (ValaTypeSymbol* base);
static char* vala_interface_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_interface_real_get_set_value_function (ValaTypeSymbol* base);
static char* vala_interface_real_get_type_id (ValaTypeSymbol* base);
static void vala_interface_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static char* vala_interface_real_get_param_spec_function (ValaTypeSymbol* base);
static gboolean vala_interface_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_interface_parent_class = NULL;
static void vala_interface_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_warn_if_fail (offset >= 0);
	} else {
		g_warn_if_fail (offset <= string_length);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_warn_if_fail ((offset + len) <= string_length);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


/**
 * Returns a copy of the list of classes.
 *
 * @return list of classes
 */
GeeList* vala_interface_get_classes (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->classes);
}


/**
 * Returns a copy of the list of structs.
 *
 * @return list of structs
 */
GeeList* vala_interface_get_structs (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->structs);
}


/**
 * Returns a copy of the list of enums.
 *
 * @return list of enums
 */
GeeList* vala_interface_get_enums (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->enums);
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return list of delegates
 */
GeeList* vala_interface_get_delegates (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->delegates);
}


/**
 * Creates a new interface.
 *
 * @param name   type name
 * @param source reference to source code
 * @return       newly created interface
 */
ValaInterface* vala_interface_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaInterface* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaInterface*) vala_object_type_symbol_construct (object_type, name, source_reference);
	return self;
}


ValaInterface* vala_interface_new (const char* name, ValaSourceReference* source_reference) {
	return vala_interface_construct (VALA_TYPE_INTERFACE, name, source_reference);
}


/**
 * Adds the specified interface or class to the list of prerequisites of
 * this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_add_prerequisite (ValaInterface* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_collection_add ((GeeCollection*) self->priv->prerequisites, type);
	vala_code_node_set_parent_node ((ValaCodeNode*) type, (ValaCodeNode*) self);
}


/**
 * Prepends the specified interface or class to the list of
 * prerequisites of this interface.
 *
 * @param type an interface or class reference
 */
void vala_interface_prepend_prerequisite (ValaInterface* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_list_insert (self->priv->prerequisites, 0, type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeList* vala_interface_get_prerequisites (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->prerequisites);
}


/**
 * Adds the specified method as a member to this interface.
 *
 * @param m a method
 */
void vala_interface_add_method (ValaInterface* self, ValaMethod* m) {
	gboolean _tmp2;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		ValaFormalParameter* _tmp1;
		ValaObjectType* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_method_set_this_parameter (m, _tmp1 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0 = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	_tmp2 = FALSE;
	if (!(VALA_IS_VOID_TYPE (vala_method_get_return_type (m)))) {
		GeeList* _tmp3;
		_tmp3 = NULL;
		_tmp2 = gee_collection_get_size ((GeeCollection*) (_tmp3 = vala_method_get_postconditions (m))) > 0;
		(_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL));
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		ValaLocalVariable* _tmp5;
		ValaDataType* _tmp4;
		_tmp5 = NULL;
		_tmp4 = NULL;
		vala_method_set_result_var (m, _tmp5 = vala_local_variable_new (_tmp4 = vala_data_type_copy (vala_method_get_return_type (m)), "result", NULL, NULL));
		(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		(_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_result_var (m)), (ValaSymbol*) vala_method_get_result_var (m));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
static GeeList* vala_interface_real_get_methods (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


/**
 * Adds the specified field as a member to this interface. The field
 * must be private and static.
 *
 * @param f a field
 */
void vala_interface_add_field (ValaInterface* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeList* vala_interface_get_fields (ValaInterface* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
}


/**
 * Adds the specified property as a member to this interface.
 *
 * @param prop a property
 */
void vala_interface_add_property (ValaInterface* self, ValaProperty* prop) {
	ValaFormalParameter* _tmp1;
	ValaObjectType* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (prop != NULL);
	gee_collection_add ((GeeCollection*) self->priv->properties, prop);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) prop), (ValaSymbol*) prop);
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_property_set_this_parameter (prop, _tmp1 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp0 = vala_object_type_new ((ValaObjectTypeSymbol*) self)), NULL));
	(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
	(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) prop), vala_symbol_get_name ((ValaSymbol*) vala_property_get_this_parameter (prop)), (ValaSymbol*) vala_property_get_this_parameter (prop));
}


/**
 * Returns a copy of the list of properties.
 *
 * @return list of properties
 */
static GeeList* vala_interface_real_get_properties (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->properties);
}


/**
 * Adds the specified signal as a member to this interface.
 *
 * @param sig a signal
 */
void vala_interface_add_signal (ValaInterface* self, ValaSignal* sig) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (sig != NULL);
	gee_collection_add ((GeeCollection*) self->priv->signals, sig);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) sig), (ValaSymbol*) sig);
}


/**
 * Returns a copy of the list of signals.
 *
 * @return list of signals
 */
static GeeList* vala_interface_real_get_signals (ValaObjectTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->signals);
}


/**
 * Adds the specified class as an inner class.
 *
 * @param cl a class
 */
void vala_interface_add_class (ValaInterface* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	gee_collection_add ((GeeCollection*) self->priv->classes, cl);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) cl), (ValaSymbol*) cl);
}


/**
 * Adds the specified struct as an inner struct.
 *
 * @param st a struct
 */
void vala_interface_add_struct (ValaInterface* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_add ((GeeCollection*) self->priv->structs, st);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st), (ValaSymbol*) st);
}


/**
 * Adds the specified enum as an inner enum.
 *
 * @param en an enum
 */
void vala_interface_add_enum (ValaInterface* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	gee_collection_add ((GeeCollection*) self->priv->enums, en);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) en), (ValaSymbol*) en);
}


/**
 * Adds the specified delegate as an inner delegate.
 *
 * @param d a delegate
 */
void vala_interface_add_delegate (ValaInterface* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	gee_collection_add ((GeeCollection*) self->priv->delegates, d);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) d), (ValaSymbol*) d);
}


static char* vala_interface_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaInterface * self;
	const char* _tmp2;
	self = (ValaInterface*) base;
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", _tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


/**
 * Returns the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_lower_case_csuffix (ValaInterface* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_interface_get_default_lower_case_csuffix (self), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


/**
 * Returns default string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @return the suffix to be used in C code
 */
char* vala_interface_get_default_lower_case_csuffix (ValaInterface* self) {
	char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self));
	/* remove underscores in some cases to avoid conflicts of type macros*/
	if (g_str_has_prefix (result, "type_")) {
		char* _tmp0;
		_tmp0 = NULL;
		result = (_tmp0 = g_strconcat ("type", g_utf8_offset_to_pointer (result, g_utf8_strlen ("type_", -1)), NULL), result = (g_free (result), NULL), _tmp0);
	} else {
		if (g_str_has_prefix (result, "is_")) {
			char* _tmp1;
			_tmp1 = NULL;
			result = (_tmp1 = g_strconcat ("is", g_utf8_offset_to_pointer (result, g_utf8_strlen ("is_", -1)), NULL), result = (g_free (result), NULL), _tmp1);
		}
	}
	if (g_str_has_suffix (result, "_class")) {
		char* _tmp3;
		char* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		result = (_tmp3 = g_strconcat (_tmp2 = string_substring (result, (glong) 0, g_utf8_strlen (result, -1) - g_utf8_strlen ("_class", -1)), "class", NULL), result = (g_free (result), NULL), _tmp3);
		_tmp2 = (g_free (_tmp2), NULL);
	}
	return result;
}


/**
 * Sets the string to be prepended to the name of members of this
 * interface when used in C code.
 *
 * @param csuffix the suffix to be used in C code
 */
void vala_interface_set_lower_case_csuffix (ValaInterface* self, const char* csuffix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (csuffix != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->lower_case_csuffix = (_tmp1 = (_tmp0 = csuffix, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp1);
}


static char* vala_interface_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaInterface * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = (ValaInterface*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", _tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1 = vala_interface_get_lower_case_csuffix (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_interface_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaInterface * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaInterface*) base;
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s_", _tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_interface_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaInterface * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaInterface*) base;
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static void vala_interface_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_interface (visitor, self);
}


static void vala_interface_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_accept ((ValaCodeNode*) type, visitor);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _p_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_p_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_accept ((ValaCodeNode*) p, visitor);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (gee_iterator_next (_en_it)) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_accept ((ValaCodeNode*) en, visitor);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_accept ((ValaCodeNode*) prop, visitor);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_accept ((ValaCodeNode*) sig, visitor);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (gee_iterator_next (_cl_it)) {
			ValaClass* cl;
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_accept ((ValaCodeNode*) cl, visitor);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (gee_iterator_next (_st_it)) {
			ValaStruct* st;
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_accept ((ValaCodeNode*) st, visitor);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (gee_iterator_next (_d_it)) {
			ValaDelegate* d;
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_accept ((ValaCodeNode*) d, visitor);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
}


static gboolean vala_interface_real_is_reference_type (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return TRUE;
}


static gboolean vala_interface_real_is_reference_counting (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return TRUE;
}


static char* vala_interface_real_get_ref_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_prerequisite_it)) {
			ValaDataType* prerequisite;
			char* ref_func;
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			ref_func = vala_typesymbol_get_ref_function (vala_data_type_get_data_type (prerequisite));
			if (ref_func != NULL) {
				char* _tmp0;
				_tmp0 = NULL;
				return (_tmp0 = ref_func, (prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL)), (_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL)), _tmp0);
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			ref_func = (g_free (ref_func), NULL);
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	return NULL;
}


static char* vala_interface_real_get_unref_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_prerequisite_it)) {
			ValaDataType* prerequisite;
			char* unref_func;
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			unref_func = vala_typesymbol_get_unref_function (vala_data_type_get_data_type (prerequisite));
			if (unref_func != NULL) {
				char* _tmp0;
				_tmp0 = NULL;
				return (_tmp0 = unref_func, (prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL)), (_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL)), _tmp0);
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			unref_func = (g_free (unref_func), NULL);
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	return NULL;
}


static gboolean vala_interface_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (self) == t) {
		return TRUE;
	}
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_prerequisite_it)) {
			ValaDataType* prerequisite;
			gboolean _tmp1;
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			_tmp1 = FALSE;
			if (vala_data_type_get_data_type (prerequisite) != NULL) {
				_tmp1 = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (prerequisite), t);
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL)), (_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL)), _tmp2);
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	return FALSE;
}


static void vala_interface_process_ccode_attribute (ValaInterface* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "type_cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_interface_set_type_cname (self, _tmp0 = vala_attribute_get_string (a, "type_cname"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp1;
				char* filename;
				_tmp1 = NULL;
				filename = (_tmp1 = *filename_it, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "lower_case_csuffix")) {
		char* _tmp2;
		_tmp2 = NULL;
		self->priv->lower_case_csuffix = (_tmp2 = vala_attribute_get_string (a, "lower_case_csuffix"), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp2);
	}
}


/**
 * Process all associated attributes.
 */
void vala_interface_process_attributes (ValaInterface* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_interface_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


/**
 * Returns the name of the type struct as it is used in C code.
 *
 * @return the type struct name to be used in C code
 */
char* vala_interface_get_type_cname (ValaInterface* self) {
	const char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->type_cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->type_cname = (_tmp1 = g_strdup_printf ("%sIface", _tmp0 = vala_typesymbol_get_cname ((ValaTypeSymbol*) self, FALSE)), self->priv->type_cname = (g_free (self->priv->type_cname), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->type_cname, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


/**
 * Sets the name of the type struct as it is used in C code.
 *
 * @param type_cname the type struct name to be used in C code
 */
void vala_interface_set_type_cname (ValaInterface* self, const char* type_cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (type_cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->type_cname = (_tmp1 = (_tmp0 = type_cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->type_cname = (g_free (self->priv->type_cname), NULL), _tmp1);
}


static char* vala_interface_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return g_strdup ("OBJECT");
}


static char* vala_interface_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return g_strdup ("g_value_get_object");
}


static char* vala_interface_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	return g_strdup ("g_value_set_object");
}


static char* vala_interface_real_get_type_id (ValaTypeSymbol* base) {
	ValaInterface * self;
	const char* _tmp1;
	self = (ValaInterface*) base;
	if (self->priv->type_id == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->type_id = (_tmp0 = vala_typesymbol_get_upper_case_cname ((ValaTypeSymbol*) self, "TYPE_"), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->type_id, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


static void vala_interface_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->prerequisites); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get ((GeeList*) self->priv->prerequisites, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set ((GeeList*) self->priv->prerequisites, i, new_type);
				return;
			}
		}
	}
}


static char* vala_interface_real_get_param_spec_function (ValaTypeSymbol* base) {
	ValaInterface * self;
	self = (ValaInterface*) base;
	{
		GeeIterator* _prerequisite_it;
		_prerequisite_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_prerequisite_it)) {
			ValaDataType* prerequisite;
			ValaObjectType* _tmp1;
			ValaDataType* _tmp0;
			ValaObjectType* prereq;
			ValaClass* _tmp3;
			ValaObjectTypeSymbol* _tmp2;
			ValaClass* cl;
			ValaInterface* _tmp6;
			ValaObjectTypeSymbol* _tmp5;
			ValaInterface* interf;
			prerequisite = (ValaDataType*) gee_iterator_get (_prerequisite_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			prereq = (_tmp1 = (_tmp0 = prerequisite, VALA_IS_OBJECT_TYPE (_tmp0) ? ((ValaObjectType*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			_tmp3 = NULL;
			_tmp2 = NULL;
			cl = (_tmp3 = (_tmp2 = vala_object_type_get_type_symbol (prereq), VALA_IS_CLASS (_tmp2) ? ((ValaClass*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
			if (cl != NULL) {
				char* _tmp4;
				_tmp4 = NULL;
				return (_tmp4 = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) cl), (prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL)), (prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL)), _tmp4);
			}
			_tmp6 = NULL;
			_tmp5 = NULL;
			interf = (_tmp6 = (_tmp5 = vala_object_type_get_type_symbol (prereq), VALA_IS_INTERFACE (_tmp5) ? ((ValaInterface*) _tmp5) : NULL), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6));
			if (interf != NULL) {
				char* param_spec_function;
				param_spec_function = vala_typesymbol_get_param_spec_function ((ValaTypeSymbol*) interf);
				if (param_spec_function != NULL) {
					char* _tmp7;
					_tmp7 = NULL;
					return (_tmp7 = param_spec_function, (prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL)), (prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL)), (cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL)), (interf == NULL) ? NULL : (interf = (vala_code_node_unref (interf), NULL)), (_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL)), _tmp7);
				}
				param_spec_function = (g_free (param_spec_function), NULL);
			}
			(prerequisite == NULL) ? NULL : (prerequisite = (vala_code_node_unref (prerequisite), NULL));
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			(interf == NULL) ? NULL : (interf = (vala_code_node_unref (interf), NULL));
		}
		(_prerequisite_it == NULL) ? NULL : (_prerequisite_it = (gee_collection_object_unref (_prerequisite_it), NULL));
	}
	return NULL;
}


static gboolean vala_interface_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaInterface * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	ValaClass* prereq_class;
	gboolean _tmp21;
	self = (ValaInterface*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_interface_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	{
		GeeList* _tmp3;
		GeeIterator* _tmp4;
		GeeIterator* _prerequisite_reference_it;
		_tmp3 = NULL;
		_tmp4 = NULL;
		_prerequisite_reference_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_interface_get_prerequisites (self))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
		while (gee_iterator_next (_prerequisite_reference_it)) {
			ValaDataType* prerequisite_reference;
			prerequisite_reference = (ValaDataType*) gee_iterator_get (_prerequisite_reference_it);
			/* check whether prerequisite is at least as accessible as the interface*/
			if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, prerequisite_reference)) {
				char* _tmp7;
				char* _tmp6;
				char* _tmp5;
				gboolean _tmp8;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp7 = g_strdup_printf ("prerequisite `%s` is less accessible than interface `%s`", _tmp5 = vala_code_node_to_string ((ValaCodeNode*) prerequisite_reference), _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) self)));
				_tmp7 = (g_free (_tmp7), NULL);
				_tmp6 = (g_free (_tmp6), NULL);
				_tmp5 = (g_free (_tmp5), NULL);
				return (_tmp8 = FALSE, (prerequisite_reference == NULL) ? NULL : (prerequisite_reference = (vala_code_node_unref (prerequisite_reference), NULL)), (_prerequisite_reference_it == NULL) ? NULL : (_prerequisite_reference_it = (gee_collection_object_unref (_prerequisite_reference_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp8);
			}
			vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), prerequisite_reference, VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			(prerequisite_reference == NULL) ? NULL : (prerequisite_reference = (vala_code_node_unref (prerequisite_reference), NULL));
		}
		(_prerequisite_reference_it == NULL) ? NULL : (_prerequisite_reference_it = (gee_collection_object_unref (_prerequisite_reference_it), NULL));
	}
	/* check prerequisites */
	prereq_class = NULL;
	{
		GeeList* _tmp9;
		GeeIterator* _tmp10;
		GeeIterator* _prereq_it;
		_tmp9 = NULL;
		_tmp10 = NULL;
		_prereq_it = (_tmp10 = gee_iterable_iterator ((GeeIterable*) (_tmp9 = vala_interface_get_prerequisites (self))), (_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL)), _tmp10);
		while (gee_iterator_next (_prereq_it)) {
			ValaDataType* prereq;
			ValaTypeSymbol* _tmp11;
			ValaTypeSymbol* class_or_interface;
			prereq = (ValaDataType*) gee_iterator_get (_prereq_it);
			_tmp11 = NULL;
			class_or_interface = (_tmp11 = vala_data_type_get_data_type (prereq), (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11));
			/* skip on previous errors */
			if (class_or_interface == NULL) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
				(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
				continue;
			}
			/* interfaces are not allowed to have multiple instantiable prerequisites */
			if (VALA_IS_CLASS (class_or_interface)) {
				ValaClass* _tmp18;
				ValaClass* _tmp17;
				if (prereq_class != NULL) {
					char* _tmp15;
					char* _tmp14;
					char* _tmp13;
					char* _tmp12;
					gboolean _tmp16;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					_tmp15 = NULL;
					_tmp14 = NULL;
					_tmp13 = NULL;
					_tmp12 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp15 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", _tmp12 = vala_symbol_get_full_name ((ValaSymbol*) self), _tmp13 = vala_symbol_get_full_name ((ValaSymbol*) class_or_interface), _tmp14 = vala_symbol_get_full_name ((ValaSymbol*) prereq_class)));
					_tmp15 = (g_free (_tmp15), NULL);
					_tmp14 = (g_free (_tmp14), NULL);
					_tmp13 = (g_free (_tmp13), NULL);
					_tmp12 = (g_free (_tmp12), NULL);
					return (_tmp16 = FALSE, (prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL)), (class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL)), (_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL)), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL)), _tmp16);
				}
				_tmp18 = NULL;
				_tmp17 = NULL;
				prereq_class = (_tmp18 = (_tmp17 = VALA_CLASS (class_or_interface), (_tmp17 == NULL) ? NULL : vala_code_node_ref (_tmp17)), (prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL)), _tmp18);
			}
			(prereq == NULL) ? NULL : (prereq = (vala_code_node_unref (prereq), NULL));
			(class_or_interface == NULL) ? NULL : (class_or_interface = (vala_code_node_unref (class_or_interface), NULL));
		}
		(_prereq_it == NULL) ? NULL : (_prereq_it = (gee_collection_object_unref (_prereq_it), NULL));
	}
	{
		GeeIterator* _type_it;
		_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->prerequisites);
		while (gee_iterator_next (_type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (_type_it);
			vala_code_node_check ((ValaCodeNode*) type, analyzer);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(_type_it == NULL) ? NULL : (_type_it = (gee_collection_object_unref (_type_it), NULL));
	}
	{
		GeeList* _tmp19;
		GeeIterator* _tmp20;
		GeeIterator* _p_it;
		_tmp19 = NULL;
		_tmp20 = NULL;
		_p_it = (_tmp20 = gee_iterable_iterator ((GeeIterable*) (_tmp19 = vala_object_type_symbol_get_type_parameters ((ValaObjectTypeSymbol*) self))), (_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL)), _tmp20);
		while (gee_iterator_next (_p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (_p_it);
			vala_code_node_check ((ValaCodeNode*) p, analyzer);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(_p_it == NULL) ? NULL : (_p_it = (gee_collection_object_unref (_p_it), NULL));
	}
	{
		GeeIterator* _en_it;
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		while (gee_iterator_next (_en_it)) {
			ValaEnum* en;
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_check ((ValaCodeNode*) en, analyzer);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_check ((ValaCodeNode*) f, analyzer);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _prop_it;
		_prop_it = gee_iterable_iterator ((GeeIterable*) self->priv->properties);
		while (gee_iterator_next (_prop_it)) {
			ValaProperty* prop;
			prop = (ValaProperty*) gee_iterator_get (_prop_it);
			vala_code_node_check ((ValaCodeNode*) prop, analyzer);
			(prop == NULL) ? NULL : (prop = (vala_code_node_unref (prop), NULL));
		}
		(_prop_it == NULL) ? NULL : (_prop_it = (gee_collection_object_unref (_prop_it), NULL));
	}
	{
		GeeIterator* _sig_it;
		_sig_it = gee_iterable_iterator ((GeeIterable*) self->priv->signals);
		while (gee_iterator_next (_sig_it)) {
			ValaSignal* sig;
			sig = (ValaSignal*) gee_iterator_get (_sig_it);
			vala_code_node_check ((ValaCodeNode*) sig, analyzer);
			(sig == NULL) ? NULL : (sig = (vala_code_node_unref (sig), NULL));
		}
		(_sig_it == NULL) ? NULL : (_sig_it = (gee_collection_object_unref (_sig_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (gee_iterator_next (_cl_it)) {
			ValaClass* cl;
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_check ((ValaCodeNode*) cl, analyzer);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (gee_iterator_next (_st_it)) {
			ValaStruct* st;
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_check ((ValaCodeNode*) st, analyzer);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (gee_iterator_next (_d_it)) {
			ValaDelegate* d;
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_check ((ValaCodeNode*) d, analyzer);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp21 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (prereq_class == NULL) ? NULL : (prereq_class = (vala_code_node_unref (prereq_class), NULL)), _tmp21);
}


static void vala_interface_class_init (ValaInterfaceClass * klass) {
	vala_interface_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_interface_finalize;
	g_type_class_add_private (klass, sizeof (ValaInterfacePrivate));
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_methods = vala_interface_real_get_methods;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_properties = vala_interface_real_get_properties;
	VALA_OBJECT_TYPE_SYMBOL_CLASS (klass)->get_signals = vala_interface_real_get_signals;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_interface_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_interface_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_interface_real_get_lower_case_cprefix;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_interface_real_get_upper_case_cname;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_interface_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_interface_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_interface_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_counting = vala_interface_real_is_reference_counting;
	VALA_TYPESYMBOL_CLASS (klass)->get_ref_function = vala_interface_real_get_ref_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_unref_function = vala_interface_real_get_unref_function;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_interface_real_is_subtype_of;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_interface_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_interface_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_interface_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_interface_real_get_type_id;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_interface_real_replace_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_param_spec_function = vala_interface_real_get_param_spec_function;
	VALA_CODE_NODE_CLASS (klass)->check = vala_interface_real_check;
}


static void vala_interface_instance_init (ValaInterface * self) {
	self->priv = VALA_INTERFACE_GET_PRIVATE (self);
	self->priv->prerequisites = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->properties = (GeeList*) gee_array_list_new (VALA_TYPE_PROPERTY, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->signals = (GeeList*) gee_array_list_new (VALA_TYPE_SIGNAL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->classes = (GeeList*) gee_array_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->structs = (GeeList*) gee_array_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->enums = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->delegates = (GeeList*) gee_array_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_interface_finalize (ValaCodeNode* obj) {
	ValaInterface * self;
	self = VALA_INTERFACE (obj);
	(self->priv->prerequisites == NULL) ? NULL : (self->priv->prerequisites = (gee_collection_object_unref (self->priv->prerequisites), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->properties == NULL) ? NULL : (self->priv->properties = (gee_collection_object_unref (self->priv->properties), NULL));
	(self->priv->signals == NULL) ? NULL : (self->priv->signals = (gee_collection_object_unref (self->priv->signals), NULL));
	(self->priv->classes == NULL) ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL));
	(self->priv->structs == NULL) ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL));
	(self->priv->enums == NULL) ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL));
	(self->priv->delegates == NULL) ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->type_cname = (g_free (self->priv->type_cname), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	VALA_CODE_NODE_CLASS (vala_interface_parent_class)->finalize (obj);
}


GType vala_interface_get_type (void) {
	static GType vala_interface_type_id = 0;
	if (vala_interface_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterface), 0, (GInstanceInitFunc) vala_interface_instance_init, NULL };
		vala_interface_type_id = g_type_register_static (VALA_TYPE_OBJECT_TYPE_SYMBOL, "ValaInterface", &g_define_type_info, 0);
	}
	return vala_interface_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




