/* valaforeachstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaforeachstatement.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/list.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valaarraytype.h>
#include <vala/valasourcefile.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valamethod.h>
#include <vala/valavoidtype.h>
#include <vala/valamethodcall.h>
#include <vala/valamemberaccess.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valastatement.h>
#include <vala/valawhilestatement.h>




struct _ValaForeachStatementPrivate {
	char* _variable_name;
	gboolean _use_iterator;
	ValaLocalVariable* _element_variable;
	ValaLocalVariable* _collection_variable;
	ValaLocalVariable* _iterator_variable;
	ValaExpression* _collection;
	ValaBlock* _body;
	ValaDataType* _data_type;
};

#define VALA_FOREACH_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FOREACH_STATEMENT, ValaForeachStatementPrivate))
enum  {
	VALA_FOREACH_STATEMENT_DUMMY_PROPERTY
};
static void vala_foreach_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_foreach_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_foreach_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_foreach_statement_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_foreach_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gboolean vala_foreach_statement_check_with_iterator (ValaForeachStatement* self, ValaSemanticAnalyzer* analyzer, ValaDataType* collection_type);
static gboolean vala_foreach_statement_check_without_iterator (ValaForeachStatement* self, ValaSemanticAnalyzer* analyzer, ValaDataType* collection_type, ValaDataType* element_type);
static void vala_foreach_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_foreach_statement_set_use_iterator (ValaForeachStatement* self, gboolean value);
static gpointer vala_foreach_statement_parent_class = NULL;
static void vala_foreach_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new foreach statement.
 *
 * @param type   element type
 * @param id     element variable name
 * @param col    loop body
 * @param source reference to source code
 * @return       newly created foreach statement
 */
ValaForeachStatement* vala_foreach_statement_construct (GType object_type, ValaDataType* type_reference, const char* variable_name, ValaExpression* collection, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaForeachStatement* self;
	g_return_val_if_fail (variable_name != NULL, NULL);
	g_return_val_if_fail (collection != NULL, NULL);
	g_return_val_if_fail (body != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaForeachStatement*) vala_block_construct (object_type, source_reference);
	vala_foreach_statement_set_variable_name (self, variable_name);
	vala_foreach_statement_set_collection (self, collection);
	vala_foreach_statement_set_body (self, body);
	vala_foreach_statement_set_type_reference (self, type_reference);
	return self;
}


ValaForeachStatement* vala_foreach_statement_new (ValaDataType* type_reference, const char* variable_name, ValaExpression* collection, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_foreach_statement_construct (VALA_TYPE_FOREACH_STATEMENT, type_reference, variable_name, collection, body, source_reference);
}


static void vala_foreach_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForeachStatement * self;
	self = (ValaForeachStatement*) base;
	g_return_if_fail (visitor != NULL);
	if (self->priv->_use_iterator) {
		VALA_CODE_NODE_CLASS (vala_foreach_statement_parent_class)->accept ((ValaCodeNode*) VALA_BLOCK (self), visitor);
		return;
	}
	vala_code_visitor_visit_foreach_statement (visitor, self);
}


static void vala_foreach_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaForeachStatement * self;
	self = (ValaForeachStatement*) base;
	g_return_if_fail (visitor != NULL);
	if (self->priv->_use_iterator) {
		VALA_CODE_NODE_CLASS (vala_foreach_statement_parent_class)->accept_children ((ValaCodeNode*) VALA_BLOCK (self), visitor);
		return;
	}
	vala_code_node_accept ((ValaCodeNode*) vala_foreach_statement_get_collection (self), visitor);
	vala_code_visitor_visit_end_full_expression (visitor, vala_foreach_statement_get_collection (self));
	if (vala_foreach_statement_get_type_reference (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_foreach_statement_get_type_reference (self), visitor);
	}
	vala_code_node_accept ((ValaCodeNode*) vala_foreach_statement_get_body (self), visitor);
}


static void vala_foreach_statement_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaForeachStatement * self;
	self = (ValaForeachStatement*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_foreach_statement_get_collection (self) == old_node) {
		vala_foreach_statement_set_collection (self, new_node);
	}
}


static void vala_foreach_statement_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaForeachStatement * self;
	self = (ValaForeachStatement*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_foreach_statement_get_type_reference (self) == old_type) {
		vala_foreach_statement_set_type_reference (self, new_type);
	}
}


static gboolean vala_foreach_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaForeachStatement * self;
	ValaDataType* collection_type;
	ValaDataType* _tmp3;
	self = (ValaForeachStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	/* analyze collection expression first, used for type inference*/
	if (!vala_code_node_check ((ValaCodeNode*) vala_foreach_statement_get_collection (self), analyzer)) {
		/* ignore inner error*/
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	} else {
		if (vala_expression_get_value_type (vala_foreach_statement_get_collection (self)) == NULL) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), "invalid collection expression");
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			return FALSE;
		}
	}
	collection_type = vala_data_type_copy (vala_expression_get_value_type (vala_foreach_statement_get_collection (self)));
	_tmp3 = NULL;
	vala_expression_set_target_type (vala_foreach_statement_get_collection (self), _tmp3 = vala_data_type_copy (collection_type));
	(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL));
	if (vala_data_type_is_array (collection_type)) {
		ValaArrayType* _tmp4;
		ValaArrayType* array_type;
		gboolean _tmp5;
		_tmp4 = NULL;
		array_type = (_tmp4 = VALA_ARRAY_TYPE (collection_type), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4));
		return (_tmp5 = vala_foreach_statement_check_without_iterator (self, analyzer, collection_type, vala_array_type_get_element_type (array_type)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), (collection_type == NULL) ? NULL : (collection_type = (vala_code_node_unref (collection_type), NULL)), _tmp5);
	} else {
		gboolean _tmp6;
		_tmp6 = FALSE;
		if (vala_data_type_compatible (collection_type, analyzer->glist_type)) {
			_tmp6 = TRUE;
		} else {
			_tmp6 = vala_data_type_compatible (collection_type, analyzer->gslist_type);
		}
		if (_tmp6) {
			GeeList* _tmp7;
			gboolean _tmp8;
			ValaDataType* _tmp11;
			GeeList* _tmp10;
			gboolean _tmp12;
			gboolean _tmp13;
			_tmp7 = NULL;
			if ((_tmp8 = gee_collection_get_size ((GeeCollection*) (_tmp7 = vala_data_type_get_type_arguments (collection_type))) != 1, (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8)) {
				gboolean _tmp9;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), "missing type argument for collection");
				return (_tmp9 = FALSE, (collection_type == NULL) ? NULL : (collection_type = (vala_code_node_unref (collection_type), NULL)), _tmp9);
			}
			_tmp11 = NULL;
			_tmp10 = NULL;
			return (_tmp13 = (_tmp12 = vala_foreach_statement_check_without_iterator (self, analyzer, collection_type, _tmp11 = (ValaDataType*) gee_list_get (_tmp10 = vala_data_type_get_type_arguments (collection_type), 0)), (_tmp11 == NULL) ? NULL : (_tmp11 = (vala_code_node_unref (_tmp11), NULL)), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp12), (collection_type == NULL) ? NULL : (collection_type = (vala_code_node_unref (collection_type), NULL)), _tmp13);
		} else {
			gboolean _tmp14;
			return (_tmp14 = vala_foreach_statement_check_with_iterator (self, analyzer, collection_type), (collection_type == NULL) ? NULL : (collection_type = (vala_code_node_unref (collection_type), NULL)), _tmp14);
		}
	}
	(collection_type == NULL) ? NULL : (collection_type = (vala_code_node_unref (collection_type), NULL));
}


static gboolean vala_foreach_statement_check_with_iterator (ValaForeachStatement* self, ValaSemanticAnalyzer* analyzer, ValaDataType* collection_type) {
	ValaSymbol* _tmp0;
	ValaMethod* iterator_method;
	GeeList* _tmp4;
	gboolean _tmp5;
	ValaDataType* iterator_type;
	ValaSymbol* _tmp12;
	ValaMethod* next_method;
	GeeList* _tmp16;
	gboolean _tmp17;
	ValaSymbol* _tmp24;
	ValaMethod* get_method;
	GeeList* _tmp28;
	gboolean _tmp29;
	ValaDataType* element_type;
	ValaMemberAccess* _tmp44;
	ValaMethodCall* _tmp45;
	ValaMethodCall* iterator_call;
	ValaDeclarationStatement* _tmp48;
	ValaLocalVariable* _tmp47;
	char* _tmp46;
	ValaMemberAccess* _tmp51;
	ValaMemberAccess* _tmp50;
	char* _tmp49;
	ValaMethodCall* _tmp52;
	ValaMethodCall* next_call;
	ValaWhileStatement* loop;
	ValaMemberAccess* _tmp55;
	ValaMemberAccess* _tmp54;
	char* _tmp53;
	ValaMethodCall* _tmp56;
	ValaMethodCall* get_call;
	ValaDeclarationStatement* _tmp58;
	ValaLocalVariable* _tmp57;
	gboolean _tmp59;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (analyzer != NULL, FALSE);
	g_return_val_if_fail (collection_type != NULL, FALSE);
	vala_foreach_statement_set_use_iterator (self, TRUE);
	_tmp0 = NULL;
	iterator_method = (_tmp0 = vala_data_type_get_member (collection_type, "iterator"), VALA_IS_METHOD (_tmp0) ? ((ValaMethod*) _tmp0) : NULL);
	if (iterator_method == NULL) {
		char* _tmp2;
		char* _tmp1;
		gboolean _tmp3;
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp2 = g_strdup_printf ("`%s' does not have an `iterator' method", _tmp1 = vala_code_node_to_string ((ValaCodeNode*) collection_type)));
		_tmp2 = (g_free (_tmp2), NULL);
		_tmp1 = (g_free (_tmp1), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp3 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), _tmp3);
	}
	_tmp4 = NULL;
	if ((_tmp5 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_method_get_parameters (iterator_method))) != 0, (_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL)), _tmp5)) {
		char* _tmp7;
		char* _tmp6;
		gboolean _tmp8;
		_tmp7 = NULL;
		_tmp6 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp7 = g_strdup_printf ("`%s' must not have any parameters", _tmp6 = vala_symbol_get_full_name ((ValaSymbol*) iterator_method)));
		_tmp7 = (g_free (_tmp7), NULL);
		_tmp6 = (g_free (_tmp6), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp8 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), _tmp8);
	}
	iterator_type = vala_data_type_get_actual_type (vala_method_get_return_type (iterator_method), collection_type, (ValaCodeNode*) self);
	if (VALA_IS_VOID_TYPE (iterator_type)) {
		char* _tmp10;
		char* _tmp9;
		gboolean _tmp11;
		_tmp10 = NULL;
		_tmp9 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp10 = g_strdup_printf ("`%s' must return an iterator", _tmp9 = vala_symbol_get_full_name ((ValaSymbol*) iterator_method)));
		_tmp10 = (g_free (_tmp10), NULL);
		_tmp9 = (g_free (_tmp9), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp11 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), _tmp11);
	}
	_tmp12 = NULL;
	next_method = (_tmp12 = vala_data_type_get_member (iterator_type, "next"), VALA_IS_METHOD (_tmp12) ? ((ValaMethod*) _tmp12) : NULL);
	if (next_method == NULL) {
		char* _tmp14;
		char* _tmp13;
		gboolean _tmp15;
		_tmp14 = NULL;
		_tmp13 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp14 = g_strdup_printf ("`%s' does not have a `next' method", _tmp13 = vala_code_node_to_string ((ValaCodeNode*) iterator_type)));
		_tmp14 = (g_free (_tmp14), NULL);
		_tmp13 = (g_free (_tmp13), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp15 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), _tmp15);
	}
	_tmp16 = NULL;
	if ((_tmp17 = gee_collection_get_size ((GeeCollection*) (_tmp16 = vala_method_get_parameters (next_method))) != 0, (_tmp16 == NULL) ? NULL : (_tmp16 = (gee_collection_object_unref (_tmp16), NULL)), _tmp17)) {
		char* _tmp19;
		char* _tmp18;
		gboolean _tmp20;
		_tmp19 = NULL;
		_tmp18 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp19 = g_strdup_printf ("`%s' must not have any parameters", _tmp18 = vala_symbol_get_full_name ((ValaSymbol*) next_method)));
		_tmp19 = (g_free (_tmp19), NULL);
		_tmp18 = (g_free (_tmp18), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp20 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), _tmp20);
	}
	if (!vala_data_type_compatible (vala_method_get_return_type (next_method), analyzer->bool_type)) {
		char* _tmp22;
		char* _tmp21;
		gboolean _tmp23;
		_tmp22 = NULL;
		_tmp21 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp22 = g_strdup_printf ("`%s' must return a boolean value", _tmp21 = vala_symbol_get_full_name ((ValaSymbol*) next_method)));
		_tmp22 = (g_free (_tmp22), NULL);
		_tmp21 = (g_free (_tmp21), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp23 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), _tmp23);
	}
	_tmp24 = NULL;
	get_method = (_tmp24 = vala_data_type_get_member (iterator_type, "get"), VALA_IS_METHOD (_tmp24) ? ((ValaMethod*) _tmp24) : NULL);
	if (get_method == NULL) {
		char* _tmp26;
		char* _tmp25;
		gboolean _tmp27;
		_tmp26 = NULL;
		_tmp25 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp26 = g_strdup_printf ("`%s' does not have a `get' method", _tmp25 = vala_code_node_to_string ((ValaCodeNode*) iterator_type)));
		_tmp26 = (g_free (_tmp26), NULL);
		_tmp25 = (g_free (_tmp25), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp27 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), _tmp27);
	}
	_tmp28 = NULL;
	if ((_tmp29 = gee_collection_get_size ((GeeCollection*) (_tmp28 = vala_method_get_parameters (get_method))) != 0, (_tmp28 == NULL) ? NULL : (_tmp28 = (gee_collection_object_unref (_tmp28), NULL)), _tmp29)) {
		char* _tmp31;
		char* _tmp30;
		gboolean _tmp32;
		_tmp31 = NULL;
		_tmp30 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp31 = g_strdup_printf ("`%s' must not have any parameters", _tmp30 = vala_symbol_get_full_name ((ValaSymbol*) get_method)));
		_tmp31 = (g_free (_tmp31), NULL);
		_tmp30 = (g_free (_tmp30), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp32 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), _tmp32);
	}
	element_type = vala_data_type_get_actual_type (vala_method_get_return_type (get_method), iterator_type, (ValaCodeNode*) self);
	if (VALA_IS_VOID_TYPE (element_type)) {
		char* _tmp34;
		char* _tmp33;
		gboolean _tmp35;
		_tmp34 = NULL;
		_tmp33 = NULL;
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) vala_foreach_statement_get_collection (self)), _tmp34 = g_strdup_printf ("`%s' must return an element", _tmp33 = vala_symbol_get_full_name ((ValaSymbol*) get_method)));
		_tmp34 = (g_free (_tmp34), NULL);
		_tmp33 = (g_free (_tmp33), NULL);
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return (_tmp35 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp35);
	}
	/* analyze element type*/
	if (vala_foreach_statement_get_type_reference (self) == NULL) {
		ValaDataType* _tmp36;
		/* var type*/
		_tmp36 = NULL;
		vala_foreach_statement_set_type_reference (self, _tmp36 = vala_data_type_copy (element_type));
		(_tmp36 == NULL) ? NULL : (_tmp36 = (vala_code_node_unref (_tmp36), NULL));
	} else {
		if (!vala_data_type_compatible (element_type, vala_foreach_statement_get_type_reference (self))) {
			char* _tmp39;
			char* _tmp38;
			char* _tmp37;
			gboolean _tmp40;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp39 = NULL;
			_tmp38 = NULL;
			_tmp37 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp39 = g_strdup_printf ("Foreach: Cannot convert from `%s' to `%s'", _tmp37 = vala_code_node_to_string ((ValaCodeNode*) element_type), _tmp38 = vala_code_node_to_string ((ValaCodeNode*) vala_foreach_statement_get_type_reference (self))));
			_tmp39 = (g_free (_tmp39), NULL);
			_tmp38 = (g_free (_tmp38), NULL);
			_tmp37 = (g_free (_tmp37), NULL);
			return (_tmp40 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp40);
		} else {
			gboolean _tmp41;
			gboolean _tmp42;
			_tmp41 = FALSE;
			_tmp42 = FALSE;
			if (vala_data_type_is_disposable (element_type)) {
				_tmp42 = vala_data_type_get_value_owned (element_type);
			} else {
				_tmp42 = FALSE;
			}
			if (_tmp42) {
				_tmp41 = !vala_data_type_get_value_owned (vala_foreach_statement_get_type_reference (self));
			} else {
				_tmp41 = FALSE;
			}
			if (_tmp41) {
				gboolean _tmp43;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Foreach: Invalid assignment from owned expression to unowned variable");
				return (_tmp43 = FALSE, (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), _tmp43);
			}
		}
	}
	_tmp44 = NULL;
	_tmp45 = NULL;
	iterator_call = (_tmp45 = vala_method_call_new ((ValaExpression*) (_tmp44 = vala_member_access_new (vala_foreach_statement_get_collection (self), "iterator", NULL)), NULL), (_tmp44 == NULL) ? NULL : (_tmp44 = (vala_code_node_unref (_tmp44), NULL)), _tmp45);
	_tmp48 = NULL;
	_tmp47 = NULL;
	_tmp46 = NULL;
	vala_block_add_statement ((ValaBlock*) self, (ValaStatement*) (_tmp48 = vala_declaration_statement_new ((ValaSymbol*) (_tmp47 = vala_local_variable_new (iterator_type, _tmp46 = g_strdup_printf ("_%s_it", self->priv->_variable_name), (ValaExpression*) iterator_call, vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp48 == NULL) ? NULL : (_tmp48 = (vala_code_node_unref (_tmp48), NULL));
	(_tmp47 == NULL) ? NULL : (_tmp47 = (vala_code_node_unref (_tmp47), NULL));
	_tmp46 = (g_free (_tmp46), NULL);
	_tmp51 = NULL;
	_tmp50 = NULL;
	_tmp49 = NULL;
	_tmp52 = NULL;
	next_call = (_tmp52 = vala_method_call_new ((ValaExpression*) (_tmp51 = vala_member_access_new ((ValaExpression*) (_tmp50 = vala_member_access_new_simple (_tmp49 = g_strdup_printf ("_%s_it", self->priv->_variable_name), vala_code_node_get_source_reference ((ValaCodeNode*) self))), "next", vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp51 == NULL) ? NULL : (_tmp51 = (vala_code_node_unref (_tmp51), NULL)), (_tmp50 == NULL) ? NULL : (_tmp50 = (vala_code_node_unref (_tmp50), NULL)), _tmp49 = (g_free (_tmp49), NULL), _tmp52);
	loop = vala_while_statement_new ((ValaExpression*) next_call, vala_foreach_statement_get_body (self), vala_code_node_get_source_reference ((ValaCodeNode*) self));
	vala_block_add_statement ((ValaBlock*) self, (ValaStatement*) loop);
	_tmp55 = NULL;
	_tmp54 = NULL;
	_tmp53 = NULL;
	_tmp56 = NULL;
	get_call = (_tmp56 = vala_method_call_new ((ValaExpression*) (_tmp55 = vala_member_access_new ((ValaExpression*) (_tmp54 = vala_member_access_new_simple (_tmp53 = g_strdup_printf ("_%s_it", self->priv->_variable_name), vala_code_node_get_source_reference ((ValaCodeNode*) self))), "get", vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp55 == NULL) ? NULL : (_tmp55 = (vala_code_node_unref (_tmp55), NULL)), (_tmp54 == NULL) ? NULL : (_tmp54 = (vala_code_node_unref (_tmp54), NULL)), _tmp53 = (g_free (_tmp53), NULL), _tmp56);
	_tmp58 = NULL;
	_tmp57 = NULL;
	vala_block_insert_statement (vala_foreach_statement_get_body (self), 0, (ValaStatement*) (_tmp58 = vala_declaration_statement_new ((ValaSymbol*) (_tmp57 = vala_local_variable_new (vala_foreach_statement_get_type_reference (self), self->priv->_variable_name, (ValaExpression*) get_call, vala_code_node_get_source_reference ((ValaCodeNode*) self))), vala_code_node_get_source_reference ((ValaCodeNode*) self))));
	(_tmp58 == NULL) ? NULL : (_tmp58 = (vala_code_node_unref (_tmp58), NULL));
	(_tmp57 == NULL) ? NULL : (_tmp57 = (vala_code_node_unref (_tmp57), NULL));
	vala_code_node_set_checked ((ValaCodeNode*) self, FALSE);
	return (_tmp59 = VALA_CODE_NODE_CLASS (vala_foreach_statement_parent_class)->check ((ValaCodeNode*) VALA_BLOCK (self), analyzer), (iterator_method == NULL) ? NULL : (iterator_method = (vala_code_node_unref (iterator_method), NULL)), (iterator_type == NULL) ? NULL : (iterator_type = (vala_code_node_unref (iterator_type), NULL)), (next_method == NULL) ? NULL : (next_method = (vala_code_node_unref (next_method), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), (element_type == NULL) ? NULL : (element_type = (vala_code_node_unref (element_type), NULL)), (iterator_call == NULL) ? NULL : (iterator_call = (vala_code_node_unref (iterator_call), NULL)), (next_call == NULL) ? NULL : (next_call = (vala_code_node_unref (next_call), NULL)), (loop == NULL) ? NULL : (loop = (vala_code_node_unref (loop), NULL)), (get_call == NULL) ? NULL : (get_call = (vala_code_node_unref (get_call), NULL)), _tmp59);
}


static gboolean vala_foreach_statement_check_without_iterator (ValaForeachStatement* self, ValaSemanticAnalyzer* analyzer, ValaDataType* collection_type, ValaDataType* element_type) {
	ValaLocalVariable* _tmp5;
	ValaLocalVariable* _tmp9;
	char* _tmp8;
	GeeList* _tmp10;
	GeeList* _tmp11;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (analyzer != NULL, FALSE);
	g_return_val_if_fail (collection_type != NULL, FALSE);
	g_return_val_if_fail (element_type != NULL, FALSE);
	/* analyze element type*/
	if (vala_foreach_statement_get_type_reference (self) == NULL) {
		ValaDataType* _tmp0;
		/* var type*/
		_tmp0 = NULL;
		vala_foreach_statement_set_type_reference (self, _tmp0 = vala_data_type_copy (element_type));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL));
	} else {
		if (!vala_data_type_compatible (element_type, vala_foreach_statement_get_type_reference (self))) {
			char* _tmp3;
			char* _tmp2;
			char* _tmp1;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp3 = NULL;
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp3 = g_strdup_printf ("Foreach: Cannot convert from `%s' to `%s'", _tmp1 = vala_code_node_to_string ((ValaCodeNode*) element_type), _tmp2 = vala_code_node_to_string ((ValaCodeNode*) vala_foreach_statement_get_type_reference (self))));
			_tmp3 = (g_free (_tmp3), NULL);
			_tmp2 = (g_free (_tmp2), NULL);
			_tmp1 = (g_free (_tmp1), NULL);
			return FALSE;
		}
	}
	vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_foreach_statement_get_type_reference (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	_tmp5 = NULL;
	vala_foreach_statement_set_element_variable (self, _tmp5 = vala_local_variable_new (vala_foreach_statement_get_type_reference (self), self->priv->_variable_name, NULL, NULL));
	(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_foreach_statement_get_body (self)), self->priv->_variable_name, (ValaSymbol*) self->priv->_element_variable);
	vala_block_add_local_variable (vala_foreach_statement_get_body (self), self->priv->_element_variable);
	vala_symbol_set_active ((ValaSymbol*) self->priv->_element_variable, TRUE);
	vala_code_node_set_checked ((ValaCodeNode*) self->priv->_element_variable, TRUE);
	/* analyze body*/
	vala_symbol_set_owner ((ValaSymbol*) self, vala_symbol_get_scope (vala_semantic_analyzer_get_current_symbol (analyzer)));
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	vala_code_node_check ((ValaCodeNode*) vala_foreach_statement_get_body (self), analyzer);
	{
		GeeList* _tmp6;
		GeeIterator* _tmp7;
		GeeIterator* _local_it;
		_tmp6 = NULL;
		_tmp7 = NULL;
		_local_it = (_tmp7 = gee_iterable_iterator ((GeeIterable*) (_tmp6 = vala_block_get_local_variables ((ValaBlock*) self))), (_tmp6 == NULL) ? NULL : (_tmp6 = (gee_collection_object_unref (_tmp6), NULL)), _tmp7);
		while (gee_iterator_next (_local_it)) {
			ValaLocalVariable* local;
			local = (ValaLocalVariable*) gee_iterator_get (_local_it);
			vala_symbol_set_active ((ValaSymbol*) local, FALSE);
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
		}
		(_local_it == NULL) ? NULL : (_local_it = (gee_collection_object_unref (_local_it), NULL));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, vala_symbol_get_parent_symbol (vala_semantic_analyzer_get_current_symbol (analyzer)));
	_tmp9 = NULL;
	_tmp8 = NULL;
	vala_foreach_statement_set_collection_variable (self, _tmp9 = vala_local_variable_new (collection_type, _tmp8 = g_strdup_printf ("%s_collection", self->priv->_variable_name), NULL, NULL));
	(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
	_tmp8 = (g_free (_tmp8), NULL);
	vala_block_add_local_variable ((ValaBlock*) self, self->priv->_collection_variable);
	vala_symbol_set_active ((ValaSymbol*) self->priv->_collection_variable, TRUE);
	_tmp10 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp10 = vala_code_node_get_error_types ((ValaCodeNode*) vala_foreach_statement_get_collection (self)));
	(_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL));
	_tmp11 = NULL;
	vala_code_node_add_error_types ((ValaCodeNode*) self, _tmp11 = vala_code_node_get_error_types ((ValaCodeNode*) vala_foreach_statement_get_body (self)));
	(_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL));
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_foreach_statement_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaForeachStatement * self;
	self = (ValaForeachStatement*) base;
	g_return_if_fail (collection != NULL);
	gee_collection_add (collection, self->priv->_element_variable);
}


ValaDataType* vala_foreach_statement_get_type_reference (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_foreach_statement_set_type_reference (ValaForeachStatement* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2);
	if (self->priv->_data_type != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
	}
}


const char* vala_foreach_statement_get_variable_name (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_variable_name;
}


void vala_foreach_statement_set_variable_name (ValaForeachStatement* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_variable_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_variable_name = (g_free (self->priv->_variable_name), NULL), _tmp2);
}


ValaExpression* vala_foreach_statement_get_collection (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_collection;
}


void vala_foreach_statement_set_collection (ValaForeachStatement* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_collection = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_collection == NULL) ? NULL : (self->priv->_collection = (vala_code_node_unref (self->priv->_collection), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_collection, (ValaCodeNode*) self);
}


ValaBlock* vala_foreach_statement_get_body (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_foreach_statement_set_body (ValaForeachStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_body, (ValaCodeNode*) self);
}


gboolean vala_foreach_statement_get_use_iterator (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_use_iterator;
}


static void vala_foreach_statement_set_use_iterator (ValaForeachStatement* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_use_iterator = value;
}


ValaLocalVariable* vala_foreach_statement_get_element_variable (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_element_variable;
}


void vala_foreach_statement_set_element_variable (ValaForeachStatement* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_element_variable = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_element_variable == NULL) ? NULL : (self->priv->_element_variable = (vala_code_node_unref (self->priv->_element_variable), NULL)), _tmp2);
}


ValaLocalVariable* vala_foreach_statement_get_collection_variable (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_collection_variable;
}


void vala_foreach_statement_set_collection_variable (ValaForeachStatement* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_collection_variable = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_collection_variable == NULL) ? NULL : (self->priv->_collection_variable = (vala_code_node_unref (self->priv->_collection_variable), NULL)), _tmp2);
}


ValaLocalVariable* vala_foreach_statement_get_iterator_variable (ValaForeachStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_iterator_variable;
}


void vala_foreach_statement_set_iterator_variable (ValaForeachStatement* self, ValaLocalVariable* value) {
	ValaLocalVariable* _tmp2;
	ValaLocalVariable* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_iterator_variable = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_iterator_variable == NULL) ? NULL : (self->priv->_iterator_variable = (vala_code_node_unref (self->priv->_iterator_variable), NULL)), _tmp2);
}


static void vala_foreach_statement_class_init (ValaForeachStatementClass * klass) {
	vala_foreach_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_foreach_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaForeachStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_foreach_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_foreach_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_foreach_statement_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_foreach_statement_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_foreach_statement_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_foreach_statement_real_get_defined_variables;
}


static void vala_foreach_statement_instance_init (ValaForeachStatement * self) {
	self->priv = VALA_FOREACH_STATEMENT_GET_PRIVATE (self);
}


static void vala_foreach_statement_finalize (ValaCodeNode* obj) {
	ValaForeachStatement * self;
	self = VALA_FOREACH_STATEMENT (obj);
	self->priv->_variable_name = (g_free (self->priv->_variable_name), NULL);
	(self->priv->_element_variable == NULL) ? NULL : (self->priv->_element_variable = (vala_code_node_unref (self->priv->_element_variable), NULL));
	(self->priv->_collection_variable == NULL) ? NULL : (self->priv->_collection_variable = (vala_code_node_unref (self->priv->_collection_variable), NULL));
	(self->priv->_iterator_variable == NULL) ? NULL : (self->priv->_iterator_variable = (vala_code_node_unref (self->priv->_iterator_variable), NULL));
	(self->priv->_collection == NULL) ? NULL : (self->priv->_collection = (vala_code_node_unref (self->priv->_collection), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	VALA_CODE_NODE_CLASS (vala_foreach_statement_parent_class)->finalize (obj);
}


GType vala_foreach_statement_get_type (void) {
	static GType vala_foreach_statement_type_id = 0;
	if (vala_foreach_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaForeachStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_foreach_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaForeachStatement), 0, (GInstanceInitFunc) vala_foreach_statement_instance_init, NULL };
		vala_foreach_statement_type_id = g_type_register_static (VALA_TYPE_BLOCK, "ValaForeachStatement", &g_define_type_info, 0);
	}
	return vala_foreach_statement_type_id;
}




