/* valaformalparameter.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valaformalparameter.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valasourcefile.h>
#include <vala/valaarraytype.h>
#include <vala/valareport.h>
#include <vala/valacodecontext.h>
#include <vala/valanullliteral.h>
#include <vala/valavaluetype.h>
#include <vala/valacreationmethod.h>
#include <vala/valamethod.h>
#include <vala/valablock.h>
#include <vala/valamemberaccess.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaassignment.h>
#include <vala/valastatement.h>




struct _ValaFormalParameterPrivate {
	ValaParameterDirection _direction;
	gboolean _ellipsis;
	gboolean _params_array;
	ValaExpression* _default_expression;
	gboolean _no_array_length;
	gboolean _construct_parameter;
	double _cparameter_position;
	double _carray_length_parameter_position;
	double _cdelegate_target_parameter_position;
	char* _ctype;
	ValaDataType* _data_type;
};

#define VALA_FORMAL_PARAMETER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterPrivate))
enum  {
	VALA_FORMAL_PARAMETER_DUMMY_PROPERTY
};
static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a);
static gboolean vala_formal_parameter_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_formal_parameter_parent_class = NULL;
static void vala_formal_parameter_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new formal parameter.
 *
 * @param name   parameter name
 * @param type   parameter type
 * @param source reference to source code
 * @return       newly created formal parameter
 */
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	ValaFormalParameter* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (parameter_type != NULL, NULL);
	self = (ValaFormalParameter*) vala_symbol_construct (object_type, name, source_reference);
	vala_formal_parameter_set_parameter_type (self, parameter_type);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference) {
	return vala_formal_parameter_construct (VALA_TYPE_FORMAL_PARAMETER, name, parameter_type, source_reference);
}


/**
 * Creates a new ellipsis parameter representing an indefinite number of
 * parameters.
 */
ValaFormalParameter* vala_formal_parameter_construct_with_ellipsis (GType object_type, ValaSourceReference* source_reference) {
	ValaFormalParameter* self;
	self = (ValaFormalParameter*) vala_symbol_construct (object_type, NULL, source_reference);
	vala_formal_parameter_set_ellipsis (self, TRUE);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaFormalParameter* vala_formal_parameter_new_with_ellipsis (ValaSourceReference* source_reference) {
	return vala_formal_parameter_construct_with_ellipsis (VALA_TYPE_FORMAL_PARAMETER, source_reference);
}


static void vala_formal_parameter_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_formal_parameter (visitor, self);
}


static void vala_formal_parameter_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (visitor != NULL);
	if (!self->priv->_ellipsis) {
		vala_code_node_accept ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), visitor);
		if (self->priv->_default_expression != NULL) {
			vala_code_node_accept ((ValaCodeNode*) self->priv->_default_expression, visitor);
		}
	}
}


static void vala_formal_parameter_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaFormalParameter * self;
	self = (ValaFormalParameter*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_formal_parameter_get_parameter_type (self) == old_type) {
		vala_formal_parameter_set_parameter_type (self, new_type);
	}
}


static void vala_formal_parameter_process_ccode_attribute (ValaFormalParameter* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "type")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_formal_parameter_set_ctype (self, _tmp0 = vala_attribute_get_string (a, "type"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "array_length_pos")) {
		vala_formal_parameter_set_carray_length_parameter_position (self, vala_attribute_get_double (a, "array_length_pos"));
	}
	if (vala_attribute_has_argument (a, "delegate_target_pos")) {
		vala_formal_parameter_set_cdelegate_target_parameter_position (self, vala_attribute_get_double (a, "delegate_target_pos"));
	}
}


/**
 * Process all associated attributes.
 */
void vala_formal_parameter_process_attributes (ValaFormalParameter* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_formal_parameter_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


ValaFormalParameter* vala_formal_parameter_copy (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (!self->priv->_ellipsis) {
		ValaFormalParameter* result;
		result = vala_formal_parameter_new (vala_symbol_get_name ((ValaSymbol*) self), vala_formal_parameter_get_parameter_type (self), vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_formal_parameter_set_params_array (result, self->priv->_params_array);
		return result;
	} else {
		return vala_formal_parameter_new_with_ellipsis (NULL);
	}
}


static gboolean vala_formal_parameter_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaFormalParameter * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	gboolean _tmp5;
	gboolean _tmp19;
	self = (ValaFormalParameter*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_formal_parameter_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, vala_symbol_get_parent_symbol ((ValaSymbol*) self));
	if (vala_formal_parameter_get_parameter_type (self) != NULL) {
		vala_code_node_check ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), analyzer);
	}
	if (!self->priv->_ellipsis) {
		gboolean _tmp3;
		vala_code_node_check ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self), analyzer);
		_tmp3 = FALSE;
		if (self->priv->_params_array) {
			_tmp3 = !(VALA_IS_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (self)));
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			gboolean _tmp4;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "parameter array expected");
			return (_tmp4 = FALSE, (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp4);
		}
		if (self->priv->_default_expression != NULL) {
			vala_code_node_check ((ValaCodeNode*) self->priv->_default_expression, analyzer);
		}
	}
	_tmp5 = FALSE;
	if (vala_code_context_get_non_null (vala_semantic_analyzer_get_context (analyzer))) {
		_tmp5 = self->priv->_default_expression != NULL;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		gboolean _tmp6;
		gboolean _tmp7;
		_tmp6 = FALSE;
		_tmp7 = FALSE;
		if (VALA_IS_NULL_LITERAL (self->priv->_default_expression)) {
			_tmp7 = !vala_data_type_get_nullable (vala_formal_parameter_get_parameter_type (self));
		} else {
			_tmp7 = FALSE;
		}
		if (_tmp7) {
			_tmp6 = self->priv->_direction != VALA_PARAMETER_DIRECTION_OUT;
		} else {
			_tmp6 = FALSE;
		}
		if (_tmp6) {
			char* _tmp9;
			char* _tmp8;
			_tmp9 = NULL;
			_tmp8 = NULL;
			vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp9 = g_strdup_printf ("`null' incompatible with parameter type `%s`", _tmp8 = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self))));
			_tmp9 = (g_free (_tmp9), NULL);
			_tmp8 = (g_free (_tmp8), NULL);
		}
	}
	if (!self->priv->_ellipsis) {
		if (!vala_symbol_is_internal_symbol ((ValaSymbol*) self)) {
			gboolean _tmp10;
			_tmp10 = FALSE;
			if (VALA_IS_VALUE_TYPE (vala_formal_parameter_get_parameter_type (self))) {
				_tmp10 = !vala_data_type_is_real_struct_type (vala_formal_parameter_get_parameter_type (self));
			} else {
				_tmp10 = FALSE;
			}
			if (_tmp10) {
				vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_formal_parameter_get_parameter_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			} else {
				vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_formal_parameter_get_parameter_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			}
		}
		vala_source_file_add_type_dependency (vala_semantic_analyzer_get_current_source_file (analyzer), vala_formal_parameter_get_parameter_type (self), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		/* check whether parameter type is at least as accessible as the method*/
		if (!vala_semantic_analyzer_is_type_accessible (analyzer, (ValaSymbol*) self, vala_formal_parameter_get_parameter_type (self))) {
			char* _tmp13;
			char* _tmp12;
			char* _tmp11;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp13 = NULL;
			_tmp12 = NULL;
			_tmp11 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp13 = g_strdup_printf ("parameter type `%s` is less accessible than method `%s`", _tmp11 = vala_code_node_to_string ((ValaCodeNode*) vala_formal_parameter_get_parameter_type (self)), _tmp12 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol ((ValaSymbol*) self))));
			_tmp13 = (g_free (_tmp13), NULL);
			_tmp12 = (g_free (_tmp12), NULL);
			_tmp11 = (g_free (_tmp11), NULL);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (self->priv->_construct_parameter) {
		ValaBlock* _tmp14;
		ValaBlock* method_body;
		ValaMemberAccess* _tmp15;
		ValaMemberAccess* _tmp16;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaExpressionStatement* _tmp18;
		ValaAssignment* _tmp17;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_parent_symbol ((ValaSymbol*) self)))) {
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "construct parameters are only allowed in type creation methods");
		}
		_tmp14 = NULL;
		method_body = (_tmp14 = vala_method_get_body ((ValaMethod*) (VALA_CREATION_METHOD (vala_symbol_get_parent_symbol ((ValaSymbol*) self)))), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
		_tmp15 = NULL;
		_tmp16 = NULL;
		left = (_tmp16 = vala_member_access_new ((ValaExpression*) (_tmp15 = vala_member_access_new_simple ("this", NULL)), vala_symbol_get_name ((ValaSymbol*) self), NULL), (_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL)), _tmp16);
		right = vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) self), NULL);
		_tmp18 = NULL;
		_tmp17 = NULL;
		vala_block_add_statement (method_body, (ValaStatement*) (_tmp18 = vala_expression_statement_new ((ValaExpression*) (_tmp17 = vala_assignment_new ((ValaExpression*) left, (ValaExpression*) right, VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL)), vala_code_node_get_source_reference ((ValaCodeNode*) self))));
		(_tmp18 == NULL) ? NULL : (_tmp18 = (vala_code_node_unref (_tmp18), NULL));
		(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_code_node_unref (_tmp17), NULL));
		(method_body == NULL) ? NULL : (method_body = (vala_code_node_unref (method_body), NULL));
		(left == NULL) ? NULL : (left = (vala_code_node_unref (left), NULL));
		(right == NULL) ? NULL : (right = (vala_code_node_unref (right), NULL));
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	return (_tmp19 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), _tmp19);
}


ValaDataType* vala_formal_parameter_get_parameter_type (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_data_type;
}


void vala_formal_parameter_set_parameter_type (ValaFormalParameter* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_data_type, (ValaCodeNode*) self);
}


ValaParameterDirection vala_formal_parameter_get_direction (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_direction;
}


void vala_formal_parameter_set_direction (ValaFormalParameter* self, ValaParameterDirection value) {
	g_return_if_fail (self != NULL);
	self->priv->_direction = value;
}


gboolean vala_formal_parameter_get_ellipsis (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_ellipsis;
}


void vala_formal_parameter_set_ellipsis (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_ellipsis = value;
}


gboolean vala_formal_parameter_get_params_array (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_params_array;
}


void vala_formal_parameter_set_params_array (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_params_array = value;
}


ValaExpression* vala_formal_parameter_get_default_expression (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_expression;
}


void vala_formal_parameter_set_default_expression (ValaFormalParameter* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL)), _tmp2);
}


gboolean vala_formal_parameter_get_no_array_length (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_no_array_length;
}


void vala_formal_parameter_set_no_array_length (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_no_array_length = value;
}


gboolean vala_formal_parameter_get_construct_parameter (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_construct_parameter;
}


void vala_formal_parameter_set_construct_parameter (ValaFormalParameter* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_construct_parameter = value;
}


double vala_formal_parameter_get_cparameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cparameter_position;
}


void vala_formal_parameter_set_cparameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cparameter_position = value;
}


double vala_formal_parameter_get_carray_length_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_carray_length_parameter_position;
}


void vala_formal_parameter_set_carray_length_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_carray_length_parameter_position = value;
}


double vala_formal_parameter_get_cdelegate_target_parameter_position (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_cdelegate_target_parameter_position;
}


void vala_formal_parameter_set_cdelegate_target_parameter_position (ValaFormalParameter* self, double value) {
	g_return_if_fail (self != NULL);
	self->priv->_cdelegate_target_parameter_position = value;
}


const char* vala_formal_parameter_get_ctype (ValaFormalParameter* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_ctype;
}


void vala_formal_parameter_set_ctype (ValaFormalParameter* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_ctype = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_ctype = (g_free (self->priv->_ctype), NULL), _tmp2);
}


static void vala_formal_parameter_class_init (ValaFormalParameterClass * klass) {
	vala_formal_parameter_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_formal_parameter_finalize;
	g_type_class_add_private (klass, sizeof (ValaFormalParameterPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_formal_parameter_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_formal_parameter_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_formal_parameter_real_replace_type;
	VALA_CODE_NODE_CLASS (klass)->check = vala_formal_parameter_real_check;
}


static void vala_formal_parameter_instance_init (ValaFormalParameter * self) {
	self->priv = VALA_FORMAL_PARAMETER_GET_PRIVATE (self);
	self->priv->_direction = VALA_PARAMETER_DIRECTION_IN;
}


static void vala_formal_parameter_finalize (ValaCodeNode* obj) {
	ValaFormalParameter * self;
	self = VALA_FORMAL_PARAMETER (obj);
	(self->priv->_default_expression == NULL) ? NULL : (self->priv->_default_expression = (vala_code_node_unref (self->priv->_default_expression), NULL));
	self->priv->_ctype = (g_free (self->priv->_ctype), NULL);
	(self->priv->_data_type == NULL) ? NULL : (self->priv->_data_type = (vala_code_node_unref (self->priv->_data_type), NULL));
	VALA_CODE_NODE_CLASS (vala_formal_parameter_parent_class)->finalize (obj);
}


GType vala_formal_parameter_get_type (void) {
	static GType vala_formal_parameter_type_id = 0;
	if (vala_formal_parameter_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaFormalParameterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_formal_parameter_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaFormalParameter), 0, (GInstanceInitFunc) vala_formal_parameter_instance_init, NULL };
		vala_formal_parameter_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaFormalParameter", &g_define_type_info, 0);
	}
	return vala_formal_parameter_type_id;
}



GType vala_parameter_direction_get_type (void) {
	static GType vala_parameter_direction_type_id = 0;
	if (G_UNLIKELY (vala_parameter_direction_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_PARAMETER_DIRECTION_IN, "VALA_PARAMETER_DIRECTION_IN", "in"}, {VALA_PARAMETER_DIRECTION_OUT, "VALA_PARAMETER_DIRECTION_OUT", "out"}, {VALA_PARAMETER_DIRECTION_REF, "VALA_PARAMETER_DIRECTION_REF", "ref"}, {0, NULL, NULL}};
		vala_parameter_direction_type_id = g_enum_register_static ("ValaParameterDirection", values);
	}
	return vala_parameter_direction_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




