/* valacodecontext.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valacodecontext.h>
#include <gee/arraylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/readonlylist.h>
#include <gee/collection.h>
#include <glib/gstdio.h>
#include <config.h>
#include <vala/valamethod.h>
#include <vala/valanamespace.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasourcefile.h>
#include <vala/valasourcefilecycle.h>
#include <vala/valacodegenerator.h>
#include <gobject/gvaluecollector.h>




struct _ValaCodeContextPrivate {
	char* _library;
	gboolean _memory_management;
	gboolean _assert;
	gboolean _checking;
	gboolean _non_null;
	gboolean _non_null_experimental;
	gboolean _dbus_transformation;
	gboolean _ccode_only;
	gboolean _compile_only;
	char* _output;
	char* _basedir;
	char* _directory;
	gboolean _debug;
	gint _optlevel;
	gboolean _thread;
	ValaMethod* _module_init_method;
	gboolean _save_temps;
	gint _target_glib_major;
	gint _target_glib_minor;
	GeeList* source_files;
	GeeList* c_source_files;
	ValaNamespace* _root;
	GeeList* cycles;
	GeeList* packages;
	ValaCodeGenerator* _codegen;
};

#define VALA_CODE_CONTEXT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CODE_CONTEXT, ValaCodeContextPrivate))
enum  {
	VALA_CODE_CONTEXT_DUMMY_PROPERTY
};
static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file);
static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeList* chain);
static gpointer vala_code_context_parent_class = NULL;
static void vala_code_context_finalize (ValaCodeContext* obj);



/**
 * Returns true if the target version of glib is greater than or 
 * equal to the specified version.
 */
gboolean vala_code_context_require_glib_version (ValaCodeContext* self, gint major, gint minor) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if ((self->priv->_target_glib_major > major)) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp1;
		_tmp1 = FALSE;
		if (self->priv->_target_glib_major == major) {
			_tmp1 = self->priv->_target_glib_minor >= minor;
		} else {
			_tmp1 = FALSE;
		}
		_tmp0 = (_tmp1);
	}
	return _tmp0;
}


ValaCodeContext* vala_code_context_construct (GType object_type) {
	ValaCodeContext* self;
	self = (ValaCodeContext*) g_type_create_instance (object_type);
	return self;
}


ValaCodeContext* vala_code_context_new (void) {
	return vala_code_context_construct (VALA_TYPE_CODE_CONTEXT);
}


/**
 * Returns a copy of the list of source files.
 *
 * @return list of source files
 */
GeeList* vala_code_context_get_source_files (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_SOURCE_FILE, (GBoxedCopyFunc) vala_source_file_ref, vala_source_file_unref, self->priv->source_files);
}


/**
 * Returns a copy of the list of C source files.
 *
 * @return list of C source files
 */
GeeList* vala_code_context_get_c_source_files (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->c_source_files);
}


/**
 * Adds the specified file to the list of source files.
 *
 * @param file a source file
 */
void vala_code_context_add_source_file (ValaCodeContext* self, ValaSourceFile* file) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (file != NULL);
	gee_collection_add ((GeeCollection*) self->priv->source_files, file);
}


/**
 * Adds the specified file to the list of C source files.
 *
 * @param file a C source file
 */
void vala_code_context_add_c_source_file (ValaCodeContext* self, const char* file) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (file != NULL);
	gee_collection_add ((GeeCollection*) self->priv->c_source_files, file);
}


/**
 * Returns a copy of the list of used packages.
 *
 * @return list of used packages
 */
GeeList* vala_code_context_get_packages (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->packages);
}


/**
 * Returns whether the specified package is being used.
 *
 * @param pkg a package name
 * @return    true if the specified package is being used
 */
gboolean vala_code_context_has_package (ValaCodeContext* self, const char* pkg) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	return gee_collection_contains ((GeeCollection*) self->priv->packages, pkg);
}


/**
 * Adds the specified package to the list of used packages.
 *
 * @param pkg a package name
 */
void vala_code_context_add_package (ValaCodeContext* self, const char* pkg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (pkg != NULL);
	gee_collection_add ((GeeCollection*) self->priv->packages, pkg);
}


/**
 * Visits the complete code tree file by file.
 *
 * @param visitor the visitor to be called when traversing
 */
void vala_code_context_accept (ValaCodeContext* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_code_context_get_root (self), visitor);
	{
		GeeIterator* file_it;
		file_it = gee_iterable_iterator ((GeeIterable*) self->priv->source_files);
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			file = (ValaSourceFile*) gee_iterator_get (file_it);
			vala_source_file_accept (file, visitor);
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(file_it == NULL) ? NULL : (file_it = (gee_collection_object_unref (file_it), NULL));
	}
}


/**
 * Find and resolve cycles in source file dependencies.
 */
void vala_code_context_find_header_cycles (ValaCodeContext* self) {
	g_return_if_fail (self != NULL);
	/* find cycles in dependencies between source files */
	{
		GeeIterator* file_it;
		/* find cycles in dependencies between source files */
		file_it = gee_iterable_iterator ((GeeIterable*) self->priv->source_files);
		/* find cycles in dependencies between source files */
		while (gee_iterator_next (file_it)) {
			ValaSourceFile* file;
			/* find cycles in dependencies between source files */
			file = (ValaSourceFile*) gee_iterator_get (file_it);
			/* we're only interested in internal source files */
			if (vala_source_file_get_external_package (file)) {
				(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
				continue;
			}
			if (vala_source_file_get_mark (file) == 0) {
				GeeArrayList* _tmp0;
				_tmp0 = NULL;
				vala_code_context_visit (self, file, (GeeList*) (_tmp0 = gee_array_list_new (VALA_TYPE_SOURCE_FILE, (GBoxedCopyFunc) vala_source_file_ref, vala_source_file_unref, g_direct_equal)));
				(_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL));
			}
			(file == NULL) ? NULL : (file = (vala_source_file_unref (file), NULL));
		}
		(file_it == NULL) ? NULL : (file_it = (gee_collection_object_unref (file_it), NULL));
	}
	/* find one head for each cycle, it must not have any
	 * hard dependencies on other files in the cycle
	 */
	{
		GeeIterator* cycle_it;
		/* find one head for each cycle, it must not have any
		 * hard dependencies on other files in the cycle
		 */
		cycle_it = gee_iterable_iterator ((GeeIterable*) self->priv->cycles);
		/* find one head for each cycle, it must not have any
		 * hard dependencies on other files in the cycle
		 */
		while (gee_iterator_next (cycle_it)) {
			ValaSourceFileCycle* cycle;
			/* find one head for each cycle, it must not have any
			 * hard dependencies on other files in the cycle
			 */
			cycle = (ValaSourceFileCycle*) gee_iterator_get (cycle_it);
			cycle->head = vala_code_context_find_cycle_head (self, (ValaSourceFile*) gee_list_get (cycle->files, 0));
			vala_source_file_set_is_cycle_head (cycle->head, TRUE);
			(cycle == NULL) ? NULL : (cycle = (vala_source_file_cycle_unref (cycle), NULL));
		}
		(cycle_it == NULL) ? NULL : (cycle_it = (gee_collection_object_unref (cycle_it), NULL));
	}
	/* connect the source files in a non-cyclic way
	 * cycle members connect to the head of their cycle
	 */
	{
		GeeIterator* file2_it;
		/* connect the source files in a non-cyclic way
		 * cycle members connect to the head of their cycle
		 */
		file2_it = gee_iterable_iterator ((GeeIterable*) self->priv->source_files);
		/* connect the source files in a non-cyclic way
		 * cycle members connect to the head of their cycle
		 */
		while (gee_iterator_next (file2_it)) {
			ValaSourceFile* file2;
			/* connect the source files in a non-cyclic way
			 * cycle members connect to the head of their cycle
			 */
			file2 = (ValaSourceFile*) gee_iterator_get (file2_it);
			/* we're only interested in internal source files */
			if (vala_source_file_get_external_package (file2)) {
				(file2 == NULL) ? NULL : (file2 = (vala_source_file_unref (file2), NULL));
				continue;
			}
			{
				GeeList* _tmp1;
				GeeIterator* _tmp2;
				GeeIterator* dep_it;
				_tmp1 = NULL;
				_tmp2 = NULL;
				dep_it = (_tmp2 = gee_iterable_iterator ((GeeIterable*) (_tmp1 = vala_source_file_get_header_internal_dependencies (file2))), (_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL)), _tmp2);
				while (gee_iterator_next (dep_it)) {
					ValaSourceFile* _tmp3;
					ValaSourceFile* dep;
					gboolean _tmp4;
					_tmp3 = NULL;
					dep = (_tmp3 = (ValaSourceFile*) gee_iterator_get (dep_it), (_tmp3 == NULL) ? NULL : vala_source_file_ref (_tmp3));
					_tmp4 = FALSE;
					if (vala_source_file_get_cycle (file2) != NULL) {
						_tmp4 = vala_source_file_get_cycle (dep) == vala_source_file_get_cycle (file2);
					} else {
						_tmp4 = FALSE;
					}
					if (_tmp4) {
						/* in the same cycle */
						if (!vala_source_file_get_is_cycle_head (file2)) {
							char* _tmp5;
							/* include header of cycle head */
							_tmp5 = NULL;
							vala_source_file_add_header_internal_include (file2, _tmp5 = vala_source_file_get_cinclude_filename (vala_source_file_get_cycle (file2)->head));
							_tmp5 = (g_free (_tmp5), NULL);
						}
					} else {
						char* _tmp6;
						/* we can just include the headers if they are not in a cycle or not in the same cycle as the current file */
						_tmp6 = NULL;
						vala_source_file_add_header_internal_include (file2, _tmp6 = vala_source_file_get_cinclude_filename (dep));
						_tmp6 = (g_free (_tmp6), NULL);
					}
					(dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL));
				}
				(dep_it == NULL) ? NULL : (dep_it = (gee_collection_object_unref (dep_it), NULL));
			}
			(file2 == NULL) ? NULL : (file2 = (vala_source_file_unref (file2), NULL));
		}
		(file2_it == NULL) ? NULL : (file2_it = (gee_collection_object_unref (file2_it), NULL));
	}
}


static ValaSourceFile* vala_code_context_find_cycle_head (ValaCodeContext* self, ValaSourceFile* file) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (file != NULL, NULL);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* dep_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		dep_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_source_file_get_header_internal_full_dependencies (file))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* _tmp2;
			ValaSourceFile* dep;
			_tmp2 = NULL;
			dep = (_tmp2 = (ValaSourceFile*) gee_iterator_get (dep_it), (_tmp2 == NULL) ? NULL : vala_source_file_ref (_tmp2));
			if (dep == file) {
				/* ignore file-internal dependencies */
				(dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL));
				continue;
			}
			{
				GeeIterator* cycle_file_it;
				cycle_file_it = gee_iterable_iterator ((GeeIterable*) vala_source_file_get_cycle (file)->files);
				while (gee_iterator_next (cycle_file_it)) {
					ValaSourceFile* _tmp3;
					ValaSourceFile* cycle_file;
					_tmp3 = NULL;
					cycle_file = (_tmp3 = (ValaSourceFile*) gee_iterator_get (cycle_file_it), (_tmp3 == NULL) ? NULL : vala_source_file_ref (_tmp3));
					if (dep == cycle_file) {
						ValaSourceFile* _tmp4;
						_tmp4 = NULL;
						return (_tmp4 = vala_code_context_find_cycle_head (self, dep), (cycle_file == NULL) ? NULL : (cycle_file = (vala_source_file_unref (cycle_file), NULL)), (cycle_file_it == NULL) ? NULL : (cycle_file_it = (gee_collection_object_unref (cycle_file_it), NULL)), (dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL)), (dep_it == NULL) ? NULL : (dep_it = (gee_collection_object_unref (dep_it), NULL)), _tmp4);
					}
					(cycle_file == NULL) ? NULL : (cycle_file = (vala_source_file_unref (cycle_file), NULL));
				}
				(cycle_file_it == NULL) ? NULL : (cycle_file_it = (gee_collection_object_unref (cycle_file_it), NULL));
			}
			(dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL));
		}
		(dep_it == NULL) ? NULL : (dep_it = (gee_collection_object_unref (dep_it), NULL));
	}
	/* no hard dependencies on members of the same cycle found
	 * source file suitable as cycle head
	 */
	return file;
}


static void vala_code_context_visit (ValaCodeContext* self, ValaSourceFile* file, GeeList* chain) {
	GeeList* l;
	g_return_if_fail (self != NULL);
	g_return_if_fail (file != NULL);
	g_return_if_fail (chain != NULL);
	l = (GeeList*) gee_array_list_new (VALA_TYPE_SOURCE_FILE, (GBoxedCopyFunc) vala_source_file_ref, vala_source_file_unref, g_direct_equal);
	{
		GeeIterator* chain_file_it;
		chain_file_it = gee_iterable_iterator ((GeeIterable*) chain);
		while (gee_iterator_next (chain_file_it)) {
			ValaSourceFile* chain_file;
			chain_file = (ValaSourceFile*) gee_iterator_get (chain_file_it);
			gee_collection_add ((GeeCollection*) l, chain_file);
			(chain_file == NULL) ? NULL : (chain_file = (vala_source_file_unref (chain_file), NULL));
		}
		(chain_file_it == NULL) ? NULL : (chain_file_it = (gee_collection_object_unref (chain_file_it), NULL));
	}
	gee_collection_add ((GeeCollection*) l, file);
	/* mark file as currently being visited */
	vala_source_file_set_mark (file, 1);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* dep_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		dep_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_source_file_get_header_internal_dependencies (file))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (dep_it)) {
			ValaSourceFile* _tmp2;
			ValaSourceFile* dep;
			_tmp2 = NULL;
			dep = (_tmp2 = (ValaSourceFile*) gee_iterator_get (dep_it), (_tmp2 == NULL) ? NULL : vala_source_file_ref (_tmp2));
			if (file == dep) {
				(dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL));
				continue;
			}
			if (vala_source_file_get_mark (dep) == 1) {
				ValaSourceFileCycle* cycle;
				gboolean cycle_start_found;
				/* found cycle */
				cycle = vala_source_file_cycle_new ();
				gee_collection_add ((GeeCollection*) self->priv->cycles, cycle);
				cycle_start_found = FALSE;
				{
					GeeIterator* cycle_file_it;
					cycle_file_it = gee_iterable_iterator ((GeeIterable*) l);
					while (gee_iterator_next (cycle_file_it)) {
						ValaSourceFile* cycle_file;
						ValaSourceFileCycle* _tmp3;
						ValaSourceFileCycle* ref_cycle_file_cycle;
						cycle_file = (ValaSourceFile*) gee_iterator_get (cycle_file_it);
						_tmp3 = NULL;
						ref_cycle_file_cycle = (_tmp3 = vala_source_file_get_cycle (cycle_file), (_tmp3 == NULL) ? NULL : vala_source_file_cycle_ref (_tmp3));
						if (!cycle_start_found) {
							if (cycle_file == dep) {
								cycle_start_found = TRUE;
							}
						}
						if (!cycle_start_found) {
							(cycle_file == NULL) ? NULL : (cycle_file = (vala_source_file_unref (cycle_file), NULL));
							(ref_cycle_file_cycle == NULL) ? NULL : (ref_cycle_file_cycle = (vala_source_file_cycle_unref (ref_cycle_file_cycle), NULL));
							continue;
						}
						if (ref_cycle_file_cycle != NULL) {
							/* file already in a cycle */
							if (vala_source_file_get_cycle (cycle_file) == cycle) {
								/* file is in the same cycle, nothing to do */
								(cycle_file == NULL) ? NULL : (cycle_file = (vala_source_file_unref (cycle_file), NULL));
								(ref_cycle_file_cycle == NULL) ? NULL : (ref_cycle_file_cycle = (vala_source_file_cycle_unref (ref_cycle_file_cycle), NULL));
								continue;
							}
							/* file is in an other cycle, merge the two cycles */
							gee_collection_remove ((GeeCollection*) self->priv->cycles, vala_source_file_get_cycle (cycle_file));
							{
								GeeIterator* inner_cycle_file_it;
								inner_cycle_file_it = gee_iterable_iterator ((GeeIterable*) vala_source_file_get_cycle (cycle_file)->files);
								while (gee_iterator_next (inner_cycle_file_it)) {
									ValaSourceFile* _tmp4;
									ValaSourceFile* inner_cycle_file;
									_tmp4 = NULL;
									inner_cycle_file = (_tmp4 = (ValaSourceFile*) gee_iterator_get (inner_cycle_file_it), (_tmp4 == NULL) ? NULL : vala_source_file_ref (_tmp4));
									if (vala_source_file_get_cycle (inner_cycle_file) != cycle) {
										/* file in inner cycle not yet added to outer cycle */
										gee_collection_add ((GeeCollection*) cycle->files, inner_cycle_file);
										vala_source_file_set_cycle (inner_cycle_file, cycle);
									}
									(inner_cycle_file == NULL) ? NULL : (inner_cycle_file = (vala_source_file_unref (inner_cycle_file), NULL));
								}
								(inner_cycle_file_it == NULL) ? NULL : (inner_cycle_file_it = (gee_collection_object_unref (inner_cycle_file_it), NULL));
							}
						} else {
							gee_collection_add ((GeeCollection*) cycle->files, cycle_file);
							vala_source_file_set_cycle (cycle_file, cycle);
						}
						(cycle_file == NULL) ? NULL : (cycle_file = (vala_source_file_unref (cycle_file), NULL));
						(ref_cycle_file_cycle == NULL) ? NULL : (ref_cycle_file_cycle = (vala_source_file_cycle_unref (ref_cycle_file_cycle), NULL));
					}
					(cycle_file_it == NULL) ? NULL : (cycle_file_it = (gee_collection_object_unref (cycle_file_it), NULL));
				}
				(cycle == NULL) ? NULL : (cycle = (vala_source_file_cycle_unref (cycle), NULL));
			} else {
				if (vala_source_file_get_mark (dep) == 0) {
					/* found not yet visited file */
					vala_code_context_visit (self, dep, l);
				}
			}
			(dep == NULL) ? NULL : (dep = (vala_source_file_unref (dep), NULL));
		}
		(dep_it == NULL) ? NULL : (dep_it = (gee_collection_object_unref (dep_it), NULL));
	}
	/* mark file as successfully visited */
	vala_source_file_set_mark (file, 2);
	(l == NULL) ? NULL : (l = (gee_collection_object_unref (l), NULL));
}


char* vala_code_context_get_package_path (ValaCodeContext* self, const char* pkg, int vapi_directories_length1, char** vapi_directories) {
	char* basename;
	char* filename;
	char* _tmp3;
	char* _tmp8;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (pkg != NULL, NULL);
	basename = g_strdup_printf ("%s.vapi", pkg);
	filename = NULL;
	if (vapi_directories != NULL) {
		{
			char** vapidir_collection;
			int vapidir_collection_length1;
			int vapidir_it;
			vapidir_collection = vapi_directories;
			vapidir_collection_length1 = vapi_directories_length1;
			for (vapidir_it = 0; ((vapi_directories_length1 != -1) && (vapidir_it < vapi_directories_length1)) || ((vapi_directories_length1 == -1) && (vapidir_collection[vapidir_it] != NULL)); vapidir_it = vapidir_it + 1) {
				const char* _tmp2;
				char* vapidir;
				_tmp2 = NULL;
				vapidir = (_tmp2 = vapidir_collection[vapidir_it], (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
				{
					char* _tmp0;
					_tmp0 = NULL;
					filename = (_tmp0 = g_build_filename (vapidir, basename, NULL), filename = (g_free (filename), NULL), _tmp0);
					if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
						char* _tmp1;
						_tmp1 = NULL;
						return (_tmp1 = filename, vapidir = (g_free (vapidir), NULL), basename = (g_free (basename), NULL), _tmp1);
					}
					vapidir = (g_free (vapidir), NULL);
				}
			}
		}
	}
	_tmp3 = NULL;
	filename = (_tmp3 = g_build_filename (PACKAGE_DATADIR, "vapi", basename, NULL), filename = (g_free (filename), NULL), _tmp3);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* _tmp4;
		_tmp4 = NULL;
		return (_tmp4 = filename, basename = (g_free (basename), NULL), _tmp4);
	}
	{
		char** vapidir_collection;
		int vapidir_collection_length1;
		char** vapidir_it;
		vapidir_collection = g_get_system_data_dirs ();
		vapidir_collection_length1 = -1;
		for (vapidir_it = vapidir_collection; *vapidir_it != NULL; vapidir_it = vapidir_it + 1) {
			const char* _tmp7;
			char* vapidir;
			_tmp7 = NULL;
			vapidir = (_tmp7 = *vapidir_it, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
			{
				char* _tmp5;
				_tmp5 = NULL;
				filename = (_tmp5 = g_build_filename (vapidir, "vala/vapi", basename, NULL), filename = (g_free (filename), NULL), _tmp5);
				if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
					char* _tmp6;
					_tmp6 = NULL;
					return (_tmp6 = filename, vapidir = (g_free (vapidir), NULL), basename = (g_free (basename), NULL), _tmp6);
				}
				vapidir = (g_free (vapidir), NULL);
			}
		}
	}
	_tmp8 = NULL;
	return (_tmp8 = NULL, basename = (g_free (basename), NULL), filename = (g_free (filename), NULL), _tmp8);
}


const char* vala_code_context_get_library (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_library;
}


void vala_code_context_set_library (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_library = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_library = (g_free (self->priv->_library), NULL), _tmp2);
}


gboolean vala_code_context_get_memory_management (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_memory_management;
}


void vala_code_context_set_memory_management (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_memory_management = value;
}


gboolean vala_code_context_get_assert (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_assert;
}


void vala_code_context_set_assert (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_assert = value;
}


gboolean vala_code_context_get_checking (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_checking;
}


void vala_code_context_set_checking (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_checking = value;
}


gboolean vala_code_context_get_non_null (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_non_null;
}


void vala_code_context_set_non_null (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_non_null = value;
}


gboolean vala_code_context_get_non_null_experimental (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_non_null_experimental;
}


void vala_code_context_set_non_null_experimental (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_non_null_experimental = value;
}


gboolean vala_code_context_get_dbus_transformation (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_dbus_transformation;
}


void vala_code_context_set_dbus_transformation (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_dbus_transformation = value;
}


gboolean vala_code_context_get_ccode_only (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_ccode_only;
}


void vala_code_context_set_ccode_only (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_ccode_only = value;
}


gboolean vala_code_context_get_compile_only (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_compile_only;
}


void vala_code_context_set_compile_only (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_compile_only = value;
}


const char* vala_code_context_get_output (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_output;
}


void vala_code_context_set_output (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_output = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_output = (g_free (self->priv->_output), NULL), _tmp2);
}


const char* vala_code_context_get_basedir (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_basedir;
}


void vala_code_context_set_basedir (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_basedir = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_basedir = (g_free (self->priv->_basedir), NULL), _tmp2);
}


const char* vala_code_context_get_directory (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_directory;
}


void vala_code_context_set_directory (ValaCodeContext* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_directory = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_directory = (g_free (self->priv->_directory), NULL), _tmp2);
}


gboolean vala_code_context_get_debug (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_debug;
}


void vala_code_context_set_debug (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_debug = value;
}


gint vala_code_context_get_optlevel (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_optlevel;
}


void vala_code_context_set_optlevel (ValaCodeContext* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_optlevel = value;
}


gboolean vala_code_context_get_thread (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_thread;
}


void vala_code_context_set_thread (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_thread = value;
}


ValaMethod* vala_code_context_get_module_init_method (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_module_init_method;
}


void vala_code_context_set_module_init_method (ValaCodeContext* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_module_init_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_module_init_method == NULL) ? NULL : (self->priv->_module_init_method = (vala_code_node_unref (self->priv->_module_init_method), NULL)), _tmp2);
}


gboolean vala_code_context_get_save_temps (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_save_temps;
}


void vala_code_context_set_save_temps (ValaCodeContext* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_save_temps = value;
}


gint vala_code_context_get_target_glib_major (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_target_glib_major;
}


void vala_code_context_set_target_glib_major (ValaCodeContext* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_target_glib_major = value;
}


gint vala_code_context_get_target_glib_minor (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_target_glib_minor;
}


void vala_code_context_set_target_glib_minor (ValaCodeContext* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_target_glib_minor = value;
}


gboolean vala_code_context_get_save_csources (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_save_temps;
}


gboolean vala_code_context_get_save_cheaders (ValaCodeContext* self) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (vala_code_context_get_save_csources (self)) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = NULL != self->priv->_library;
	}
	return _tmp0;
}


ValaNamespace* vala_code_context_get_root (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_root;
}


ValaCodeGenerator* vala_code_context_get_codegen (ValaCodeContext* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_codegen;
}


void vala_code_context_set_codegen (ValaCodeContext* self, ValaCodeGenerator* value) {
	ValaCodeGenerator* _tmp2;
	ValaCodeGenerator* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_codegen = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_visitor_ref (_tmp1)), (self->priv->_codegen == NULL) ? NULL : (self->priv->_codegen = (vala_code_visitor_unref (self->priv->_codegen), NULL)), _tmp2);
}


static void vala_value_code_context_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_code_context_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_code_context_unref (value->data[0].v_pointer);
	}
}


static void vala_value_code_context_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_code_context_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_code_context_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_code_context_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCodeContext* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_code_context_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_code_context_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCodeContext** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_code_context_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_code_context (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCodeContext* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_CODE_CONTEXT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_code_context (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CODE_CONTEXT), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_code_context (GValue* value, gpointer v_object) {
	ValaCodeContext* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_CODE_CONTEXT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_CODE_CONTEXT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_code_context_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_code_context_unref (old);
	}
}


static void vala_code_context_class_init (ValaCodeContextClass * klass) {
	vala_code_context_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_CONTEXT_CLASS (klass)->finalize = vala_code_context_finalize;
	g_type_class_add_private (klass, sizeof (ValaCodeContextPrivate));
}


static void vala_code_context_instance_init (ValaCodeContext * self) {
	self->priv = VALA_CODE_CONTEXT_GET_PRIVATE (self);
	self->priv->source_files = (GeeList*) gee_array_list_new (VALA_TYPE_SOURCE_FILE, (GBoxedCopyFunc) vala_source_file_ref, vala_source_file_unref, g_direct_equal);
	self->priv->c_source_files = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->_root = vala_namespace_new (NULL, NULL);
	self->priv->cycles = (GeeList*) gee_array_list_new (VALA_TYPE_SOURCE_FILE_CYCLE, (GBoxedCopyFunc) vala_source_file_cycle_ref, vala_source_file_cycle_unref, g_direct_equal);
	self->priv->packages = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_equal);
	self->priv->_codegen = vala_code_generator_new ();
	self->ref_count = 1;
}


static void vala_code_context_finalize (ValaCodeContext* obj) {
	ValaCodeContext * self;
	self = VALA_CODE_CONTEXT (obj);
	self->priv->_library = (g_free (self->priv->_library), NULL);
	self->priv->_output = (g_free (self->priv->_output), NULL);
	self->priv->_basedir = (g_free (self->priv->_basedir), NULL);
	self->priv->_directory = (g_free (self->priv->_directory), NULL);
	(self->priv->_module_init_method == NULL) ? NULL : (self->priv->_module_init_method = (vala_code_node_unref (self->priv->_module_init_method), NULL));
	(self->priv->source_files == NULL) ? NULL : (self->priv->source_files = (gee_collection_object_unref (self->priv->source_files), NULL));
	(self->priv->c_source_files == NULL) ? NULL : (self->priv->c_source_files = (gee_collection_object_unref (self->priv->c_source_files), NULL));
	(self->priv->_root == NULL) ? NULL : (self->priv->_root = (vala_code_node_unref (self->priv->_root), NULL));
	(self->priv->cycles == NULL) ? NULL : (self->priv->cycles = (gee_collection_object_unref (self->priv->cycles), NULL));
	(self->priv->packages == NULL) ? NULL : (self->priv->packages = (gee_collection_object_unref (self->priv->packages), NULL));
	(self->priv->_codegen == NULL) ? NULL : (self->priv->_codegen = (vala_code_visitor_unref (self->priv->_codegen), NULL));
}


GType vala_code_context_get_type (void) {
	static GType vala_code_context_type_id = 0;
	if (vala_code_context_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_code_context_init, vala_value_code_context_free_value, vala_value_code_context_copy_value, vala_value_code_context_peek_pointer, "p", vala_value_code_context_collect_value, "p", vala_value_code_context_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCodeContextClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_code_context_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCodeContext), 0, (GInstanceInitFunc) vala_code_context_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_code_context_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCodeContext", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_code_context_type_id;
}


gpointer vala_code_context_ref (gpointer instance) {
	ValaCodeContext* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_code_context_unref (gpointer instance) {
	ValaCodeContext* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_CODE_CONTEXT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




