/* valastruct.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valastruct.h>
#include <gee/arraylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/readonlylist.h>
#include <gee/collection.h>
#include <vala/valatypeparameter.h>
#include <vala/valafield.h>
#include <vala/valamember.h>
#include <vala/valadatatype.h>
#include <vala/valacodevisitor.h>
#include <vala/valamethod.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaattribute.h>
#include <vala/valasourcefile.h>
#include <vala/valasymbol.h>
#include <vala/valasourcereference.h>
#include <vala/valaconstant.h>
#include <vala/valareport.h>
#include <vala/valascope.h>
#include <vala/valacreationmethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valavaluetype.h>
#include <vala/valavoidtype.h>
#include <vala/valalocalvariable.h>
#include <vala/valaexpression.h>




struct _ValaStructPrivate {
	GeeList* type_parameters;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* base_types;
	char* cname;
	char* const_cname;
	char* type_id;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	gboolean integer_type;
	gboolean floating_type;
	gint rank;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* default_value;
	char* type_signature;
	char* copy_function;
	char* destroy_function;
	ValaMethod* _default_construction_method;
};

#define VALA_STRUCT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRUCT, ValaStructPrivate))
enum  {
	VALA_STRUCT_DUMMY_PROPERTY
};
static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_struct_real_get_cname (ValaTypeSymbol* base, gboolean const_type);
static void vala_struct_set_const_cname (ValaStruct* self, const char* cname);
static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_struct_get_lower_case_csuffix (ValaStruct* self);
static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_struct_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix);
static char* vala_struct_real_get_type_signature (ValaTypeSymbol* base);
static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a);
static char* vala_struct_real_get_type_id (ValaTypeSymbol* base);
static char* vala_struct_real_get_marshaller_type_name (ValaTypeSymbol* base);
static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name);
static char* vala_struct_real_get_get_value_function (ValaTypeSymbol* base);
static char* vala_struct_real_get_set_value_function (ValaTypeSymbol* base);
static void vala_struct_set_get_value_function (ValaStruct* self, const char* function);
static void vala_struct_set_set_value_function (ValaStruct* self, const char* function);
static char* vala_struct_real_get_default_value (ValaTypeSymbol* base);
static void vala_struct_set_default_value (ValaStruct* self, const char* value);
static gint vala_struct_real_get_type_parameter_index (ValaTypeSymbol* base, const char* name);
static void vala_struct_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_struct_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t);
static char* vala_struct_real_get_copy_function (ValaTypeSymbol* base);
static char* vala_struct_real_get_destroy_function (ValaTypeSymbol* base);
static gboolean vala_struct_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_struct_parent_class = NULL;
static void vala_struct_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new struct.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created struct
 */
ValaStruct* vala_struct_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaStruct* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaStruct*) vala_typesymbol_construct (object_type, name, source_reference);
	return self;
}


ValaStruct* vala_struct_new (const char* name, ValaSourceReference* source_reference) {
	return vala_struct_construct (VALA_TYPE_STRUCT, name, source_reference);
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_struct_add_type_parameter (ValaStruct* self, ValaTypeParameter* p) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (p != NULL);
	gee_collection_add ((GeeCollection*) self->priv->type_parameters, p);
	p->type = (ValaTypeSymbol*) self;
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) p), (ValaSymbol*) p);
}


/**
 * Returns a copy of the type parameter list.
 *
 * @return list of type parameters
 */
GeeList* vala_struct_get_type_parameters (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_TYPEPARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->type_parameters);
}


/**
 * Adds the specified constant as a member to this struct.
 *
 * @param c a constant
 */
void vala_struct_add_constant (ValaStruct* self, ValaConstant* c) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (c != NULL);
	gee_collection_add ((GeeCollection*) self->priv->constants, c);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) c), (ValaSymbol*) c);
}


/**
 * Adds the specified field as a member to this struct.
 *
 * @param f a field
 */
void vala_struct_add_field (ValaStruct* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeList* vala_struct_get_fields (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
}


/**
 * Returns a copy of the list of constants.
 *
 * @return list of constants
 */
GeeList* vala_struct_get_constants (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->constants);
}


/**
 * Adds the specified method as a member to this struct.
 *
 * @param m a method
 */
void vala_struct_add_method (ValaStruct* self, ValaMethod* m) {
	gboolean _tmp0;
	gboolean _tmp3;
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	g_return_if_fail (m != NULL);
	_tmp0 = FALSE;
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = VALA_IS_CREATION_METHOD (m);
	}
	if (_tmp0) {
		ValaFormalParameter* _tmp2;
		ValaValueType* _tmp1;
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_method_set_this_parameter (m, _tmp2 = vala_formal_parameter_new ("this", (ValaDataType*) (_tmp1 = vala_value_type_new ((ValaTypeSymbol*) self)), NULL));
		(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_code_node_unref (_tmp2), NULL));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (vala_code_node_unref (_tmp1), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_this_parameter (m)), (ValaSymbol*) vala_method_get_this_parameter (m));
	}
	_tmp3 = FALSE;
	if (!(VALA_IS_VOID_TYPE (vala_method_get_return_type (m)))) {
		GeeList* _tmp4;
		_tmp4 = NULL;
		_tmp3 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_method_get_postconditions (m))) > 0;
		(_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL));
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		ValaLocalVariable* _tmp6;
		ValaDataType* _tmp5;
		_tmp6 = NULL;
		_tmp5 = NULL;
		vala_method_set_result_var (m, _tmp6 = vala_local_variable_new (_tmp5 = vala_data_type_copy (vala_method_get_return_type (m)), "result", NULL, NULL));
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL));
		(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_code_node_unref (_tmp5), NULL));
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) m), vala_symbol_get_name ((ValaSymbol*) vala_method_get_result_var (m)), (ValaSymbol*) vala_method_get_result_var (m));
	}
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaCreationMethod* _tmp7;
		ValaCreationMethod* cm;
		gboolean _tmp8;
		if (vala_symbol_get_name ((ValaSymbol*) m) == NULL) {
			vala_struct_set_default_construction_method (self, m);
			vala_symbol_set_name ((ValaSymbol*) m, "new");
		}
		_tmp7 = NULL;
		cm = (_tmp7 = VALA_CREATION_METHOD (m), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
		_tmp8 = FALSE;
		if (vala_creation_method_get_type_name (cm) != NULL) {
			_tmp8 = _vala_strcmp0 (vala_creation_method_get_type_name (cm), vala_symbol_get_name ((ValaSymbol*) self)) != 0;
		} else {
			_tmp8 = FALSE;
		}
		if (_tmp8) {
			char* _tmp10;
			char* _tmp9;
			/* type_name is null for constructors generated by GIdlParser*/
			_tmp10 = NULL;
			_tmp9 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), _tmp10 = g_strdup_printf ("missing return type in method `%s.%s´", _tmp9 = vala_symbol_get_full_name ((ValaSymbol*) self), vala_creation_method_get_type_name (cm)));
			_tmp10 = (g_free (_tmp10), NULL);
			_tmp9 = (g_free (_tmp9), NULL);
			vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
			(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
			return;
		}
		(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeList* vala_struct_get_methods (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_struct (visitor, self);
}


static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* type_it;
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (type_it);
			vala_code_node_accept ((ValaCodeNode*) type, visitor);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	{
		GeeIterator* p_it;
		p_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_parameters);
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (p_it);
			vala_code_node_accept ((ValaCodeNode*) p, visitor);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(p_it == NULL) ? NULL : (p_it = (gee_collection_object_unref (p_it), NULL));
	}
	{
		GeeIterator* f_it;
		f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(f_it == NULL) ? NULL : (f_it = (gee_collection_object_unref (f_it), NULL));
	}
	{
		GeeIterator* c_it;
		c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = (ValaConstant*) gee_iterator_get (c_it);
			vala_code_node_accept ((ValaCodeNode*) c, visitor);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(c_it == NULL) ? NULL : (c_it = (gee_collection_object_unref (c_it), NULL));
	}
	{
		GeeIterator* m_it;
		m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(m_it == NULL) ? NULL : (m_it = (gee_collection_object_unref (m_it), NULL));
	}
}


static char* vala_struct_real_get_cname (ValaTypeSymbol* base, gboolean const_type) {
	ValaStruct * self;
	gboolean _tmp0;
	const char* _tmp5;
	self = (ValaStruct*) base;
	_tmp0 = FALSE;
	if (const_type) {
		_tmp0 = self->priv->const_cname != NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		const char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = self->priv->const_cname, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
	}
	if (self->priv->cname == NULL) {
		ValaAttribute* attr;
		attr = vala_code_node_get_attribute ((ValaCodeNode*) self, "CCode");
		if (attr != NULL) {
			char* _tmp3;
			_tmp3 = NULL;
			self->priv->cname = (_tmp3 = vala_attribute_get_string (attr, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp3);
		}
		if (self->priv->cname == NULL) {
			char* _tmp4;
			_tmp4 = NULL;
			self->priv->cname = (_tmp4 = vala_struct_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp4);
		}
		(attr == NULL) ? NULL : (attr = (vala_code_node_unref (attr), NULL));
	}
	_tmp5 = NULL;
	return (_tmp5 = self->priv->cname, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
}


/**
 * Returns the default name of this struct as it is used in C code.
 *
 * @return the name to be used in C code by default
 */
char* vala_struct_get_default_cname (ValaStruct* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strdup_printf ("%s%s", _tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), vala_symbol_get_name ((ValaSymbol*) self)), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static void vala_struct_set_const_cname (ValaStruct* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->const_cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->const_cname = (g_free (self->priv->const_cname), NULL), _tmp1);
}


static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaStruct * self;
	const char* _tmp2;
	self = (ValaStruct*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = g_strdup_printf ("%s_", _tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, NULL)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->lower_case_cprefix, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
}


static char* vala_struct_get_lower_case_csuffix (ValaStruct* self) {
	const char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->lower_case_csuffix = (_tmp0 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self)), self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->lower_case_csuffix, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix) {
	ValaStruct * self;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	self = (ValaStruct*) base;
	if (infix == NULL) {
		infix = "";
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s%s%s", _tmp0 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), infix, _tmp1 = vala_struct_get_lower_case_csuffix (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


static char* vala_struct_real_get_upper_case_cname (ValaTypeSymbol* base, const char* infix) {
	ValaStruct * self;
	char* _tmp0;
	char* _tmp1;
	self = (ValaStruct*) base;
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_utf8_strup (_tmp0 = vala_symbol_get_lower_case_cname ((ValaSymbol*) self, infix), -1), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_struct_real_get_type_signature (ValaTypeSymbol* base) {
	ValaStruct * self;
	const char* _tmp3;
	self = (ValaStruct*) base;
	if (self->priv->type_signature == NULL) {
		GString* str;
		const char* _tmp1;
		char* _tmp2;
		str = g_string_new ("");
		g_string_append_c (str, '(');
		{
			GeeIterator* f_it;
			f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
			while (gee_iterator_next (f_it)) {
				ValaField* f;
				f = (ValaField*) gee_iterator_get (f_it);
				if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
					char* _tmp0;
					_tmp0 = NULL;
					g_string_append (str, _tmp0 = vala_data_type_get_type_signature (vala_field_get_field_type (f)));
					_tmp0 = (g_free (_tmp0), NULL);
				}
				(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			}
			(f_it == NULL) ? NULL : (f_it = (gee_collection_object_unref (f_it), NULL));
		}
		g_string_append_c (str, ')');
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = str->str, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->type_signature, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


/**
 * Returns whether this is an integer type.
 *
 * @return true if this is an integer type, false otherwise
 */
gboolean vala_struct_is_integer_type (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GeeIterator* type_it;
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			ValaStruct* _tmp1;
			ValaTypeSymbol* _tmp0;
			ValaStruct* st;
			gboolean _tmp2;
			type = (ValaDataType*) gee_iterator_get (type_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			_tmp2 = FALSE;
			if (st != NULL) {
				_tmp2 = vala_struct_is_integer_type (st);
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				gboolean _tmp3;
				return (_tmp3 = TRUE, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp3);
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	return self->priv->integer_type;
}


/**
 * Returns whether this is a floating point type.
 *
 * @return true if this is a floating point type, false otherwise
 */
gboolean vala_struct_is_floating_type (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GeeIterator* type_it;
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			ValaStruct* _tmp1;
			ValaTypeSymbol* _tmp0;
			ValaStruct* st;
			gboolean _tmp2;
			type = (ValaDataType*) gee_iterator_get (type_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			_tmp2 = FALSE;
			if (st != NULL) {
				_tmp2 = vala_struct_is_floating_type (st);
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				gboolean _tmp3;
				return (_tmp3 = TRUE, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp3);
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	return self->priv->floating_type;
}


/**
 * Returns the rank of this integer or floating point type.
 *
 * @return the rank if this is an integer or floating point type
 */
gint vala_struct_get_rank (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->rank;
}


static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_struct_set_const_cname (self, _tmp0 = vala_attribute_get_string (a, "const_cname"));
		_tmp0 = (g_free (_tmp0), NULL);
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* _tmp1;
		_tmp1 = NULL;
		self->priv->lower_case_cprefix = (_tmp1 = vala_attribute_get_string (a, "cprefix"), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			int filename_collection_length1;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp2;
				char* filename;
				_tmp2 = NULL;
				filename = (_tmp2 = *filename_it, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2));
				{
					vala_typesymbol_add_cheader_filename ((ValaTypeSymbol*) self, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_struct_set_type_id (self, _tmp3 = vala_attribute_get_string (a, "type_id"));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* _tmp4;
		_tmp4 = NULL;
		vala_struct_set_marshaller_type_name (self, _tmp4 = vala_attribute_get_string (a, "marshaller_type_name"));
		_tmp4 = (g_free (_tmp4), NULL);
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* _tmp5;
		_tmp5 = NULL;
		vala_struct_set_get_value_function (self, _tmp5 = vala_attribute_get_string (a, "get_value_function"));
		_tmp5 = (g_free (_tmp5), NULL);
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* _tmp6;
		_tmp6 = NULL;
		vala_struct_set_set_value_function (self, _tmp6 = vala_attribute_get_string (a, "set_value_function"));
		_tmp6 = (g_free (_tmp6), NULL);
	}
	if (vala_attribute_has_argument (a, "default_value")) {
		char* _tmp7;
		_tmp7 = NULL;
		vala_struct_set_default_value (self, _tmp7 = vala_attribute_get_string (a, "default_value"));
		_tmp7 = (g_free (_tmp7), NULL);
	}
	if (vala_attribute_has_argument (a, "type_signature")) {
		char* _tmp8;
		_tmp8 = NULL;
		self->priv->type_signature = (_tmp8 = vala_attribute_get_string (a, "type_signature"), self->priv->type_signature = (g_free (self->priv->type_signature), NULL), _tmp8);
	}
	if (vala_attribute_has_argument (a, "copy_function")) {
		char* _tmp9;
		_tmp9 = NULL;
		vala_struct_set_copy_function (self, _tmp9 = vala_attribute_get_string (a, "copy_function"));
		_tmp9 = (g_free (_tmp9), NULL);
	}
	if (vala_attribute_has_argument (a, "destroy_function")) {
		char* _tmp10;
		_tmp10 = NULL;
		vala_struct_set_destroy_function (self, _tmp10 = vala_attribute_get_string (a, "destroy_function"));
		_tmp10 = (g_free (_tmp10), NULL);
	}
}


static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	self->priv->integer_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	self->priv->floating_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


/**
 * Process all associated attributes.
 */
void vala_struct_process_attributes (ValaStruct* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_struct_process_ccode_attribute (self, a);
				} else {
					if (_vala_strcmp0 (vala_attribute_get_name (a), "IntegerType") == 0) {
						vala_struct_process_integer_type_attribute (self, a);
					} else {
						if (_vala_strcmp0 (vala_attribute_get_name (a), "FloatingType") == 0) {
							vala_struct_process_floating_type_attribute (self, a);
						}
					}
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static char* vala_struct_real_get_type_id (ValaTypeSymbol* base) {
	ValaStruct * self;
	const char* _tmp6;
	self = (ValaStruct*) base;
	if (self->priv->type_id == NULL) {
		{
			GeeIterator* type_it;
			type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				ValaStruct* _tmp1;
				ValaTypeSymbol* _tmp0;
				ValaStruct* st;
				type = (ValaDataType*) gee_iterator_get (type_it);
				_tmp1 = NULL;
				_tmp0 = NULL;
				st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
				if (st != NULL) {
					char* _tmp2;
					_tmp2 = NULL;
					return (_tmp2 = vala_typesymbol_get_type_id ((ValaTypeSymbol*) st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp2);
				}
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
		}
		if (vala_struct_is_simple_type (self)) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("The type `%s` doesn't declare a type id", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
		} else {
			return g_strdup ("G_TYPE_POINTER");
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = self->priv->type_id, (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
}


void vala_struct_set_type_id (ValaStruct* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->type_id = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->type_id = (g_free (self->priv->type_id), NULL), _tmp1);
}


static char* vala_struct_real_get_marshaller_type_name (ValaTypeSymbol* base) {
	ValaStruct * self;
	const char* _tmp6;
	self = (ValaStruct*) base;
	if (self->priv->marshaller_type_name == NULL) {
		{
			GeeIterator* type_it;
			type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				ValaStruct* _tmp1;
				ValaTypeSymbol* _tmp0;
				ValaStruct* st;
				type = (ValaDataType*) gee_iterator_get (type_it);
				_tmp1 = NULL;
				_tmp0 = NULL;
				st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
				if (st != NULL) {
					char* _tmp2;
					_tmp2 = NULL;
					return (_tmp2 = vala_typesymbol_get_marshaller_type_name ((ValaTypeSymbol*) st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp2);
				}
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
		}
		if (vala_struct_is_simple_type (self)) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("The type `%s` doesn't declare a marshaller type name", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
		} else {
			return g_strdup ("POINTER");
		}
	}
	_tmp6 = NULL;
	return (_tmp6 = self->priv->marshaller_type_name, (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
}


static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->marshaller_type_name = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL), _tmp1);
}


static char* vala_struct_real_get_get_value_function (ValaTypeSymbol* base) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	if (self->priv->get_value_function == NULL) {
		{
			GeeIterator* type_it;
			type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				ValaStruct* _tmp1;
				ValaTypeSymbol* _tmp0;
				ValaStruct* st;
				type = (ValaDataType*) gee_iterator_get (type_it);
				_tmp1 = NULL;
				_tmp0 = NULL;
				st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
				if (st != NULL) {
					char* _tmp2;
					_tmp2 = NULL;
					return (_tmp2 = vala_typesymbol_get_get_value_function ((ValaTypeSymbol*) st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp2);
				}
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
		}
		if (vala_struct_is_simple_type (self)) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("The value type `%s` doesn't declare a GValue get function", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
			return NULL;
		} else {
			return g_strdup ("g_value_get_pointer");
		}
	} else {
		const char* _tmp7;
		_tmp7 = NULL;
		return (_tmp7 = self->priv->get_value_function, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
	}
}


static char* vala_struct_real_get_set_value_function (ValaTypeSymbol* base) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	if (self->priv->set_value_function == NULL) {
		{
			GeeIterator* type_it;
			type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				ValaStruct* _tmp1;
				ValaTypeSymbol* _tmp0;
				ValaStruct* st;
				type = (ValaDataType*) gee_iterator_get (type_it);
				_tmp1 = NULL;
				_tmp0 = NULL;
				st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
				if (st != NULL) {
					char* _tmp2;
					_tmp2 = NULL;
					return (_tmp2 = vala_typesymbol_get_set_value_function ((ValaTypeSymbol*) st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp2);
				}
				(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
		}
		if (vala_struct_is_simple_type (self)) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("The value type `%s` doesn't declare a GValue set function", _tmp3 = vala_symbol_get_full_name ((ValaSymbol*) self)));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
			return NULL;
		} else {
			return g_strdup ("g_value_set_pointer");
		}
	} else {
		const char* _tmp7;
		_tmp7 = NULL;
		return (_tmp7 = self->priv->set_value_function, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7));
	}
}


static void vala_struct_set_get_value_function (ValaStruct* self, const char* function) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->get_value_function = (_tmp1 = (_tmp0 = function, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL), _tmp1);
}


static void vala_struct_set_set_value_function (ValaStruct* self, const char* function) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->set_value_function = (_tmp1 = (_tmp0 = function, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL), _tmp1);
}


static char* vala_struct_real_get_default_value (ValaTypeSymbol* base) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	if (self->priv->default_value != NULL) {
		const char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = self->priv->default_value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	}
	/* inherit default value from base type*/
	{
		GeeIterator* type_it;
		/* inherit default value from base type*/
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		/* inherit default value from base type*/
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			ValaStruct* _tmp3;
			ValaTypeSymbol* _tmp2;
			ValaStruct* st;
			/* inherit default value from base type*/
			type = (ValaDataType*) gee_iterator_get (type_it);
			_tmp3 = NULL;
			_tmp2 = NULL;
			st = (_tmp3 = (_tmp2 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp2) ? ((ValaStruct*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
			if (st != NULL) {
				char* _tmp4;
				_tmp4 = NULL;
				return (_tmp4 = vala_typesymbol_get_default_value ((ValaTypeSymbol*) st), (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp4);
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	return NULL;
}


static void vala_struct_set_default_value (ValaStruct* self, const char* value) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->default_value = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->default_value = (g_free (self->priv->default_value), NULL), _tmp1);
}


/**
 * Adds the specified struct to the list of base types of this struct.
 *
 * @param type a class or interface reference
 */
void vala_struct_add_base_type (ValaStruct* self, ValaDataType* type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	gee_collection_add ((GeeCollection*) self->priv->base_types, type);
	vala_code_node_set_parent_node ((ValaCodeNode*) type, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeList* vala_struct_get_base_types (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->base_types);
}


static gint vala_struct_real_get_type_parameter_index (ValaTypeSymbol* base, const char* name) {
	ValaStruct * self;
	gint i;
	self = (ValaStruct*) base;
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeIterator* p_it;
		p_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_parameters);
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (p_it);
			if (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) p), name) == 0) {
				gint _tmp0;
				return (_tmp0 = (i), (p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL)), (p_it == NULL) ? NULL : (p_it = (gee_collection_object_unref (p_it), NULL)), _tmp0);
			}
			i++;
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(p_it == NULL) ? NULL : (p_it = (gee_collection_object_unref (p_it), NULL));
	}
	return -1;
}


/**
 * Returns whether this struct is a simple type, i.e. whether
 * instances are passed by value.
 */
gboolean vala_struct_is_simple_type (ValaStruct* self) {
	ValaAttribute* _tmp4;
	gboolean _tmp5;
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GeeIterator* type_it;
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			ValaStruct* _tmp1;
			ValaTypeSymbol* _tmp0;
			ValaStruct* st;
			gboolean _tmp2;
			type = (ValaDataType*) gee_iterator_get (type_it);
			_tmp1 = NULL;
			_tmp0 = NULL;
			st = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (type), VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1));
			_tmp2 = FALSE;
			if (st != NULL) {
				_tmp2 = vala_struct_is_simple_type (st);
			} else {
				_tmp2 = FALSE;
			}
			if (_tmp2) {
				gboolean _tmp3;
				return (_tmp3 = TRUE, (type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL)), _tmp3);
			}
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	_tmp4 = NULL;
	return (_tmp5 = (_tmp4 = vala_code_node_get_attribute ((ValaCodeNode*) self, "SimpleType")) != NULL, (_tmp4 == NULL) ? NULL : (_tmp4 = (vala_code_node_unref (_tmp4), NULL)), _tmp5);
}


/**
 * Marks this struct as simple type, i.e. instances will be passed by
 * value.
 */
void vala_struct_set_simple_type (ValaStruct* self, gboolean simple_type) {
	g_return_if_fail (self != NULL);
	((ValaCodeNode*) self)->attributes = g_list_append (((ValaCodeNode*) self)->attributes, vala_attribute_new ("SimpleType", NULL));
}


static void vala_struct_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->base_types); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaDataType*) gee_list_get ((GeeList*) self->priv->base_types, i)) == old_type, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set ((GeeList*) self->priv->base_types, i, new_type);
				return;
			}
		}
	}
}


static gboolean vala_struct_real_is_subtype_of (ValaTypeSymbol* base, ValaTypeSymbol* t) {
	ValaStruct * self;
	self = (ValaStruct*) base;
	g_return_val_if_fail (t != NULL, FALSE);
	if (VALA_TYPESYMBOL (self) == t) {
		return TRUE;
	}
	{
		GeeIterator* base_type_it;
		base_type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (base_type_it)) {
			ValaDataType* base_type;
			gboolean _tmp1;
			base_type = (ValaDataType*) gee_iterator_get (base_type_it);
			_tmp1 = FALSE;
			if (vala_data_type_get_data_type (base_type) != NULL) {
				_tmp1 = vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (base_type), t);
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL)), (base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL)), _tmp2);
			}
			(base_type == NULL) ? NULL : (base_type = (vala_code_node_unref (base_type), NULL));
		}
		(base_type_it == NULL) ? NULL : (base_type_it = (gee_collection_object_unref (base_type_it), NULL));
	}
	return FALSE;
}


char* vala_struct_get_default_copy_function (ValaStruct* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat (_tmp0 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self), "copy", NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_struct_real_get_copy_function (ValaTypeSymbol* base) {
	ValaStruct * self;
	const char* _tmp1;
	self = (ValaStruct*) base;
	if (self->priv->copy_function == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->copy_function = (_tmp0 = vala_struct_get_default_copy_function (self), self->priv->copy_function = (g_free (self->priv->copy_function), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->copy_function, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


void vala_struct_set_copy_function (ValaStruct* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->copy_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->copy_function = (g_free (self->priv->copy_function), NULL), _tmp1);
}


char* vala_struct_get_default_destroy_function (ValaStruct* self) {
	char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat (_tmp0 = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) self), "destroy", NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp1);
}


static char* vala_struct_real_get_destroy_function (ValaTypeSymbol* base) {
	ValaStruct * self;
	const char* _tmp1;
	self = (ValaStruct*) base;
	if (self->priv->destroy_function == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		self->priv->destroy_function = (_tmp0 = vala_struct_get_default_destroy_function (self), self->priv->destroy_function = (g_free (self->priv->destroy_function), NULL), _tmp0);
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->destroy_function, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1));
}


void vala_struct_set_destroy_function (ValaStruct* self, const char* name) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (name != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->destroy_function = (_tmp1 = (_tmp0 = name, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->destroy_function = (g_free (self->priv->destroy_function), NULL), _tmp1);
}


gboolean vala_struct_is_disposable (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	if (self->priv->destroy_function != NULL) {
		return TRUE;
	}
	{
		GeeIterator* f_it;
		f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			gboolean _tmp1;
			f = (ValaField*) gee_iterator_get (f_it);
			_tmp1 = FALSE;
			if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
				_tmp1 = vala_data_type_is_disposable (vala_field_get_field_type (f));
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				gboolean _tmp2;
				return (_tmp2 = TRUE, (f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL)), (f_it == NULL) ? NULL : (f_it = (gee_collection_object_unref (f_it), NULL)), _tmp2);
			}
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(f_it == NULL) ? NULL : (f_it = (gee_collection_object_unref (f_it), NULL));
	}
	return FALSE;
}


static gboolean vala_struct_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaStruct * self;
	ValaSourceFile* _tmp1;
	ValaSourceFile* old_source_file;
	ValaSymbol* _tmp2;
	ValaSymbol* old_symbol;
	ValaStruct* _tmp3;
	ValaStruct* old_struct;
	ValaStruct* _tmp5;
	ValaStruct* _tmp4;
	gboolean _tmp6;
	gboolean _tmp7;
	gboolean _tmp8;
	ValaStruct* _tmp12;
	ValaStruct* _tmp11;
	gboolean _tmp13;
	self = (ValaStruct*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_struct_process_attributes (self);
	_tmp1 = NULL;
	old_source_file = (_tmp1 = vala_semantic_analyzer_get_current_source_file (analyzer), (_tmp1 == NULL) ? NULL : vala_source_file_ref (_tmp1));
	_tmp2 = NULL;
	old_symbol = (_tmp2 = vala_semantic_analyzer_get_current_symbol (analyzer), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp3 = NULL;
	old_struct = (_tmp3 = analyzer->current_struct, (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3));
	if (vala_code_node_get_source_reference ((ValaCodeNode*) self) != NULL) {
		vala_semantic_analyzer_set_current_source_file (analyzer, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	_tmp5 = NULL;
	_tmp4 = NULL;
	analyzer->current_struct = (_tmp5 = (_tmp4 = self, (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4)), (analyzer->current_struct == NULL) ? NULL : (analyzer->current_struct = (vala_code_node_unref (analyzer->current_struct), NULL)), _tmp5);
	{
		GeeIterator* type_it;
		type_it = gee_iterable_iterator ((GeeIterable*) self->priv->base_types);
		while (gee_iterator_next (type_it)) {
			ValaDataType* type;
			type = (ValaDataType*) gee_iterator_get (type_it);
			vala_code_node_check ((ValaCodeNode*) type, analyzer);
			(type == NULL) ? NULL : (type = (vala_code_node_unref (type), NULL));
		}
		(type_it == NULL) ? NULL : (type_it = (gee_collection_object_unref (type_it), NULL));
	}
	{
		GeeIterator* p_it;
		p_it = gee_iterable_iterator ((GeeIterable*) self->priv->type_parameters);
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = (ValaTypeParameter*) gee_iterator_get (p_it);
			vala_code_node_check ((ValaCodeNode*) p, analyzer);
			(p == NULL) ? NULL : (p = (vala_code_node_unref (p), NULL));
		}
		(p_it == NULL) ? NULL : (p_it = (gee_collection_object_unref (p_it), NULL));
	}
	{
		GeeIterator* f_it;
		f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (f_it);
			vala_code_node_check ((ValaCodeNode*) f, analyzer);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(f_it == NULL) ? NULL : (f_it = (gee_collection_object_unref (f_it), NULL));
	}
	{
		GeeIterator* c_it;
		c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = (ValaConstant*) gee_iterator_get (c_it);
			vala_code_node_check ((ValaCodeNode*) c, analyzer);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(c_it == NULL) ? NULL : (c_it = (gee_collection_object_unref (c_it), NULL));
	}
	{
		GeeIterator* m_it;
		m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (m_it);
			vala_code_node_check ((ValaCodeNode*) m, analyzer);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(m_it == NULL) ? NULL : (m_it = (gee_collection_object_unref (m_it), NULL));
	}
	_tmp6 = FALSE;
	_tmp7 = FALSE;
	_tmp8 = FALSE;
	if (!vala_symbol_get_external ((ValaSymbol*) self)) {
		_tmp8 = !vala_symbol_get_external_package ((ValaSymbol*) self);
	} else {
		_tmp8 = FALSE;
	}
	if (_tmp8) {
		GeeList* _tmp9;
		_tmp9 = NULL;
		_tmp7 = gee_collection_get_size ((GeeCollection*) (_tmp9 = vala_struct_get_base_types (self))) == 0;
		(_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL));
	} else {
		_tmp7 = FALSE;
	}
	if (_tmp7) {
		GeeList* _tmp10;
		_tmp10 = NULL;
		_tmp6 = gee_collection_get_size ((GeeCollection*) (_tmp10 = vala_struct_get_fields (self))) == 0;
		(_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL));
	} else {
		_tmp6 = FALSE;
	}
	if (_tmp6) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "structs cannot be empty");
	}
	vala_semantic_analyzer_set_current_source_file (analyzer, old_source_file);
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	_tmp12 = NULL;
	_tmp11 = NULL;
	analyzer->current_struct = (_tmp12 = (_tmp11 = old_struct, (_tmp11 == NULL) ? NULL : vala_code_node_ref (_tmp11)), (analyzer->current_struct == NULL) ? NULL : (analyzer->current_struct = (vala_code_node_unref (analyzer->current_struct), NULL)), _tmp12);
	return (_tmp13 = !vala_code_node_get_error ((ValaCodeNode*) self), (old_source_file == NULL) ? NULL : (old_source_file = (vala_source_file_unref (old_source_file), NULL)), (old_symbol == NULL) ? NULL : (old_symbol = (vala_code_node_unref (old_symbol), NULL)), (old_struct == NULL) ? NULL : (old_struct = (vala_code_node_unref (old_struct), NULL)), _tmp13);
}


ValaMethod* vala_struct_get_default_construction_method (ValaStruct* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_default_construction_method;
}


void vala_struct_set_default_construction_method (ValaStruct* self, ValaMethod* value) {
	ValaMethod* _tmp2;
	ValaMethod* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_default_construction_method = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_default_construction_method == NULL) ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL)), _tmp2);
}


static void vala_struct_class_init (ValaStructClass * klass) {
	vala_struct_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_struct_finalize;
	g_type_class_add_private (klass, sizeof (ValaStructPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_struct_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_struct_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_struct_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_struct_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_struct_real_get_lower_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_upper_case_cname = vala_struct_real_get_upper_case_cname;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_signature = vala_struct_real_get_type_signature;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_struct_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_struct_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_struct_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_struct_real_get_set_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_default_value = vala_struct_real_get_default_value;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_parameter_index = vala_struct_real_get_type_parameter_index;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_struct_real_replace_type;
	VALA_TYPESYMBOL_CLASS (klass)->is_subtype_of = vala_struct_real_is_subtype_of;
	VALA_TYPESYMBOL_CLASS (klass)->get_copy_function = vala_struct_real_get_copy_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_destroy_function = vala_struct_real_get_destroy_function;
	VALA_CODE_NODE_CLASS (klass)->check = vala_struct_real_check;
}


static void vala_struct_instance_init (ValaStruct * self) {
	self->priv = VALA_STRUCT_GET_PRIVATE (self);
	self->priv->type_parameters = (GeeList*) gee_array_list_new (VALA_TYPE_TYPEPARAMETER, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->constants = (GeeList*) gee_array_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->base_types = (GeeList*) gee_array_list_new (VALA_TYPE_DATA_TYPE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->default_value = NULL;
}


static void vala_struct_finalize (ValaCodeNode* obj) {
	ValaStruct * self;
	self = VALA_STRUCT (obj);
	(self->priv->type_parameters == NULL) ? NULL : (self->priv->type_parameters = (gee_collection_object_unref (self->priv->type_parameters), NULL));
	(self->priv->constants == NULL) ? NULL : (self->priv->constants = (gee_collection_object_unref (self->priv->constants), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->base_types == NULL) ? NULL : (self->priv->base_types = (gee_collection_object_unref (self->priv->base_types), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	self->priv->const_cname = (g_free (self->priv->const_cname), NULL);
	self->priv->type_id = (g_free (self->priv->type_id), NULL);
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL);
	self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL);
	self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL);
	self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL);
	self->priv->default_value = (g_free (self->priv->default_value), NULL);
	self->priv->type_signature = (g_free (self->priv->type_signature), NULL);
	self->priv->copy_function = (g_free (self->priv->copy_function), NULL);
	self->priv->destroy_function = (g_free (self->priv->destroy_function), NULL);
	(self->priv->_default_construction_method == NULL) ? NULL : (self->priv->_default_construction_method = (vala_code_node_unref (self->priv->_default_construction_method), NULL));
	VALA_CODE_NODE_CLASS (vala_struct_parent_class)->finalize (obj);
}


GType vala_struct_get_type (void) {
	static GType vala_struct_type_id = 0;
	if (vala_struct_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStructClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_struct_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStruct), 0, (GInstanceInitFunc) vala_struct_instance_init, NULL };
		vala_struct_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaStruct", &g_define_type_info, 0);
	}
	return vala_struct_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




