/* valasourcefile.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valasourcefile.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <glib/gstdio.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/hashset.h>
#include <gee/readonlylist.h>
#include <gee/readonlyset.h>
#include <vala/valasourcefilecycle.h>
#include <vala/valacodecontext.h>
#include <vala/valadatatype.h>
#include <vala/valasymbol.h>
#include <vala/valaerrorcode.h>
#include <vala/valatypesymbol.h>
#include <vala/valamethod.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valaconstant.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>
#include <vala/valausingdirective.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valareport.h>
#include <gobject/gvaluecollector.h>




static glong string_get_length (const char* self);
struct _ValaSourceFilePrivate {
	char* _filename;
	char* _comment;
	gboolean _external_package;
	ValaSourceFileCycle* _cycle;
	gboolean _is_cycle_head;
	gint _mark;
	ValaCodeContext* _context;
	GeeList* using_directives;
	GeeList* nodes;
	char* cheader_filename;
	char* csource_filename;
	char* cinclude_filename;
	GeeList* header_external_includes;
	GeeList* header_internal_includes;
	GeeList* source_external_includes;
	GeeList* source_internal_includes;
	GeeList* header_internal_full_dependencies;
	GeeList* header_internal_dependencies;
	GeeSet* source_symbol_dependencies;
	GeeArrayList* source_array;
	GMappedFile* mapped_file;
	char* _content;
};

#define VALA_SOURCE_FILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_FILE, ValaSourceFilePrivate))
enum  {
	VALA_SOURCE_FILE_DUMMY_PROPERTY
};
static gboolean vala_source_file_same_symbol (ValaSourceFile* self, ValaSymbol* sym1, ValaSymbol* sym2);
static char* vala_source_file_get_subdir (ValaSourceFile* self);
static char* vala_source_file_get_destination_directory (ValaSourceFile* self);
static char* vala_source_file_get_basename (ValaSourceFile* self);
static void vala_source_file_read_source_file (ValaSourceFile* self);
static void vala_source_file_read_source_lines (ValaSourceFile* self, const char* cont);
static gpointer vala_source_file_parent_class = NULL;
static void vala_source_file_finalize (ValaSourceFile* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static glong string_get_length (const char* self) {
	g_return_val_if_fail (self != NULL, 0L);
	return g_utf8_strlen (self, -1);
}


/**
 * Creates a new source file.
 *
 * @param filename source file name
 * @param pkg      true if this is a VAPI package file
 * @return         newly created source file
 */
ValaSourceFile* vala_source_file_construct (GType object_type, ValaCodeContext* context, const char* filename, gboolean pkg, const char* content) {
	ValaSourceFile* self;
	g_return_val_if_fail (context != NULL, NULL);
	g_return_val_if_fail (filename != NULL, NULL);
	self = (ValaSourceFile*) g_type_create_instance (object_type);
	vala_source_file_set_filename (self, filename);
	vala_source_file_set_external_package (self, pkg);
	vala_source_file_set_context (self, context);
	vala_source_file_set_content (self, content);
	return self;
}


ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg, const char* content) {
	return vala_source_file_construct (VALA_TYPE_SOURCE_FILE, context, filename, pkg, content);
}


/**
 * Adds a new using directive with the specified namespace.
 *
 * @param ns reference to namespace
 */
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaUsingDirective* ns) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	{
		GeeIterator* using_directive_it;
		using_directive_it = gee_iterable_iterator ((GeeIterable*) self->priv->using_directives);
		while (gee_iterator_next (using_directive_it)) {
			ValaUsingDirective* using_directive;
			using_directive = (ValaUsingDirective*) gee_iterator_get (using_directive_it);
			if (vala_source_file_same_symbol (self, vala_using_directive_get_namespace_symbol (using_directive), vala_using_directive_get_namespace_symbol (ns))) {
				/* ignore duplicates*/
				(using_directive == NULL) ? NULL : (using_directive = (vala_code_node_unref (using_directive), NULL));
				(using_directive_it == NULL) ? NULL : (using_directive_it = (gee_collection_object_unref (using_directive_it), NULL));
				return;
			}
			(using_directive == NULL) ? NULL : (using_directive = (vala_code_node_unref (using_directive), NULL));
		}
		(using_directive_it == NULL) ? NULL : (using_directive_it = (gee_collection_object_unref (using_directive_it), NULL));
	}
	gee_collection_add ((GeeCollection*) self->priv->using_directives, ns);
}


static gboolean vala_source_file_same_symbol (ValaSourceFile* self, ValaSymbol* sym1, ValaSymbol* sym2) {
	ValaUnresolvedSymbol* _tmp2;
	ValaSymbol* _tmp1;
	ValaUnresolvedSymbol* unresolved_symbol1;
	ValaUnresolvedSymbol* _tmp4;
	ValaSymbol* _tmp3;
	ValaUnresolvedSymbol* unresolved_symbol2;
	gboolean _tmp5;
	gboolean _tmp7;
	g_return_val_if_fail (self != NULL, FALSE);
	if (sym1 == sym2) {
		return TRUE;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	unresolved_symbol1 = (_tmp2 = (_tmp1 = sym1, VALA_IS_UNRESOLVED_SYMBOL (_tmp1) ? ((ValaUnresolvedSymbol*) _tmp1) : NULL), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp4 = NULL;
	_tmp3 = NULL;
	unresolved_symbol2 = (_tmp4 = (_tmp3 = sym2, VALA_IS_UNRESOLVED_SYMBOL (_tmp3) ? ((ValaUnresolvedSymbol*) _tmp3) : NULL), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4));
	_tmp5 = FALSE;
	if (unresolved_symbol1 != NULL) {
		_tmp5 = unresolved_symbol2 != NULL;
	} else {
		_tmp5 = FALSE;
	}
	if (_tmp5) {
		if (vala_source_file_same_symbol (self, (ValaSymbol*) vala_unresolved_symbol_get_inner (unresolved_symbol1), (ValaSymbol*) vala_unresolved_symbol_get_inner (unresolved_symbol2))) {
			gboolean _tmp6;
			return (_tmp6 = (_vala_strcmp0 (vala_symbol_get_name ((ValaSymbol*) unresolved_symbol1), vala_symbol_get_name ((ValaSymbol*) unresolved_symbol2)) == 0), (unresolved_symbol1 == NULL) ? NULL : (unresolved_symbol1 = (vala_code_node_unref (unresolved_symbol1), NULL)), (unresolved_symbol2 == NULL) ? NULL : (unresolved_symbol2 = (vala_code_node_unref (unresolved_symbol2), NULL)), _tmp6);
		}
	}
	return (_tmp7 = FALSE, (unresolved_symbol1 == NULL) ? NULL : (unresolved_symbol1 = (vala_code_node_unref (unresolved_symbol1), NULL)), (unresolved_symbol2 == NULL) ? NULL : (unresolved_symbol2 = (vala_code_node_unref (unresolved_symbol2), NULL)), _tmp7);
}


/**
 * Returns a copy of the list of using directives.
 *
 * @return using directive list
 */
GeeList* vala_source_file_get_using_directives (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_USING_DIRECTIVE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->using_directives);
}


/**
 * Adds the specified code node to this source file.
 *
 * @param node a code node
 */
void vala_source_file_add_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	gee_collection_add ((GeeCollection*) self->priv->nodes, node);
}


void vala_source_file_remove_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	gee_collection_remove ((GeeCollection*) self->priv->nodes, node);
}


/**
 * Returns a copy of the list of code nodes.
 *
 * @return code node list
 */
GeeList* vala_source_file_get_nodes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->nodes);
}


void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_source_file (visitor, self);
}


void vala_source_file_accept_children (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* ns_ref_it;
		ns_ref_it = gee_iterable_iterator ((GeeIterable*) self->priv->using_directives);
		while (gee_iterator_next (ns_ref_it)) {
			ValaUsingDirective* ns_ref;
			ns_ref = (ValaUsingDirective*) gee_iterator_get (ns_ref_it);
			vala_code_node_accept ((ValaCodeNode*) ns_ref, visitor);
			(ns_ref == NULL) ? NULL : (ns_ref = (vala_code_node_unref (ns_ref), NULL));
		}
		(ns_ref_it == NULL) ? NULL : (ns_ref_it = (gee_collection_object_unref (ns_ref_it), NULL));
	}
	{
		GeeIterator* node_it;
		node_it = gee_iterable_iterator ((GeeIterable*) self->priv->nodes);
		while (gee_iterator_next (node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (node_it);
			vala_code_node_accept (node, visitor);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(node_it == NULL) ? NULL : (node_it = (gee_collection_object_unref (node_it), NULL));
	}
}


static char* vala_source_file_get_subdir (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_code_context_get_basedir (self->priv->_context) == NULL) {
		return g_strdup ("");
	}
	/* filename and basedir are already canonicalized*/
	if (g_str_has_prefix (self->priv->_filename, vala_code_context_get_basedir (self->priv->_context))) {
		char* basename;
		char* _tmp1;
		char* subdir;
		char* _tmp4;
		basename = g_path_get_basename (self->priv->_filename);
		_tmp1 = NULL;
		subdir = (_tmp1 = g_utf8_offset_to_pointer (self->priv->_filename, g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1)), g_strndup (_tmp1, g_utf8_offset_to_pointer (_tmp1, (g_utf8_strlen (self->priv->_filename, -1) - g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1)) - g_utf8_strlen (basename, -1)) - _tmp1));
		while (g_utf8_get_char (g_utf8_offset_to_pointer (subdir, 0)) == '/') {
			char* _tmp3;
			const char* _tmp2;
			_tmp3 = NULL;
			_tmp2 = NULL;
			subdir = (_tmp3 = (_tmp2 = g_utf8_offset_to_pointer (subdir, (glong) 1), (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), subdir = (g_free (subdir), NULL), _tmp3);
		}
		_tmp4 = NULL;
		return (_tmp4 = subdir, basename = (g_free (basename), NULL), _tmp4);
	}
	return g_strdup ("");
}


static char* vala_source_file_get_destination_directory (ValaSourceFile* self) {
	char* _tmp1;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	if (vala_code_context_get_directory (self->priv->_context) == NULL) {
		return vala_source_file_get_subdir (self);
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s/%s", vala_code_context_get_directory (self->priv->_context), _tmp1 = vala_source_file_get_subdir (self)), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
}


static char* vala_source_file_get_basename (ValaSourceFile* self) {
	glong dot;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	dot = g_utf8_pointer_to_offset (self->priv->_filename, g_utf8_strrchr (self->priv->_filename, (glong) (-1), (gunichar) '.'));
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_path_get_basename (_tmp1 = (_tmp0 = g_utf8_offset_to_pointer (self->priv->_filename, (glong) 0), g_strndup (_tmp0, g_utf8_offset_to_pointer (_tmp0, dot) - _tmp0))), _tmp1 = (g_free (_tmp1), NULL), _tmp2);
}


char* vala_source_file_get_relative_filename (ValaSourceFile* self) {
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strconcat (_tmp0 = vala_source_file_get_subdir (self), _tmp1 = g_path_get_basename (self->priv->_filename), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
}


/**
 * Returns the filename to use when generating C header files.
 *
 * @return generated C header filename
 */
char* vala_source_file_get_cheader_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cheader_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cheader_filename = (_tmp2 = g_strdup_printf ("%s%s.h", _tmp0 = vala_source_file_get_destination_directory (self), _tmp1 = vala_source_file_get_basename (self)), self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cheader_filename, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


/**
 * Returns the filename to use when generating C source files.
 *
 * @return generated C source filename
 */
char* vala_source_file_get_csource_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->csource_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->csource_filename = (_tmp2 = g_strdup_printf ("%s%s.c", _tmp0 = vala_source_file_get_destination_directory (self), _tmp1 = vala_source_file_get_basename (self)), self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->csource_filename, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


/**
 * Returns the filename to use when including the generated C header
 * file.
 *
 * @return C header filename to include
 */
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cinclude_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cinclude_filename = (_tmp2 = g_strdup_printf ("%s%s.h", _tmp0 = vala_source_file_get_subdir (self), _tmp1 = vala_source_file_get_basename (self)), self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cinclude_filename, (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
}


/**
 * Adds the specified symbol to the list of symbols code in this source
 * file depends on.
 *
 * TODO Move source and header file dependency analysis to
 * code generator.
 *
 * @param sym      a symbol
 * @param dep_type type of dependency
 */
void vala_source_file_add_symbol_dependency (ValaSourceFile* self, ValaSymbol* sym, ValaSourceFileDependencyType dep_type) {
	ValaSymbol* s;
	g_return_if_fail (self != NULL);
	if (self->priv->_external_package) {
		return;
	}
	s = NULL;
	if (VALA_IS_ERROR_CODE (sym)) {
		ValaSymbol* _tmp1;
		ValaSymbol* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		s = (_tmp1 = (_tmp0 = vala_symbol_get_parent_symbol (sym), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0)), (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp1);
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean _tmp5;
		_tmp2 = FALSE;
		_tmp3 = FALSE;
		_tmp4 = FALSE;
		_tmp5 = FALSE;
		if (VALA_IS_TYPESYMBOL (sym)) {
			_tmp5 = TRUE;
		} else {
			_tmp5 = VALA_IS_METHOD (sym);
		}
		if (_tmp5) {
			_tmp4 = TRUE;
		} else {
			_tmp4 = VALA_IS_FIELD (sym);
		}
		if (_tmp4) {
			_tmp3 = TRUE;
		} else {
			_tmp3 = VALA_IS_PROPERTY (sym);
		}
		if (_tmp3) {
			_tmp2 = TRUE;
		} else {
			_tmp2 = VALA_IS_CONSTANT (sym);
		}
		if (_tmp2) {
			ValaSymbol* _tmp7;
			ValaSymbol* _tmp6;
			_tmp7 = NULL;
			_tmp6 = NULL;
			s = (_tmp7 = (_tmp6 = sym, (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp7);
		} else {
			if (VALA_IS_FORMAL_PARAMETER (sym)) {
				ValaFormalParameter* _tmp8;
				ValaFormalParameter* fp;
				ValaSymbol* _tmp10;
				ValaSymbol* _tmp9;
				_tmp8 = NULL;
				fp = (_tmp8 = VALA_FORMAL_PARAMETER (sym), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
				_tmp10 = NULL;
				_tmp9 = NULL;
				s = (_tmp10 = (_tmp9 = (ValaSymbol*) vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (fp)), (_tmp9 == NULL) ? NULL : vala_code_node_ref (_tmp9)), (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), _tmp10);
				if (s == NULL) {
					/* generic type parameter */
					(fp == NULL) ? NULL : (fp = (vala_code_node_unref (fp), NULL));
					(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
					return;
				}
				(fp == NULL) ? NULL : (fp = (vala_code_node_unref (fp), NULL));
			} else {
				(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
				return;
			}
		}
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE) {
		gee_collection_add ((GeeCollection*) self->priv->source_symbol_dependencies, s);
		if (vala_symbol_get_external_package (s)) {
			{
				GeeList* _tmp11;
				GeeIterator* _tmp12;
				GeeIterator* fn_it;
				_tmp11 = NULL;
				_tmp12 = NULL;
				fn_it = (_tmp12 = gee_iterable_iterator ((GeeIterable*) (_tmp11 = vala_symbol_get_cheader_filenames (s))), (_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL)), _tmp12);
				while (gee_iterator_next (fn_it)) {
					char* fn;
					fn = (char*) gee_iterator_get (fn_it);
					gee_collection_add ((GeeCollection*) self->priv->source_external_includes, fn);
					fn = (g_free (fn), NULL);
				}
				(fn_it == NULL) ? NULL : (fn_it = (gee_collection_object_unref (fn_it), NULL));
			}
		} else {
			{
				GeeList* _tmp13;
				GeeIterator* _tmp14;
				GeeIterator* fn_it;
				_tmp13 = NULL;
				_tmp14 = NULL;
				fn_it = (_tmp14 = gee_iterable_iterator ((GeeIterable*) (_tmp13 = vala_symbol_get_cheader_filenames (s))), (_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL)), _tmp14);
				while (gee_iterator_next (fn_it)) {
					char* fn;
					fn = (char*) gee_iterator_get (fn_it);
					gee_collection_add ((GeeCollection*) self->priv->source_internal_includes, fn);
					fn = (g_free (fn), NULL);
				}
				(fn_it == NULL) ? NULL : (fn_it = (gee_collection_object_unref (fn_it), NULL));
			}
		}
		(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
		return;
	}
	if (vala_symbol_get_external_package (s)) {
		/* external package */
		{
			GeeList* _tmp15;
			GeeIterator* _tmp16;
			GeeIterator* fn_it;
			/* external package */
			_tmp15 = NULL;
			_tmp16 = NULL;
			fn_it = (_tmp16 = gee_iterable_iterator ((GeeIterable*) (_tmp15 = vala_symbol_get_cheader_filenames (s))), (_tmp15 == NULL) ? NULL : (_tmp15 = (gee_collection_object_unref (_tmp15), NULL)), _tmp16);
			/* external package */
			while (gee_iterator_next (fn_it)) {
				char* fn;
				/* external package */
				fn = (char*) gee_iterator_get (fn_it);
				gee_collection_add ((GeeCollection*) self->priv->header_external_includes, fn);
				fn = (g_free (fn), NULL);
			}
			(fn_it == NULL) ? NULL : (fn_it = (gee_collection_object_unref (fn_it), NULL));
		}
		(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
		return;
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL) {
		{
			GeeList* _tmp17;
			GeeIterator* _tmp18;
			GeeIterator* fn_it;
			_tmp17 = NULL;
			_tmp18 = NULL;
			fn_it = (_tmp18 = gee_iterable_iterator ((GeeIterable*) (_tmp17 = vala_symbol_get_cheader_filenames (s))), (_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL)), _tmp18);
			while (gee_iterator_next (fn_it)) {
				char* fn;
				fn = (char*) gee_iterator_get (fn_it);
				gee_collection_add ((GeeCollection*) self->priv->header_internal_includes, fn);
				fn = (g_free (fn), NULL);
			}
			(fn_it == NULL) ? NULL : (fn_it = (gee_collection_object_unref (fn_it), NULL));
		}
		gee_collection_add ((GeeCollection*) self->priv->header_internal_full_dependencies, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) s)));
	}
	gee_collection_add ((GeeCollection*) self->priv->header_internal_dependencies, vala_source_reference_get_file (vala_code_node_get_source_reference ((ValaCodeNode*) s)));
	(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
}


/**
 * Adds the symbols that define the specified type to the list of
 * symbols code in this source file depends on.
 *
 * TODO Move source and header file dependency analysis to
 * code generator.
 *
 * @param type     a data type
 * @param dep_type type of dependency
 */
void vala_source_file_add_type_dependency (ValaSourceFile* self, ValaDataType* type, ValaSourceFileDependencyType dep_type) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (type != NULL);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* type_symbol_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		type_symbol_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_data_type_get_symbols (type))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (type_symbol_it)) {
			ValaSymbol* type_symbol;
			type_symbol = (ValaSymbol*) gee_iterator_get (type_symbol_it);
			vala_source_file_add_symbol_dependency (self, type_symbol, dep_type);
			(type_symbol == NULL) ? NULL : (type_symbol = (vala_code_node_unref (type_symbol), NULL));
		}
		(type_symbol_it == NULL) ? NULL : (type_symbol_it = (gee_collection_object_unref (type_symbol_it), NULL));
	}
}


/**
 * Returns the list of external includes the generated C header file
 * requires.
 *
 * @return external include list for C header file
 */
GeeList* vala_source_file_get_header_external_includes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->header_external_includes);
}


/**
 * Adds the specified filename to the list of package-internal includes
 * the generated C header file requires.
 *
 * @param include internal include for C header file
 */
void vala_source_file_add_header_internal_include (ValaSourceFile* self, const char* include) {
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	g_return_if_fail (include != NULL);
	/* skip includes to self */
	_tmp0 = NULL;
	if ((_tmp1 = _vala_strcmp0 (include, _tmp0 = vala_source_file_get_cinclude_filename (self)) != 0, _tmp0 = (g_free (_tmp0), NULL), _tmp1)) {
		gee_collection_add ((GeeCollection*) self->priv->header_internal_includes, include);
	}
}


/**
 * Returns the list of package-internal includes the generated C header
 * file requires.
 *
 * @return internal include list for C header file
 */
GeeList* vala_source_file_get_header_internal_includes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->header_internal_includes);
}


/**
 * Returns the list of external includes the generated C source file
 * requires.
 *
 * @return include list for C source file
 */
GeeList* vala_source_file_get_source_external_includes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->source_external_includes);
}


/**
 * Returns the list of package-internal includes the generated C source
 * file requires.
 *
 * @return include list for C source file
 */
GeeList* vala_source_file_get_source_internal_includes (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->source_internal_includes);
}


/**
 * Returns the list of source files the generated C header file requires
 * definitely.
 *
 * @return definite source file dependencies
 */
GeeList* vala_source_file_get_header_internal_full_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, self->priv->header_internal_full_dependencies);
}


/**
 * Returns the list of source files the generated C header file loosely
 * depends on.
 *
 * @return loose source file dependencies
 */
GeeList* vala_source_file_get_header_internal_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, self->priv->header_internal_dependencies);
}


GeeSet* vala_source_file_get_source_symbol_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeSet*) gee_read_only_set_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->source_symbol_dependencies);
}


/**
 * Returns the requested line from this file, loading it if needed.
 *
 * @param lineno 1-based line number
 * @return       the specified source line
 */
char* vala_source_file_get_source_line (ValaSourceFile* self, gint lineno) {
	gboolean _tmp0;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->source_array == NULL) {
		if (vala_source_file_get_content (self) != NULL) {
			vala_source_file_read_source_lines (self, vala_source_file_get_content (self));
		} else {
			vala_source_file_read_source_file (self);
		}
	}
	_tmp0 = FALSE;
	if (lineno < 1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = lineno > gee_collection_get_size ((GeeCollection*) self->priv->source_array);
	}
	if (_tmp0) {
		return NULL;
	}
	return (char*) gee_list_get ((GeeList*) self->priv->source_array, lineno - 1);
}


/**
 * Parses the input file into ::source_array.
 */
static void vala_source_file_read_source_file (ValaSourceFile* self) {
	GError * inner_error;
	char* cont;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	cont = NULL;
	{
		char* _tmp2;
		gboolean _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp0 = NULL;
		_tmp1 = g_file_get_contents (self->priv->_filename, &_tmp0, NULL, &inner_error);
		cont = (_tmp2 = _tmp0, cont = (g_free (cont), NULL), _tmp2);
		_tmp1;
		if (inner_error != NULL) {
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch6_g_file_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	goto __finally6;
	__catch6_g_file_error:
	{
		GError * fe;
		fe = inner_error;
		inner_error = NULL;
		{
			(fe == NULL) ? NULL : (fe = (g_error_free (fe), NULL));
			cont = (g_free (cont), NULL);
			return;
		}
	}
	__finally6:
	;
	vala_source_file_read_source_lines (self, cont);
	cont = (g_free (cont), NULL);
}


static void vala_source_file_read_source_lines (ValaSourceFile* self, const char* cont) {
	GeeArrayList* _tmp0;
	char** _tmp1;
	gint lines_length1;
	char** lines;
	gint idx;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cont != NULL);
	_tmp0 = NULL;
	self->priv->source_array = (_tmp0 = gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal), (self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL)), _tmp0);
	_tmp1 = NULL;
	lines = (_tmp1 = g_strsplit (cont, "\n", 0), lines_length1 = -1, _tmp1);
	idx = 0;
	for (idx = 0; lines[idx] != NULL; (idx = idx + 1)) {
		gee_collection_add ((GeeCollection*) self->priv->source_array, lines[idx]);
	}
	lines = (_vala_array_free (lines, lines_length1, (GDestroyNotify) g_free), NULL);
}


gchar* vala_source_file_get_mapped_contents (ValaSourceFile* self) {
	GError * inner_error;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	if (vala_source_file_get_content (self) != NULL) {
		return vala_source_file_get_content (self);
	}
	if (self->priv->mapped_file == NULL) {
		{
			GMappedFile* _tmp1;
			_tmp1 = NULL;
			self->priv->mapped_file = (_tmp1 = g_mapped_file_new (self->priv->_filename, FALSE, &inner_error), (self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL)), _tmp1);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch7_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
		}
		goto __finally7;
		__catch7_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp2;
				gchar* _tmp3;
				_tmp2 = NULL;
				vala_report_error (NULL, _tmp2 = g_strdup_printf ("Unable to map file `%s': %s", self->priv->_filename, e->message));
				_tmp2 = (g_free (_tmp2), NULL);
				return (_tmp3 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), _tmp3);
			}
		}
		__finally7:
		;
	}
	return g_mapped_file_get_contents (self->priv->mapped_file);
}


gsize vala_source_file_get_mapped_length (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, 0UL);
	if (vala_source_file_get_content (self) != NULL) {
		return (gsize) string_get_length (vala_source_file_get_content (self));
	}
	return (gsize) g_mapped_file_get_length (self->priv->mapped_file);
}


gboolean vala_source_file_check (ValaSourceFile* self, ValaSemanticAnalyzer* analyzer) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (analyzer != NULL, FALSE);
	{
		GeeIterator* node_it;
		node_it = gee_iterable_iterator ((GeeIterable*) self->priv->nodes);
		while (gee_iterator_next (node_it)) {
			ValaCodeNode* node;
			node = (ValaCodeNode*) gee_iterator_get (node_it);
			vala_code_node_check (node, analyzer);
			(node == NULL) ? NULL : (node = (vala_code_node_unref (node), NULL));
		}
		(node_it == NULL) ? NULL : (node_it = (gee_collection_object_unref (node_it), NULL));
	}
	return TRUE;
}


const char* vala_source_file_get_filename (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_filename;
}


void vala_source_file_set_filename (ValaSourceFile* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_filename = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_filename = (g_free (self->priv->_filename), NULL), _tmp2);
}


const char* vala_source_file_get_comment (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_comment;
}


void vala_source_file_set_comment (ValaSourceFile* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_comment = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_comment = (g_free (self->priv->_comment), NULL), _tmp2);
}


gboolean vala_source_file_get_external_package (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_external_package;
}


void vala_source_file_set_external_package (ValaSourceFile* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_external_package = value;
}


ValaSourceFileCycle* vala_source_file_get_cycle (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_cycle;
}


void vala_source_file_set_cycle (ValaSourceFile* self, ValaSourceFileCycle* value) {
	ValaSourceFileCycle* _tmp2;
	ValaSourceFileCycle* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_cycle = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_source_file_cycle_ref (_tmp1)), (self->priv->_cycle == NULL) ? NULL : (self->priv->_cycle = (vala_source_file_cycle_unref (self->priv->_cycle), NULL)), _tmp2);
}


gboolean vala_source_file_get_is_cycle_head (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_is_cycle_head;
}


void vala_source_file_set_is_cycle_head (ValaSourceFile* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_is_cycle_head = value;
}


gint vala_source_file_get_mark (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_mark;
}


void vala_source_file_set_mark (ValaSourceFile* self, gint value) {
	g_return_if_fail (self != NULL);
	self->priv->_mark = value;
}


ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_context;
}


void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value) {
	g_return_if_fail (self != NULL);
	self->priv->_context = value;
}


const char* vala_source_file_get_content (ValaSourceFile* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_content;
}


void vala_source_file_set_content (ValaSourceFile* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	GeeArrayList* _tmp3;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_content = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_content = (g_free (self->priv->_content), NULL), _tmp2);
	_tmp3 = NULL;
	self->priv->source_array = (_tmp3 = NULL, (self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL)), _tmp3);
}


static void vala_value_source_file_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_source_file_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_source_file_unref (value->data[0].v_pointer);
	}
}


static void vala_value_source_file_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_source_file_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_source_file_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_source_file_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaSourceFile* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_source_file_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_source_file_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaSourceFile** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_source_file_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecSourceFile* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_SOURCE_FILE), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_source_file (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_FILE), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_source_file (GValue* value, gpointer v_object) {
	ValaSourceFile* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_SOURCE_FILE));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_SOURCE_FILE));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_source_file_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_source_file_unref (old);
	}
}


static void vala_source_file_class_init (ValaSourceFileClass * klass) {
	vala_source_file_parent_class = g_type_class_peek_parent (klass);
	VALA_SOURCE_FILE_CLASS (klass)->finalize = vala_source_file_finalize;
	g_type_class_add_private (klass, sizeof (ValaSourceFilePrivate));
}


static void vala_source_file_instance_init (ValaSourceFile * self) {
	self->priv = VALA_SOURCE_FILE_GET_PRIVATE (self);
	self->priv->using_directives = (GeeList*) gee_array_list_new (VALA_TYPE_USING_DIRECTIVE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->nodes = (GeeList*) gee_array_list_new (VALA_TYPE_CODE_NODE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->cheader_filename = NULL;
	self->priv->csource_filename = NULL;
	self->priv->cinclude_filename = NULL;
	self->priv->header_external_includes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->header_internal_includes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->source_external_includes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->source_internal_includes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->header_internal_full_dependencies = (GeeList*) gee_array_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, g_direct_equal);
	self->priv->header_internal_dependencies = (GeeList*) gee_array_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, g_direct_equal);
	self->priv->source_symbol_dependencies = (GeeSet*) gee_hash_set_new (VALA_TYPE_SYMBOL, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_hash, g_direct_equal);
	self->priv->source_array = NULL;
	self->priv->mapped_file = NULL;
	self->priv->_content = NULL;
	self->ref_count = 1;
}


static void vala_source_file_finalize (ValaSourceFile* obj) {
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
	(self->priv->_cycle == NULL) ? NULL : (self->priv->_cycle = (vala_source_file_cycle_unref (self->priv->_cycle), NULL));
	(self->priv->using_directives == NULL) ? NULL : (self->priv->using_directives = (gee_collection_object_unref (self->priv->using_directives), NULL));
	(self->priv->nodes == NULL) ? NULL : (self->priv->nodes = (gee_collection_object_unref (self->priv->nodes), NULL));
	self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL);
	self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL);
	self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL);
	(self->priv->header_external_includes == NULL) ? NULL : (self->priv->header_external_includes = (gee_collection_object_unref (self->priv->header_external_includes), NULL));
	(self->priv->header_internal_includes == NULL) ? NULL : (self->priv->header_internal_includes = (gee_collection_object_unref (self->priv->header_internal_includes), NULL));
	(self->priv->source_external_includes == NULL) ? NULL : (self->priv->source_external_includes = (gee_collection_object_unref (self->priv->source_external_includes), NULL));
	(self->priv->source_internal_includes == NULL) ? NULL : (self->priv->source_internal_includes = (gee_collection_object_unref (self->priv->source_internal_includes), NULL));
	(self->priv->header_internal_full_dependencies == NULL) ? NULL : (self->priv->header_internal_full_dependencies = (gee_collection_object_unref (self->priv->header_internal_full_dependencies), NULL));
	(self->priv->header_internal_dependencies == NULL) ? NULL : (self->priv->header_internal_dependencies = (gee_collection_object_unref (self->priv->header_internal_dependencies), NULL));
	(self->priv->source_symbol_dependencies == NULL) ? NULL : (self->priv->source_symbol_dependencies = (gee_collection_object_unref (self->priv->source_symbol_dependencies), NULL));
	(self->priv->source_array == NULL) ? NULL : (self->priv->source_array = (gee_collection_object_unref (self->priv->source_array), NULL));
	(self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL));
	self->priv->_content = (g_free (self->priv->_content), NULL);
}


GType vala_source_file_get_type (void) {
	static GType vala_source_file_type_id = 0;
	if (vala_source_file_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_source_file_init, vala_value_source_file_free_value, vala_value_source_file_copy_value, vala_value_source_file_peek_pointer, "p", vala_value_source_file_collect_value, "p", vala_value_source_file_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceFileClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_file_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceFile), 0, (GInstanceInitFunc) vala_source_file_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_source_file_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaSourceFile", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_source_file_type_id;
}


gpointer vala_source_file_ref (gpointer instance) {
	ValaSourceFile* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_source_file_unref (gpointer instance) {
	ValaSourceFile* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_SOURCE_FILE_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}



GType vala_source_file_dependency_type_get_type (void) {
	static GType vala_source_file_dependency_type_type_id = 0;
	if (G_UNLIKELY (vala_source_file_dependency_type_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL", "header-full"}, {VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW", "header-shallow"}, {VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE", "source"}, {0, NULL, NULL}};
		vala_source_file_dependency_type_type_id = g_enum_register_static ("ValaSourceFileDependencyType", values);
	}
	return vala_source_file_dependency_type_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




