/* valamethodcall.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valamethodcall.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valamemberaccess.h>
#include <vala/valareport.h>
#include <vala/valasymbol.h>
#include <vala/valaobjecttype.h>
#include <vala/valacreationmethod.h>
#include <vala/valastruct.h>
#include <vala/valaobjectcreationexpression.h>
#include <vala/valaformalparameter.h>
#include <vala/valaarraytype.h>
#include <vala/valamethodtype.h>
#include <vala/valamethod.h>
#include <vala/valastringliteral.h>
#include <vala/valapointertype.h>
#include <vala/valavoidtype.h>
#include <vala/valaexpressionstatement.h>
#include <vala/valaforstatement.h>
#include <vala/valayieldstatement.h>
#include <vala/valalocalvariable.h>
#include <vala/valablock.h>
#include <vala/valadeclarationstatement.h>
#include <vala/valastatement.h>




struct _ValaMethodCallPrivate {
	ValaCCodeExpression* _delegate_target;
	GeeList* argument_list;
	GeeList* array_sizes;
};

#define VALA_METHOD_CALL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_METHOD_CALL, ValaMethodCallPrivate))
enum  {
	VALA_METHOD_CALL_DUMMY_PROPERTY
};
static void vala_method_call_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_call_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_method_call_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_method_call_real_is_pure (ValaExpression* base);
static gboolean vala_method_call_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_method_call_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_method_call_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_method_call_parent_class = NULL;
static void vala_method_call_finalize (ValaCodeNode* obj);



/**
 * Creates a new invocation expression.
 *
 * @param call             method to call
 * @param source_reference reference to source code
 * @return                 newly created invocation expression
 */
ValaMethodCall* vala_method_call_construct (GType object_type, ValaExpression* call, ValaSourceReference* source_reference) {
	ValaMethodCall* self;
	g_return_val_if_fail (call != NULL, NULL);
	self = (ValaMethodCall*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_method_call_set_call (self, call);
	return self;
}


ValaMethodCall* vala_method_call_new (ValaExpression* call, ValaSourceReference* source_reference) {
	return vala_method_call_construct (VALA_TYPE_METHOD_CALL, call, source_reference);
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_method_call_add_argument (ValaMethodCall* self, ValaExpression* arg) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (arg != NULL);
	gee_collection_add ((GeeCollection*) self->priv->argument_list, arg);
	vala_code_node_set_parent_node ((ValaCodeNode*) arg, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeList* vala_method_call_get_argument_list (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->argument_list);
}


/**
 * Add an array size C code expression.
 */
void vala_method_call_append_array_size (ValaMethodCall* self, ValaCCodeExpression* size) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (size != NULL);
	gee_collection_add ((GeeCollection*) self->priv->array_sizes, size);
}


/**
 * Get the C code expression for array sizes for all dimensions
 * ascending from left to right.
 */
GeeList* vala_method_call_get_array_sizes (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, self->priv->array_sizes);
}


static void vala_method_call_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_method_call (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_method_call_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_method_call_get_call (self), visitor);
	{
		GeeIterator* expr_it;
		expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (expr_it);
			vala_code_node_accept ((ValaCodeNode*) expr, visitor);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(expr_it == NULL) ? NULL : (expr_it = (gee_collection_object_unref (expr_it), NULL));
	}
}


static void vala_method_call_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaMethodCall * self;
	gint index;
	gboolean _tmp0;
	self = (ValaMethodCall*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_method_call_get_call (self) == old_node) {
		vala_method_call_set_call (self, new_node);
	}
	index = gee_list_index_of (self->priv->argument_list, old_node);
	_tmp0 = FALSE;
	if (index >= 0) {
		_tmp0 = vala_code_node_get_parent_node ((ValaCodeNode*) new_node) == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		gee_list_set ((GeeList*) self->priv->argument_list, index, new_node);
		vala_code_node_set_parent_node ((ValaCodeNode*) new_node, (ValaCodeNode*) self);
	}
}


static gboolean vala_method_call_real_is_pure (ValaExpression* base) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	return FALSE;
}


static gboolean vala_method_call_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaMethodCall * self;
	ValaDataType* target_object_type;
	ValaDataType* _tmp8;
	ValaDataType* mtype;
	gboolean _tmp11;
	GeeList* params;
	gboolean _tmp20;
	ValaExpression* last_arg;
	GeeList* args;
	GeeIterator* arg_it;
	gboolean _tmp27;
	ValaDataType* ret_type;
	GeeList* _tmp85;
	ValaDataType* _tmp89;
	gboolean may_throw;
	GeeList* _tmp93;
	gboolean _tmp103;
	self = (ValaMethodCall*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (!vala_code_node_check ((ValaCodeNode*) vala_method_call_get_call (self), analyzer)) {
		/* if method resolving didn't succeed, skip this check */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	/* type of target object*/
	target_object_type = NULL;
	if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* ma;
		_tmp2 = NULL;
		ma = (_tmp2 = VALA_MEMBER_ACCESS (vala_method_call_get_call (self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp4;
			char* _tmp3;
			gboolean _tmp5;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			_tmp4 = NULL;
			_tmp3 = NULL;
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp4 = g_strdup_printf ("Access to instance member `%s' denied", _tmp3 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_method_call_get_call (self)))));
			_tmp4 = (g_free (_tmp4), NULL);
			_tmp3 = (g_free (_tmp3), NULL);
			return (_tmp5 = FALSE, (ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), _tmp5);
		}
		if (vala_member_access_get_inner (ma) != NULL) {
			ValaDataType* _tmp7;
			ValaDataType* _tmp6;
			_tmp7 = NULL;
			_tmp6 = NULL;
			target_object_type = (_tmp7 = (_tmp6 = vala_expression_get_value_type (vala_member_access_get_inner (ma)), (_tmp6 == NULL) ? NULL : vala_code_node_ref (_tmp6)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), _tmp7);
		}
		(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
	}
	_tmp8 = NULL;
	mtype = (_tmp8 = vala_expression_get_value_type (vala_method_call_get_call (self)), (_tmp8 == NULL) ? NULL : vala_code_node_ref (_tmp8));
	if (VALA_IS_OBJECT_TYPE (mtype)) {
		ValaMethod* _tmp9;
		ValaCreationMethod* cm;
		/* constructor chain-up*/
		_tmp9 = NULL;
		cm = (_tmp9 = vala_semantic_analyzer_find_current_method (analyzer), VALA_IS_CREATION_METHOD (_tmp9) ? ((ValaCreationMethod*) _tmp9) : NULL);
		g_assert (cm != NULL);
		if (vala_creation_method_get_chain_up (cm)) {
			gboolean _tmp10;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Multiple constructor calls in the same constructor are not permitted");
			return (_tmp10 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp10);
		}
		vala_creation_method_set_chain_up (cm, TRUE);
		(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
	}
	_tmp11 = FALSE;
	if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
		gboolean _tmp12;
		gboolean _tmp13;
		_tmp12 = FALSE;
		_tmp13 = FALSE;
		if (VALA_IS_CREATION_METHOD (vala_expression_get_symbol_reference (vala_method_call_get_call (self)))) {
			_tmp13 = VALA_IS_STRUCT (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_method_call_get_call (self))));
		} else {
			_tmp13 = FALSE;
		}
		if ((_tmp13)) {
			_tmp12 = TRUE;
		} else {
			_tmp12 = VALA_IS_STRUCT (vala_expression_get_symbol_reference (vala_method_call_get_call (self)));
		}
		_tmp11 = (_tmp12);
	} else {
		_tmp11 = FALSE;
	}
	/* check for struct construction*/
	if (_tmp11) {
		ValaObjectCreationExpression* struct_creation_expression;
		gboolean _tmp16;
		struct_creation_expression = vala_object_creation_expression_new (VALA_MEMBER_ACCESS (vala_method_call_get_call (self)), vala_code_node_get_source_reference ((ValaCodeNode*) self));
		vala_object_creation_expression_set_struct_creation (struct_creation_expression, TRUE);
		{
			GeeList* _tmp14;
			GeeIterator* _tmp15;
			GeeIterator* arg_it;
			_tmp14 = NULL;
			_tmp15 = NULL;
			arg_it = (_tmp15 = gee_iterable_iterator ((GeeIterable*) (_tmp14 = vala_method_call_get_argument_list (self))), (_tmp14 == NULL) ? NULL : (_tmp14 = (gee_collection_object_unref (_tmp14), NULL)), _tmp15);
			while (gee_iterator_next (arg_it)) {
				ValaExpression* arg;
				arg = (ValaExpression*) gee_iterator_get (arg_it);
				vala_object_creation_expression_add_argument (struct_creation_expression, arg);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
		}
		vala_expression_set_target_type ((ValaExpression*) struct_creation_expression, vala_expression_get_target_type ((ValaExpression*) self));
		gee_collection_add ((GeeCollection*) analyzer->replaced_nodes, (ValaCodeNode*) self);
		vala_code_node_replace_expression (vala_code_node_get_parent_node ((ValaCodeNode*) self), (ValaExpression*) self, (ValaExpression*) struct_creation_expression);
		vala_code_node_check ((ValaCodeNode*) struct_creation_expression, analyzer);
		return (_tmp16 = TRUE, (struct_creation_expression == NULL) ? NULL : (struct_creation_expression = (vala_code_node_unref (struct_creation_expression), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp16);
	} else {
		gboolean _tmp17;
		_tmp17 = FALSE;
		if (VALA_IS_MEMBER_ACCESS (vala_method_call_get_call (self))) {
			_tmp17 = VALA_IS_CREATION_METHOD (vala_expression_get_symbol_reference (vala_method_call_get_call (self)));
		} else {
			_tmp17 = FALSE;
		}
		if (_tmp17) {
			ValaMethod* _tmp18;
			ValaCreationMethod* cm;
			/* constructor chain-up*/
			_tmp18 = NULL;
			cm = (_tmp18 = vala_semantic_analyzer_find_current_method (analyzer), VALA_IS_CREATION_METHOD (_tmp18) ? ((ValaCreationMethod*) _tmp18) : NULL);
			g_assert (cm != NULL);
			if (vala_creation_method_get_chain_up (cm)) {
				gboolean _tmp19;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Multiple constructor calls in the same constructor are not permitted");
				return (_tmp19 = FALSE, (cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), _tmp19);
			}
			vala_creation_method_set_chain_up (cm, TRUE);
			(cm == NULL) ? NULL : (cm = (vala_code_node_unref (cm), NULL));
		}
	}
	params = NULL;
	_tmp20 = FALSE;
	if (mtype != NULL) {
		_tmp20 = vala_data_type_is_invokable (mtype);
	} else {
		_tmp20 = FALSE;
	}
	if (_tmp20) {
		GeeList* _tmp21;
		_tmp21 = NULL;
		params = (_tmp21 = vala_data_type_get_parameters (mtype), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp21);
	} else {
		gboolean _tmp22;
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "invocation not supported in this context");
		return (_tmp22 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp22);
	}
	last_arg = NULL;
	args = vala_method_call_get_argument_list (self);
	arg_it = gee_iterable_iterator ((GeeIterable*) args);
	{
		GeeIterator* param_it;
		param_it = gee_iterable_iterator ((GeeIterable*) params);
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = (ValaFormalParameter*) gee_iterator_get (param_it);
			if (vala_formal_parameter_get_ellipsis (param)) {
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				break;
			}
			if (vala_formal_parameter_get_params_array (param)) {
				ValaArrayType* _tmp23;
				ValaArrayType* array_type;
				_tmp23 = NULL;
				array_type = (_tmp23 = VALA_ARRAY_TYPE (vala_formal_parameter_get_parameter_type (param)), (_tmp23 == NULL) ? NULL : vala_code_node_ref (_tmp23));
				while (gee_iterator_next (arg_it)) {
					ValaExpression* arg;
					arg = (ValaExpression*) gee_iterator_get (arg_it);
					/* store expected type for callback parameters */
					vala_expression_set_target_type (arg, vala_array_type_get_element_type (array_type));
					vala_data_type_set_value_owned (vala_expression_get_target_type (arg), vala_data_type_get_value_owned ((ValaDataType*) array_type));
					(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
				}
				(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
				(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
				break;
			}
			if (gee_iterator_next (arg_it)) {
				ValaExpression* arg;
				ValaDataType* _tmp24;
				ValaExpression* _tmp26;
				ValaExpression* _tmp25;
				arg = (ValaExpression*) gee_iterator_get (arg_it);
				/* store expected type for callback parameters */
				vala_expression_set_formal_target_type (arg, vala_formal_parameter_get_parameter_type (param));
				_tmp24 = NULL;
				vala_expression_set_target_type (arg, _tmp24 = vala_data_type_get_actual_type (vala_expression_get_formal_target_type (arg), target_object_type, (ValaCodeNode*) self));
				(_tmp24 == NULL) ? NULL : (_tmp24 = (vala_code_node_unref (_tmp24), NULL));
				_tmp26 = NULL;
				_tmp25 = NULL;
				last_arg = (_tmp26 = (_tmp25 = arg, (_tmp25 == NULL) ? NULL : vala_code_node_ref (_tmp25)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), _tmp26);
				(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
			}
			(param == NULL) ? NULL : (param = (vala_code_node_unref (param), NULL));
		}
		(param_it == NULL) ? NULL : (param_it = (gee_collection_object_unref (param_it), NULL));
	}
	_tmp27 = FALSE;
	if (VALA_IS_METHOD_TYPE (mtype)) {
		_tmp27 = vala_method_get_printf_format (vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype))));
	} else {
		_tmp27 = FALSE;
	}
	/* printf arguments*/
	if (_tmp27) {
		ValaStringLiteral* format_literal;
		format_literal = NULL;
		if (last_arg != NULL) {
			ValaStringLiteral* _tmp30;
			ValaStringLiteral* _tmp29;
			ValaExpression* _tmp28;
			/* use last argument as format string*/
			_tmp30 = NULL;
			_tmp29 = NULL;
			_tmp28 = NULL;
			format_literal = (_tmp30 = (_tmp29 = (_tmp28 = last_arg, VALA_IS_STRING_LITERAL (_tmp28) ? ((ValaStringLiteral*) _tmp28) : NULL), (_tmp29 == NULL) ? NULL : vala_code_node_ref (_tmp29)), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), _tmp30);
		} else {
			ValaMemberAccess* _tmp32;
			ValaExpression* _tmp31;
			ValaMemberAccess* ma;
			/* use instance as format string for string.printf (...)*/
			_tmp32 = NULL;
			_tmp31 = NULL;
			ma = (_tmp32 = (_tmp31 = vala_method_call_get_call (self), VALA_IS_MEMBER_ACCESS (_tmp31) ? ((ValaMemberAccess*) _tmp31) : NULL), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32));
			if (ma != NULL) {
				ValaStringLiteral* _tmp35;
				ValaStringLiteral* _tmp34;
				ValaExpression* _tmp33;
				_tmp35 = NULL;
				_tmp34 = NULL;
				_tmp33 = NULL;
				format_literal = (_tmp35 = (_tmp34 = (_tmp33 = vala_member_access_get_inner (ma), VALA_IS_STRING_LITERAL (_tmp33) ? ((ValaStringLiteral*) _tmp33) : NULL), (_tmp34 == NULL) ? NULL : vala_code_node_ref (_tmp34)), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), _tmp35);
			}
			(ma == NULL) ? NULL : (ma = (vala_code_node_unref (ma), NULL));
		}
		if (format_literal != NULL) {
			char* format;
			gboolean unsupported_format;
			const char* format_it;
			gunichar c;
			gboolean _tmp81;
			format = vala_string_literal_eval (format_literal);
			unsupported_format = FALSE;
			format_it = format;
			c = g_utf8_get_char (format_it);
			while (c != '\0') {
				gint length;
				ValaDataType* param_type;
				gboolean _tmp42;
				gboolean _tmp43;
				if (c != '%') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					continue;
				}
				format_it = g_utf8_next_char (format_it);
				c = g_utf8_get_char (format_it);
				while (TRUE) {
					gboolean _tmp36;
					gboolean _tmp37;
					gboolean _tmp38;
					gboolean _tmp39;
					_tmp36 = FALSE;
					_tmp37 = FALSE;
					_tmp38 = FALSE;
					_tmp39 = FALSE;
					if (c == '#') {
						_tmp39 = TRUE;
					} else {
						_tmp39 = c == '0';
					}
					if (_tmp39) {
						_tmp38 = TRUE;
					} else {
						_tmp38 = c == '-';
					}
					if (_tmp38) {
						_tmp37 = TRUE;
					} else {
						_tmp37 = c == ' ';
					}
					if (_tmp37) {
						_tmp36 = TRUE;
					} else {
						_tmp36 = c == '+';
					}
					if (!_tmp36) {
						break;
					}
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				while (TRUE) {
					gboolean _tmp40;
					_tmp40 = FALSE;
					if (c >= '0') {
						_tmp40 = c <= '9';
					} else {
						_tmp40 = FALSE;
					}
					if (!_tmp40) {
						break;
					}
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				/* precision*/
				if (c == '.') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					while (TRUE) {
						gboolean _tmp41;
						_tmp41 = FALSE;
						if (c >= '0') {
							_tmp41 = c <= '9';
						} else {
							_tmp41 = FALSE;
						}
						if (!_tmp41) {
							break;
						}
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					}
				}
				/* length modifier*/
				length = 0;
				if (c == 'h') {
					length = -1;
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
					if (c == 'h') {
						length = -2;
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					}
				} else {
					if (c == 'l') {
						length = 1;
						format_it = g_utf8_next_char (format_it);
						c = g_utf8_get_char (format_it);
					} else {
						if (c == 'z') {
							length = 2;
							format_it = g_utf8_next_char (format_it);
							c = g_utf8_get_char (format_it);
						}
					}
				}
				/* conversion specifier*/
				param_type = NULL;
				_tmp42 = FALSE;
				_tmp43 = FALSE;
				if (c == 'd') {
					_tmp43 = TRUE;
				} else {
					_tmp43 = c == 'i';
				}
				if (_tmp43) {
					_tmp42 = TRUE;
				} else {
					_tmp42 = c == 'c';
				}
				if (_tmp42) {
					/* integer*/
					if (length == (-2)) {
						ValaDataType* _tmp45;
						ValaDataType* _tmp44;
						_tmp45 = NULL;
						_tmp44 = NULL;
						param_type = (_tmp45 = (_tmp44 = analyzer->int8_type, (_tmp44 == NULL) ? NULL : vala_code_node_ref (_tmp44)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp45);
					} else {
						if (length == (-1)) {
							ValaDataType* _tmp47;
							ValaDataType* _tmp46;
							_tmp47 = NULL;
							_tmp46 = NULL;
							param_type = (_tmp47 = (_tmp46 = analyzer->short_type, (_tmp46 == NULL) ? NULL : vala_code_node_ref (_tmp46)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp47);
						} else {
							if (length == 0) {
								ValaDataType* _tmp49;
								ValaDataType* _tmp48;
								_tmp49 = NULL;
								_tmp48 = NULL;
								param_type = (_tmp49 = (_tmp48 = analyzer->int_type, (_tmp48 == NULL) ? NULL : vala_code_node_ref (_tmp48)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp49);
							} else {
								if (length == 1) {
									ValaDataType* _tmp51;
									ValaDataType* _tmp50;
									_tmp51 = NULL;
									_tmp50 = NULL;
									param_type = (_tmp51 = (_tmp50 = analyzer->long_type, (_tmp50 == NULL) ? NULL : vala_code_node_ref (_tmp50)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp51);
								} else {
									if (length == 2) {
										ValaDataType* _tmp53;
										ValaDataType* _tmp52;
										_tmp53 = NULL;
										_tmp52 = NULL;
										param_type = (_tmp53 = (_tmp52 = analyzer->ssize_t_type, (_tmp52 == NULL) ? NULL : vala_code_node_ref (_tmp52)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp53);
									}
								}
							}
						}
					}
				} else {
					gboolean _tmp54;
					gboolean _tmp55;
					gboolean _tmp56;
					_tmp54 = FALSE;
					_tmp55 = FALSE;
					_tmp56 = FALSE;
					if (c == 'o') {
						_tmp56 = TRUE;
					} else {
						_tmp56 = c == 'u';
					}
					if (_tmp56) {
						_tmp55 = TRUE;
					} else {
						_tmp55 = c == 'x';
					}
					if (_tmp55) {
						_tmp54 = TRUE;
					} else {
						_tmp54 = c == 'X';
					}
					if (_tmp54) {
						/* unsigned integer*/
						if (length == (-2)) {
							ValaDataType* _tmp58;
							ValaDataType* _tmp57;
							_tmp58 = NULL;
							_tmp57 = NULL;
							param_type = (_tmp58 = (_tmp57 = analyzer->uchar_type, (_tmp57 == NULL) ? NULL : vala_code_node_ref (_tmp57)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp58);
						} else {
							if (length == (-1)) {
								ValaDataType* _tmp60;
								ValaDataType* _tmp59;
								_tmp60 = NULL;
								_tmp59 = NULL;
								param_type = (_tmp60 = (_tmp59 = analyzer->ushort_type, (_tmp59 == NULL) ? NULL : vala_code_node_ref (_tmp59)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp60);
							} else {
								if (length == 0) {
									ValaDataType* _tmp62;
									ValaDataType* _tmp61;
									_tmp62 = NULL;
									_tmp61 = NULL;
									param_type = (_tmp62 = (_tmp61 = analyzer->uint_type, (_tmp61 == NULL) ? NULL : vala_code_node_ref (_tmp61)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp62);
								} else {
									if (length == 1) {
										ValaDataType* _tmp64;
										ValaDataType* _tmp63;
										_tmp64 = NULL;
										_tmp63 = NULL;
										param_type = (_tmp64 = (_tmp63 = analyzer->ulong_type, (_tmp63 == NULL) ? NULL : vala_code_node_ref (_tmp63)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp64);
									} else {
										if (length == 2) {
											ValaDataType* _tmp66;
											ValaDataType* _tmp65;
											_tmp66 = NULL;
											_tmp65 = NULL;
											param_type = (_tmp66 = (_tmp65 = analyzer->size_t_type, (_tmp65 == NULL) ? NULL : vala_code_node_ref (_tmp65)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp66);
										}
									}
								}
							}
						}
					} else {
						gboolean _tmp67;
						gboolean _tmp68;
						gboolean _tmp69;
						gboolean _tmp70;
						gboolean _tmp71;
						gboolean _tmp72;
						gboolean _tmp73;
						_tmp67 = FALSE;
						_tmp68 = FALSE;
						_tmp69 = FALSE;
						_tmp70 = FALSE;
						_tmp71 = FALSE;
						_tmp72 = FALSE;
						_tmp73 = FALSE;
						if (c == 'e') {
							_tmp73 = TRUE;
						} else {
							_tmp73 = c == 'E';
						}
						if (_tmp73) {
							_tmp72 = TRUE;
						} else {
							_tmp72 = c == 'f';
						}
						if (_tmp72) {
							_tmp71 = TRUE;
						} else {
							_tmp71 = c == 'F';
						}
						if (_tmp71) {
							_tmp70 = TRUE;
						} else {
							_tmp70 = c == 'g';
						}
						if (_tmp70) {
							_tmp69 = TRUE;
						} else {
							_tmp69 = c == 'G';
						}
						if (_tmp69) {
							_tmp68 = TRUE;
						} else {
							_tmp68 = c == 'a';
						}
						if (_tmp68) {
							_tmp67 = TRUE;
						} else {
							_tmp67 = c == 'A';
						}
						if (_tmp67) {
							ValaDataType* _tmp75;
							ValaDataType* _tmp74;
							/* double*/
							_tmp75 = NULL;
							_tmp74 = NULL;
							param_type = (_tmp75 = (_tmp74 = analyzer->double_type, (_tmp74 == NULL) ? NULL : vala_code_node_ref (_tmp74)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp75);
						} else {
							if (c == 's') {
								ValaDataType* _tmp77;
								ValaDataType* _tmp76;
								/* string*/
								_tmp77 = NULL;
								_tmp76 = NULL;
								param_type = (_tmp77 = (_tmp76 = analyzer->string_type, (_tmp76 == NULL) ? NULL : vala_code_node_ref (_tmp76)), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp77);
							} else {
								if (c == 'p') {
									ValaDataType* _tmp79;
									ValaVoidType* _tmp78;
									/* pointer*/
									_tmp79 = NULL;
									_tmp78 = NULL;
									param_type = (_tmp79 = (ValaDataType*) vala_pointer_type_new ((ValaDataType*) (_tmp78 = vala_void_type_new ()), NULL), (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), _tmp79);
									(_tmp78 == NULL) ? NULL : (_tmp78 = (vala_code_node_unref (_tmp78), NULL));
								} else {
									if (c == '%') {
									} else {
										/* literal %*/
										unsupported_format = TRUE;
										(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
										break;
									}
								}
							}
						}
					}
				}
				if (c != '\0') {
					format_it = g_utf8_next_char (format_it);
					c = g_utf8_get_char (format_it);
				}
				if (param_type != NULL) {
					if (gee_iterator_next (arg_it)) {
						ValaExpression* arg;
						arg = (ValaExpression*) gee_iterator_get (arg_it);
						vala_expression_set_target_type (arg, param_type);
						(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
					} else {
						gboolean _tmp80;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Too few arguments for specified format");
						return (_tmp80 = FALSE, (param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL)), format = (g_free (format), NULL), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), _tmp80);
					}
				}
				(param_type == NULL) ? NULL : (param_type = (vala_code_node_unref (param_type), NULL));
			}
			_tmp81 = FALSE;
			if (!unsupported_format) {
				_tmp81 = gee_iterator_next (arg_it);
			} else {
				_tmp81 = FALSE;
			}
			if (_tmp81) {
				gboolean _tmp82;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Too many arguments for specified format");
				return (_tmp82 = FALSE, format = (g_free (format), NULL), (format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL)), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), _tmp82);
			}
			format = (g_free (format), NULL);
		}
		(format_literal == NULL) ? NULL : (format_literal = (vala_code_node_unref (format_literal), NULL));
	}
	{
		GeeList* _tmp83;
		GeeIterator* _tmp84;
		GeeIterator* arg_it;
		_tmp83 = NULL;
		_tmp84 = NULL;
		arg_it = (_tmp84 = gee_iterable_iterator ((GeeIterable*) (_tmp83 = vala_method_call_get_argument_list (self))), (_tmp83 == NULL) ? NULL : (_tmp83 = (gee_collection_object_unref (_tmp83), NULL)), _tmp84);
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (arg_it);
			vala_code_node_check ((ValaCodeNode*) arg, analyzer);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
	}
	ret_type = vala_data_type_get_return_type (mtype);
	_tmp85 = NULL;
	params = (_tmp85 = vala_data_type_get_parameters (mtype), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), _tmp85);
	if (VALA_IS_VOID_TYPE (ret_type)) {
		gboolean _tmp86;
		gboolean _tmp87;
		_tmp86 = FALSE;
		_tmp87 = FALSE;
		if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)))) {
			_tmp87 = !(VALA_IS_FOR_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
		} else {
			_tmp87 = FALSE;
		}
		if (_tmp87) {
			_tmp86 = !(VALA_IS_YIELD_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
		} else {
			_tmp86 = FALSE;
		}
		/* void return type*/
		if (_tmp86) {
			gboolean _tmp88;
			/* A void method invocation can be in the initializer or
			 iterator of a for statement*/
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "invocation of void method not allowed as expression");
			return (_tmp88 = FALSE, (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp88);
		}
	}
	vala_expression_set_formal_value_type ((ValaExpression*) self, ret_type);
	_tmp89 = NULL;
	vala_expression_set_value_type ((ValaExpression*) self, _tmp89 = vala_data_type_get_actual_type (vala_expression_get_formal_value_type ((ValaExpression*) self), target_object_type, (ValaCodeNode*) self));
	(_tmp89 == NULL) ? NULL : (_tmp89 = (vala_code_node_unref (_tmp89), NULL));
	may_throw = FALSE;
	if (VALA_IS_METHOD_TYPE (mtype)) {
		ValaMethod* _tmp90;
		ValaMethod* m;
		_tmp90 = NULL;
		m = (_tmp90 = vala_method_type_get_method_symbol ((VALA_METHOD_TYPE (mtype))), (_tmp90 == NULL) ? NULL : vala_code_node_ref (_tmp90));
		{
			GeeList* _tmp91;
			GeeIterator* _tmp92;
			GeeIterator* error_type_it;
			_tmp91 = NULL;
			_tmp92 = NULL;
			error_type_it = (_tmp92 = gee_iterable_iterator ((GeeIterable*) (_tmp91 = vala_code_node_get_error_types ((ValaCodeNode*) m))), (_tmp91 == NULL) ? NULL : (_tmp91 = (gee_collection_object_unref (_tmp91), NULL)), _tmp92);
			while (gee_iterator_next (error_type_it)) {
				ValaDataType* error_type;
				ValaDataType* call_error_type;
				error_type = (ValaDataType*) gee_iterator_get (error_type_it);
				may_throw = TRUE;
				/* ensure we can trace back which expression may throw errors of this type*/
				call_error_type = vala_data_type_copy (error_type);
				vala_code_node_set_source_reference ((ValaCodeNode*) call_error_type, vala_code_node_get_source_reference ((ValaCodeNode*) self));
				vala_code_node_add_error_type ((ValaCodeNode*) self, call_error_type);
				(error_type == NULL) ? NULL : (error_type = (vala_code_node_unref (error_type), NULL));
				(call_error_type == NULL) ? NULL : (call_error_type = (vala_code_node_unref (call_error_type), NULL));
			}
			(error_type_it == NULL) ? NULL : (error_type_it = (gee_collection_object_unref (error_type_it), NULL));
		}
		(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
	}
	_tmp93 = NULL;
	vala_semantic_analyzer_check_arguments (analyzer, (ValaExpression*) self, mtype, params, _tmp93 = vala_method_call_get_argument_list (self));
	(_tmp93 == NULL) ? NULL : (_tmp93 = (gee_collection_object_unref (_tmp93), NULL));
	if (may_throw) {
		gboolean _tmp94;
		_tmp94 = FALSE;
		if (VALA_IS_LOCAL_VARIABLE (vala_code_node_get_parent_node ((ValaCodeNode*) self))) {
			_tmp94 = TRUE;
		} else {
			_tmp94 = VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node ((ValaCodeNode*) self));
		}
		if (_tmp94) {
		} else {
			ValaBlock* _tmp95;
			ValaBlock* old_insert_block;
			ValaBlock* _tmp96;
			ValaCodeNode* _tmp97;
			ValaCodeNode* old_parent_node;
			char* _tmp98;
			ValaLocalVariable* _tmp99;
			ValaLocalVariable* local;
			ValaDeclarationStatement* decl;
			ValaExpression* temp_access;
			ValaBlock* _tmp100;
			ValaBlock* block;
			ValaBlock* _tmp102;
			ValaBlock* _tmp101;
			/* simple statements, no side effects after method call*/
			_tmp95 = NULL;
			old_insert_block = (_tmp95 = analyzer->insert_block, (_tmp95 == NULL) ? NULL : vala_code_node_ref (_tmp95));
			_tmp96 = NULL;
			analyzer->insert_block = (_tmp96 = vala_expression_prepare_condition_split ((ValaExpression*) self, analyzer), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp96);
			/* store parent_node as we need to replace the expression in the old parent node later on*/
			_tmp97 = NULL;
			old_parent_node = (_tmp97 = vala_code_node_get_parent_node ((ValaCodeNode*) self), (_tmp97 == NULL) ? NULL : vala_code_node_ref (_tmp97));
			_tmp98 = NULL;
			_tmp99 = NULL;
			local = (_tmp99 = vala_local_variable_new (vala_expression_get_value_type ((ValaExpression*) self), _tmp98 = vala_code_node_get_temp_name ((ValaCodeNode*) self), NULL, vala_code_node_get_source_reference ((ValaCodeNode*) self)), _tmp98 = (g_free (_tmp98), NULL), _tmp99);
			/* use floating variable to avoid unnecessary (and sometimes impossible) copies*/
			vala_local_variable_set_floating (local, TRUE);
			decl = vala_declaration_statement_new ((ValaSymbol*) local, vala_code_node_get_source_reference ((ValaCodeNode*) self));
			vala_expression_insert_statement ((ValaExpression*) self, analyzer->insert_block, (ValaStatement*) decl);
			temp_access = (ValaExpression*) vala_member_access_new_simple (vala_symbol_get_name ((ValaSymbol*) local), vala_code_node_get_source_reference ((ValaCodeNode*) self));
			vala_expression_set_target_type (temp_access, vala_expression_get_target_type ((ValaExpression*) self));
			/* don't set initializer earlier as this changes parent_node and parent_statement*/
			vala_local_variable_set_initializer (local, (ValaExpression*) self);
			vala_code_node_check ((ValaCodeNode*) decl, analyzer);
			vala_code_node_check ((ValaCodeNode*) temp_access, analyzer);
			/* move temp variable to insert block to ensure the
			 variable is in the same block as the declaration
			 otherwise there will be scoping issues in the generated code*/
			_tmp100 = NULL;
			block = (_tmp100 = VALA_BLOCK (vala_semantic_analyzer_get_current_symbol (analyzer)), (_tmp100 == NULL) ? NULL : vala_code_node_ref (_tmp100));
			vala_block_remove_local_variable (block, local);
			vala_block_add_local_variable (analyzer->insert_block, local);
			_tmp102 = NULL;
			_tmp101 = NULL;
			analyzer->insert_block = (_tmp102 = (_tmp101 = old_insert_block, (_tmp101 == NULL) ? NULL : vala_code_node_ref (_tmp101)), (analyzer->insert_block == NULL) ? NULL : (analyzer->insert_block = (vala_code_node_unref (analyzer->insert_block), NULL)), _tmp102);
			vala_code_node_replace_expression (old_parent_node, (ValaExpression*) self, temp_access);
			(old_insert_block == NULL) ? NULL : (old_insert_block = (vala_code_node_unref (old_insert_block), NULL));
			(old_parent_node == NULL) ? NULL : (old_parent_node = (vala_code_node_unref (old_parent_node), NULL));
			(local == NULL) ? NULL : (local = (vala_code_node_unref (local), NULL));
			(decl == NULL) ? NULL : (decl = (vala_code_node_unref (decl), NULL));
			(temp_access == NULL) ? NULL : (temp_access = (vala_code_node_unref (temp_access), NULL));
			(block == NULL) ? NULL : (block = (vala_code_node_unref (block), NULL));
		}
	}
	return (_tmp103 = !vala_code_node_get_error ((ValaCodeNode*) self), (target_object_type == NULL) ? NULL : (target_object_type = (vala_code_node_unref (target_object_type), NULL)), (mtype == NULL) ? NULL : (mtype = (vala_code_node_unref (mtype), NULL)), (params == NULL) ? NULL : (params = (gee_collection_object_unref (params), NULL)), (last_arg == NULL) ? NULL : (last_arg = (vala_code_node_unref (last_arg), NULL)), (args == NULL) ? NULL : (args = (gee_collection_object_unref (args), NULL)), (arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL)), (ret_type == NULL) ? NULL : (ret_type = (vala_code_node_unref (ret_type), NULL)), _tmp103);
}


static void vala_method_call_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_method_call_get_call (self), collection);
	{
		GeeIterator* arg_it;
		arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (arg_it);
			vala_code_node_get_defined_variables ((ValaCodeNode*) arg, collection);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
	}
}


static void vala_method_call_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaMethodCall * self;
	self = (ValaMethodCall*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_method_call_get_call (self), collection);
	{
		GeeIterator* arg_it;
		arg_it = gee_iterable_iterator ((GeeIterable*) self->priv->argument_list);
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = (ValaExpression*) gee_iterator_get (arg_it);
			vala_code_node_get_used_variables ((ValaCodeNode*) arg, collection);
			(arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL));
		}
		(arg_it == NULL) ? NULL : (arg_it = (gee_collection_object_unref (arg_it), NULL));
	}
}


ValaExpression* vala_method_call_get_call (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->_call;
}


void vala_method_call_set_call (ValaMethodCall* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->_call = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->_call == NULL) ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->_call, (ValaCodeNode*) self);
}


ValaCCodeExpression* vala_method_call_get_delegate_target (ValaMethodCall* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_delegate_target;
}


void vala_method_call_set_delegate_target (ValaMethodCall* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2;
	ValaCCodeExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_delegate_target = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_ccode_node_ref (_tmp1)), (self->priv->_delegate_target == NULL) ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL)), _tmp2);
}


static void vala_method_call_class_init (ValaMethodCallClass * klass) {
	vala_method_call_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_method_call_finalize;
	g_type_class_add_private (klass, sizeof (ValaMethodCallPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_method_call_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_method_call_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_method_call_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_method_call_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_method_call_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_method_call_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_method_call_real_get_used_variables;
}


static void vala_method_call_instance_init (ValaMethodCall * self) {
	self->priv = VALA_METHOD_CALL_GET_PRIVATE (self);
	self->priv->argument_list = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->array_sizes = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_method_call_finalize (ValaCodeNode* obj) {
	ValaMethodCall * self;
	self = VALA_METHOD_CALL (obj);
	(self->priv->_delegate_target == NULL) ? NULL : (self->priv->_delegate_target = (vala_ccode_node_unref (self->priv->_delegate_target), NULL));
	(self->_call == NULL) ? NULL : (self->_call = (vala_code_node_unref (self->_call), NULL));
	(self->priv->argument_list == NULL) ? NULL : (self->priv->argument_list = (gee_collection_object_unref (self->priv->argument_list), NULL));
	(self->priv->array_sizes == NULL) ? NULL : (self->priv->array_sizes = (gee_collection_object_unref (self->priv->array_sizes), NULL));
	VALA_CODE_NODE_CLASS (vala_method_call_parent_class)->finalize (obj);
}


GType vala_method_call_get_type (void) {
	static GType vala_method_call_type_id = 0;
	if (vala_method_call_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMethodCallClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_method_call_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMethodCall), 0, (GInstanceInitFunc) vala_method_call_instance_init, NULL };
		vala_method_call_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaMethodCall", &g_define_type_info, 0);
	}
	return vala_method_call_type_id;
}




