/* valaconstructor.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaconstructor.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valacodevisitor.h>
#include <vala/valablock.h>
#include <vala/valaformalparameter.h>
#include <vala/valasourcereference.h>




struct _ValaConstructorPrivate {
	ValaBlock* _body;
	ValaFormalParameter* _this_parameter;
	MemberBinding _binding;
};

#define VALA_CONSTRUCTOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CONSTRUCTOR, ValaConstructorPrivate))
enum  {
	VALA_CONSTRUCTOR_DUMMY_PROPERTY
};
static void vala_constructor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_constructor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_constructor_parent_class = NULL;
static void vala_constructor_finalize (ValaCodeNode* obj);



/**
 * Creates a new constructor.
 *
 * @param source reference to source code
 * @return       newly created constructor
 */
ValaConstructor* vala_constructor_construct (GType object_type, ValaSourceReference* source) {
	ValaConstructor* self;
	g_return_val_if_fail (source != NULL, NULL);
	self = ((ValaConstructor*) (g_type_create_instance (object_type)));
	self = ((ValaConstructor*) (vala_symbol_construct (object_type, NULL, source)));
	return self;
}


ValaConstructor* vala_constructor_new (ValaSourceReference* source) {
	return vala_constructor_construct (VALA_TYPE_CONSTRUCTOR, source);
}


static void vala_constructor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstructor * self;
	self = ((ValaConstructor*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_constructor (visitor, self);
}


static void vala_constructor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConstructor * self;
	self = ((ValaConstructor*) (base));
	g_return_if_fail (visitor != NULL);
	if (self->priv->_body != NULL) {
		vala_code_node_accept (((ValaCodeNode*) (self->priv->_body)), visitor);
	}
}


ValaBlock* vala_constructor_get_body (ValaConstructor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_constructor_set_body (ValaConstructor* self, ValaBlock* value) {
	ValaBlock* _tmp2;
	ValaBlock* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_body = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL))), _tmp2);
}


ValaFormalParameter* vala_constructor_get_this_parameter (ValaConstructor* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_this_parameter;
}


void vala_constructor_set_this_parameter (ValaConstructor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp2;
	ValaFormalParameter* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_this_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL))), _tmp2);
}


MemberBinding vala_constructor_get_binding (ValaConstructor* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_binding;
}


void vala_constructor_set_binding (ValaConstructor* self, MemberBinding value) {
	g_return_if_fail (self != NULL);
	self->priv->_binding = value;
}


static void vala_constructor_class_init (ValaConstructorClass * klass) {
	vala_constructor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_constructor_finalize;
	g_type_class_add_private (klass, sizeof (ValaConstructorPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_constructor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_constructor_real_accept_children;
}


static void vala_constructor_instance_init (ValaConstructor * self) {
	self->priv = VALA_CONSTRUCTOR_GET_PRIVATE (self);
	self->priv->_binding = MEMBER_BINDING_INSTANCE;
}


static void vala_constructor_finalize (ValaCodeNode* obj) {
	ValaConstructor * self;
	self = VALA_CONSTRUCTOR (obj);
	(self->priv->_body == NULL ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)));
	(self->priv->_this_parameter == NULL ? NULL : (self->priv->_this_parameter = (vala_code_node_unref (self->priv->_this_parameter), NULL)));
	VALA_CODE_NODE_CLASS (vala_constructor_parent_class)->finalize (obj);
}


GType vala_constructor_get_type (void) {
	static GType vala_constructor_type_id = 0;
	if (vala_constructor_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaConstructorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_constructor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaConstructor), 0, (GInstanceInitFunc) vala_constructor_instance_init, NULL };
		vala_constructor_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaConstructor", &g_define_type_info, 0);
	}
	return vala_constructor_type_id;
}




