/*
 * wsdl-schema.c: Build typecodes from XML describing one of the known
 * type systems
 *
 * Authors:
 *	Dick Porter (dick@ximian.com)
 *
 * Copyright (C) 2001, Ximian, Inc.
 */

#include <glib.h>
#include <string.h>

#include <tree.h>

#include "wsdl-schema.h"
#include "wsdl-schema-glib.h"

static void (*schema_func_start)(const xmlDocPtr, const xmlNodePtr,
				 const CHAR **, const guchar *,
				 const guchar *)=NULL;
static void (*schema_func_end)(const xmlNodePtr)=NULL;

/*
 * check that the node name is `localname` and belongs to the namespace `ns`
 *
 * returns TRUE for "match", FALSE otherwise
 */
gboolean wsdl_qnamecmp(const xmlNodePtr node, const CHAR *ns,
		       const CHAR *localname)
{
	int ret;
	
	g_assert(node!=NULL);
	g_assert(node->name!=NULL);
	g_assert(node->ns!=NULL);
	g_assert(node->ns->href!=NULL);
	g_assert(ns!=NULL);
	g_assert(localname!=NULL);
	
	/* If the local parts don't match, return straight away */
	ret=strcmp(localname, node->name);
	if(ret!=0) {
		return(FALSE);
	}

	/* Now check that the node namespace matches ns */
	ret=strcmp(ns, node->ns->href);
	if(ret!=0) {
		return(FALSE);
	}
	
	return(TRUE);
}

/*
 * check that the namespace prefix of attr matches the defined
 * namespace for `ns_href`.
 *
 * returns TRUE for "match", FALSE otherwise.
 */
gboolean wsdl_attrnscmp(const xmlNodePtr node, const guchar *attr,
			const guchar *ns_href)
{
	xmlNodePtr nptr;
	xmlNsPtr ns;
	guchar *colon;
	guchar *copy;
	gboolean ret=FALSE;

	g_assert(node!=NULL);
	g_assert(attr!=NULL);
	g_assert(ns_href!=NULL);
	
	copy=g_strdup(attr);
	colon=strchr(copy, ':');
	if(colon!=NULL) {
		/* There is a namespace prefix. Now 'copy' contains
		 * the prefix and 'colon' the local part.
		 */
		*colon++='\0';

		/* Need to find all namespace definitions from this
		 * node up the parent chain, to the root.  (If theres
		 * an easier way to find all the currently in-scope
		 * namespace definitions, I'd like to know)
		 */

		nptr=node;
		
		do {
			ns=nptr->nsDef;
			
			while(ns!=NULL) {
				/* ns->prefix can be NULL, denoting
				 * the global namespace.
				 */
				if(ns->prefix!=NULL &&
				   !strcmp(copy, ns->prefix) &&
				   !strcmp(ns_href, ns->href)) {
					ret=TRUE;
					goto end;
				}

				ns=ns->next;
			}
			
			nptr=nptr->parent;
		} while(nptr!=NULL);
	} else {
		/* No namespace prefix, so only the blank namespace
		 * matches
		 * (should the global namespace match too? - RHP)
		 */
		if(!strcmp(ns_href, "")) {
			ret=TRUE;
			goto end;
		}
	}
	
end:
	g_free(copy);
	return(ret);
}

const guchar *wsdl_prefix_to_namespace(xmlDocPtr doc, xmlNodePtr node,
				       const guchar *str, gboolean defns)
{
	xmlNsPtr nsptr;
	guchar *colon;
	guchar *ns;
	
	ns=g_strdup(str);
	
	if((colon=strchr(ns, ':'))!=NULL) {
		*colon='\0';
	} else {
		g_free(ns);
		ns=NULL;
	}
	
	if(ns==NULL && defns==FALSE) {
		return(NULL);
	}
		
	nsptr=xmlSearchNs(doc, node, ns);

	if(ns!=NULL) {
		g_free(ns);
	}
	
	
	if(nsptr!=NULL) {
		return(nsptr->href);
	} else {
		return(NULL);
	}
}

static gboolean wsdl_schema_glib_parse_attrs(const CHAR **attrs)
{
	int i=0;
	
	if(attrs!=NULL) {
		while(attrs[i]!=NULL) {
			if(!strcmp(attrs[i], "fill-in-here")) {
				/* g_strdup(attrs[i+1]) */
			} else if (!strcmp(attrs[i], "xmlns") ||
				   !strncmp(attrs[i], "xmlns:", 6)) {
				/* Do nothing */
			} else {
				/* unknown element attribute */
			}
			
			i+=2;
		}
	}
	
	/* Check that all required attrs are set */
	if(1) {
		/* set pointer */
		return(TRUE);
	} else {
		/* free anything that has been alloced */
		return(FALSE);
	}
}

gboolean wsdl_schema_init(const xmlNodePtr node, const CHAR **attrs)
{
	if(0 && wsdl_qnamecmp(node, XSDNS, "schema")==TRUE) {
		/* This section will be filled in sometime, honest! */
		return(TRUE);
	} else if (wsdl_qnamecmp(node, GLIBNS, "type")==TRUE) {
		schema_func_start=wsdl_schema_glib_start_element;
		schema_func_end=wsdl_schema_glib_end_element;
		
		wsdl_schema_glib_parse_attrs(attrs);

		return(TRUE);
	} else {
		return(FALSE);
	}
}

void wsdl_schema_start_element(const xmlDocPtr doc,const xmlNodePtr node,
			       const CHAR **attrs, const guchar *ns,
			       const guchar *nsuri)
{
	if(schema_func_start!=NULL) {
		schema_func_start(doc, node, attrs, ns, nsuri);
	}
}

void wsdl_schema_end_element(const xmlNodePtr node)
{
	if(schema_func_end!=NULL) {
		schema_func_end(node);
	}
}
