/*
 * Copyright (C) 2008 Zeeshan Ali <zeenix@gmail.com>.
 *
 * Author: Zeeshan Ali <zeenix@gmail.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <rygel-media-object.h>
#include <gobject/gvaluecollector.h>




enum  {
	RYGEL_MEDIA_OBJECT_DUMMY_PROPERTY
};
static void rygel_media_object_real_serialize (RygelMediaObject* self, GUPnPDIDLLiteWriter* didl_writer, GError** error);
static gpointer rygel_media_object_parent_class = NULL;
static void rygel_media_object_finalize (RygelMediaObject* obj);



static void rygel_media_object_real_serialize (RygelMediaObject* self, GUPnPDIDLLiteWriter* didl_writer, GError** error) {
	g_return_if_fail (self != NULL);
	g_critical ("Type `%s' does not implement abstract method `rygel_media_object_serialize'", g_type_name (G_TYPE_FROM_INSTANCE (self)));
	return;
}


void rygel_media_object_serialize (RygelMediaObject* self, GUPnPDIDLLiteWriter* didl_writer, GError** error) {
	RYGEL_MEDIA_OBJECT_GET_CLASS (self)->serialize (self, didl_writer, error);
}


static void rygel_value_media_object_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void rygel_value_media_object_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		rygel_media_object_unref (value->data[0].v_pointer);
	}
}


static void rygel_value_media_object_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = rygel_media_object_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer rygel_value_media_object_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* rygel_value_media_object_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		RygelMediaObject* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = rygel_media_object_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* rygel_value_media_object_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	RygelMediaObject** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = rygel_media_object_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* rygel_param_spec_media_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	RygelParamSpecMediaObject* spec;
	g_return_val_if_fail (g_type_is_a (object_type, RYGEL_TYPE_MEDIA_OBJECT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer rygel_value_get_media_object (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, RYGEL_TYPE_MEDIA_OBJECT), NULL);
	return value->data[0].v_pointer;
}


void rygel_value_set_media_object (GValue* value, gpointer v_object) {
	RygelMediaObject* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, RYGEL_TYPE_MEDIA_OBJECT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, RYGEL_TYPE_MEDIA_OBJECT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		rygel_media_object_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		rygel_media_object_unref (old);
	}
}


static void rygel_media_object_class_init (RygelMediaObjectClass * klass) {
	rygel_media_object_parent_class = g_type_class_peek_parent (klass);
	RYGEL_MEDIA_OBJECT_CLASS (klass)->finalize = rygel_media_object_finalize;
	RYGEL_MEDIA_OBJECT_CLASS (klass)->serialize = rygel_media_object_real_serialize;
}


static void rygel_media_object_instance_init (RygelMediaObject * self) {
	self->ref_count = 1;
}


static void rygel_media_object_finalize (RygelMediaObject* obj) {
	RygelMediaObject * self;
	self = RYGEL_MEDIA_OBJECT (obj);
	self->id = (g_free (self->id), NULL);
	self->parent_id = (g_free (self->parent_id), NULL);
	self->title = (g_free (self->title), NULL);
}


GType rygel_media_object_get_type (void) {
	static GType rygel_media_object_type_id = 0;
	if (rygel_media_object_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { rygel_value_media_object_init, rygel_value_media_object_free_value, rygel_value_media_object_copy_value, rygel_value_media_object_peek_pointer, "p", rygel_value_media_object_collect_value, "p", rygel_value_media_object_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (RygelMediaObjectClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_media_object_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelMediaObject), 0, (GInstanceInitFunc) rygel_media_object_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		rygel_media_object_type_id = g_type_register_fundamental (g_type_fundamental_next (), "RygelMediaObject", &g_define_type_info, &g_define_type_fundamental_info, G_TYPE_FLAG_ABSTRACT);
	}
	return rygel_media_object_type_id;
}


gpointer rygel_media_object_ref (gpointer instance) {
	RygelMediaObject* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void rygel_media_object_unref (gpointer instance) {
	RygelMediaObject* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		RYGEL_MEDIA_OBJECT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




