/*
 * Copyright (C) 2008 Zeeshan Ali <zeenix@gmail.com>.
 * Copyright (C) 2007 OpenedHand Ltd.
 *
 * Author: Zeeshan Ali <zeenix@gmail.com>
 *         Jorn Baayen <jorn@openedhand.com>
 *
 * This file is part of Rygel.
 *
 * Rygel is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rygel is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "rygel-browse.h"
#include <libgupnp-av/gupnp-av.h>
#include <libsoup/soup.h>
#include <gee/list.h>
#include <gee/collection.h>
#include "rygel-media-object.h"
#include "rygel-media-container.h"
#include "rygel-didl-lite-writer.h"
#include "rygel-http-server.h"




struct _RygelBrowsePrivate {
	gboolean fetch_metadata;
	RygelMediaObject* media_object;
	RygelMediaContainer* root_container;
	guint32 system_update_id;
	GUPnPServiceAction* action;
	RygelDIDLLiteWriter* didl_writer;
	GCancellable* cancellable;
};

#define RYGEL_BROWSE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), RYGEL_TYPE_BROWSE, RygelBrowsePrivate))
enum  {
	RYGEL_BROWSE_DUMMY_PROPERTY
};
static void rygel_browse_real_run (RygelStateMachine* base, GCancellable* cancellable);
static void rygel_browse_got_media_object (RygelBrowse* self);
static void _rygel_browse_on_media_object_found_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self);
static void rygel_browse_fetch_media_object (RygelBrowse* self);
static void rygel_browse_on_media_object_found (RygelBrowse* self, GObject* source_object, GAsyncResult* res);
static void rygel_browse_handle_metadata_request (RygelBrowse* self);
static void rygel_browse_handle_children_request (RygelBrowse* self);
static void rygel_browse_parse_args (RygelBrowse* self);
static void rygel_browse_conclude (RygelBrowse* self);
static void rygel_browse_handle_error (RygelBrowse* self, GError* error);
static void rygel_browse_serialize_children (RygelBrowse* self, GeeList* children);
static void _rygel_browse_on_children_fetched_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self);
static void rygel_browse_fetch_children (RygelBrowse* self);
static void rygel_browse_on_children_fetched (RygelBrowse* self, GObject* source_object, GAsyncResult* res);
static gpointer rygel_browse_parent_class = NULL;
static RygelStateMachineIface* rygel_browse_rygel_state_machine_parent_iface = NULL;
static void rygel_browse_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



RygelBrowse* rygel_browse_construct (GType object_type, RygelContentDirectory* content_dir, GUPnPServiceAction* action) {
	RygelBrowse * self;
	RygelMediaContainer* _tmp1;
	RygelMediaContainer* _tmp0;
	GUPnPServiceAction* _tmp2;
	RygelDIDLLiteWriter* _tmp3;
	g_return_val_if_fail (content_dir != NULL, NULL);
	g_return_val_if_fail (action != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->root_container = (_tmp1 = (_tmp0 = content_dir->root_container, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL)), _tmp1);
	self->priv->system_update_id = content_dir->system_update_id;
	_tmp2 = NULL;
	self->priv->action = (_tmp2 = action, action = NULL, _tmp2);
	_tmp3 = NULL;
	self->priv->didl_writer = (_tmp3 = rygel_didl_lite_writer_new (content_dir->http_server), (self->priv->didl_writer == NULL) ? NULL : (self->priv->didl_writer = (g_object_unref (self->priv->didl_writer), NULL)), _tmp3);
	return self;
}


RygelBrowse* rygel_browse_new (RygelContentDirectory* content_dir, GUPnPServiceAction* action) {
	return rygel_browse_construct (RYGEL_TYPE_BROWSE, content_dir, action);
}


static void rygel_browse_real_run (RygelStateMachine* base, GCancellable* cancellable) {
	RygelBrowse * self;
	GCancellable* _tmp1;
	GCancellable* _tmp0;
	self = (RygelBrowse*) base;
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cancellable = (_tmp1 = (_tmp0 = cancellable, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL)), _tmp1);
	/* Start DIDL-Lite fragment */
	gupnp_didl_lite_writer_start_didl_lite ((GUPnPDIDLLiteWriter*) self->priv->didl_writer, NULL, NULL, TRUE);
	/* Start by parsing the 'in' arguments */
	rygel_browse_parse_args (self);
}


static void rygel_browse_got_media_object (RygelBrowse* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->media_object == NULL) {
		GError* _tmp0;
		_tmp0 = NULL;
		rygel_browse_handle_error (self, _tmp0 = g_error_new_literal (RYGEL_CONTENT_DIRECTORY_ERROR, RYGEL_CONTENT_DIRECTORY_ERROR_NO_SUCH_OBJECT, "No such object"));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (g_error_free (_tmp0), NULL));
		return;
	}
	if (self->priv->fetch_metadata) {
		/* BrowseMetadata*/
		rygel_browse_handle_metadata_request (self);
	} else {
		/* BrowseDirectChildren*/
		rygel_browse_handle_children_request (self);
	}
}


static void _rygel_browse_on_media_object_found_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self) {
	rygel_browse_on_media_object_found (self, source_object, res);
}


static void rygel_browse_fetch_media_object (RygelBrowse* self) {
	g_return_if_fail (self != NULL);
	if (_vala_strcmp0 (self->object_id, ((RygelMediaObject*) self->priv->root_container)->id) == 0) {
		RygelMediaObject* _tmp1;
		RygelMediaObject* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->media_object = (_tmp1 = (_tmp0 = (RygelMediaObject*) self->priv->root_container, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)), (self->priv->media_object == NULL) ? NULL : (self->priv->media_object = (g_object_unref (self->priv->media_object), NULL)), _tmp1);
		rygel_browse_got_media_object (self);
		return;
	}
	rygel_media_container_find_object (self->priv->root_container, self->object_id, self->priv->cancellable, _rygel_browse_on_media_object_found_gasync_ready_callback, self);
}


static void rygel_browse_on_media_object_found (RygelBrowse* self, GObject* source_object, GAsyncResult* res) {
	GError * inner_error;
	RygelMediaContainer* _tmp0;
	RygelMediaContainer* container;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_object != NULL);
	g_return_if_fail (res != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	container = (_tmp0 = RYGEL_MEDIA_CONTAINER (source_object), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	{
		RygelMediaObject* _tmp1;
		RygelMediaObject* _tmp2;
		_tmp1 = rygel_media_container_find_object_finish (container, res, &inner_error);
		if (inner_error != NULL) {
			goto __catch7_g_error;
			goto __finally7;
		}
		_tmp2 = NULL;
		self->priv->media_object = (_tmp2 = _tmp1, (self->priv->media_object == NULL) ? NULL : (self->priv->media_object = (g_object_unref (self->priv->media_object), NULL)), _tmp2);
	}
	goto __finally7;
	__catch7_g_error:
	{
		GError * err;
		err = inner_error;
		inner_error = NULL;
		{
			rygel_browse_handle_error (self, err);
			(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
			(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
			return;
		}
	}
	__finally7:
	if (inner_error != NULL) {
		(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	rygel_browse_got_media_object (self);
	(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
}


static void rygel_browse_handle_metadata_request (RygelBrowse* self) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	if (RYGEL_IS_MEDIA_CONTAINER (self->priv->media_object)) {
		self->update_id = (guint) RYGEL_MEDIA_CONTAINER (self->priv->media_object)->update_id;
	} else {
		self->update_id = (guint) G_MAXUINT32;
	}
	{
		rygel_didl_lite_writer_serialize (self->priv->didl_writer, self->priv->media_object, &inner_error);
		if (inner_error != NULL) {
			goto __catch8_g_error;
			goto __finally8;
		}
	}
	goto __finally8;
	__catch8_g_error:
	{
		GError * err;
		err = inner_error;
		inner_error = NULL;
		{
			rygel_browse_handle_error (self, err);
			(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
			return;
		}
	}
	__finally8:
	if (inner_error != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	self->number_returned = (guint) 1;
	self->total_matches = (guint) 1;
	/* Conclude the successful Browse action*/
	rygel_browse_conclude (self);
}


static void rygel_browse_handle_children_request (RygelBrowse* self) {
	RygelMediaContainer* _tmp1;
	RygelMediaContainer* container;
	g_return_if_fail (self != NULL);
	if (!RYGEL_IS_MEDIA_CONTAINER (self->priv->media_object)) {
		GError* _tmp0;
		_tmp0 = NULL;
		rygel_browse_handle_error (self, _tmp0 = g_error_new_literal (RYGEL_CONTENT_DIRECTORY_ERROR, RYGEL_CONTENT_DIRECTORY_ERROR_NO_SUCH_OBJECT, "No such object"));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (g_error_free (_tmp0), NULL));
		return;
	}
	_tmp1 = NULL;
	container = (_tmp1 = RYGEL_MEDIA_CONTAINER (self->priv->media_object), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
	self->total_matches = container->child_count;
	self->update_id = (guint) container->update_id;
	if (self->requested_count == 0) {
		/* No max count requested, try to fetch all children*/
		self->requested_count = self->total_matches;
	}
	rygel_browse_fetch_children (self);
	(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
}


static void rygel_browse_parse_args (RygelBrowse* self) {
	gboolean _tmp0;
	g_return_if_fail (self != NULL);
	gupnp_service_action_get (self->priv->action, "ObjectID", G_TYPE_STRING, &self->object_id, "BrowseFlag", G_TYPE_STRING, &self->browse_flag, "Filter", G_TYPE_STRING, &self->filter, "StartingIndex", G_TYPE_UINT, &self->index, "RequestedCount", G_TYPE_UINT, &self->requested_count, "SortCriteria", G_TYPE_STRING, &self->sort_criteria, NULL);
	_tmp0 = FALSE;
	if (self->browse_flag != NULL) {
		_tmp0 = _vala_strcmp0 (self->browse_flag, "BrowseDirectChildren") == 0;
	} else {
		_tmp0 = FALSE;
	}
	/* BrowseFlag */
	if (_tmp0) {
		self->priv->fetch_metadata = FALSE;
	} else {
		gboolean _tmp1;
		_tmp1 = FALSE;
		if (self->browse_flag != NULL) {
			_tmp1 = _vala_strcmp0 (self->browse_flag, "BrowseMetadata") == 0;
		} else {
			_tmp1 = FALSE;
		}
		if (_tmp1) {
			self->priv->fetch_metadata = TRUE;
		} else {
			GError* _tmp2;
			self->priv->fetch_metadata = FALSE;
			_tmp2 = NULL;
			rygel_browse_handle_error (self, _tmp2 = g_error_new_literal (RYGEL_CONTENT_DIRECTORY_ERROR, RYGEL_CONTENT_DIRECTORY_ERROR_INVALID_ARGS, "Invalid Args"));
			(_tmp2 == NULL) ? NULL : (_tmp2 = (g_error_free (_tmp2), NULL));
			return;
		}
	}
	/* ObjectID */
	if (self->object_id == NULL) {
		/* Stupid Xbox */
		gupnp_service_action_get (self->priv->action, "ContainerID", G_TYPE_STRING, &self->object_id, NULL);
	}
	if (self->object_id == NULL) {
		GError* _tmp3;
		/* Sorry we can't do anything without ObjectID*/
		_tmp3 = NULL;
		rygel_browse_handle_error (self, _tmp3 = g_error_new_literal (RYGEL_CONTENT_DIRECTORY_ERROR, RYGEL_CONTENT_DIRECTORY_ERROR_NO_SUCH_OBJECT, "No such object"));
		(_tmp3 == NULL) ? NULL : (_tmp3 = (g_error_free (_tmp3), NULL));
		return;
	}
	rygel_browse_fetch_media_object (self);
}


static void rygel_browse_conclude (RygelBrowse* self) {
	const char* _tmp0;
	char* didl;
	g_return_if_fail (self != NULL);
	/* End DIDL-Lite fragment */
	gupnp_didl_lite_writer_end_didl_lite ((GUPnPDIDLLiteWriter*) self->priv->didl_writer);
	/* Retrieve generated string */
	_tmp0 = NULL;
	didl = (_tmp0 = gupnp_didl_lite_writer_get_string ((GUPnPDIDLLiteWriter*) self->priv->didl_writer), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	if (self->update_id == G_MAXUINT32) {
		self->update_id = (guint) self->priv->system_update_id;
	}
	/* Set action return arguments */
	gupnp_service_action_set (self->priv->action, "Result", G_TYPE_STRING, didl, "NumberReturned", G_TYPE_UINT, self->number_returned, "TotalMatches", G_TYPE_UINT, self->total_matches, "UpdateID", G_TYPE_UINT, self->update_id, NULL);
	gupnp_service_action_return (self->priv->action);
	g_signal_emit_by_name ((RygelStateMachine*) self, "completed");
	didl = (g_free (didl), NULL);
}


static void rygel_browse_handle_error (RygelBrowse* self, GError* error) {
	g_return_if_fail (self != NULL);
	if (error->domain == RYGEL_CONTENT_DIRECTORY_ERROR) {
		g_warning ("rygel-browse.vala:226: Failed to browse '%s': %s\n", self->object_id, error->message);
		gupnp_service_action_return_error (self->priv->action, (guint) error->code, error->message);
	} else {
		g_warning ("rygel-browse.vala:231: Failed to browse '%s': %s\n", self->object_id, error->message);
		gupnp_service_action_return_error (self->priv->action, (guint) 701, error->message);
	}
	g_signal_emit_by_name ((RygelStateMachine*) self, "completed");
}


static void rygel_browse_serialize_children (RygelBrowse* self, GeeList* children) {
	GError * inner_error;
	g_return_if_fail (self != NULL);
	inner_error = NULL;
	if (children == NULL) {
		GError* _tmp0;
		_tmp0 = NULL;
		rygel_browse_handle_error (self, _tmp0 = g_error_new_literal (RYGEL_CONTENT_DIRECTORY_ERROR, RYGEL_CONTENT_DIRECTORY_ERROR_NO_SUCH_OBJECT, "No such object"));
		(_tmp0 == NULL) ? NULL : (_tmp0 = (g_error_free (_tmp0), NULL));
		return;
	}
	{
		gint i;
		/* serialize all children */
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) children); i++) {
			{
				RygelMediaObject* _tmp1;
				_tmp1 = NULL;
				rygel_didl_lite_writer_serialize (self->priv->didl_writer, _tmp1 = (RygelMediaObject*) gee_list_get (children, i), &inner_error);
				if (inner_error != NULL) {
					goto __catch9_g_error;
					goto __finally9;
				}
				(_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL));
			}
			goto __finally9;
			__catch9_g_error:
			{
				GError * err;
				err = inner_error;
				inner_error = NULL;
				{
					rygel_browse_handle_error (self, err);
					(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
					return;
				}
			}
			__finally9:
			if (inner_error != NULL) {
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return;
			}
		}
	}
	/* Conclude the successful Browse action*/
	rygel_browse_conclude (self);
}


static void _rygel_browse_on_children_fetched_gasync_ready_callback (GObject* source_object, GAsyncResult* res, gpointer self) {
	rygel_browse_on_children_fetched (self, source_object, res);
}


static void rygel_browse_fetch_children (RygelBrowse* self) {
	RygelMediaContainer* _tmp0;
	RygelMediaContainer* container;
	g_return_if_fail (self != NULL);
	_tmp0 = NULL;
	container = (_tmp0 = RYGEL_MEDIA_CONTAINER (self->priv->media_object), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	rygel_media_container_get_children (container, self->index, self->requested_count, self->priv->cancellable, _rygel_browse_on_children_fetched_gasync_ready_callback, self);
	(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
}


static void rygel_browse_on_children_fetched (RygelBrowse* self, GObject* source_object, GAsyncResult* res) {
	GError * inner_error;
	RygelMediaContainer* _tmp0;
	RygelMediaContainer* container;
	g_return_if_fail (self != NULL);
	g_return_if_fail (source_object != NULL);
	g_return_if_fail (res != NULL);
	inner_error = NULL;
	_tmp0 = NULL;
	container = (_tmp0 = RYGEL_MEDIA_CONTAINER (source_object), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	{
		GeeList* children;
		children = rygel_media_container_get_children_finish (container, res, &inner_error);
		if (inner_error != NULL) {
			goto __catch10_g_error;
			goto __finally10;
		}
		self->number_returned = (guint) gee_collection_get_size ((GeeCollection*) children);
		rygel_browse_serialize_children (self, children);
		(children == NULL) ? NULL : (children = (g_object_unref (children), NULL));
	}
	goto __finally10;
	__catch10_g_error:
	{
		GError * err;
		err = inner_error;
		inner_error = NULL;
		{
			rygel_browse_handle_error (self, err);
			(err == NULL) ? NULL : (err = (g_error_free (err), NULL));
		}
	}
	__finally10:
	if (inner_error != NULL) {
		(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	(container == NULL) ? NULL : (container = (g_object_unref (container), NULL));
}


static void rygel_browse_class_init (RygelBrowseClass * klass) {
	rygel_browse_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (RygelBrowsePrivate));
	G_OBJECT_CLASS (klass)->finalize = rygel_browse_finalize;
}


static void rygel_browse_rygel_state_machine_interface_init (RygelStateMachineIface * iface) {
	rygel_browse_rygel_state_machine_parent_iface = g_type_interface_peek_parent (iface);
	iface->run = rygel_browse_real_run;
}


static void rygel_browse_instance_init (RygelBrowse * self) {
	self->priv = RYGEL_BROWSE_GET_PRIVATE (self);
}


static void rygel_browse_finalize (GObject* obj) {
	RygelBrowse * self;
	self = RYGEL_BROWSE (obj);
	self->object_id = (g_free (self->object_id), NULL);
	self->browse_flag = (g_free (self->browse_flag), NULL);
	self->filter = (g_free (self->filter), NULL);
	self->sort_criteria = (g_free (self->sort_criteria), NULL);
	(self->priv->media_object == NULL) ? NULL : (self->priv->media_object = (g_object_unref (self->priv->media_object), NULL));
	(self->priv->root_container == NULL) ? NULL : (self->priv->root_container = (g_object_unref (self->priv->root_container), NULL));
	(self->priv->didl_writer == NULL) ? NULL : (self->priv->didl_writer = (g_object_unref (self->priv->didl_writer), NULL));
	(self->priv->cancellable == NULL) ? NULL : (self->priv->cancellable = (g_object_unref (self->priv->cancellable), NULL));
	G_OBJECT_CLASS (rygel_browse_parent_class)->finalize (obj);
}


GType rygel_browse_get_type (void) {
	static GType rygel_browse_type_id = 0;
	if (rygel_browse_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (RygelBrowseClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) rygel_browse_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (RygelBrowse), 0, (GInstanceInitFunc) rygel_browse_instance_init, NULL };
		static const GInterfaceInfo rygel_state_machine_info = { (GInterfaceInitFunc) rygel_browse_rygel_state_machine_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		rygel_browse_type_id = g_type_register_static (G_TYPE_OBJECT, "RygelBrowse", &g_define_type_info, 0);
		g_type_add_interface_static (rygel_browse_type_id, RYGEL_TYPE_STATE_MACHINE, &rygel_state_machine_info);
	}
	return rygel_browse_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




