#define __GNOME_PRINT_PAPER_SELECTOR_C__

/*
 * GNOME Paper Selector
 *
 * Copyright (C) 1999 James Henstridge <james@daa.com.au>
 * All rights reserved.
 *
 * This replaces the paper selector by Dirk Luetjens.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_svp.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libart_lgpl/art_rect_svp.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkframe.h>
#include <gtk/gtktable.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <libgnomecanvas/gnome-canvas.h>
#include <libgnomeprint/gnome-print-paper.h>
#include "gnome-print-i18n.h"
#include "gnome-print-paper-selector.h"

#define noGPP_VERBOSE

/* Paper preview CanvasItem */

#define GNOME_TYPE_PAPER_PREVIEW_ITEM (gnome_paper_preview_item_get_type ())
#define GNOME_PAPER_PREVIEW_ITEM(o) (G_TYPE_CHECK_INSTANCE_CAST ((o), GNOME_TYPE_PAPER_PREVIEW_ITEM, GnomePaperPreviewItem))
#define GNOME_PAPER_PREVIEW_ITEM_CLASS(k) (G_TYPE_CHECK_CLASS_CAST ((k), GNOME_TYPE_PAPER_PREVIEW_ITEM, GnomePaperPreviewItemClass))
#define GNOME_IS_PAPER_PREVIEW_ITEM(o) (G_TYPE_CHECK_INSTANCE_TYPE ((o), GNOME_TYPE_PAPER_PREVIEW_ITEM))
#define GNOME_IS_PAPER_PREVIEW_ITEM_CLASS(k) (G_TYPE_CHECK_CLASS_TYPE ((k), GNOME_TYPE_PAPER_PREVIEW_ITEM))
#define GNOME_PAPER_PREVIEW_ITEM_GET_CLASS(o) (G_TYPE_INSTANCE_GET_CLASS ((o), GNOME_TYPE_PAPER_PREVIEW_ITEM, GnomePaperPreviewItemClass))

typedef struct _GnomePaperPreviewItem GnomePaperPreviewItem;
typedef struct _GnomePaperPreviewItemClass GnomePaperPreviewItemClass;

GType gnome_paper_preview_item_get_type (void);

GnomeCanvasItem *gnome_paper_preview_item_new (GnomePrintConfig *config, GnomeCanvasGroup *parent);

/* EXPERIMENTAL */
/* These will be remodelled via properties, if makes sense */

void gnome_paper_preview_item_set_physical_size (GnomePaperPreviewItem *item, gdouble width, gdouble height);
/* NB! Only first 4 values are used */
void gnome_paper_preview_item_set_physical_orientation (GnomePaperPreviewItem *item, const gdouble *orientation);
/* NB! Only first 4 values are used */
void gnome_paper_preview_item_set_logical_orientation (GnomePaperPreviewItem *item, const gdouble *orientation);
void gnome_paper_preview_item_set_physical_margins (GnomePaperPreviewItem *item, gdouble l, gdouble r, gdouble t, gdouble b);
void gnome_paper_preview_item_set_logical_margins (GnomePaperPreviewItem *item, gdouble l, gdouble r, gdouble t, gdouble b);
/* NB! Layout size is applicable width and height in layout subpage */
void gnome_paper_preview_item_set_layout (GnomePaperPreviewItem *item, const GSList *affines, gdouble width, gdouble height);

#define MM(v) ((v) * 72.0 / 25.4)
#define CM(v) ((v) * 72.0 / 2.54)
#define M(v) ((v) * 72.0 / 0.0254)

#define EPSILON 1e-9
#define PAD 4
#define SHADOW_SIZE 5
#define BORDER_COLOR  0x000000ff
#define SHADOW_COLOR  0x00003fff
#define PAGE_COLOR    0xffffdfff
#define PMARGIN_COLOR 0xff0000bf
#define LMARGIN_COLOR 0x0000ffbf
#define ARROW_COLOR   0xff7f003f

struct _GnomePaperPreviewItem {
	GnomeCanvasItem canvasitem;

	/* Width and height of physical page */
	gdouble pw, ph;
	/* Transform from 1x1 abstract physical page to 1x1 abstract printing area */
	gdouble porient[6];

	gdouble lorient[6];
	gdouble pml, pmr, pmt, pmb;
	gdouble lml, lmr, lmt, lmb;
	gdouble lyw, lyh;
	gint num_affines;
	gdouble *affines;

	/* fixme: remove this (Lauris) */
	/* Physical page -> Printed area */
	gdouble PP2PA[6];
	/* Printed area width and height */
	gdouble PAW, PAH;
	/* Layout page width and height */
	gdouble LYW, LYH;
	/* Logical page -> Layout page */
	gdouble LP2LY[6];
	/* Logical width and height */
	gdouble LW, LH;
	/* State data */
	gdouble PPDP2C[6], PP2C[6];
	ArtIRect area;

	/* Render data */
	ArtSVP *up, *right;
};

struct _GnomePaperPreviewItemClass {
	GnomeCanvasItemClass parent_class;
};

static void gnome_paper_preview_item_class_init (GnomePaperPreviewItemClass *klass);
static void gnome_paper_preview_item_init (GnomePaperPreviewItem *item);
static void gnome_paper_preview_item_finalize (GObject *object);

static void gnome_paper_preview_item_update (GnomeCanvasItem *item, gdouble *affine, ArtSVP *clip, gint flags);
static void gnome_paper_preview_item_render (GnomeCanvasItem *item, GnomeCanvasBuf *buf);

static void gppi_hline (GnomeCanvasBuf *buf, gint y, gint xs, gint xe, guint32 rgba);
static void gppi_vline (GnomeCanvasBuf *buf, gint x, gint ys, gint ye, guint32 rgba);
static void gppi_rect (GnomeCanvasBuf *buf, gint xs, gint ys, gint xe, gint ye, guint32 rgba);
static void gppi_tvline (GnomeCanvasBuf *buf, gdouble x, gdouble sy, gdouble ey, gdouble *affine, guint32 rgba);
static void gppi_thline (GnomeCanvasBuf *buf, gdouble y, gdouble sx, gdouble ex, gdouble *affine, guint32 rgba);

static GnomeCanvasItemClass *item_parent_class;

GType
gnome_paper_preview_item_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePaperPreviewItemClass),
			NULL, NULL,
			(GClassInitFunc) gnome_paper_preview_item_class_init,
			NULL, NULL,
			sizeof (GnomePaperPreviewItem),
			0,
			(GInstanceInitFunc) gnome_paper_preview_item_init
		};
		type = g_type_register_static (GNOME_TYPE_CANVAS_ITEM, "GnomePaperPreviewItem", &info, 0);
	}
	return type;
}

static void
gnome_paper_preview_item_class_init (GnomePaperPreviewItemClass *klass)
{
	GObjectClass *object_class;
	GnomeCanvasItemClass *item_class;

	object_class = G_OBJECT_CLASS (klass);
	item_class = GNOME_CANVAS_ITEM_CLASS (klass);

	item_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_paper_preview_item_finalize;

	item_class->update = gnome_paper_preview_item_update;
	item_class->render = gnome_paper_preview_item_render;
}

static void
gnome_paper_preview_item_init (GnomePaperPreviewItem *pp)
{
	pp->pw = MM(210);
	pp->ph = MM(297);
	pp->lyw = 1.0;
	pp->lyh = 1.0;
	art_affine_identity (pp->porient);
	art_affine_identity (pp->lorient);
	pp->pml = pp->pmr = pp->pmt = pp->pmb = MM(5);
	pp->lml = pp->lmr = pp->lmt = pp->lmb = MM(15);
	pp->num_affines = 1;
	pp->affines = g_new (gdouble, 6);
	art_affine_identity (pp->affines);
}

static void
gnome_paper_preview_item_finalize (GObject *object)
{
	GnomePaperPreviewItem *pp;

	pp = GNOME_PAPER_PREVIEW_ITEM (object);

	if (pp->affines) {
		g_free (pp->affines);
		pp->affines = NULL;
		pp->num_affines = 0;
	}

	if (pp->up) {
		art_svp_free (pp->up);
		pp->up = NULL;
	}

	if (pp->right) {
		art_svp_free (pp->right);
		pp->right = NULL;
	}

	G_OBJECT_CLASS (item_parent_class)->finalize (object);
}

#define WIDGET_WIDTH(w) (GTK_WIDGET (w)->allocation.width)
#define WIDGET_HEIGHT(w) (GTK_WIDGET (w)->allocation.height)

#ifdef GPP_VERBOSE
#define PRINT_2(s,a,b) g_print ("%s %g %g\n", (s), (a), (b))
#define PRINT_DRECT(s,a) g_print ("%s %g %g %g %g\n", (s), (a)->x0, (a)->y0, (a)->x1, (a)->y1)
#define PRINT_AFFINE(s,a) g_print ("%s %g %g %g %g %g %g\n", (s), *(a), *((a) + 1), *((a) + 2), *((a) + 3), *((a) + 4), *((a) + 5))
#else
#define PRINT_2(s,a,b)
#define PRINT_DRECT(s,a)
#define PRINT_AFFINE(s,a)
#endif

#define SQ(v) ((v) * (v))

static void
gnome_paper_preview_item_update (GnomeCanvasItem *item, gdouble *affine, ArtSVP *clip, gint flags)
{
	GnomePaperPreviewItem *pp;
	ArtDRect area, r, PPDC;
	gdouble xscale, yscale, scale, t;
	gdouble PPDP2C[6], a[6], b[6];
	ArtVpath arrow[] = {{ART_MOVETO, -1, 0.001}, {ART_LINETO, 0, 1}, {ART_LINETO, 1, 0}, {ART_LINETO, -1, -0.001}, {ART_END, 0, 0}};
	ArtVpath *vpath;
	ArtSVP *svp;

	pp = GNOME_PAPER_PREVIEW_ITEM (item);

	if (((GnomeCanvasItemClass *) item_parent_class)->update)
		((GnomeCanvasItemClass *) item_parent_class)->update (item, affine, clip, flags);

	gnome_canvas_item_reset_bounds (item);

	if (pp->up) {
		art_svp_free (pp->up);
		pp->up = NULL;
	}

	if (pp->right) {
		art_svp_free (pp->right);
		pp->right = NULL;
	}

	/* Req redraw old */
	gnome_canvas_request_redraw (item->canvas,
				     pp->area.x0 - 1, pp->area.y0 - 1,
				     pp->area.x1 + SHADOW_SIZE + 1, pp->area.y1 + SHADOW_SIZE + 1);

	/* Now comes the fun part */

	if (pp->num_affines < 1) return;
	if ((fabs (pp->pw) < EPSILON) || (fabs (pp->ph) < EPSILON)) return;

	/* Initial setup */
	/* Calculate PP2PA */
	/* We allow only rectilinear setups, so we can cheat */
	pp->PP2PA[0] = pp->porient[0];
	pp->PP2PA[1] = pp->porient[1];
	pp->PP2PA[2] = pp->porient[2];
	pp->PP2PA[3] = pp->porient[3];
	t = pp->pw * pp->PP2PA[0] + pp->ph * pp->PP2PA[2];
	pp->PP2PA[4] = (t < 0) ? -t : 0.0;
	t = pp->pw * pp->PP2PA[1] + pp->ph * pp->PP2PA[3];
	pp->PP2PA[5] = (t < 0) ? -t : 0.0;
	PRINT_AFFINE ("PP2PA:", &pp->PP2PA[0]);

	/* PPDP - Physical Page Dimensions in Printer */
	/* A: PhysicalPage X PhysicalOrientation X TRANSLATE -> Physical Page in Printer */
	area.x0 = 0.0;
	area.y0 = 0.0;
	area.x1 = pp->pw;
	area.y1 = pp->ph;
	art_drect_affine_transform (&r, &area, pp->PP2PA);
	pp->PAW = r.x1 - r.x0;
	pp->PAH = r.y1 - r.y0;
	if ((pp->PAW < EPSILON) || (pp->PAH < EPSILON)) return;

	/* Now we have to find the size of layout page */
	/* Again, knowing that layouts are rectilinear helps us */
	art_affine_invert (a, pp->affines);
	PRINT_AFFINE ("INV LY:", &a[0]);
	pp->LYW = pp->lyw * fabs (pp->pw * a[0] + pp->ph * a[2]);
	pp->LYH = pp->lyh * fabs (pp->pw * a[1] + pp->ph * a[3]);
	PRINT_2 ("LY Dimensions:", pp->LYW, pp->LYH);

	/* Calculate LP2LY */
	/* We allow only rectilinear setups, so we can cheat */
	pp->LP2LY[0] = pp->lorient[0];
	pp->LP2LY[1] = pp->lorient[1];
	pp->LP2LY[2] = pp->lorient[2];
	pp->LP2LY[3] = pp->lorient[3];
	/* Delay */
	pp->LP2LY[4] = 0.0;
	pp->LP2LY[5] = 0.0;
	/* Meanwhile find logical width and height */
	area.x0 = 0.0;
	area.y0 = 0.0;
	area.x1 = pp->LYW;
	area.y1 = pp->LYH;
	art_affine_invert (a, pp->LP2LY);
	art_drect_affine_transform (&r, &area, a);
	pp->LW = r.x1 - r.x0;
	pp->LH = r.y1 - r.y0;
	if ((pp->LW < EPSILON) || (pp->LH < EPSILON)) return;
	PRINT_2 ("L Dimensions", pp->LW, pp->LH);
	/* Now complete matrix calculation */
	t = pp->LW * pp->LP2LY[0] + pp->LH * pp->LP2LY[2];
	pp->LP2LY[4] = (t < 0) ? -t : 0.0;
	t = pp->LW * pp->LP2LY[1] + pp->LH * pp->LP2LY[3];
	pp->LP2LY[5] = (t < 0) ? -t : 0.0;
	PRINT_AFFINE ("LP2LY:", &pp->LP2LY[0]);

#if 0
	/* Find actual size */
	/* Do not forget, that lw and lh are relative to pw and ph */
	/* Also, do not forget layout affine rescaling in layout section */
	if ((fabs (pp->lorient[1]) + fabs (pp->lorient[2])) > (fabs (pp->lorient[0]) + fabs (pp->lorient[3]))) {
		pp->aw = pp->LYH;
		pp->ah = pp->LYW;
	} else {
		pp->aw = pp->LYW;
		pp->ah = pp->LYH;
	}

	/* Write correct translation to Logical orientation */
	t = pp->aw * pp->lorient[0] + pp->ah * pp->lorient[2];
	if (t < 0) pp->lorient[4] = -t;
	t = pp->aw * pp->lorient[1] + pp->ah * pp->lorient[3];
	if (t < 0) pp->lorient[5] = -t;
	PRINT_AFFINE ("Logical Orientation:", &pp->lorient[0]);
#endif

	/* PPDC - Physical Page Dimensions on Canvas */

	/* Now we calculate PPDP -> Canvas transformation */
	/* fixme: This should be done by group or something? (Lauris)  */

	/* Find view box in canvas coordinates */
	gnome_canvas_window_to_world (item->canvas, 0, 0, &PPDC.x0, &PPDC.y0);
	gnome_canvas_w2c_d (item->canvas, PPDC.x0, PPDC.y0, &PPDC.x0, &PPDC.y0);
	gnome_canvas_window_to_world (item->canvas, WIDGET_WIDTH (item->canvas), WIDGET_HEIGHT (item->canvas), &PPDC.x1, &PPDC.y1);
	gnome_canvas_w2c_d (item->canvas, PPDC.x1, PPDC.y1, &PPDC.x1, &PPDC.y1);
	PRINT_DRECT ("Visible area:", &PPDC);
	/* Clip it by shadow and stuff */
	PPDC.x0 += PAD;
	PPDC.y0 += PAD;
	PPDC.x1 -= (SHADOW_SIZE + PAD);
	PPDC.y1 -= (SHADOW_SIZE + PAD);
	PRINT_DRECT ("Drawable area:", &PPDC);
	/* Check for too small drawing area */
	if ((PPDC.x0 >= PPDC.x1) || (PPDC.y0 >= PPDC.y1)) return;
	/* Crop to right aspect ratio */
	xscale = (PPDC.x1 - PPDC.x0) / pp->PAW;
	yscale = (PPDC.y1 - PPDC.y0) / pp->PAH;
	scale = MIN (xscale, yscale);
	t = 0.5 * ((PPDC.x1 - PPDC.x0) - scale * pp->PAW);
	PPDC.x0 += t;
	PPDC.x1 -= t;
	t = 0.5 * ((PPDC.y1 - PPDC.y0) - scale * pp->PAH);
	PPDC.y0 += t;
	PPDC.y1 -= t;
	PRINT_DRECT ("Actual page area:", &PPDC);

	/* Find physical page area -> canvas transformation */
	PPDP2C[0] = scale;
	PPDP2C[1] = 0.0;
	PPDP2C[2] = 0.0;
	PPDP2C[3] = -scale;
	PPDP2C[4] = PPDC.x0;
	PPDP2C[5] = PPDC.y0 + (PPDC.y1 - PPDC.y0);
	PRINT_AFFINE ("PPDP -> Canvas:", &PPDP2C[0]);

	/* Find canvas area in integers */
	art_drect_to_irect (&pp->area, &PPDC);

	memcpy (pp->PPDP2C, PPDP2C, 6 * sizeof (gdouble));
	art_affine_multiply (pp->PP2C, pp->PP2PA, pp->PPDP2C);
	PRINT_AFFINE ("Physical Page to Canvas:", &pp->PP2C[0]);

	/* Create SVP-s */
	a[0] = 0.2 * pp->pw;
	a[1] = 0.0;
	a[2] = 0.0;
	a[3] = 0.2 * pp->pw;
	a[4] = 0.5 * pp->pw;
	a[5] = 0.0;
	art_affine_multiply (b, a, pp->PP2C);
	vpath = art_vpath_affine_transform (arrow, b);
	svp = art_svp_from_vpath (vpath);
	art_free (vpath);
	pp->up = art_svp_rewind_uncrossed (svp, ART_WIND_RULE_NONZERO);
	art_svp_free (svp);
	art_drect_svp (&area, pp->up);
	PRINT_DRECT ("Up arrow:", &area);
	a[0] = 0.0;
	a[1] = -0.2 * pp->pw;
	a[2] = -0.2 * pp->pw;
	a[3] = 0.0;
	a[4] = pp->pw;
	a[5] = 0.5 * pp->ph;
	art_affine_multiply (b, a, pp->PP2C);
	vpath = art_vpath_affine_transform (arrow, b);
	svp = art_svp_from_vpath (vpath);
	pp->right = art_svp_rewind_uncrossed (svp, ART_WIND_RULE_NONZERO);
	art_svp_free (svp);
	art_free (vpath);
	art_drect_svp (&area, pp->up);
	PRINT_DRECT ("Up arrow:", &area);

	/* Req redraw new */
	gnome_canvas_request_redraw (item->canvas,
				     pp->area.x0 - 1, pp->area.y0 - 1,
				     pp->area.x1 + SHADOW_SIZE + 1, pp->area.y1 + SHADOW_SIZE + 1);

	item->x1 = pp->area.x0 - 1;
	item->y1 = pp->area.y0 - 1;
	item->x2 = pp->area.x1 + SHADOW_SIZE + 1;
	item->y2 = pp->area.y1 + SHADOW_SIZE + 1;
}

static void
gnome_paper_preview_item_render (GnomeCanvasItem *item, GnomeCanvasBuf *buf)
{
	GnomePaperPreviewItem *pp;

	pp = GNOME_PAPER_PREVIEW_ITEM (item);

	if ((pp->area.x0 < buf->rect.x1) &&
	    (pp->area.y0 < buf->rect.y1) &&
	    ((pp->area.x1 + SHADOW_SIZE) >= buf->rect.x0) &&
	    ((pp->area.y1 + SHADOW_SIZE) >= buf->rect.y0)) {
		gint imargin, i;
		/* Initialize buffer, if needed */
		gnome_canvas_buf_ensure_buf (buf);
		buf->is_buf = TRUE;
		buf->is_bg = FALSE;
		/* Top */
		gppi_hline (buf, pp->area.y0, pp->area.x0, pp->area.x1, BORDER_COLOR);
		/* Bottom */
		gppi_hline (buf, pp->area.y1, pp->area.x0, pp->area.x1, BORDER_COLOR);
		/* Left */
		gppi_vline (buf, pp->area.x0, pp->area.y0 + 1, pp->area.y1 - 1, BORDER_COLOR);
		/* Right */
		gppi_vline (buf, pp->area.x1, pp->area.y0 + 1, pp->area.y1 - 1, BORDER_COLOR);
		/* Shadow */
		if (SHADOW_SIZE > 0) {
			/* Right */
			gppi_rect (buf, pp->area.x1 + 1, pp->area.y0 + SHADOW_SIZE,
				   pp->area.x1 + SHADOW_SIZE, pp->area.y1 + SHADOW_SIZE, SHADOW_COLOR);
			/* Bottom */
			gppi_rect (buf, pp->area.x0 + SHADOW_SIZE, pp->area.y1 + 1,
				   pp->area.x1, pp->area.y1 + SHADOW_SIZE, SHADOW_COLOR);
		}
		/* Fill */
		gppi_rect (buf, pp->area.x0 + 1, pp->area.y0 + 1,
			   pp->area.x1 - 1, pp->area.y1 - 1, PAGE_COLOR);

		/* Arrows */
		if (pp->up) gnome_canvas_render_svp (buf, pp->up, ARROW_COLOR);
		if (pp->right) gnome_canvas_render_svp (buf, pp->right, ARROW_COLOR);

		/* Fun part */

		/* Physical margins */
		imargin = (gint) fabs (pp->pml * pp->PPDP2C[0]);
		if (imargin > 0) gppi_vline (buf, pp->area.x0 + imargin, pp->area.y0 + 1, pp->area.y1 - 1, PMARGIN_COLOR);
		imargin = (gint) fabs (pp->pmr * pp->PPDP2C[0]);
		if (imargin > 0) gppi_vline (buf, pp->area.x1 - imargin, pp->area.y0 + 1, pp->area.y1 - 1, PMARGIN_COLOR);
		imargin = (gint) fabs (pp->pmt * pp->PPDP2C[3]);
		if (imargin > 0) gppi_hline (buf, pp->area.y0 + imargin, pp->area.x0 + 1, pp->area.x1 - 1, PMARGIN_COLOR);
		imargin = (gint) fabs (pp->pmb * pp->PPDP2C[3]);
		if (imargin > 0) gppi_hline (buf, pp->area.y1 - imargin, pp->area.x0 + 1, pp->area.x1 - 1, PMARGIN_COLOR);

		/* Extra fun */

		for (i = 0; i < pp->num_affines; i++) {
			gdouble l2p[6], l2c[6], lp2c[6];
			gdouble w, h, y;

			/* Calculate Layout -> Physical Page affine */
			memcpy (l2p, pp->affines + 6 * i, 6 * sizeof (gdouble));
			l2p[4] *= pp->pw;
			l2p[5] *= pp->ph;
			/* PRINT_AFFINE ("Layout -> Physical:", &l2p[0]); */
			/* Calculate Layout -> Canvas affine */
			art_affine_multiply (l2c, l2p, pp->PP2C);
			/* PRINT_AFFINE ("Layout -> Canvas:", &l2c[0]); */
			/* Calcualte Logical Page -> Canvas affine */
			art_affine_multiply (lp2c, pp->LP2LY, l2c);
			/* PRINT_AFFINE ("Logical Page -> Canvas:", &lp2c[0]); */

			/* Draw logical margins */
			gppi_tvline (buf, pp->lml, 0, pp->LH, lp2c, LMARGIN_COLOR);
			gppi_tvline (buf, pp->LW - pp->lmr, 0, pp->LH, lp2c, LMARGIN_COLOR);
			gppi_thline (buf, pp->LH - pp->lmt, 0, pp->LW, lp2c, LMARGIN_COLOR);
			gppi_thline (buf, pp->lmb, 0, pp->LW, lp2c, LMARGIN_COLOR);
			/* Render fancy page */
			w = pp->LW - pp->lml - pp->lmr;
			h = pp->LH - pp->lmt - pp->lmb;
			if ((w > 0) && (h > 0)) {
				ArtDRect a, r;
				y = h;
				if ((y >= CM(5)) && (w > CM(5))) {
					/* 5CM x 5CM box */
					r.x0 = pp->lml + 0;
					r.y0 = pp->lmb + y - CM(5);
					r.x1 = pp->lml + CM(5);
					r.y1 = pp->lmb + y;
					art_drect_affine_transform (&a, &r, lp2c);
					gppi_rect (buf, a.x0, a.y0, a.x1, a.y1, 0x0000007f);
					if (w >= CM(7)) {
						gint l;
						for (l = 0; l < 3; l++) {
							/* Short line */
							r.x0 = pp->lml + CM(6);
							r.y0 = pp->lmb + y - l * CM(2) - CM(1.5);
							r.x1 = pp->lml + w;
							r.y1 = pp->lmb + y - l * CM(2) - CM(0.5);
							art_drect_affine_transform (&a, &r, lp2c);
							gppi_rect (buf, a.x0, a.y0, a.x1, a.y1, 0x0000005f);
						}
					}
					y -= CM(6.5);
				}
				while (y > CM(1)) {
					/* Long line */
					r.x0 = pp->lml + 0;
					r.y0 = pp->lmb + y - CM(1);
					r.x1 = pp->lml + w;
					r.y1 = pp->lmb + y;
					art_drect_affine_transform (&a, &r, lp2c);
					gppi_rect (buf, a.x0, a.y0, a.x1, a.y1, 0x0000005f);
					y -= CM(2);
				}
			}
		}
	}
}

GnomeCanvasItem *
gnome_paper_preview_item_new (GnomePrintConfig *config, GnomeCanvasGroup *parent)
{
	GnomeCanvasItem *item;

	item = gnome_canvas_item_new (parent, GNOME_TYPE_PAPER_PREVIEW_ITEM, NULL);

	return item;
}

/* EXPERIMENTAL */
/* These will be remodelled via properties, if makes sense */

void
gnome_paper_preview_item_set_physical_size (GnomePaperPreviewItem *pp, gdouble width, gdouble height)
{
	pp->pw = CLAMP (MM(1), width, M(10));
	pp->ph = CLAMP (MM(1), height, M(10));

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

void
gnome_paper_preview_item_set_physical_orientation (GnomePaperPreviewItem *pp, const gdouble *orientation)
{
	memcpy (pp->porient, orientation, 6 * sizeof (gdouble));

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

void
gnome_paper_preview_item_set_logical_orientation (GnomePaperPreviewItem *pp, const gdouble *orientation)
{
	memcpy (pp->lorient, orientation, 6 * sizeof (gdouble));

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

void
gnome_paper_preview_item_set_physical_margins (GnomePaperPreviewItem *pp, gdouble l, gdouble r, gdouble t, gdouble b)
{
	l = MAX (0, l);
	r = MAX (0, r);
	t = MAX (0, t);
	b = MAX (0, b);

	if ((l + r) > 0.0 && (l + r) > pp->pw) {
		l = l * pp->pw / (l + r);
		r = r * pp->pw / (l + r);
	}

	if ((t + b) > 0.0 && (t + b) > pp->ph) {
		t = t * pp->ph / (t + b);
		b = b * pp->ph / (t + b);
	}

	pp->pml = l;
	pp->pmr = r;
	pp->pmt = t;
	pp->pmb = b;

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

void
gnome_paper_preview_item_set_logical_margins (GnomePaperPreviewItem *pp, gdouble l, gdouble r, gdouble t, gdouble b)
{
	l = MAX (0, l);
	r = MAX (0, r);
	t = MAX (0, t);
	b = MAX (0, b);

#if 0
	if ((l + r) > 0.0 && (l + r) > pp->lw) {
		l = l * pp->lw / (l + r);
		r = r * pp->lw / (l + r);
	}

	if ((t + b) > 0.0 && (t + b) > pp->lh) {
		t = t * pp->lh / (t + b);
		b = b * pp->lh / (t + b);
	}
#endif

	pp->lml = l;
	pp->lmr = r;
	pp->lmt = t;
	pp->lmb = b;

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

void
gnome_paper_preview_item_set_layout (GnomePaperPreviewItem *pp, const GSList *affines, gdouble width, gdouble height)
{
	if (pp->affines) {
		g_free (pp->affines);
		pp->affines = NULL;
	}

	pp->num_affines = g_slist_length ((GSList *) affines);

	if (pp->num_affines > 0) {
		const GSList *l;
		gint i;
		pp->affines = g_new (gdouble, pp->num_affines * 6);
		i = 0;
		for (l = affines; l != NULL; l = l->next) {
			memcpy (pp->affines + 6 * i, l->data, 6 * sizeof (gdouble));
			i += 1;
		}
	}

	pp->lyw = CLAMP (0.001, width, 1000.0);
	pp->lyh = CLAMP (0.001, height, 1000.0);

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (pp));
}

/* Drawing helpers */

#define RGBA_R(v) ((v) >> 24)
#define RGBA_G(v) (((v) >> 16) & 0xff)
#define RGBA_B(v) (((v) >> 8) & 0xff)
#define RGBA_A(v) ((v) & 0xff)

/* Non-premultiplied alpha composition */

#define COMPOSE(b,f,a) (((255 - (a)) * b + (f * a) + 127) / 255)

static void
gppi_hline (GnomeCanvasBuf *buf, gint y, gint xs, gint xe, guint32 rgba)
{
	if ((y >= buf->rect.y0) && (y < buf->rect.y1)) {
		guint r, g, b, a;
		gint x0, x1, x;
		guchar *p;
		r = RGBA_R (rgba);
		g = RGBA_G (rgba);
		b = RGBA_B (rgba);
		a = RGBA_A (rgba);
		x0 = MAX (buf->rect.x0, xs);
		x1 = MIN (buf->rect.x1, xe + 1);
		p = buf->buf + (y - buf->rect.y0) * buf->buf_rowstride + (x0 - buf->rect.x0) * 3;
		for (x = x0; x < x1; x++) {
			p[0] = COMPOSE (p[0], r, a);
			p[1] = COMPOSE (p[1], g, a);
			p[2] = COMPOSE (p[2], b, a);
			p += 3;
		}
	}
}

static void
gppi_vline (GnomeCanvasBuf *buf, gint x, gint ys, gint ye, guint32 rgba)
{
	if ((x >= buf->rect.x0) && (x < buf->rect.x1)) {
		guint r, g, b, a;
		gint y0, y1, y;
		guchar *p;
		r = RGBA_R (rgba);
		g = RGBA_G (rgba);
		b = RGBA_B (rgba);
		a = RGBA_A (rgba);
		y0 = MAX (buf->rect.y0, ys);
		y1 = MIN (buf->rect.y1, ye + 1);
		p = buf->buf + (y0 - buf->rect.y0) * buf->buf_rowstride + (x - buf->rect.x0) * 3;
		for (y = y0; y < y1; y++) {
			p[0] = COMPOSE (p[0], r, a);
			p[1] = COMPOSE (p[1], g, a);
			p[2] = COMPOSE (p[2], b, a);
			p += buf->buf_rowstride;
		}
	}
}

static void
gppi_rect (GnomeCanvasBuf *buf, gint xs, gint ys, gint xe, gint ye, guint32 rgba)
{
	guint r, g, b, a;
	gint x0, x1, x;
	gint y0, y1, y;
	guchar *p;
	r = RGBA_R (rgba);
	g = RGBA_G (rgba);
	b = RGBA_B (rgba);
	a = RGBA_A (rgba);
	x0 = MAX (buf->rect.x0, xs);
	x1 = MIN (buf->rect.x1, xe + 1);
	y0 = MAX (buf->rect.y0, ys);
	y1 = MIN (buf->rect.y1, ye + 1);
	for (y = y0; y < y1; y++) {
		p = buf->buf + (y - buf->rect.y0) * buf->buf_rowstride + (x0 - buf->rect.x0) * 3;
		for (x = x0; x < x1; x++) {
			p[0] = COMPOSE (p[0], r, a);
			p[1] = COMPOSE (p[1], g, a);
			p[2] = COMPOSE (p[2], b, a);
			p += 3;
		}
	}
}

static void
gppi_tvline (GnomeCanvasBuf *buf, gdouble x, gdouble sy, gdouble ey, gdouble *affine, guint32 rgba)
{
	gdouble x0, y0, x1, y1;

	x0 = affine[0] * x + affine[2] * sy + affine[4];
	y0 = affine[1] * x + affine[3] * sy + affine[5];
	x1 = affine[0] * x + affine[2] * ey + affine[4];
	y1 = affine[1] * x + affine[3] * ey + affine[5];

	if (fabs (x1 - x0) > fabs (y1 - y0)) {
		gppi_hline (buf, 0.5 * (y0 + y1), MIN (x0, x1), MAX (x0, x1), rgba);
	} else {
		gppi_vline (buf, 0.5 * (x0 + x1), MIN (y0, y1), MAX (y0, y1), rgba);
	}
}

static void
gppi_thline (GnomeCanvasBuf *buf, gdouble y, gdouble sx, gdouble ex, gdouble *affine, guint32 rgba)
{
	gdouble x0, y0, x1, y1;

	x0 = affine[0] * sx + affine[2] * y + affine[4];
	y0 = affine[1] * sx + affine[3] * y + affine[5];
	x1 = affine[0] * ex + affine[2] * y + affine[4];
	y1 = affine[1] * ex + affine[3] * y + affine[5];

	if (fabs (x1 - x0) > fabs (y1 - y0)) {
		gppi_hline (buf, 0.5 * (y0 + y1), MIN (x0, x1), MAX (x0, x1), rgba);
	} else {
		gppi_vline (buf, 0.5 * (x0 + x1), MIN (y0, y1), MAX (y0, y1), rgba);
	}
}


/*
 * GnomePaperPreview widget
 */

struct _GnomePaperPreview {
	GtkHBox box;

	GtkWidget *canvas;

	GnomeCanvasItem *item;
};

struct _GnomePaperPreviewClass {
	GtkHBoxClass parent_class;
};

static void gnome_paper_preview_class_init (GnomePaperPreviewClass *klass);
static void gnome_paper_preview_init (GnomePaperPreview *preview);
static void gnome_paper_preview_finalize (GObject *object);

static void gnome_paper_preview_size_allocate (GtkWidget *widget, GtkAllocation *allocation);

static GtkHBoxClass *preview_parent_class;

GType
gnome_paper_preview_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePaperPreviewClass),
			NULL, NULL,
			(GClassInitFunc) gnome_paper_preview_class_init,
			NULL, NULL,
			sizeof (GnomePaperPreview),
			0,
			(GInstanceInitFunc) gnome_paper_preview_init
		};
		type = g_type_register_static (GTK_TYPE_HBOX, "GnomePaperPreview", &info, 0);
	}
	return type;
}

static void
gnome_paper_preview_class_init (GnomePaperPreviewClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);

	preview_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_paper_preview_finalize;

	widget_class->size_allocate = gnome_paper_preview_size_allocate;
}

static void
gnome_paper_preview_init (GnomePaperPreview *preview)
{
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	preview->canvas = gnome_canvas_new_aa ();
	gtk_widget_pop_colormap ();

	preview->item = gnome_paper_preview_item_new (NULL, gnome_canvas_root (GNOME_CANVAS (preview->canvas)));

	gtk_box_pack_start (GTK_BOX (preview), GTK_WIDGET (preview->canvas), TRUE, TRUE, 0);
	gtk_widget_show (GTK_WIDGET (preview->canvas));
}

static void
gnome_paper_preview_finalize (GObject *object)
{
	GnomePaperPreview *preview;

	preview = GNOME_PAPER_PREVIEW (object);

	preview->item = NULL;
	preview->canvas = NULL;

	G_OBJECT_CLASS (item_parent_class)->finalize (object);
}

static void
gnome_paper_preview_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	GnomePaperPreview *pp;

	pp = GNOME_PAPER_PREVIEW (widget);

	gnome_canvas_set_scroll_region (GNOME_CANVAS (pp->canvas), 0, 0, allocation->width + 50, allocation->height + 50);

	if (((GtkWidgetClass *) preview_parent_class)->size_allocate)
		((GtkWidgetClass *) preview_parent_class)->size_allocate (widget, allocation);

	gnome_canvas_item_request_update (pp->item);
}

GtkWidget *
gnome_paper_preview_new (GnomePrintConfig *config)
{
	return gtk_type_new (GNOME_TYPE_PAPER_PREVIEW);
}

/*
 * GnomePaperSelector widget
 */

struct _GnomePaperSelector {
	GtkHBox box;

	GnomePrintConfig *config;

	GtkWidget *preview;
};

struct _GnomePaperSelectorClass {
	GtkHBoxClass parent_class;
};

static void gnome_paper_selector_class_init (GnomePaperSelectorClass *klass);
static void gnome_paper_selector_init (GnomePaperSelector *selector);
static void gnome_paper_selector_finalize (GObject *object);

static GtkHBoxClass *selector_parent_class;

GType
gnome_paper_selector_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePaperSelectorClass),
			NULL, NULL,
			(GClassInitFunc) gnome_paper_selector_class_init,
			NULL, NULL,
			sizeof (GnomePaperSelector),
			0,
			(GInstanceInitFunc) gnome_paper_selector_init
		};
		type = g_type_register_static (GTK_TYPE_HBOX, "GnomePaperSelector", &info, 0);
	}
	return type;
}

static void
gnome_paper_selector_class_init (GnomePaperSelectorClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);

	selector_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_paper_selector_finalize;
}

static void
gps_paper_activate (GtkWidget *widget, GnomePaperSelector *ps)
{
	GnomePrintPaper *paper;

	paper = gtk_object_get_data (GTK_OBJECT (widget), "paper");

	gnome_paper_preview_item_set_physical_size (GNOME_PAPER_PREVIEW_ITEM (GNOME_PAPER_PREVIEW (ps->preview)->item),
						    paper->width, paper->height);

	/* fixme: */
	if (ps->config) {
		gnome_print_config_set (ps->config, GNOME_PRINT_KEY_PAPER_SIZE, paper->name);
	}
}

static void
gps_paper_menu_create (GnomePaperSelector *selector, GtkWidget *optionmenu)
{
	GtkWidget *m, *i;
	GList *papers, *l;

	papers = gnome_print_paper_get_list ();

	m = gtk_menu_new ();
	gtk_widget_show (m);

	for (l = papers; l != NULL; l = l->next) {
		GnomePrintPaper *paper;
		paper = (GnomePrintPaper *) l->data;
		i = gtk_menu_item_new_with_label (paper->name);
		gtk_object_set_data (GTK_OBJECT (i), "paper", paper);
		gtk_signal_connect (GTK_OBJECT (i), "activate", GTK_SIGNAL_FUNC (gps_paper_activate), selector);
		gtk_widget_show (i);
		gtk_menu_shell_append (GTK_MENU_SHELL (m), i);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), m);

	gnome_print_paper_free_list (l);
}

typedef struct {
	guchar *id, *name;
	gdouble affine[6];
} GPPOrientation;

static const GPPOrientation porient[] = {
	{"R0", "Straight", {1, 0, 0, 1, 0, 0}},
	{"R90", "Rotated 90 degrees", {0, -1, 1, 0, 0, 1}},
	{"R180", "Rotated 180 degrees", {-1, 0, 0, -1, 1, 1}},
	{"R270", "Rotated 270 degrees", {0, 1, -1, 0, 1, 0}}
};

static const GPPOrientation lorient[] = {
	{"R0", "Portrait", {1, 0, 0, 1, 0, 0}},
	{"R90", "Landscape", {0, 1, -1, 0, 0, 1}},
	{"R180", "Upside down portrait", {-1, 0, 0, -1, 1, 1}},
	{"R270", "Upside down landscape", {0, -1, 1, 0, 1, 0}}
};

static void
gps_feed_orientation_activate (GtkWidget *widget, GnomePaperSelector *ps)
{
	GPPOrientation *o;

	o = gtk_object_get_data (GTK_OBJECT (widget), "orientation");

	gnome_paper_preview_item_set_physical_orientation (GNOME_PAPER_PREVIEW_ITEM (GNOME_PAPER_PREVIEW (ps->preview)->item), o->affine);

	/* fixme: */
	if (ps->config) {
		gnome_print_config_set (ps->config, "Settings.Output.Media.PhysicalOrientation", o->id);
	}
}

static void
gps_feed_orientation_menu_create (GnomePaperSelector *selector, GtkWidget *optionmenu)
{
	GtkWidget *m, *i;
	gint n;

	m = gtk_menu_new ();
	gtk_widget_show (m);

	for (n = 0; n < 4; n++) {
		i = gtk_menu_item_new_with_label (porient[n].name);
		gtk_object_set_data (GTK_OBJECT (i), "orientation", (gpointer) &porient[n]);
		gtk_signal_connect (GTK_OBJECT (i), "activate", GTK_SIGNAL_FUNC (gps_feed_orientation_activate), selector);
		gtk_widget_show (i);
		gtk_menu_shell_append (GTK_MENU_SHELL (m), i);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), m);
}

static void
gps_page_orientation_activate (GtkWidget *widget, GnomePaperSelector *ps)
{
	GPPOrientation *o;

	o = gtk_object_get_data (GTK_OBJECT (widget), "orientation");

	gnome_paper_preview_item_set_logical_orientation (GNOME_PAPER_PREVIEW_ITEM (GNOME_PAPER_PREVIEW (ps->preview)->item), o->affine);

	/* fixme: */
	if (ps->config) {
		gnome_print_config_set (ps->config, "Settings.Document.Page.LogicalOrientation", o->id);
	}
}

static void
gps_page_orientation_menu_create (GnomePaperSelector *selector, GtkWidget *optionmenu)
{
	GtkWidget *m, *i;
	gint n;

	m = gtk_menu_new ();
	gtk_widget_show (m);

	for (n = 0; n < 4; n++) {
		i = gtk_menu_item_new_with_label (lorient[n].name);
		gtk_object_set_data (GTK_OBJECT (i), "orientation", (gpointer) &lorient[n]);
		gtk_signal_connect (GTK_OBJECT (i), "activate", GTK_SIGNAL_FUNC (gps_page_orientation_activate), selector);
		gtk_widget_show (i);
		gtk_menu_shell_append (GTK_MENU_SHELL (m), i);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), m);
}

typedef struct {
	guchar *id;
	guchar *name;
	gdouble width, height;
	gint num_affines;
	const gdouble *affines;
} GPPLayout;

static const gdouble lyid[] = {1, 0, 0, 1, 0, 0};
static const gdouble ly21[] = {0, -0.707, 0.707, 0, 0, 1,
			       0, -0.707, 0.707, 0, 0, 0.5};
static const gdouble ly41[] = {0.5, 0, 0, 0.5, 0, 0.5,
			       0.5, 0, 0, 0.5, 0.5, 0.5,
			       0.5, 0, 0, 0.5, 0, 0,
			       0.5, 0, 0, 0.5, 0.5, 0};
static const gdouble ly2[] = {0, -1, 1, 0, 0, 1,
			       0, -1, 1, 0, 0, 0.5};
static const gdouble ly2f[] = {0, -1, 1, 0, 0, 0.5,
			       0, 1, -1, 0, 1, 0.5};

static const GPPLayout layout[] = {
	{"Plain", "Plain", 1, 1, 1, lyid},
	{"2_1", "2 Pages to 1", 0.5, 1, 2, ly21},
	{"4_1", "4 Pages to 1", 0.5, 0.5, 4, ly41},
	{"I2_1", "Divided", 0.5, 1, 2, ly2},
	{"IM2_1", "Folded", 0.5, 1, 2, ly2f}
};
#define NUM_LAYOUTS (sizeof (layout) / sizeof (layout[0]))

static void
gps_layout_activate (GtkWidget *widget, GnomePaperSelector *ps)
{
	GPPLayout *ly;
	GSList *l;
	gint i;

	ly = gtk_object_get_data (GTK_OBJECT (widget), "layout");

	l = NULL;
	for (i = 0; i < ly->num_affines; i++) {
		l = g_slist_prepend (l, (gpointer) (ly->affines + 6 * i));
	}
	l = g_slist_reverse (l);

	gnome_paper_preview_item_set_layout (GNOME_PAPER_PREVIEW_ITEM (GNOME_PAPER_PREVIEW (ps->preview)->item), l, ly->width, ly->height);

	/* fixme: */
	if (ps->config) {
		gnome_print_config_set (ps->config, "Settings.Document.Page.Layout", ly->id);
	}
}

static void
gps_layout_menu_create (GnomePaperSelector *selector, GtkWidget *optionmenu)
{
	GtkWidget *m, *i;
	gint n;

	m = gtk_menu_new ();
	gtk_widget_show (m);

	for (n = 0; n < NUM_LAYOUTS; n++) {
		i = gtk_menu_item_new_with_label (layout[n].name);
		gtk_object_set_data (GTK_OBJECT (i), "layout", (gpointer) &layout[n]);
		gtk_signal_connect (GTK_OBJECT (i), "activate", GTK_SIGNAL_FUNC (gps_layout_activate), selector);
		gtk_widget_show (i);
		gtk_menu_shell_append (GTK_MENU_SHELL (m), i);
	}

	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), m);
}

static void
gnome_paper_selector_init (GnomePaperSelector *selector)
{
	GtkWidget *vb, *f, *t, *l, *m;

	selector->config = NULL;

	/* VBox for controls */
	vb = gtk_vbox_new (FALSE, 4);
	gtk_widget_show (vb);
	gtk_box_pack_start (GTK_BOX (selector), vb, FALSE, FALSE, 4);

	/* Create frame for selection menus */
	f = gtk_frame_new (_("Paper and layout"));
	gtk_widget_show (f);
	gtk_box_pack_start (GTK_BOX (vb), f, FALSE, FALSE, 0);

	/* Create table for packing menus */
	t = gtk_table_new (2, 4, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (t), 4);
	gtk_widget_show (t);
	gtk_container_add (GTK_CONTAINER (f), t);

	l = gtk_label_new (_("Paper size"));
	gtk_widget_show (l);
	gtk_table_attach_defaults (GTK_TABLE (t), l, 0, 1, 0, 1);

	m = gtk_option_menu_new ();
	gtk_widget_show (m);
	gtk_table_attach_defaults (GTK_TABLE (t), m, 1, 2, 0, 1);

	gps_paper_menu_create (selector, m);

	l = gtk_label_new (_("Feed orientation"));
	gtk_widget_show (l);
	gtk_table_attach_defaults (GTK_TABLE (t), l, 0, 1, 1, 2);

	m = gtk_option_menu_new ();
	gtk_widget_show (m);
	gtk_table_attach_defaults (GTK_TABLE (t), m, 1, 2, 1, 2);

	gps_feed_orientation_menu_create (selector, m);

	l = gtk_label_new (_("Page orientation"));
	gtk_widget_show (l);
	gtk_table_attach_defaults (GTK_TABLE (t), l, 0, 1, 2, 3);

	m = gtk_option_menu_new ();
	gtk_widget_show (m);
	gtk_table_attach_defaults (GTK_TABLE (t), m, 1, 2, 2, 3);

	gps_page_orientation_menu_create (selector, m);

	l = gtk_label_new (_("Layout"));
	gtk_widget_show (l);
	gtk_table_attach_defaults (GTK_TABLE (t), l, 0, 1, 3, 4);

	m = gtk_option_menu_new ();
	gtk_widget_show (m);
	gtk_table_attach_defaults (GTK_TABLE (t), m, 1, 2, 3, 4);

	gps_layout_menu_create (selector, m);

	/* Preview frame */
	f = gtk_frame_new (_("Preview"));
	gtk_widget_show (f);
	gtk_box_pack_start (GTK_BOX (selector), f, TRUE, TRUE, 4);

	selector->preview = gnome_paper_preview_new (NULL);
	gtk_widget_set_usize (selector->preview, 320, 320);
	gtk_widget_show (selector->preview);
	gtk_container_add (GTK_CONTAINER (f), selector->preview);
}

static void
gnome_paper_selector_finalize (GObject *object)
{
	GnomePaperSelector *selector;

	selector = GNOME_PAPER_SELECTOR (object);

	selector->preview = NULL;

	if (selector->config) {
		selector->config = gnome_print_config_unref (selector->config);
	}

	G_OBJECT_CLASS (item_parent_class)->finalize (object);
}

GtkWidget *
gnome_paper_selector_new (GnomePrintConfig *config)
{
	GnomePaperSelector *selector;

	selector = gtk_type_new (GNOME_TYPE_PAPER_SELECTOR);

	/* fixme: */
	if (config) {
		selector->config = gnome_print_config_ref (config);
	}

	return (GtkWidget *) selector;
}

#if 0

/* #define PAGELAYOUT_TEST */

#define DEFAULT_PAPER "A4"
#define DEFAULT_UNIT "centimeter"

#include <string.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-uidefs.h>

#include "gnome-paper-selector.h"
#include "gnome-unit-spinner.h"

#include "portrait.xpm"
#include "landscape.xpm"

struct _GnomePaperSelectorPrivate {
  const GnomeUnit *unit;
  const GnomePaper *paper;

  GtkWidget *paper_size, *paper_label;
  GtkWidget *orient_portrait, *orient_landscape;
  GtkWidget *tmargin, *bmargin, *lmargin, *rmargin;
  GtkWidget *scaling;
  GtkWidget *fittopage;

  GtkWidget *darea;

  GdkGC *gc;

  /* position of paper preview */
  gint16 x, y, width, height;

  gboolean block_changed : 1;
};

static const GnomeUnit *default_unit = NULL;
static const GnomeUnit *point_unit = NULL;
static const GnomePaper *default_paper = NULL;

enum {
  CHANGED,
  FITTOPAGE,
  LAST_SIGNAL
};

static guint ps_signals[LAST_SIGNAL] = { 0 };
static GtkTableClass *parent_class;

static void gnome_paper_selector_class_init(GnomePaperSelectorClass *class);
static void gnome_paper_selector_init(GnomePaperSelector *self);
static void gnome_paper_selector_destroy(GtkObject *object);
static void gnome_paper_selector_finalize(GObject *object);

GtkType
gnome_paper_selector_get_type(void)
{
  static GtkType ps_type = 0;

  if (!ps_type) {
    GtkTypeInfo ps_info = {
      "GnomePaperSelector",
      sizeof(GnomePaperSelector),
      sizeof(GnomePaperSelectorClass),
      (GtkClassInitFunc) gnome_paper_selector_class_init,
      (GtkObjectInitFunc) gnome_paper_selector_init,
      NULL,
      NULL,
      NULL
    };
    ps_type = gtk_type_unique(gtk_table_get_type(), &ps_info);
  }
  return ps_type;
}

static void
gnome_paper_selector_class_init(GnomePaperSelectorClass *class)
{
  GtkObjectClass *object_class;
  GObjectClass *gobject_class;
  
  object_class = (GtkObjectClass*) class;
  gobject_class = (GObjectClass*) class;
  parent_class = gtk_type_class(gtk_table_get_type());

  ps_signals[CHANGED] =
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   GTK_CLASS_TYPE(object_class),
		   GTK_SIGNAL_OFFSET(GnomePaperSelectorClass, changed),
		   gtk_signal_default_marshaller,
		   GTK_TYPE_NONE, 0);
  ps_signals[FITTOPAGE] =
    gtk_signal_new("fittopage",
		   GTK_RUN_FIRST,
		   GTK_CLASS_TYPE(object_class),
		   GTK_SIGNAL_OFFSET(GnomePaperSelectorClass, fittopage),
		   gtk_signal_default_marshaller,
		   GTK_TYPE_NONE, 0);

  object_class->destroy = gnome_paper_selector_destroy;
  gobject_class->finalize = gnome_paper_selector_finalize;
}

static void fittopage_pressed(GnomePaperSelector *self);
static void darea_size_allocate(GnomePaperSelector *self,GtkAllocation *alloc);
static gint darea_expose_event(GnomePaperSelector *self, GdkEventExpose *ev);
static void paper_size_change(GtkMenuItem *item, GnomePaperSelector *self);
static void orient_changed(GnomePaperSelector *self);
static void margin_changed(GnomePaperSelector *self);
static void scale_changed(GnomePaperSelector *self);

static void
gnome_paper_selector_init(GnomePaperSelector *self)
{
  GtkWidget *frame, *box, *table, *menu, *menuitem, *vbox, *wid;
  GdkPixmap *pix;
  GdkBitmap *mask;
  const GList *papers;

  self->_priv = g_new0(GnomePaperSelectorPrivate, 1);

  if (default_unit == NULL)
    default_unit = gnome_unit_with_name(DEFAULT_UNIT);
  if (point_unit == NULL)
    point_unit = gnome_unit_with_name("point");
  if (default_paper == NULL)
    default_paper = gnome_paper_with_name(DEFAULT_PAPER);

  gtk_table_resize(GTK_TABLE(self), 3, 2);
  gtk_table_set_row_spacings(GTK_TABLE(self), 5);
  gtk_table_set_col_spacings(GTK_TABLE(self), 5);

  /* paper size */
  frame = gtk_frame_new(_("Paper Size"));
  gtk_table_attach(GTK_TABLE(self), frame, 0,1, 0,1,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(frame);

  box = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);

  self->_priv->paper_size = gtk_option_menu_new();
  gtk_box_pack_start(GTK_BOX(box), self->_priv->paper_size, TRUE, FALSE, 0);

  menu = gtk_menu_new();
  for (papers = gnome_paper_name_list(); papers; papers = papers->next) {
    const GnomePaper *paper=gnome_paper_with_name((const gchar *)papers->data);

    menuitem = gtk_menu_item_new_with_label((const gchar *)papers->data);
    gtk_object_set_user_data(GTK_OBJECT(menuitem), (gpointer) paper);
    gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
		       GTK_SIGNAL_FUNC(paper_size_change), self);
    gtk_container_add(GTK_CONTAINER(menu), menuitem);
    gtk_widget_show(menuitem);
  }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(self->_priv->paper_size), menu);
  gtk_widget_show(self->_priv->paper_size);

  self->_priv->paper_label = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(box), self->_priv->paper_label, TRUE, TRUE, 0);
  gtk_widget_show(self->_priv->paper_label);

  /* orientation */
  frame = gtk_frame_new(_("Orientation"));
  gtk_table_attach(GTK_TABLE(self), frame, 1,2, 0,1,
		   GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
  gtk_widget_show(frame);

  box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);

  self->_priv->orient_portrait = gtk_radio_button_new(NULL);
  vbox = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_add(GTK_CONTAINER(self->_priv->orient_portrait), vbox);
  gtk_widget_show(vbox);
  pix = gdk_pixmap_colormap_create_from_xpm_d(NULL,
		gtk_widget_get_colormap(GTK_WIDGET(self)), &mask, NULL,
		portrait_xpm);
  wid = gtk_pixmap_new(pix, mask);
  gdk_pixmap_unref(pix);
  gdk_bitmap_unref(mask);
  gtk_box_pack_start(GTK_BOX(vbox), wid, TRUE, TRUE, 0);
  gtk_widget_show(wid);
  wid = gtk_label_new(_("Portrait"));
  gtk_box_pack_start(GTK_BOX(vbox), wid, TRUE, TRUE, 0);
  gtk_widget_show(wid);

  gtk_box_pack_start(GTK_BOX(box), self->_priv->orient_portrait, TRUE, TRUE, 0);
  gtk_widget_show(self->_priv->orient_portrait);

  self->_priv->orient_landscape = gtk_radio_button_new(
	gtk_radio_button_group(GTK_RADIO_BUTTON(self->_priv->orient_portrait)));
  vbox = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_add(GTK_CONTAINER(self->_priv->orient_landscape), vbox);
  gtk_widget_show(vbox);
  pix = gdk_pixmap_colormap_create_from_xpm_d(NULL,
		gtk_widget_get_colormap(GTK_WIDGET(self)), &mask, NULL,
		landscape_xpm);
  wid = gtk_pixmap_new(pix, mask);
  gdk_pixmap_unref(pix);
  gdk_bitmap_unref(mask);
  gtk_box_pack_start(GTK_BOX(vbox), wid, TRUE, TRUE, 0);
  gtk_widget_show(wid);
  wid = gtk_label_new(_("Landscape"));
  gtk_box_pack_start(GTK_BOX(vbox), wid, TRUE, TRUE, 0);
  gtk_widget_show(wid);

  gtk_box_pack_start(GTK_BOX(box), self->_priv->orient_landscape, TRUE, TRUE, 0);
  gtk_widget_show(self->_priv->orient_landscape);

  /* margins */
  frame = gtk_frame_new(_("Margins"));
  gtk_table_attach(GTK_TABLE(self), frame, 0,1, 1,2,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(4, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);

  wid = gtk_label_new(_("Top:"));
  gtk_misc_set_alignment(GTK_MISC(wid), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), wid, 0,1, 0,1,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);

  self->_priv->tmargin = gnome_unit_spinner_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(1, 0,100, 0.1,10,10)),
	2, default_unit);
  gtk_table_attach(GTK_TABLE(table), self->_priv->tmargin, 1,2, 0,1,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->_priv->tmargin);

  wid = gtk_label_new(_("Bottom:"));
  gtk_misc_set_alignment(GTK_MISC(wid), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), wid, 0,1, 1,2,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);

  self->_priv->bmargin = gnome_unit_spinner_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(1, 0,100, 0.1,10,10)),
	2, default_unit);
  gtk_table_attach(GTK_TABLE(table), self->_priv->bmargin, 1,2, 1,2,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->_priv->bmargin);

  wid = gtk_label_new(_("Left:"));
  gtk_misc_set_alignment(GTK_MISC(wid), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), wid, 0,1, 2,3,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);

  self->_priv->lmargin = gnome_unit_spinner_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(1, 0,100, 0.1,10,10)),
	2, default_unit);
  gtk_table_attach(GTK_TABLE(table), self->_priv->lmargin, 1,2, 2,3,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->_priv->lmargin);

  wid = gtk_label_new(_("Right:"));
  gtk_misc_set_alignment(GTK_MISC(wid), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), wid, 0,1, 3,4,
		   GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(wid);

  self->_priv->rmargin = gnome_unit_spinner_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(1, 0,100, 0.1,10,10)),
	2, default_unit);
  gtk_table_attach(GTK_TABLE(table), self->_priv->rmargin, 1,2, 3,4,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->_priv->rmargin);

  /* Scaling */
  frame = gtk_frame_new(_("Scaling"));
  gtk_table_attach(GTK_TABLE(self), frame, 0,1, 2,3,
		   GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show(frame);

  box = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add(GTK_CONTAINER(frame), box);
  gtk_widget_show(box);

  self->_priv->scaling = gtk_spin_button_new(
	GTK_ADJUSTMENT(gtk_adjustment_new(100,1,10000, 1,10,10)), 1, 1);
  gtk_box_pack_start(GTK_BOX(box), self->_priv->scaling, TRUE, FALSE, 0);
  gtk_widget_show(self->_priv->scaling);

  self->_priv->fittopage = gtk_button_new_with_label(_("Fit to Page"));
  gtk_box_pack_start(GTK_BOX(box), self->_priv->fittopage, TRUE, FALSE, 0);
  gtk_widget_show(self->_priv->fittopage);

  /* the drawing area */
  self->_priv->darea = gtk_drawing_area_new();
  gtk_table_attach(GTK_TABLE(self), self->_priv->darea, 1,2, 1,3,
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
  gtk_widget_show(self->_priv->darea);

  /* connect the signal handlers */
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->orient_portrait), "toggled",
			    GTK_SIGNAL_FUNC(orient_changed), GTK_OBJECT(self));

  gtk_signal_connect_object(GTK_OBJECT(self->_priv->tmargin), "changed",
			    GTK_SIGNAL_FUNC(margin_changed), GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->bmargin), "changed",
			    GTK_SIGNAL_FUNC(margin_changed), GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->lmargin), "changed",
			    GTK_SIGNAL_FUNC(margin_changed), GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->rmargin), "changed",
			    GTK_SIGNAL_FUNC(margin_changed), GTK_OBJECT(self));

  gtk_signal_connect_object(GTK_OBJECT(self->_priv->scaling), "changed",
			    GTK_SIGNAL_FUNC(scale_changed), GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->fittopage), "pressed",
			    GTK_SIGNAL_FUNC(fittopage_pressed),
			    GTK_OBJECT(self));

  gtk_signal_connect_object(GTK_OBJECT(self->_priv->darea), "size_allocate",
			    GTK_SIGNAL_FUNC(darea_size_allocate),
			    GTK_OBJECT(self));
  gtk_signal_connect_object(GTK_OBJECT(self->_priv->darea), "expose_event",
			    GTK_SIGNAL_FUNC(darea_expose_event),
			    GTK_OBJECT(self));

  self->_priv->unit = default_unit;
  self->_priv->paper = default_paper;

  self->_priv->gc = NULL;
  self->_priv->block_changed = FALSE;
}

/**
 * gnome_paper_selector_new:
 * 
 * createa GnomePaperSelector, using centimeters as the display unit for
 * the spin buttons.
 * 
 * Return value: the GnomePaperSelector
 **/
GtkWidget *
gnome_paper_selector_new(void)
{
  return gnome_paper_selector_new_with_unit(default_unit);
}

/**
 * gnome_paper_selector_new_with_unit:
 * @unit: the display units for the spin buttons
 * 
 * creates a new GnomePaperSelector
 * 
 * Return value: the GnomePaperSelector
 **/
GtkWidget *
gnome_paper_selector_new_with_unit(const GnomeUnit *unit)
{
  GnomePaperSelector *self = gtk_type_new(gnome_paper_selector_get_type());

  if (unit) {
    self->_priv->unit = unit;
    gnome_unit_spinner_change_units(GNOME_UNIT_SPINNER(self->_priv->tmargin), unit);
    gnome_unit_spinner_change_units(GNOME_UNIT_SPINNER(self->_priv->bmargin), unit);
    gnome_unit_spinner_change_units(GNOME_UNIT_SPINNER(self->_priv->lmargin), unit);
    gnome_unit_spinner_change_units(GNOME_UNIT_SPINNER(self->_priv->rmargin), unit);
  }
  gnome_paper_selector_set_paper(self, DEFAULT_PAPER);
  return GTK_WIDGET(self);
}

/**
 * gnome_paper_selector_get_paper:
 * @self: the GnomePaperSelector
 * 
 * get the currently selected paper.
 * 
 * Return value: the GnomePaper representing the currently selected paper.
 **/
const GnomePaper *
gnome_paper_selector_get_paper(GnomePaperSelector *self)
{
  g_return_val_if_fail(self != NULL, NULL);

  return self->_priv->paper;
}

/**
 * gnome_paper_selector_set_paper:
 * @self: the GnomePaperSelector
 * @paper: the name of the paper.
 * 
 * change the currently selected paper.  This will also reset the margins
 * for the new paper.
 **/
void
gnome_paper_selector_set_paper(GnomePaperSelector *self, const gchar *paper)
{
  const GList *l;
  gint i;
  const GnomePaper *p = NULL;

  g_return_if_fail(self != NULL);

  if (paper)
    p = gnome_paper_with_name(paper);
  if (!p)
    p = default_paper;
  paper = gnome_paper_name(p);

  for (l = gnome_paper_name_list(), i = 0; l; l = l->next, i++)
    if (!strcmp((const gchar *)l->data, paper))
      break;
  gtk_option_menu_set_history(GTK_OPTION_MENU(self->_priv->paper_size), i);
  gtk_menu_item_activate(
	GTK_MENU_ITEM(GTK_OPTION_MENU(self->_priv->paper_size)->menu_item));
}

/**
 * gnome_paper_selector_get_margins:
 * @self: the GnomePaperSelector
 * @unit: the units to return the margin widths in.
 * @tmargin: a variable for the top margin, or NULL
 * @bmargin: a variable for the bottom margin, or NULL
 * @lmargin: a variable for the left margin, or NULL
 * @rmargin: a variable for the right margin or NULL
 * 
 * Get the currently set margins for the paper.  If the unit argument
 * is NULL, then the paper selector's default unit is used.
 **/
void
gnome_paper_selector_get_margins(GnomePaperSelector *self,
				 const GnomeUnit *unit,
				 gfloat *tmargin, gfloat *bmargin,
				 gfloat *lmargin, gfloat *rmargin)
{
  g_return_if_fail(self != NULL);

  if (!unit)
    unit = self->_priv->unit;

  if (tmargin)
    *tmargin = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->tmargin),
					    unit);
  if (bmargin)
    *bmargin = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->bmargin),
					    unit);
  if (lmargin)
    *lmargin = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->lmargin),
					    unit);
  if (rmargin)
    *rmargin = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->rmargin),
					    unit);
}

/**
 * gnome_paper_selector_set_margins:
 * @self: the GnomePaperSelector
 * @unit: the unit the margin widths are in terms of, or NULL
 * @tmargin: the new top margin
 * @bmargin: the new bottom margin
 * @lmargin: the new left margin
 * @rmargin: the new right margin
 * 
 * Set the margins for the paper selector.  If the unit argument is NULL,
 * the paper selector's default unit is used.
 **/
void
gnome_paper_selector_set_margins(GnomePaperSelector *self,
				 const GnomeUnit *unit,
				 gfloat tmargin, gfloat bmargin,
				 gfloat lmargin, gfloat rmargin)
{
  g_return_if_fail(self != NULL);

  if ( ! unit)
	  unit = self->_priv->unit;

  self->_priv->block_changed = TRUE;
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->tmargin), tmargin,
			       unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->bmargin), bmargin,
			       unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->lmargin), lmargin,
			       unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->rmargin), rmargin,
			       unit);
  self->_priv->block_changed = FALSE;

  gtk_signal_emit(GTK_OBJECT(self), ps_signals[CHANGED]);
}

/**
 * gnome_paper_selector_get_orientation:
 * @self: the GnomePaperSelector
 * 
 * get the currently set orientation of the paper.
 * 
 * Return value: the paper orientation.
 **/
GnomePaperOrient
gnome_paper_selector_get_orientation(GnomePaperSelector *self)
{
  g_return_val_if_fail(self != NULL, GNOME_PAPER_ORIENT_PORTRAIT);

  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active)
    return GNOME_PAPER_ORIENT_PORTRAIT;
  else
    return GNOME_PAPER_ORIENT_LANDSCAPE;
}

/**
 * gnome_paper_selector_set_orientation:
 * @self: the GnomePaperSelector
 * @orient: the new paper orientation.
 * 
 * Set the paper orientation for the paper selector.
 **/
void
gnome_paper_selector_set_orientation(GnomePaperSelector *self,
				     GnomePaperOrient orient)
{
  g_return_if_fail(self != NULL);

  switch (orient) {
  case GNOME_PAPER_ORIENT_PORTRAIT:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(self->_priv->orient_portrait),
				 TRUE);
    break;
  case GNOME_PAPER_ORIENT_LANDSCAPE:
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(self->_priv->orient_landscape),
				 TRUE);
    break;
  }
}

/**
 * gnome_paper_selector_get_scaling:
 * @self: the GnomePaperSelector
 * 
 * Get the scaling factor for the paper selector
 * 
 * Return value: the scaling factor.
 **/
gfloat
gnome_paper_selector_get_scaling(GnomePaperSelector *self)
{
  g_return_val_if_fail(self != NULL, 0.0);

  return GTK_SPIN_BUTTON(self->_priv->scaling)->adjustment->value / 100.0;
}

/**
 * gnome_paper_selector_set_scaling:
 * @self: the GnomePaperSelector
 * @scaling: the new scaling factor
 * 
 * Set the scaling factor.
 **/
void
gnome_paper_selector_set_scaling(GnomePaperSelector *self, gfloat scaling)
{
  g_return_if_fail(self != NULL);

  GTK_SPIN_BUTTON(self->_priv->scaling)->adjustment->value = scaling * 100.0;
  gtk_adjustment_value_changed(GTK_SPIN_BUTTON(self->_priv->scaling)->adjustment);
}

/**
 * gnome_paper_selector_get_effective_area:
 * @self: the GnomePaperSelector
 * @unit: the unit the width/height should be in, or NULL
 * @width: a variable to return the width in
 * @height: a variable to return the height in 
 * 
 * Returns the effective area of the paper.  This takes the paper size,
 * margin widths and scaling factor into account.
 **/
void
gnome_paper_selector_get_effective_area(GnomePaperSelector *self,
					const GnomeUnit *unit,
					gfloat *width, gfloat *height)
{
  gfloat h, w, scaling;

  g_return_if_fail(self != NULL);

  if ( ! unit)
	  unit = self->_priv->unit;

  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active) {
    w = gnome_paper_pswidth(self->_priv->paper);
    h = gnome_paper_psheight(self->_priv->paper);
  } else {
    h = gnome_paper_pswidth(self->_priv->paper);
    w = gnome_paper_psheight(self->_priv->paper);
  }
  w = gnome_unit_convert(w, point_unit, unit);
  h = gnome_unit_convert(h, point_unit, unit);

  h -= gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->tmargin), unit);
  h -= gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->bmargin), unit);
  w -= gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->lmargin), unit);
  w -= gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->rmargin), unit);
  scaling = GTK_SPIN_BUTTON(self->_priv->scaling)->adjustment->value / 100.0;
  h /= scaling;
  w /= scaling;

  if (width)  *width = w;
  if (height) *height = h;
}

static void
fittopage_pressed(GnomePaperSelector *self)
{
  gtk_signal_emit(GTK_OBJECT(self), ps_signals[FITTOPAGE]);
}

static void size_page(GnomePaperSelector *self, GtkAllocation *a)
{
  self->_priv->width = a->width - 3;
  self->_priv->height = a->height - 3;

  /* change to correct metrics */
  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active) {
    if (self->_priv->width * gnome_paper_psheight(self->_priv->paper) >
	self->_priv->height * gnome_paper_pswidth(self->_priv->paper))
      self->_priv->width = self->_priv->height * gnome_paper_pswidth(self->_priv->paper) /
	gnome_paper_psheight(self->_priv->paper);
    else
      self->_priv->height = self->_priv->width * gnome_paper_psheight(self->_priv->paper) /
	gnome_paper_pswidth(self->_priv->paper);
  } else {
    if (self->_priv->width * gnome_paper_pswidth(self->_priv->paper) >
	self->_priv->height * gnome_paper_psheight(self->_priv->paper))
      self->_priv->width = self->_priv->height * gnome_paper_psheight(self->_priv->paper) /
	gnome_paper_pswidth(self->_priv->paper);
    else
      self->_priv->height = self->_priv->width * gnome_paper_pswidth(self->_priv->paper) /
	gnome_paper_psheight(self->_priv->paper);
  }

  self->_priv->x = (a->width - self->_priv->width - 3) / 2;
  self->_priv->y = (a->height - self->_priv->height - 3) / 2;
}

static void
darea_size_allocate(GnomePaperSelector *self, GtkAllocation *allocation)
{
  size_page(self, allocation);
}

static gint
darea_expose_event(GnomePaperSelector *self, GdkEventExpose *event)
{
  GdkWindow *window= self->_priv->darea->window;
  gfloat val;
  gint num;
  GdkGC *black_gc = gtk_widget_get_style(GTK_WIDGET(self))->black_gc;
  GdkGC *white_gc = gtk_widget_get_style(GTK_WIDGET(self))->white_gc;

  if (!window)
    return FALSE;

  /* setup gc ... */
  if (!self->_priv->gc) {
    GdkColor blue;

    self->_priv->gc = gdk_gc_new(window);
    blue.red = 0;
    blue.green = 0;
    blue.blue = 0x7fff;
    gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(self)), &blue);
    gdk_gc_set_foreground(self->_priv->gc, &blue);
  }

  gdk_window_clear_area (window,
                         0, 0,
                         self->_priv->darea->allocation.width,
                         self->_priv->darea->allocation.height);

  /* draw the page image */
  gdk_draw_rectangle(window, black_gc, TRUE, self->_priv->x+3, self->_priv->y+3,
		     self->_priv->width, self->_priv->height);
  gdk_draw_rectangle(window, white_gc, TRUE, self->_priv->x, self->_priv->y,
		     self->_priv->width, self->_priv->height);
  gdk_draw_rectangle(window, black_gc, FALSE, self->_priv->x, self->_priv->y,
		     self->_priv->width-1, self->_priv->height-1);

  /* draw margins */
  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active) {
    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->tmargin),
				       point_unit);
    num = self->_priv->y + val * self->_priv->height /gnome_paper_psheight(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, self->_priv->x+1, num, self->_priv->x+self->_priv->width-2,num);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->bmargin),
				       point_unit);
    num = self->_priv->y + self->_priv->height -
      val * self->_priv->height / gnome_paper_psheight(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, self->_priv->x+1, num, self->_priv->x+self->_priv->width-2,num);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->lmargin),
				       point_unit);
    num = self->_priv->x + val * self->_priv->width / gnome_paper_pswidth(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, num, self->_priv->y+1,num,self->_priv->y+self->_priv->height-2);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->rmargin),
				       point_unit);
    num = self->_priv->x + self->_priv->width -
      val * self->_priv->width / gnome_paper_pswidth(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, num, self->_priv->y+1,num,self->_priv->y+self->_priv->height-2);
  } else {
    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->tmargin),
				       point_unit);
    num = self->_priv->y + val * self->_priv->height /gnome_paper_pswidth(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, self->_priv->x+1, num, self->_priv->x+self->_priv->width-2,num);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->bmargin),
				       point_unit);
    num = self->_priv->y + self->_priv->height -
      val * self->_priv->height / gnome_paper_pswidth(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, self->_priv->x+1, num, self->_priv->x+self->_priv->width-2,num);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->lmargin),
				       point_unit);
    num = self->_priv->x + val * self->_priv->width / gnome_paper_psheight(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, num, self->_priv->y+1,num,self->_priv->y+self->_priv->height-2);

    val = gnome_unit_spinner_get_value(GNOME_UNIT_SPINNER(self->_priv->rmargin),
				       point_unit);
    num = self->_priv->x + self->_priv->width -
      val * self->_priv->width / gnome_paper_psheight(self->_priv->paper);
    gdk_draw_line(window, self->_priv->gc, num, self->_priv->y+1,num,self->_priv->y+self->_priv->height-2);
  }

  return FALSE;
}

static void
paper_size_change(GtkMenuItem *item, GnomePaperSelector *self)
{
  gchar buf[512];

  self->_priv->paper = (const GnomePaper *)gtk_object_get_user_data(GTK_OBJECT(item));
  size_page(self, &self->_priv->darea->allocation);
  gtk_widget_queue_draw(self->_priv->darea);

  self->_priv->block_changed = TRUE;
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->tmargin),
			     gnome_paper_tmargin(self->_priv->paper), point_unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->bmargin),
			     gnome_paper_bmargin(self->_priv->paper), point_unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->lmargin),
			     gnome_paper_lmargin(self->_priv->paper), point_unit);
  gnome_unit_spinner_set_value(GNOME_UNIT_SPINNER(self->_priv->rmargin),
			     gnome_paper_rmargin(self->_priv->paper), point_unit);

  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active) {
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->tmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->bmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->lmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->rmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
  } else {
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->tmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->bmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->lmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->rmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
  }
  self->_priv->block_changed = FALSE;

  g_snprintf(buf, sizeof(buf), _("%0.3g%s x %0.3g%s"),
	     gnome_unit_convert(gnome_paper_pswidth(self->_priv->paper),
				point_unit, self->_priv->unit),
	     gnome_unit_abbrev(self->_priv->unit),
	     gnome_unit_convert(gnome_paper_psheight(self->_priv->paper),
				point_unit, self->_priv->unit),
	     gnome_unit_abbrev(self->_priv->unit));
  gtk_label_set(GTK_LABEL(self->_priv->paper_label), buf);

  gtk_signal_emit(GTK_OBJECT(self), ps_signals[CHANGED]);
}

static void
orient_changed(GnomePaperSelector *self)
{
  size_page(self, &self->_priv->darea->allocation);
  gtk_widget_queue_draw(self->_priv->darea);

  if (GTK_TOGGLE_BUTTON(self->_priv->orient_portrait)->active) {
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->tmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->bmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->lmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->rmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
  } else {
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->tmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->bmargin))->upper =
      gnome_paper_pswidth(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->lmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
    gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(self->_priv->rmargin))->upper =
      gnome_paper_psheight(self->_priv->paper);
  }

  if (!self->_priv->block_changed)
    gtk_signal_emit(GTK_OBJECT(self), ps_signals[CHANGED]);
}

static void
margin_changed(GnomePaperSelector *self)
{
  gtk_widget_queue_draw(self->_priv->darea);
  if (!self->_priv->block_changed)
    gtk_signal_emit(GTK_OBJECT(self), ps_signals[CHANGED]);
}

static void
scale_changed(GnomePaperSelector *self)
{
  if (!self->_priv->block_changed)
    gtk_signal_emit(GTK_OBJECT(self), ps_signals[CHANGED]);
}

static void
gnome_paper_selector_destroy(GtkObject *object)
{
  GnomePaperSelector *self;

  g_return_if_fail(object != NULL);

  /* remember, destroy can be run multiple times! */

  self = GNOME_PAPER_SELECTOR(object);
  if (self->_priv->gc)
    gdk_gc_unref(self->_priv->gc);
  self->_priv->gc = NULL;

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (* GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

static void
gnome_paper_selector_finalize(GObject *object)
{
  GnomePaperSelector *self;

  g_return_if_fail(object != NULL);

  self = GNOME_PAPER_SELECTOR(object);

  g_free(self->_priv);
  self->_priv = NULL;

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (* G_OBJECT_CLASS(parent_class)->finalize)(object);
}



#ifdef OLD_PAGE_SELECTOR_API
/**
 * gnome_paper_selector_get_width:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the paper width in points.  This function is provided for backward
 * compatibility.
 * 
 * Return value: the paper width in points.
 **/
gfloat
gnome_paper_selector_get_width(GnomePaperSelector *gspaper)
{
  g_return_val_if_fail(gspaper != NULL, 0);

  return gnome_paper_pswidth(gnome_paper_selector_get_paper(gspaper));
}
/**
 * gnome_paper_selector_get_height:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the paper height in points.  This function is provided for backward
 * compatibility.
 * 
 * Return value: the paper height in points.
 **/
gfloat
gnome_paper_selector_get_height(GnomePaperSelector *gspaper);
{
  g_return_val_if_fail(gspaper != NULL, 0);

  return gnome_paper_psheight(gnome_paper_selector_get_paper(gspaper));
}
/**
 * gnome_paper_selector_get_left_margin:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the left margin width in points.  This function is provided for
 * backward compatibility.
 * 
 * Return value: the left margin width in points.
 **/
gfloat
gnome_paper_selector_get_left_margin(GnomePaperSelector *gspaper);
{
  gfloat margin;

  g_return_val_if_fail(gspaper != NULL, 0);

  gnome_paper_selector_get_margins(gspaper, point_unit,
				   NULL, NULL, &margin, NULL);
  return margin;
}
/**
 * gnome_paper_selector_get_right_margin:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the right margin width in points.  This function is provided for
 * backward compatibility.
 * 
 * Return value: the right margin width in points.
 **/
gfloat
gnome_paper_selector_get_right_margin(GnomePaperSelector *gspaper);
{
  gfloat margin;

  g_return_val_if_fail(gspaper != NULL, 0);

  gnome_paper_selector_get_margins(gspaper, point_unit,
				   NULL, NULL, NULL, &margin);
  return margin;
}
/**
 * gnome_paper_selector_get_top_margin:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the top margin width in points.  This function is provided for
 * backward compatibility.
 * 
 * Return value: the top margin width in points.
 **/
gfloat
gnome_paper_selector_get_top_margin(GnomePaperSelector *gspaper);
{
  gfloat margin;

  g_return_val_if_fail(gspaper != NULL, 0);

  gnome_paper_selector_get_margins(gspaper, point_unit,
				   &margin, NULL, NULL, NULL);
  return margin;
}
/**
 * gnome_paper_selector_get_bottom_margin:
 * @gspaper: the GnomePaperSelector
 * 
 * Get the bottom margin width in points.  This function is provided for
 * backward compatibility.
 * 
 * Return value: the bottom margin width in points.
 **/
gfloat
gnome_paper_selector_get_bottom_margin(GnomePaperSelector *gspaper);
{
  gfloat margin;

  g_return_val_if_fail(gspaper != NULL, 0);

  gnome_paper_selector_get_margins(gspaper, point_unit,
				   NULL, &margin, NULL, NULL);
  return margin;
}
#endif


#ifdef PAGELAYOUT_TEST

void
changed_signal(GnomePaperSelector *self)
{
  g_message("changed");
}
void
fittopage_signal(GnomePaperSelector *self)
{
  g_message("fit to page");
}

void
main(int argc, char **argv)
{
  GtkWidget *win, *ps;

  gtk_init(&argc, &argv);

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(win), _("Page Setup"));
  gtk_signal_connect(GTK_OBJECT(win), "destroy",
		     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

  ps = gnome_paper_selector_new();
  gtk_container_set_border_width(GTK_CONTAINER(ps), 5);
  gtk_container_add(GTK_CONTAINER(win), ps);
  gtk_widget_show(ps);

  gtk_signal_connect(GTK_OBJECT(ps), "changed",
		     GTK_SIGNAL_FUNC(changed_signal), NULL);
  gtk_signal_connect(GTK_OBJECT(ps), "fittopage",
		     GTK_SIGNAL_FUNC(fittopage_signal), NULL);

  gtk_widget_show(win);
  gtk_main();
}

#endif

#endif /* The whole thing */

