/* gnome-db-basic-form.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-basic-form.h"
#include "gnome-db-tools.h"
#include "marshal.h"
#include "gnome-db-data-entry.h"
#include <libgnomedb/data-entries/gnome-db-entry-combo.h>

static void gnome_db_basic_form_class_init (GnomeDbBasicFormClass * class);
static void gnome_db_basic_form_init (GnomeDbBasicForm *wid);
static void gnome_db_basic_form_dispose (GObject *object);

static void gnome_db_basic_form_set_property (GObject *object,
					      guint param_id,
					      const GValue *value,
					      GParamSpec *pspec);
static void gnome_db_basic_form_get_property (GObject *object,
					      guint param_id,
					      GValue *value,
					      GParamSpec *pspec);

static void layout_spec_free (GnomeDbFormLayoutSpec *spec);

static void gnome_db_basic_form_fill (GnomeDbBasicForm *form);
static void gnome_db_basic_form_clean (GnomeDbBasicForm *form);

static void paramlist_destroyed_cb (GdaParameterList *paramlist, GnomeDbBasicForm *form);
static void paramlist_public_data_changed_cb (GdaParameterList *paramlist, GnomeDbBasicForm *form);

static void entry_contents_modified (GnomeDbDataEntry *entry, GnomeDbBasicForm *form);
static void parameter_changed_cb (GdaParameter *param, GnomeDbDataEntry *entry);

static void mark_not_null_entry_labels (GnomeDbBasicForm *form, gboolean show_mark);
enum
{
	PARAM_CHANGED,
	LAST_SIGNAL
};

/* properties */
enum
{
        PROP_0,
	PROP_LAYOUT_SPEC,
	PROP_PARAMLIST,
	PROP_HEADERS_SENSITIVE
};

struct _GnomeDbBasicFormPriv
{
	GdaParameterList       *paramlist;/* parameters */
	gulong                 *signal_ids; /* array of signal ids */

	GSList                 *entries;/* list of GnomeDbDataEntry widgets */
	GSList                 *not_null_labels;/* list of GtkLabel widgets corresponding to NOT NULL entries */

	GnomeDbFormLayoutSpec  *layout_spec;
	GtkWidget              *entries_table;
	GtkWidget              *entries_glade;
	GSList                 *hidden_entries;

	gboolean                headers_sensitive;
	gboolean                forward_param_updates; /* forward them to the GnomeDbDataEntry widgets ? */
	GtkTooltips            *tooltips;
};


static gint gnome_db_basic_form_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

GType
gnome_db_basic_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBasicFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_basic_form_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBasicForm),
			0,
			(GInstanceInitFunc) gnome_db_basic_form_init
		};		
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbBasicForm", &info, 0);
	}

	return type;
}

static void
gnome_db_basic_form_class_init (GnomeDbBasicFormClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	/* signals */
	gnome_db_basic_form_signals[PARAM_CHANGED] =
		g_signal_new ("param_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbBasicFormClass, param_changed),
			      NULL, NULL,
			      marshal_VOID__OBJECT_BOOLEAN, G_TYPE_NONE, 2,
			      G_TYPE_OBJECT, G_TYPE_BOOLEAN);

	class->param_changed = NULL;
	object_class->dispose = gnome_db_basic_form_dispose;

	/* Properties */
        object_class->set_property = gnome_db_basic_form_set_property;
        object_class->get_property = gnome_db_basic_form_get_property;
	g_object_class_install_property (object_class, PROP_LAYOUT_SPEC,
					 g_param_spec_pointer ("layout_spec", 
							       _("Pointer to a GnomeDbFormLayoutSpec structure"), NULL,
							       G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_PARAMLIST,
					 g_param_spec_pointer ("paramlist", 
							       _("List of parameters to show in the form"), NULL,
                                                               G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_HEADERS_SENSITIVE,
					 g_param_spec_boolean ("headers_sensitive",
							       _("Defines if entry headers are sensitive with entries"), 
							       NULL, FALSE,
							       G_PARAM_READABLE | G_PARAM_WRITABLE));
}

static void
gnome_db_basic_form_init (GnomeDbBasicForm * wid)
{
	wid->priv = g_new0 (GnomeDbBasicFormPriv, 1);
	wid->priv->paramlist = NULL;
	wid->priv->entries = NULL;
	wid->priv->not_null_labels = NULL;
	wid->priv->layout_spec = NULL;
	wid->priv->entries_glade = NULL;
	wid->priv->entries_table = NULL;
	wid->priv->hidden_entries = NULL;
	wid->priv->signal_ids = NULL;

	wid->priv->headers_sensitive = FALSE;

	wid->priv->forward_param_updates = TRUE;
	wid->priv->tooltips = gtk_tooltips_new ();
}

static void widget_shown_cb (GtkWidget *wid, GnomeDbBasicForm *form);

/**
 * gnome_db_basic_form_new
 * @paramlist: a #GdaParameterList structure
 *
 * Creates a new #GnomeDbBasicForm widget using all the parameters provided in @paramlist.
 *
 * The global layout is rendered using a table (a #GtkTable), and an entry is created for each
 * node of @paramlist.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_basic_form_new (GdaParameterList *paramlist)
{
	GObject *obj;

	obj = g_object_new (GNOME_DB_TYPE_BASIC_FORM, "paramlist", paramlist, NULL);

	return (GtkWidget *) obj;
}

/**
 * gnome_db_basic_form_new_custom
 * @paramlist: a #GdaParameterList structure
 * @glade_file: a Glade XML file name
 * @root_element: the name of the top-most widget in @glade_file to use in the new form
 * @form_prefix: the prefix used to look for widgets to add entries in
 *
 * Creates a new #GnomeDbBasicForm widget using all the parameters provided in @paramlist.
 *
 * The layout is specified in the @glade_file specification, and an entry is created for each
 * node of @paramlist.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_basic_form_new_custom (GdaParameterList *paramlist, const gchar *glade_file, 
				const gchar *root_element, const gchar *form_prefix)
{
	GnomeDbFormLayoutSpec spec;
	GObject *obj;

	spec.xml_object = NULL;
	spec.xml_file = (gchar *) glade_file;
	spec.root_element = (gchar *) root_element;
	spec.form_prefix = (gchar *) form_prefix;
	obj = g_object_new (GNOME_DB_TYPE_BASIC_FORM, "layout_spec", &spec, "paramlist", paramlist, NULL);

	return (GtkWidget *) obj;	
}


static void
widget_shown_cb (GtkWidget *wid, GnomeDbBasicForm *form)
{
	if (g_slist_find (form->priv->hidden_entries, wid)) {
		if (form->priv->entries_table && g_slist_find (form->priv->entries, wid)) {
			gint row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (wid), "row_no"));
			gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 0);
		}
		
		gtk_widget_hide (wid);
	}
}

static void
paramlist_destroyed_cb (GdaParameterList *paramlist, GnomeDbBasicForm *form)
{
	GSList *list;
	gint i = 0;

	g_assert (paramlist == form->priv->paramlist);

	/* disconnect from parameters */
	list = form->priv->paramlist->parameters;;
	while (list) {
		g_signal_handler_disconnect (G_OBJECT (list->data), form->priv->signal_ids[i]);
		
		list = g_slist_next (list);
		i++;
	}
	g_free (form->priv->signal_ids);
	form->priv->signal_ids = NULL;

	/* unref the paramlist */
	g_signal_handlers_disconnect_by_func (paramlist,
					      G_CALLBACK (paramlist_destroyed_cb), form);
	g_signal_handlers_disconnect_by_func (paramlist,
					      G_CALLBACK (paramlist_public_data_changed_cb), form);
	g_object_unref (form->priv->paramlist);
	form->priv->paramlist = NULL;

	/* render all the entries non sensitive */
	list = form->priv->entries;
	while (list) {
		gtk_widget_set_sensitive (GTK_WIDGET (list->data), FALSE);
		list = g_slist_next (list);
	}
}

static void
paramlist_public_data_changed_cb (GdaParameterList *paramlist, GnomeDbBasicForm *form)
{
	/* here we want to re-define all the data entry widgets */
	gnome_db_basic_form_clean (form);
	gnome_db_basic_form_fill (form);
}

static void
gnome_db_basic_form_dispose (GObject *object)
{
	GnomeDbBasicForm *form;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_BASIC_FORM (object));
	form = GNOME_DB_BASIC_FORM (object);

	if (form->priv) {
		/* paramlist */
		if (form->priv->paramlist) 
			paramlist_destroyed_cb (form->priv->paramlist, form);

		gnome_db_basic_form_clean (form);

		/* the private area itself */
		g_free (form->priv);
		form->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_basic_form_set_property (GObject *object,
				  guint param_id,
				  const GValue *value,
				  GParamSpec *pspec)
{
	GnomeDbBasicForm *form;
	GnomeDbFormLayoutSpec *lspec, *new_spec = NULL;

        form = GNOME_DB_BASIC_FORM (object);
        if (form->priv) {
                switch (param_id) {
		case PROP_LAYOUT_SPEC:
			lspec = g_value_get_pointer (value);
			if (lspec) {
				g_return_if_fail (lspec->xml_file || lspec->xml_object);
				g_return_if_fail (lspec->root_element);
				
				/* spec copy */
				new_spec = g_new0 (GnomeDbFormLayoutSpec, 1);
				if (lspec->xml_file)
					new_spec->xml_file = g_strdup (lspec->xml_file);
				if (lspec->xml_object) {
					new_spec->xml_object = lspec->xml_object;
					g_object_ref (new_spec->xml_object);
				}
				if (lspec->root_element)
					new_spec->root_element = g_strdup (lspec->root_element);
				if (lspec->form_prefix)
					new_spec->form_prefix = g_strdup (lspec->form_prefix);
				
				/* spec verify */
				if (!new_spec->xml_object) {
					new_spec->xml_object = glade_xml_new (new_spec->xml_file, new_spec->root_element, NULL);
					if (! new_spec->xml_object) {
						layout_spec_free (new_spec);
						g_warning (_("Could not load file '%s'"), new_spec->xml_file);
						return;
					}
				}
			}

			gnome_db_basic_form_clean (form);
			if (new_spec) {
				form->priv->layout_spec = new_spec;
				g_print ("Loaded Glade file, reinit interface\n");
			}
			gnome_db_basic_form_fill (form);
			break;
		case PROP_PARAMLIST:
			if (form->priv->paramlist) {
				TO_IMPLEMENT;
				g_assert_not_reached ();
			}

			form->priv->paramlist = g_value_get_pointer (value);
			if (form->priv->paramlist) {
				GError *error = NULL;
				g_return_if_fail (GDA_IS_PARAMETER_LIST (form->priv->paramlist));
				
				if (! gda_parameter_list_is_coherent (form->priv->paramlist, &error)) {
					g_warning ("gda_parameter_list_is_coherent() returned FALSE: %s", 
						   error->message);
					form->priv->paramlist = NULL;
					g_error_free (error);
					return;
				}
				
				g_object_ref (form->priv->paramlist);
				gda_object_connect_destroy (form->priv->paramlist,
							    G_CALLBACK (paramlist_destroyed_cb), form);
				g_signal_connect (form->priv->paramlist, "public_data_changed",
						  G_CALLBACK (paramlist_public_data_changed_cb), form);
				
				gnome_db_basic_form_fill (form);
			}
			break;
		case PROP_HEADERS_SENSITIVE:
			form->priv->headers_sensitive = g_value_get_boolean (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
		}
	}
}

static void
gnome_db_basic_form_get_property (GObject *object,
				  guint param_id,
				  GValue *value,
				  GParamSpec *pspec)
{
	GnomeDbBasicForm *form;

        form = GNOME_DB_BASIC_FORM (object);
        if (form->priv) {
                switch (param_id) {
		case PROP_PARAMLIST:
			g_value_set_pointer (value, form->priv->paramlist);
			break;
		case PROP_HEADERS_SENSITIVE:
			g_value_set_boolean (value, form->priv->headers_sensitive);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}

static void
gnome_db_basic_form_clean (GnomeDbBasicForm *form)
{
	GSList *list;
	gint i = 0;

	if (form->priv->paramlist) {
		list = form->priv->paramlist->parameters;
		while (list) {
			g_signal_handler_disconnect (G_OBJECT (list->data), form->priv->signal_ids[i]);
			
			list = g_slist_next (list);
			i++;
		}
		g_free (form->priv->signal_ids);
		form->priv->signal_ids = NULL;
	}

	/* destroy all the widgets */
	if (form->priv->entries_table) {
		gtk_widget_destroy (form->priv->entries_table);
		form->priv->entries_table = NULL;
	}
	if (form->priv->entries_glade) {
		gtk_widget_destroy (form->priv->entries_glade);
		form->priv->entries_glade = NULL;
	}
	if (form->priv->layout_spec) {
		layout_spec_free (form->priv->layout_spec);
		form->priv->layout_spec = NULL;
	}
	g_slist_free (form->priv->entries);
	form->priv->entries = NULL;

	g_slist_free (form->priv->not_null_labels);
	form->priv->not_null_labels = NULL;

	g_slist_free (form->priv->hidden_entries);
	form->priv->hidden_entries = NULL;
}

static void
layout_spec_free (GnomeDbFormLayoutSpec *spec)
{
	if (spec->xml_object)
		g_object_unref (spec->xml_object);
	g_free (spec->xml_file);
	g_free (spec->root_element);
	g_free (spec->form_prefix);
	g_free (spec);
}

static void entry_destroyed_cb (GtkWidget *entry, GnomeDbBasicForm *form);

/*
 * create the entries in the widget
 */
static void 
gnome_db_basic_form_fill (GnomeDbBasicForm *form)
{
	GSList *list;
	gint i;
	
	/* parameters list management */
	if (!form->priv->paramlist || !form->priv->paramlist->groups_list)
		/* nothing to do */
		return;

	/* allocating space for the signal ids and connect to the parameter's changes */
	form->priv->signal_ids = g_new0 (gulong, g_slist_length (form->priv->paramlist->parameters));
	i = 0;

	/* creating all the entries, and putting them into the form->priv->entries list */
	list = form->priv->paramlist->groups_list;
	while (list) {
		GdaParameterListGroup *group;
		GtkWidget *entry = NULL;

		group = GDA_PARAMETER_LIST_GROUP (list->data);
		if (! group->nodes_source) { 
			/* there is only one non-constrained parameter */
			GdaParameter *param;
			GdaValueType type;
			const GdaValue *val, *default_val, *value;
			gboolean nnul;
			gchar *plugin = NULL;

			g_assert (g_slist_length (group->nodes) == 1);

			param = GDA_PARAMETER (GDA_PARAMETER_LIST_NODE (group->nodes->data)->param);

			val = gda_parameter_get_value (param);
			default_val = gda_parameter_get_default_value (param);
			nnul = gda_parameter_get_not_null (param);

			/* determine initial value */
			type = gda_parameter_get_gda_type (param);
			value = val;
			if (!value && default_val && 
			    (gda_value_get_type ((GdaValue *) default_val) == type))
				value = default_val;
			
			/* create entry */
			g_object_get (G_OBJECT (param), "handler_plugin", &plugin, NULL);
			entry = GTK_WIDGET (gnome_db_util_new_data_entry (type, plugin));


			/* set current value */
			gnome_db_data_entry_set_value (GNOME_DB_DATA_ENTRY (entry), val);

			if (!nnul ||
			    (nnul && value && 
			     (gda_value_get_type ((GdaValue *) value) != GDA_VALUE_TYPE_NULL)))
				gnome_db_data_entry_set_value_orig (GNOME_DB_DATA_ENTRY (entry), value);
			
			if (default_val) {
				gnome_db_data_entry_set_value_default (GNOME_DB_DATA_ENTRY (entry), default_val);
				gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
								    GDA_VALUE_ATTR_CAN_BE_DEFAULT,
								    GDA_VALUE_ATTR_CAN_BE_DEFAULT);
			}

			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
							    nnul ? 0 : GDA_VALUE_ATTR_CAN_BE_NULL,
							    GDA_VALUE_ATTR_CAN_BE_NULL);
			    
			g_object_set_data (G_OBJECT (entry), "param", param);
			g_object_set_data (G_OBJECT (entry), "form", form);
			form->priv->entries = g_slist_append (form->priv->entries, entry);
			g_signal_connect (entry, "destroy", G_CALLBACK (entry_destroyed_cb), form);

			/* connect to the parameter's changes */
                        form->priv->signal_ids[i] = g_signal_connect (G_OBJECT (param), "changed",
                                                                      G_CALLBACK (parameter_changed_cb), 
								      entry);
                        i++;
		}
		else { 
			/* several parameters depending on the values of a GdaDataModel object */
			GSList *plist;
			gboolean nnul = TRUE;

			entry = gnome_db_entry_combo_new (form->priv->paramlist, group->nodes_source);
			g_object_set_data (G_OBJECT (entry), "group", group);
			g_object_set_data (G_OBJECT (entry), "form", form);
			form->priv->entries = g_slist_append (form->priv->entries, entry);
			g_signal_connect (entry, "destroy", G_CALLBACK (entry_destroyed_cb), form);

			/* connect to the parameter's changes */
			plist = group->nodes;
			while (plist) {
				GdaParameter *param;

				param = GDA_PARAMETER_LIST_NODE (plist->data)->param;
				if (!gda_parameter_get_not_null (param))
					nnul = FALSE;
				form->priv->signal_ids[i] = g_signal_connect (param, "changed",
                                                                              G_CALLBACK (parameter_changed_cb), 
									      entry);
                                i++;
				plist = g_slist_next (plist);
			}
			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entry),
							    nnul ? 0 : GDA_VALUE_ATTR_CAN_BE_NULL,
							    GDA_VALUE_ATTR_CAN_BE_NULL);
		}

		/* connect the entry's changes */
		g_signal_connect (G_OBJECT (entry), "contents_modified",
				  G_CALLBACK (entry_contents_modified), form);
		list = g_slist_next (list);
	}


	/*
	 * If there is a layout spec, then try to use it
	 */
	if (form->priv->layout_spec) {
		GtkWidget *layout = NULL;
		
		layout = glade_xml_get_widget (form->priv->layout_spec->xml_object, form->priv->layout_spec->root_element);
		if (!layout) {
			g_warning (_("Can't find widget named '%s', returning to basic layout"), 
				   form->priv->layout_spec->root_element);
			layout_spec_free (form->priv->layout_spec);
			form->priv->layout_spec = NULL;
		}
		else {
			/* really use the provided layout */
			GtkWidget *box;
			GSList *groups;
			
			gtk_box_pack_start (GTK_BOX (form), layout,  TRUE, TRUE, 0);
			list = form->priv->entries;
			groups = form->priv->paramlist->groups_list;
			while (groups && list) {
				gint param_no;
				gchar *box_name;

				param_no = g_slist_index (form->priv->paramlist->parameters,
							  ((GdaParameterListNode *)(((GdaParameterListGroup *)groups->data)->nodes->data))->param);
				box_name = g_strdup_printf ("%s_%d", form->priv->layout_spec->form_prefix, param_no);
				box = glade_xml_get_widget (form->priv->layout_spec->xml_object, box_name);
				g_print ("Box named %s => %p\n", box_name, box);
				g_free (box_name);
				if (box) {
					gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET (list->data),
							    gnome_db_data_entry_expand_in_layout (GNOME_DB_DATA_ENTRY (list->data)),
							    TRUE, 0);
					gtk_widget_show (GTK_WIDGET (list->data));
					if (! g_object_get_data (G_OBJECT (box), "show_actions")) 
						gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (list->data),
										    0, GDA_VALUE_ATTR_ACTIONS_SHOWN);
				}
				list = g_slist_next (list);
				groups = g_slist_next (groups);
			}
			g_assert (!groups && !list);
			gtk_widget_show (layout);
		}
	}

	/* 
	 * There is no layout spec (or the provided one could not be used),
	 * so use the default tables arrangment
	 */
	if (!form->priv->layout_spec) {
		GtkWidget *table, *label;

		/* creating a table for all the entries */
		table = gtk_table_new (g_slist_length (form->priv->entries), 2, FALSE);
		gtk_table_set_row_spacings (GTK_TABLE (table), 5);
		gtk_table_set_col_spacings (GTK_TABLE (table), 5);
		form->priv->entries_table = table;
		gtk_box_pack_start (GTK_BOX (form), table,  FALSE, TRUE, 0);
		list = form->priv->entries;
		i = 0;
		while (list) {
			gboolean expand;
			GtkWidget *entry_label;
			GdaParameter *param;
			
			/* label for the entry */
			param = g_object_get_data (G_OBJECT (list->data), "param");
			if (param) {
				const gchar *str;
				gchar *str2;
				GtkWidget *evbox;
				
				str = gda_object_get_name (GDA_OBJECT (param));
				if (!str)
					str = _("Value");
				str2 = g_strdup_printf ("%s:", str);
				evbox = gtk_event_box_new ();
				label = gtk_label_new (str2);
				if (gda_parameter_get_not_null (param)) {
					gpointer data = NULL;
					form->priv->not_null_labels = g_slist_prepend (form->priv->not_null_labels, label);
					data = form->priv->not_null_labels->data;
				}
				
				gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
				g_free (str2);
				gtk_container_add (GTK_CONTAINER (evbox), label);
				
				gtk_table_attach (GTK_TABLE (table), evbox, 0, 1, i, i+1,
						  GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
				gtk_widget_show (evbox);
				gtk_widget_show (label);
				entry_label = evbox;
				
				str = gda_object_get_description (GDA_OBJECT (param));
				if (str && *str)
					gtk_tooltips_set_tip (form->priv->tooltips, evbox, str, NULL);
			}
			else {
				/* FIXME: find a better label and tooltip and improve data entry attributes */
				const gchar *str, *title = NULL;
				gchar *str2;
				GtkWidget *evbox;
				gboolean nullok = TRUE;
				GSList *params;
				GdaParameterListGroup *group;

				group = g_object_get_data (G_OBJECT (list->data), "group");
				params = group->nodes;
				while (params) {
					if (nullok && gda_parameter_get_not_null (GDA_PARAMETER_LIST_NODE (params->data)->param))
						nullok = FALSE;
					if (!title)
						title = gda_object_get_name (GDA_OBJECT (GDA_PARAMETER_LIST_NODE (params->data)->param));
					params = g_slist_next (params);
				}
				
				if (!title)
					title = gda_object_get_name (GDA_OBJECT (group->nodes_source->data_model));
				if (!title)
					title = _("Value");
				str2 = g_strdup_printf ("%s:", title);
				evbox = gtk_event_box_new ();
				label = gtk_label_new (str2);
				if (!nullok) 
					form->priv->not_null_labels = g_slist_prepend (form->priv->not_null_labels, label);
				gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
				g_free (str2);
				gtk_container_add (GTK_CONTAINER (evbox), label);
				
				gtk_table_attach (GTK_TABLE (table), evbox, 0, 1, i, i+1,
						  GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
				gtk_widget_show (evbox);
				gtk_widget_show (label);
				entry_label = evbox;
				
				str = gda_object_get_description (GDA_OBJECT (group->nodes_source->data_model));
				if (str && *str)
					gtk_tooltips_set_tip (form->priv->tooltips, evbox, str, NULL);
			}

			/* add the entry itself to the table */
			expand = gnome_db_data_entry_expand_in_layout (GNOME_DB_DATA_ENTRY (list->data));
			gtk_table_attach (GTK_TABLE (table), GTK_WIDGET (list->data), 1, 2, i, i+1,
					  GTK_FILL | GTK_SHRINK | GTK_EXPAND, 
					  GTK_FILL | GTK_SHRINK | expand? GTK_EXPAND : 0, 0, 0);
			gtk_widget_show (GTK_WIDGET (list->data));
			g_object_set_data (G_OBJECT (list->data), "entry_label", entry_label);
			g_object_set_data (G_OBJECT (list->data), "row_no", GINT_TO_POINTER (i));
			
			list = g_slist_next (list);
			i++;
		}
		mark_not_null_entry_labels (form, TRUE);
		gtk_widget_show (table);
	}
}

static void
entry_destroyed_cb (GtkWidget *entry, GnomeDbBasicForm *form)
{
	/* debug only function */
}

/*
 * if @show_mark is TRUE, display the label as bold 
 */
static void
mark_not_null_entry_labels (GnomeDbBasicForm *form, gboolean show_mark)
{
	PangoAttrList *attrs = NULL;
	PangoAttribute *att;
	GSList *list;

	if (show_mark) {
		attrs = pango_attr_list_new ();
		att = pango_attr_weight_new (PANGO_WEIGHT_BOLD);
		att->start_index = 0;
		att->end_index = G_MAXUINT;
		pango_attr_list_insert (attrs, att);
	}

	list = form->priv->not_null_labels;
	while (list) {
		gtk_label_set_attributes (GTK_LABEL (list->data), attrs);
		list = g_slist_next (list);
	}

	if (show_mark)
		pango_attr_list_unref (attrs);
}

static void
entry_contents_modified (GnomeDbDataEntry *entry, GnomeDbBasicForm *form)
{
	GdaParameter *param;
	guint attr;

	attr = gnome_db_data_entry_get_attributes (entry);
	param = g_object_get_data (G_OBJECT (entry), "param");
	if (param) { /* single parameter */
		GdaValue *value;
		
		form->priv->forward_param_updates = FALSE;

		/* parameter's value */
		value = gnome_db_data_entry_get_value (entry);
		if ((!value || gda_value_is_null (value)) &&
		    (attr & GDA_VALUE_ATTR_IS_DEFAULT))
			g_object_set (G_OBJECT (param), "use_default_value", TRUE, NULL);
		else
			g_object_set (G_OBJECT (param), "use_default_value", FALSE, NULL);
		gda_parameter_set_value (param, value);
#ifdef debug_signal
		g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 0, param, TRUE);
#ifdef debug_signal
		g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		form->priv->forward_param_updates = TRUE;
		gda_value_free (value);
	}
	else { /* multiple parameters */
		GSList *params;
		GSList *values, *list;
		GdaParameterListGroup *group;

		group = g_object_get_data (G_OBJECT (entry), "group");
		params = group->nodes;
		values = gnome_db_entry_combo_get_values (GNOME_DB_ENTRY_COMBO (entry));
		g_assert (g_slist_length (params) == g_slist_length (values));

		list = values;
		while (list) {
			/* REM: if there is more than one value in 'params', then a 
			 * signal is emitted for each param that is changed, 
			 * and there is no way for the listener of that signal to know if it
			 * the end of the "param_changed" sequence. What could be done is:
			 * - adding another boolean to tell if that signal is the 
			 *   last one in the "param_changed" sequence, or
			 * - modify the signal to add a list of parameters which are changed 
			 *   and emit only one signal.
			 */
			GdaParameter *param;
			form->priv->forward_param_updates = FALSE;

			/* parameter's value */
			param = GDA_PARAMETER_LIST_NODE (params->data)->param;
			gda_parameter_set_value (param, (GdaValue *)(list->data));
#ifdef debug_signal
			g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
			g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 
				       0, param, TRUE);
#ifdef debug_signal
			g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
			form->priv->forward_param_updates = TRUE;;

			list = g_slist_next (list);
			params = g_slist_next (params);
		}
		g_slist_free (values);

		/* updating the GdaDataProxy if there is one */
		if (GDA_IS_DATA_MODEL_ITER (form->priv->paramlist)) {
			GdaDataProxy *proxy;
			gint proxy_row;
			
			proxy_row = gda_data_model_iter_get_row ((GdaDataModelIter *) form->priv->paramlist);

			g_object_get (G_OBJECT (form->priv->paramlist), "data_model", &proxy, NULL);
			if (GDA_IS_DATA_PROXY (proxy)) {
				GSList *all_new_values;
				gint i, col;

				all_new_values = gnome_db_entry_combo_get_all_values (GNOME_DB_ENTRY_COMBO (entry));
				for (i = 0; i < group->nodes_source->shown_n_cols; i++) {
					GdaValue *value;
					
					col = group->nodes_source->shown_cols_index[i];
					value = (GdaValue *) g_slist_nth_data (all_new_values, col);
					gda_data_proxy_set_model_row_value (proxy, 
									    group->nodes_source->data_model,
									    proxy_row, col, value);
				}
				g_slist_free (all_new_values);
			}
		}
	}
}


/*
 * Called when a parameter changes
 * We emit a "param_changed" signal only if the 'form->priv->forward_param_updates' is TRUE, which means
 * the param change does not come from a GnomeDbDataEntry change.
 */ 
static void
parameter_changed_cb (GdaParameter *param, GnomeDbDataEntry *entry)
{
	GnomeDbBasicForm *form = g_object_get_data (G_OBJECT (entry), "form");
	GdaParameterListGroup *group = g_object_get_data (G_OBJECT (entry), "group");
	const GdaValue *value = gda_parameter_get_value (param);

	if (form->priv->forward_param_updates) {
		gboolean param_valid;
		gboolean default_if_invalid = FALSE;

		/* There can be a feedback from the entry if the param is invalid and "set_default_if_invalid"
		   exists and is TRUE */
		param_valid = gda_parameter_is_valid (param);
		if (!param_valid) 
			if (g_object_class_find_property (G_OBJECT_GET_CLASS (entry), "set_default_if_invalid"))
				g_object_get (G_OBJECT (entry), "set_default_if_invalid", &default_if_invalid, NULL);

		/* updating the corresponding entry */
		if (! default_if_invalid)
			g_signal_handlers_block_by_func (G_OBJECT (entry),
							 G_CALLBACK (entry_contents_modified), form);
		if (group) {
			GSList *values = NULL;
			GSList *list = group->nodes;
			gboolean allnull = TRUE;

			while (list) {
				const GdaValue *pvalue;
				pvalue = gda_parameter_get_value (GDA_PARAMETER_LIST_NODE (list->data)->param);
				values = g_slist_append (values, (GdaValue *) pvalue);
				if (allnull && pvalue && 
				    (gda_value_get_type ((GdaValue *) pvalue) != GDA_VALUE_TYPE_NULL))
					allnull = FALSE;

				list = g_slist_next (list);
			}
			
			if (!allnull) 
				gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (entry), values);
			else 
				gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (entry), NULL);

			g_slist_free (values);
		}
		else
			gnome_db_data_entry_set_value (entry, value);

		if (! default_if_invalid)
			g_signal_handlers_unblock_by_func (G_OBJECT (entry),
							   G_CALLBACK (entry_contents_modified), form);

#ifdef debug_signal
		g_print (">> 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (form), gnome_db_basic_form_signals[PARAM_CHANGED], 0, param, FALSE);
#ifdef debug_signal
		g_print ("<< 'PARAM_CHANGED' from %s\n", __FUNCTION__);
#endif
	}
}

/**
 * gnome_db_basic_form_get_paramlist
 * @form: a #GnomeDbBasicForm widget
 *
 * Get a pointer to the #GdaParameterList used internally by @form to store
 * values
 *
 * Returns:
 */
GdaParameterList *
gnome_db_basic_form_get_paramlist (GnomeDbBasicForm *form)
{
	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), NULL);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->paramlist;
}

/**
 * gnome_db_basic_form_set_current_as_orig
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells @form that the current values in the different entries are
 * to be considered as the original values for all the entries; the immediate
 * consequence is that any sub-sequent call to gnome_db_basic_form_has_been_changed()
 * will return FALSE (of course until any entry is changed).
 */
void
gnome_db_basic_form_set_current_as_orig (GnomeDbBasicForm *form)
{
	GSList *list;
	GdaParameter *param;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	list = form->priv->entries;
	while (list) {
		GdaParameterListGroup *group;

		group = g_object_get_data (G_OBJECT (list->data), "group");

		if (group) {
			/* Combo entry */
			GSList *values = NULL;
			GSList *params = group->nodes;
			gboolean allnull = TRUE;
			
			while (params) {
				const GdaValue *pvalue;
				pvalue = gda_parameter_get_value (GDA_PARAMETER_LIST_NODE (params->data)->param);
				values = g_slist_append (values, (GdaValue *) pvalue);
				if (allnull && pvalue && 
				    (gda_value_get_type ((GdaValue *) pvalue) != GDA_VALUE_TYPE_NULL))
					allnull = FALSE;
				
				params = g_slist_next (params);
			}
			
			if (!allnull) 
				gnome_db_entry_combo_set_values_orig (GNOME_DB_ENTRY_COMBO (list->data), values);
			else 
				gnome_db_entry_combo_set_values_orig (GNOME_DB_ENTRY_COMBO (list->data), NULL);
			
			g_slist_free (values);
		}
		else {
			/* non combo entry */
			param = g_object_get_data (G_OBJECT (list->data), "param");
			gnome_db_data_entry_set_value_orig (GNOME_DB_DATA_ENTRY (list->data), gda_parameter_get_value (param));
		}
		list = g_slist_next (list);
	}
}

/**
 * gnome_db_basic_form_is_valid 
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells if the form can be used as-is (if all the parameters do have some valid values)
 *
 * Returns: TRUE if the form is valid
 */
gboolean
gnome_db_basic_form_is_valid (GnomeDbBasicForm *form)
{
	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), FALSE);
	g_return_val_if_fail (form->priv, FALSE);

	return gda_parameter_list_is_valid (form->priv->paramlist);
}

/**
 * gnome_db_basic_form_get_data_set
 * @form: a #GnomeDbBasicForm widget
 *
 * Get a pointer to the #GdaParameterList object which
 * is modified by @form
 *
 * Returns:
 */
GdaParameterList *
gnome_db_basic_form_get_data_set (GnomeDbBasicForm *form)
{
	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), NULL);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->paramlist;
}

/**
 * gnome_db_basic_form_has_been_changed
 * @form: a #GnomeDbBasicForm widget
 *
 * Tells if the form has had at least on entry changed, or not
 *
 * Returns:
 */
gboolean
gnome_db_basic_form_has_been_changed (GnomeDbBasicForm *form)
{
	gboolean changed = FALSE;
	GSList *list;

	g_return_val_if_fail (form && IS_GNOME_DB_BASIC_FORM (form), FALSE);
	g_return_val_if_fail (form->priv, FALSE);
	
	list = form->priv->entries;
	while (list && !changed) {
		if (! (gnome_db_data_entry_get_attributes (GNOME_DB_DATA_ENTRY (list->data)) & GDA_VALUE_ATTR_IS_UNCHANGED))
			changed = TRUE;
		list = g_slist_next (list);
	}

	return changed;
}

/**
 * gnome_db_basic_form_show_entries_actions
 * @form: a #GnomeDbBasicForm widget
 * @show_actions: a boolean
 *
 * Show or hide the actions button available at the end of each data entry
 * in the form
 */
void
gnome_db_basic_form_show_entries_actions (GnomeDbBasicForm *form, gboolean show_actions)
{
	GSList *entries;
	guint show;
	
	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	show = show_actions ? GDA_VALUE_ATTR_ACTIONS_SHOWN : 0;

	entries = form->priv->entries;
	while (entries) {
		gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entries->data), show, 
						    GDA_VALUE_ATTR_ACTIONS_SHOWN);
		entries = g_slist_next (entries);
	}

	/* mark_not_null_entry_labels (form, show_actions); */
}

/**
 * gnome_db_basic_form_reset
 * @form: a #GnomeDbBasicForm widget
 *
 * Resets all the entries in the form to their
 * original values
 */
void
gnome_db_basic_form_reset (GnomeDbBasicForm *form)
{
	GSList *list;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);
	
	list = form->priv->entries;
	while (list) {
		GdaParameterListNode *node = g_object_get_data (G_OBJECT (list->data), "node");

		if (node) {
			/* Combo entry */
			GSList *values = NULL;

			values = gnome_db_entry_combo_get_values_orig (GNOME_DB_ENTRY_COMBO (list->data));
			gnome_db_entry_combo_set_values (GNOME_DB_ENTRY_COMBO (list->data), values);
			g_slist_free (values);
		}
		else {
			/* non combo entry */
			const GdaValue *value;

			value = gnome_db_data_entry_get_value_orig (GNOME_DB_DATA_ENTRY (list->data));
			gnome_db_data_entry_set_value (GNOME_DB_DATA_ENTRY (list->data), value);
		}
		list = g_slist_next (list);
	}
}


/**
 * gnome_db_basic_form_entry_show
 * @form: a #GnomeDbBasicForm widget
 * @param: a #GdaParameter object
 * @show:
 *
 * Shows or hides the #GnomeDbDataEntry in @form which corresponds to the
 * @param parameter
 */
void
gnome_db_basic_form_entry_show (GnomeDbBasicForm *form, GdaParameter *param, gboolean show)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		GtkWidget *entry = NULL;
		GdaParameter *thisparam = g_object_get_data (G_OBJECT (entries->data), "param");

		if (thisparam) {
			if (thisparam == param)
				entry = GTK_WIDGET (entries->data);
		}
		else {
			/* multiple parameters */
			GSList *params;
			GdaParameterListGroup *group;

			group = g_object_get_data (G_OBJECT (entries->data), "group");
			params = group->nodes;
			while (params && !entry) {
				if (GDA_PARAMETER_LIST_NODE (params->data)->param == (gpointer) param)
					entry = GTK_WIDGET (entries->data);
				params = g_slist_next (params);
			}
		}

		if (entry) {
			gint row = -1;
			GtkWidget *entry_label = g_object_get_data (G_OBJECT (entry), "entry_label");

			if (form->priv->entries_table)
				row = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (entry), "row_no"));

			if (show) {
				if (g_slist_find (form->priv->hidden_entries, entry)) {
					form->priv->hidden_entries = g_slist_remove (form->priv->hidden_entries, entry);
					g_signal_handlers_disconnect_by_func (G_OBJECT (entry), 
									      G_CALLBACK (widget_shown_cb), form);
				}
				gtk_widget_show (entry);

				if (entry_label) {
					if (g_slist_find (form->priv->hidden_entries, entry_label)) {
						form->priv->hidden_entries = g_slist_remove (form->priv->hidden_entries, 
											     entry_label);
						g_signal_handlers_disconnect_by_func (G_OBJECT (entry_label), 
										      G_CALLBACK (widget_shown_cb), form);
					}
					gtk_widget_show (entry_label);
				}
				if (row > -1) 
					gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 5);
			}
			else {
				if (!g_slist_find (form->priv->hidden_entries, entry)) {
					form->priv->hidden_entries = g_slist_append (form->priv->hidden_entries, entry);
					g_signal_connect_after (G_OBJECT (entry), "show", 
								G_CALLBACK (widget_shown_cb), form);
				}
				gtk_widget_hide (entry);

				if (entry_label) {
					if (!g_slist_find (form->priv->hidden_entries, entry_label)) {
						form->priv->hidden_entries = g_slist_append (form->priv->hidden_entries, 
											     entry_label);
						g_signal_connect_after (G_OBJECT (entry_label), "show", 
									G_CALLBACK (widget_shown_cb), form);
					}
					gtk_widget_hide (entry_label);
				}
				if (row > -1)
					gtk_table_set_row_spacing (GTK_TABLE (form->priv->entries_table), row, 0);
			}
		}

		entries = g_slist_next (entries);
	}
}

/**
 * gnome_db_basic_form_entry_set_sensitive
 * @form: a #GnomeDbBasicForm widget
 * @param: a #GdaParameter object
 * @sensitive:
 *
 * Shows or hides the #GnomeDbDataEntry in @form which corresponds to the
 * @param parameter
 */
void
gnome_db_basic_form_entry_set_sensitive (GnomeDbBasicForm *form, GdaParameter *param, gboolean sensitive)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		GtkWidget *entry = NULL;
		GdaParameter *thisparam = g_object_get_data (G_OBJECT (entries->data), "param");

		if (thisparam) {
			if (thisparam == param)
				entry = GTK_WIDGET (entries->data);
		}
		else {
			/* multiple parameters */
			GSList *params;
			GdaParameterListGroup *group;

			group = g_object_get_data (G_OBJECT (entries->data), "group");
			params = group->nodes;
			while (params && !entry) {
				if (GDA_PARAMETER_LIST_NODE (params->data)->param == (gpointer) param)
					entry = GTK_WIDGET (entries->data);
				params = g_slist_next (params);
			}
		}

		if (entry) {
			GtkWidget *entry_label = g_object_get_data (G_OBJECT (entry), "entry_label");

			gtk_widget_set_sensitive (entry, sensitive);
			if (entry_label)
				gtk_widget_set_sensitive (entry_label, sensitive || !form->priv->headers_sensitive);
		}

		entries = g_slist_next (entries);
	}
}


/**
 * gnome_db_basic_form_set_entries_auto_default
 * @form: a #GnomeDbBasicForm widget
 * @auto_default:
 *
 * Sets weather all the #GnomeDbDataEntry entries in the form must default
 * to a default value if they are assigned a non valid value.
 * Depending on the real type of entry, it will provide a default value
 * which the user does not need to modify if it is OK.
 *
 * For example a date entry can by default display the current date.
 */
void
gnome_db_basic_form_set_entries_auto_default (GnomeDbBasicForm *form, gboolean auto_default)
{
	GSList *entries;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		if (g_object_class_find_property (G_OBJECT_GET_CLASS (entries->data), "set_default_if_invalid"))
			g_object_set (G_OBJECT (entries->data), "set_default_if_invalid", auto_default, NULL);
		entries = g_slist_next (entries);
	}	
}

/**
 * gnome_db_basic_form_set_entries_default
 * @form: a #GnomeDbBasicForm widget
 *
 * For each entry in the form, sets it to a default value if it is possible to do so.
 */
void
gnome_db_basic_form_set_entries_default (GnomeDbBasicForm *form)
{
	GSList *entries;
	guint attrs;

	g_return_if_fail (form && IS_GNOME_DB_BASIC_FORM (form));
	g_return_if_fail (form->priv);

	entries = form->priv->entries;
	while (entries) {
		attrs = gnome_db_data_entry_get_attributes (GNOME_DB_DATA_ENTRY (entries->data));
		if (attrs & GDA_VALUE_ATTR_CAN_BE_DEFAULT)
			gnome_db_data_entry_set_attributes (GNOME_DB_DATA_ENTRY (entries->data), 
						      GDA_VALUE_ATTR_IS_DEFAULT, GDA_VALUE_ATTR_IS_DEFAULT);
		entries = g_slist_next (entries);
	}
}

static void form_param_changed (GnomeDbBasicForm *form, GdaParameter *param, gboolean is_user_modif, GtkDialog *dlg);

/**
 * gnome_db_basic_form_new_in_dialog
 * @dict: a #GdaDict object
 * @paramlist: a #GdaParameterList structure
 * @parent: the parent window for the new dialog, or %NULL
 * @title: the title of the dialog window, or %NULL
 * @header: a helper text displayed at the top of the dialog, or %NULL
 *
 * Creates a new #GnomeDbBasicForm widget in the same way as gnome_db_basic_form_new()
 * and puts it into a #GtkDialog widget. The returned dialog has the "Ok" and "Cancel" buttons
 * which respectively return GTK_RESPONSE_ACCEPT and GTK_RESPONSE_REJECT.
 *
 * The #GnomeDbBasicForm widget is attached to the dialog using the user property
 * "form".
 *
 * Returns: the new #GtkDialog widget
 */
GtkWidget *
gnome_db_basic_form_new_in_dialog (GdaParameterList *paramlist, GtkWindow *parent,
				   const gchar *title, const gchar *header)
{
	GtkWidget *form;
	GtkWidget *dlg;
	const gchar *rtitle;

	form = gnome_db_basic_form_new (paramlist);
 
	rtitle = title;
	if (!rtitle)
		rtitle = _("Values to be filled");
		
	dlg = gtk_dialog_new_with_buttons (rtitle, parent,
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK,
					   GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
	if (header && *header) {
		GtkWidget *label;

		label = gtk_label_new (NULL);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
		gtk_label_set_markup (GTK_LABEL (label), header);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), label, FALSE, FALSE, 5);
		gtk_widget_show (label);
	}

	gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (dlg)->vbox), 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), form, TRUE, TRUE, 10);

	g_signal_connect (G_OBJECT (form), "param_changed",
			  G_CALLBACK (form_param_changed), dlg);
	g_object_set_data (G_OBJECT (dlg), "form", form);

	gtk_widget_show_all (form);
	form_param_changed (GNOME_DB_BASIC_FORM (form), NULL, FALSE, GTK_DIALOG (dlg));

	return dlg;
}

static void
form_param_changed (GnomeDbBasicForm *form, GdaParameter *param, gboolean is_user_modif, GtkDialog *dlg)
{
	gboolean valid;

	valid = gnome_db_basic_form_is_valid (form);

	gtk_dialog_set_response_sensitive (dlg, GTK_RESPONSE_ACCEPT, valid);
}
