/* gnome-db-entry-passmd5.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-entry-passmd5.h"
#include <libgda/gda-data-handler.h>
#include <openssl/evp.h>
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_entry_passmd5_class_init (GnomeDbEntryPassmd5Class * class);
static void gnome_db_entry_passmd5_init (GnomeDbEntryPassmd5 * srv);
static void gnome_db_entry_passmd5_finalize (GObject   * object);

/* virtual functions */
static GtkWidget *create_entry (GnomeDbEntryWrapper *mgwrap);
static void       real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value);
static GdaValue  *real_get_value (GnomeDbEntryWrapper *mgwrap);
static void       connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback);
static gboolean   expand_in_layout (GnomeDbEntryWrapper *mgwrap);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _GnomeDbEntryPassmd5Private
{
	GtkWidget  *entry;
};


GType
gnome_db_entry_passmd5_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntryPassmd5Class),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_entry_passmd5_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbEntryPassmd5),
			0,
			(GInstanceInitFunc) gnome_db_entry_passmd5_init
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_ENTRY_WRAPPER, "GnomeDbEntryPassmd5", &info, 0);
	}
	return type;
}

static void
gnome_db_entry_passmd5_class_init (GnomeDbEntryPassmd5Class * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->finalize = gnome_db_entry_passmd5_finalize;

	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->create_entry = create_entry;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_set_value = real_set_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_get_value = real_get_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->connect_signals = connect_signals;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->expand_in_layout = expand_in_layout;
}

static void
gnome_db_entry_passmd5_init (GnomeDbEntryPassmd5 * gnome_db_entry_passmd5)
{
	gnome_db_entry_passmd5->priv = g_new0 (GnomeDbEntryPassmd5Private, 1);
	gnome_db_entry_passmd5->priv->entry = NULL;
}

/**
 * gnome_db_entry_passmd5_new
 * @dh: the data handler to be used by the new widget
 * @type: the requested data type (compatible with @dh)
 *
 * Creates a new widget which is mainly a GtkEntry
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_entry_passmd5_new (GdaDataHandler *dh, GdaValueType type)
{
	GObject *obj;
	GnomeDbEntryPassmd5 *mgtxt;

	g_return_val_if_fail (dh && GDA_IS_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, NULL);
	g_return_val_if_fail (gda_data_handler_accepts_gda_type (dh, type), NULL);

	obj = g_object_new (GNOME_DB_ENTRY_PASSMD5_TYPE, "handler", dh, NULL);
	mgtxt = GNOME_DB_ENTRY_PASSMD5 (obj);
	gnome_db_data_entry_set_value_type (GNOME_DB_DATA_ENTRY (mgtxt), type);

	return GTK_WIDGET (obj);
}

static void
gnome_db_entry_passmd5_finalize (GObject   * object)
{
	GnomeDbEntryPassmd5 *gnome_db_entry_passmd5;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_ENTRY_PASSMD5 (object));

	gnome_db_entry_passmd5 = GNOME_DB_ENTRY_PASSMD5 (object);
	if (gnome_db_entry_passmd5->priv) {
		g_free (gnome_db_entry_passmd5->priv);
		gnome_db_entry_passmd5->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static GtkWidget *
create_entry (GnomeDbEntryWrapper *mgwrap)
{
	GtkWidget *entry;
        GnomeDbEntryPassmd5 *mgstr;

        g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_PASSMD5 (mgwrap), NULL);
        mgstr = GNOME_DB_ENTRY_PASSMD5 (mgwrap);
        g_return_val_if_fail (mgstr->priv, NULL);

        entry = gtk_entry_new ();
        mgstr->priv->entry = entry;
	gtk_entry_set_visibility (GTK_ENTRY (entry), FALSE);

        return entry;
}

static void
real_set_value (GnomeDbEntryWrapper *mgwrap, const GdaValue *value)
{
	GnomeDbEntryPassmd5 *mgstr;

        g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_PASSMD5 (mgwrap));
        mgstr = GNOME_DB_ENTRY_PASSMD5 (mgwrap);
        g_return_if_fail (mgstr->priv);

        if (value) {
                if (gda_value_is_null ((GdaValue *) value))
                        gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), "");
                else {
                        GdaDataHandler *dh;
                        gchar *str;

                        dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
                        str = gda_data_handler_get_str_from_value (dh, value);
                        gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), str);
			g_free (str);
                }
        }
        else
                gtk_entry_set_text (GTK_ENTRY (mgstr->priv->entry), "");
}

static GdaValue *
real_get_value (GnomeDbEntryWrapper *mgwrap)
{
	GdaValue *value;
        GnomeDbEntryPassmd5 *mgstr;
        GdaDataHandler *dh;
        const gchar *str;

	EVP_MD_CTX mdctx;
        const EVP_MD *md;
        unsigned char md5str [EVP_MAX_MD_SIZE+1];
        int md_len, i;
        GString *md5pass;

        g_return_val_if_fail (mgwrap && IS_GNOME_DB_ENTRY_PASSMD5 (mgwrap), NULL);
        mgstr = GNOME_DB_ENTRY_PASSMD5 (mgwrap);
        g_return_val_if_fail (mgstr->priv, NULL);

        dh = gnome_db_data_entry_get_handler (GNOME_DB_DATA_ENTRY (mgwrap));
        str = gtk_entry_get_text (GTK_ENTRY (mgstr->priv->entry));

        /* calcul MD5 */
        OpenSSL_add_all_digests ();
        md = EVP_md5 ();
        EVP_DigestInit(&mdctx, md);
        EVP_DigestUpdate(&mdctx, str, strlen(str));
        EVP_DigestFinal(&mdctx, md5str, &md_len);
        md5str [md_len] = 0;
        md5pass = g_string_new ("");
        for(i = 0; i < md_len; i++)
                g_string_append_printf (md5pass, "%02x", md5str[i]);

        value = gda_data_handler_get_value_from_sql (dh, md5pass->str, gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgwrap)));
        g_string_free (md5pass, TRUE);

        if (!value) {
                /* in case the gda_data_handler_get_value_from_sql() returned an error because
                   the contents of the GtkEntry cannot be interpreted as a GdaValue */
                value = gda_value_new_null ();
        }

        return value;
}

static void
connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback)
{
	GnomeDbEntryPassmd5 *mgstr;

        g_return_if_fail (mgwrap && IS_GNOME_DB_ENTRY_PASSMD5 (mgwrap));
        mgstr = GNOME_DB_ENTRY_PASSMD5 (mgwrap);
        g_return_if_fail (mgstr->priv);

        g_signal_connect (G_OBJECT (mgstr->priv->entry), "changed",
                          callback, mgwrap);
}

static gboolean
expand_in_layout (GnomeDbEntryWrapper *mgwrap)
{
	return FALSE;
}
