/* gnome-db-qf-field.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-qf-field.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-ref-base.h"
#include "marshal.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_qf_field_class_init (GnomeDbQfFieldClass * class);
static void gnome_db_qf_field_init (GnomeDbQfField *qf);
static void gnome_db_qf_field_dispose (GObject *object);
static void gnome_db_qf_field_finalize (GObject *object);

static void gnome_db_qf_field_set_property (GObject              *object,
				      guint                 param_id,
				      const GValue         *value,
				      GParamSpec           *pspec);
static void gnome_db_qf_field_get_property (GObject              *object,
				      guint                 param_id,
				      GValue               *value,
				      GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_qf_field_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_qf_field_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_qf_field_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_qf_field_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void              gnome_db_qf_field_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_qf_field_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_qf_field_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_qf_field_renderer_init   (GnomeDbRendererIface *iface);
static gchar          *gnome_db_qf_field_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_qf_field_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_qf_field_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_qf_field_activate            (GnomeDbReferer *iface);
static void        gnome_db_qf_field_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_qf_field_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_qf_field_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_qf_field_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* virtual functions */
static GObject          *gnome_db_qf_field_copy           (GnomeDbQfield *orig);
static gboolean          gnome_db_qf_field_is_equal       (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2);

#ifdef debug
static void        gnome_db_qf_field_dump           (GnomeDbQfField *field, guint offset);
#endif


/* When the GnomeDbQuery is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbQfField *field);
static void target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbQfField *field);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_qf_field_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY,
	PROP_VALUE_PROVIDER_OBJECT,
	PROP_VALUE_PROVIDER_XML_ID,
	PROP_HANDLER_PLUGIN
};


/* private structure */
struct _GnomeDbQfFieldPrivate
{
	GnomeDbQuery    *query;
	GnomeDbRefBase  *target_ref; /* references a GnomeDbTarget */
	GnomeDbRefBase  *field_ref;  /* references a GnomeDbField in the entity behind the GnomeDbTarget */
	GnomeDbRefBase  *value_prov_ref;

	gchar       *plugin;       /* specific plugin to be used */
};


/* module error */
GQuark gnome_db_qf_field_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qf_field_error");
	return quark;
}


GType
gnome_db_qf_field_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qf_field_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfField),
			0,
			(GInstanceInitFunc) gnome_db_qf_field_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_qf_field_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_qf_field_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_qf_field_renderer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_qf_field_referer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_TYPE_QFIELD, "GnomeDbQfField", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_FIELD, &field_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_RENDERER, &renderer_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
	}
	return type;
}

static void 
gnome_db_qf_field_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_qf_field_get_xml_id;
	iface->save_to_xml = gnome_db_qf_field_save_to_xml;
	iface->load_from_xml = gnome_db_qf_field_load_from_xml;
}

static void
gnome_db_qf_field_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_qf_field_get_entity;
	iface->get_data_type = gnome_db_qf_field_get_data_type;
}

static void
gnome_db_qf_field_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_sql = gnome_db_qf_field_render_as_sql;
	iface->render_as_str = gnome_db_qf_field_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_qf_field_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_qf_field_activate;
        iface->deactivate = gnome_db_qf_field_deactivate;
        iface->is_active = gnome_db_qf_field_is_active;
        iface->get_ref_objects = gnome_db_qf_field_get_ref_objects;
        iface->replace_refs = gnome_db_qf_field_replace_refs;
}

static void
gnome_db_qf_field_class_init (GnomeDbQfFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_qf_field_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQfFieldClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_qf_field_dispose;
	object_class->finalize = gnome_db_qf_field_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qf_field_set_property;
	object_class->get_property = gnome_db_qf_field_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_VALUE_PROVIDER_OBJECT,
					 g_param_spec_pointer ("value_provider", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_VALUE_PROVIDER_XML_ID,
					 g_param_spec_string ("value_provider_xml_id", NULL, NULL, NULL,
							      (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_HANDLER_PLUGIN,
                                         g_param_spec_string ("handler_plugin", NULL, NULL, NULL,
                                                              (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_qf_field_dump;
#endif
	GNOME_DB_QFIELD_CLASS (class)->copy = gnome_db_qf_field_copy;
	GNOME_DB_QFIELD_CLASS (class)->is_equal = gnome_db_qf_field_is_equal;
	GNOME_DB_QFIELD_CLASS (class)->is_list = NULL;
	GNOME_DB_QFIELD_CLASS (class)->get_params = NULL;
}

static void
gnome_db_qf_field_init (GnomeDbQfField *gnome_db_qf_field)
{
	gnome_db_qf_field->priv = g_new0 (GnomeDbQfFieldPrivate, 1);
	gnome_db_qf_field->priv->query = NULL;
	gnome_db_qf_field->priv->target_ref = NULL;
	gnome_db_qf_field->priv->field_ref = NULL;
	gnome_db_qf_field->priv->value_prov_ref = NULL;
	gnome_db_qf_field->priv->plugin = NULL;
}


/**
 * gnome_db_qf_field_new_with_objects
 * @query: a #GnomeDbQuery in which the new object will be
 * @target: a #GnomeDbTarget object
 * @field: an object which implements the #GnomeDbField interface
 *
 * Creates a new GnomeDbQfField object which represents the @field of the entity represented
 * by @target. For example if @target represents my_table, and @field represents the "name" field,
 * then the created object would represent 'my_table.name' in SQL notation
 *
 * REM: @field MUST belong to the #GnomeDbEntity represented by @target (we need the @target since
 * an entity can appear several times within a query, for self joins for example).
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_field_new_with_objects (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbField *field)
{
	GObject   *obj;
	GnomeDbQfField *gnome_db_qf_field;
	GnomeDbDict *dict;
	GnomeDbEntity *ent_f, *ent_t;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target && IS_GNOME_DB_TARGET (target), NULL);
	g_return_val_if_fail (gnome_db_target_get_query (target) == query, NULL);
	g_return_val_if_fail (gnome_db_referer_activate (GNOME_DB_REFERER (target)), NULL);
	g_return_val_if_fail (field && IS_GNOME_DB_FIELD (field), NULL);
	ent_f = gnome_db_field_get_entity (field);
	ent_t = gnome_db_target_get_represented_entity (target);
	g_return_val_if_fail (ent_f == ent_t, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_TYPE_QF_FIELD, "dict", dict, NULL);
	gnome_db_qf_field = GNOME_DB_QF_FIELD (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_field), id);

	gnome_db_qf_field->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_field);
	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_qf_field);

	gnome_db_qf_field->priv->target_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_qf_field->priv->target_ref, GNOME_DB_BASE (target));

	gnome_db_qf_field->priv->field_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_qf_field->priv->field_ref, GNOME_DB_BASE (field));
	
	return obj;
}

/**
 * gnome_db_qf_field_new_with_xml_ids
 * @query: a #GnomeDbQuery in which the new object will be
 * @target_xml_id: the XML Id of a #GnomeDbTarget object
 * @field_xml_id: the XML Id of an object which implements the #GnomeDbField interface
 *
 * Creates a new GnomeDbQfField object which represents a given field of the entity represented
 * by the target identified by @target_xml_id. 
 *
 * REM: the aimed field MUST belong the the entity represented by the aimed target. This test is
 * not done here but when this object will be activated (activation wll be refused if that test fails).
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_field_new_with_xml_ids (GnomeDbQuery *query, const gchar *target_xml_id, const gchar *field_xml_id)
{
	GObject   *obj;
	GnomeDbQfField *gnome_db_qf_field;
	GnomeDbDict *dict;
	gchar *str, *ptr, *tok, *qid;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (target_xml_id && *target_xml_id, NULL);
	g_return_val_if_fail (field_xml_id && *field_xml_id, NULL);

	qid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (query));
	str = g_strdup (target_xml_id);
	ptr = strtok_r (str, ":", &tok);
        g_return_val_if_fail (!strcmp (ptr, qid), NULL);
	g_free (qid);
	g_free (str);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_TYPE_QF_FIELD, "dict", dict, NULL);
	gnome_db_qf_field = GNOME_DB_QF_FIELD (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_field), id);

	gnome_db_qf_field->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_field);
	g_signal_connect (G_OBJECT (query), "target_removed",
			  G_CALLBACK (target_removed_cb), gnome_db_qf_field);

	gnome_db_qf_field->priv->target_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_qf_field->priv->target_ref, GNOME_DB_TYPE_TARGET, REFERENCE_BY_XML_ID, target_xml_id);

	gnome_db_qf_field->priv->field_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_qf_field->priv->field_ref, GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, field_xml_id);
	
	return obj;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbQfField *field)
{
	gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static void
target_removed_cb (GnomeDbQuery *query, GnomeDbTarget *target, GnomeDbQfField *field)
{
	GnomeDbBase *base;

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base && (GNOME_DB_TARGET (base) == target)) 
		gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static void
gnome_db_qf_field_dispose (GObject *object)
{
	GnomeDbQfField *gnome_db_qf_field;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_FIELD (object));

	gnome_db_qf_field = GNOME_DB_QF_FIELD (object);
	if (gnome_db_qf_field->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_qf_field->priv->value_prov_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_field->priv->value_prov_ref));
			gnome_db_qf_field->priv->value_prov_ref = NULL;
		}

		if (gnome_db_qf_field->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_field->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_field);
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_field->priv->query),
							      G_CALLBACK (target_removed_cb), gnome_db_qf_field);
			gnome_db_qf_field->priv->query = NULL;
		}
		if (gnome_db_qf_field->priv->target_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_field->priv->target_ref));
			gnome_db_qf_field->priv->target_ref = NULL;
		}
		if (gnome_db_qf_field->priv->field_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_field->priv->field_ref));
			gnome_db_qf_field->priv->field_ref = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qf_field_finalize (GObject   * object)
{
	GnomeDbQfField *gnome_db_qf_field;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_FIELD (object));

	gnome_db_qf_field = GNOME_DB_QF_FIELD (object);
	if (gnome_db_qf_field->priv) {
		if (gnome_db_qf_field->priv->plugin)
			g_free (gnome_db_qf_field->priv->plugin);

		g_free (gnome_db_qf_field->priv);
		gnome_db_qf_field->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qf_field_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbQfField *gnome_db_qf_field;
	gpointer ptr;
	const gchar *val;

	gnome_db_qf_field = GNOME_DB_QF_FIELD (object);
	if (gnome_db_qf_field->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

			if (gnome_db_qf_field->priv->query) {
				if (gnome_db_qf_field->priv->query == GNOME_DB_QUERY (ptr))
					return;

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_field->priv->query),
								      G_CALLBACK (nullified_object_cb), gnome_db_qf_field);
				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_field->priv->query),
								      G_CALLBACK (target_removed_cb), gnome_db_qf_field);
			}

			gnome_db_qf_field->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr, 
						 G_CALLBACK (nullified_object_cb), gnome_db_qf_field);
			g_signal_connect (G_OBJECT (ptr), "target_removed",
					  G_CALLBACK (target_removed_cb), gnome_db_qf_field);
			break;
		case PROP_VALUE_PROVIDER_OBJECT:
			ptr = g_value_get_pointer (value);
			if (ptr) {
				g_return_if_fail (IS_GNOME_DB_QFIELD (ptr));
				g_return_if_fail (gnome_db_field_get_entity (GNOME_DB_FIELD (ptr)) == 
						  GNOME_DB_ENTITY (gnome_db_qf_field->priv->query));
				if (!gnome_db_qf_field->priv->value_prov_ref)
					gnome_db_qf_field->priv->value_prov_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (gnome_db_qf_field))));
				gnome_db_ref_base_set_ref_object_type (gnome_db_qf_field->priv->value_prov_ref,
								 ptr, GNOME_DB_TYPE_FIELD);
			}
			else {
				if (gnome_db_qf_field->priv->value_prov_ref) {
					g_object_unref (G_OBJECT (gnome_db_qf_field->priv->value_prov_ref));
					gnome_db_qf_field->priv->value_prov_ref = NULL;
				}
			}
			break;
		case PROP_VALUE_PROVIDER_XML_ID:
			val = g_value_get_string (value);
			if (val && *val) {
				gchar *qid, *str, *start, *tok;

				str = g_strdup (val);
				start = strtok_r (str, ":", &tok);
				qid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (gnome_db_qf_field->priv->query));
				g_return_if_fail (!strcmp (start, qid));
				g_free (str);
				g_free (qid);

				if (!gnome_db_qf_field->priv->value_prov_ref)
					gnome_db_qf_field->priv->value_prov_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (gnome_db_qf_field))));
				gnome_db_ref_base_set_ref_name (gnome_db_qf_field->priv->value_prov_ref,
							  GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, val);
			}
			else {
				if (gnome_db_qf_field->priv->value_prov_ref) {
					g_object_unref (G_OBJECT (gnome_db_qf_field->priv->value_prov_ref));
					gnome_db_qf_field->priv->value_prov_ref = NULL;
				}
			}
			break;
		case PROP_HANDLER_PLUGIN:
			val =  g_value_get_string (value);
			if (gnome_db_qf_field->priv->plugin) {
				g_free (gnome_db_qf_field->priv->plugin);
				gnome_db_qf_field->priv->plugin = NULL;
			}
			if (val)
				gnome_db_qf_field->priv->plugin = g_strdup (val);
			break;
		}
	}
}

static void
gnome_db_qf_field_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfField *gnome_db_qf_field;
	gnome_db_qf_field = GNOME_DB_QF_FIELD (object);
	
	if (gnome_db_qf_field->priv) {
		switch (param_id) {
		case PROP_QUERY:
			g_value_set_pointer (value, gnome_db_qf_field->priv->query);
			break;
		case PROP_VALUE_PROVIDER_OBJECT:
			if (gnome_db_qf_field->priv->value_prov_ref)
				g_value_set_pointer (value, 
						     gnome_db_ref_base_get_ref_object (gnome_db_qf_field->priv->value_prov_ref));
			else
				g_value_set_pointer (value, NULL);
			break;
		case PROP_VALUE_PROVIDER_XML_ID:
			if (gnome_db_qf_field->priv->value_prov_ref)
				g_value_set_string (value, 
						    gnome_db_ref_base_get_ref_name (gnome_db_qf_field->priv->value_prov_ref,
									      NULL, NULL));
			else
				g_value_set_string (value, NULL);
			break;
		case PROP_HANDLER_PLUGIN:
			g_value_set_string (value, gnome_db_qf_field->priv->plugin);
			break;
		}	
	}
}

static GObject *
gnome_db_qf_field_copy (GnomeDbQfield *orig)
{
	GnomeDbQfField *qf;
	GObject *obj;
	g_assert (IS_GNOME_DB_QF_FIELD (orig));
	qf = GNOME_DB_QF_FIELD (orig);

	obj = gnome_db_qf_field_new_with_xml_ids (qf->priv->query, 
					    gnome_db_ref_base_get_ref_name (qf->priv->target_ref, NULL, NULL),
					    gnome_db_ref_base_get_ref_name (qf->priv->field_ref, NULL, NULL));
	if (gnome_db_base_get_name (GNOME_DB_BASE (orig)))
		gnome_db_base_set_name (GNOME_DB_BASE (obj), gnome_db_base_get_name (GNOME_DB_BASE (orig)));

	if (gnome_db_base_get_description (GNOME_DB_BASE (orig)))
		gnome_db_base_set_description (GNOME_DB_BASE (obj), gnome_db_base_get_description (GNOME_DB_BASE (orig)));

	if (qf->priv->value_prov_ref) {
		GnomeDbBase *ref = gnome_db_ref_base_get_ref_object (qf->priv->value_prov_ref);
		if (ref)
			g_object_set (obj, "value_provider", ref, NULL);
		else
			g_object_set (obj, "value_provider_xml_id",
				      gnome_db_ref_base_get_ref_name (qf->priv->value_prov_ref, NULL, NULL), NULL);
	}

	if (qf->priv->plugin)
		GNOME_DB_QF_FIELD (obj)->priv->plugin = g_strdup (qf->priv->plugin);

	return obj;
}

static gboolean
gnome_db_qf_field_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	GnomeDbBase *ref1, *ref2;
	gboolean retval;
	g_assert (IS_GNOME_DB_QF_FIELD (qfield1));
	g_assert (IS_GNOME_DB_QF_FIELD (qfield2));
	
	/* it is here assumed that qfield1 and qfield2 are of the same type and refer to the same
	   query */
	ref1 = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FIELD (qfield1)->priv->target_ref);
	ref2 = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FIELD (qfield2)->priv->target_ref);

	retval = (ref1 == ref2) ? TRUE : FALSE;
	if (retval) {
		ref1 = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FIELD (qfield1)->priv->field_ref);
		ref2 = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FIELD (qfield2)->priv->field_ref);
		retval = (ref1 == ref2) ? TRUE : FALSE;
	}

	return retval;
}

/**
 * gnome_db_qf_field_get_ref_field
 * @field: a #GnomeDbQfField object
 *
 * Get the real #GnomeDbField object (well, the object which implements that interface)
 * referenced by @field
 *
 * Returns: the #GnomeDbField object, or NULL if @field is not active
 */
GnomeDbField *
gnome_db_qf_field_get_ref_field (GnomeDbQfField *field)
{
	GnomeDbBase *base;
	g_return_val_if_fail (field && IS_GNOME_DB_QF_FIELD (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (field->priv->field_ref);
	if (base)
		return GNOME_DB_FIELD (base);
	else
		return NULL;
}

/**
 * gnome_db_qf_field_get_target
 * @field: a #GnomeDbQfField object
 *
 * Get the #GnomeDbTarget object @field 'belongs' to
 *
 * Returns: the #GnomeDbTarget object
 */
GnomeDbTarget *
gnome_db_qf_field_get_target (GnomeDbQfField *field)
{
	GnomeDbBase *base;
	g_return_val_if_fail (field && IS_GNOME_DB_QF_FIELD (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	if (base)
		return GNOME_DB_TARGET (base);
	else
		return NULL;
}

#ifdef debug
static void
gnome_db_qf_field_dump (GnomeDbQfField *field, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (field && IS_GNOME_DB_QF_FIELD (field));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (field->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbQfField" D_COL_NOR " \"%s\" (%p, id=%d) ",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (field)), field, gnome_db_base_get_id (GNOME_DB_BASE (field)));
		if (! gnome_db_qf_field_is_active (GNOME_DB_REFERER (field)))
			g_print (D_COL_ERR "Inactive" D_COL_NOR ", ");
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
			g_print ("Visible, ");
		if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
			g_print ("Internal, ");
		g_print ("references Target %s & Field %s", 
			 /* gnome_db_ref_base_get_ref_object (field->priv->target_ref), */
			 gnome_db_ref_base_get_ref_name (field->priv->target_ref, NULL, NULL),
			 /* gnome_db_ref_base_get_ref_object (field->priv->field_ref), */
			 gnome_db_ref_base_get_ref_name (field->priv->field_ref, NULL, NULL));

		if (field->priv->value_prov_ref) 
			g_print (" Value prov: %p (%s)\n",
				 gnome_db_ref_base_get_ref_object (field->priv->value_prov_ref),
				 gnome_db_ref_base_get_ref_name (field->priv->value_prov_ref, NULL, NULL));
		else
			g_print ("\n");
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, field);
}
#endif


/* 
 * GnomeDbField interface implementation
 */
static GnomeDbEntity *
gnome_db_qf_field_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_QF_FIELD (iface)->priv->query);
}

static GnomeDbServerDataType *
gnome_db_qf_field_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);
	
	if (gnome_db_qf_field_activate (GNOME_DB_REFERER (iface))) {
		GnomeDbField *field;
		field = GNOME_DB_FIELD (gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FIELD (iface)->priv->field_ref));
		return gnome_db_field_get_data_type (field);
	}

	return NULL;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_qf_field_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *q_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);

	q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_QF_FIELD (iface)->priv->query));
	xml_id = g_strdup_printf ("%s:QF%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
	g_free (q_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_qf_field_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbQfField *field;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);

	field = GNOME_DB_QF_FIELD (iface);

	node = xmlNewNode (NULL, "GNOME_DB_QF");
	
	str = gnome_db_qf_field_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);

	xmlSetProp (node, "type", "FIELD");
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (field)));
	if (gnome_db_base_get_description (GNOME_DB_BASE (field)) && *gnome_db_base_get_description (GNOME_DB_BASE (field)))
		xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (field)));
	xmlSetProp (node, "target", gnome_db_ref_base_get_ref_name (field->priv->target_ref, NULL, NULL));
	xmlSetProp (node, "object", gnome_db_ref_base_get_ref_name (field->priv->field_ref, NULL, NULL));
	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_visible",  "f");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_internal", "t");

	if (field->priv->value_prov_ref)
		xmlSetProp (node, "value_prov", gnome_db_ref_base_get_ref_name (field->priv->value_prov_ref, NULL, NULL));

	str = gnome_db_qfield_get_alias (GNOME_DB_QFIELD (field));
	if (str && *str) 
		xmlSetProp (node, "alias", str);

	if (field->priv->plugin)
		xmlSetProp (node, "plugin", field->priv->plugin);

	return node;
}

static gboolean
gnome_db_qf_field_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQfField *field;
	gchar *prop;
	gboolean target = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	field = GNOME_DB_QF_FIELD (iface);
	if (strcmp (node->name, "GNOME_DB_QF")) {
		g_set_error (error,
			     GNOME_DB_QF_FIELD_ERROR,
			     GNOME_DB_QF_FIELD_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QF>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		if (strcmp (prop, "FIELD")) {
			g_set_error (error,
				     GNOME_DB_QF_FIELD_ERROR,
				     GNOME_DB_QF_FIELD_XML_LOAD_ERROR,
				     _("Wrong type of field in <GNOME_DB_QF>"));
			return FALSE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "id");
	if (prop) {
		gchar *ptr, *tok;
		ptr = strtok_r (prop, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (strlen (ptr) < 3) {
			g_set_error (error,
				     GNOME_DB_QF_FIELD_ERROR,
				     GNOME_DB_QF_FIELD_XML_LOAD_ERROR,
				     _("Wrong 'id' attribute in <GNOME_DB_QF>"));
			return FALSE;
		}
		gnome_db_base_set_id (GNOME_DB_BASE (field), atoi (ptr+2));
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "target");
	if (prop) {
		target = TRUE;
		gnome_db_ref_base_set_ref_name (field->priv->target_ref, GNOME_DB_TYPE_TARGET, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "object");
	if (prop) {
		target = TRUE;
		gnome_db_ref_base_set_ref_name (field->priv->field_ref, GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_visible");
	if (prop) {
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_internal");
	if (prop) {
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "value_prov");
	if (prop) {
		g_object_set (G_OBJECT (iface), "value_provider_xml_id", prop, NULL);
		g_free (prop);
	}

	prop = xmlGetProp (node, "alias");
	if (prop) {
		gnome_db_qfield_set_alias (GNOME_DB_QFIELD (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "plugin");
	if (prop) 
		field->priv->plugin = prop;

	if (target)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_QF_FIELD_ERROR,
			     GNOME_DB_QF_FIELD_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_QF>"));
		return FALSE;
	}
}


/*
 * GnomeDbRenderer interface implementation
 */

static gchar *
gnome_db_qf_field_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;
	GnomeDbBase *base, *base2;
	GnomeDbQfField *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);
	field = GNOME_DB_QF_FIELD (iface);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	base2 = gnome_db_ref_base_get_ref_object (field->priv->field_ref);

	if (base && base2) {
		GnomeDbQuery *query;

		/* if the query is a SELECT query, then we qualify the field with the target */
		query = gnome_db_target_get_query (GNOME_DB_TARGET (base));
		if (gnome_db_query_is_modif_query (query))
			str = g_strdup_printf ("%s", gnome_db_field_get_name (GNOME_DB_FIELD (base2)));
		else
			str = g_strdup_printf ("%s.%s", gnome_db_target_get_alias (GNOME_DB_TARGET (base)),
					       gnome_db_field_get_name (GNOME_DB_FIELD (base2)));
	}
	else {
		if (base)
			g_set_error (error,
				     GNOME_DB_QF_FIELD_ERROR,
				     GNOME_DB_QF_FIELD_RENDER_ERROR,
				     _("Can't find target '%s'"), gnome_db_ref_base_get_ref_name (field->priv->target_ref,
											    NULL, NULL));
		else
			g_set_error (error,
				     GNOME_DB_QF_FIELD_ERROR,
				     GNOME_DB_QF_FIELD_RENDER_ERROR,
				     _("Can't find field '%s'"), gnome_db_ref_base_get_ref_name (field->priv->field_ref,
											   NULL, NULL));
	}

	
	return str;
}

static gchar *
gnome_db_qf_field_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;
	GnomeDbBase *base, *base2;
	GnomeDbQfField *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);
	field = GNOME_DB_QF_FIELD (iface);

	base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
	base2 = gnome_db_ref_base_get_ref_object (field->priv->field_ref);

	if (base && base2) {
		GnomeDbEntity *ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (base));
		str = g_strdup_printf ("%s(%s).%s", gnome_db_base_get_name (GNOME_DB_BASE (ent)), 
				       gnome_db_target_get_alias (GNOME_DB_TARGET (base)),
				       gnome_db_field_get_name (GNOME_DB_FIELD (base2)));
	}
	else 
		str = g_strdup (_("Non activated field"));
	
	return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_qf_field_activate (GnomeDbReferer *iface)
{
	gboolean act1, act2;
	gboolean active = FALSE;
	GnomeDbQfField *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, FALSE);
	field = GNOME_DB_QF_FIELD (iface);

	act1 = gnome_db_ref_base_activate (field->priv->target_ref);
	act2 = gnome_db_ref_base_activate (field->priv->field_ref);
	if (act1 && act2) {
		/* coherence test */
		GnomeDbTarget *target;
		GnomeDbField *rfield;

		target = GNOME_DB_TARGET (gnome_db_ref_base_get_ref_object (field->priv->target_ref));
		rfield = GNOME_DB_FIELD (gnome_db_ref_base_get_ref_object (field->priv->field_ref));
		if (gnome_db_target_get_represented_entity (target) != gnome_db_field_get_entity (rfield))
			gnome_db_ref_base_deactivate (field->priv->field_ref);
		else
			active = TRUE;
	}

	if (active && field->priv->value_prov_ref)
			active = gnome_db_ref_base_activate (field->priv->value_prov_ref);

	return active;
}

static void
gnome_db_qf_field_deactivate (GnomeDbReferer *iface)
{
	GnomeDbQfField *field;
	g_return_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface));
	g_return_if_fail (GNOME_DB_QF_FIELD (iface)->priv);
	field = GNOME_DB_QF_FIELD (iface);

	gnome_db_ref_base_deactivate (field->priv->target_ref);
	gnome_db_ref_base_deactivate (field->priv->field_ref);
	if (field->priv->value_prov_ref)
		gnome_db_ref_base_deactivate (field->priv->value_prov_ref);
}

static gboolean
gnome_db_qf_field_is_active (GnomeDbReferer *iface)
{
	gboolean active;
	GnomeDbQfField *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, FALSE);
	field = GNOME_DB_QF_FIELD (iface);

	active =  gnome_db_ref_base_is_active (field->priv->target_ref) &&
		gnome_db_ref_base_is_active (field->priv->field_ref);

	if (active && field->priv->value_prov_ref)
		active = gnome_db_ref_base_is_active (field->priv->value_prov_ref);

	return active;
}

static GSList *
gnome_db_qf_field_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
        GnomeDbBase *base;
	GnomeDbQfField *field;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FIELD (iface)->priv, NULL);
	field = GNOME_DB_QF_FIELD (iface);

        base = gnome_db_ref_base_get_ref_object (field->priv->target_ref);
        if (base)
                list = g_slist_append (list, base);

        base = gnome_db_ref_base_get_ref_object (field->priv->field_ref);
        if (base)
                list = g_slist_append (list, base);

	if (field->priv->value_prov_ref) {
		base = gnome_db_ref_base_get_ref_object (field->priv->value_prov_ref);
		if (base)
			list = g_slist_append (list, base);
	}

        return list;
}

static void
gnome_db_qf_field_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbQfField *field;

        g_return_if_fail (iface && IS_GNOME_DB_QF_FIELD (iface));
        g_return_if_fail (GNOME_DB_QF_FIELD (iface)->priv);

        field = GNOME_DB_QF_FIELD (iface);
        if (field->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, field->priv->query);
                if (query) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->query),
                                                              G_CALLBACK (nullified_object_cb), field);
                        g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->query),
                                                              G_CALLBACK (target_removed_cb), field);
                        field->priv->query = query;
			gnome_db_base_connect_nullify (query, 
						 G_CALLBACK (nullified_object_cb), field);
                        g_signal_connect (G_OBJECT (query), "target_removed",
                                          G_CALLBACK (target_removed_cb), field);
                }
        }

        gnome_db_ref_base_replace_ref_object (field->priv->target_ref, replacements);
        gnome_db_ref_base_replace_ref_object (field->priv->field_ref, replacements);
	if (field->priv->value_prov_ref)
		gnome_db_ref_base_replace_ref_object (field->priv->value_prov_ref, replacements);
}
