/* gnome-db-constraint.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-constraint.h"
#include "gnome-db-referer.h"
#include "gnome-db-database.h"
#include "gnome-db-table.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-entity.h"
#include "gnome-db-table-field.h"
#include "gnome-db-field.h"
#include "marshal.h"
#include "utility.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_constraint_class_init (GnomeDbConstraintClass * class);
static void gnome_db_constraint_init (GnomeDbConstraint * srv);
static void gnome_db_constraint_dispose (GObject   * object);
static void gnome_db_constraint_finalize (GObject   * object);

static void gnome_db_constraint_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_constraint_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_constraint_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_constraint_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_constraint_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_constraint_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);


/* GnomeDbReferer interface */
static void        gnome_db_constraint_referer_init (GnomeDbRefererIface *iface);
static gboolean    gnome_db_constraint_activate            (GnomeDbReferer *iface);
static void        gnome_db_constraint_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_constraint_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_constraint_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_constraint_replace_refs        (GnomeDbReferer *iface, GHashTable *replacaments);

#ifdef debug
static void        gnome_db_constraint_dump                (GnomeDbConstraint *cstr, guint offset);
#endif

/* When a DbTable or GnomeDbTableField is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbConstraint *cstr);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_constraint_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_USER_CSTR
};


/* private structure */
struct _GnomeDbConstraintPrivate
{
	GnomeDbConstraintType      type;
	GnomeDbTable              *table;
	gboolean                user_defined; /* FALSE if the constraint is hard coded into the database */
	
	/* Only used if single field constraint */
	GnomeDbTableField              *single_field;

	/* Only used if Primary key */
	GSList                 *multiple_fields; 

	/* Only used for foreign keys */
	GnomeDbTable              *ref_table; 
	GSList                 *fk_pairs;  
	GnomeDbConstraintFkAction  on_delete;
	GnomeDbConstraintFkAction  on_update;
	
};


/* module error */
GQuark gnome_db_constraint_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_constraint_error");
	return quark;
}


GType
gnome_db_constraint_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbConstraintClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_constraint_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbConstraint),
			0,
			(GInstanceInitFunc) gnome_db_constraint_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_constraint_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_constraint_referer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbConstraint", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
	}
	return type;
}

static void 
gnome_db_constraint_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_constraint_get_xml_id;
	iface->save_to_xml = gnome_db_constraint_save_to_xml;
	iface->load_from_xml = gnome_db_constraint_load_from_xml;
}

static void
gnome_db_constraint_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = gnome_db_constraint_activate;
	iface->deactivate = gnome_db_constraint_deactivate;
	iface->is_active = gnome_db_constraint_is_active;
	iface->get_ref_objects = gnome_db_constraint_get_ref_objects;
	iface->replace_refs = gnome_db_constraint_replace_refs;
}

static void
gnome_db_constraint_class_init (GnomeDbConstraintClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_constraint_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbConstraintClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_constraint_dispose;
	object_class->finalize = gnome_db_constraint_finalize;

	/* Properties */
	object_class->set_property = gnome_db_constraint_set_property;
	object_class->get_property = gnome_db_constraint_get_property;
	g_object_class_install_property (object_class, PROP_USER_CSTR,
					 g_param_spec_boolean ("user_constraint", NULL, NULL, FALSE,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_constraint_dump;
#endif
}

static void
gnome_db_constraint_init (GnomeDbConstraint * gnome_db_constraint)
{
	gnome_db_constraint->priv = g_new0 (GnomeDbConstraintPrivate, 1);
	gnome_db_constraint->priv->type = CONSTRAINT_PRIMARY_KEY;
	gnome_db_constraint->priv->table = NULL;
	gnome_db_constraint->priv->user_defined = FALSE;
	gnome_db_constraint->priv->single_field = NULL;
	gnome_db_constraint->priv->multiple_fields = NULL;
	gnome_db_constraint->priv->ref_table = NULL;
	gnome_db_constraint->priv->fk_pairs = NULL;
	gnome_db_constraint->priv->on_delete = CONSTRAINT_FK_ACTION_NO_ACTION;
	gnome_db_constraint->priv->on_update = CONSTRAINT_FK_ACTION_NO_ACTION;
}


/**
 * gnome_db_constraint_new
 * @table: the #GnomeDbTable to which the constraint is attached
 * @type: the type of constraint
 *
 * Creates a new GnomeDbConstraint object
 *
 * Returns: the new object
 */
GObject*
gnome_db_constraint_new (GnomeDbTable *table, GnomeDbConstraintType type)
{
	GObject   *obj;
	GnomeDbConstraint *gnome_db_constraint;
	GnomeDbDict *dict = NULL;

	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (table));
	
	obj = g_object_new (GNOME_DB_TYPE_CONSTRAINT, "dict", dict, NULL);

	gnome_db_constraint = GNOME_DB_CONSTRAINT (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_constraint), 0);

	gnome_db_constraint->priv->type = type;
	gnome_db_constraint->priv->table = table;

	gnome_db_base_connect_nullify (table, G_CALLBACK (nullified_object_cb), gnome_db_constraint);

	return obj;
}

/**
 * gnome_db_constraint_new_with_db
 * @db: a #GnomeDbDatabase object
 * 
 * Creates a new #GnomeDbConstraint object without specifying anything about the
 * constraint except the database it is attached to. This is usefull only
 * when the object is going to be loaded from an XML node.
 *
 * Returns: the new uninitialized object
 */
GObject *
gnome_db_constraint_new_with_db (GnomeDbDatabase *db)
{
	GObject   *obj;
	GnomeDbConstraint *gnome_db_constraint;
	GnomeDbDict *dict = NULL;

	/* here priv->table will be NULL and a "db" data is attached to the object */

	g_return_val_if_fail (db && IS_GNOME_DB_DATABASE (db), NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (db));
	
	obj = g_object_new (GNOME_DB_TYPE_CONSTRAINT, "dict", dict, NULL);

	gnome_db_constraint = GNOME_DB_CONSTRAINT (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_constraint), 0);
	
	g_object_set_data (obj, "db", db);

	gnome_db_base_connect_nullify (db, G_CALLBACK (nullified_object_cb), gnome_db_constraint);
	return obj;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbConstraint *cstr)
{
	gnome_db_base_nullify (GNOME_DB_BASE (cstr));
}

static void
gnome_db_constraint_dispose (GObject *object)
{
	GnomeDbConstraint *cstr;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CONSTRAINT (object));

	cstr = GNOME_DB_CONSTRAINT (object);
	if (cstr->priv) {
		GSList *list;

		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		switch (cstr->priv->type) {
		case CONSTRAINT_PRIMARY_KEY:
		case CONSTRAINT_UNIQUE:
			list = cstr->priv->multiple_fields;
			while (list) {
				g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
								      G_CALLBACK (nullified_object_cb), cstr);
				list = g_slist_next (list);
			}
			g_slist_free (cstr->priv->multiple_fields);
			cstr->priv->multiple_fields = NULL;
			break;
		case CONSTRAINT_FOREIGN_KEY:
			if (cstr->priv->ref_table)
				g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->ref_table),
								      G_CALLBACK (nullified_object_cb), cstr);
			cstr->priv->ref_table = NULL;

			list = cstr->priv->fk_pairs;
			while (list) {
				GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);

				g_signal_handlers_disconnect_by_func (G_OBJECT (pair->fkey),
								      G_CALLBACK (nullified_object_cb), cstr);
				if (pair->ref_pkey)
					g_signal_handlers_disconnect_by_func (G_OBJECT (pair->ref_pkey),
									      G_CALLBACK (nullified_object_cb), cstr);
				if (pair->ref_pkey_repl)
					g_object_unref (G_OBJECT (pair->ref_pkey_repl));
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_slist_free (cstr->priv->fk_pairs);
			cstr->priv->fk_pairs = NULL;
			break;
		case CONSTRAINT_NOT_NULL:
			if (cstr->priv->single_field)
				g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->single_field),
								      G_CALLBACK (nullified_object_cb), cstr);
			cstr->priv->single_field = NULL;
			break;
		case CONSTRAINT_CHECK_EXPR:
		default:
			TO_IMPLEMENT;
		}

		if (g_object_get_data (G_OBJECT (cstr), "db")) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (g_object_get_data (G_OBJECT (cstr), "db")),
							      G_CALLBACK (nullified_object_cb), cstr);
			g_object_set_data (G_OBJECT (cstr), "db", NULL);
		}
		
		if (cstr->priv->table) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->table),
							      G_CALLBACK (nullified_object_cb), cstr);
			cstr->priv->table = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_constraint_finalize (GObject   * object)
{
	GnomeDbConstraint *gnome_db_constraint;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_CONSTRAINT (object));

	gnome_db_constraint = GNOME_DB_CONSTRAINT (object);
	if (gnome_db_constraint->priv) {

		g_free (gnome_db_constraint->priv);
		gnome_db_constraint->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_constraint_set_property (GObject              *object,
			       guint                 param_id,
			       const GValue         *value,
			       GParamSpec           *pspec)
{
	GnomeDbConstraint *gnome_db_constraint;

	gnome_db_constraint = GNOME_DB_CONSTRAINT (object);
	if (gnome_db_constraint->priv) {
		switch (param_id) {
		case PROP_USER_CSTR:
			gnome_db_constraint->priv->user_defined = g_value_get_boolean (value);
			break;
		}
	}
}

static void
gnome_db_constraint_get_property (GObject              *object,
			       guint                 param_id,
			       GValue               *value,
			       GParamSpec           *pspec)
{
	GnomeDbConstraint *gnome_db_constraint;
	gnome_db_constraint = GNOME_DB_CONSTRAINT (object);
	
	if (gnome_db_constraint->priv) {
		switch (param_id) {
		case PROP_USER_CSTR:
			g_value_set_boolean (value, gnome_db_constraint->priv->user_defined);
			break;
		}	
	}
}


/**
 * gnome_db_constraint_get_constraint_type
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the type of constraint the @cstr object represents
 *
 * Returns: the constraint type
 */
GnomeDbConstraintType
gnome_db_constraint_get_constraint_type (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), CONSTRAINT_UNKNOWN);
	g_return_val_if_fail (cstr->priv, CONSTRAINT_UNKNOWN);
	g_return_val_if_fail (cstr->priv->table, CONSTRAINT_UNKNOWN);

	return cstr->priv->type;
}

/**
 * gnome_db_constraint_equal
 * @cstr1: the first #GnomeDbConstraint to compare
 * @cstr2: the second #GnomeDbConstraint to compare
 *
 * Compares two #GnomeDbConstraint objects to see if they are equal, without taking into account the
 * name of the constraints or weather they are user or system defined
 *
 * Returns: TRUE if the two constraints are equal and FALSE otherwise
 */
gboolean
gnome_db_constraint_equal (GnomeDbConstraint *cstr1, GnomeDbConstraint *cstr2)
{
	gboolean equal = TRUE;
	GSList *list1, *list2;

	g_return_val_if_fail (cstr1 && IS_GNOME_DB_CONSTRAINT (cstr1), FALSE);
	g_return_val_if_fail (cstr1->priv, FALSE);
	g_return_val_if_fail (cstr2 && IS_GNOME_DB_CONSTRAINT (cstr2), FALSE);
	g_return_val_if_fail (cstr2->priv, FALSE);
	g_return_val_if_fail (cstr1->priv->table, FALSE);
	g_return_val_if_fail (cstr2->priv->table, FALSE);

	if (cstr1->priv->type != cstr2->priv->type)
		return FALSE;

	if (cstr1->priv->table != cstr2->priv->table)
		return FALSE;
	
	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr1));
	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr2));

	switch (cstr1->priv->type) {
	case CONSTRAINT_PRIMARY_KEY:
	case CONSTRAINT_UNIQUE:
		list1 = cstr1->priv->multiple_fields;
		list2 = cstr2->priv->multiple_fields;
		while (list1 && list2 && equal) {
			if (list1->data != list2->data)
				equal = FALSE;
			list1 = g_slist_next (list1);
			list2 = g_slist_next (list2);
		}
		if (list1 || list2)
			equal = FALSE;
		break;
	case CONSTRAINT_FOREIGN_KEY:
		list1 = cstr1->priv->fk_pairs;
		list2 = cstr2->priv->fk_pairs;
		while (list1 && list2 && equal) {
			if (GNOME_DB_CONSTRAINT_FK_PAIR (list1->data)->fkey != 
			    GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->fkey)
				equal = FALSE;
			if (GNOME_DB_CONSTRAINT_FK_PAIR (list1->data)->ref_pkey != 
			    GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey)
				equal = FALSE;

			if (GNOME_DB_CONSTRAINT_FK_PAIR (list1->data)->ref_pkey_repl &&
			    GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey_repl) {
				GType type1, type2;
				GnomeDbRefBaseType itype1, itype2;
				const gchar *name1, *name2;
				
				name1 = gnome_db_ref_base_get_ref_name (GNOME_DB_CONSTRAINT_FK_PAIR (list1->data)->ref_pkey_repl,
								  &type1, &itype1);
				name2 = gnome_db_ref_base_get_ref_name (GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey_repl,
								  &type2, &itype2);
				if ((type1 != type2) || (itype1 != itype2) ||
				    strcmp (name1, name2))
					equal = FALSE;
			}
			else {
				if (GNOME_DB_CONSTRAINT_FK_PAIR (list1->data)->ref_pkey_repl ||
				    GNOME_DB_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey_repl)
					equal = FALSE;
			}

			list1 = g_slist_next (list1);
			list2 = g_slist_next (list2);
		}
		if (list1 || list2)
			equal = FALSE;
		break;
	case CONSTRAINT_NOT_NULL:
		if (cstr1->priv->single_field != cstr2->priv->single_field)
			equal = FALSE;
		break;
	case CONSTRAINT_CHECK_EXPR:
		TO_IMPLEMENT;
		break;
	default:
		equal = FALSE;
		break;
	}

	return equal;
}


/**
 * gnome_db_constraint_get_table
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the table to which the constraint is attached
 *
 * Returns: the #GnomeDbTable
 */
GnomeDbTable *
gnome_db_constraint_get_table (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);

	return cstr->priv->table;
}

/**
 * gnome_db_constraint_uses_field
 * @cstr: a #GnomeDbConstraint object
 * @field: a #GnomeDbTableField object
 *
 * Tests if @field is part of the @cstr constraint
 *
 * Returns: TRUE if @cstr uses @field
 */
gboolean
gnome_db_constraint_uses_field (GnomeDbConstraint *cstr, GnomeDbTableField *field)
{
	gboolean retval = FALSE;
	GSList *list;

	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), FALSE);
	g_return_val_if_fail (cstr->priv, FALSE);
	g_return_val_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field), FALSE);

	switch (gnome_db_constraint_get_constraint_type (cstr)) {
	case CONSTRAINT_PRIMARY_KEY:
	case CONSTRAINT_UNIQUE:
		if (g_slist_find (cstr->priv->multiple_fields, field))
			retval = TRUE;
		break;
	case CONSTRAINT_FOREIGN_KEY:
		list = cstr->priv->fk_pairs;
		while (list && !retval) {
			if (GNOME_DB_CONSTRAINT_FK_PAIR (list->data)->fkey == field)
				retval = TRUE;
			list = g_slist_next (list);
		}
		break;
	case CONSTRAINT_NOT_NULL:
		if (cstr->priv->single_field == field)
			retval = TRUE;
		break;
	case CONSTRAINT_CHECK_EXPR:
	default:
		TO_IMPLEMENT;
	}
	
	return retval;
}


static void gnome_db_constraint_multiple_set_fields (GnomeDbConstraint *cstr, const GSList *fields);

/**
* gnome_db_constraint_pkey_set_fields
* @cstr: a #GnomeDbConstraint object
* @fields: a list of #GnomeDbTableField objects
*
* Sets the fields which make the primary key represented by @cstr. All the fields
* must belong to the same #GnomeDbTable to which the constraint is attached
*/
void
gnome_db_constraint_pkey_set_fields (GnomeDbConstraint *cstr, const GSList *fields)
{
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_PRIMARY_KEY);
	g_return_if_fail (cstr->priv->table);
	g_return_if_fail (fields);

	gnome_db_constraint_multiple_set_fields (cstr, fields);
}


static void
gnome_db_constraint_multiple_set_fields (GnomeDbConstraint *cstr, const GSList *fields)
{
	const GSList *list;
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));

	/* testing for errors */
	list = fields;
	while (list) {
		if (!list->data) {
			g_warning ("List contains a NULL value, not a field");
			return;
		}
		if (!IS_GNOME_DB_TABLE_FIELD (list->data)) {
			g_warning ("List item %p is not a field\n", list->data);
			return;
		}
		if (gnome_db_field_get_entity (GNOME_DB_FIELD (list->data)) != GNOME_DB_ENTITY (cstr->priv->table)) {
			g_warning ("Field %p belongs to a table different from the constraint\n", list->data);
			return;
		}

		list = g_slist_next (list);
	}

	/* if a fields list is already here, get rid of it */
	if (cstr->priv->multiple_fields) {
		list = cstr->priv->multiple_fields;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (nullified_object_cb), cstr);
			list = g_slist_next (list);
		}
		g_slist_free (cstr->priv->multiple_fields);
		cstr->priv->multiple_fields = NULL;
	}

	/* make a copy of the new fields list */
	list = fields;
	while (list) {
		gnome_db_base_connect_nullify (list->data, G_CALLBACK (nullified_object_cb), cstr);
		cstr->priv->multiple_fields = g_slist_append (cstr->priv->multiple_fields, list->data);
		list = g_slist_next (list);
	}
}

/**
 * gnome_db_constraint_pkey_get_fields
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the list of fields composing the primary key constraint which @cstr represents. The
 * returned list is allocated and must be de-allocated by the caller.
 *
 * Returns: a new list of fields
 */
GSList *
gnome_db_constraint_pkey_get_fields (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->type == CONSTRAINT_PRIMARY_KEY, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);

	return g_slist_copy (cstr->priv->multiple_fields);
}


/** 
 * gnome_db_constraint_fkey_set_fields
 * @cstr: a #GnomeDbConstraint object
 * @pairs: a list of #GnomeDbTableField objects
 *
 * Sets the field pairs which make the foreign key represented by @cstr. All the field pairs
 * must list a field which belong to the same #GnomeDbTable to which the constraint is attached
 * and a field which belongs to a #GnomeDbTable which is different from the one just mentionned and which
 * is within the same database.
 * The pairs are of type #GnomeDbConstraintFkeyPair.
 */
void
gnome_db_constraint_fkey_set_fields (GnomeDbConstraint *cstr, const GSList *pairs)
{
	const GSList *list;
	GnomeDbTable *ref_table = NULL;
	GSList *oldlist;

	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_FOREIGN_KEY);
	g_return_if_fail (cstr->priv->table);

	/* testing for errors */
	list = pairs;
	while (list) {
		GnomeDbConstraintFkeyPair *pair;

		if (!list->data) {
			g_warning ("List contains a NULL value, not a pair of fields");
			return;
		}

		pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
		if (!IS_GNOME_DB_TABLE_FIELD (pair->fkey)) {
			g_warning ("Pair item %p has fkey which is not a is not a field", list->data);
			return;
		}
		if (pair->ref_pkey_repl) { /* we have a GnomeDbRefBase object */
			if (!IS_GNOME_DB_REF_BASE (pair->ref_pkey_repl)) {
				g_warning ("Pair item %p has ref_pkey_repl which is not a is not a GnomeDbRefBase", list->data);
				return;	
			}
			if (gnome_db_ref_base_get_ref_type (pair->ref_pkey_repl) !=
			    GNOME_DB_TYPE_TABLE_FIELD) {
				g_warning ("Pair item %p has ref_pkey_repl which does not reference a field", list->data);
				return;	
			}
		}
		else { /* we have a GnomeDbTableField object */
			if (!pair->ref_pkey || !IS_GNOME_DB_TABLE_FIELD (pair->ref_pkey)) {
				g_warning ("Pair item %p has ref_pkey which is not a is not a field", list->data);
				return;
			}
			if (!ref_table)
				ref_table = GNOME_DB_TABLE (gnome_db_field_get_entity (GNOME_DB_FIELD (pair->ref_pkey)));
			else
				if (gnome_db_field_get_entity (GNOME_DB_FIELD (pair->ref_pkey)) != 
				    GNOME_DB_ENTITY (ref_table)) {
					g_warning ("Referenced table is not the same for all pairs");
					return;
				}
		}

		if (gnome_db_field_get_entity (GNOME_DB_FIELD (pair->fkey)) != 
					 GNOME_DB_ENTITY (cstr->priv->table)) {
			g_warning ("Field %p belongs to a table different from the constraint", 
				   pair->fkey);
			return;
		}

		list = g_slist_next (list);
	}

	/* if a fields list is already here, we treat in two parts: first (before the new list is analysed)
	 * we diocsonnect signals from objects of the old list, and then (after the new list has been analysed)
	 * we remove any reference to unused objects and clear the list */
	oldlist = cstr->priv->fk_pairs;
	list = cstr->priv->fk_pairs;
	while (list) {
		GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
		
		g_signal_handlers_disconnect_by_func (G_OBJECT (pair->fkey),
						      G_CALLBACK (nullified_object_cb), cstr);
		if (pair->ref_pkey)
			g_signal_handlers_disconnect_by_func (G_OBJECT (pair->ref_pkey),
							      G_CALLBACK (nullified_object_cb), cstr);
		
		list = g_slist_next (list);
	}
	if (cstr->priv->ref_table) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->ref_table),
						      G_CALLBACK (nullified_object_cb), cstr);
		cstr->priv->ref_table = NULL;
	}

	/* make a copy of the new fields list */
	cstr->priv->fk_pairs = NULL;
	list = pairs;
	while (list) {
		GnomeDbConstraintFkeyPair *pair;
		pair = g_new0 (GnomeDbConstraintFkeyPair, 1);
		*pair = *(GNOME_DB_CONSTRAINT_FK_PAIR (list->data));

		gnome_db_base_connect_nullify (pair->fkey, 
					 G_CALLBACK (nullified_object_cb), cstr);
		
		if (!pair->ref_pkey_repl) 
			/* here we have tested that pair->ref_pkey is OK */
			gnome_db_base_connect_nullify (pair->ref_pkey, 
						 G_CALLBACK (nullified_object_cb), cstr);
		else 
			g_object_ref (G_OBJECT (pair->ref_pkey_repl));

		cstr->priv->fk_pairs = g_slist_append (cstr->priv->fk_pairs, pair);
		list = g_slist_next (list);
	}
	cstr->priv->ref_table = ref_table;
	if (ref_table)
		gnome_db_base_connect_nullify (ref_table, 
					 G_CALLBACK (nullified_object_cb), cstr);

	/* second part of getting rid of the ols list of fields */
	if (oldlist) {
		list = oldlist;
		while (list) {
			GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);

			if (pair->ref_pkey_repl)
				g_object_unref (G_OBJECT (pair->ref_pkey_repl));
				
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (oldlist);
	}


	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr));
}


/**
 * gnome_db_constraint_fkey_get_ref_table
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the #GnomeDbTable at the other end of the foreign key relation represented by this
 * constraint
 *
 * Returns: the #GnomeDbTable
 */
GnomeDbTable *
gnome_db_constraint_fkey_get_ref_table (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->type == CONSTRAINT_FOREIGN_KEY, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);
	
	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr));

	return cstr->priv->ref_table;
}

/**
 * gnome_db_constraint_fkey_get_fields
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the list of field pairs composing the foreign key constraint which @cstr represents. In the returned
 * list, each pair item is allocated and it's up to the caller to free the list and each pair, and the
 * reference count for each pointer to GObjects in each pair is NOT INCREASED, which means the caller of this
 * function DOES NOT hold any reference on the mentionned GObjects (if he needs to, it has to call g_object_ref())
 *
 * Returns: a new list of #GnomeDbConstraintFkeyPair pairs
 */
GSList *
gnome_db_constraint_fkey_get_fields (GnomeDbConstraint *cstr)
{
	GSList *list, *retval = NULL;

	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->type == CONSTRAINT_FOREIGN_KEY, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);

	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr));
	
	/* copy the list of pairs */
	list = cstr->priv->fk_pairs;
	while (list) {
		GnomeDbConstraintFkeyPair *pair;
		pair = g_new0 (GnomeDbConstraintFkeyPair, 1);
		*pair = * GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
		retval = g_slist_append (retval, pair);
		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_constraint_fkey_set_actions
 * @cstr: a #GnomeDbConstraint object
 * @on_update: the action undertaken when an UPDATE occurs
 * @on_delete: the action undertaken when a DELETE occurs
 *
 * Sets the actions undertaken by the DBMS when some actions occur on the referenced data
 */
void
gnome_db_constraint_fkey_set_actions (GnomeDbConstraint *cstr, 
				   GnomeDbConstraintFkAction on_update, GnomeDbConstraintFkAction on_delete)
{
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_FOREIGN_KEY);
	g_return_if_fail (cstr->priv->table);

	cstr->priv->on_update = on_update;
	cstr->priv->on_delete = on_delete;
}

/**
 * gnome_db_constraint_fkey_get_actions
 * @cstr: a #GnomeDbConstraint object
 * @on_update: an address to store the action undertaken when an UPDATE occurs
 * @on_delete: an address to store the action undertaken when a DELETE occurs
 *
 * Get the actions undertaken by the DBMS when some actions occur on the referenced data
 */
void
gnome_db_constraint_fkey_get_actions (GnomeDbConstraint *cstr, 
				   GnomeDbConstraintFkAction *on_update, GnomeDbConstraintFkAction *on_delete)
{
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_FOREIGN_KEY);
	g_return_if_fail (cstr->priv->table);

	if (on_update)
		*on_update = cstr->priv->on_update;
	if (on_delete)
		*on_delete = cstr->priv->on_delete;
}


/**
 * gnome_db_constraint_unique_set_fields
 * @cstr: a #GnomeDbConstraint object
 * @fields:
 *
 */
void
gnome_db_constraint_unique_set_fields (GnomeDbConstraint *cstr, const GSList *fields)
{
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_UNIQUE);
	g_return_if_fail (cstr->priv->table);
	g_return_if_fail (fields);

	gnome_db_constraint_multiple_set_fields (cstr, fields);
}

/**
 * gnome_db_constraint_unique_get_fields
 * @cstr: a #GnomeDbConstraint object
 *
 * Get the list of fields represented by this UNIQUE constraint. It's up to the caller to free the list.
 *
 * Returns: a new list of fields
 */
GSList *
gnome_db_constraint_unique_get_fields (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->type == CONSTRAINT_UNIQUE, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);

	return g_slist_copy (cstr->priv->multiple_fields);
}

/**
 * gnome_db_constraint_non_null_set_field
 * @cstr: a #GnomeDbConstraint object
 * @field:
 *
 */
void
gnome_db_constraint_not_null_set_field (GnomeDbConstraint *cstr, GnomeDbTableField *field)
{
	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	g_return_if_fail (cstr->priv);
	g_return_if_fail (cstr->priv->type == CONSTRAINT_NOT_NULL);
	g_return_if_fail (cstr->priv->table);
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (gnome_db_field_get_entity (GNOME_DB_FIELD (field)) == GNOME_DB_ENTITY (cstr->priv->table));


	/* if a field  is already here, get rid of it */
	if (cstr->priv->single_field) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->single_field),
						      G_CALLBACK (nullified_object_cb), cstr);
		cstr->priv->single_field = NULL;
	}

	gnome_db_base_connect_nullify (field, G_CALLBACK (nullified_object_cb), cstr);
	cstr->priv->single_field = field;
}

/**
 * gnome_db_constraint_non_null_get_field
 * @cstr: a #GnomeDbConstraint object
 *
 */
GnomeDbTableField *
gnome_db_constraint_not_null_get_field (GnomeDbConstraint *cstr)
{
	g_return_val_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr), NULL);
	g_return_val_if_fail (cstr->priv, NULL);
	g_return_val_if_fail (cstr->priv->type == CONSTRAINT_NOT_NULL, NULL);
	g_return_val_if_fail (cstr->priv->table, NULL);

	return cstr->priv->single_field;
}



#ifdef debug
static void
gnome_db_constraint_dump (GnomeDbConstraint *cstr, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (cstr && IS_GNOME_DB_CONSTRAINT (cstr));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (cstr->priv && cstr->priv->table)
		g_print ("%s" D_COL_H1 "GnomeDbConstraint" D_COL_NOR " %p type=%d name=%s\n",
			 str, cstr, cstr->priv->type, gnome_db_base_get_description (GNOME_DB_BASE (cstr)));
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, cstr);
}
#endif








/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_constraint_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *t_xml_id, *tmp, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), NULL);
	g_return_val_if_fail (GNOME_DB_CONSTRAINT (iface)->priv, NULL);
	g_return_val_if_fail (GNOME_DB_CONSTRAINT (iface)->priv->table, NULL);

	t_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_CONSTRAINT (iface)->priv->table));
	tmp = utility_build_encoded_id ("FI", gnome_db_base_get_name (GNOME_DB_BASE (iface)));
	xml_id = g_strdup_printf ("%s:%s", t_xml_id, tmp);
	g_free (t_xml_id);
	g_free (tmp);
	g_assert_not_reached ();
	
	return xml_id;
}

static const gchar *
constraint_type_to_str (GnomeDbConstraintType type)
{
	switch (type) {
	case CONSTRAINT_PRIMARY_KEY:
		return "PKEY";
		break;
	case CONSTRAINT_FOREIGN_KEY:
		return "FKEY";
		break;
	case CONSTRAINT_UNIQUE:
		return "UNIQ";
		break;
	case CONSTRAINT_NOT_NULL:
		return "NNUL";
		break;
	case CONSTRAINT_CHECK_EXPR:
		return "CHECK";
		break;
	default:
		return "???";
		break;
	}
}

static GnomeDbConstraintType
constraint_str_to_type (const gchar *str)
{
	g_return_val_if_fail (str, CONSTRAINT_UNKNOWN);

	switch (*str) {
	case 'P':
		return CONSTRAINT_PRIMARY_KEY;
		break;
	case 'F':
		return CONSTRAINT_FOREIGN_KEY;
		break;
	case 'U':
		return CONSTRAINT_UNIQUE;
		break;
	case 'N':
		return CONSTRAINT_NOT_NULL;
		break;
	case 'C':
		return CONSTRAINT_CHECK_EXPR;
		break;
	default:
		return CONSTRAINT_UNKNOWN;
		break;
	}
}


static const gchar *
constraint_action_to_str (GnomeDbConstraintFkAction action)
{
	switch (action) {
	case CONSTRAINT_FK_ACTION_CASCADE:
		return "CAS";
		break;
	case CONSTRAINT_FK_ACTION_SET_NULL:
		return "NULL";
		break;
	case CONSTRAINT_FK_ACTION_SET_DEFAULT:
		return "DEF";
		break;
	case CONSTRAINT_FK_ACTION_SET_VALUE:
		return "VAL";
		break;
	case CONSTRAINT_FK_ACTION_NO_ACTION:
		return "RESTRICT";
		break;
	default:
		return "???";
		break;	
	}
}

static GnomeDbConstraintFkAction
constraint_str_to_action (const gchar *str)
{
	g_return_val_if_fail (str, CONSTRAINT_FK_ACTION_NO_ACTION);
	switch (*str) {
	case 'C':
		return CONSTRAINT_FK_ACTION_CASCADE;
		break;
	case 'N':
		return CONSTRAINT_FK_ACTION_SET_NULL;
		break;
	case 'D':
		return CONSTRAINT_FK_ACTION_SET_DEFAULT;
		break;
	case 'V':
		return CONSTRAINT_FK_ACTION_SET_VALUE;
		break;
	case 'R':
		return CONSTRAINT_FK_ACTION_NO_ACTION;
		break;
	default:
		return CONSTRAINT_FK_ACTION_NO_ACTION;
		break;	
	}
}

static xmlNodePtr
gnome_db_constraint_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	xmlNodePtr child;
	GnomeDbConstraint *cstr;
	gchar *str;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), NULL);
	g_return_val_if_fail (GNOME_DB_CONSTRAINT (iface)->priv, NULL);
	g_return_val_if_fail (GNOME_DB_CONSTRAINT (iface)->priv->table, NULL);

	cstr = GNOME_DB_CONSTRAINT (iface);
	gnome_db_constraint_activate (GNOME_DB_REFERER (cstr));

	if (! gnome_db_constraint_is_active (GNOME_DB_REFERER (cstr))) {
		g_set_error (error,
			     GNOME_DB_CONSTRAINT_ERROR,
			     GNOME_DB_CONSTRAINT_XML_SAVE_ERROR,
			     _("Constraint cannot be activated!"));
		return NULL;
	}
	node = xmlNewNode (NULL, "GNOME_DB_CONSTRAINT");
	
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (cstr)));
	xmlSetProp (node, "user_defined", cstr->priv->user_defined ? "t" : "f");
	xmlSetProp (node, "type", constraint_type_to_str (cstr->priv->type));

	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (cstr->priv->table));
	xmlSetProp (node, "table", str);
	g_free (str);

	switch (cstr->priv->type) {
	case CONSTRAINT_UNIQUE:
	case CONSTRAINT_PRIMARY_KEY:
		list = cstr->priv->multiple_fields;
		while (list) {
			child = xmlNewChild (node, NULL, "GNOME_DB_CONSTRAINT_FIELD", NULL);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
			xmlSetProp (child, "field", str);
			g_free (str);
			list = g_slist_next (list);
		}
		break;
	case CONSTRAINT_FOREIGN_KEY:
		list = cstr->priv->fk_pairs;
		while (list) {
			child = xmlNewChild (node, NULL, "GNOME_DB_CONSTRAINT_PAIR", NULL);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_CONSTRAINT_FK_PAIR (list->data)->fkey));
			xmlSetProp (child, "field", str);
			g_free (str);
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_CONSTRAINT_FK_PAIR (list->data)->ref_pkey));
			xmlSetProp (child, "ref", str);
			g_free (str);
			list = g_slist_next (list);
		}

		xmlSetProp (node, "on_update", constraint_action_to_str (cstr->priv->on_update));
		xmlSetProp (node, "on_delete", constraint_action_to_str (cstr->priv->on_delete));
		break;
	case CONSTRAINT_NOT_NULL:
		child = xmlNewChild (node, NULL, "GNOME_DB_CONSTRAINT_FIELD", NULL);
		if (cstr->priv->single_field)
			str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (cstr->priv->single_field));
		xmlSetProp (child, "field", str);
		g_free (str);
		break;
	case CONSTRAINT_CHECK_EXPR:
		TO_IMPLEMENT;
		break;
	default:
		break;
	}

	return node;
}

static gboolean
gnome_db_constraint_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbConstraint *cstr;
	gchar *prop;
	gboolean type = FALSE, field_error = FALSE;
	GnomeDbTable *table = NULL;
	GnomeDbDatabase *db;
	gchar *ref_pb = NULL;
	xmlNodePtr subnode;
	GSList *contents, *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_CONSTRAINT (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);
	g_return_val_if_fail (!GNOME_DB_CONSTRAINT (iface)->priv->table, FALSE);
	db = g_object_get_data (G_OBJECT (iface), "db");
	g_return_val_if_fail (db, FALSE);

	cstr = GNOME_DB_CONSTRAINT (iface);
	if (strcmp (node->name, "GNOME_DB_CONSTRAINT")) {
		g_set_error (error,
			     GNOME_DB_CONSTRAINT_ERROR,
			     GNOME_DB_CONSTRAINT_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_CONSTRAINT>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (cstr), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "table");
	if (prop) {
		table = gnome_db_database_get_table_by_xml_id (db, prop);
		if (table) {
			g_object_set_data (G_OBJECT (iface), "db", NULL);
			g_signal_handlers_disconnect_by_func (G_OBJECT (db),
							      G_CALLBACK (nullified_object_cb), cstr);
			cstr->priv->table = table;
			gnome_db_base_connect_nullify (table, 
						 G_CALLBACK (nullified_object_cb), cstr);
			g_free (prop);
		}
		else 
			ref_pb = prop;
	}
	
	prop = xmlGetProp (node, "user_defined");
	if (prop) {
		cstr->priv->user_defined = (*prop && (*prop == 't')) ? TRUE : FALSE;
		g_free (prop);
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		cstr->priv->type = constraint_str_to_type (prop);
		g_free (prop);
		type = TRUE;
	}
	
	prop = xmlGetProp (node, "on_update");
	if (prop) {
		cstr->priv->on_update = constraint_str_to_action (prop);
		g_free (prop);
		type = TRUE;
	}

	prop = xmlGetProp (node, "on_delete");
	if (prop) {
		cstr->priv->on_delete = constraint_str_to_action (prop);
		g_free (prop);
		type = TRUE;
	}

	/* contents of the constraint if applicable */
	contents = NULL;
	subnode = node->children;
	while (subnode) {
		if (!strcmp (subnode->name, "GNOME_DB_CONSTRAINT_FIELD")) {
			GnomeDbTableField *field = NULL;
			prop = xmlGetProp (subnode, "field");
			if (prop) {
				field = gnome_db_database_get_field_by_xml_id (db, prop);
				g_free (prop);
			}
			if (field)
				contents = g_slist_append (contents, field);
			else {
				field_error = TRUE;
				ref_pb = xmlGetProp (subnode, "field");
			}
		}
		
		if (!strcmp (subnode->name, "GNOME_DB_CONSTRAINT_PAIR")) {
			GnomeDbTableField *field = NULL, *ref = NULL;
			prop = xmlGetProp (subnode, "field");
			if (prop) {
				field = gnome_db_database_get_field_by_xml_id (db, prop);
				g_free (prop);
			}
			prop = xmlGetProp (subnode, "ref");
			if (prop) {
				ref = gnome_db_database_get_field_by_xml_id (db, prop);
				g_free (prop);
			}
			if (field && ref) {
				GnomeDbConstraintFkeyPair *pair;

				pair = g_new0 (GnomeDbConstraintFkeyPair, 1);
				pair->fkey = field;
				pair->ref_pkey = ref;
				pair->ref_pkey_repl = NULL;
				contents = g_slist_append (contents, pair);
			}
			else {
				field_error = TRUE;
				if (!field)
					ref_pb = xmlGetProp (subnode, "field");
				else
					ref_pb = xmlGetProp (subnode, "ref");
			}
		}
		subnode = subnode->next;
	}

	switch (cstr->priv->type) {
	case CONSTRAINT_PRIMARY_KEY:
		gnome_db_constraint_pkey_set_fields (cstr, contents);
		g_slist_free (contents);
		break;
        case CONSTRAINT_FOREIGN_KEY:
		gnome_db_constraint_fkey_set_fields (cstr, contents);
		list = contents;
		while (list) {
			g_free (list->data);
			list = g_slist_next (list);
		}
		g_slist_free (contents);
		break;
        case CONSTRAINT_UNIQUE:
		gnome_db_constraint_unique_set_fields (cstr, contents);
		g_slist_free (contents);
		break;
        case CONSTRAINT_NOT_NULL:
		if (contents) {
			gnome_db_constraint_not_null_set_field (cstr, contents->data);
			g_slist_free (contents);
		}
		else
			field_error = TRUE;
		break;
        case CONSTRAINT_CHECK_EXPR:
		TO_IMPLEMENT;
		break;
	default:
		break;
	}

	if (type && table && !field_error)
		return TRUE;
	else {
		if (!type)
			g_set_error (error,
				     GNOME_DB_CONSTRAINT_ERROR,
				     GNOME_DB_CONSTRAINT_XML_LOAD_ERROR,
				     _("Missing required attributes for <GNOME_DB_CONSTRAINT>"));
		if (!table) {
			g_set_error (error,
				     GNOME_DB_CONSTRAINT_ERROR,
				     GNOME_DB_CONSTRAINT_XML_LOAD_ERROR,
				     _("Referenced table (%s) not found"), ref_pb);
			if (ref_pb)
				g_free (ref_pb);
		}
		if (field_error) {
			g_set_error (error,
				     GNOME_DB_CONSTRAINT_ERROR,
				     GNOME_DB_CONSTRAINT_XML_LOAD_ERROR,
				     _("Referenced field in constraint (%s) not found"), ref_pb);
			if (ref_pb)
				g_free (ref_pb);
		}
		return FALSE;
	}
}




/*
 * GnomeDbReferer interface implementation
 */

/*
 * gnome_db_constraint_activate
 *
 * Tries to convert any GnomeDbRefBase object to the pointed GnomeDbTableField instead
 */
static gboolean
gnome_db_constraint_activate (GnomeDbReferer *iface)
{
	gboolean activated = TRUE;
	GnomeDbConstraint *cstr;
	GnomeDbTable *ref_table = NULL;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), FALSE);
	cstr = GNOME_DB_CONSTRAINT (iface);
	g_return_val_if_fail (cstr->priv, FALSE);
	g_return_val_if_fail (cstr->priv->table, FALSE);

	if (gnome_db_constraint_is_active (GNOME_DB_REFERER (cstr)))
		return TRUE;

	if (cstr->priv->type == CONSTRAINT_FOREIGN_KEY) {
		GSList *list;
		GnomeDbConstraintFkeyPair *pair;
		GnomeDbBase *ref;

		list = cstr->priv->fk_pairs;
		while (list) {
			pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
			if (!pair->ref_pkey) {
				g_assert (pair->ref_pkey_repl);
				ref = gnome_db_ref_base_get_ref_object (pair->ref_pkey_repl);
				if (ref) {
					pair->ref_pkey = GNOME_DB_TABLE_FIELD (ref);
					g_object_unref (G_OBJECT (pair->ref_pkey_repl));
					pair->ref_pkey_repl = NULL;
					gnome_db_base_connect_nullify (pair->ref_pkey,
								 G_CALLBACK (nullified_object_cb), cstr);

					if (!ref_table)
						ref_table = GNOME_DB_TABLE (gnome_db_field_get_entity (GNOME_DB_FIELD (pair->ref_pkey)));
					else
						if (gnome_db_field_get_entity (GNOME_DB_FIELD (pair->ref_pkey)) != 
						    GNOME_DB_ENTITY (ref_table)) {
							g_warning ("Referenced table is not the same for all pairs");
							return FALSE;
						}
				}
			}
			
			if (!pair->ref_pkey)
				activated = FALSE;
			
			list = g_slist_next (list);
		}

		if (ref_table != cstr->priv->ref_table) {
			if (cstr->priv->ref_table)
				g_signal_handlers_disconnect_by_func (G_OBJECT (cstr->priv->ref_table),
								      G_CALLBACK (nullified_object_cb), cstr);
			cstr->priv->ref_table = ref_table;
			if (ref_table)
				gnome_db_base_connect_nullify (ref_table,
							 G_CALLBACK (nullified_object_cb), cstr);
		}
	}
	
	return activated;
}

static void
gnome_db_constraint_deactivate (GnomeDbReferer *iface)
{
	g_return_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface));
	/* do nothing */
}

/*
 * gnome_db_constraint_is_active
 *
 * Tells wether the GnomeDbConstraint object has found all the
 * GnomeDbTableField objects it needs
 */
static gboolean
gnome_db_constraint_is_active (GnomeDbReferer *iface)
{
	gboolean activated = TRUE;
	GnomeDbConstraint *cstr;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), FALSE);
	cstr = GNOME_DB_CONSTRAINT (iface);
	g_return_val_if_fail (cstr->priv, FALSE);
	g_return_val_if_fail (cstr->priv->table, FALSE);

	if (cstr->priv->type == CONSTRAINT_FOREIGN_KEY) {
		GSList *list = cstr->priv->fk_pairs;
		while (list && activated) {
			GnomeDbConstraintFkeyPair *pair = GNOME_DB_CONSTRAINT_FK_PAIR (list->data);
			if (pair->ref_pkey_repl)
				activated = FALSE;
			list = g_slist_next (list);
		}

		if (!cstr->priv->ref_table)
			activated = FALSE;
	}

	return activated;
}

static GSList *
gnome_db_constraint_get_ref_objects (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface), NULL);
	/* do nothing */
	return NULL;
}

static void gnome_db_constraint_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	g_return_if_fail (iface && IS_GNOME_DB_CONSTRAINT (iface));
	/* do nothing */
}
