/* gnome-db-server-aggregate.c
 *
 * Copyright (C) 2003 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-server.h"
#include "gnome-db-server-aggregate.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-data-handler.h"
#include "marshal.h"
#include <libgda/libgda.h>
#include <string.h>
#include "utility.h"

/* 
 * Main static functions 
 */
static void gnome_db_server_aggregate_class_init (GnomeDbServerAggregateClass * class);
static void gnome_db_server_aggregate_init (GnomeDbServerAggregate * srv);
static void gnome_db_server_aggregate_dispose (GObject   * object);
static void gnome_db_server_aggregate_finalize (GObject   * object);

static void gnome_db_server_aggregate_set_property (GObject              *object,
					     guint                 param_id,
					     const GValue         *value,
					     GParamSpec           *pspec);
static void gnome_db_server_aggregate_get_property (GObject              *object,
					     guint                 param_id,
					     GValue               *value,
					     GParamSpec           *pspec);

#ifdef debug
static void gnome_db_server_aggregate_dump (GnomeDbServerAggregate *agg, guint offset);
#endif

static void        gnome_db_aggregate_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_aggregate_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_aggregate_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_aggregate_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_server_aggregate_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbServerAggregatePrivate
{
	GnomeDbServer         *srv;
	gchar                 *objectid;       /* unique id for the aggregate */
	GnomeDbServerDataType *result_type;
	GnomeDbServerDataType *arg_type;
};


/* module error */
GQuark gnome_db_server_aggregate_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_server_aggregate_error");
	return quark;
}


GType
gnome_db_server_aggregate_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbServerAggregateClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_server_aggregate_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbServerAggregate),
			0,
			(GInstanceInitFunc) gnome_db_server_aggregate_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_aggregate_xml_storage_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbServerAggregate", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
	}
	return type;
}

static void 
gnome_db_aggregate_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_aggregate_get_xml_id;
	iface->save_to_xml = gnome_db_aggregate_save_to_xml;
	iface->load_from_xml = gnome_db_aggregate_load_from_xml;
}


static void
gnome_db_server_aggregate_class_init (GnomeDbServerAggregateClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_server_aggregate_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbServerAggregateClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_server_aggregate_dispose;
	object_class->finalize = gnome_db_server_aggregate_finalize;

	/* Properties */
	object_class->set_property = gnome_db_server_aggregate_set_property;
	object_class->get_property = gnome_db_server_aggregate_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_server_aggregate_dump;
#endif

}

static void
gnome_db_server_aggregate_init (GnomeDbServerAggregate * gnome_db_server_aggregate)
{
	gnome_db_server_aggregate->priv = g_new0 (GnomeDbServerAggregatePrivate, 1);
	gnome_db_server_aggregate->priv->srv = NULL;
	gnome_db_server_aggregate->priv->objectid = NULL;
	gnome_db_server_aggregate->priv->result_type = NULL;
	gnome_db_server_aggregate->priv->arg_type = NULL;
}


/**
 * gnome_db_server_aggregate_new
 * @srv: a #GnomeDbServer object
 *
 * Creates a new GnomeDbServerAggregate object
 *
 * Returns: the new object
 */
GObject*
gnome_db_server_aggregate_new (GnomeDbServer *srv)
{
	GObject   *obj;
	GnomeDbServerAggregate *gnome_db_server_aggregate;

	g_return_val_if_fail (srv && GNOME_DB_SERVER (srv), NULL);

	obj = g_object_new (GNOME_DB_TYPE_SERVER_AGGREGATE, "dict",
			    gnome_db_server_get_dict (srv), NULL);
	gnome_db_server_aggregate = GNOME_DB_SERVER_AGGREGATE (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_server_aggregate), 0);
	g_object_add_weak_pointer (G_OBJECT (srv), (gpointer) &(gnome_db_server_aggregate->priv->srv));
	gnome_db_server_aggregate->priv->srv = srv;

	return obj;
}


static void
gnome_db_server_aggregate_dispose (GObject *object)
{
	GnomeDbServerAggregate *gnome_db_server_aggregate;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_AGGREGATE (object));

	gnome_db_server_aggregate = GNOME_DB_SERVER_AGGREGATE (object);
	if (gnome_db_server_aggregate->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		g_object_remove_weak_pointer (G_OBJECT (gnome_db_server_aggregate->priv->srv), 
					      (gpointer) &(gnome_db_server_aggregate->priv->srv));

		gnome_db_server_aggregate_set_ret_type (gnome_db_server_aggregate, NULL);
		gnome_db_server_aggregate_set_arg_type (gnome_db_server_aggregate, NULL);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_server_aggregate_finalize (GObject   * object)
{
	GnomeDbServerAggregate *gnome_db_server_aggregate;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER_AGGREGATE (object));

	gnome_db_server_aggregate = GNOME_DB_SERVER_AGGREGATE (object);
	if (gnome_db_server_aggregate->priv) {

		g_free (gnome_db_server_aggregate->priv);
		gnome_db_server_aggregate->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_server_aggregate_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbServerAggregate *gnome_db_server_aggregate;

	gnome_db_server_aggregate = GNOME_DB_SERVER_AGGREGATE (object);
	if (gnome_db_server_aggregate->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}

static void
gnome_db_server_aggregate_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbServerAggregate *gnome_db_server_aggregate;
	gnome_db_server_aggregate = GNOME_DB_SERVER_AGGREGATE (object);
	
	if (gnome_db_server_aggregate->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			g_value_set_pointer (value, NULL);
			break;
		}	
	}
}

#ifdef debug
static void
gnome_db_server_aggregate_dump (GnomeDbServerAggregate *agg, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg));
	g_return_if_fail (agg->priv);
	
	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

	/* return type */
	g_print ("%s" D_COL_H1 "GnomeDbServerAggregate" D_COL_NOR " %s (%p)\n",
		 str, gnome_db_base_get_name (GNOME_DB_BASE (agg)), agg);
	if (agg->priv->result_type)
		g_print ("%sReturn data type: %s\n", str, gnome_db_server_data_type_get_sqlname (agg->priv->result_type));
	else
		g_print ("%s" D_COL_ERR "No return type defined" D_COL_NOR "\n", str);

	if (agg->priv->arg_type)
		g_print ("%sParameter data type: %s\n", str, gnome_db_server_data_type_get_sqlname (agg->priv->arg_type));
	else
		g_print ("%sAny argument type allowed\n", str);

	g_free (str);
}
#endif



/* GnomeDbXmlStorage interface implementation */
static gchar *
gnome_db_aggregate_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_AGGREGATE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_AGGREGATE (iface)->priv, NULL);

	return g_strconcat ("AG", GNOME_DB_SERVER_AGGREGATE (iface)->priv->objectid, NULL);
}

static xmlNodePtr
gnome_db_aggregate_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL, subnode;
	GnomeDbServerAggregate *agg;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_AGGREGATE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER_AGGREGATE (iface)->priv, NULL);

	agg = GNOME_DB_SERVER_AGGREGATE (iface);

	node = xmlNewNode (NULL, "GNOME_DB_AGGREGATE");

	str = gnome_db_aggregate_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (agg)));
	xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (agg)));
	xmlSetProp (node, "owner", gnome_db_base_get_owner (GNOME_DB_BASE (agg)));

	/* return type */
	if (agg->priv->result_type) {
		subnode = xmlNewChild (node, NULL, "GNOME_DB_FUNC_PARAM", NULL);
		
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (agg->priv->result_type));
		xmlSetProp (subnode, "type", str);
		g_free (str);
		xmlSetProp (subnode, "way", "out");
	}

	/* argument type */
	if (agg->priv->arg_type) {
		subnode = xmlNewChild (node, NULL, "GNOME_DB_FUNC_PARAM", NULL);
		
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (agg->priv->arg_type));
		xmlSetProp (subnode, "type", str);
		g_free (str);
		xmlSetProp (subnode, "way", "in");
	}

	return node;
}

static gboolean
gnome_db_aggregate_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbServerAggregate *agg;
	gchar *prop;
	gboolean pname = FALSE, pid = FALSE;
	xmlNodePtr subnode;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER_AGGREGATE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_SERVER_AGGREGATE (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	agg = GNOME_DB_SERVER_AGGREGATE (iface);
	if (strcmp (node->name, "GNOME_DB_AGGREGATE")) {
		g_set_error (error,
			     GNOME_DB_SERVER_AGGREGATE_ERROR,
			     GNOME_DB_SERVER_AGGREGATE_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_AGGREGATE>"));
		return FALSE;
	}

	/* aggregate's attributes */
	prop = xmlGetProp (node, "id");
	if (prop) {
		if ((*prop == 'A') && (*(prop+1)=='G')) {
			pid = TRUE;
			if (agg->priv->objectid)
				g_free (agg->priv->objectid);
			agg->priv->objectid = g_strdup (prop+2);
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		pname = TRUE;
		gnome_db_base_set_name (GNOME_DB_BASE (agg), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (agg), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "owner");
	if (prop) {
		gnome_db_base_set_owner (GNOME_DB_BASE (agg), prop);
		g_free (prop);
	}
	
	/* arguments and return type */
	subnode = node->children;
	while (subnode) {
		if (!strcmp (subnode->name, "GNOME_DB_FUNC_PARAM")) {
			GnomeDbServerDataType *dt = NULL;
			prop = xmlGetProp (subnode, "type");
			if (prop) {
				dt = gnome_db_server_get_data_type_by_xml_id (agg->priv->srv, prop);
				g_free (prop);
			}

			if (!dt) {
				g_set_error (error,
					     GNOME_DB_SERVER_AGGREGATE_ERROR,
					     GNOME_DB_SERVER_AGGREGATE_XML_LOAD_ERROR,
					     _("Can't find data type for aggregate '%s'"), 
					     gnome_db_base_get_name (GNOME_DB_BASE (agg)));
				return FALSE;
			}
			
			prop = xmlGetProp (subnode, "way");
			if (prop) {
				if (*prop == 'o') {
					if (agg->priv->result_type) {
						g_set_error (error,
							     GNOME_DB_SERVER_AGGREGATE_ERROR,
							     GNOME_DB_SERVER_AGGREGATE_XML_LOAD_ERROR,
							     _("More than one return type for aggregate '%s'"), 
							     gnome_db_base_get_name (GNOME_DB_BASE (agg)));
						return FALSE;
					}
					gnome_db_server_aggregate_set_ret_type (agg, dt);
				}
				else {
					if (agg->priv->arg_type) {
						g_set_error (error,
							     GNOME_DB_SERVER_AGGREGATE_ERROR,
							     GNOME_DB_SERVER_AGGREGATE_XML_LOAD_ERROR,
							     _("More than one argument type for aggregate '%s'"), 
							     gnome_db_base_get_name (GNOME_DB_BASE (agg)));
						return FALSE;
					}
					gnome_db_server_aggregate_set_arg_type (agg, dt);
				}
				g_free (prop);
			}
		}
		subnode = subnode->next;
	}

	if (pname && pid)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_SERVER_AGGREGATE_ERROR,
			     GNOME_DB_SERVER_AGGREGATE_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_AGGREGATE>"));
		return FALSE;
	}
}




/**
 * gnome_db_server_aggregate_set_dbms_id
 * @agg: a #GnomeDbServerAggregate object
 * @id: the DBMS identifier
 *
 * Set the DBMS identifier of the aggregate
 */
void
gnome_db_server_aggregate_set_dbms_id (GnomeDbServerAggregate *agg, const gchar *id)
{
	g_return_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg));
	g_return_if_fail (agg->priv);
	g_return_if_fail (id && *id);

	if (agg->priv->objectid)
		g_free (agg->priv->objectid);

	agg->priv->objectid = utility_build_encoded_id (NULL, id);
}


/**
 * gnome_db_server_aggregate_get_dbms_id
 * @agg: a #GnomeDbServerAggregate object
 *
 * Get the DBMS identifier of the aggregate
 *
 * Returns: the aggregate's id
 */
gchar *
gnome_db_server_aggregate_get_dbms_id (GnomeDbServerAggregate *agg)
{
	g_return_val_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	return utility_build_decoded_id (NULL, agg->priv->objectid);
}


/**
 * gnome_db_server_aggregate_set_sqlname
 * @agg: a #GnomeDbServerAggregate object
 * @sqlname: 
 *
 * Set the SQL name of the data type.
 */
void
gnome_db_server_aggregate_set_sqlname (GnomeDbServerAggregate *agg, const gchar *sqlname)
{
	g_return_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg));
	g_return_if_fail (agg->priv);

	gnome_db_base_set_name (GNOME_DB_BASE (agg), sqlname);
}


/**
 * gnome_db_server_aggregate_get_sqlname
 * @agg: a #GnomeDbServerAggregate object
 *
 * Get the DBMS's name of a data type.
 *
 * Returns: the name of the data type
 */
const gchar *
gnome_db_server_aggregate_get_sqlname (GnomeDbServerAggregate *agg)
{
	g_return_val_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	return gnome_db_base_get_name (GNOME_DB_BASE (agg));
}

static void nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServerAggregate *agg);

/**
 * gnome_db_server_aggregate_set_arg_type
 * @agg: a #GnomeDbServerAggregate object
 * @dt: a #GnomeDbServerDataType objects or #NULL value to represent the data type
 * of the aggregate's unique argument .
 *
 * Set the argument type of a aggregate
 */
void 
gnome_db_server_aggregate_set_arg_type (GnomeDbServerAggregate *agg, GnomeDbServerDataType *dt)
{
	g_return_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg));
	g_return_if_fail (agg->priv);
	if (dt)
		g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	
	if (agg->priv->arg_type) { 
		g_signal_handlers_disconnect_by_func (G_OBJECT (agg->priv->arg_type), 
						      G_CALLBACK (nullified_data_type_cb), agg);
		g_object_unref (G_OBJECT (agg->priv->arg_type));
	}

	agg->priv->arg_type = dt;
	if (dt) {
		gnome_db_base_connect_nullify (dt, G_CALLBACK (nullified_data_type_cb), agg);
		g_object_ref (G_OBJECT (dt));
	}
}

/**
 * gnome_db_server_aggregate_get_arg_type
 * @agg: a #GnomeDbServerAggregate object
 * 
 * To consult the list of arguments types (and number) of a aggregate.
 *
 * Returns: a list of #GnomeDbServerDataType objects, the list MUST NOT be modified.
 */
GnomeDbServerDataType *
gnome_db_server_aggregate_get_arg_type (GnomeDbServerAggregate *agg)
{
	g_return_val_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	return agg->priv->arg_type;
}

/**
 * gnome_db_server_aggregate_set_ret_type
 * @agg: a #GnomeDbServerAggregate object
 * @dt: a #GnomeDbServerDataType object or #NULL
 *
 * Set the return type of a aggregate
 */
void 
gnome_db_server_aggregate_set_ret_type  (GnomeDbServerAggregate *agg, GnomeDbServerDataType *dt)
{
	g_return_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg));
	g_return_if_fail (agg->priv);
	if (dt)
		g_return_if_fail (dt && IS_GNOME_DB_SERVER_DATA_TYPE (dt));
	
	if (agg->priv->result_type) { 
		g_signal_handlers_disconnect_by_func (G_OBJECT (agg->priv->result_type), 
						      G_CALLBACK (nullified_data_type_cb), agg);
		g_object_unref (G_OBJECT (agg->priv->result_type));
	}

	agg->priv->result_type = dt;
	if (dt) {
		gnome_db_base_connect_nullify (dt, G_CALLBACK (nullified_data_type_cb), agg);
		g_object_ref (G_OBJECT (dt));
	}
}

static void
nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServerAggregate *agg)
{
	gnome_db_base_nullify (GNOME_DB_BASE (agg));
}

/**
 * gnome_db_server_aggregate_get_ret_type
 * @agg: a #GnomeDbServerAggregate object
 * 
 * To consult the return type of a aggregate.
 *
 * Returns: a #GnomeDbServerDataType object.
 */
GnomeDbServerDataType *
gnome_db_server_aggregate_get_ret_type  (GnomeDbServerAggregate *agg)
{
	g_return_val_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	return agg->priv->result_type;
}
