/* gnome-db-qf-agg.c
 *
 * Copyright (C) 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-qf-agg.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-ref-base.h"
#include "marshal.h"
#include "gnome-db-query.h"
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_qf_agg_class_init (GnomeDbQfAggClass * class);
static void gnome_db_qf_agg_init (GnomeDbQfAgg *qf);
static void gnome_db_qf_agg_dispose (GObject *object);
static void gnome_db_qf_agg_finalize (GObject *object);

static void gnome_db_qf_agg_set_property (GObject              *object,
					  guint                 param_id,
					  const GValue         *value,
					  GParamSpec           *pspec);
static void gnome_db_qf_agg_get_property (GObject              *object,
					  guint                 param_id,
					  GValue               *value,
					  GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_qf_agg_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_qf_agg_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_qf_agg_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_qf_agg_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void              gnome_db_qf_agg_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_qf_agg_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_qf_agg_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_qf_agg_renderer_init   (GnomeDbRendererIface *iface);
static gchar          *gnome_db_qf_agg_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_qf_agg_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_qf_agg_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_qf_agg_activate            (GnomeDbReferer *iface);
static void        gnome_db_qf_agg_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_qf_agg_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_qf_agg_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_qf_agg_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* virtual functions */
static GObject    *gnome_db_qf_agg_copy           (GnomeDbQfield *orig);
static gboolean    gnome_db_qf_agg_is_equal       (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2);


#ifdef debug
static void        gnome_db_qf_agg_dump           (GnomeDbQfAgg *agg, guint offset);
#endif

/* When the GnomeDbQuery or GnomeDbTarget is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbQfAgg *agg);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_qf_agg_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY,
};


/* private structure */
struct _GnomeDbQfAggPrivate
{
	GnomeDbQuery    *query;
	GnomeDbRefBase  *agg_ref;  /* references a GnomeDbServerAggregate */
	GnomeDbRefBase  *arg;
};


/* module error */
GQuark gnome_db_qf_agg_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qf_agg_error");
	return quark;
}


GType
gnome_db_qf_agg_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfAggClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qf_agg_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfAgg),
			0,
			(GInstanceInitFunc) gnome_db_qf_agg_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_qf_agg_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_qf_agg_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_qf_agg_renderer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_qf_agg_referer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_TYPE_QFIELD, "GnomeDbQfAgg", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_FIELD, &field_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_RENDERER, &renderer_info);
		g_type_add_interface_static (type, GNOME_DB_TYPE_REFERER, &referer_info);
	}
	return type;
}

static void 
gnome_db_qf_agg_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_qf_agg_get_xml_id;
	iface->save_to_xml = gnome_db_qf_agg_save_to_xml;
	iface->load_from_xml = gnome_db_qf_agg_load_from_xml;
}

static void
gnome_db_qf_agg_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_qf_agg_get_entity;
	iface->get_data_type = gnome_db_qf_agg_get_data_type;
}

static void
gnome_db_qf_agg_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_sql = gnome_db_qf_agg_render_as_sql;
	iface->render_as_str = gnome_db_qf_agg_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_qf_agg_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_qf_agg_activate;
        iface->deactivate = gnome_db_qf_agg_deactivate;
        iface->is_active = gnome_db_qf_agg_is_active;
        iface->get_ref_objects = gnome_db_qf_agg_get_ref_objects;
        iface->replace_refs = gnome_db_qf_agg_replace_refs;
}

static void
gnome_db_qf_agg_class_init (GnomeDbQfAggClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_qf_agg_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQfAggClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_qf_agg_dispose;
	object_class->finalize = gnome_db_qf_agg_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qf_agg_set_property;
	object_class->get_property = gnome_db_qf_agg_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_qf_agg_dump;
#endif
	GNOME_DB_QFIELD_CLASS (class)->copy = gnome_db_qf_agg_copy;
	GNOME_DB_QFIELD_CLASS (class)->is_equal = gnome_db_qf_agg_is_equal;
	GNOME_DB_QFIELD_CLASS (class)->is_list = NULL;
	GNOME_DB_QFIELD_CLASS (class)->get_params = NULL;
}

static void
gnome_db_qf_agg_init (GnomeDbQfAgg *gnome_db_qf_agg)
{
	gnome_db_qf_agg->priv = g_new0 (GnomeDbQfAggPrivate, 1);
	gnome_db_qf_agg->priv->query = NULL;
	gnome_db_qf_agg->priv->agg_ref = NULL;
	gnome_db_qf_agg->priv->arg = NULL;
}


/**
 * gnome_db_qf_agg_new_with_agg
 * @query: a #GnomeDbQuery in which the new object will be
 * @agg: a #GnomeDbServerAggregate object
 *
 * Creates a new GnomeDbQfAgg object which represents the @agg aggregate
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_agg_new_with_agg (GnomeDbQuery *query, GnomeDbServerAggregate *agg)
{
	GObject   *obj;
	GnomeDbQfAgg *gnome_db_qf_agg;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (agg && IS_GNOME_DB_SERVER_AGGREGATE (agg), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_TYPE_QF_AGG, "dict", dict, NULL);
	gnome_db_qf_agg = GNOME_DB_QF_AGG (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_agg), id);

	gnome_db_qf_agg->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_agg);

	gnome_db_qf_agg->priv->agg_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_qf_agg->priv->agg_ref, GNOME_DB_BASE (agg));
	
	return obj;
}


/**
 * gnome_db_qf_agg_new_with_xml_id
 * @query: a #GnomeDbQuery in which the new object will be
 * @agg_xml_id: the XML Id of a #GnomeDbServerAggregate object
 *
 * Creates a new GnomeDbQfAgg object which represents a given aggregate
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_agg_new_with_xml_id (GnomeDbQuery *query, const gchar *agg_xml_id)
{
	GObject   *obj;
	GnomeDbQfAgg *gnome_db_qf_agg;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (agg_xml_id && *agg_xml_id, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_TYPE_QF_AGG, "dict", dict, NULL);
	gnome_db_qf_agg = GNOME_DB_QF_AGG (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_agg), id);

	gnome_db_qf_agg->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_agg);

	gnome_db_qf_agg->priv->agg_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_qf_agg->priv->agg_ref, GNOME_DB_TYPE_SERVER_AGGREGATE, 
				  REFERENCE_BY_XML_ID, agg_xml_id);
	
	return obj;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbQfAgg *agg)
{
	gnome_db_base_nullify (GNOME_DB_BASE (agg));
}

static void
gnome_db_qf_agg_dispose (GObject *object)
{
	GnomeDbQfAgg *gnome_db_qf_agg;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_AGG (object));

	gnome_db_qf_agg = GNOME_DB_QF_AGG (object);
	if (gnome_db_qf_agg->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_qf_agg->priv->arg) {
			g_object_unref (G_OBJECT (gnome_db_qf_agg->priv->arg));
			gnome_db_qf_agg->priv->arg = NULL;
		}

		if (gnome_db_qf_agg->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_agg->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_agg);
			gnome_db_qf_agg->priv->query = NULL;
		}

		if (gnome_db_qf_agg->priv->agg_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_agg->priv->agg_ref));
			gnome_db_qf_agg->priv->agg_ref = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qf_agg_finalize (GObject   * object)
{
	GnomeDbQfAgg *gnome_db_qf_agg;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_AGG (object));

	gnome_db_qf_agg = GNOME_DB_QF_AGG (object);
	if (gnome_db_qf_agg->priv) {
		g_free (gnome_db_qf_agg->priv);
		gnome_db_qf_agg->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qf_agg_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbQfAgg *gnome_db_qf_agg;
	gpointer ptr;

	gnome_db_qf_agg = GNOME_DB_QF_AGG (object);
	if (gnome_db_qf_agg->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

			if (gnome_db_qf_agg->priv->query) {
				if (gnome_db_qf_agg->priv->query == GNOME_DB_QUERY (ptr))
					return;

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_agg->priv->query),
								      G_CALLBACK (nullified_object_cb), gnome_db_qf_agg);
			}

			gnome_db_qf_agg->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr,
						 G_CALLBACK (nullified_object_cb), gnome_db_qf_agg);
			break;
		}
	}
}

static void
gnome_db_qf_agg_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfAgg *gnome_db_qf_agg;
	gnome_db_qf_agg = GNOME_DB_QF_AGG (object);
	
	if (gnome_db_qf_agg->priv) {
		switch (param_id) {
		case PROP_QUERY:
			g_value_set_pointer (value, gnome_db_qf_agg->priv->query);
			break;
		}	
	}
}

static GObject *
gnome_db_qf_agg_copy (GnomeDbQfield *orig)
{
	GnomeDbQfAgg *qf;
	GObject *obj;
	GnomeDbDict *dict;

	g_assert (IS_GNOME_DB_QF_AGG (orig));
	qf = GNOME_DB_QF_AGG (orig);

	obj = gnome_db_qf_agg_new_with_xml_id (qf->priv->query, 
					  gnome_db_ref_base_get_ref_name (qf->priv->agg_ref, NULL, NULL));
	if (gnome_db_base_get_name (GNOME_DB_BASE (orig)))
		gnome_db_base_set_name (GNOME_DB_BASE (obj), gnome_db_base_get_name (GNOME_DB_BASE (orig)));

	if (gnome_db_base_get_description (GNOME_DB_BASE (orig)))
		gnome_db_base_set_description (GNOME_DB_BASE (obj), gnome_db_base_get_description (GNOME_DB_BASE (orig)));


	/* argument */
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	if (qf->priv->arg) {
		GnomeDbRefBase *ref;
		const gchar *refname;
		GType type;

		refname = gnome_db_ref_base_get_ref_name (qf->priv->arg, &type, NULL);
		ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
		gnome_db_ref_base_set_ref_name (ref, type, REFERENCE_BY_XML_ID, refname);
		GNOME_DB_QF_AGG (obj)->priv->arg = ref;
	}

	return obj;
}

static gboolean
gnome_db_qf_agg_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	const gchar *ref1, *ref2;
	gboolean retval;
	g_assert (IS_GNOME_DB_QF_AGG (qfield1));
	g_assert (IS_GNOME_DB_QF_AGG (qfield2));
	
	/* it is here assumed that qfield1 and qfield2 are of the same type and refer to the same
	   query */
	ref1 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_AGG (qfield1)->priv->agg_ref, NULL, NULL);
	ref2 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_AGG (qfield2)->priv->agg_ref, NULL, NULL);

	retval = !strcmp (ref1, ref2) ? TRUE : FALSE;
	if (retval) {
		TO_IMPLEMENT; /* arguments */
	}

	return retval;
}

/**
 * gnome_db_qf_agg_get_ref_agg
 * @agg: a #GnomeDbQfAgg object
 *
 * Get the real #GnomeDbServerAggregate object used by @agg
 *
 * Returns: the #GnomeDbServerAggregate object, or NULL if @agg is not active
 */
GnomeDbServerAggregate *
gnome_db_qf_agg_get_ref_agg (GnomeDbQfAgg *agg)
{
	GnomeDbBase *base;
	g_return_val_if_fail (agg && IS_GNOME_DB_QF_AGG (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (agg->priv->agg_ref);
	if (base)
		return GNOME_DB_SERVER_AGGREGATE (base);
	else
		return NULL;
}

/**
 * gnome_db_qf_agg_set_arg
 * @agg: a #GnomeDbQfAgg object
 * @arg: a #GnomeDbQfield object
 *
 * Sets the argument of @agg. If @arg is %NULL, then
 * the argument (if there was one) is removed.
 *
 * If @agg is not active, then no check on the provided arg
 * is performed.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_qf_agg_set_arg (GnomeDbQfAgg *agg, GnomeDbQfield *arg)
{
	gboolean arg_ok = TRUE;
	g_return_val_if_fail (agg && IS_GNOME_DB_QF_AGG (agg), FALSE);
	g_return_val_if_fail (agg->priv, FALSE);

	if (arg && gnome_db_ref_base_activate (agg->priv->agg_ref)) {
		/* test for the right argument */
		GnomeDbServerInfo *sinfo = gnome_db_server_get_server_info (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (agg))));
		GnomeDbServerAggregate *sagg = GNOME_DB_SERVER_AGGREGATE (gnome_db_ref_base_get_ref_object (agg->priv->agg_ref));
		GnomeDbServerDataType *argtype = gnome_db_server_aggregate_get_arg_type (sagg);

		if (GNOME_DB_FUNCTIONS_TEST_PARAMS_DO_TEST && arg && argtype) {
			if (!sinfo || !sinfo->implicit_data_types_casts) {
				/* Strict test */
				if (gnome_db_qfield_get_data_type (arg) != argtype)
					arg_ok = FALSE;
			}
			else {
				/* GdaValueType compatibility test */
				if (gnome_db_server_data_type_get_gda_type (gnome_db_qfield_get_data_type (GNOME_DB_QFIELD (arg))) !=
				    gnome_db_server_data_type_get_gda_type (argtype))
					arg_ok = FALSE;
			}
		}
	}

	if (!arg || arg_ok) {
		/* Delete any pre-existing argument */
		if (agg->priv->arg) {
			g_object_unref (G_OBJECT (agg->priv->arg));
			agg->priv->arg = NULL;
		}
	}

	if (arg && arg_ok) {
		/* add the new argument */
		GnomeDbRefBase *ref;
				
		ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (agg))));
		gnome_db_ref_base_set_ref_object (ref, GNOME_DB_BASE (arg));
		agg->priv->arg = ref;
	}

	if (arg_ok)
		gnome_db_referer_activate (GNOME_DB_REFERER (agg));

	return arg_ok;
}

/**
 * gnome_db_qf_agg_get_arg
 * @agg: a #GnomeDbQfAgg object
 *
 * Get a list of the other #GnomeDbQfield objects which are arguments of @agg. If some
 * of them are missing, then a %NULL is inserted where it should have been.
 *
 * Returns: the #GnomeDbQField argument
 */
GnomeDbQfield *
gnome_db_qf_agg_get_args (GnomeDbQfAgg *agg)
{
	g_return_val_if_fail (agg && IS_GNOME_DB_QF_AGG (agg), NULL);
	g_return_val_if_fail (agg->priv, NULL);

	if (agg->priv->arg)
		return (GnomeDbQfield *) gnome_db_ref_base_get_ref_object (agg->priv->arg);
	else
		return NULL;
}

#ifdef debug
static void
gnome_db_qf_agg_dump (GnomeDbQfAgg *agg, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (agg && IS_GNOME_DB_QF_AGG (agg));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (agg->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbQfAgg" D_COL_NOR " \"%s\" (%p, id=%d) ",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (agg)), agg, gnome_db_base_get_id (GNOME_DB_BASE (agg)));
		if (gnome_db_qf_agg_is_active (GNOME_DB_REFERER (agg)))
			g_print ("Active");
		else
			g_print (D_COL_ERR "Inactive" D_COL_NOR);
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (agg)))
			g_print (", Visible\n");
		g_print ("\n");
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, agg);
}
#endif


/* 
 * GnomeDbEntity interface implementation
 */
static GnomeDbEntity *
gnome_db_qf_agg_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_QF_AGG (iface)->priv->query);
}

static GnomeDbServerDataType *
gnome_db_qf_agg_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);
	
	if (gnome_db_qf_agg_activate (GNOME_DB_REFERER (iface))) {
		GnomeDbServerAggregate *agg;
		agg = GNOME_DB_SERVER_AGGREGATE (gnome_db_ref_base_get_ref_object (GNOME_DB_QF_AGG (iface)->priv->agg_ref));
		return gnome_db_server_aggregate_get_ret_type (agg);
	}

	return NULL;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_qf_agg_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *q_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);

	q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_QF_AGG (iface)->priv->query));
	xml_id = g_strdup_printf ("%s:QF%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
	g_free (q_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_qf_agg_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbQfAgg *agg;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);

	agg = GNOME_DB_QF_AGG (iface);

	node = xmlNewNode (NULL, "GNOME_DB_QF");
	
	str = gnome_db_qf_agg_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);

	xmlSetProp (node, "type", "AGG");
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (agg)));
	if (gnome_db_base_get_description (GNOME_DB_BASE (agg)) && *gnome_db_base_get_description (GNOME_DB_BASE (agg)))
		xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (agg)));
	xmlSetProp (node, "object", gnome_db_ref_base_get_ref_name (agg->priv->agg_ref, NULL, NULL));
	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (agg)))
		xmlSetProp (node, "is_visible",  "f");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (agg)))
		xmlSetProp (node, "is_internal", "t");

	str = gnome_db_qfield_get_alias (GNOME_DB_QFIELD (agg));
	if (str && *str) 
		xmlSetProp (node, "alias", str);

	/* aggregate's argument */
	if (agg->priv->arg) {
		xmlNodePtr argnode;
		
		argnode = xmlNewChild (node, NULL, "GNOME_DB_QF_REF", NULL);
		xmlSetProp (argnode, "object", gnome_db_ref_base_get_ref_name (agg->priv->arg, NULL, NULL));
	}

	return node;
}

static gboolean
gnome_db_qf_agg_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQfAgg *agg;
	gchar *prop;
	gboolean aggref = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	agg = GNOME_DB_QF_AGG (iface);
	if (strcmp (node->name, "GNOME_DB_QF")) {
		g_set_error (error,
			     GNOME_DB_QF_AGG_ERROR,
			     GNOME_DB_QF_AGG_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QF>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		if (strcmp (prop, "AGG")) {
			g_set_error (error,
				     GNOME_DB_QF_AGG_ERROR,
				     GNOME_DB_QF_AGG_XML_LOAD_ERROR,
				     _("Wrong type of aggregate in <GNOME_DB_QF>"));
			return FALSE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "id");
	if (prop) {
		gchar *ptr, *tok;
		ptr = strtok_r (prop, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (strlen (ptr) < 3) {
			g_set_error (error,
				     GNOME_DB_QF_AGG_ERROR,
				     GNOME_DB_QF_AGG_XML_LOAD_ERROR,
				     _("Wrong 'id' attribute in <GNOME_DB_QF>"));
			return FALSE;
		}
		gnome_db_base_set_id (GNOME_DB_BASE (agg), atoi (ptr+2));
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (agg), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (agg), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "object");
	if (prop) {
		aggref = TRUE;
		gnome_db_ref_base_set_ref_name (agg->priv->agg_ref, GNOME_DB_TYPE_SERVER_AGGREGATE, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_visible");
	if (prop) {
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (agg), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_internal");
	if (prop) {
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (agg), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "alias");
	if (prop) {
		gnome_db_qfield_set_alias (GNOME_DB_QFIELD (agg), prop);
		g_free (prop);
	}
	
	/* aggregate's argument */
	if (node->children) {
		GnomeDbDict *dict;
		xmlNodePtr argnode = node->children;
		dict = gnome_db_base_get_dict (GNOME_DB_BASE (agg));
		while (argnode) {
			if (!strcmp (argnode->name, "GNOME_DB_QF_REF")) {
				if (! agg->priv->arg) {
					prop = xmlGetProp (argnode, "object");
					if (prop) {
						GnomeDbRefBase *ref;
						
						ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
						gnome_db_ref_base_set_ref_name (ref, GNOME_DB_TYPE_FIELD, REFERENCE_BY_XML_ID, prop);
						g_free (prop);
						agg->priv->arg = ref;
					}
				}
				else {
					GnomeDbBase *ref;
		
					ref = gnome_db_ref_base_get_ref_object (agg->priv->agg_ref);
					g_set_error (error,
						     GNOME_DB_QF_AGG_ERROR,
						     GNOME_DB_QF_AGG_XML_LOAD_ERROR,
						     _("More than one argument for aggregate %s"), 
						     gnome_db_base_get_name (ref));
					return FALSE;
				}
			}
			argnode = argnode->next;
		}
	}

	if (! agg->priv->arg) {
		g_set_error (error,
			     GNOME_DB_QF_AGG_ERROR,
			     GNOME_DB_QF_AGG_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_QF>"));
		return FALSE;
	}
	else
		return TRUE;
}


/*
 * GnomeDbRenderer interface implementation
 */

static gchar *
gnome_db_qf_agg_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfAgg *agg;
	gboolean err = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);
	agg = GNOME_DB_QF_AGG (iface);

	base = gnome_db_ref_base_get_ref_object (agg->priv->agg_ref);

	if (base) {
		GString *string;

		string = g_string_new (gnome_db_base_get_name (base));
		g_string_append (string, " (");
		if (agg->priv->arg) {
			gchar *argstr;
			GnomeDbBase *argbase;

			argbase = gnome_db_ref_base_get_ref_object (agg->priv->arg);
			if (argbase) {
				argstr = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (argbase), context, 
									  options, error);
				if (argstr) {
					g_string_append (string, argstr);
					g_free (argstr);
				}
				else
					err = TRUE;
			}
			else {
				const gchar *tmpstr;
				tmpstr = gnome_db_ref_base_get_ref_name (agg->priv->arg, NULL, NULL);
				g_set_error (error,
					     GNOME_DB_QF_AGG_ERROR,
					     GNOME_DB_QF_AGG_RENDER_ERROR,
					     _("Can't find referenced field '%s'"), tmpstr);
				err = TRUE;
			}
		}
		else {
			g_set_error (error,
				     GNOME_DB_QF_AGG_ERROR,
				     GNOME_DB_QF_AGG_RENDER_ERROR,
				     _("Aggregate '%s' has no argument"), gnome_db_base_get_name (base));
			err = TRUE;
		}
		
		g_string_append (string, ")");
		str = string->str;
		g_string_free (string, FALSE);
	}
	else {
		g_set_error (error,
			     GNOME_DB_QF_AGG_ERROR,
			     GNOME_DB_QF_AGG_RENDER_ERROR,
			     _("Can't find aggregate '%s'"), gnome_db_ref_base_get_ref_name (agg->priv->agg_ref,
											     NULL, NULL));
		err = TRUE;
	}

	if (err) {
		if (str)
			g_free (str);
		return NULL;
	}

	return str;
}

static gchar *
gnome_db_qf_agg_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfAgg *agg;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);
	agg = GNOME_DB_QF_AGG (iface);

	base = gnome_db_ref_base_get_ref_object (agg->priv->agg_ref);

	if (base) {
		GString *string;
		GSList *list;

		string = g_string_new (gnome_db_base_get_name (base));
		g_string_append (string, " (");
		if (agg->priv->arg) {
			gchar *argstr;
			GnomeDbBase *argbase;
			
			argbase = gnome_db_ref_base_get_ref_object (agg->priv->arg);
			if (argbase) {
				argstr = gnome_db_renderer_render_as_str (GNOME_DB_RENDERER (argbase), context);
				g_assert (argstr);
				g_string_append (string, argstr);
				g_free (argstr);
			}
			else {
				const gchar *tmpstr;
				tmpstr = gnome_db_ref_base_get_ref_name (agg->priv->arg, NULL, NULL);
				g_string_append (string, tmpstr);
			}

			list = g_slist_next (list);
		}
		g_string_append (string, ")");

		str = string->str;
		g_string_free (string, FALSE);

	}
	else 
		str = g_strdup (_("Non activated aggregate"));
	
	return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_qf_agg_activate (GnomeDbReferer *iface)
{
	gboolean active = FALSE;
	GnomeDbQfAgg *agg;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, FALSE);
	agg = GNOME_DB_QF_AGG (iface);

	active = gnome_db_ref_base_activate (agg->priv->agg_ref);
	active = active && agg->priv->arg && gnome_db_ref_base_activate (agg->priv->arg);

	return active;
}

static void
gnome_db_qf_agg_deactivate (GnomeDbReferer *iface)
{
	GnomeDbQfAgg *agg;

	g_return_if_fail (iface && IS_GNOME_DB_QF_AGG (iface));
	g_return_if_fail (GNOME_DB_QF_AGG (iface)->priv);
	agg = GNOME_DB_QF_AGG (iface);

	gnome_db_ref_base_deactivate (agg->priv->agg_ref);
	if (agg->priv->arg)
		gnome_db_ref_base_deactivate (agg->priv->arg);
}

static gboolean
gnome_db_qf_agg_is_active (GnomeDbReferer *iface)
{
	gboolean active;
	GnomeDbQfAgg *agg;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, FALSE);
	agg = GNOME_DB_QF_AGG (iface);

	active = gnome_db_ref_base_is_active (agg->priv->agg_ref);
	active = active && agg->priv->arg && gnome_db_ref_base_is_active (agg->priv->arg);

	return active;
}

static GSList *
gnome_db_qf_agg_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
        GnomeDbBase *base;
	GnomeDbQfAgg *agg;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_AGG (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_AGG (iface)->priv, NULL);
	agg = GNOME_DB_QF_AGG (iface);

        base = gnome_db_ref_base_get_ref_object (agg->priv->agg_ref);
        if (base)
                list = g_slist_append (list, base);
	if (agg->priv->arg) {
		base = gnome_db_ref_base_get_ref_object (agg->priv->arg);
		if (base)
			list = g_slist_append (list, base);
	}

        return list;
}

static void
gnome_db_qf_agg_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbQfAgg *agg;

        g_return_if_fail (iface && IS_GNOME_DB_QF_AGG (iface));
        g_return_if_fail (GNOME_DB_QF_AGG (iface)->priv);

        agg = GNOME_DB_QF_AGG (iface);
        if (agg->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, agg->priv->query);
                if (query) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (agg->priv->query),
                                                              G_CALLBACK (nullified_object_cb), agg);
                        agg->priv->query = query;
			gnome_db_base_connect_nullify (query, 
						 G_CALLBACK (nullified_object_cb), agg);
                }
        }

        gnome_db_ref_base_replace_ref_object (agg->priv->agg_ref, replacements);
	if (agg->priv->arg)
		gnome_db_ref_base_replace_ref_object (agg->priv->arg, replacements);
}
