/* gnome-db-layout.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-layout.h"
#include "gnome-db-data-widget.h"
#include "gnome-db-custom-layout.h"
#include "gnome-db-referer.h"
#include "gnome-db-target.h"
#include "gnome-db-table.h"
#include "gnome-db-grid.h"
#include "gnome-db-form.h"
#include "gnome-db-matrix.h"
#include "gnome-db-qfield.h"
#include "gnome-db-parameter.h"

static void gnome_db_layout_class_init (GnomeDbLayoutClass * class);
static void gnome_db_layout_init (GnomeDbLayout * wid);
static void gnome_db_layout_dispose (GObject   * object);

static void gnome_db_layout_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void gnome_db_layout_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void gnome_db_layout_initialize (GnomeDbLayout *form);

static void nullified_custom_layout_cb (GnomeDbCustomLayout *layout, GnomeDbLayout *form);

/* GnomeDbDataWidget interface */
static void            gnome_db_layout_widget_init         (GnomeDbDataWidgetIface *iface);
static void            gnome_db_layout_run                 (GnomeDbDataWidget *iface, guint mode);
static void            gnome_db_layout_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions);
static GnomeDbParameter    *gnome_db_layout_get_param_for_field (GnomeDbDataWidget *iface, GnomeDbQfield *field, const gchar *field_name, 
							   gboolean in_exec_context);
static GnomeDbDataSet      *gnome_db_layout_get_exec_context    (GnomeDbDataWidget *iface);

struct _GnomeDbLayoutPrivate {
	GnomeDbCustomLayout  *layout;

	/* global context */
	GnomeDbDataSet       *exec_context; /* garbage of parameters not dependant on each others */

	/* sub layouts */
	GSList          *work_widgets; /* list of GnomeDbLayout widgets */

	/* glade layout */
	GladeXML        *glade_instance;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* properties */
enum
{
        PROP_0,
};

GType
gnome_db_layout_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbLayoutClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_layout_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbLayout),
			0,
			(GInstanceInitFunc) gnome_db_layout_init
		};		
		
		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) gnome_db_layout_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbLayout", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_DATA_WIDGET, &work_widget_info);
	}

	return type;
}

static void
gnome_db_layout_widget_init (GnomeDbDataWidgetIface *iface)
{
	iface->set_mode = NULL;
	iface->show_column_actions = NULL;
	iface->show_global_actions = gnome_db_layout_show_global_actions;
	iface->get_actions_group = NULL;
}


static void
gnome_db_layout_class_init (GnomeDbLayoutClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = gnome_db_layout_dispose;
	
	/* Properties */
        object_class->set_property = gnome_db_layout_set_property;
        object_class->get_property = gnome_db_layout_get_property;
}

static void
gnome_db_layout_init (GnomeDbLayout * wid)
{
	wid->priv = g_new0 (GnomeDbLayoutPrivate, 1);
}

/**
 * gnome_db_layout_new
 * @layout: a #GnomeDbLayout object
 *
 * Creates a new #GnomeDbLayout widget.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_layout_new (GnomeDbCustomLayout *layout)
{
	GObject *obj;
	GnomeDbLayout *wl;

	g_return_val_if_fail (layout && IS_GNOME_DB_CUSTOM_LAYOUT (layout), NULL);

	obj = g_object_new (GNOME_DB_TYPE_LAYOUT, NULL);
	wl = GNOME_DB_LAYOUT (obj);

	wl->priv->layout = layout;
	g_object_ref (layout);
	gnome_db_base_connect_nullify (wl->priv->layout,
				 G_CALLBACK (nullified_custom_layout_cb), wl);

	gnome_db_layout_initialize (wl);

	return GTK_WIDGET (obj);
}

static void
nullified_custom_layout_cb (GnomeDbCustomLayout *layout, GnomeDbLayout *wl)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (layout),
					      G_CALLBACK (nullified_custom_layout_cb), wl);
	
	g_object_unref (G_OBJECT (wl->priv->layout));
	wl->priv->layout = NULL;
}

static void
gnome_db_layout_dispose (GObject *object)
{
	GnomeDbLayout *wl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_LAYOUT (object));
	wl = GNOME_DB_LAYOUT (object);

	if (wl->priv) {
		/* core */
		if (wl->priv->layout)
			nullified_custom_layout_cb (wl->priv->layout, wl);

		if (wl->priv->exec_context)
			g_object_unref (wl->priv->exec_context);

		g_slist_free (wl->priv->work_widgets);

		if (wl->priv->glade_instance)
			g_object_unref (wl->priv->glade_instance);

		/* the private area itself */
		g_free (wl->priv);
		wl->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_layout_set_property (GObject              *object,
			     guint                 param_id,
			     const GValue         *value,
			     GParamSpec           *pspec)
{
	GnomeDbLayout *wl;

        wl = GNOME_DB_LAYOUT (object);
        if (wl->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
gnome_db_layout_get_property (GObject              *object,
			     guint                 param_id,
			     GValue               *value,
			     GParamSpec           *pspec)
{
	GnomeDbLayout *wl;

        wl = GNOME_DB_LAYOUT (object);
        if (wl->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}



/*
 * Real initialization
 */
static void init_work_form_grid (GnomeDbLayout *wl);
static void init_matrix (GnomeDbLayout *wl);
static void init_super_layout (GnomeDbLayout *wl);
static void 
gnome_db_layout_initialize (GnomeDbLayout *wl)
{
	if (! gnome_db_referer_activate (GNOME_DB_REFERER (wl->priv->layout))) {
		GtkWidget *wid = gtk_label_new (_("missing required objects!"));
		gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
	}
	else {
		switch (gnome_db_custom_layout_get_layout_type (wl->priv->layout)) {
		case GNOME_DB_CUSTOM_LAYOUT_LAYOUT:
			init_super_layout (wl);
			break;
		case GNOME_DB_CUSTOM_LAYOUT_GRID:
		case GNOME_DB_CUSTOM_LAYOUT_FORM:
			init_work_form_grid (wl);
			break;
		case GNOME_DB_CUSTOM_LAYOUT_MATRIX:
			init_matrix (wl);
			break;
		}
	}
}

static void
init_work_form_grid (GnomeDbLayout *wl)
{
	GtkWidget *wid, *root;
	GnomeDbCustomLayout *cl = wl->priv->layout;
	GladeXML *glade;
	GHashTable *hash = NULL;
	GError *error = NULL;
	GnomeDbCustomLayoutData *cldata;
	GnomeDbTarget *target = NULL;
	gboolean err = FALSE;

	glade = gnome_db_custom_layout_get_glade_instance (cl, &root, &hash, &error);
	wl->priv->glade_instance = glade;
	if (error) {
		g_warning ("Error building GladeXML file: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	cldata = gnome_db_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using GnomeDbCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	/* testing */
	if (!cldata->contents.work_iface.query ||
	    ! gnome_db_query_is_select_query (cldata->contents.work_iface.query))
		err = TRUE;

	if (!err) {
		target = (GnomeDbTarget *) cldata->contents.work_iface.modified;
		if (!IS_GNOME_DB_TARGET (target) ||
		    (gnome_db_target_get_query (target) != cldata->contents.work_iface.query)) 
			target = NULL;
	}

	if (! err) {
		/* actual widget creation */
		if (cldata->type == GNOME_DB_CUSTOM_LAYOUT_GRID) 
			wid = gnome_db_grid_new_with_select_query (cldata->contents.work_iface.query, target);
		else 
			wid = gnome_db_form_new_in_layout (cldata->contents.work_iface.query, target, root, hash);
		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		gnome_db_data_widget_set_mode (GNOME_DB_DATA_WIDGET (wid), cldata->contents.work_iface.mode);
		g_object_set (G_OBJECT (wid), "title_visible", FALSE, NULL);
	}
	else 
		wid = gtk_label_new (_("Error creating widget"));
	
	gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	if (hash)
		g_hash_table_destroy (hash);
	g_free (cldata);
}

static void
init_matrix (GnomeDbLayout *wl)
{
	GtkWidget *wid;
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (wl->priv->layout));
	GnomeDbCustomLayout *cl = wl->priv->layout;
	GnomeDbCustomLayoutData *cldata;
	GnomeDbTarget *target;
	GnomeDbTable *table;
	gboolean err = FALSE;
	GError *error = NULL;
	
	cldata = gnome_db_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using GnomeDbCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}

	
	/* tests */
	table = (GnomeDbTable *) cldata->contents.work_iface.modified;
	if (!IS_GNOME_DB_TABLE (table))
		err = TRUE;

	if (!cldata->contents.work_iface.query ||
	    ! gnome_db_query_is_select_query (cldata->contents.work_iface.query))
		err = TRUE;

	if (!cldata->contents.work_iface.query_extra ||
	    ! gnome_db_query_is_select_query (cldata->contents.work_iface.query_extra))
		err = TRUE;


	if (!err) {
		target = cldata->contents.work_iface.rows_target;
		if (!IS_GNOME_DB_TARGET (target) ||
		    (gnome_db_target_get_query (target) != cldata->contents.work_iface.query)) 
			err = TRUE;
	}

	if (!err) {
		target = cldata->contents.work_iface.cols_target;
		if (!IS_GNOME_DB_TARGET (target) ||
		    (gnome_db_target_get_query (target) != cldata->contents.work_iface.query_extra)) 
			err = TRUE;
	}
	    
	/* actual widget creation */
	if (!err) {
		wid = gnome_db_matrix_new (dict,
					   cldata->contents.work_iface.query,
					   cldata->contents.work_iface.rows_target,
					   cldata->contents.work_iface.query_extra,
					   cldata->contents.work_iface.cols_target,
					   (GnomeDbTable *) cldata->contents.work_iface.modified,
					   NULL);
		gnome_db_matrix_set_view_type (GNOME_DB_MATRIX (wid), cldata->contents.work_iface.view_type);
		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		gnome_db_data_widget_set_mode (GNOME_DB_DATA_WIDGET (wid), cldata->contents.work_iface.mode);
		g_object_set (G_OBJECT (wid), "title_visible", FALSE, NULL);
	}
	else
		wid = gtk_label_new (_("Error creating widget"));

	gtk_box_pack_start (GTK_BOX (wl), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	g_free (cldata);
}

static GnomeDbLayout *
find_by_custom_layout (GnomeDbLayout *super_layout, GnomeDbCustomLayout *cl)
{
	GnomeDbLayout *retval = NULL;
	GSList *list = super_layout->priv->work_widgets;
	while (list && !retval) {
		if (GNOME_DB_LAYOUT (list->data)->priv->layout == cl)
			retval = GNOME_DB_LAYOUT (list->data);
		else
			list = g_slist_next (list);
	}

	return retval;
}

static void
init_super_layout (GnomeDbLayout *wl)
{
	GtkWidget *wid, *box, *root;
	GnomeDbCustomLayout *cl = wl->priv->layout;
	GladeXML *glade;
	GHashTable *hash = NULL;
	GError *error = NULL;
	GnomeDbCustomLayoutData *cldata;
	GSList *list;

	glade = gnome_db_custom_layout_get_glade_instance (cl, &root, &hash, &error);
	wl->priv->glade_instance = glade;
	if (error) {
		g_warning ("Error building GladeXML file: %s", error->message);
		g_error_free (error);
		error = NULL;
	}

	if (!root) {
		/* If there is no Glade layout, or if there was an error, then we use
		   a table layout */
		TO_IMPLEMENT;
	}
	
	cldata = gnome_db_custom_layout_get_data (cl, &error);
	if (error) {
		g_warning ("Error using GnomeDbCustomLayout: %s", error->message);
		g_error_free (error);
		error = NULL;
	}
	
	/* actual widget creation */
	list = cldata->contents.layout.children;
	while (list) {
		wid = gnome_db_layout_new (GNOME_DB_CUSTOM_LAYOUT (list->data));

		if (hash) {
			box = g_hash_table_lookup (hash, list->data);
			gtk_box_pack_start (GTK_BOX (box), wid, TRUE, TRUE, 0);
			gtk_widget_show (wid);
			if (! g_object_get_data (G_OBJECT (box), "show_actions")) 
				gnome_db_data_widget_global_show_actions (GNOME_DB_DATA_WIDGET (wid), FALSE);
		}
		else {
			TO_IMPLEMENT;
		}
		
		wl->priv->work_widgets = g_slist_prepend (wl->priv->work_widgets, wid);
		list = g_slist_next (list);
	}

	list = cldata->contents.layout.connects;
	while (list) {
		GnomeDbCustomLayoutConnect *lc = (GnomeDbCustomLayoutConnect*) list->data;
		GnomeDbDataWidget *src, *dest;
		GnomeDbParameter *param_src, *param_dest;
		GnomeDbLayout *tmp;

		tmp = find_by_custom_layout (wl, lc->src_layout);
		g_assert (tmp->priv && (g_slist_length (tmp->priv->work_widgets) == 1));
		src = GNOME_DB_DATA_WIDGET (tmp->priv->work_widgets->data);

		tmp = find_by_custom_layout (wl, lc->dest_layout);
		g_assert (tmp->priv && (g_slist_length (tmp->priv->work_widgets) == 1));
		dest = GNOME_DB_DATA_WIDGET (tmp->priv->work_widgets->data);

		/* param_src = gnome_db_data_widget_get_param_for_field_data (src, GNOME_DB_QFIELD (lc->src_field)); */
		if (!param_src) 
			g_warning (_("Cannot find a parameter for source field connection"));
		/* param_dest = gnome_db_data_widget_get_param_for_field_exec (dest, GNOME_DB_QFIELD (lc->dest_field)); */
		if (!param_dest) 
			g_warning (_("Cannot find a parameter for destination field connection"));
		
		if (param_src && param_dest)
			gnome_db_parameter_bind_to_param (param_dest, param_src);
		
		list = g_slist_next (list);
	}
	
	gtk_box_pack_start (GTK_BOX (wl), root, TRUE, TRUE, 0);
	gtk_widget_show (root);
	if (hash)
		g_hash_table_destroy (hash);
	g_free (cldata);
}

/**
 * gnome_db_layout_lookup_widget
 * @layout: a #GnomeDbLayout widget
 * @widget_name: a string
 *
 * Retreives the #GtkWidget which name is @widget_name, if @layout
 * uses a Glade file as layout.
 *
 * Returns: the requested #GtkWidget, or %NULL if not found
 */
GtkWidget *
gnome_db_layout_lookup_widget (GnomeDbLayout *layout, const gchar *widget_name)
{
	GtkWidget *retval = NULL;

	g_return_val_if_fail (layout && IS_GNOME_DB_LAYOUT (layout), NULL);
	g_return_val_if_fail (layout->priv, NULL);

	if (layout->priv->glade_instance)
		retval = glade_xml_get_widget (layout->priv->glade_instance, widget_name);

	return retval;
}

/*
 * GnomeDbDataWidget interface implementation
 */
static void
gnome_db_layout_run (GnomeDbDataWidget *iface, guint mode)
{
	GSList *list;
	GnomeDbLayout *wl;

	g_return_if_fail (iface && IS_GNOME_DB_LAYOUT (iface));
	wl = GNOME_DB_LAYOUT (iface);
	g_return_if_fail (wl->priv);

	list = wl->priv->work_widgets;
	while (list) {
		/* gnome_db_data_widget_run (GNOME_DB_DATA_WIDGET (list->data), 0); */ /* don't change previous mode */
		list = g_slist_next (list);
	}
}

static void
gnome_db_layout_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions)
{
	GSList *list;
	GnomeDbLayout *wl;

	g_return_if_fail (iface && IS_GNOME_DB_LAYOUT (iface));
	wl = GNOME_DB_LAYOUT (iface);
	g_return_if_fail (wl->priv);

	list = wl->priv->work_widgets;
	while (list) {
		gnome_db_data_widget_global_show_actions (GNOME_DB_DATA_WIDGET (list->data), show_actions);
		list = g_slist_next (list);
	}
}

static GnomeDbParameter *
gnome_db_layout_get_param_for_field (GnomeDbDataWidget *iface, GnomeDbQfield *field, const gchar *field_name, gboolean in_exec_context)
{
	GnomeDbLayout *wl;
	GnomeDbParameter *param = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_LAYOUT (iface), NULL);
	wl = GNOME_DB_LAYOUT (iface);
	g_return_val_if_fail (wl->priv, NULL);
	g_return_val_if_fail (field || (field_name && *field_name), NULL);
	
	TO_IMPLEMENT;
	return param;
}


static GnomeDbDataSet *
gnome_db_layout_get_exec_context (GnomeDbDataWidget *iface)
{
	GnomeDbLayout *wl;

	g_return_val_if_fail (iface && IS_GNOME_DB_LAYOUT (iface), NULL);
	wl = GNOME_DB_LAYOUT (iface);
	g_return_val_if_fail (wl->priv, NULL);
	
	return wl->priv->exec_context;
}
