/* gnome-db-database.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-database.h"
#include "gnome-db-table.h"
#include "gnome-db-field.h"
#include "gnome-db-constraint.h"
#include "gnome-db-referer.h"
#include "gnome-db-entity.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-result-set.h"
#include "marshal.h"
#include <string.h>
#include "gnome-db-server.h"

/* 
 * Main static functions 
 */
static void gnome_db_database_class_init (GnomeDbDatabaseClass * class);
static void gnome_db_database_init (GnomeDbDatabase * srv);
static void gnome_db_database_dispose (GObject   * object);
static void gnome_db_database_finalize (GObject   * object);

static void gnome_db_database_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_database_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

static void        gnome_db_database_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_database_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_database_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_database_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

static void gnome_db_database_add_table (GnomeDbDatabase *mgdb, GnomeDbTable *table, gint pos);
static void gnome_db_database_add_sequence (GnomeDbDatabase *mgdb, GnomeDbSequence *seq, gint pos);

static void nullified_table_cb (GnomeDbTable *table, GnomeDbDatabase *mgdb);
static void nullified_sequence_cb (GnomeDbSequence *seq, GnomeDbDatabase *mgdb);
static void nullified_constraint_cb (GnomeDbConstraint *cons, GnomeDbDatabase *mgdb);

static void updated_table_cb (GnomeDbTable *table, GnomeDbDatabase *mgdb);
static void updated_sequence_cb (GnomeDbSequence *seq, GnomeDbDatabase *mgdb);
static void updated_constraint_cb (GnomeDbConstraint *cons, GnomeDbDatabase *mgdb);

#ifdef debug
static void gnome_db_database_dump (GnomeDbDatabase *mgdb, gint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	DATA_UPDATE_STARTED,
	DATA_UPDATE_FINISHED,
	UPDATE_PROGRESS,
	TABLE_ADDED,
	TABLE_REMOVED,
	TABLE_UPDATED,
	FIELD_ADDED,
	FIELD_REMOVED,
	FIELD_UPDATED,
	SEQUENCE_ADDED,
	SEQUENCE_REMOVED,
	SEQUENCE_UPDATED,
	CONSTRAINT_ADDED,
	CONSTRAINT_REMOVED,
	CONSTRAINT_UPDATED,
	FS_LINK_ADDED,
	FS_LINK_REMOVED,
	LAST_SIGNAL
};

static gint gnome_db_database_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP
};


/* private structure */
struct _GnomeDbDatabasePrivate
{
	/* Db structure */
	GSList                 *tables;
	GSList                 *sequences;
	GSList                 *constraints;
	GHashTable             *constraints_hash; /* key=table, value=GSList of constraints on that table */
	GHashTable             *tables_hash;

	/* XML loading attributes */
	gboolean                xml_loading;

	/* DBMS update related information */
	gboolean                update_in_progress;
	gboolean                stop_update; /* TRUE if a DBMS data update must be stopped */

	/* the names are set to lower case before any comparison */
	gboolean                lc_names;
};


/* module error */
GQuark gnome_db_database_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_database_error");
	return quark;
}

/**
 * gnome_db_database_get_type
 *
 * Returns: the type id
 */
GType
gnome_db_database_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbDatabaseClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_database_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbDatabase),
			0,
			(GInstanceInitFunc) gnome_db_database_init
		};
		

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_database_xml_storage_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GNOME_DB_TYPE_BASE, "GnomeDbDatabase", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_XML_STORAGE, &xml_storage_info);
	}
	return type;
}

static void 
gnome_db_database_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_database_get_xml_id;
	iface->save_to_xml = gnome_db_database_save_to_xml;
	iface->load_from_xml = gnome_db_database_load_from_xml;
}

static void
gnome_db_database_class_init (GnomeDbDatabaseClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

        gnome_db_database_signals[TABLE_ADDED] =
                g_signal_new ("table_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, table_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_database_signals[TABLE_REMOVED] =
                g_signal_new ("table_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, table_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_database_signals[TABLE_UPDATED] =
                g_signal_new ("table_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, table_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[FIELD_ADDED] =
                g_signal_new ("field_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, field_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[FIELD_REMOVED] =
                g_signal_new ("field_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, field_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[FIELD_UPDATED] =
                g_signal_new ("field_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, field_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[SEQUENCE_ADDED] =
                g_signal_new ("sequence_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, sequence_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[SEQUENCE_REMOVED] =
                g_signal_new ("sequence_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, sequence_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[SEQUENCE_UPDATED] =
                g_signal_new ("sequence_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, sequence_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[CONSTRAINT_ADDED] =
                g_signal_new ("constraint_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, constraint_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[CONSTRAINT_REMOVED] =
                g_signal_new ("constraint_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, constraint_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_database_signals[CONSTRAINT_UPDATED] =
                g_signal_new ("constraint_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, constraint_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_database_signals[FS_LINK_ADDED] =
                g_signal_new ("fs_link_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, fs_link_added),
                              NULL, NULL,
                              marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER);
	gnome_db_database_signals[FS_LINK_REMOVED] =
                g_signal_new ("fs_link_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, fs_link_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER_POINTER,
                              G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_POINTER);
	gnome_db_database_signals[DATA_UPDATE_STARTED] =
                g_signal_new ("data_update_started",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, data_update_started),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gnome_db_database_signals[UPDATE_PROGRESS] =
                g_signal_new ("update_progress",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, update_progress),
                              NULL, NULL,
                              marshal_VOID__POINTER_UINT_UINT,
                              G_TYPE_NONE, 3, G_TYPE_POINTER, G_TYPE_UINT, G_TYPE_UINT);
	gnome_db_database_signals[DATA_UPDATE_FINISHED] =
                g_signal_new ("data_update_finished",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbDatabaseClass, data_update_finished),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);

	class->table_added = NULL;
	class->table_removed = NULL;
	class->table_updated = NULL;
	class->field_added = NULL;
	class->field_removed = NULL;
	class->field_updated = NULL;
	class->sequence_added = NULL;
	class->sequence_removed = NULL;
	class->sequence_updated = NULL;
	class->constraint_added = NULL;
	class->constraint_removed = NULL;
	class->constraint_updated = NULL;
	class->fs_link_added = NULL;
	class->fs_link_removed = NULL;
        class->data_update_started = NULL;
        class->data_update_finished = NULL;
        class->update_progress = NULL;

	object_class->dispose = gnome_db_database_dispose;
	object_class->finalize = gnome_db_database_finalize;

	/* Properties */
	object_class->set_property = gnome_db_database_set_property;
	object_class->get_property = gnome_db_database_get_property;
	g_object_class_install_property (object_class, PROP,
					 g_param_spec_pointer ("prop", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_database_dump;
#endif

}


static void
gnome_db_database_init (GnomeDbDatabase * gnome_db_database)
{
	gnome_db_database->priv = g_new0 (GnomeDbDatabasePrivate, 1);
	gnome_db_database->priv->tables = NULL;
        gnome_db_database->priv->sequences = NULL;
        gnome_db_database->priv->constraints = NULL;
	gnome_db_database->priv->constraints_hash = g_hash_table_new (NULL, NULL);

	gnome_db_database->priv->tables_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	gnome_db_database->priv->xml_loading = FALSE;

	gnome_db_database->priv->update_in_progress = FALSE;
	gnome_db_database->priv->stop_update = FALSE;

	gnome_db_database->priv->lc_names = TRUE;
}


/**
 * gnome_db_database_new
 * @dict: a #GnomeDbDict object
 * 
 * Creates a new GnomeDbDatabase object
 *
 * Returns: the new object
 */
GObject   *
gnome_db_database_new (GnomeDbDict *dict)
{
	GObject   *obj;
	GnomeDbDatabase *gnome_db_database;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_TYPE_DATABASE, NULL);
	gnome_db_database = GNOME_DB_DATABASE (obj);

	gnome_db_base_set_dict (GNOME_DB_BASE (gnome_db_database), ASSERT_DICT (dict));

	return obj;
}


static void
constraints_hash_foreach (GnomeDbTable *table, GSList *constraints, gpointer data)
{
	g_slist_free (constraints);
}

static void
gnome_db_database_dispose (GObject   *object)
{
	GnomeDbDatabase *gnome_db_database;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATABASE (object));

	gnome_db_database = GNOME_DB_DATABASE (object);
	if (gnome_db_database->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_database->priv->tables_hash) {
			g_hash_table_destroy (gnome_db_database->priv->tables_hash);
			gnome_db_database->priv->tables_hash = NULL;
		}

		if (gnome_db_database->priv->constraints_hash) {
			g_hash_table_foreach (gnome_db_database->priv->constraints_hash, (GHFunc) constraints_hash_foreach, NULL);
			g_hash_table_destroy (gnome_db_database->priv->constraints_hash);
			gnome_db_database->priv->constraints_hash = NULL;
		}

		/* getting rid of the constraints */
		while (gnome_db_database->priv->constraints) 
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_database->priv->constraints->data));

		/* getting rid of the sequences */
		while (gnome_db_database->priv->sequences) 
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_database->priv->sequences->data));
		
		/* getting rid of the tables */
		while (gnome_db_database->priv->tables) 
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_database->priv->tables->data));
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_database_finalize (GObject *object)
{
	GnomeDbDatabase *gnome_db_database;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_DATABASE (object));

	gnome_db_database = GNOME_DB_DATABASE (object);
	if (gnome_db_database->priv) {
		g_free (gnome_db_database->priv);
		gnome_db_database->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_database_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbDatabase *gnome_db_database;

	gnome_db_database = GNOME_DB_DATABASE (object);
	if (gnome_db_database->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			ptr = g_value_get_pointer (value);
			break;
		}
	}
}
static void
gnome_db_database_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbDatabase *gnome_db_database;
	gnome_db_database = GNOME_DB_DATABASE (object);
	
	if (gnome_db_database->priv) {
		switch (param_id) {
		case PROP:
			/* FIXME */
			g_value_set_pointer (value, NULL);
			break;
		}	
	}
}


/* GnomeDbXmlStorage interface implementation */
static gchar *
gnome_db_database_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_DATABASE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_DATABASE (iface)->priv, NULL);

	return NULL;
}

static xmlNodePtr
gnome_db_database_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr toptree, tree;
	GnomeDbDatabase *mgdb;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_DATABASE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_DATABASE (iface)->priv, NULL);

	mgdb = GNOME_DB_DATABASE (iface);

	/* main node */
        toptree = xmlNewNode (NULL, "GNOME_DB_DATABASE");
	
	/* Tables */
	tree = xmlNewChild (toptree, NULL, "GNOME_DB_TABLES", NULL);
	list = mgdb->priv->tables;
	while (list) {
		xmlNodePtr table;
		
		table = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
				
		if (table)
			xmlAddChild (tree, table);
		else {
			xmlFreeNode (tree);
			return NULL;
		}

		list = g_slist_next (list);
	}

	/* Sequences */
	if (mgdb->priv->sequences) {
		tree = xmlNewChild (toptree, NULL, "GNOME_DB_SEQUENCES", NULL);
		list = mgdb->priv->sequences;
		while (list) {
			xmlNodePtr table;
			
			table = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			
			if (table)
				xmlAddChild (tree, table);
			else {
				xmlFreeNode (tree);
				return NULL;
			}
			
			list = g_slist_next (list);
		}
	}

	/* Constraints */
	tree = xmlNewChild (toptree, NULL, "GNOME_DB_CONSTRAINTS", NULL);
	list = mgdb->priv->constraints;
	while (list) {
		xmlNodePtr cstr;
		
		cstr = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), NULL);
				
		if (cstr)
			xmlAddChild (tree, cstr); /* otherwise, just ignore the error */
#ifdef debug
		else
			g_print (D_COL_ERR "ERROR\n" D_COL_NOR);
#endif
		list = g_slist_next (list);
	}

	return toptree;
}

static gboolean gnome_db_database_load_from_xml_tables (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);
static gboolean gnome_db_database_load_from_xml_constraints (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

static gboolean
gnome_db_database_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbDatabase *mgdb;
	xmlNodePtr subnode;

	g_return_val_if_fail (iface && IS_GNOME_DB_DATABASE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_DATABASE (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	mgdb = GNOME_DB_DATABASE (iface);

	if (mgdb->priv->tables || mgdb->priv->sequences || mgdb->priv->constraints) {
		g_set_error (error,
			     GNOME_DB_DATABASE_ERROR,
			     GNOME_DB_DATABASE_XML_LOAD_ERROR,
			     _("Database already contains data"));
		return FALSE;
	}
	if (strcmp (node->name, "GNOME_DB_DATABASE")) {
		g_set_error (error,
			     GNOME_DB_DATABASE_ERROR,
			     GNOME_DB_DATABASE_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_DATABASE>"));
		return FALSE;
	}
	mgdb->priv->xml_loading = TRUE;
	subnode = node->children;
	while (subnode) {
		gboolean done = FALSE;

		if (!strcmp (subnode->name, "GNOME_DB_TABLES")) {
			if (!gnome_db_database_load_from_xml_tables (iface, subnode, error)) {
				mgdb->priv->xml_loading = FALSE;
				return FALSE;
			}

			done = TRUE;
		}

		if (!done && !strcmp (subnode->name, "GNOME_DB_SEQUENCES")) {
			TO_IMPLEMENT;
			done = TRUE;
		}

		if (!done && !strcmp (subnode->name, "GNOME_DB_CONSTRAINTS")) {
			if (!gnome_db_database_load_from_xml_constraints (iface, subnode, error)) {
				mgdb->priv->xml_loading = FALSE;
				return FALSE;
			}

			done = TRUE;
		}

		subnode = subnode->next;
	}
	mgdb->priv->xml_loading = FALSE;
	
	return TRUE;
}

static gboolean
gnome_db_database_load_from_xml_tables (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	gboolean allok = TRUE;
	xmlNodePtr subnode = node->children;

	while (subnode && allok) {
		if (! xmlNodeIsText (subnode)) {
			if (!strcmp (subnode->name, "GNOME_DB_TABLE")) {
				GnomeDbTable *table;
				
				table = GNOME_DB_TABLE (gnome_db_table_new (gnome_db_base_get_dict (GNOME_DB_BASE (iface))));
				allok = gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (table), subnode, error);
				if (allok)
					gnome_db_database_add_table (GNOME_DB_DATABASE (iface), table, -1);
				g_object_unref (G_OBJECT (table));
			}
			else {
				allok = FALSE;
				g_set_error (error,
					     GNOME_DB_DATABASE_ERROR,
					     GNOME_DB_DATABASE_XML_LOAD_ERROR,
					     _("XML Tag below <GNOME_DB_TABLES> is not <GNOME_DB_TABLE>"));
			}
		}			

		subnode = subnode->next;
	}	

	return allok;
}

static void gnome_db_database_add_constraint_real (GnomeDbDatabase *mgdb, GnomeDbConstraint *cstr, gboolean force_user_constraint);
static gboolean
gnome_db_database_load_from_xml_constraints (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	gboolean allok = TRUE;
	xmlNodePtr subnode = node->children;

	while (subnode && allok) {
		if (! xmlNodeIsText (subnode)) {
			if (!strcmp (subnode->name, "GNOME_DB_CONSTRAINT")) {
				GnomeDbConstraint *cstr;
				
				cstr = GNOME_DB_CONSTRAINT (gnome_db_constraint_new_with_db (GNOME_DB_DATABASE (iface)));
				allok = gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (cstr), subnode, error);
				gnome_db_database_add_constraint_real (GNOME_DB_DATABASE (iface), cstr, FALSE);
				g_object_unref (G_OBJECT (cstr));
			}
			else {
				allok = FALSE;
				g_set_error (error,
					     GNOME_DB_DATABASE_ERROR,
					     GNOME_DB_DATABASE_XML_LOAD_ERROR,
					     _("XML Tag below <GNOME_DB_CONSTRAINTS> is not <GNOME_DB_CONSTRAINT>"));
			}
		}			

		subnode = subnode->next;
	}	

	return allok;
}

/*
 * pos = -1 to append the table to the list
 */
static void
gnome_db_database_add_table (GnomeDbDatabase *mgdb, GnomeDbTable *table, gint pos)
{
	gchar *str;
	g_return_if_fail (table);
	g_return_if_fail (!g_slist_find (mgdb->priv->tables, table));

	g_object_set (G_OBJECT (table), "database", mgdb, NULL);
	mgdb->priv->tables = g_slist_insert (mgdb->priv->tables, table, pos);

	g_object_ref (G_OBJECT (table));
	gnome_db_base_connect_nullify (table, G_CALLBACK (nullified_table_cb), mgdb);
	g_signal_connect (G_OBJECT (table), "changed",
			  G_CALLBACK (updated_table_cb), mgdb);


	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (table));
	g_hash_table_insert (mgdb->priv->tables_hash, str, table);

#ifdef debug_signal
	g_print (">> 'TABLE_ADDED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (mgdb), gnome_db_database_signals[TABLE_ADDED], 0, table);
#ifdef debug_signal
	g_print ("<< 'TABLE_ADDED' from %s\n", __FUNCTION__);
#endif
}

/*
 * pos = -1 to append the table to the list
 */
static void
gnome_db_database_add_sequence (GnomeDbDatabase *mgdb, GnomeDbSequence *seq, gint pos)
{
	g_return_if_fail (seq);
	g_return_if_fail (!g_slist_find (mgdb->priv->sequences, seq));

	g_object_set (G_OBJECT (seq), "database", mgdb, NULL);
	mgdb->priv->sequences = g_slist_insert (mgdb->priv->sequences, seq, pos);

	g_object_ref (G_OBJECT (seq));
	gnome_db_base_connect_nullify (seq, G_CALLBACK (nullified_sequence_cb), mgdb);
	g_signal_connect (G_OBJECT (seq), "changed",
			  G_CALLBACK (updated_sequence_cb), mgdb);


#ifdef debug_signal
	g_print (">> 'SEQUENCE_ADDED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (mgdb), gnome_db_database_signals[SEQUENCE_ADDED], 0, seq);
#ifdef debug_signal
	g_print ("<< 'SEQUENCE_ADDED' from %s\n", __FUNCTION__);
#endif
}

/**
 * gnome_db_database_add_constraint
 * @mgdb: a #GnomeDbDatabase object
 * @cstr: a #GnomeDbConstraint
 *
 * Add the @cstr constraint to the database. The @cstr constraint is a user-defined constraint
 * (which is not part of the database structure itself).
 */
void
gnome_db_database_add_constraint (GnomeDbDatabase *mgdb, GnomeDbConstraint *cstr)
{
	gnome_db_database_add_constraint_real (mgdb, cstr, TRUE);
}

static void
gnome_db_database_add_constraint_real (GnomeDbDatabase *mgdb, GnomeDbConstraint *cstr, gboolean force_user_constraint)
{
	GnomeDbConstraint *ptr = NULL;

	g_return_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb));
	g_return_if_fail (mgdb->priv);
	g_return_if_fail (cstr);

	/* Try to activate the constraints here */
	gnome_db_referer_activate (GNOME_DB_REFERER (cstr));

	/* try to find if a similar constraint is there, if we are not loading from an XML file */
	if (!mgdb->priv->xml_loading) {
		GSList *list = mgdb->priv->constraints;
		while (list && !ptr) {
			if (gnome_db_constraint_equal (cstr, GNOME_DB_CONSTRAINT (list->data)))
				ptr = GNOME_DB_CONSTRAINT (list->data);
			list = g_slist_next (list);
		}
	}

	if (ptr) {
		/* update the existing constraint with name and description */
		gnome_db_base_set_name (GNOME_DB_BASE (ptr), gnome_db_base_get_name (GNOME_DB_BASE (cstr)));
		gnome_db_base_set_description (GNOME_DB_BASE (ptr), gnome_db_base_get_description (GNOME_DB_BASE (cstr)));
		gnome_db_base_set_owner (GNOME_DB_BASE (ptr), gnome_db_base_get_owner (GNOME_DB_BASE (cstr)));
	}
	else {
		GSList *list;
		GnomeDbTable *table;

		/* user defined constraint */
		if (force_user_constraint)
			g_object_set (G_OBJECT (cstr), "user_constraint", TRUE, NULL);

		/* add @cstr to the list of constraints */
		mgdb->priv->constraints = g_slist_append (mgdb->priv->constraints, cstr);
		g_object_ref (G_OBJECT (cstr));
		gnome_db_base_connect_nullify (cstr, G_CALLBACK (nullified_constraint_cb), mgdb);
		g_signal_connect (G_OBJECT (cstr), "changed",
				  G_CALLBACK (updated_constraint_cb), mgdb);

		/* add the constraint to the 'constraints_hash' */
		table = gnome_db_constraint_get_table (cstr);
		list = g_hash_table_lookup (mgdb->priv->constraints_hash, table);
		list = g_slist_append (list, cstr);
		g_hash_table_insert (mgdb->priv->constraints_hash, table, list);

#ifdef debug_signal
		g_print (">> 'CONSTRAINT_ADDED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (mgdb), gnome_db_database_signals[CONSTRAINT_ADDED], 0, cstr);
#ifdef debug_signal
		g_print ("<< 'CONSTRAINT_ADDED' from %s\n", __FUNCTION__);
#endif
	}
}

#ifdef debug
static void 
gnome_db_database_dump (GnomeDbDatabase *mgdb, gint offset)
{
	gchar *str;
	guint i;
	GSList *list;

	g_return_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb));
	
	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;
	
	/* dump */
	if (mgdb->priv) 
		g_print ("%s" D_COL_H1 "GnomeDbDatabase %p\n" D_COL_NOR, str, mgdb);
	else
		g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, mgdb);

	/* tables */
	list = mgdb->priv->tables;
	if (list) {
		g_print ("%sTables:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Table defined\n", str);

	/* sequences */
	list = mgdb->priv->sequences;
	if (list) {
		g_print ("%sSequences:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Sequence defined\n", str);


	/* constraints */
	list = mgdb->priv->constraints;
	if (list) {
		g_print ("%sConstraints:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Constraint defined\n", str);


	g_free (str);

}
#endif


/**
 * gnome_db_database_get_dict
 * @mgdb: a #GnomeDbDatabase object
 *
 * Fetch the GnomeDbDict object to which the GnomeDbDatabase belongs.
 *
 * Returns: the GnomeDbDict object
 */
GnomeDbDict *
gnome_db_database_get_dict (GnomeDbDatabase *mgdb)
{
	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);

	return gnome_db_base_get_dict (GNOME_DB_BASE (mgdb));
}



static gboolean database_tables_update_list (GnomeDbDatabase * mgdb, GError **error);
static gboolean database_sequences_update_list (GnomeDbDatabase * mgdb, GError **error);
static gboolean database_constraints_update_list (GnomeDbDatabase * mgdb, GError **error);
/**
 * gnome_db_database_update_dbms_data
 * @mgdb: a #GnomeDbDatabase object
 * @error: location to store error, or %NULL
 *
 * Synchronises the Database representation with the database structure which is stored in
 * the DBMS. For this operation to succeed, the connection to the DBMS server MUST be opened
 * (using the corresponding #GnomeDbServer object).
 *
 * Returns: TRUE if no error
 */
gboolean
gnome_db_database_update_dbms_data (GnomeDbDatabase *mgdb, GError **error)
{
	gboolean retval = TRUE;
	GnomeDbServer *srv;
	GnomeDbDict *dict;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), FALSE);
	g_return_val_if_fail (mgdb->priv, FALSE);
	
	if (mgdb->priv->update_in_progress) {
		g_set_error (error, GNOME_DB_DATABASE_ERROR, GNOME_DB_DATABASE_META_DATA_UPDATE,
			     _("Update already started!"));
		return FALSE;
	}

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (mgdb));
	srv = gnome_db_dict_get_server (dict);
	if (!gnome_db_server_conn_is_opened (srv)) {
		g_set_error (error, GNOME_DB_DATABASE_ERROR, GNOME_DB_DATABASE_META_DATA_UPDATE,
			     _("Connection is not opened!"));
		return FALSE;
	}

	mgdb->priv->update_in_progress = TRUE;
	mgdb->priv->stop_update = FALSE;

#ifdef debug_signal
	g_print (">> 'DATA_UPDATE_STARTED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (mgdb), gnome_db_database_signals[DATA_UPDATE_STARTED], 0);
#ifdef debug_signal
	g_print ("<< 'DATA_UPDATE_STARTED' from %s\n", __FUNCTION__);
#endif

	retval = database_tables_update_list (mgdb, error);
	if (retval && !mgdb->priv->stop_update) 
		retval = database_sequences_update_list (mgdb, error);
	if (retval && !mgdb->priv->stop_update) 
		retval = database_constraints_update_list (mgdb, error);

#ifdef debug_signal
	g_print (">> 'DATA_UPDATE_FINISHED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (mgdb), gnome_db_database_signals[DATA_UPDATE_FINISHED], 0);
#ifdef debug_signal
	g_print ("<< 'DATA_UPDATE_FINISHED' from %s\n", __FUNCTION__);
#endif

	mgdb->priv->update_in_progress = FALSE;
	if (mgdb->priv->stop_update) {
		g_set_error (error, GNOME_DB_DATABASE_ERROR, GNOME_DB_DATABASE_META_DATA_UPDATE_USER_STOPPED,
			     _("Update stopped!"));
		return FALSE;
	}

	return retval;
}


/**
 * gnome_db_database_stop_update_dbms_data
 * @mgdb: a #GnomeDbDatabase object
 *
 * When the database updates its internal lists of DBMS objects, a call to this function will 
 * stop that update process. It has no effect when the database is not updating its DBMS data.
 */
void
gnome_db_database_stop_update_dbms_data (GnomeDbDatabase *mgdb)
{
	g_return_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb));
	g_return_if_fail (mgdb->priv);

	mgdb->priv->stop_update = TRUE;
}

static gboolean
database_tables_update_list (GnomeDbDatabase *mgdb, GError **error)
{
	GSList *list;
	GdaDataModel *rs;
	gchar *str;
	guint now, total;
	GSList *updated_tables = NULL;
	GnomeDbServer *srv;
	GnomeDbTable *table;
	GSList *constraints;

	srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (mgdb)));
	rs = gnome_db_server_get_gda_schema (srv, GDA_CONNECTION_SCHEMA_TABLES, NULL);
	if (!rs) {
		g_set_error (error, GNOME_DB_DATABASE_ERROR, GNOME_DB_DATABASE_TABLES_ERROR,
			     _("Can't get list of tables"));
		return FALSE;
	}

	if (!gnome_db_result_set_check_data_model (rs, 4, 
						   GDA_VALUE_TYPE_STRING, 
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING)) {
		g_set_error (error, GNOME_DB_DATABASE_ERROR, GNOME_DB_DATABASE_TABLES_ERROR,
			     _("Schema for list of tables is wrong"));
		g_object_unref (G_OBJECT (rs));
		return FALSE;
	}

	total = gda_data_model_get_n_rows (rs);
	now = 0;		
	while ((now < total) && !mgdb->priv->stop_update) {
		const GdaValue *value;
		gboolean newtable = FALSE;
		gint i = -1;

		value = gda_data_model_get_value_at (rs, 0, now);
		str = gda_value_stringify (value);
		table = gnome_db_database_get_table_by_name (mgdb, str);
		if (!table) {
			gboolean found = FALSE;

			i = 0;
			/* table name */
			table = GNOME_DB_TABLE (gnome_db_table_new (gnome_db_base_get_dict (GNOME_DB_BASE (mgdb))));
			gnome_db_base_set_name (GNOME_DB_BASE (table), str);
			newtable = TRUE;
			
			/* finding the right position for the table */
			list = mgdb->priv->tables;
			while (list && !found) {
				if (strcmp (str, gnome_db_base_get_name (GNOME_DB_BASE (list->data))) < 0)
					found = TRUE;
				else
					i++;
				list = g_slist_next (list);
			}
		}
		g_free (str);
		
		updated_tables = g_slist_append (updated_tables, table);

		/* description */
		value = gda_data_model_get_value_at (rs, 2, now);
		if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
			str = gda_value_stringify (value);
			gnome_db_base_set_description (GNOME_DB_BASE (table), str);
			g_free (str);
		}
		else 
			gnome_db_base_set_description (GNOME_DB_BASE (table), NULL);

		/* owner */
		value = gda_data_model_get_value_at (rs, 1, now);
		if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
			str = gda_value_stringify (value);
			gnome_db_base_set_owner (GNOME_DB_BASE (table), str);
			g_free (str);
		}
		else
			gnome_db_base_set_owner (GNOME_DB_BASE (table), NULL);
				
		/* fields */
		g_object_set (G_OBJECT (table), "database", mgdb, NULL);
		if (!gnome_db_table_update_dbms_data (table, error)) {
			g_object_unref (G_OBJECT (rs));
			return FALSE;
		}

		/* signal if the DT is new */
		if (newtable) {
			gnome_db_database_add_table (mgdb, table, i);
			g_object_unref (G_OBJECT (table));
		}

		/* Taking care of the constraints coming attached to the GnomeDbTable */
		constraints = g_object_get_data (G_OBJECT (table), "pending_constraints");
		if (constraints) {
			GSList *list = constraints;

			while (list) {
				gnome_db_database_add_constraint_real (mgdb, GNOME_DB_CONSTRAINT (list->data), FALSE);
				g_object_set (G_OBJECT (list->data), "user_constraint", FALSE, NULL);
				g_object_unref (G_OBJECT (list->data));
				list = g_slist_next (list);
			}
			g_slist_free (constraints);
			g_object_set_data (G_OBJECT (table), "pending_constraints", NULL);
		}

		g_signal_emit_by_name (G_OBJECT (mgdb), "update_progress", "TABLES",
				       now, total);
		now++;
	}
	
	g_object_unref (G_OBJECT (rs));
	
	/* remove the tables not existing anymore */
	list = mgdb->priv->tables;
	while (list) {
		if (!g_slist_find (updated_tables, list->data)) {
			gnome_db_base_nullify (GNOME_DB_BASE (list->data));
			list = mgdb->priv->tables;
		}
		else
			list = g_slist_next (list);
	}
	g_slist_free (updated_tables);
	
	g_signal_emit_by_name (G_OBJECT (mgdb), "update_progress", NULL, 0, 0);

	/* activate all the constraints here */
	list = mgdb->priv->constraints;
	while (list) {
		if (!gnome_db_referer_activate (GNOME_DB_REFERER (list->data))) {
			gnome_db_base_nullify (GNOME_DB_BASE (list->data));
			list = mgdb->priv->constraints;
		}
		else
			list = g_slist_next (list);
	}
	
	return TRUE;
}


static void
nullified_table_cb (GnomeDbTable *table, GnomeDbDatabase *mgdb)
{
	gchar *str;
	g_return_if_fail (g_slist_find (mgdb->priv->tables, table));

	mgdb->priv->tables = g_slist_remove (mgdb->priv->tables, table);

	g_signal_handlers_disconnect_by_func (G_OBJECT (table), 
					      G_CALLBACK (nullified_table_cb), mgdb);
	g_signal_handlers_disconnect_by_func (G_OBJECT (table), 
					      G_CALLBACK (updated_table_cb), mgdb);

	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (table));
	g_hash_table_remove (mgdb->priv->tables_hash, str);
	g_free (str);

#ifdef debug_signal
	g_print (">> 'TABLE_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "table_removed", table);
#ifdef debug_signal
	g_print ("<< 'TABLE_REMOVED' from %s\n", __FUNCTION__);
#endif

	g_object_unref (G_OBJECT (table));
}

static void
updated_table_cb (GnomeDbTable *table, GnomeDbDatabase *mgdb)
{
#ifdef debug_signal
	g_print (">> 'TABLE_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "table_updated", table);
#ifdef debug_signal
	g_print ("<< 'TABLE_UPDATED' from %s\n", __FUNCTION__);
#endif	
}

static gboolean
database_sequences_update_list (GnomeDbDatabase *mgdb, GError **error)
{
	TO_IMPLEMENT;
	
	return TRUE;
}

static void
nullified_sequence_cb (GnomeDbSequence *seq, GnomeDbDatabase *mgdb)
{
	g_return_if_fail (g_slist_find (mgdb->priv->sequences, seq));
	mgdb->priv->sequences = g_slist_remove (mgdb->priv->sequences, seq);

	g_signal_handlers_disconnect_by_func (G_OBJECT (seq), 
					      G_CALLBACK (nullified_sequence_cb), mgdb);
	g_signal_handlers_disconnect_by_func (G_OBJECT (seq), 
					      G_CALLBACK (updated_sequence_cb), mgdb);

#ifdef debug_signal
	g_print (">> 'SEQUENCE_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "sequence_removed", seq);
#ifdef debug_signal
	g_print ("<< 'SEQUENCE_REMOVED' from %s\n", __FUNCTION__);
#endif

	g_object_unref (G_OBJECT (seq));
}


static void
updated_sequence_cb (GnomeDbSequence *seq, GnomeDbDatabase *mgdb)
{
#ifdef debug_signal
	g_print (">> 'SEQUENCE_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "sequence_updated", seq);
#ifdef debug_signal
	g_print ("<< 'SEQUENCE_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


static gboolean
database_constraints_update_list (GnomeDbDatabase *mgdb, GError **error)
{
	TO_IMPLEMENT;
	/* To be implemented when Libgda has a dedicated schema to fetch constraints */

	return TRUE;
}


static void
nullified_constraint_cb (GnomeDbConstraint *cons, GnomeDbDatabase *mgdb)
{
	g_return_if_fail (g_slist_find (mgdb->priv->constraints, cons));
	mgdb->priv->constraints = g_slist_remove (mgdb->priv->constraints, cons);

	g_signal_handlers_disconnect_by_func (G_OBJECT (cons), 
					      G_CALLBACK (nullified_constraint_cb), mgdb);
	g_signal_handlers_disconnect_by_func (G_OBJECT (cons), 
					      G_CALLBACK (updated_constraint_cb), mgdb);
	
	if (mgdb->priv->constraints_hash) {
		GnomeDbTable *table;
		GSList *list;

		table = gnome_db_constraint_get_table (cons);
		list = g_hash_table_lookup (mgdb->priv->constraints_hash, table);
		list = g_slist_remove (list, cons);
		if (list)
			g_hash_table_insert (mgdb->priv->constraints_hash, table, list);
		else
			g_hash_table_remove (mgdb->priv->constraints_hash, table);
	}

#ifdef debug_signal
	g_print (">> 'CONSTRAINT_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "constraint_removed", cons);
#ifdef debug_signal
	g_print ("<< 'CONSTRAINT_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_object_unref (G_OBJECT (cons));
}

static void
updated_constraint_cb (GnomeDbConstraint *cons, GnomeDbDatabase *mgdb)
{
#ifdef debug_signal
	g_print (">> 'CONSTRAINT_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (mgdb), "constraint_updated", cons);
#ifdef debug_signal
	g_print ("<< 'CONSTRAINT_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


/**
 * gnome_db_database_get_tables
 * @mgdb: a #GnomeDbDatabase object
 *
 * Get a list of all the tables within @mgdb
 *
 * Returns: a new list of all the #GnomeDbTable objects
 */
GSList *
gnome_db_database_get_tables(GnomeDbDatabase *mgdb)
{
	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);

	return g_slist_copy (mgdb->priv->tables);
}

/**
 * gnome_db_database_get_table_by_name
 * @mgdb: a #GnomeDbDatabase object
 * @name: the name of the requested table
 *
 * Get a reference to a GnomeDbTable using its name.
 *
 * Returns: The GnomeDbTable pointer or NULL if the requested table does not exist.
 */
GnomeDbTable *
gnome_db_database_get_table_by_name (GnomeDbDatabase *mgdb, const gchar *name)
{
	GnomeDbTable *table = NULL;
	GSList *tables;
	gchar *cmpstr = NULL;
	gchar *cmpstr2;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);
	g_return_val_if_fail (name && *name, NULL);

	if (mgdb->priv->lc_names)
		cmpstr = g_utf8_strdown (name, -1);
	else
		cmpstr = name;

	tables = mgdb->priv->tables;
	while (!table && tables) {
		if (mgdb->priv->lc_names) {
			cmpstr2 = g_utf8_strdown (gnome_db_base_get_name (GNOME_DB_BASE (tables->data)), -1);
			if (!strcmp (cmpstr2, cmpstr))
				table = GNOME_DB_TABLE (tables->data);
			g_free (cmpstr2);
		}
		else
			if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (tables->data)), cmpstr))
				table = GNOME_DB_TABLE (tables->data);

		tables = g_slist_next (tables);
	}

	if (mgdb->priv->lc_names)
		g_free (cmpstr);
	
	return table;
}

/**
 * gnome_db_database_get_table_by_xml_id
 * @mgdb: a #GnomeDbDatabase object
 * @xml_id: the XML id of the requested table
 *
 * Get a reference to a GnomeDbTable using its XML id.
 *
 * Returns: The GnomeDbTable pointer or NULL if the requested table does not exist.
 */
GnomeDbTable *
gnome_db_database_get_table_by_xml_id (GnomeDbDatabase *mgdb, const gchar *xml_id)
{
	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);

	return g_hash_table_lookup (mgdb->priv->tables_hash, xml_id);
}

/**
 * gnome_db_database_get_field_by_name
 * @mgdb: a #GnomeDbDatabase object
 * @fullname: the name of the requested table field
 *
 * Get a reference to a GnomeDbField specifying the full name (table_name.field_name)
 * of the requested field.
 *
 * Returns: The GnomeDbField pointer or NULL if the requested field does not exist.
 */
GnomeDbField *
gnome_db_database_get_field_by_name (GnomeDbDatabase *mgdb, const gchar *fullname)
{
	gchar *str, *tok, *ptr;
	GnomeDbTable *ref_table;
	GnomeDbField *ref_field = NULL;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (fullname && *fullname, NULL);

	str = g_strdup (fullname);
	ptr = strtok_r (str, ".", &tok);
	ref_table = gnome_db_database_get_table_by_name (mgdb, ptr);
	if (ref_table) {
		gpointer data;
		ptr = strtok_r (NULL, ".", &tok);
		if (ptr) {
			data = gnome_db_entity_get_field_by_name (GNOME_DB_ENTITY (ref_table), ptr);
			if (data)
				ref_field = GNOME_DB_FIELD (data);
		}
	}
	g_free (str);

	return ref_field;
}


/**
 * gnome_db_database_get_field_by_xml_id
 * @mgdb: a #GnomeDbDatabase object
 * @xml_id: the XML id of the requested table field
 *
 * Get a reference to a GnomeDbField specifying its XML id
 *
 * Returns: The GnomeDbField pointer or NULL if the requested field does not exist.
 */
GnomeDbField *
gnome_db_database_get_field_by_xml_id (GnomeDbDatabase *mgdb, const gchar *xml_id)
{
	GnomeDbField *field = NULL;
	GSList *tables;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);

	tables = mgdb->priv->tables;
	while (tables && !field) {
		gpointer data;
		data = gnome_db_entity_get_field_by_xml_id (GNOME_DB_ENTITY (tables->data), xml_id);
		if (data)
			field = GNOME_DB_FIELD (data);
		tables = g_slist_next (tables);
	}

	return field;
}

/**
 * gnome_db_database_get_sequence_by_name
 * @mgdb: a #GnomeDbDatabase object
 * @name: the name of the requested sequence
 *
 * Get a reference to a GnomeDbSequence specifying its name
 *
 * Returns: The GnomeDbSequence pointer or NULL if the requested sequence does not exist.
 */
GnomeDbSequence *
gnome_db_database_get_sequence_by_name (GnomeDbDatabase *mgdb, const gchar *name)
{
	GnomeDbSequence *seq = NULL;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (name && *name, NULL);

	TO_IMPLEMENT;
	return seq;
}

/**
 * gnome_db_database_get_sequence_by_xml_id
 * @mgdb: a #GnomeDbDatabase object
 * @xml_id: the XML id of the requested sequence
 *
 * Get a reference to a GnomeDbSequence specifying its XML id.
 *
 * Returns: The GnomeDbSequence pointer or NULL if the requested sequence does not exist.
 */
GnomeDbSequence *
gnome_db_database_get_sequence_by_xml_id (GnomeDbDatabase *mgdb, const gchar *xml_id)
{
	GnomeDbSequence *seq = NULL;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);

	TO_IMPLEMENT;
	return seq;
}

/** 
 * gnome_db_database_get_sequence_to_field
 * @mgdb: a #GnomeDbDatabase object
 * @field: a #GnomeDbField object
 *
 * Get a reference to a GnomeDbSequence which is "linked" to the GnomeDbField given as parameter.
 * This "link" means that each new value of the field will be given by the returned sequence
 *
 * Returns: The GnomeDbSequence pointer or NULL if there is no sequence for this job.
 */
GnomeDbSequence *
gnome_db_database_get_sequence_to_field  (GnomeDbDatabase *mgdb, GnomeDbField *field)
{
	GnomeDbSequence *seq = NULL;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (field && IS_GNOME_DB_FIELD (field), NULL);

	TO_IMPLEMENT;
	return seq;
}


/**
 * gnome_db_database_link_sequence
 * @mgdb: a #GnomeDbDatabase object
 * @seq:
 * @field:
 *
 * Tells the database that each new value of the field given as argument should be
 * obtained from the specified sequence (this is usefull when the field is a simple
 * primary key for example).
 */
void
gnome_db_database_link_sequence (GnomeDbDatabase *mgdb, GnomeDbSequence *seq, GnomeDbField *field)
{
	g_return_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb));
	/*g_return_if_fail (seq && IS_GNOME_DB_SEQUENCE (seq));*/
	g_return_if_fail (field && IS_GNOME_DB_FIELD (field));

	TO_IMPLEMENT;	
}

/**
 * gnome_db_database_unlink_sequence
 * @mgdb: a #GnomeDbDatabase object
 * @seq:
 * @field:
 *
 * Tells the database that each new value of the field given as argument should not be
 * obtained from the specified sequence (this is usefull when the field is a simple
 * primary key for example). This is the opposite of the gnome_db_database_link_sequence() method.
 */
void
gnome_db_database_unlink_sequence (GnomeDbDatabase *mgdb, GnomeDbSequence *seq, GnomeDbField *field)
{
	g_return_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb));
	/*g_return_if_fail (seq && IS_GNOME_DB_SEQUENCE (seq));*/
	g_return_if_fail (field && IS_GNOME_DB_FIELD (field));

	TO_IMPLEMENT;	
}

/**
 * gnome_db_database_get_all_constraints
 * @mgdb: a #GnomeDbDatabase object
 *
 * Get a list of all the constraints applied to the database. Constraints are represented
 * as #GnomeDbConstraint objects and represent any type of constraint.
 *
 * Returns: a new list of the constraints
 */
GSList *
gnome_db_database_get_all_constraints (GnomeDbDatabase *mgdb)
{
	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);

	return g_slist_copy (mgdb->priv->constraints);
}

/**
 * gnome_db_database_get_table_constraints
 * @mgdb: a #GnomeDbDatabase object
 * @table: a #GnomeDbTable, part of @mgdb
 *
 * Get all the constraints applicable to @table
 *
 * Returns: a new GSList of #GnomeDbConstraint objects
 */
GSList *
gnome_db_database_get_table_constraints (GnomeDbDatabase *mgdb, GnomeDbTable *table)
{
	GSList *list;
	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);

	list = g_hash_table_lookup (mgdb->priv->constraints_hash, table);
	if (list) 
		return g_slist_copy (list);
	else
		return NULL;
}

/**
 * gnome_db_database_get_all_fk_constraints
 * @mgdb: a #GnomeDbDatabase object
 *
 * Get a list of all the constraints applied to the database which represent a foreign constrains. 
 * Constraints are represented as #GnomeDbConstraint objects.
 *
 * Returns: a new list of the constraints
 */
GSList *
gnome_db_database_get_all_fk_constraints (GnomeDbDatabase *mgdb)
{
	GSList *retval = NULL, *list;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);

	list = mgdb->priv->constraints;
	while (list) {
		if (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == 
		    CONSTRAINT_FOREIGN_KEY)
			retval = g_slist_append (retval, list->data);

		list = g_slist_next (list);
	}

	return retval;
}

/**
 * gnome_db_database_get_tables_fk_constraints
 * @mgdb: a #GnomeDbDatabase object
 * @table1: a #GnomeDbTable, or %NULL
 * @table2: a #GnomeDbTable, or %NULL
 * @table1_has_fk: TRUE if the returned constraints are the one for which @table1 contains the foreign key
 *
 * Get a list of all the constraints applied to the database which represent a foreign key constrains, between
 * @table1 and @table2. If @table1 or @table2 are %NULL, then the returned foreign key constraints are the ones
 * between any table and @table1 or @table2.
 *
 * Constraints are represented as #GnomeDbConstraint objects.
 *
 * Returns: a new list of the constraints
 */
GSList *
gnome_db_database_get_tables_fk_constraints (GnomeDbDatabase *mgdb, GnomeDbTable *table1, GnomeDbTable *table2, 
				       gboolean table1_has_fk)
{
	GSList *retval = NULL, *list;
	GnomeDbConstraint *fkptr;

	g_return_val_if_fail (mgdb && IS_GNOME_DB_DATABASE (mgdb), NULL);
	g_return_val_if_fail (mgdb->priv, NULL);
	if (table1)
		g_return_val_if_fail (IS_GNOME_DB_TABLE (table1), NULL);
	if (table2)
		g_return_val_if_fail (IS_GNOME_DB_TABLE (table2), NULL);
	if (!table1 && !table2)
		return NULL;

	if (table1_has_fk) {
		if (table1) {
			/* we want the FK constraints where @table1 is the FK table */
			list = g_hash_table_lookup (mgdb->priv->constraints_hash, table1);
			while (list) {
				fkptr = GNOME_DB_CONSTRAINT (list->data);
				if ((gnome_db_constraint_get_constraint_type (fkptr) == CONSTRAINT_FOREIGN_KEY) &&
				    (!table2 || (gnome_db_constraint_fkey_get_ref_table (fkptr) ==  (GnomeDbTable *) table2)))
					retval = g_slist_append (retval, fkptr);
				list = g_slist_next (list);
			}
		}
		else {
			/* we want all the FK constraints where @table2 is the ref_pk table */
			list = mgdb->priv->constraints;
			while (list) {
				fkptr = GNOME_DB_CONSTRAINT (list->data);
				if ((gnome_db_constraint_get_constraint_type (fkptr) == CONSTRAINT_FOREIGN_KEY) &&
				    (gnome_db_constraint_fkey_get_ref_table (fkptr) ==  (GnomeDbTable *) table2))
					retval = g_slist_append (retval, fkptr);
				list = g_slist_next (list);
			}
		}
	}
	else {
		list = mgdb->priv->constraints;
		while (list) {
			fkptr = GNOME_DB_CONSTRAINT (list->data);
			if (gnome_db_constraint_get_constraint_type (fkptr) == CONSTRAINT_FOREIGN_KEY) {
				GnomeDbTable *fk_table = gnome_db_constraint_get_table (fkptr);
				GnomeDbTable *ref_pk_table = gnome_db_constraint_fkey_get_ref_table (fkptr);
								
				if (((!table1 || (fk_table == table1)) && (!table2 || (ref_pk_table == table2))) ||
				    ((!table1 || (ref_pk_table == table1)) && (!table2 || (fk_table == table2))) )
					retval = g_slist_append (retval, fkptr);
			}
				
			list = g_slist_next (list);
		}
	}

	return retval;
}
