/* gnome-db-form.c
 *
 * Copyright (C) 2002 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <libgnomedb/libgnomedb.h>
#include "gnome-db-server.h"
#include "gnome-db-form.h"
#include "gnome-db-data-widget.h"
#include "gnome-db-query.h"
#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-result-set.h"
#include "gnome-db-basic-form.h"
#include "gnome-db-parameter.h"
#include "gnome-db-data-set.h"
#include "gnome-db-qfield.h"
#include "utility.h"
#include "gnome-db-util.h"

#ifdef debug
#include "gnome-db-graphviz.h"
#endif

static void gnome_db_form_class_init (GnomeDbFormClass * class);
static void gnome_db_form_init (GnomeDbForm * wid);
static void gnome_db_form_dispose (GObject   * object);

static void gnome_db_form_set_property (GObject              *object,
				       guint                 param_id,
				       const GValue         *value,
				       GParamSpec           *pspec);
static void gnome_db_form_get_property (GObject              *object,
				       guint                 param_id,
				       GValue               *value,
				       GParamSpec           *pspec);

static void gnome_db_form_initialize (GnomeDbForm *form, GnomeDbQuery *orig_query, GtkWidget *layout, GHashTable *box_widgets);

static GtkWidget *modif_buttons_make (GnomeDbForm *form);
static void       modif_buttons_update (GnomeDbForm *form);

static void       gnome_db_form_set_params_from_cursor (GnomeDbForm *form);
static void       data_set_param_changed_cb (GnomeDbBasicForm *simple_form, GnomeDbParameter *param, gboolean is_user_modif, GnomeDbForm *form);

static void model_row_changed_cb  (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, GnomeDbForm *form);
static void model_row_deleted_cb  (GtkTreeModel *treemodel, GtkTreePath *path, GnomeDbForm *form);
static void model_row_inserted_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, GnomeDbForm *form);
static void model_rows_reordered_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, gint *new_order, GnomeDbForm *form);


static void arrow_actions_real_do (GnomeDbForm *form, gint movement);

/* GtkTreeIter <-> cursor conversions */
static gboolean get_iter_for_cursor (GnomeDbForm *form, GtkTreeIter *iter);
static gint     get_cursor_from_iter (GnomeDbForm *form, GtkTreeIter *iter);
static gint     get_cursor_from_path (GnomeDbForm *form, GtkTreePath *path);

/* GnomeDbDataWidget interface */
static void            gnome_db_form_widget_init         (GnomeDbDataWidgetIface *iface);
static void            gnome_db_form_set_mode            (GnomeDbDataWidget *iface, guint mode);
static void            gnome_db_form_col_set_show        (GnomeDbDataWidget *iface, gint column, gboolean shown);
static void            gnome_db_form_set_column_editable (GnomeDbDataWidget *iface, gint column, gboolean editable);
static void            gnome_db_form_show_column_actions (GnomeDbDataWidget *iface, gint column, gboolean show_actions);
static void            gnome_db_form_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions);
static GtkActionGroup *gnome_db_form_get_actions_group   (GnomeDbDataWidget *iface);
static GnomeDbDataSet *gnome_db_form_widget_get_params_set (GnomeDbDataWidget *iface);
static GnomeDbDataSet *gnome_db_form_widget_get_data_set   (GnomeDbDataWidget *iface);


struct _GnomeDbFormPriv
{
	GnomeDbDataProxy  *proxy;
	GnomeDbDataModel  *data_model;
	GnomeDbDataSet    *data_set;

	guint              mode;
	GtkTooltips       *tooltips;
	guint              cursor;   /* row number of priv->proxy which is displayed */

	GtkWidget         *title;
	GtkWidget         *notebook;
	GtkWidget         *basic_form;
	
	GtkUIManager      *uimanager;
	GtkActionGroup    *actions_group;
	GtkWidget         *modif_all;
	GtkWidget         *nav_scale;
	GtkWidget         *nav_current;
	
	gboolean           intern_form_changes;
};

#define PAGE_NO_DATA 0
#define PAGE_FORM    1

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

/* properties */
enum
{
        PROP_0,
	PROP_TITLE_VISIBLE,
	PROP_TITLE_STRING,
	PROP_ACTIONS_VISIBLE
};

GType
gnome_db_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_form_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbForm),
			0,
			(GInstanceInitFunc) gnome_db_form_init
		};		
		
		static const GInterfaceInfo work_widget_info = {
                        (GInterfaceInitFunc) gnome_db_form_widget_init,
                        NULL,
                        NULL
                };
		
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbForm", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_TYPE_DATA_WIDGET, &work_widget_info);
	}

	return type;
}

static void
gnome_db_form_widget_init (GnomeDbDataWidgetIface *iface)
{
	iface->set_mode = gnome_db_form_set_mode;
	iface->col_set_show = gnome_db_form_col_set_show;
	iface->set_column_editable = gnome_db_form_set_column_editable;
	iface->show_column_actions = gnome_db_form_show_column_actions;
	iface->show_global_actions = gnome_db_form_show_global_actions;
	iface->get_actions_group = gnome_db_form_get_actions_group;
	iface->get_params_set = gnome_db_form_widget_get_params_set;
	iface->get_data_set = gnome_db_form_widget_get_data_set;
}


static void
gnome_db_form_class_init (GnomeDbFormClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	object_class->dispose = gnome_db_form_dispose;

	/* Properties */
        object_class->set_property = gnome_db_form_set_property;
        object_class->get_property = gnome_db_form_get_property;
	g_object_class_install_property (object_class, PROP_ACTIONS_VISIBLE,
                                         g_param_spec_boolean ("actions_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_VISIBLE,
                                         g_param_spec_boolean ("title_visible", NULL, NULL, FALSE,
                                                               G_PARAM_WRITABLE));
	g_object_class_install_property (object_class, PROP_TITLE_STRING,
                                         g_param_spec_string ("title_string", NULL, NULL, NULL,
							      G_PARAM_WRITABLE));
}

static void
gnome_db_form_init (GnomeDbForm * wid)
{
	wid->priv = g_new0 (GnomeDbFormPriv, 1);

	wid->priv->notebook = NULL;
	wid->priv->basic_form = NULL;

	wid->priv->cursor = 0;

	wid->priv->mode = 0;
	wid->priv->tooltips = NULL;
}

/**
 * gnome_db_form_new
 *
 * Creates a new #GnomeDbForm widget
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_form_new (void)
{
	GObject *obj;
	
	obj = g_object_new (GNOME_DB_TYPE_FORM, NULL);
	
	return GTK_WIDGET (obj);	
}

/**
 * gnome_db_form_new_with_gda_model
 * @dict: a #GnomeDbDict dictionary, or %NULL
 * @model: a #GdaDataModel
 *
 * Creates a new #GnomeDbForm widget suitable to display the data in @model
 *
 *  Returns: the new widget
 */
GtkWidget *
gnome_db_form_new_with_gda_model (GnomeDbDict *dict, GdaDataModel *model)
{
	TO_IMPLEMENT;
	return NULL;
}


/**
 * gnome_db_form_new_with_select_query
 * @query: a #GnomeDbQuery object
 * @modified: a #GnomeDbTarget object, or %NULL
 *
 * Creates a new #GnomeDbForm widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #GnomeDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_form_new_with_select_query (GnomeDbQuery *query, GnomeDbTarget *modified)
{
	return gnome_db_form_new_in_layout (query, modified, NULL, NULL);
}

/**
 * gnome_db_form_new_in_layout
 * @query: a #GnomeDbQuery object
 * @modified: a #GnomeDbTarget object, or %NULL
 * @layout: a #GtkWidget object
 * @box_widgets: a #GHashTable of #GtkBox widgets
 *
 * Creates a new #GnomeDbForm widget.
 *
 * @query must be a SELECT query (no union, etc selection query)
 *
 * The @modified target must belong to @query and represent
 * modifiable entity (a #GnomeDbTable for example). If @modified is %NULL then
 * no modification will be allowed.
 *
 * This function is similar to gnome_db_form_new() but provides a #GtkWidget to pack
 * entries in. The @box_widgets hash table has keys corresponding to the
 * query fields of @query, and corresponding values pointing to the #GtkBox widgets
 * where the #MGDataEntry widgets will be packed.
 *
 * If any of @layout or @box_widgets is %NULL, then this function is equivalent to gnome_db_form_new().
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_form_new_in_layout (GnomeDbQuery *query, GnomeDbTarget *modified, GtkWidget *layout, GHashTable *box_widgets)
{
	GObject *obj;
	GnomeDbForm *form;
	GnomeDbResultSet *rs;
	GnomeDbDataProxy *proxy;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (gnome_db_query_get_query_type (query) == GNOME_DB_QUERY_TYPE_SELECT, NULL);
	
	if (modified) {
		g_return_val_if_fail (IS_GNOME_DB_TARGET (modified), NULL);
		g_return_val_if_fail (gnome_db_target_get_query (modified) == query, NULL);
		g_return_val_if_fail (gnome_db_entity_is_writable (gnome_db_target_get_represented_entity (modified)), NULL);
	}
	
	obj = g_object_new (GNOME_DB_TYPE_FORM, NULL);
	form = GNOME_DB_FORM (obj);

	rs = GNOME_DB_RESULT_SET (gnome_db_result_set_new (query, modified));
	proxy = GNOME_DB_DATA_PROXY (gnome_db_data_proxy_new (GNOME_DB_DATA_MODEL (rs)));
	g_object_unref (rs);
	form->priv->data_model = GNOME_DB_DATA_MODEL (rs);
	form->priv->proxy = proxy;

	form->priv->data_set = gnome_db_data_model_get_new_data_set (GNOME_DB_DATA_MODEL (rs));
	
	/* catch signals from the GtkTreeModel */
	g_signal_connect (G_OBJECT (form->priv->proxy), "row-changed",
			  G_CALLBACK (model_row_changed_cb), form);
	g_signal_connect (G_OBJECT (form->priv->proxy), "row-deleted",
			  G_CALLBACK (model_row_deleted_cb), form);
	g_signal_connect (G_OBJECT (form->priv->proxy), "row-inserted",
			  G_CALLBACK (model_row_inserted_cb), form);
	g_signal_connect (G_OBJECT (form->priv->proxy), "rows-reordered",
			  G_CALLBACK (model_rows_reordered_cb), form);

	gnome_db_form_initialize (form, query, layout, box_widgets);

	return GTK_WIDGET (obj);
}

static void
gnome_db_form_dispose (GObject *object)
{
	GnomeDbForm *form;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_FORM (object));
	form = GNOME_DB_FORM (object);

	if (form->priv) {
		/* private data set */
		if (form->priv->data_set) {
			g_object_unref (form->priv->data_set);
			form->priv->data_set = NULL;
		}
		
		/* proxy */
		if (form->priv->proxy) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->proxy), G_CALLBACK (model_row_changed_cb), form);
			g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->proxy), G_CALLBACK (model_row_deleted_cb), form);
			g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->proxy), G_CALLBACK (model_row_inserted_cb), form);
			g_signal_handlers_disconnect_by_func (G_OBJECT (form->priv->proxy), G_CALLBACK (model_rows_reordered_cb), form);

			g_object_unref (form->priv->proxy);
			form->priv->proxy = NULL;
		}

		/* tooltips */
		if (form->priv->tooltips) {
			gtk_object_destroy (GTK_OBJECT (form->priv->tooltips));
			form->priv->tooltips = NULL;
		}

		/* UI */
		if (form->priv->actions_group) 
			g_object_unref (G_OBJECT (form->priv->actions_group));

		if (form->priv->uimanager)
			g_object_unref (G_OBJECT (form->priv->uimanager));
		
		/* the private area itself */
		g_free (form->priv);
		form->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

static void
gnome_db_form_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	GnomeDbForm *form;

        form = GNOME_DB_FORM (object);
        if (form->priv) {
                switch (param_id) {
                case PROP_TITLE_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (form->priv->title);
			else
				gtk_widget_hide (form->priv->title);
                        break;
                case PROP_TITLE_STRING:
			gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (form->priv->title), g_value_get_string (value));
			gtk_widget_show (form->priv->title);
                        break;
		case PROP_ACTIONS_VISIBLE:
			if (g_value_get_boolean (value))
				gtk_widget_show (form->priv->modif_all);
			else
				gtk_widget_hide (form->priv->modif_all);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }
}

static void
gnome_db_form_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	GnomeDbForm *form;

        form = GNOME_DB_FORM (object);
        if (form->priv) {
                switch (param_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
                }
        }	
}

/*
 * Fills @iter fith to point to the current form->priv->cursor
 */
static gboolean
get_iter_for_cursor (GnomeDbForm *form, GtkTreeIter *iter)
{
	GtkTreePath *path;
	gboolean retval;

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, form->priv->cursor);
	retval = gtk_tree_model_get_iter (GTK_TREE_MODEL (form->priv->proxy), iter, path);
	gtk_tree_path_free (path);

	return retval;
}

/*
 * get the cursor row corresponding to the @iter row 
 */
static gint
get_cursor_from_iter (GnomeDbForm *form, GtkTreeIter *iter)
{
	GtkTreePath *path;
	gint *indices;
	gint retval;
	
	path = gtk_tree_model_get_path (GTK_TREE_MODEL (form->priv->proxy), iter);
	indices = gtk_tree_path_get_indices (path);
	retval = indices[0];
	gtk_tree_path_free (path);

	return retval;
}

/*
 * get the cursor row corresponding to the @path row 
 */
static gint
get_cursor_from_path (GnomeDbForm *form, GtkTreePath *path)
{
	gint *indices;

	indices = gtk_tree_path_get_indices (path);
	return indices[0];
}


/*
 * Called when a parameter is changed within the GnomeDbBasicForm
 */
static void
data_set_param_changed_cb (GnomeDbBasicForm *simple_form, GnomeDbParameter *param, 
			   gboolean is_user_modif, GnomeDbForm *form)
{
	if (! form->priv->intern_form_changes) {
		/* propagate the param changes to the proxy */
		gint col;
		GtkTreeIter iter;
		guint hint;

		hint = GPOINTER_TO_UINT (g_hash_table_lookup (form->priv->data_set->hints, param));
		if (hint & GNOME_DB_DATA_SET_PARAM_READ_ONLY)
			return;

		if (!get_iter_for_cursor (form, &iter))
			return;

		col = gnome_db_data_model_get_column_at_param (form->priv->data_model, form->priv->data_set, param);
#ifdef debug
		if (col < 0) {
			g_print ("_______________ COL < 0 for param %p_____________\n", param);
			g_print ("_______________ data set _____________\n");
			gnome_db_base_dump (GNOME_DB_BASE (form->priv->data_set), 0);
			g_print ("_______________ model _____________\n");
			gnome_db_base_dump (GNOME_DB_BASE (form->priv->data_model), 0);
			
		}
#endif
		g_assert (col >= 0);
		gnome_db_data_proxy_set_value (form->priv->proxy, &iter, col, gnome_db_parameter_get_value (param));
		modif_buttons_update (form);

		/* REM:
		 * if @form->priv->data_model is a GnomeDbResultSet, then we should also change the values
		 * of @form->priv->proxy which are copies of values displayed in the GnomeDbEntryCombo
		 * (use form->priv->data_set->data_set_data_for_params_source);
		 * see GnomeDbGrid::data_cell_values_changed() for more information;
		 *
		 * OR
		 * Add a "changed" signal to GnomeDbEntryCombo with the same arguments as the "changed" signal
		 * of GnomeDbDataCellRendererCombo, and connect to that instead of monitoring the parameters change
		 * for parameters constrained by value in a GnomeDbDataModel .
		 */
	}
}

/*
 * Real initialization
 */
static void 
gnome_db_form_initialize (GnomeDbForm *form, GnomeDbQuery *orig_query, GtkWidget *layout, GHashTable *box_widgets)
{
	GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (form->priv->data_model));
	GtkWidget *realform, *wid, *nb, *group;
	GSList *list;

	/* title */
	form->priv->title = gnome_db_gray_bar_new (_("No title"));
        gtk_box_pack_start (GTK_BOX (form), form->priv->title, FALSE, TRUE, 2);
	gtk_widget_show (form->priv->title);


	/*
	 * the "No Data" notice
	 */
	nb = gtk_notebook_new ();
	form->priv->notebook = nb;
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (form), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	wid = gtk_label_new (_("No data to be displayed"));
	gtk_widget_show (wid);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), wid, NULL);

	/* 
	 * the form's attributes
	 */
	gtk_widget_hide (form->priv->title);

	/*
	 * the actions part
	 */
	group = modif_buttons_make (form);
	gtk_box_pack_start (GTK_BOX (form), group, FALSE, FALSE, 0);
	gtk_widget_show (group);

	/*
	 * handling of the layout
	 */
	if (layout && box_widgets) {
		/* REM: the @box_widgets has keys for query fields, and we need to provide
		   gnome_db_basic_form_new_in_layout() with a hash table which has keys for GnomeDbDataSetNode;
		   this is why we need to use another hash table: 'fbw' */

		TO_IMPLEMENT;
		/* GSList *list; */
/* 		GnomeDbParameter *param; */
/* 		GnomeDbDataSetNode *node; */
/* 		gpointer widget; */

/* 		fbw = g_hash_table_new (NULL, NULL); */
/* 		g_object_get (G_OBJECT (orig_query), "really_all_fields", &list, NULL); */
/* 		while (list) { */
/* 			widget = g_hash_table_lookup (box_widgets, list->data); */
/* 			if (widget) { */
/* 				node = gnome_db_work_core_find_context_node (form->priv->core, GNOME_DB_QFIELD (list->data)); */
/* 				if (node) */
/* 					g_hash_table_insert (fbw, node, widget); */
/* 			} */
/* 			list = g_slist_next (list); */
/* 		} */
	}

	/* 
	 * the form itself 
	 */
	realform = gnome_db_basic_form_new_in_layout (dict, form->priv->data_set, layout, box_widgets);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), realform, NULL);
	gtk_widget_show (realform);
	if (!layout && gnome_db_data_proxy_is_read_only (form->priv->proxy))
		gnome_db_basic_form_show_entries_actions (GNOME_DB_BASIC_FORM (realform), FALSE);
	form->priv->basic_form = realform;

	g_signal_connect (G_OBJECT (form->priv->basic_form), "param_changed",
			  G_CALLBACK (data_set_param_changed_cb), form);
	list = form->priv->data_set->parameters;
	while (list) {
		guint hint;
		hint = GPOINTER_TO_UINT (g_hash_table_lookup (form->priv->data_set->hints, list->data));
		if (hint & GNOME_DB_DATA_SET_PARAM_HIDE)
			gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (realform),
							GNOME_DB_PARAMETER (list->data), FALSE);
		list = g_slist_next (list);
	}
	

	/* tooltips */
	form->priv->tooltips = gtk_tooltips_new ();

	/* data display update */
	gnome_db_form_set_params_from_cursor (form);
	modif_buttons_update (form);
}

/*
 * updates all the parameters in form->priv->data_set to the values of form->priv->proxy
 * pointed by the form->priv->cursor cursor
 */
static void
gnome_db_form_set_params_from_cursor (GnomeDbForm *form)
{
	GtkTreeIter iter;
	GSList *list;
	
	if (! form->priv->data_set->nodes || !get_iter_for_cursor (form, &iter)) {
		gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_NO_DATA);
		return;
	}

	/* actual work */
	form->priv->intern_form_changes = TRUE;
	gnome_db_basic_form_set_entries_auto_default (GNOME_DB_BASIC_FORM (form->priv->basic_form), FALSE);

	if (gnome_db_data_proxy_get_n_rows (form->priv->proxy) == 0) {
		/* no data to be displayed */
		gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_NO_DATA);
		list = form->priv->data_set->parameters;
		while (list) {
			gnome_db_parameter_declare_invalid (GNOME_DB_PARAMETER (list->data));
			list = g_slist_next (list);
		}
	}
	else {
		/* there are some data to display */
		gint offset = gnome_db_data_proxy_get_n_columns (form->priv->proxy);

		gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_FORM);			

		/* preparing the original values */
		list = form->priv->data_set->parameters;
		while (list) {
			GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
			GdaValue *value;
			GnomeDbParameter *tmp;
			
			g_object_get (G_OBJECT (param), "full_bind", &tmp, NULL);
			if (! tmp) {
				gint col;
				
				col = gnome_db_data_model_get_column_at_param (form->priv->data_model, 
									       form->priv->data_set, param);
				gtk_tree_model_get (GTK_TREE_MODEL (form->priv->proxy), &iter, 
						    2*offset + col, &value, -1);
				gnome_db_parameter_set_value (param, value);
			}
			
			list = g_slist_next (list);
		}
		/* Tell the real form that the parameters, as they are now set, are the new default values */
		gnome_db_basic_form_set_current_as_orig (GNOME_DB_BASIC_FORM (form->priv->basic_form));

		/* actual values */
		list = form->priv->data_set->parameters;
		while (list) {
			GnomeDbParameter *param = GNOME_DB_PARAMETER (list->data);
			GdaValue *value;
			guint attributes;
			GnomeDbParameter *tmp;
			
			g_object_get (G_OBJECT (param), "full_bind", &tmp, NULL);
			if (! tmp) {
				guint hint;

				hint = GPOINTER_TO_UINT (g_hash_table_lookup (form->priv->data_set->hints, list->data));
				if (! (hint & GNOME_DB_DATA_SET_PARAM_READ_ONLY)) {
					gint col;
					col = gnome_db_data_model_get_column_at_param (form->priv->data_model, 
										       form->priv->data_set, param);
					gtk_tree_model_get (GTK_TREE_MODEL (form->priv->proxy), &iter, 
							    col, &value, 
							    offset + col, &attributes, -1);
					gnome_db_parameter_set_value (param, value);
					
					g_object_set (G_OBJECT (param), "use_default_value", 
						      attributes & GNOME_DB_VALUE_IS_DEFAULT ? TRUE : FALSE, NULL);
				}
			}
			
			list = g_slist_next (list);
		}
	}
	
	form->priv->intern_form_changes = FALSE;
}




/*
 *
 * Modification buttons (Commit changes, Reset form, New entry, Delete)
 *
 */
static void action_new_cb (GtkAction *action, GnomeDbForm *form);
static void action_delete_cb (GtkAction *action, GnomeDbForm *form);
static void action_undelete_cb (GtkAction *action, GnomeDbForm *form);
static void action_commit_cb (GtkAction *action, GnomeDbForm *form);
static void action_reset_cb (GtkAction *action, GnomeDbForm *form);
static void action_first_record_cb (GtkAction *action, GnomeDbForm *form);
static void action_prev_record_cb (GtkAction *action, GnomeDbForm *form);
static void action_next_record_cb (GtkAction *action, GnomeDbForm *form);
static void action_last_record_cb (GtkAction *action, GnomeDbForm *form);

static GtkActionEntry ui_actions[] = {
	{ "ActionNew", GTK_STOCK_NEW, "_New", NULL, "Create a new data entry", G_CALLBACK (action_new_cb)},
	{ "ActionDelete", GTK_STOCK_DELETE, "_Delete", NULL, "Delete the selected entry", G_CALLBACK (action_delete_cb)},
	{ "ActionUndelete", GTK_STOCK_UNDELETE, "_Undelete", NULL, "Cancels the deletion of the current entry", 
	  G_CALLBACK (action_undelete_cb)},	
	{ "ActionCommit", GTK_STOCK_SAVE, "_Commit", NULL, "Commit the latest changes", G_CALLBACK (action_commit_cb)},
	{ "ActionReset", GTK_STOCK_REFRESH, "_Reset", NULL, "Reset the data", G_CALLBACK (action_reset_cb)},
	{ "ActionFirstRecord", GTK_STOCK_GOTO_FIRST, "_First record", NULL, "Go to first record of records", 
	  G_CALLBACK (action_first_record_cb)},
	{ "ActionLastRecord", GTK_STOCK_GOTO_LAST, "_Last record", NULL, "Go to last record of records", 
	  G_CALLBACK (action_last_record_cb)},
	{ "ActionPrevRecord", GTK_STOCK_GO_BACK, "_Previous record", NULL, "Go to previous record of records", 
	  G_CALLBACK (action_prev_record_cb)},
	{ "ActionNextRecord", GTK_STOCK_GO_FORWARD, "Ne_xt record", NULL, "Go to next record of records",
	  G_CALLBACK (action_next_record_cb)}
};

static const gchar *ui_actions_info =
"<ui>"
"  <toolbar  name='ToolBar'>"
"    <toolitem action='ActionNew'/>"
"    <toolitem action='ActionDelete'/>"
"    <toolitem action='ActionUndelete'/>"
"    <toolitem action='ActionCommit'/>"
"    <toolitem action='ActionReset'/>"
"    <separator/>"
"    <toolitem action='ActionFirstRecord'/>"
"    <toolitem action='ActionPrevRecord'/>"
"    <toolitem action='ActionNextRecord'/>"
"    <toolitem action='ActionLastRecord'/>"
"  </toolbar>"
"</ui>";

static void actions_arrows_value_changed_cb (GtkRange *range, GnomeDbForm *form);
static GtkWidget *
modif_buttons_make (GnomeDbForm *form)
{
	GtkActionGroup *actions;
	GtkUIManager *ui;
	GtkWidget *hbox, *wid;

	hbox = gtk_hbox_new (FALSE, 0);
	
	/* action buttons */
	actions = gtk_action_group_new ("Actions");
	form->priv->actions_group = actions;

	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), form);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, actions, 0);
	gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);
	form->priv->uimanager = ui;
	form->priv->modif_all = gtk_ui_manager_get_widget (ui, "/ToolBar");
	g_object_set (G_OBJECT (form->priv->modif_all), "toolbar-style", GTK_TOOLBAR_ICONS, NULL);
	gtk_widget_show (form->priv->modif_all);
	gtk_box_pack_start (GTK_BOX (hbox), form->priv->modif_all, TRUE, TRUE, 0);

	/* Nav Scale */
	wid = gtk_hscale_new_with_range (0, 1, 1);
	gtk_range_set_update_policy (GTK_RANGE (wid), GTK_UPDATE_DELAYED);
	gtk_scale_set_draw_value (GTK_SCALE (wid), TRUE);
	gtk_scale_set_digits (GTK_SCALE (wid), 0);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 2);
	gtk_widget_show (wid);
	gtk_widget_set_sensitive (wid, FALSE);
	form->priv->nav_scale = wid;
	g_signal_connect (G_OBJECT (wid), "value_changed",
			  G_CALLBACK (actions_arrows_value_changed_cb), form);

	/* rows counter */
	wid = gtk_label_new ("? / ?");
	gtk_widget_show (wid);
	form->priv->nav_current = wid;
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 2);

	return hbox;
}

static void
action_new_cb (GtkAction *action, GnomeDbForm *form)
{
	GSList *list;
	GtkTreeIter iter;

	/* append a row in the proxy */
	gnome_db_data_proxy_append (form->priv->proxy, &iter);
	
	/* move to that new row */
	form->priv->cursor = get_cursor_from_iter (form, &iter);;

	/* set parameters to their default values */
	list = form->priv->data_set->parameters;
	while (list) {
		GnomeDbParameter *param;
		const GdaValue *value;
		
		g_object_get (G_OBJECT (list->data), "full_bind", &param, NULL);
		if (! param) {
			value = gnome_db_data_set_get_param_default_value (form->priv->data_set,
									   GNOME_DB_PARAMETER (list->data));
			gnome_db_parameter_set_value (GNOME_DB_PARAMETER (list->data), value);
		}
		
		list = g_slist_next (list);
	}
	
	gnome_db_basic_form_set_entries_default (GNOME_DB_BASIC_FORM (form->priv->basic_form));
	gnome_db_basic_form_set_entries_auto_default (GNOME_DB_BASIC_FORM (form->priv->basic_form), TRUE);

	/* updates */
	gtk_notebook_set_current_page (GTK_NOTEBOOK (form->priv->notebook), PAGE_FORM);
	modif_buttons_update (form);
}

static void
action_delete_cb (GtkAction *action, GnomeDbForm *form)
{
	GtkTreeIter iter;
	g_assert (get_iter_for_cursor (form, &iter));
	gnome_db_data_proxy_delete (form->priv->proxy, &iter);
}

static void
action_undelete_cb (GtkAction *action, GnomeDbForm *form)
{
	GtkTreeIter iter;
	g_assert (get_iter_for_cursor (form, &iter));
	gnome_db_data_proxy_undelete (form->priv->proxy, &iter);
}

static void
action_commit_cb (GtkAction *action, GnomeDbForm *form)
{
	gnome_db_data_proxy_commit_all (form->priv->proxy, NULL);
}

static void
action_reset_cb (GtkAction *action, GnomeDbForm *form)
{
	GnomeDbDataModel *model;

	/* reset the proxy */
	gnome_db_data_proxy_reset_all (form->priv->proxy);

	/* refresh the data if possible */
	model = gnome_db_data_proxy_get_model (form->priv->proxy);
	if (gnome_db_data_model_get_status (model) & GNOME_DB_DATA_MODEL_CAN_BE_REFRESHED)
		gnome_db_data_model_refresh (model, NULL);
}

static void
action_first_record_cb (GtkAction *action, GnomeDbForm *form)
{
	arrow_actions_real_do (form, -2);
}

static void
action_prev_record_cb (GtkAction *action, GnomeDbForm *form)
{
	arrow_actions_real_do (form, -1);
}

static void
action_next_record_cb (GtkAction *action, GnomeDbForm *form)
{
	arrow_actions_real_do (form, 1);
}

static void
action_last_record_cb (GtkAction *action, GnomeDbForm *form)
{
	arrow_actions_real_do (form, 2);
}

static void
actions_arrows_value_changed_cb (GtkRange *range, GnomeDbForm *form)
{
	gint value = gtk_range_get_value (GTK_RANGE (form->priv->nav_scale));
	if ((value >= 1) &&
	    (value <= gnome_db_data_proxy_get_n_rows (form->priv->proxy)))
		form->priv->cursor = value - 1;

	/* updates */
	gnome_db_form_set_params_from_cursor (form);
	modif_buttons_update (form);
}

static void
modif_buttons_update (GnomeDbForm *form)
{
	gint nrows = 0;
	gboolean changed;
	GtkAction *action;
	gboolean can_modif;
	gboolean to_be_deleted = FALSE;

	if (! form->priv->basic_form)
		return;

	/* action buttons */
	changed = gnome_db_data_proxy_has_been_modified (form->priv->proxy);
	can_modif = ! gnome_db_data_proxy_is_read_only (form->priv->proxy);

	if (form->priv->proxy) {
		GtkTreeIter iter;

		nrows = gnome_db_data_proxy_get_n_rows (form->priv->proxy);
		if (get_iter_for_cursor (form, &iter))
			gtk_tree_model_get (GTK_TREE_MODEL (form->priv->proxy), &iter, PROXY_COL_TO_DELETE, &to_be_deleted, -1);
		else
			nrows = 0;
	}

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionCommit");
	g_object_set (G_OBJECT (action), "sensitive", changed && can_modif ? TRUE : FALSE, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionReset");
	g_object_set (G_OBJECT (action), "sensitive", TRUE, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionNew");
	g_object_set (G_OBJECT (action), "sensitive", can_modif, NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionDelete");
	g_object_set (G_OBJECT (action), "sensitive", can_modif && !to_be_deleted && (nrows > 0), NULL);

	action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionUndelete");
	g_object_set (G_OBJECT (action), "sensitive", can_modif && to_be_deleted && (nrows > 0), NULL);

	/* navigation buttons */
	/* global mode */
	if (form->priv->mode & GNOME_DB_ACTION_NAVIGATION_ARROWS) {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionFirstRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionPrevRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionNextRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionLastRecord");
		g_object_set (G_OBJECT (action), "visible", TRUE, NULL);

		if (form->priv->mode & GNOME_DB_ACTION_NAVIGATION_SCROLL) 
			gtk_widget_show (form->priv->nav_scale);
		else 
			gtk_widget_hide (form->priv->nav_scale);
	}
	else {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionFirstRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionPrevRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionNextRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionLastRecord");
		g_object_set (G_OBJECT (action), "visible", FALSE, NULL);

		gtk_widget_hide (form->priv->nav_scale);
		gtk_widget_hide (form->priv->nav_current);
		return; /* END HERE */
	}

	/* sensitiveness of the widgets */
	if (nrows > 0) {
		gchar *str;
		
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionFirstRecord");
		g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == 0) ? FALSE : TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionPrevRecord");
		g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == 0) ? FALSE : TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionNextRecord");
		g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == nrows -1) ? FALSE : TRUE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionLastRecord");
		g_object_set (G_OBJECT (action), "sensitive", (form->priv->cursor == nrows -1) ? FALSE : TRUE, NULL);
		
		str = g_strdup_printf ("%d / %d", form->priv->cursor+1, nrows);
		gtk_label_set_text (GTK_LABEL (form->priv->nav_current), str);
		g_free (str);
		
		gtk_widget_set_sensitive (form->priv->nav_scale, nrows == 1 ? FALSE : TRUE);
		if (nrows == 1)
			gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 1, 2);
		else
			gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 1, nrows);
		
		if (gtk_range_get_value (GTK_RANGE (form->priv->nav_scale)) != form->priv->cursor+1)
			gtk_range_set_value (GTK_RANGE (form->priv->nav_scale), form->priv->cursor+1);
	}
	else {
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionFirstRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionPrevRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionNextRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		action = gtk_ui_manager_get_action (form->priv->uimanager, "/ToolBar/ActionLastRecord");
		g_object_set (G_OBJECT (action), "sensitive", FALSE, NULL);
		gtk_label_set_text (GTK_LABEL (form->priv->nav_current), "? / ?");
		gtk_widget_set_sensitive (form->priv->nav_scale, FALSE);
		gtk_range_set_range (GTK_RANGE (form->priv->nav_scale), 0, 1);
	}
}


static void
arrow_actions_real_do (GnomeDbForm *form, gint movement)
{
	/* see if some data have been modified and need to be written to the DBMS */
	if ((form->priv->mode & GNOME_DB_ACTION_MODIF_AUTO_COMMIT) &&
	    gnome_db_data_proxy_has_been_modified (form->priv->proxy))
		action_commit_cb (NULL, form);
	
	switch (movement) {
	case -2:
		form->priv->cursor = 0;
		break;
	case -1:
		if (form->priv->cursor > 0)
			form->priv->cursor--;
		break;
	case 1:
		if (form->priv->proxy &&
		    (form->priv->cursor < gnome_db_data_proxy_get_n_rows (form->priv->proxy) - 1 ))
			form->priv->cursor++;
		break;
	case 2:
		if (form->priv->proxy)
			form->priv->cursor = gnome_db_data_proxy_get_n_rows (form->priv->proxy) - 1;
		break;
	default:
		g_assert_not_reached ();
	}

	/* updates */
	gnome_db_form_set_params_from_cursor (form);
	modif_buttons_update (form);
}

static void
model_row_changed_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, GnomeDbForm *form)
{
	gint row;

	row = get_cursor_from_iter (form, iter);
	if (row == form->priv->cursor) {
		gnome_db_form_set_params_from_cursor (form);
		modif_buttons_update (form);
	}
}

static void
model_row_deleted_cb (GtkTreeModel *treemodel, GtkTreePath *path, GnomeDbForm *form)
{
	if (get_cursor_from_path (form, path) <= form->priv->cursor) {
		form->priv->cursor --;

		gnome_db_form_set_params_from_cursor (form);
		modif_buttons_update (form);
	}
}

static void
model_row_inserted_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, GnomeDbForm *form)
{
	TO_IMPLEMENT;
}

static void
model_rows_reordered_cb (GtkTreeModel *treemodel, GtkTreePath *path, GtkTreeIter *iter, gint *new_order, GnomeDbForm *form)
{
	TO_IMPLEMENT;
}


/*
 * GnomeDbDataWidget interface implementation
 */
static void
gnome_db_form_set_mode (GnomeDbDataWidget *iface, guint mode)
{
	GnomeDbForm *form;

	g_return_if_fail (iface && IS_GNOME_DB_FORM (iface));
	form = GNOME_DB_FORM (iface);
	g_return_if_fail (form->priv);

	form->priv->mode = mode;

	/* updating the various possible actions */
	modif_buttons_update (form);
}

static void
gnome_db_form_col_set_show (GnomeDbDataWidget *iface, gint column, gboolean shown)
{
	GnomeDbForm *form;
	GnomeDbParameter *param;

	g_return_if_fail (iface && IS_GNOME_DB_FORM (iface));
	form = GNOME_DB_FORM (iface);
	g_return_if_fail (form->priv);

	param = gnome_db_data_model_get_param_at_column (form->priv->data_model, form->priv->data_set, column);
	g_return_if_fail (param);
	gnome_db_basic_form_entry_show (GNOME_DB_BASIC_FORM (form->priv->basic_form), param, shown);
}

void
gnome_db_form_set_column_editable (GnomeDbDataWidget *iface, gint column, gboolean editable)
{
	GnomeDbForm *form;

	g_return_if_fail (iface && IS_GNOME_DB_FORM (iface));
	form = GNOME_DB_FORM (iface);
	g_return_if_fail (form->priv);

	TO_IMPLEMENT;
	/* What needs to be done:
	 * - create a gnome_db_basic_form_set_entry_editable() function for GnomeDbBasicForm, and call it
	 * - in the GnomeDbDataEntry, add a GNOME_DB_VALUE_EDITABLE property which defaults to TRUE.
	 * - imtplement the gnome_db_basic_form_set_entry_editable() in the same way as gnome_db_basic_form_set_entries_default()
	 *   by setting that new property
	 * - implement the new property in GnomeDbEntryWrapper and GnomeDbEntryCombo.
	 */
}

static void
gnome_db_form_show_column_actions (GnomeDbDataWidget *iface, gint column, gboolean show_actions)
{
	GnomeDbForm *form;
	
	g_return_if_fail (iface && IS_GNOME_DB_FORM (iface));
	form = GNOME_DB_FORM (iface);
	g_return_if_fail (form->priv);
	
	/* REM: don't take care of the @column argument */
	gnome_db_basic_form_show_entries_actions (GNOME_DB_BASIC_FORM (form->priv->basic_form), show_actions);
}

static void
gnome_db_form_show_global_actions (GnomeDbDataWidget *iface, gboolean show_actions)
{
	GnomeDbForm *form;

	g_return_if_fail (iface && IS_GNOME_DB_FORM (iface));
	form = GNOME_DB_FORM (iface);
	g_return_if_fail (form->priv);

	if (show_actions)
		gtk_widget_show (form->priv->modif_all);
	else
		gtk_widget_hide (form->priv->modif_all);
}

static GtkActionGroup *
gnome_db_form_get_actions_group (GnomeDbDataWidget *iface)
{
	GnomeDbForm *form;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_FORM (iface), NULL);
	form = GNOME_DB_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->actions_group;
}

static GnomeDbDataSet *
gnome_db_form_widget_get_params_set (GnomeDbDataWidget *iface)
{
	GnomeDbForm *form;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_FORM (iface), NULL);
	form = GNOME_DB_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);

	return gnome_db_data_model_get_params (form->priv->data_model);
}

static GnomeDbDataSet *
gnome_db_form_widget_get_data_set (GnomeDbDataWidget *iface)
{
	GnomeDbForm *form;
	
	g_return_val_if_fail (iface && IS_GNOME_DB_FORM (iface), NULL);
	form = GNOME_DB_FORM (iface);
	g_return_val_if_fail (form->priv, NULL);

	return form->priv->data_set;
}
