/* gnome-db-server.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-server.h"
#include "gnome-db-xml-storage.h"
#include "marshal.h"
#include <gmodule.h>
#include "gnome-db-result-set.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-server-function.h"
#include "gnome-db-server-aggregate.h"
#include "handlers/gnome-db-handler-string.h"
#include "handlers/gnome-db-handler-numerical.h"
#include "handlers/gnome-db-handler-boolean.h"
#include "handlers/gnome-db-handler-time.h"
#include "handlers/gnome-db-handler-none.h"
#include <string.h>
#include <dirent.h>

/* 
 * Main static functions 
 */
static void gnome_db_server_class_init (GnomeDbServerClass * class);
static void gnome_db_server_init (GnomeDbServer * srv);
static void gnome_db_server_dispose (GObject   * object);
static void gnome_db_server_finalize (GObject   * object);

static void gnome_db_server_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_server_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

static void        gnome_db_server_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_server_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_server_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_server_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

static GSList *handlers_get_initial_list (GnomeDbServer *srv);
static GSList *handlers_load_plugins (GnomeDbServer *srv, const gchar *path);

static void nullified_handler_cb (GnomeDbDataHandler *hdl, GnomeDbServer *srv);
static void nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServer *srv);
static void nullified_function_cb (GnomeDbServerFunction *func, GnomeDbServer *srv);
static void nullified_aggregate_cb (GnomeDbServerAggregate *agg, GnomeDbServer *srv);

static void updated_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServer *srv);
static void updated_function_cb (GnomeDbServerFunction *func, GnomeDbServer *srv);
static void updated_aggregate_cb (GnomeDbServerAggregate *agg, GnomeDbServer *srv);


static GnomeDbServerFunction *gnome_db_server_get_function_by_name_arg_real (GnomeDbServer *srv, GSList *functions, const gchar *funcname, const GSList *argtypes);
GnomeDbServerAggregate *gnome_db_server_get_aggregate_by_name_arg_real (GnomeDbServer *srv, GSList *aggregates, const gchar *aggname, GnomeDbServerDataType *argtype);


#define HANDLER_FUNCTION(x) ((GnomeDbDataHandler *(*) (GnomeDbServer *, GObject *)) x)
static GnomeDbDataHandler *func_handler_data_types (GnomeDbServer *srv, GObject *obj);

#define LC_NAMES(srv) ((srv)->priv->info && (srv)->priv->info->is_case_insensitive)


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	CONN_OPENED,
	CONN_TO_CLOSE,
	CONN_CLOSED,
	DATA_TYPE_ADDED,
	DATA_TYPE_REMOVED,
	DATA_TYPE_UPDATED,
	DATA_FUNCTION_ADDED,
	DATA_FUNCTION_REMOVED,
	DATA_FUNCTION_UPDATED,
	DATA_AGGREGATE_ADDED,
	DATA_AGGREGATE_REMOVED,
	DATA_AGGREGATE_UPDATED,
	DATA_UPDATE_STARTED,
	UPDATE_PROGRESS,
	DATA_UPDATE_FINISHED,
	OBJECT_HANDLER_UPDATED,
	LAST_SIGNAL
};

static gint gnome_db_server_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* properties */
enum
{
	PROP_0,
	PROP_WITH_FUNCTIONS
};

/* private structure */
struct _GnomeDbServerPrivate
{
	GnomeDbDict                 *dict;

	/* server description */
        GdaConnection          *cnc;
	gboolean                with_functions;
	GnomeDbServerInfo      *info;

	/* DBMS update related information */
	gboolean                update_in_progress;
	gboolean                stop_update; /* TRUE if a DBMS data update must be stopped */

	/* DBMS connectivity */
        GString                *gda_datasource;
        GString                *user_name;
        GString                *password;

	/* DBMS objects */
	GSList                 *users;
        GSList                 *data_types;
	GSList                 *custom_types; /* types here also listed in @data_types, not refed in this list */
        GSList                 *functions;
        GSList                 *aggregates;

	/* Data type handlers and plugins management */
        GSList                 *handlers;            /* list of DataHandler objects */
	GnomeDbDataHandler     *fallback_handler;    /* to be used when no other handler can do */
        GHashTable             *types_objects_hash;  /* hash to store the bindings of
							DataHandler objects to several objects */
        GSList                 *handlers_functions;
};


/* NOTE about referencing data type objects:
 *
 * There are 2 kinds of data types: the ones returned by the GdaConnection schema data types list and
 * the ones added by gnome_db_server_declare_custom_data_type().
 * 
 * Data types of the first category are:
 * -> listed in priv->data_types
 * -> referenced (using g_object_ref())
 * -> not listed in priv->custom_types
 *
 * Data types of the 2nd category (custom ones) are:
 * -> listed in priv->data_types
 * -> NOT referenced
 * -> listed in priv->custom_types
 *
 * When a database sync occurs, if a data type which was a custom data type is found, then
 * it becomes a data type of the 1st category.
 */

/* module error */
GQuark gnome_db_server_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_server_error");
	return quark;
}


GType
gnome_db_server_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbServerClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_server_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbServer),
			0,
			(GInstanceInitFunc) gnome_db_server_init
		};
		

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_server_xml_storage_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GDA_TYPE_CLIENT, "GnomeDbServer", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void 
gnome_db_server_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_server_get_xml_id;
	iface->save_to_xml = gnome_db_server_save_to_xml;
	iface->load_from_xml = gnome_db_server_load_from_xml;
}

static void
gnome_db_server_class_init (GnomeDbServerClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_server_signals[CONN_OPENED] =
                g_signal_new ("conn_opened",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, conn_opened),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gnome_db_server_signals[CONN_TO_CLOSE] =
                g_signal_new ("conn_to_close",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, conn_to_close),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gnome_db_server_signals[CONN_CLOSED] =    /* runs after user handlers */
                g_signal_new ("conn_closed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, conn_closed),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gnome_db_server_signals[DATA_TYPE_ADDED] =
                g_signal_new ("data_type_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_type_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_server_signals[DATA_TYPE_REMOVED] =
                g_signal_new ("data_type_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_type_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_server_signals[DATA_TYPE_UPDATED] =
                g_signal_new ("data_type_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_type_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_FUNCTION_ADDED] =
                g_signal_new ("data_function_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_function_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_FUNCTION_REMOVED] =
                g_signal_new ("data_function_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_function_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_FUNCTION_UPDATED] =
                g_signal_new ("data_function_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_function_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_AGGREGATE_ADDED] =
                g_signal_new ("data_aggregate_added",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_aggregate_added),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_AGGREGATE_REMOVED] =
                g_signal_new ("data_aggregate_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_aggregate_removed),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
        gnome_db_server_signals[DATA_AGGREGATE_UPDATED] =
                g_signal_new ("data_aggregate_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_aggregate_updated),
                              NULL, NULL,
                              marshal_VOID__POINTER,
                              G_TYPE_NONE, 1, G_TYPE_POINTER);
	gnome_db_server_signals[DATA_UPDATE_STARTED] =
                g_signal_new ("data_update_started",
                              G_TYPE_FROM_CLASS (object_class),
                                G_SIGNAL_RUN_FIRST,
                            G_STRUCT_OFFSET (GnomeDbServerClass, data_update_started),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
        gnome_db_server_signals[UPDATE_PROGRESS] =
                g_signal_new ("update_progress",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, update_progress),
                              NULL, NULL,
                              marshal_VOID__POINTER_UINT_UINT,
                              G_TYPE_NONE, 3, G_TYPE_POINTER, G_TYPE_UINT, G_TYPE_UINT);
	gnome_db_server_signals[DATA_UPDATE_FINISHED] =
                g_signal_new ("data_update_finished",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, data_update_finished),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);
	gnome_db_server_signals[OBJECT_HANDLER_UPDATED] =
                g_signal_new ("object_handler_updated",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_FIRST,
                              G_STRUCT_OFFSET (GnomeDbServerClass, object_handler_updated),
                              NULL, NULL,
                              marshal_VOID__VOID,
                              G_TYPE_NONE, 0);

        class->conn_opened = NULL;
        class->conn_to_close = NULL;
        class->conn_closed = NULL;
        class->data_type_added = NULL;
        class->data_type_removed = NULL;
        class->data_type_updated = NULL;
        class->data_function_added = NULL;
        class->data_function_removed = NULL;
        class->data_function_updated = NULL;
        class->data_aggregate_added = NULL;
        class->data_aggregate_removed = NULL;
        class->data_aggregate_updated = NULL;
        class->data_update_started = NULL;
        class->data_update_finished = NULL;
        class->update_progress = NULL;
        class->object_handler_updated = NULL;


	object_class->dispose = gnome_db_server_dispose;
	object_class->finalize = gnome_db_server_finalize;

	/* Properties */
	object_class->set_property = gnome_db_server_set_property;
	object_class->get_property = gnome_db_server_get_property;
	g_object_class_install_property (object_class, PROP_WITH_FUNCTIONS,
					 g_param_spec_boolean ("with_functions", NULL, NULL, FALSE,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}


static void
gnome_db_server_init (GnomeDbServer * gnome_db_server)
{
	GSList *plugins, *list;

	gnome_db_server->priv = g_new0 (GnomeDbServerPrivate, 1);
	gnome_db_server->priv->gda_datasource = g_string_new ("");
        gnome_db_server->priv->user_name = g_string_new (getenv ("USER"));
        gnome_db_server->priv->password = g_string_new ("");
        gnome_db_server->priv->cnc = NULL;
	gnome_db_server->priv->dict = NULL;
	gnome_db_server->priv->info = gnome_db_server_info_get (NULL);
	gnome_db_server->priv->with_functions = FALSE;

	gnome_db_server->priv->update_in_progress = FALSE;
	gnome_db_server->priv->stop_update = FALSE;

        /*
	 * creation of the data types known by the DBMS 
	 */
        gnome_db_server->priv->data_types = NULL;
	gnome_db_server->priv->custom_types = NULL;
        gnome_db_server->priv->functions = NULL;
        gnome_db_server->priv->aggregates = NULL;


	/* 
	 * data handlers
	 */
        gnome_db_server->priv->handlers = handlers_get_initial_list (gnome_db_server);
        /* data handlers plugins */
        if (!g_module_supported ()) {
                g_warning ("GModule is required to use Libgnomedb!");
                exit (1);
        }
	/* FIXME: also load plugins from a $HOME/.libgnomedb location */
	plugins = handlers_load_plugins (gnome_db_server, LIBGNOMEDB_PLUGINSDIR);
	if (plugins) 
		gnome_db_server->priv->handlers = g_slist_concat (gnome_db_server->priv->handlers, plugins);
	list = gnome_db_server->priv->handlers;
	while (list) {
		gnome_db_base_connect_nullify (list->data,
					 G_CALLBACK (nullified_handler_cb), gnome_db_server);
		list = g_slist_next (list);
	}

	gnome_db_server->priv->fallback_handler = GNOME_DB_DATA_HANDLER (gnome_db_handler_none_new (gnome_db_server));
        gnome_db_server->priv->types_objects_hash = g_hash_table_new (NULL, NULL);
        gnome_db_server->priv->handlers_functions = NULL;

	/* registering functions for the handler finding */
	gnome_db_server_set_object_func_handler (gnome_db_server, func_handler_data_types);
	/* FIXME: Functions, Aggregates */
}

static void
nullified_handler_cb (GnomeDbDataHandler *hdl, GnomeDbServer *srv)
{
	g_return_if_fail (g_slist_find (srv->priv->handlers, hdl));
	srv->priv->handlers = g_slist_remove (srv->priv->handlers, hdl);
	g_object_unref (G_OBJECT (hdl));
}

static GSList *
handlers_get_initial_list (GnomeDbServer *srv)
{
	GSList *list = NULL;
	GnomeDbDataHandler *dh;

	/* Strings */
	dh = GNOME_DB_DATA_HANDLER (gnome_db_handler_string_new (srv));
	list = g_slist_append (list, dh);

	/* Numerical values */
	dh = GNOME_DB_DATA_HANDLER (gnome_db_handler_numerical_new (srv));
	list = g_slist_append (list, dh);

	/* Booleans */
	dh = GNOME_DB_DATA_HANDLER (gnome_db_handler_boolean_new (srv));
	list = g_slist_append (list, dh);

	/* Times, dates, ... */
	dh = GNOME_DB_DATA_HANDLER (gnome_db_handler_time_new (srv));
	list = g_slist_append (list, dh);

	return list;
}

static GSList *
handlers_load_plugins (GnomeDbServer *srv, const gchar *path)
{
        DIR *dstream;
	GSList *list = NULL;

        /* FIXME: it appears that the program hangs if we try to access
           not a lib but an executable program. maybe try to use the "standard"
           libtool calls. */
        dstream = opendir (path);
        if (!dstream) {
                g_print ("Cannot open %s\n", path);
        }
        else {
		struct dirent *dir;	

#ifdef debug
		g_print ("Looking for plugins in %s\n", path);
#endif
		dir = readdir (dstream);
                while (dir) {
                        if ((strlen (dir->d_name) > 3) &&       /* only .so files */
                            !strcmp (dir->d_name + strlen (dir->d_name) - 3, ".so")) {
				gchar *str;
                                GModule *lib;
                                GnomeDbDataHandler *hdl;
                                GnomeDbDataHandler *(*plugin_init) (GnomeDbServer *srv, GModule *module);

                                str = g_strdup_printf ("%s/%s", path, dir->d_name);
                                if ((lib = g_module_open (str, G_MODULE_BIND_LAZY))) {
                                        g_module_symbol (lib, "plugin_init", (gpointer *) &plugin_init);
                                        if (plugin_init) {
                                                hdl = (plugin_init) (srv, lib);
                                                list = g_slist_append (list, hdl);

#ifdef debug
						g_print ("\tLoaded '%s' from file %s\n",
							 gnome_db_data_handler_get_plugin_name (hdl),
							 dir->d_name);
#endif

						/* we won't release this module.
						 * REM: we could make a hash table to release a module
						 * when the corresponding GnomeDbDataHandler is nullified
						 */
						g_module_make_resident (lib);
                                        }
#ifdef debug
                                        else {
						g_print ("\tFile %s does not have plugin_init()\n",
							 dir->d_name);
                                                g_module_close (lib);
                                        }
#endif
                                }
                                g_free (str);
                        }
                        dir = readdir (dstream);
                }
                closedir (dstream);
	}

	return list;
}

/**
 * gnome_db_server_new
 * @dict: a #GnomeDbDict object
 * 
 * Creates a new GnomeDbServer object
 *
 * Returns: the new object
 */
GObject   *
gnome_db_server_new (GnomeDbDict *dict)
{
	GObject   *obj;
	GnomeDbServer *gnome_db_server;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_SERVER_TYPE, NULL);
	gnome_db_server = GNOME_DB_SERVER (obj);

	gnome_db_server->priv->dict = ASSERT_DICT (dict);

	return obj;
}


static void
gnome_db_server_dispose (GObject   * object)
{
	GnomeDbServer *gnome_db_server;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER (object));

	gnome_db_server = GNOME_DB_SERVER (object);
	if (gnome_db_server->priv) {
		gnome_db_server_reset (gnome_db_server);

		/* getting rid of the data handlers */
		if (gnome_db_server->priv->fallback_handler) {
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_server->priv->fallback_handler));
			gnome_db_server->priv->fallback_handler = NULL;
		}
		
		while (gnome_db_server->priv->handlers) 
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_server->priv->handlers->data));
		
		/* getting rid of the hash table for handlers */
		if (gnome_db_server->priv->types_objects_hash) {
			g_hash_table_destroy (gnome_db_server->priv->types_objects_hash);
			gnome_db_server->priv->types_objects_hash = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_server_finalize (GObject *object)
{
	GnomeDbServer *gnome_db_server;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_SERVER (object));

	gnome_db_server = GNOME_DB_SERVER (object);
	if (gnome_db_server->priv) {
		if (gnome_db_server->priv->info) 
			gnome_db_server_info_free (gnome_db_server->priv->info);

		g_string_free (gnome_db_server->priv->gda_datasource, TRUE);
		g_string_free (gnome_db_server->priv->user_name, TRUE);
		g_string_free (gnome_db_server->priv->password, TRUE);

		g_free (gnome_db_server->priv);
		gnome_db_server->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_server_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbServer *gnome_db_server;

	gnome_db_server = GNOME_DB_SERVER (object);
	if (gnome_db_server->priv) {
		switch (param_id) {
		case PROP_WITH_FUNCTIONS:
			gnome_db_server->priv->with_functions = g_value_get_boolean (value);
			break;
		}
	}
}
static void
gnome_db_server_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbServer *gnome_db_server;
	gnome_db_server = GNOME_DB_SERVER (object);
	
	if (gnome_db_server->priv) {
		switch (param_id) {
		case PROP_WITH_FUNCTIONS:
			g_value_set_boolean (value, gnome_db_server->priv->with_functions);
			break;
		}	
	}
}


/* GnomeDbXmlStorage interface implementation */
static gchar *
gnome_db_server_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER (iface)->priv, NULL);

	return NULL;
}

static xmlNodePtr
gnome_db_server_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr toptree, tree;
	GnomeDbServer *srv;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_SERVER (iface)->priv, NULL);

	srv = GNOME_DB_SERVER (iface);

	/* main node */
        toptree = xmlNewNode (NULL, "GNOME_DB_SERVER");
	xmlSetProp (toptree, "with_functions", srv->priv->with_functions ? "t" : "f");

	/* gda_datasource + username */
	tree = xmlNewChild (toptree, NULL, "GNOME_DB_GDA_DATASOURCE", srv->priv->gda_datasource->str);
	xmlAddChild (toptree, tree);
	tree = xmlNewChild (toptree, NULL, "GNOME_DB_USERNAME", srv->priv->user_name->str);
	xmlAddChild (toptree, tree);
	
	/* data types */
	tree = xmlNewChild (toptree, NULL, "GNOME_DB_DATATYPES", NULL);
	list = srv->priv->data_types;
	while (list) {
		xmlNodePtr type;

		type = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
		if (!type) {
			xmlFreeNode (toptree);
			return NULL;
		}
		else {
			if (!xmlAddChild (tree, type)) {
				g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_XML_SAVE_ERROR,
					     _("Error saving data type %s"), 
					     gnome_db_server_data_type_get_sqlname 
					     (GNOME_DB_SERVER_DATA_TYPE (list->data)));
				xmlFreeNode (toptree);
				return NULL;
			}
			if (g_slist_find (srv->priv->custom_types, list->data))
				xmlSetProp (type, "custom", "t");

			list = g_slist_next (list);
		}
	}

	if (srv->priv->with_functions) {
		/* procedures */
		tree = xmlNewChild (toptree, NULL, "GNOME_DB_PROCEDURES", NULL);
		list = srv->priv->functions;
		while (list) {
			xmlNodePtr type;
			
			type = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (!type) {
				xmlFreeNode (toptree);
				return NULL;
			}
			else {
				if (!xmlAddChild (tree, type)) {
					g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_XML_SAVE_ERROR,
						     _("Error saving function %s"), 
						     gnome_db_server_function_get_sqlname 
						     (GNOME_DB_SERVER_FUNCTION (list->data))); 
					xmlFreeNode (toptree);
					return NULL;
				}
				list = g_slist_next (list);
			}
		}
		
		/* aggregates */
		tree = xmlNewChild (toptree, NULL, "GNOME_DB_AGGREGATES", NULL);
		list = srv->priv->aggregates;
		while (list) {
			xmlNodePtr type;
			
			type = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);
			if (!type) {
				xmlFreeNode (toptree);
				return NULL;
			}
			else {
				if (!xmlAddChild (tree, type)) {
					g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_XML_SAVE_ERROR,
						     _("Error saving aggregate %s"), 
						     gnome_db_server_aggregate_get_sqlname 
						     (GNOME_DB_SERVER_AGGREGATE (list->data))); 
					xmlFreeNode (toptree);
					return NULL;
				}
				list = g_slist_next (list);
			}
		}
	}
	
	return toptree;
}

static gboolean
gnome_db_server_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbServer *srv;
	xmlNodePtr subnode;
	gchar *str;

	g_return_val_if_fail (iface && IS_GNOME_DB_SERVER (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_SERVER (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	srv = GNOME_DB_SERVER (iface);

	if (srv->priv->data_types || srv->priv->functions || srv->priv->aggregates) {
		g_set_error (error,
			     GNOME_DB_SERVER_ERROR,
			     GNOME_DB_SERVER_XML_LOAD_ERROR,
			     _("Server already contains data"));
		return FALSE;
	}
	if (strcmp (node->name, "GNOME_DB_SERVER")) {
		g_set_error (error,
			     GNOME_DB_SERVER_ERROR,
			     GNOME_DB_SERVER_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_SERVER>"));
		return FALSE;
	}

	str = xmlGetProp (node, "with_functions");
	if (str) {
		srv->priv->with_functions = (*str && (*str == 't')) ? TRUE : FALSE;
		g_free (str);
	}

	subnode = node->children;
	while (subnode) {
		gboolean done = FALSE;

		if (!strcmp (subnode->name, "GNOME_DB_GDA_DATASOURCE")) {
			gchar *txt;
			txt = xmlNodeGetContent (subnode);
			if (! gnome_db_server_set_datasource (srv, txt)) {
				g_set_error (error,
					     GNOME_DB_SERVER_ERROR,
					     GNOME_DB_SERVER_XML_LOAD_ERROR,
					     _("Datasource '%s' not found"), txt);
				g_free (txt);
				return FALSE;
			}
			g_free (txt);
			done = TRUE;
		}

		if (!done && !strcmp (subnode->name, "GNOME_DB_USERNAME")) {
			gchar *txt;
			txt = xmlNodeGetContent (subnode);
			gnome_db_server_set_user_name (srv, txt);
			g_free (txt);
			done = TRUE;
		}

		if (!done && !strcmp (subnode->name, "GNOME_DB_DATATYPES")) {
			xmlNodePtr type;
			GnomeDbServerDataType *dt;
			gchar *prop;

			type = subnode->children;
			while (type) {
				if (! xmlNodeIsText (type)) {
					gboolean load = TRUE; /* don't load custom data types */
					prop = xmlGetProp (type, "custom");
					if (prop) {
						if (*prop == 't')
							load = FALSE;
						g_free (prop);
					}
					
					if (load) {
						dt = GNOME_DB_SERVER_DATA_TYPE (gnome_db_server_data_type_new (srv));
						if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (dt), type, error)) {
							srv->priv->data_types = g_slist_append (srv->priv->data_types, dt);
							
							gnome_db_base_connect_nullify (dt,
										       G_CALLBACK (nullified_data_type_cb), srv);
							g_signal_connect (G_OBJECT (dt), "changed",
									  G_CALLBACK (updated_data_type_cb), srv);
#ifdef debug_signal
							g_print (">> 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
							g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[DATA_TYPE_ADDED], 
								       0, dt);
#ifdef debug_signal
							g_print ("<< 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
						}
						else {
							return FALSE;
						}
					}
				}
				type = type->next;
			}
			done = TRUE;
		}
		
		if (!done && srv->priv->with_functions && !strcmp (subnode->name, "GNOME_DB_PROCEDURES")) {
			xmlNodePtr fnode;
			GnomeDbServerFunction *func;
			
			fnode = subnode->children;
			while (fnode) {
				if (! xmlNodeIsText (fnode)) {
					func = GNOME_DB_SERVER_FUNCTION (gnome_db_server_function_new (srv));
					if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (func), fnode, error)) {
						srv->priv->functions = g_slist_append (srv->priv->functions, func);
						gnome_db_base_connect_nullify (func,
									 G_CALLBACK (nullified_function_cb), srv);
						g_signal_connect (G_OBJECT (func), "changed",
								  G_CALLBACK (updated_function_cb), srv);
#ifdef debug_signal
						g_print (">> 'DATA_FUNCTION_ADDED' from %s\n", __FUNCTION__);
#endif
						g_signal_emit_by_name (G_OBJECT (srv), "data_function_added", func);
#ifdef debug_signal
						g_print ("<< 'DATA_FUNCTION_ADDED' from %s\n", __FUNCTION__);
#endif
					}
					else {
						return FALSE;
					}
				}
				fnode = fnode->next;
			}
			done = TRUE;
		}

		if (!done && srv->priv->with_functions && !strcmp (subnode->name, "GNOME_DB_AGGREGATES")) {
			xmlNodePtr fnode;
			GnomeDbServerAggregate *agg;
			
			fnode = subnode->children;
			while (fnode) {
				if (! xmlNodeIsText (fnode)) {
					agg = GNOME_DB_SERVER_AGGREGATE (gnome_db_server_aggregate_new (srv));
					if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (agg), fnode, error)) {
						srv->priv->aggregates = g_slist_append (srv->priv->aggregates, agg);
						gnome_db_base_connect_nullify (agg,
									 G_CALLBACK (nullified_aggregate_cb), srv);
						g_signal_connect (G_OBJECT (agg), "changed",
								  G_CALLBACK (updated_aggregate_cb), srv);
#ifdef debug_signal
						g_print (">> 'DATA_AGGREGATE_ADDED' from %s\n", __FUNCTION__);
#endif
						g_signal_emit_by_name (G_OBJECT (srv), "data_aggregate_added", agg);
#ifdef debug_signal
						g_print ("<< 'DATA_AGGREGATE_ADDED' from %s\n", __FUNCTION__);
#endif
					}
					else {
						return FALSE;
					}
				}
				fnode = fnode->next;
			}
			done = TRUE;
		}

		subnode = subnode->next;
	}
	
	return TRUE;
}

#ifdef debug
/**
 * gnome_db_server_dump
 * @srv: a #GnomeDbServer object
 * @offset: the offset (in caracters) at which the dump will start
 *
 * Writes a textual description of the object to STDOUT. This function only
 * exists if libgnomedb is compiled with the "--enable-debug" option.
 */
void 
gnome_db_server_dump (GnomeDbServer *srv, gint offset)
{
	gchar *str;
	guint i;
	GSList *list;

	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	
	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;
	
	/* dump */
	if (srv->priv) 
		g_print ("%s" D_COL_H1 "GnomeDbServer %p\n" D_COL_NOR, 
			 str, srv);
	else
		g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, srv);

	/* data types */
	list = srv->priv->data_types;
	if (list) {
		g_print ("%sData types:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Data type defined\n", str);

	/* functions */
	list = srv->priv->functions;
	if (list) {
		g_print ("%sFunctions:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Function defined\n", str);


	/* aggregates */
	list = srv->priv->aggregates;
	if (list) {
		g_print ("%sAggregates:\n", str);
		while (list) {
			gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%sNo Aggregate defined\n", str);


	g_free (str);

}
#endif


/**
 * gnome_db_server_get_dict
 * @srv: a #GnomeDbServer object
 *
 * Fetch the GnomeDbDict object to which the GnomeDbServer belongs.
 *
 * Returns: the GnomeDbDict object
 */
GnomeDbDict *
gnome_db_server_get_dict (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	return srv->priv->dict;
}

/**
 * gnome_db_server_set_datasource
 * @srv: a #GnomeDbServer object
 * @datasource: a gda datasource
 * 
 * Sets the data source of the server. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * If the requested datasource does not exist, then nothing is done ans FALSE
 * is returned.
 *
 * If the default XML filename to save the dictionary has not yet been
 * specified, then the default one is specified, so it is possible to call
 * gnome_db_dict_load_xml() right after that.
 *
 * Returns: TRUE on success
 */
gboolean
gnome_db_server_set_datasource (GnomeDbServer *srv, const gchar *datasource)
{
	GdaDataSourceInfo *dsn;
	gchar *cstr;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	g_return_val_if_fail (datasource && *datasource, FALSE);

	if (srv->priv->cnc)
		return FALSE;

	dsn = gda_config_find_data_source (datasource);
	if (!dsn)
		return FALSE;

	g_string_assign (srv->priv->gda_datasource, datasource);

	/* private information on the provider */
	if (srv->priv->info)
		gnome_db_server_info_free (srv->priv->info);
	srv->priv->info = gnome_db_server_info_get (dsn->provider);
	gda_data_source_info_free (dsn);

	/* set the default XML filename for the dictionary */
	cstr = gnome_db_dict_get_xml_filename (srv->priv->dict);
	if (!cstr) {
		gchar *str;

		str = gnome_db_dict_compute_xml_filename (srv->priv->dict, datasource, NULL, NULL);
		if (str) {
			gnome_db_dict_set_xml_filename (srv->priv->dict, str);
			g_free (str);
		}
	}

	return TRUE;
}

/**
 * gnome_db_server_get_datasource
 * @srv: a #GnomeDbServer object
 * 
 * Get the data source of the server. 
 *
 * Returns: a new string with the datasource, or NULL
 */
gchar *
gnome_db_server_get_datasource (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);

	if (srv->priv->gda_datasource && srv->priv->gda_datasource->str && *(srv->priv->gda_datasource->str))
		return g_strdup (srv->priv->gda_datasource->str);
	else
		return NULL;
}

/**
 * gnome_db_server_set_user_name
 * @srv: a #GnomeDbServer object
 * @username: 
 * 
 * Sets the user name for the connection to the server. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * Returns: TRUE on success
 */
gboolean
gnome_db_server_set_user_name (GnomeDbServer *srv, const gchar *username)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	g_return_val_if_fail (username, FALSE);

	if (srv->priv->cnc)
		return FALSE;

	g_string_assign (srv->priv->user_name, username);
	return TRUE;	
}

/**
 * gnome_db_server_get_user_name
 * @srv: a #GnomeDbServer object
 * 
 * Get the user name for the connection to the server.
 *
 * Returns: a new string with the user name, or NULL
 */
gchar*
gnome_db_server_get_user_name (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	if (srv->priv->user_name && srv->priv->user_name->str && *(srv->priv->user_name->str))
		return g_strdup (srv->priv->user_name->str);
	else
		return NULL;
}

/**
 * gnome_db_server_set_user_password
 * @srv: a #GnomeDbServer object
 * @password: 
 * 
 * Sets the user password for the connection to the server. If the connection is already opened,
 * then no action is performed at all and FALSE is returned.
 *
 * Returns: TRUE on success
 */
gboolean
gnome_db_server_set_user_password (GnomeDbServer *srv, const gchar *password)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	g_return_val_if_fail (password, FALSE);

	if (srv->priv->cnc)
		return FALSE;

	g_string_assign (srv->priv->password, password);
	return TRUE;	
}

/**
 * gnome_db_server_reset
 * @srv: a #GnomeDbServer object
 *
 * Reset the GnomeDbServer as it was when created; that is: close the connection if opened,
 * and get rid of any data type, function and aggregate it has.
 */
void
gnome_db_server_reset (GnomeDbServer *srv)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);
	
	/* getting rid of the function */
	while (srv->priv->functions)
		gnome_db_base_nullify (GNOME_DB_BASE (srv->priv->functions->data));
	
	/* getting rid of the aggregates */
	while (srv->priv->aggregates)
		gnome_db_base_nullify (GNOME_DB_BASE (srv->priv->aggregates->data));
	
	/* getting rid of the data types */
	while (srv->priv->data_types)
		gnome_db_base_nullify (GNOME_DB_BASE (srv->priv->data_types->data));
	if (srv->priv->custom_types) {
		g_slist_free (srv->priv->custom_types);
		srv->priv->custom_types = NULL;
	}

	/* close the connection if necessary */
	if (srv->priv->cnc)
		gnome_db_server_close_connect_no_warn (srv);
}

/**
 * gnome_db_server_get_gda_connection
 * @srv: a #GnomeDbServer object
 *
 * Get the #GdaConnection object used by @srv, if the connection
 * is opened.
 *
 * Returns: the #GdaConnection, or %NULL
 */
GdaConnection *
gnome_db_server_get_gda_connection (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	return srv->priv->cnc;
}

/** 
 * gnome_db_server_open_connect
 * @srv: a #GnomeDbServer object
 * @error: location to store error, or %NULL
 *
 * Opens the connection to the DBMS.
 *
 * Returns: TRUE if success and FALSE otherwise (and error is positionned)
 */
gboolean
gnome_db_server_open_connect (GnomeDbServer *srv, GError **error)
{
	GdaDataSourceInfo *dsn;
	gboolean retval = FALSE;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);

	if (srv->priv->cnc)
		return TRUE;

	dsn = gda_config_find_data_source (srv->priv->gda_datasource->str);
	if (!dsn) {
		GdaError *gdaerror;
		gchar *str;

		gdaerror = gda_error_new ();
		str = g_strdup_printf (_("No datasource '%s' defined in your GDA configuration"),
				       srv->priv->gda_datasource->str);

		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_CONN_OPEN_ERROR, str);
		gda_error_set_description (gdaerror, str);

		g_free (str);
		
		gda_error_set_source (gdaerror, _("[LibMergeant]"));
		gda_connection_add_error (srv->priv->cnc, gdaerror);

		return FALSE;
	}

	srv->priv->cnc = gda_client_open_connection (GDA_CLIENT (srv), dsn->name,
						     srv->priv->user_name->str, 
						     srv->priv->password->str, 0);

	gda_data_source_info_free (dsn);
	if (srv->priv->cnc) { 
		/* connection is now opened */
#ifdef debug_signal
		g_print (">> 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif
		g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[CONN_OPENED], 0);
#ifdef debug_signal
		g_print ("<< 'CONN_OPENED' from %s\n", __FUNCTION__);
#endif

		retval = TRUE;
	}
	else 
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_CONN_OPEN_ERROR,
			     _("Could not open the connection to the DBMS for datasource '%s'"),
			     srv->priv->gda_datasource->str);

	return retval;
}


/**
 * gnome_db_server_conn_is_opened
 * @srv: a #GnomeDbServer object
 *
 * Checks wether the connection to the DBMS is opened or not
 *
 * Returns: TRUE if the connection is opened
 */
gboolean
gnome_db_server_conn_is_opened (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);

	return srv->priv->cnc ? TRUE : FALSE;
}

/** 
 * gnome_db_server_close_connect
 * @srv: a #GnomeDbServer object
 *
 * Closes the connection to the DBMS. First the "conn_to_close" signal is emitted.
 * This function should preferably be called instead of the gnome_db_server_close_connect_no_warn() function.
 */
void
gnome_db_server_close_connect (GnomeDbServer *srv)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);

	if (!srv->priv->cnc)
		return;

#ifdef debug_signal
	g_print (">> 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[CONN_TO_CLOSE], 0);
#ifdef debug_signal
	g_print ("<< 'CONN_TO_CLOSE' from %s\n", __FUNCTION__);
#endif

	gnome_db_server_close_connect_no_warn (srv);
}


/** 
 * gnome_db_server_close_connect_no_warn
 * @srv: a #GnomeDbServer object
 *
 * Closes the connection to the DBMS. Warning: "conn_to_close" signal is NOT emitted.
 */
void
gnome_db_server_close_connect_no_warn (GnomeDbServer *srv)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);

	if (!srv->priv->cnc)
		return;
	
	gda_connection_close (srv->priv->cnc);
	srv->priv->cnc = NULL;

#ifdef debug_signal
	g_print (">> 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[CONN_CLOSED], 0);
#ifdef debug_signal
	g_print ("<< 'CONN_CLOSED' from %s\n", __FUNCTION__);
#endif	
}

/**
 * gnome_db_server_get_server_info
 * @srv: a #GnomeDbServer object
 *
 * Fetch the #GnomeDbServerInfo structure describing @srv's features
 * Do not free that structure!
 *
 * Returns: a #GnomeDbServerInfo structure, or %NULL if none exists
 */
GnomeDbServerInfo *
gnome_db_server_get_server_info (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	return srv->priv->info;
}


/**
 * gnome_db_server_do_query_as_data_model
 * @srv: a #GnomeDbServer object
 * @query: the query to be executed
 * @type: the query type (SQL or XML)
 * @error: location to store error, or %NULL
 *
 * Sends a query to the DBMS to which the connection is established. If the query is a SELECT
 * one, then a new #GdaDataModel is returned (it's up to the caller to unref that object); 
 * otherwise NULL is returned. The error variable contains the error code if an error occurred.
 *
 * Returns: a new #GdaDataModel object or NULL
 */
GdaDataModel *
gnome_db_server_do_query_as_data_model (GnomeDbServer *srv, const gchar *query,
					GnomeDbServerQueryType type, GError **error)
{
	GdaCommandType gtype;
	GdaDataModel *res = NULL;
	GdaCommand *cmd;
	GnomeDbServerOpMode mode = GNOME_DB_SERVER_UNKNOWN_OP;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	if (!srv->priv->cnc) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_DO_QUERY_ERROR,
			     _("Connection is not opened"));
		return NULL;
	}
	
	switch (type) {
	case GNOME_DB_SERVER_QUERY_XML:
		gtype = GDA_COMMAND_TYPE_XML;
		/* FIXME: get the op mode of that query */
		break;
	default:
		gtype = GDA_COMMAND_TYPE_SQL;
		mode = gnome_db_server_get_sql_op_mode (srv, query);
		break;
	}
	
	if (mode == GNOME_DB_SERVER_UNKNOWN_OP) 
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_DO_QUERY_ERROR,
			     _("Unknown mode of operation for this query"));
	else {
		GList *errors_before, *errors_after;
		errors_before = gda_connection_get_errors (srv->priv->cnc);
		cmd = gda_command_new (query, gtype, GDA_COMMAND_OPTION_STOP_ON_ERRORS);
		res = gda_connection_execute_single_command (srv->priv->cnc, cmd, NULL);
		errors_after = gda_connection_get_errors (srv->priv->cnc);
		if (errors_after) {
			if (!errors_before ||
			    (errors_before && 
			     ((g_list_last (errors_before))->data != (g_list_last (errors_after))->data)))
				g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_DO_QUERY_ERROR,
					     gda_error_get_description (GDA_ERROR ((g_list_last (errors_after))->data)));
		}
	}

	return res;
}


/**
 * gnome_db_server_get_sql_op_mode
 * @srv: a #GnomeDbServer object
 * @query: an SQL query
 * 
 * Get the operation type (= mode) which is performed by the query given as argument.
 * The query MUST contain only one statement, not several separated by ';'
 *
 * Returns: the query type (mode).
 */
GnomeDbServerOpMode
gnome_db_server_get_sql_op_mode (GnomeDbServer *srv, const gchar *query)
{
	GnomeDbServerOpMode mode = GNOME_DB_SERVER_UNKNOWN_OP;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), GNOME_DB_SERVER_UNKNOWN_OP);
	g_return_val_if_fail (srv->priv, GNOME_DB_SERVER_UNKNOWN_OP);
	g_return_val_if_fail (query && *query, GNOME_DB_SERVER_UNKNOWN_OP);

	/* FIXME: return an error if there is more than ONE query statement in the query string */
	if (! g_ascii_strncasecmp (query, "SELECT", 6))
		mode = GNOME_DB_SERVER_SELECT_OP;
	if (! g_ascii_strncasecmp (query, "INSERT", 6))
		mode = GNOME_DB_SERVER_INSERT_OP;
	if (! g_ascii_strncasecmp (query, "UPDATE", 6))
		mode = GNOME_DB_SERVER_UPDATE_OP;
	if (! g_ascii_strncasecmp (query, "DELETE", 6))
		mode = GNOME_DB_SERVER_DELETE_OP;

	if (! g_ascii_strncasecmp (query, "CREATE", 6))
		mode = GNOME_DB_SERVER_DDL_OP;
	if (! g_ascii_strncasecmp (query, "DROP", 4))
		mode = GNOME_DB_SERVER_DDL_OP;
	if (! g_ascii_strncasecmp (query, "ALTER", 5))
		mode = GNOME_DB_SERVER_DDL_OP;

	return mode;
}


static gboolean server_data_type_update_list (GnomeDbServer * srv, GError **error);
static gboolean server_functions_update_list (GnomeDbServer * srv, GError **error);
static gboolean server_aggregates_update_list (GnomeDbServer * srv, GError **error);
/**
 * gnome_db_server_update_dbms_data
 * @srv: a #GnomeDbServer object
 * @error: location to store error, or %NULL
 *
 * Synchronise the list of data types, functions, etc the GnomeDbServer object has
 * with what is in the DBMS the connection is opened to. The connection to the DBMS 
 * MUST be opened.
 *
 * Returns: TRUE if no error
 */
gboolean
gnome_db_server_update_dbms_data (GnomeDbServer *srv, GError **error)
{
	gboolean retval = TRUE;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	
	if (srv->priv->update_in_progress) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_META_DATA_UPDATE,
			     _("Update already started!"));
		return FALSE;
	}

	if (!srv->priv->cnc) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_META_DATA_UPDATE,
			     _("Connection is not opened!"));
		return FALSE;
	}

	srv->priv->update_in_progress = TRUE;
	srv->priv->stop_update = FALSE;

#ifdef debug_signal
	g_print (">> 'DATA_UPDATE_STARTED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[DATA_UPDATE_STARTED], 0);
#ifdef debug_signal
	g_print ("<< 'DATA_UPDATE_STARTED' from %s\n", __FUNCTION__);
#endif

	retval = server_data_type_update_list (srv, error);
	if (retval && srv->priv->with_functions && !srv->priv->stop_update) 
		retval = server_functions_update_list (srv, error);
	if (retval && srv->priv->with_functions && !srv->priv->stop_update) 
		retval = server_aggregates_update_list (srv, error);

#ifdef debug_signal
	g_print (">> 'DATA_UPDATE_FINISHED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[DATA_UPDATE_FINISHED], 0);
#ifdef debug_signal
	g_print ("<< 'DATA_UPDATE_FINISHED' from %s\n", __FUNCTION__);
#endif

	srv->priv->update_in_progress = FALSE;
	if (srv->priv->stop_update) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_META_DATA_UPDATE_USER_STOPPED,
			     _("Update stopped!"));
		return FALSE;
	}

	return retval;
}


/**
 * gnome_db_server_stop_update_dbms_data
 * @srv: a #GnomeDbServer object
 *
 * When the server updates its internal lists of DBMS objects, a call to this function will 
 * stop that update process. It has no effect when the server is not updating its DBMS data.
 */
void
gnome_db_server_stop_update_dbms_data (GnomeDbServer *srv)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);

	srv->priv->stop_update = TRUE;
}

static gboolean
server_data_type_update_list (GnomeDbServer * srv, GError **error)
{
	GSList *dtl = srv->priv->data_types;
	GnomeDbServerDataType *dt;
	GdaDataModel *rs;
	gchar *str;
	guint now, total;
	GSList *updated_dt = NULL;
	gboolean has_synonyms;

	/* here we get the complete list of types, and for each type, update or
	   create the entry in the list if not yet there. */
	rs = gda_connection_get_schema (GDA_CONNECTION (srv->priv->cnc),
					GDA_CONNECTION_SCHEMA_TYPES, NULL);

	if (!rs) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_DATATYPE_ERROR,
			     _("Can't get list of data types"));
		return FALSE;
	}


	if (!gnome_db_result_set_check_data_model (rs, 4, 
						   GDA_VALUE_TYPE_STRING, 
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_TYPE)) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_DATATYPE_ERROR,
			     _("Schema for list of data types is wrong"));
		g_object_unref (G_OBJECT (rs));
		return FALSE;
	}
	has_synonyms = gnome_db_result_set_check_data_model (rs, 5, 
							     GDA_VALUE_TYPE_STRING, 
							     GDA_VALUE_TYPE_STRING,
							     GDA_VALUE_TYPE_STRING,
							     GDA_VALUE_TYPE_TYPE,
							     GDA_VALUE_TYPE_STRING);

	total = gda_data_model_get_n_rows (rs);
	now = 0;		
	while ((now < total) && !srv->priv->stop_update) {
		const GdaValue *value;
		gboolean newdt = FALSE;

		value = gda_data_model_get_value_at (rs, 0, now);
		str = gda_value_stringify (value);
		dt = gnome_db_server_get_data_type_by_name (srv, str);
		if (!dt) {
			gint i = 0;
			GSList *list;
			gboolean found = FALSE;

			/* type name */
			dt = GNOME_DB_SERVER_DATA_TYPE (gnome_db_server_data_type_new (srv));
			gnome_db_server_data_type_set_sqlname (dt, str);
			newdt = TRUE;
			
			/* finding the right position for the data type */
			list = srv->priv->data_types;
			while (list && !found) {
				if (strcmp (str, 
					    gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (list->data))) < 0)
					found = TRUE;
				else
					i++;
				list = g_slist_next (list);
			}

			srv->priv->data_types = g_slist_insert (srv->priv->data_types, dt, i);
			/* REM: we don't connect the "nullified" and "changed" signals from the new GnomeDbServerDataType right
			   now because otherwise the GnomeDbServer object will forward the "changed" signal 
			   (into "data_type_updated") before we have had the chance to emit the "data_type_added" signal */
		}
		g_free (str);
		
		updated_dt = g_slist_append (updated_dt, dt);

		/* FIXME: number of params */
		/*dt->numparams = 0;*/

		/* description */
		value = gda_data_model_get_value_at (rs, 2, now);
		if (value && !gda_value_is_null (value) && gda_value_get_string(value) && (* gda_value_get_string(value))) {
			str = gda_value_stringify (value);
			gnome_db_base_set_description (GNOME_DB_BASE (dt), str);
			g_free (str);
		}
		else 
			gnome_db_base_set_description (GNOME_DB_BASE (dt), NULL);

		/* owner */
		value = gda_data_model_get_value_at (rs, 1, now);
		if (value && !gda_value_is_null (value) && gda_value_get_string(value) && (* gda_value_get_string(value))) {
			str = gda_value_stringify (value);
			gnome_db_base_set_owner (GNOME_DB_BASE (dt), str);
			g_free (str);
		}
		else
			gnome_db_base_set_owner (GNOME_DB_BASE (dt), NULL);
				
		/* gda_type */
		value = gda_data_model_get_value_at (rs, 3, now);
		if (value && !gda_value_is_null (value)) 
			gnome_db_server_data_type_set_gda_type (dt, gda_value_get_vtype (value));
		
		/* data type synomyms */
		gnome_db_server_data_type_clear_synonyms (dt);
		if (has_synonyms) {
			value = gda_data_model_get_value_at (rs, 4, now);
			if (value && !gda_value_is_null (value) && gda_value_get_string(value) && 
			    (* gda_value_get_string(value))) {
				gchar *tok, *buf;

				str = gda_value_stringify (value);
				tok = strtok_r (str, ",", &buf);
				if (tok) {
					if (*tok) 
						gnome_db_server_data_type_add_synonym (dt, tok);
					tok = strtok_r (NULL, ",", &buf);
					while (tok) {
						if (*tok) 
							gnome_db_server_data_type_add_synonym (dt, tok);

						tok = strtok_r (NULL, ",", &buf);				
					}
				}
				g_free (str);
			}
		}

		/* signal if the DT is new */
		if (newdt) {
			gnome_db_base_connect_nullify (dt,
						 G_CALLBACK (nullified_data_type_cb), srv);
			g_signal_connect (G_OBJECT (dt), "changed",
					  G_CALLBACK (updated_data_type_cb), srv);
#ifdef debug_signal
			g_print (">> 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
			g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[DATA_TYPE_ADDED], 0, dt);
#ifdef debug_signal
			g_print ("<< 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
		}

		g_signal_emit_by_name (G_OBJECT (srv), "update_progress", "DATA_TYPES",
				       now, total);
		now++;
	}
	
	g_object_unref (G_OBJECT (rs));
	
	/* if a data type has been updated, make sure it's not anymore in the custom
	 * data types list */
	dtl = srv->priv->custom_types;
	while (dtl) {
		if (g_slist_find (updated_dt, dtl->data)) {
			GSList *tmp = g_slist_next (dtl);

			g_object_ref (dtl->data);
			srv->priv->custom_types = g_slist_delete_link (srv->priv->custom_types, dtl);
			dtl = tmp;
		}
		else
			dtl = g_slist_next (dtl);
	}

	/* remove the data types not existing anymore and not in the custom list */
	dtl = srv->priv->data_types;
	while (dtl) {
		if (!g_slist_find (updated_dt, dtl->data) && 
		    !g_slist_find (srv->priv->custom_types, dtl->data)) {
			gnome_db_base_nullify (GNOME_DB_BASE (dtl->data));
			dtl = srv->priv->data_types;
		}
		else
			dtl = g_slist_next (dtl);
	}
	g_slist_free (updated_dt);
	
	g_signal_emit_by_name (G_OBJECT (srv), "update_progress", NULL, 0, 0);
	
	return TRUE;
}


static void
nullified_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServer *srv)
{
	g_return_if_fail (g_slist_find (srv->priv->data_types, dt));

	srv->priv->data_types = g_slist_remove (srv->priv->data_types, dt);

	g_signal_handlers_disconnect_by_func (G_OBJECT (dt), 
					      G_CALLBACK (nullified_data_type_cb), srv);
	g_signal_handlers_disconnect_by_func (G_OBJECT (dt), 
					      G_CALLBACK (updated_data_type_cb), srv);

#ifdef debug_signal
	g_print (">> 'DATA_TYPE_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_type_removed", dt);
#ifdef debug_signal
	g_print ("<< 'DATA_TYPE_REMOVED' from %s\n", __FUNCTION__);
#endif

	if (! g_slist_find (srv->priv->custom_types, dt))
		g_object_unref (G_OBJECT (dt));
	else
		srv->priv->custom_types = g_slist_remove (srv->priv->custom_types, dt);	
}

static void
updated_data_type_cb (GnomeDbServerDataType *dt, GnomeDbServer *srv)
{
#ifdef debug_signal
	g_print (">> 'DATA_TYPE_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_type_updated", dt);
#ifdef debug_signal
	g_print ("<< 'DATA_TYPE_UPDATED' from %s\n", __FUNCTION__);
#endif	
}

static gboolean
server_functions_update_list (GnomeDbServer *srv, GError **error)
{
	GnomeDbServerFunction *func;
	GdaDataModel *rs;
	gchar *str;
	guint now, total;
	GSList *list, *updated_fn = NULL, *todelete_fn = NULL;
	GSList *original_functions;
	gboolean insert;
	gint current_position = 0;

	/* here we get the complete list of functions, and for each function, update or
	   create the entry in the list if not yet there. */
	rs = gda_connection_get_schema (GDA_CONNECTION (srv->priv->cnc),
					GDA_CONNECTION_SCHEMA_PROCEDURES, NULL);

	if (!rs) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_FUNCTIONS_ERROR,
			     _("Can't get list of functions"));
		return FALSE;
	}


	if (!gnome_db_result_set_check_data_model (rs, 8, 
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_INTEGER,
					    GDA_VALUE_TYPE_STRING,
					    GDA_VALUE_TYPE_STRING)) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_FUNCTIONS_ERROR,
			     _("Schema for list of functions is wrong"));
		g_object_unref (G_OBJECT (rs));
		return FALSE;
	}

	original_functions = gnome_db_server_get_functions (srv);
	total = gda_data_model_get_n_rows (rs);
	now = 0;		
	while ((now < total) && !srv->priv->stop_update) {
		GnomeDbServerDataType *rettype = NULL; /* return type for the function */
		GSList *dtl = NULL;     /* list of params for the function */
		const GdaValue *value;
		gchar *ptr;
		gchar *tok;
		
		insert = TRUE;

		/* fetch return type */
		value = gda_data_model_get_value_at (rs, 4, now);
		str = gda_value_stringify (value);
		if (*str != '-') {
			rettype = gnome_db_server_get_data_type_by_name (srv, str);
			if (!rettype)
				insert = FALSE;
		}
		else 
			insert = FALSE;
		g_free (str);

		/* fetch argument types */
		value = gda_data_model_get_value_at (rs, 6, now);
		str = gda_value_stringify (value);
		if (str) {
			ptr = strtok_r (str, ",", &tok);
			while (ptr && *ptr) {
				GnomeDbServerDataType *indt;

				if (*ptr == '-')
					dtl = g_slist_append (dtl, NULL); /* any data type will do */
				else {
					indt = gnome_db_server_get_data_type_by_name (srv, ptr);
					if (indt)
						dtl = g_slist_append (dtl, indt);
					else 
						insert = FALSE;
				}
				ptr = strtok_r (NULL, ",", &tok);
			}
			g_free (str);
		}

		/* fetch a func if there is already one with the same id */
		value = gda_data_model_get_value_at (rs, 1, now);
		str = gda_value_stringify (value);
		func = gnome_db_server_get_function_by_dbms_id (srv, str);
		g_free (str);

		if (!func) {
			/* try to find the function using its name, return type and argument type, 
			   and not its DBMS id, this is usefull if the DBMS has changed and the
			   DBMS id have changed */
			value =  gda_data_model_get_value_at (rs, 0, now);
			str = gda_value_stringify (value);
			func = gnome_db_server_get_function_by_name_arg_real (srv, original_functions, str, dtl);
			g_free (str);

			if (func && (gnome_db_server_function_get_ret_type (func) != rettype))
				func = NULL;
		}

		if (!insert) {
			if (func)
				/* If no insertion, then func MUST be updated */
				todelete_fn = g_slist_append (todelete_fn, func);
			func = NULL;
		}
		else {
			if (func) {
				/* does the function we found have the same rettype and params 
				   as the one we have now? */
				gboolean isequal = TRUE;
				GSList *hlist;
				
				list = gnome_db_server_function_get_arg_types (func);
				hlist = dtl;
				while (list && hlist && isequal) {
					if (list->data != hlist->data)
						isequal = FALSE;
					list = g_slist_next (list);
					hlist = g_slist_next (hlist);
				}
				if (isequal && (gnome_db_server_function_get_ret_type (func) != rettype)) 
					isequal = FALSE;
				
				if (isequal) {
					updated_fn = g_slist_append (updated_fn, func);
					current_position = g_slist_index (srv->priv->functions, func) + 1;
					insert = FALSE;
				}
				else {
					todelete_fn = g_slist_append (todelete_fn, func);
					func = NULL;
				}
			}

			if (!func) {
				/* creating new ServerFunction object */
				func = GNOME_DB_SERVER_FUNCTION (gnome_db_server_function_new (srv));
				gnome_db_server_function_set_ret_type (func, rettype);
				gnome_db_server_function_set_arg_types (func, dtl);

				/* mark function as updated */
				updated_fn = g_slist_append (updated_fn, func);
			}
		}
	
		if (dtl)
			g_slist_free (dtl);
		
		/* function's attributes update */
		if (func) {
			/* unique id */
			value = gda_data_model_get_value_at (rs, 1, now);
			str = gda_value_stringify (value);
			gnome_db_server_function_set_dbms_id (func, str);
			g_free (str);

			/* description */
			value = gda_data_model_get_value_at (rs, 3, now);
			if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
				str = gda_value_stringify (value);
				gnome_db_base_set_description (GNOME_DB_BASE (func), str);
				g_free (str);
			}
			
			/* sqlname */
			value =  gda_data_model_get_value_at (rs, 0, now);
			str = gda_value_stringify (value);
			gnome_db_server_function_set_sqlname (func, str);
			g_free (str);
			
			/* owner */
			value = gda_data_model_get_value_at (rs, 2, now);
			if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
				str = gda_value_stringify (value);
				gnome_db_base_set_owner (GNOME_DB_BASE (func), str);
				g_free (str);
			}
			else
				gnome_db_base_set_owner (GNOME_DB_BASE (func), NULL);
		}

		/* Real insertion */
		if (insert) {
			/* insertion in the list: finding where to insert the function */
			srv->priv->functions = g_slist_insert (srv->priv->functions, func, current_position++);
			gnome_db_base_connect_nullify (func,
						 G_CALLBACK (nullified_function_cb), srv);
			g_signal_connect (G_OBJECT (func), "changed",
					  G_CALLBACK (updated_function_cb), srv);

#ifdef debug_signal
			g_print (">> 'DATA_FUNCTION_ADDED' from %s\n", __FUNCTION__);
#endif
			g_signal_emit_by_name (G_OBJECT (srv), "data_function_added", func);
#ifdef debug_signal
			g_print ("<< 'DATA_FUNCTION_ADDED' from %s\n", __FUNCTION__);
#endif
		}

		g_signal_emit_by_name (G_OBJECT (srv), "update_progress", "FUNCTIONS", now, total);
		now ++;
	}

	g_object_unref (G_OBJECT (rs));
	if (original_functions)
		g_slist_free (original_functions);

	/* cleanup for the functions which do not exist anymore */
        list = srv->priv->functions;
        while (list && !srv->priv->stop_update) {
		if (!g_slist_find (updated_fn, list->data)) 
			todelete_fn = g_slist_append (todelete_fn, list->data);
		list = g_slist_next (list);
        }
	g_slist_free (updated_fn);

	list = todelete_fn;
	while (list) {
		gnome_db_base_nullify (GNOME_DB_BASE (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (todelete_fn);
	
	g_signal_emit_by_name (G_OBJECT (srv), "update_progress", NULL, 0, 0);
	
	return TRUE;
}

static void
nullified_function_cb (GnomeDbServerFunction *func, GnomeDbServer *srv)
{
	g_return_if_fail (g_slist_find (srv->priv->functions, func));
	srv->priv->functions = g_slist_remove (srv->priv->functions, func);

	g_signal_handlers_disconnect_by_func (G_OBJECT (func), 
					      G_CALLBACK (nullified_function_cb), srv);
	g_signal_handlers_disconnect_by_func (G_OBJECT (func), 
					      G_CALLBACK (updated_function_cb), srv);

#ifdef debug_signal
	g_print (">> 'DATA_FUNCTION_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_function_removed", func);
#ifdef debug_signal
	g_print ("<< 'DATA_FUNCTION_REMOVED' from %s\n", __FUNCTION__);
#endif

	g_object_unref (G_OBJECT (func));
}


static void
updated_function_cb (GnomeDbServerFunction *func, GnomeDbServer *srv)
{
#ifdef debug_signal
	g_print (">> 'DATA_FUNCTION_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_function_updated", func);
#ifdef debug_signal
	g_print ("<< 'DATA_FUNCTION_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


static gboolean
server_aggregates_update_list (GnomeDbServer *srv, GError **error)
{
	GnomeDbServerAggregate *agg;
	GdaDataModel *rs;
	gchar *str;
	guint now, total;
	GSList *list, *updated_aggs = NULL, *todelete_aggs = NULL;
	GSList *original_aggregates;
	gboolean insert;
	gint current_position = 0;

	/* here we get the complete list of aggregates, and for each aggregate, update or
	   create the entry in the list if not yet there. */
	rs = gda_connection_get_schema (GDA_CONNECTION (srv->priv->cnc),
					GDA_CONNECTION_SCHEMA_AGGREGATES, NULL);

	if (!rs) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_AGGREGATES_ERROR,
			     _("Can't get list of aggregates"));
		return FALSE;
	}


	if (!gnome_db_result_set_check_data_model (rs, 7, 
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING)) {
		g_set_error (error, GNOME_DB_SERVER_ERROR, GNOME_DB_SERVER_AGGREGATES_ERROR,
			     _("Schema for list of aggregates is wrong"));
		g_object_unref (G_OBJECT (rs));
		return FALSE;
	}

	original_aggregates = gnome_db_server_get_aggregates (srv);
	total = gda_data_model_get_n_rows (rs);
	now = 0;		
	while ((now < total) && !srv->priv->stop_update) {
		GnomeDbServerDataType *outdt = NULL; /* return type for the aggregate */
		GnomeDbServerDataType *indt = NULL;  /* argument for the aggregate */
		const GdaValue *value;
		
		insert = TRUE;

		/* fetch return type */
		value = gda_data_model_get_value_at (rs, 4, now);
		str = gda_value_stringify (value);
		if (*str != '-') {
			outdt = gnome_db_server_get_data_type_by_name (srv, str);
			if (!outdt)
				insert = FALSE;
		}
		else 
			insert = FALSE;
		g_free (str);

		/* fetch argument type */
		value = gda_data_model_get_value_at (rs, 5, now);
		str = gda_value_stringify (value);
		if (str) {
			if (*str != '-') {
				indt = gnome_db_server_get_data_type_by_name (srv, str);
				if (!indt)
					insert = FALSE;
			}
			g_free (str);
		}

		/* fetch a agg if there is already one with the same id */
		value = gda_data_model_get_value_at (rs, 1, now);
		str = gda_value_stringify (value);
		agg = gnome_db_server_get_aggregate_by_dbms_id (srv, str);
		g_free (str);

		if (!agg) {
			/* try to find the aggregate using its name, return type and argument type, 
			   and not its DBMS id, this is usefull if the DBMS has changed and the
			   DBMS id have changed */
			value =  gda_data_model_get_value_at (rs, 0, now);
			str = gda_value_stringify (value);
			agg = gnome_db_server_get_aggregate_by_name_arg_real (srv, original_aggregates, str, indt);
			g_free (str);

			if (agg && (gnome_db_server_aggregate_get_ret_type (agg) != outdt))
				agg = NULL;
		}

		if (!insert) 
			agg = NULL;
		else {
			if (agg) {
				/* does the aggregate we found have the same outdt and indt
				   as the one we have now? */
				gboolean isequal = TRUE;
				if (gnome_db_server_aggregate_get_arg_type (agg) != indt)
					isequal = FALSE;
				if (isequal && (gnome_db_server_aggregate_get_ret_type (agg) != outdt)) 
					isequal = FALSE;
				
				if (isequal) {
					updated_aggs = g_slist_append (updated_aggs, agg);
					current_position = g_slist_index (srv->priv->aggregates, agg) + 1;
					insert = FALSE;
				}
				else 
					agg = NULL;
			}

			if (!agg) {
				/* creating new ServerAggregate object */
				agg = GNOME_DB_SERVER_AGGREGATE (gnome_db_server_aggregate_new (srv));
				gnome_db_server_aggregate_set_ret_type (agg, outdt);
				gnome_db_server_aggregate_set_arg_type (agg, indt);
				
				/* mark aggregate as updated */
				updated_aggs = g_slist_append (updated_aggs, agg);
			}
		}
	
		/* aggregate's attributes update */
		if (agg) {
			/* unique id */
			value = gda_data_model_get_value_at (rs, 1, now);
			str = gda_value_stringify (value);
			gnome_db_server_aggregate_set_dbms_id (agg, str);
			g_free (str);

			/* description */
			value = gda_data_model_get_value_at (rs, 3, now);
			if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
				str = gda_value_stringify (value);
				gnome_db_base_set_description (GNOME_DB_BASE (agg), str);
				g_free (str);
			}
			
			/* sqlname */
			value =  gda_data_model_get_value_at (rs, 0, now);
			str = gda_value_stringify (value);
			gnome_db_server_aggregate_set_sqlname (agg, str);
			g_free (str);
			
			/* owner */
			value = gda_data_model_get_value_at (rs, 2, now);
			if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
				str = gda_value_stringify (value);
				gnome_db_base_set_owner (GNOME_DB_BASE (agg), str);
				g_free (str);
			}
			else
				gnome_db_base_set_owner (GNOME_DB_BASE (agg), NULL);
		}

		/* Real insertion */
		if (insert) {
			/* insertion in the list: finding where to insert the aggregate */
			srv->priv->aggregates = g_slist_insert (srv->priv->aggregates, agg, current_position++);
			gnome_db_base_connect_nullify (agg,
						 G_CALLBACK (nullified_aggregate_cb), srv);
			g_signal_connect (G_OBJECT (agg), "changed",
					  G_CALLBACK (updated_aggregate_cb), srv);

#ifdef debug_signal
			g_print (">> 'DATA_AGGREGATE_ADDED' from %s\n", __FUNCTION__);
#endif
			g_signal_emit_by_name (G_OBJECT (srv), "data_aggregate_added", agg);
#ifdef debug_signal
			g_print ("<< 'DATA_AGGREGATE_ADDED' from %s\n", __FUNCTION__);
#endif
		}

		g_signal_emit_by_name (G_OBJECT (srv), "update_progress", "AGGREGATES", now, total);
		now ++;
	}

	g_object_unref (G_OBJECT (rs));
	if (original_aggregates)
		g_slist_free (original_aggregates);

	/* cleanup for the aggregates which do not exist anymore */
        list = srv->priv->aggregates;
        while (list && !srv->priv->stop_update) {
		if (!g_slist_find (updated_aggs, list->data)) 
			todelete_aggs = g_slist_prepend (todelete_aggs, list->data);
		list = g_slist_next (list);
        }
	g_slist_free (updated_aggs);

	list = todelete_aggs;
	while (list) {
		gnome_db_base_nullify (GNOME_DB_BASE (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (todelete_aggs);
		
	g_signal_emit_by_name (G_OBJECT (srv), "update_progress", NULL, 0, 0);
	
	return TRUE;
}


static void
nullified_aggregate_cb (GnomeDbServerAggregate *agg, GnomeDbServer *srv)
{
	g_return_if_fail (g_slist_find (srv->priv->aggregates, agg));
	srv->priv->aggregates = g_slist_remove (srv->priv->aggregates, agg);

	g_signal_handlers_disconnect_by_func (G_OBJECT (agg), 
					      G_CALLBACK (nullified_aggregate_cb), srv);
	g_signal_handlers_disconnect_by_func (G_OBJECT (agg), 
					      G_CALLBACK (updated_aggregate_cb), srv);

#ifdef debug_signal
	g_print (">> 'DATA_AGGREGATE_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_aggregate_removed", agg);
#ifdef debug_signal
	g_print ("<< 'DATA_AGGREGATE_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_object_unref (G_OBJECT (agg));
}

static void
updated_aggregate_cb (GnomeDbServerAggregate *agg, GnomeDbServer *srv)
{
#ifdef debug_signal
	g_print (">> 'DATA_AGGREGATE_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (srv), "data_aggregate_updated", agg);
#ifdef debug_signal
	g_print ("<< 'DATA_AGGREGATE_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


/**
 * gnome_db_server_get_handler_by_name
 * @srv: a #GnomeDbServer object
 * @name: 
 *
 * Get the GnomeDbDataHandler from its name.
 *
 * Returns: the GnomeDbDataHandler object
 */
GnomeDbDataHandler *
gnome_db_server_get_handler_by_name (GnomeDbServer *srv, const gchar *name)
{
	GnomeDbDataHandler *dh = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (name && *name, NULL);
	
	list = srv->priv->handlers;
	while (list && !dh) {
		if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), name))
			dh = GNOME_DB_DATA_HANDLER (list->data);
		list = g_slist_next (list);
	}

	if (!dh) {
		if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (srv->priv->fallback_handler)), name))
			dh = srv->priv->fallback_handler;
	}
	
	return dh;
}


/**
 * gnome_db_server_get_handler_by_type
 * @srv: a #GnomeDbServer object
 * @type: 
 *
 * Get the GnomeDbDataHandler for a data type
 *
 * Returns: the GnomeDbDataHandler object
 */
GnomeDbDataHandler *
gnome_db_server_get_handler_by_type (GnomeDbServer *srv, GnomeDbServerDataType *type)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type), NULL);	

	return gnome_db_server_data_type_get_handler (type);
}


/**
 * gnome_db_server_get_handler_by_gda
 * @srv: a #GnomeDbServer object
 * @gda_type: 
 *
 * Get the GnomeDbDataHandler for a gda type: the first DataHandler which can handle
 * the requested type is returned.  If no good handler can be found, then a default
 * one will be provided. This function never returns NULL.
 *
 * Returns: the GnomeDbDataHandler object
 */
GnomeDbDataHandler *
gnome_db_server_get_handler_by_gda (GnomeDbServer *srv, GdaValueType gda_type)
{
	GSList *list;
	GnomeDbDataHandler *dh = NULL;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (gda_type, NULL);
	
	list = srv->priv->handlers;
	while (list && !dh) {
		if (gnome_db_data_handler_accepts_gda_type (GNOME_DB_DATA_HANDLER (list->data), gda_type))
			dh = GNOME_DB_DATA_HANDLER (list->data);
		list = g_slist_next (list);
	}

	if (!dh)
		dh = srv->priv->fallback_handler;

	return dh;
}


/**
 * gnome_db_server_get_plugin_handlers
 * @srv: a #GnomeDbServer object
 *
 * Get a list of all the GnomeDbDataHandler plugins used by the GnomeDbServer.
 *
 * Returns: an allocated list of plugins
 */
GSList *
gnome_db_server_get_plugin_handlers (GnomeDbServer *srv)
{
	GSList *list, *retval = NULL;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	
	list = srv->priv->handlers;	
	while (list) {
		if (gnome_db_data_handler_is_plugin (GNOME_DB_DATA_HANDLER (list->data)))
			retval = g_slist_append (retval, list->data);
		list = g_slist_next (list);
	}

	return retval;
}


/**
 * gnome_db_server_get_user_by_name
 * @srv: a #GnomeDbServer object
 * @username: 
 *
 * Find a GnomeDbUser from its name.
 *
 * Returns: a pointer to the requested object, or NULL if the object cannot be found.
 */
GnomeDbUser *
gnome_db_server_get_user_by_name (GnomeDbServer *srv, const gchar *username)
{
	GnomeDbUser *user = NULL;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (username && *username, NULL);
	
	TO_IMPLEMENT;

	return user;
}

/**
 * gnome_db_server_get_data_types
 * @srv: a #GnomeDbServer object
 *
 * Get the list of data types;
 *
 * Returns: the list (the caller must free the list after usage)
 */
GSList *
gnome_db_server_get_data_types (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	if (srv->priv->data_types)
		return g_slist_copy (srv->priv->data_types);
	else
		return NULL;
}

/**
 * gnome_db_server_get_data_type_by_name
 * @srv: a #GnomeDbServer object
 * @typename: the name of the requested data type
 *
 * Find a data type from its DBMS name or from one of its synonyms if it has some.
 *
 * Returns: the data type or %NULL if it cannot be found
 */
GnomeDbServerDataType  *
gnome_db_server_get_data_type_by_name (GnomeDbServer *srv, const gchar *typename)
{
	GSList *list;
	GnomeDbServerDataType *datatype = NULL;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (typename && *typename, NULL);
	
	/* compare the data types names */
	list = srv->priv->data_types;
	while (list && !datatype) {
		if (!strcmp (gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (list->data)),
			     typename))
			datatype = GNOME_DB_SERVER_DATA_TYPE (list->data);
		list = g_slist_next (list);
	}

	/* if not found, then compare with the synonyms */
	list = srv->priv->data_types;
	while (list && !datatype) {
		GSList *synlist = gnome_db_server_data_type_get_synonyms (GNOME_DB_SERVER_DATA_TYPE (list->data));
		while (synlist && !datatype) {
			if (!strcmp ((gchar *) (synlist->data), typename))
				datatype = GNOME_DB_SERVER_DATA_TYPE (list->data);
			synlist = g_slist_next (synlist);
		}
		list = g_slist_next (list);
	}

	return datatype;
}

/**
 * gnome_db_server_add_data_type 
 * @srv:
 * @datatype:
 *
 * Adds a data type to @srv
 *
 * Reserved for internal usage only.
 */
void
gnome_db_server_add_data_type (GnomeDbServer *srv, GnomeDbServerDataType *datatype)
{
	gint i = 0;
	GSList *list;
	gboolean found = FALSE;
	const gchar *str;

	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);
	g_return_if_fail (IS_GNOME_DB_SERVER_DATA_TYPE (datatype));
	str = gnome_db_server_data_type_get_sqlname (datatype);
	g_return_if_fail (! gnome_db_server_get_data_type_by_name (srv, str));

	/* finding the right position for the data type */
	list = srv->priv->data_types;
	while (list && !found) {
		if (strcmp (str,
			    gnome_db_server_data_type_get_sqlname (GNOME_DB_SERVER_DATA_TYPE (list->data))) < 0)
			found = TRUE;
		else
			i++;
		list = g_slist_next (list);
	}
	srv->priv->data_types = g_slist_insert (srv->priv->data_types, datatype, i);
	g_object_ref (datatype);

	/* signals */
	gnome_db_base_connect_nullify (datatype, G_CALLBACK (nullified_data_type_cb), srv);
	g_signal_connect (G_OBJECT (datatype), "changed",
			  G_CALLBACK (updated_data_type_cb), srv);
#ifdef debug_signal
	g_print (">> 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (srv), gnome_db_server_signals[DATA_TYPE_ADDED], 0, datatype);
#ifdef debug_signal
	g_print ("<< 'DATA_TYPE_ADDED' from %s\n", __FUNCTION__);
#endif
}

/**
 * gnome_db_server_get_data_type_by_xml_id
 * @srv: a #GnomeDbServer object
 * @xml_id: the XML identifier of the data type to be found
 *
 * To find a #GnomeDbServerDataType using its XML id.
 *
 * Returns: the data type or %NULL if it cannot be found
 */
GnomeDbServerDataType  *
gnome_db_server_get_data_type_by_xml_id (GnomeDbServer *srv, const gchar *xml_id)
{
	GSList *list;
	GnomeDbServerDataType *datatype = NULL;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);
	
	list = srv->priv->data_types;
	while (list && !datatype) {
		gchar *id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		if (!strcmp (id, xml_id))
			datatype = GNOME_DB_SERVER_DATA_TYPE (list->data);
		g_free (id);
		list = g_slist_next (list);
	}

	return datatype;
}

/**
 * gnome_db_server_declare_custom_data_type
 * @srv: a #GnomeDbServer object
 * @type: a #GnomeDbServerDataType object
 *
 * Forces @srv to consider @type as a new data type even though that data type
 * is not declared by the database to which @srv can be connected to.
 *
 * Returns: TRUE if the data type does not already exist, and FALSE if it does.
 */
gboolean
gnome_db_server_declare_custom_data_type (GnomeDbServer *srv, GnomeDbServerDataType *type)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	g_return_val_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type), FALSE);

	if (gnome_db_server_get_data_type_by_name (srv, gnome_db_server_data_type_get_sqlname (type)))
		return FALSE;
	else {
		gnome_db_server_add_data_type (srv, type);
		srv->priv->custom_types = g_slist_append (srv->priv->custom_types, type);
		g_object_unref (type);
	}
}

/**
 * gnome_db_server_get_functions
 * @srv: a #GnomeDbServer object
 *
 * To get the complete list of functions
 *
 * Returns: the allocated list of functions
 */
GSList *
gnome_db_server_get_functions (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	if (srv->priv->functions)
		return g_slist_copy (srv->priv->functions);
	else
		return NULL;
}


/**
 * gnome_db_server_get_functions_by_name
 * @srv: a #GnomeDbServer object
 * @funcname: name of the function
 *
 * To get the list of DBMS functions which match the given name.
 *
 * Returns: the allocated list of functions
 */
GSList *
gnome_db_server_get_functions_by_name (GnomeDbServer *srv, const gchar *funcname)
{
	GSList *retval = NULL, *list;
	gchar *cmpstr = NULL;
	gchar *cmpstr2;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (funcname && *funcname, NULL);

	if (LC_NAMES (srv))
		cmpstr = g_utf8_strdown (funcname, -1);
	else
		cmpstr = funcname;

	list = srv->priv->functions;
	while (list) {
		if (LC_NAMES (srv)) {
			cmpstr2 = g_utf8_strdown (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), -1);
			if (!strcmp (cmpstr2, cmpstr))
				retval = g_slist_prepend (retval, list->data);
			g_free (cmpstr2);
		}
		else
			if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), cmpstr))
				retval = g_slist_prepend (retval, list->data);
		list = g_slist_next (list);
	}

	if (LC_NAMES (srv))
		g_free (cmpstr);

	return retval;
}


/**
 * gnome_db_server_get_function_by_name_arg
 * @srv: a #GnomeDbServer object
 * @funcname: name of the function
 * @argtypes: a list of #GnomeDbServerDataType objects
 *
 * To find a DBMS function which is uniquely identified by its name and the type(s)
 * of its argument(s).
 *
 * About the functions accepting any data type for one of their argument: if the corresponding data type in
 * @argtypes is not %NULL, then such a function will be a candidate, and if the corresponding data type in
 * @argtypes is %NULL, then only such a function will be a candidate.
 *
 * Returns: The function or NULL if not found
 */
GnomeDbServerFunction *
gnome_db_server_get_function_by_name_arg (GnomeDbServer *srv, const gchar *funcname, const GSList *argtypes)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (funcname && *funcname, NULL);

	return gnome_db_server_get_function_by_name_arg_real (srv, srv->priv->functions, funcname, argtypes);
}

static GnomeDbServerFunction *
gnome_db_server_get_function_by_name_arg_real (GnomeDbServer *srv, GSList *functions, const gchar *funcname, const GSList *argtypes)
{
	GnomeDbServerFunction *func = NULL; /* prefectely matching function */
	GnomeDbServerFunction *anytypefunc = NULL; /* matching because it accepts any data type for one of its args */
	GnomeDbServerFunction *gdatypefunc = NULL; /* matching because its arg type is the same gda type as requested for one of its args */
	GSList *list;
	gchar *cmpstr = NULL;
	gchar *cmpstr2;

	if (LC_NAMES (srv))
		cmpstr = g_utf8_strdown (funcname, -1);
	else
		cmpstr = funcname;

	list = functions;
	while (list && !func) {
		gboolean argsok = TRUE;
		gboolean func_any_type = FALSE;
		gboolean func_gda_type = FALSE;
		GSList *funcargs, *list2;
		GnomeDbServerFunction *tmp = NULL;

		/* arguments comparison */
		list2 = argtypes;
		funcargs = gnome_db_server_function_get_arg_types (GNOME_DB_SERVER_FUNCTION (list->data));
		while (funcargs && list2 && argsok) {
			gboolean tmpok = FALSE;

			if (funcargs->data == list2->data)
				tmpok = TRUE;
			else {
				if (!funcargs->data) {
					tmpok = TRUE;
					func_any_type = TRUE;
				}
				else {
					if (funcargs->data && list2->data &&
					    srv->priv->info && 
					    srv->priv->info->implicit_data_types_casts &&
					    (gnome_db_server_data_type_get_gda_type (funcargs->data) == 
					     gnome_db_server_data_type_get_gda_type (list2->data))) {
						tmpok = TRUE;
						func_gda_type = TRUE;
					}
				}
			}
			
			if (!tmpok)
				argsok = FALSE;
			funcargs = g_slist_next (funcargs);
			list2 = g_slist_next (list2);
		}
		if (list2 || funcargs)
			argsok = FALSE;

		/* names comparison */
		if (argsok) {
			if (LC_NAMES (srv)) {
				cmpstr2 = g_utf8_strdown (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), -1);
				if (!strcmp (cmpstr2, cmpstr))
					tmp = GNOME_DB_SERVER_FUNCTION (list->data);
				g_free (cmpstr2);
			}
			else
				if (!strcmp (cmpstr, gnome_db_base_get_name (GNOME_DB_BASE (list->data))))
					tmp = GNOME_DB_SERVER_FUNCTION (list->data);
		}

		if (tmp) {
			if (func_any_type)
				anytypefunc = tmp;
			else {
				if (func_gda_type)
					gdatypefunc = tmp;
				else
					func = tmp;
			}
		}
		
		list = g_slist_next (list);
	}

	if (!func && gdatypefunc)
		func = gdatypefunc;
	if (!func && anytypefunc)
		func = anytypefunc;

	if (LC_NAMES (srv))
		g_free (cmpstr);

	return func;
}

/**
 * gnome_db_server_get_function_by_dbms_id
 * @srv: a #GnomeDbServer object
 * @dbms_id: 
 *
 * To find a DBMS functions which is uniquely identified by its DBMS identifier
 *
 * Returns: The function or NULL if not found
 */
GnomeDbServerFunction *
gnome_db_server_get_function_by_dbms_id (GnomeDbServer *srv, const gchar *dbms_id)
{
	GnomeDbServerFunction *func = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (dbms_id && *dbms_id, NULL);

	list = srv->priv->functions;
	while (list && !func) {
		const gchar *str = gnome_db_server_function_get_dbms_id (GNOME_DB_SERVER_FUNCTION (list->data));
		if (!str || ! *str) {
			str = gnome_db_server_function_get_sqlname (GNOME_DB_SERVER_FUNCTION (list->data));
			g_error ("Function %p (%s) has no dbms_id", list->data, str);
		}
		if (str && !strcmp (dbms_id, str))
			func = GNOME_DB_SERVER_FUNCTION (list->data);
		list = g_slist_next (list);
	}

	return func;
}

/**
 * gnome_db_server_get_function_by_xml_id
 * @srv: a #GnomeDbServer object
 * @xml_id: 
 *
 * To find a DBMS functions which is uniquely identified by its XML identifier
 *
 * Returns: The function or NULL if not found
 */
GnomeDbServerFunction *
gnome_db_server_get_function_by_xml_id (GnomeDbServer *srv, const gchar *xml_id)
{
	GnomeDbServerFunction *func = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);

	list = srv->priv->functions;
	while (list && !func) {
		const gchar *str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		if (!strcmp (xml_id, str))
			func = GNOME_DB_SERVER_FUNCTION (list->data);
		list = g_slist_next (list);
	}

	return func;
}



/**
 * gnome_db_server_get_aggregates
 * @srv: a #GnomeDbServer object
 *
 * To get the complete list of aggregates
 *
 * Returns: the allocated list of aggregates
 */
GSList *
gnome_db_server_get_aggregates (GnomeDbServer *srv)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	if (srv->priv->functions)
		return g_slist_copy (srv->priv->aggregates);
	else
		return NULL;
}

/**
 * gnome_db_server_get_aggregates_by_name
 * @srv: a #GnomeDbServer object
 * @aggname: the name of the aggregate
 *
 * To get the list of DBMS aggregates which match the given name.
 *
 * Returns: the allocated list of aggregates
 */
GSList *
gnome_db_server_get_aggregates_by_name (GnomeDbServer *srv, const gchar *aggname)
{
	GSList *retval = NULL, *list;
	gchar *cmpstr = NULL;
	gchar *cmpstr2;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (aggname && *aggname, NULL);

	if (LC_NAMES (srv))
		cmpstr = g_utf8_strdown (aggname, -1);
	else
		cmpstr = aggname;

	list = srv->priv->aggregates;
	while (list) {
		if (LC_NAMES (srv)) {
			cmpstr2 = g_utf8_strdown (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), -1);
			if (!strcmp (cmpstr2, cmpstr))
				retval = g_slist_prepend (retval, list->data);
			g_free (cmpstr2);
		}
		else
			if (!strcmp (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), cmpstr))
				retval = g_slist_prepend (retval, list->data);
		list = g_slist_next (list);
	}

	if (LC_NAMES (srv))
		g_free (cmpstr);

	return retval;;
}

/**
 * gnome_db_server_get_aggregate_by_name_arg
 * @srv: a #GnomeDbServer object
 * @aggname: the name of the aggregate
 * @argtype: the type of argument or %NULL
 *
 * To find a DBMS aggregate which is uniquely identified by its name and the type
 * of its argument.
 *
 * About the aggregates accepting any data type for their argument: if @argtype is not %NULL
 * then such an aggregate will be a candidate, and if @argtype is %NULL
 * then only such an aggregate will be a candidate.
 *
 * If several aggregates are found, then the aggregate completely matching will be returned, or
 * an aggregate where the argument type has the same GDA typa as the @argtype, or lastly an
 * aggregate accepting any data type as argument.
 *
 * Returns: The aggregate or NULL if not found
 */
GnomeDbServerAggregate *
gnome_db_server_get_aggregate_by_name_arg (GnomeDbServer *srv, const gchar *aggname, GnomeDbServerDataType *argtype)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (aggname && *aggname, NULL);
	if (argtype)
		g_return_val_if_fail (IS_GNOME_DB_SERVER_DATA_TYPE (argtype), NULL);

	return gnome_db_server_get_aggregate_by_name_arg_real (srv, srv->priv->aggregates, aggname, argtype);
}

GnomeDbServerAggregate *
gnome_db_server_get_aggregate_by_name_arg_real (GnomeDbServer *srv, GSList *aggregates, const gchar *aggname, 
					  GnomeDbServerDataType *argtype)
{
	GnomeDbServerAggregate *agg = NULL; /* prefectely matching aggregate */
	GnomeDbServerAggregate *anytypeagg = NULL; /* matching because it accepts any data type */
	GnomeDbServerAggregate *gdatypeagg = NULL; /* matching because its arg type is the same gda type as requested */
	GSList *list;
	gchar *cmpstr = NULL;
	gchar *cmpstr2;

	if (LC_NAMES (srv))
		cmpstr = g_utf8_strdown (aggname, -1);
	else
		cmpstr = aggname;

	list = aggregates;
	while (list && !agg) {
		GnomeDbServerDataType *testdt = gnome_db_server_aggregate_get_arg_type (GNOME_DB_SERVER_AGGREGATE (list->data));
		GnomeDbServerAggregate *tmp = NULL;
		gint mode = 0;

		if (argtype == testdt) {
			tmp = GNOME_DB_SERVER_AGGREGATE (list->data);
			mode = 1;
		}
		else {
			if (!testdt) {
				tmp = GNOME_DB_SERVER_AGGREGATE (list->data);
				mode = 2;
			}
			else {
				if (argtype && testdt &&
				    srv->priv->info && 
				    srv->priv->info->implicit_data_types_casts &&
				    (gnome_db_server_data_type_get_gda_type (testdt) == 
				     gnome_db_server_data_type_get_gda_type (argtype))) {
					tmp = GNOME_DB_SERVER_AGGREGATE (list->data);
					mode = 3;
				}
			}
		}

		if (tmp) {
			if (LC_NAMES (srv)) {
				cmpstr2 = g_utf8_strdown (gnome_db_base_get_name (GNOME_DB_BASE (list->data)), -1);
				if (strcmp (cmpstr2, cmpstr))
					tmp = NULL;
				g_free (cmpstr2);
			}
			else
				if (strcmp (cmpstr, gnome_db_base_get_name (GNOME_DB_BASE (list->data))))
					tmp = NULL;
		}

		if (tmp) {
			switch (mode) {
			case 1:
				agg = tmp;
				break;
			case 2:
				anytypeagg = tmp;
				break;
			case 3:
				gdatypeagg = tmp;
				break;
			default:
				g_assert_not_reached ();
				break;
			}
		}
		
		list = g_slist_next (list);
	}

	if (!agg && gdatypeagg)
		agg = gdatypeagg;

	if (!agg && anytypeagg)
		agg = anytypeagg;

	if (LC_NAMES (srv))
		g_free (cmpstr);

	return agg;	
}

/**
 * gnome_db_server_get_aggregate_by_dbms_id
 * @srv: a #GnomeDbServer object
 * @dbms_id: 
 *
 * To find a DBMS functions which is uniquely identified by its name and the type
 * of its argument.
 *
 * Returns: The aggregate or NULL if not found
 */
GnomeDbServerAggregate *
gnome_db_server_get_aggregate_by_dbms_id (GnomeDbServer *srv, const gchar *dbms_id)
{
	GnomeDbServerAggregate *agg = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (dbms_id && *dbms_id, NULL);

	list = srv->priv->aggregates;
	while (list && !agg) {
		if (!strcmp (dbms_id, gnome_db_server_aggregate_get_dbms_id (GNOME_DB_SERVER_AGGREGATE (list->data))))
			agg = GNOME_DB_SERVER_AGGREGATE (list->data);
		list = g_slist_next (list);
	}

	return agg;
}

/**
 * gnome_db_server_get_aggregate_by_xml_id
 * @srv: a #GnomeDbServer object
 * @xml_id: 
 *
 * To find a DBMS aggregates which is uniquely identified by its XML identifier
 *
 * Returns: The aggregate or NULL if not found
 */
GnomeDbServerAggregate *
gnome_db_server_get_aggregate_by_xml_id (GnomeDbServer *srv, const gchar *xml_id)
{
	GnomeDbServerAggregate *agg = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (xml_id && *xml_id, NULL);

	list = srv->priv->aggregates;
	while (list && !agg) {
		const gchar *str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		if (!strcmp (xml_id, str))
			agg = GNOME_DB_SERVER_AGGREGATE (list->data);
		list = g_slist_next (list);
	}

	return agg;
}


/**
 * gnome_db_server_get_object_handler
 * @srv: a #GnomeDbServer object
 * @object: a #GObject object
 *
 * Get the right GnomeDbDataHandler object reference to manage data which is
 * "linked" to the object. The object will usually be a GnomeDbServerDataType, 
 * a GnomeDbServerFunction, or a GnomeDbServerAggregate.
 *
 * The returned GnomeDbDataHandler depends on the loaded plugins and on the user
 * preferences regarding how these plugins are to be used. If the user has not
 * set any preference for the object, then some default rules are used:
 * <ul>
 * <li> for GnomeDbServerDataType, the GnomeDbDataHandler corresponding to the gda type
 *    of the GnomeDbServerDataType is returned</li></ul>
 * -> for GnomeDbServerFunction, the GnomeDbDataHandler corresponding to the GnomeDbServerDataType
 *    returned by the function is returned
 * -> ...
 *
 * Returns: the GnomeDbDataHandler associated to the given object, NEVER returns NULL.
 */
GnomeDbDataHandler *
gnome_db_server_get_object_handler (GnomeDbServer *srv, GObject *object)
{
        GnomeDbDataHandler *dh = NULL;
	GSList *list;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (object && G_IS_OBJECT (object), NULL);
	
	if ((dh = g_hash_table_lookup (srv->priv->types_objects_hash, object)))
		return dh;

	/* going through all the function handlers we have */
	list = srv->priv->handlers_functions;
	while (list && !dh) {
		dh = (HANDLER_FUNCTION (list->data)) (srv, object);
		list = g_slist_next (list);
	}

	if (!dh)
		return srv->priv->fallback_handler;
	else
		return dh;
}

/* called when an object which is a key in the hash table is destroyed */
static void
hash_object_destroyed_n (GnomeDbServer *srv, GObject *object) 
{
	g_hash_table_remove (srv->priv->types_objects_hash, object);
}

/**
 * gnome_db_server_set_object_handler
 * @srv: a #GnomeDbServer object
 * @object: a #GObject
 * @handler: 
 *
 * This function is the opposite of the gnome_db_server_get_object_handler() function:
 * it "attaches" a GnomeDbDataHandler object to any given object, and a subsequent call
 * to gnome_db_server_get_object_handler(object) will return the GnomeDbDataHandler object.
 */
void
gnome_db_server_set_object_handler (GnomeDbServer *srv, GObject *object, GnomeDbDataHandler *handler)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);
	g_return_if_fail (object && G_IS_OBJECT (object));
	if (handler)
		g_return_if_fail (handler && IS_GNOME_DB_DATA_HANDLER (handler));
	
	if (handler) {
		g_hash_table_insert (srv->priv->types_objects_hash, object, handler);
		g_object_weak_ref (object, (GWeakNotify) hash_object_destroyed_n, srv);
	}
	else {
		g_hash_table_remove (srv->priv->types_objects_hash, object);
		g_object_weak_ref (object, (GWeakNotify) hash_object_destroyed_n, srv);
	}
}

/**
 * gnome_db_server_unset_object_handler
 * @srv: a #GnomeDbServer object
 * @object: a #GObject
 *
 * Shortcut to gnome_db_server_set_object_handler() with NULL as "handler" argument.
 */
void
gnome_db_server_unset_object_handler (GnomeDbServer *srv, GObject *object)
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);
	g_return_if_fail (object && G_IS_OBJECT (object));
	
	gnome_db_server_set_object_handler (srv, object, NULL);	
}

/**
 * gnome_db_server_object_has_handler
 * @srv: a #GnomeDbServer object
 * @object: a #GObject
 *
 * Tells if a GnomeDbDataHandler object has been assigned to the object given as argument, or
 * if the GnomeDbDataHandler which would be returned by the gnome_db_server_get_object_handler()
 * function is a default one.
 *
 * Returns: TRUE if a GnomeDbDataHandler object has been assigned to the object.
 */
gboolean
gnome_db_server_object_has_handler (GnomeDbServer *srv, GObject *object)
{
	gboolean found = FALSE;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), FALSE);
	g_return_val_if_fail (srv->priv, FALSE);
	g_return_val_if_fail (object && G_IS_OBJECT (object), FALSE);

	if (g_hash_table_lookup (srv->priv->types_objects_hash, object))
		found = TRUE;

	return found;
}

/**
 * gnome_db_server_set_object_func_handler
 * @srv: a #GnomeDbServer object
 * @func: a function
 *
 * This function provides a way for the GnomeDbServer function to apply rules to
 * find an appropriate GnomeDbDataHandler for an object.
 */
void
gnome_db_server_set_object_func_handler (GnomeDbServer *srv, GnomeDbDataHandler *(*func) (GnomeDbServer *, GObject *))
{
	g_return_if_fail (srv && IS_GNOME_DB_SERVER (srv));
	g_return_if_fail (srv->priv);
	g_return_if_fail (func);
	
	if (g_slist_find (srv->priv->handlers_functions, func))
		return;

	srv->priv->handlers_functions = g_slist_append (srv->priv->handlers_functions, func);
}

static GnomeDbDataHandler *
func_handler_data_types (GnomeDbServer *srv, GObject *obj)
{
	if (IS_GNOME_DB_SERVER_DATA_TYPE (obj)) {
		GnomeDbServerDataType *dt = GNOME_DB_SERVER_DATA_TYPE (obj);
		return gnome_db_server_get_handler_by_gda (srv, 
						     gnome_db_server_data_type_get_gda_type (dt));
	}
	else
		return NULL;
}

/**
 * gnome_db_server_stringify_value
 * @srv: a #GnomeDbServer object
 * @value: a value to be stringified
 *
 * Renders a value as a string, and ensures the string is UTF-8 encoded.
 *
 * Returns: the new string
 */
gchar *
gnome_db_server_stringify_value (GnomeDbServer *srv, const GdaValue * value)
{
	gchar *retval, *str;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	g_return_val_if_fail (value, NULL);

        str = gda_value_stringify (value);
        retval = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
        g_free (str);

        return retval;
}

/**
 * gnome_db_server_escape_chars
 * @srv: a #GnomeDbServer object
 * @string: 
 *
 * Escapes the special caracters from a string. The new string can then safely be sent
 * to the DBMS.
 *
 * Returns: the escaped string
 */
gchar *
gnome_db_server_escape_chars (GnomeDbServer *srv, const gchar *string)
{
	gchar *str, *ptr, *ret, *retptr;
	gint size;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);
	if (!string)
		return NULL;
	
	str = g_strdup (string);
	ptr = str;

	/* determination of the new string size */
	size = 1;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str)
				size += 2;
			else {
				if (*(ptr - 1) == '\\')
					size += 1;
				else
					size += 2;
			}
		}
		else
			size += 1;
		ptr++;
	}

	ptr = str;
	ret = (gchar *) malloc (sizeof (gchar) * size);
	retptr = ret;
	while (*ptr != '\0') {
		if (*ptr == '\'') {
			if (ptr == str) {
				*retptr = '\\';
				retptr++;
			}
			else if (*(ptr - 1) != '\\') {
				*retptr = '\\';
				retptr++;
			}
		}
		*retptr = *ptr;
		retptr++;
		ptr++;
	}
	*retptr = '\0';
	g_free (str);

	return ret;
}


/**
 * gnome_db_server_get_gda_schema
 * @srv: a #GnomeDbServer object
 * @schema: the requested schema
 * @params: some parameters for the requested schema, or NULL
 *
 * Get a direct access to the libgda's function call to get a DBMS schema. It should not directely be used
 * since libgnomedb hides any necessary call to it.
 *
 * Returns: the data model, or NULL if an error occurred
 */
GdaDataModel *
gnome_db_server_get_gda_schema (GnomeDbServer *srv, GdaConnectionSchema schema, GdaParameterList *params)
{
	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	g_return_val_if_fail (srv->priv, NULL);

	return gda_connection_get_schema (GDA_CONNECTION (srv->priv->cnc), schema, params);
}
