/* gnome-db-qf-value.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-qf-value.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "marshal.h"
#include "gnome-db-entity.h"
#include "gnome-db-query.h"
#include <string.h>
#include "gnome-db-server.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-parameter.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-data-set.h"

/* 
 * Main static functions 
 */
static void gnome_db_qf_value_class_init (GnomeDbQfValueClass * class);
static void gnome_db_qf_value_init (GnomeDbQfValue *qf);
static void gnome_db_qf_value_dispose (GObject *object);
static void gnome_db_qf_value_finalize (GObject *object);

static void gnome_db_qf_value_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_qf_value_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_qf_value_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_qf_value_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_qf_value_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_qf_value_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void              gnome_db_qf_value_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_qf_value_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_qf_value_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_qf_value_renderer_init      (GnomeDbRendererIface *iface);
static GdaXqlItem     *gnome_db_qf_value_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar          *gnome_db_qf_value_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_qf_value_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_qf_value_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_qf_value_activate            (GnomeDbReferer *iface);
static void        gnome_db_qf_value_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_qf_value_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_qf_value_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_qf_value_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* virtual functions */
static GObject    *gnome_db_qf_value_copy           (GnomeDbQfield *orig);
static gboolean    gnome_db_qf_value_is_equal       (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2);


/* When the GnomeDbQuery or GnomeDbTarget is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbQfValue *field);

static GSList   *gnome_db_qf_value_get_params (GnomeDbQfield *qfield);
static gboolean  gnome_db_qf_value_restrict_with_field (GnomeDbQfValue *field, GnomeDbQfield *provider, GError **error);
static gboolean  gnome_db_qf_value_restrict_with_field_xml (GnomeDbQfValue *field, const gchar *prov_xml_id, GError **error);

static gboolean gnome_db_qf_value_render_find_value (GnomeDbQfValue *field, GnomeDbDataSet *context,
					       const GdaValue **value_found, GnomeDbParameter **param_source);

#ifdef debug
static void gnome_db_qf_value_dump (GnomeDbQfValue *field, guint offset);
#endif

/* get a pointer to the parents to be able to cvalue their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_qf_value_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY,
	PROP_VALUE_PROVIDER_OBJECT,
        PROP_VALUE_PROVIDER_XML_ID,
	PROP_HANDLER_PLUGIN
};


/* private structure */
struct _GnomeDbQfValuePrivate
{
	GnomeDbQuery          *query;
	GdaValueType      gda_type;
	GnomeDbServerDataType *srv_type;
	GdaValue         *value;        /* MUST either be NULL, or of type GDA_VALUE_NULL or 'type' */
	GdaValue         *default_value;/* CAN either be NULL, or of any type */
	gboolean          is_parameter;
	gboolean          is_null_allowed;
	GnomeDbRefBase        *value_prov_ref; /* to a GnomeDbQfield of another query */

	gchar            *plugin;       /* specific plugin to be used */
};


/* module error */
GQuark gnome_db_qf_value_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qf_value_error");
	return quark;
}


guint
gnome_db_qf_value_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfValueClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qf_value_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfValue),
			0,
			(GInstanceInitFunc) gnome_db_qf_value_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_qf_value_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_qf_value_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_qf_value_renderer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_qf_value_referer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_QFIELD_TYPE, "GnomeDbQfValue", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_FIELD_TYPE, &field_info);
		g_type_add_interface_static (type, GNOME_DB_RENDERER_TYPE, &renderer_info);
		g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void 
gnome_db_qf_value_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_qf_value_get_xml_id;
	iface->save_to_xml = gnome_db_qf_value_save_to_xml;
	iface->load_from_xml = gnome_db_qf_value_load_from_xml;
}

static void
gnome_db_qf_value_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_qf_value_get_entity;
	iface->get_data_type = gnome_db_qf_value_get_data_type;
}

static void
gnome_db_qf_value_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_qf_value_render_as_xql;
	iface->render_as_sql = gnome_db_qf_value_render_as_sql;
	iface->render_as_str = gnome_db_qf_value_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_qf_value_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_qf_value_activate;
        iface->deactivate = gnome_db_qf_value_deactivate;
        iface->is_active = gnome_db_qf_value_is_active;
        iface->get_ref_objects = gnome_db_qf_value_get_ref_objects;
        iface->replace_refs = gnome_db_qf_value_replace_refs;
}

static void
gnome_db_qf_value_class_init (GnomeDbQfValueClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_qf_value_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQfValueClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_qf_value_dispose;
	object_class->finalize = gnome_db_qf_value_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qf_value_set_property;
	object_class->get_property = gnome_db_qf_value_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_VALUE_PROVIDER_OBJECT,
                                         g_param_spec_pointer ("value_provider", NULL, NULL,
                                                               (G_PARAM_READABLE | G_PARAM_WRITABLE)));
        g_object_class_install_property (object_class, PROP_VALUE_PROVIDER_XML_ID,
                                         g_param_spec_string ("value_provider_xml_id", NULL, NULL, NULL,
                                                              (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_HANDLER_PLUGIN,
                                         g_param_spec_string ("handler_plugin", NULL, NULL, NULL,
                                                              (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_qf_value_dump;
#endif
	GNOME_DB_QFIELD_CLASS (class)->copy = gnome_db_qf_value_copy;
	GNOME_DB_QFIELD_CLASS (class)->is_equal = gnome_db_qf_value_is_equal;
	GNOME_DB_QFIELD_CLASS (class)->is_list = NULL;
	GNOME_DB_QFIELD_CLASS (class)->get_params = gnome_db_qf_value_get_params;
}

static void
gnome_db_qf_value_init (GnomeDbQfValue *gnome_db_qf_value)
{
	gnome_db_qf_value->priv = g_new0 (GnomeDbQfValuePrivate, 1);
	gnome_db_qf_value->priv->query = NULL;
	gnome_db_qf_value->priv->gda_type = GDA_VALUE_TYPE_UNKNOWN;
	gnome_db_qf_value->priv->srv_type = NULL;
	gnome_db_qf_value->priv->value = NULL;
	gnome_db_qf_value->priv->default_value = NULL;
	gnome_db_qf_value->priv->is_parameter = FALSE;
	gnome_db_qf_value->priv->is_null_allowed = FALSE;
	gnome_db_qf_value->priv->value_prov_ref = NULL;
	gnome_db_qf_value->priv->plugin = NULL;
}


/**
 * gnome_db_qf_value_new
 * @query: a #GnomeDbQuery in which the new object will be
 * @type: the requested type for the value
 *
 * Creates a new GnomeDbQfValue object which represents a value or a parameter.
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_value_new (GnomeDbQuery *query, GnomeDbServerDataType *type)
{
	GObject   *obj;
	GnomeDbQfValue *gnome_db_qf_value;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type), NULL);
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_QF_VALUE_TYPE, "dict", dict, NULL);
	gnome_db_qf_value = GNOME_DB_QF_VALUE (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_value), id);

	gnome_db_qf_value->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_value);

	gnome_db_qf_value->priv->srv_type = type;
	gnome_db_base_connect_nullify (type,
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_value);

	gnome_db_qf_value->priv->gda_type = gnome_db_server_data_type_get_gda_type (type);
	
	return obj;
}


static void 
nullified_object_cb (GObject *obj, GnomeDbQfValue *field)
{
	gnome_db_base_nullify (GNOME_DB_BASE (field));
}

static void
gnome_db_qf_value_dispose (GObject *object)
{
	GnomeDbQfValue *gnome_db_qf_value;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_VALUE (object));

	gnome_db_qf_value = GNOME_DB_QF_VALUE (object);
	if (gnome_db_qf_value->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));
		if (gnome_db_qf_value->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_value->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_value);
			gnome_db_qf_value->priv->query = NULL;
		}
		if (gnome_db_qf_value->priv->srv_type) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_value->priv->srv_type),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_value);
			gnome_db_qf_value->priv->srv_type = NULL;
		}
		if (gnome_db_qf_value->priv->value) {
			gda_value_free (gnome_db_qf_value->priv->value);
			gnome_db_qf_value->priv->value = NULL;
		}
		if (gnome_db_qf_value->priv->default_value) {
			gda_value_free (gnome_db_qf_value->priv->default_value);
			gnome_db_qf_value->priv->default_value = NULL;
		}
		if (gnome_db_qf_value->priv->value_prov_ref)
			gnome_db_qf_value_restrict_with_field (gnome_db_qf_value, NULL, NULL);
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qf_value_finalize (GObject   * object)
{
	GnomeDbQfValue *gnome_db_qf_value;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_VALUE (object));

	gnome_db_qf_value = GNOME_DB_QF_VALUE (object);
	if (gnome_db_qf_value->priv) {
		if (gnome_db_qf_value->priv->plugin)
			g_free (gnome_db_qf_value->priv->plugin);

		g_free (gnome_db_qf_value->priv);
		gnome_db_qf_value->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qf_value_set_property (GObject              *object,
			  guint                 param_id,
			  const GValue         *value,
			  GParamSpec           *pspec)
{
	GnomeDbQfValue *gnome_db_qf_value;
	gpointer ptr;
	const gchar *str;

	gnome_db_qf_value = GNOME_DB_QF_VALUE (object);
	if (gnome_db_qf_value->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

			if (gnome_db_qf_value->priv->query) {
				if (gnome_db_qf_value->priv->query == GNOME_DB_QUERY (ptr))
					return;

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_value->priv->query),
								      G_CALLBACK (nullified_object_cb), gnome_db_qf_value);
			}

			gnome_db_qf_value->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr,
						 G_CALLBACK (nullified_object_cb), gnome_db_qf_value);
			break;
		case PROP_VALUE_PROVIDER_OBJECT:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (gnome_db_qf_value_restrict_with_field (gnome_db_qf_value, (GnomeDbQfield *) ptr, NULL));
			break;
		case PROP_VALUE_PROVIDER_XML_ID:
			str =  g_value_get_string (value);
			g_return_if_fail (gnome_db_qf_value_restrict_with_field_xml (gnome_db_qf_value, str, NULL));
			break;
		case PROP_HANDLER_PLUGIN:
			str =  g_value_get_string (value);
			if (gnome_db_qf_value->priv->plugin) {
				g_free (gnome_db_qf_value->priv->plugin);
				gnome_db_qf_value->priv->plugin = NULL;
			}
			if (str)
				gnome_db_qf_value->priv->plugin = g_strdup (str);
			break;
		}
	}
}

static void
gnome_db_qf_value_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfValue *gnome_db_qf_value = GNOME_DB_QF_VALUE (object);
	
	if (gnome_db_qf_value->priv) {
		switch (param_id) {
		case PROP_QUERY:
			g_value_set_pointer (value, gnome_db_qf_value->priv->query);
			break;
		case PROP_VALUE_PROVIDER_OBJECT:
			if (gnome_db_qf_value->priv->value_prov_ref)
				g_value_set_pointer (value, gnome_db_ref_base_get_ref_object (gnome_db_qf_value->priv->value_prov_ref));
			else
				g_value_set_pointer (value, NULL);
			break;
		case PROP_VALUE_PROVIDER_XML_ID:
			if (gnome_db_qf_value->priv->value_prov_ref)
                                g_value_set_string (value,
                                                    gnome_db_ref_base_get_ref_name (gnome_db_qf_value->priv->value_prov_ref,
                                                                              NULL, NULL));
                        else
                                g_value_set_string (value, NULL);
			break;
		case PROP_HANDLER_PLUGIN:
			g_value_set_string (value, gnome_db_qf_value->priv->plugin);
			break;
		}	
	}
}

static GObject *
gnome_db_qf_value_copy (GnomeDbQfield *orig)
{
	GnomeDbQfValue *qf, *nqf;
	GObject *obj;
	g_assert (IS_GNOME_DB_QF_VALUE (orig));
	qf = GNOME_DB_QF_VALUE (orig);

	obj = gnome_db_qf_value_new (qf->priv->query, qf->priv->srv_type);
	nqf = GNOME_DB_QF_VALUE (obj);
	if (qf->priv->value)
		nqf->priv->value = gda_value_copy (qf->priv->value);
	if (qf->priv->default_value)
		nqf->priv->default_value = gda_value_copy (qf->priv->default_value);
	nqf->priv->is_parameter = qf->priv->is_parameter;
	nqf->priv->is_null_allowed = qf->priv->is_null_allowed;

	if (qf->priv->value_prov_ref) {
		GnomeDbBase *ref = gnome_db_ref_base_get_ref_object (qf->priv->value_prov_ref);
		if (ref) {
			g_assert (IS_GNOME_DB_QFIELD (ref));
			gnome_db_qf_value_restrict_with_field (nqf, GNOME_DB_QFIELD (ref), NULL);
		}
		else
			gnome_db_qf_value_restrict_with_field_xml (nqf, 
							 gnome_db_ref_base_get_ref_name (qf->priv->value_prov_ref, NULL, NULL),
							 NULL);
	}

	if (gnome_db_base_get_name (GNOME_DB_BASE (orig)))
		gnome_db_base_set_name (GNOME_DB_BASE (obj), gnome_db_base_get_name (GNOME_DB_BASE (orig)));

	if (gnome_db_base_get_description (GNOME_DB_BASE (orig)))
		gnome_db_base_set_description (GNOME_DB_BASE (obj), gnome_db_base_get_description (GNOME_DB_BASE (orig)));

	if (qf->priv->plugin)
		nqf->priv->plugin = g_strdup (qf->priv->plugin);


	return obj;
}

static gboolean
gnome_db_qf_value_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	gboolean retval;
	GnomeDbQfValue *qf1, *qf2;
	GdaValue *val1, *val2;
	GdaValueType t1 = GDA_VALUE_TYPE_NULL, t2 = GDA_VALUE_TYPE_NULL;

	/* it is here assumed that qfield1 and qfield2 are of the same type and refer to the same
	   query */
	g_assert (IS_GNOME_DB_QF_VALUE (qfield1));
	g_assert (IS_GNOME_DB_QF_VALUE (qfield2));
	qf1 = GNOME_DB_QF_VALUE (qfield1);
	qf2 = GNOME_DB_QF_VALUE (qfield2);

	/* comparing values */
	val1 = qf1->priv->value;
	val2 = qf2->priv->value;
	if (val1)
		t1 = gda_value_get_type (val1);
	if (val2)
		t2 = gda_value_get_type (val2);

	retval = qf1->priv->srv_type == qf2->priv->srv_type ? TRUE : FALSE;

	if (retval) 
		retval = (t1 == t2) ? TRUE : FALSE;

	if (retval && (t1 != GDA_VALUE_TYPE_NULL)) 
		retval = gda_value_compare (val1, val2) ? FALSE : TRUE;

	return retval;
}

/**
 * gnome_db_qf_value_set_value
 * @field: a #GnomeDbQfValue object
 * @val: the value to be set, or %NULL
 *
 * Sets the value of @field, or removes it (if @val is %NULL)
 */
void
gnome_db_qf_value_set_value (GnomeDbQfValue *field, const GdaValue *val)
{
	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	g_return_if_fail (field->priv);	

	if (val)
		g_return_if_fail (gda_value_get_type (val) == field->priv->gda_type);

	if (field->priv->value) {
		gda_value_free (field->priv->value);
		field->priv->value = NULL;
	}

	if (val)
		field->priv->value = gda_value_copy (val);
	gnome_db_base_changed (GNOME_DB_BASE (field));
}

/**
 * gnome_db_qf_value_get_value
 * @field: a #GnomeDbQfValue object
 *
 * Get the value stored by @field. If there is no value, but a default value exists, then the
 * default value is returned.n it's up to the caller to test if there is a default value for @field.
 * The default value can be of a different type than the one expected by @field.
 *
 * Returns: the value or NULL
 */
const GdaValue *
gnome_db_qf_value_get_value (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	return field->priv->value;

	return NULL;
}

/**
 * gnome_db_qf_value_set_default_value
 * @field: a #GnomeDbQfValue object
 * @default_val: the default value to be set, or %NULL
 *
 * Sets the default value of @field, or removes it (if @default_val is %NULL)
 */
void
gnome_db_qf_value_set_default_value (GnomeDbQfValue *field, const GdaValue *default_val)
{
	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	g_return_if_fail (field->priv);
	
	if (field->priv->default_value) {
		gda_value_free (field->priv->default_value);
		field->priv->default_value = NULL;
	}

	if (default_val) 
		field->priv->default_value = gda_value_copy (default_val);
}

/**
 * gnome_db_qf_value_get_default_value
 * @field: a #GnomeDbQfValue object
 *
 * Get the default value stored by @field.
 *
 * Returns: the value or NULL
 */
const GdaValue *
gnome_db_qf_value_get_default_value (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	return field->priv->default_value;
}

/**
 * gnome_db_qf_value_get_value_type
 * @field: a #GnomeDbQfValue object
 *
 * Get the GDA type of value stored within @field
 *
 * Returns: the type
 */
GdaValueType
gnome_db_qf_value_get_value_type (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (field->priv, GDA_VALUE_TYPE_UNKNOWN);

	return field->priv->gda_type;
}

/**
 * gnome_db_qf_value_set_is_parameter
 * @field: a #GnomeDbQfValue object
 * @is_param:
 *
 * Sets wether @field can be considered as a parameter
 */
void
gnome_db_qf_value_set_is_parameter (GnomeDbQfValue *field, gboolean is_param)
{
	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	g_return_if_fail (field->priv);

	field->priv->is_parameter = is_param;
}


/**
 * gnome_db_qf_value_is_parameter
 * @field: a #GnomeDbQfValue object
 *
 * Tells if @field can be considered as a parameter
 *
 * Returns: TRUE if @field can be considered as a parameter
 */
gboolean
gnome_db_qf_value_is_parameter (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);

	return field->priv->is_parameter;
}

static GSList *
gnome_db_qf_value_get_params (GnomeDbQfield *qfield)
{
	GSList *list = NULL;
	GnomeDbQfValue *field = GNOME_DB_QF_VALUE (qfield);
	
	if (field->priv->is_parameter) {
		GnomeDbParameter *param;

		param = GNOME_DB_PARAMETER (gnome_db_parameter_new_with_dest_field (qfield, field->priv->srv_type));
		
		/* parameter's attributes */
		gnome_db_base_set_name (GNOME_DB_BASE (param), gnome_db_base_get_name (GNOME_DB_BASE (field)));
		gnome_db_base_set_description (GNOME_DB_BASE (param), gnome_db_base_get_description (GNOME_DB_BASE (field)));

		gnome_db_parameter_set_value (param, field->priv->value);

		if (field->priv->default_value)
			gnome_db_parameter_set_default_value (param, field->priv->default_value);
		gnome_db_parameter_set_not_null (param, !field->priv->is_null_allowed);
		gnome_db_parameter_set_user_input_required (param, field->priv->is_parameter);
			

		/* specified plugin */
		if (field->priv->plugin)
			g_object_set (G_OBJECT (param), "handler_plugin", field->priv->plugin, NULL);

		/* possible values in a sub query */
		if (field->priv->value_prov_ref) {
			GnomeDbBase *ref;

			ref = gnome_db_ref_base_get_ref_object (field->priv->value_prov_ref);
			if (ref) {
				GnomeDbField *provf;
				GnomeDbEntity *prove;
				GSList *tmplist;

				provf = GNOME_DB_FIELD (ref);
				prove = gnome_db_field_get_entity (provf);

				tmplist = gnome_db_entity_get_parameters (prove);
				if (tmplist) {
					GSList *dlist = tmplist;
					while (dlist) {
						gnome_db_parameter_add_dependency (param, GNOME_DB_PARAMETER (dlist->data));
						dlist = g_slist_next (dlist);
					}
					list = g_slist_concat (list, tmplist);
				}
				gnome_db_parameter_set_source_field (param, GNOME_DB_FIELD (ref), NULL);
			}
		}

		list = g_slist_append (list, param);
	}
	
	return list;
}

/**
 * gnome_db_qf_value_set_not_null
 * @field: a #GnomeDbQfValue object
 * @not_null:
 *
 * Sets if a NULL value is acceptable for @field. If @not_null is TRUE, then @field
 * can't have a NULL value.
 */
void
gnome_db_qf_value_set_not_null (GnomeDbQfValue *field, gboolean not_null)
{
	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	g_return_if_fail (field->priv);

	field->priv->is_null_allowed = !not_null;
}

/**
 * gnome_db_qf_value_get_not_null
 * @field: a #GnomeDbQfValue object
 *
 * Tells if @field can receive a NULL value.
 *
 * Returns: TRUE if @field can't have a NULL value
 */
gboolean
gnome_db_qf_value_get_not_null (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);

	return !field->priv->is_null_allowed;
}


/**
 * gnome_db_qf_value_is_value_null
 * @field: a #GnomeDbQfValue object
 * @context: a #GnomeDbDataSet object
 *
 * Tells if @field represents a NULL value.
 *
 * Returns:
 */
gboolean
gnome_db_qf_value_is_value_null (GnomeDbQfValue *field, GnomeDbDataSet *context)
{
	gboolean value_found;
	const GdaValue *value;

	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);

	value_found = gnome_db_qf_value_render_find_value (field, context, &value, NULL);
	if (!value_found) 
		value = field->priv->value;

	if (!value || gda_value_is_null (value))
		return TRUE;
	else
		return FALSE;
}

/*
 * gnome_db_qf_value_restrict_with_field
 * @field: a #GnomeDbQfValue object
 * @provider: a #GnomeDbQfield object, or NULL
 * @error: location to store error, or %NULL
 *
 * Restrict the @field's value to be in the possible values represented by @provider.
 * @provider must be a field in another entity (a #GnomeDbQuery listed as a parameter source for @field's
 * own query).
 *
 * Returns: TRUE if no error occurred
 */
static gboolean
gnome_db_qf_value_restrict_with_field (GnomeDbQfValue *field, GnomeDbQfield *provider, GError **error)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);

	if (field->priv->value_prov_ref) {
		g_object_unref (G_OBJECT (field->priv->value_prov_ref));
		field->priv->value_prov_ref = NULL;
	}

	if (provider) {
		GnomeDbEntity *entity;
		GnomeDbQuery *query;
		GnomeDbServerDataType *dt;

		/* Specific tests */
		/* FIXME */
		g_return_val_if_fail (IS_GNOME_DB_QFIELD (provider), FALSE);
		
		if (!gnome_db_qfield_is_visible (provider)) {
			g_set_error (error,
				     GNOME_DB_QF_VALUE_ERROR,
				     GNOME_DB_QF_VALUE_PARAM_ERROR,
				     _("A field providing a parameter's value must be visible"));
			return FALSE;
		}

		entity = gnome_db_field_get_entity (GNOME_DB_FIELD (provider));
		g_return_val_if_fail (IS_GNOME_DB_QUERY (entity), FALSE);
		query = GNOME_DB_QUERY (entity);
		
		if (!gnome_db_query_is_select_query (query)) {
			g_set_error (error,
				     GNOME_DB_QF_VALUE_ERROR,
				     GNOME_DB_QF_VALUE_PARAM_ERROR,
				     _("A query providing a parameter must be a selection query"));
			return FALSE;
		}
		
		dt = gnome_db_field_get_data_type (GNOME_DB_FIELD (provider));
		if (dt && (dt != field->priv->srv_type)) {
			g_set_error (error,
				     GNOME_DB_QF_VALUE_ERROR,
				     GNOME_DB_QF_VALUE_PARAM_ERROR,
				     _("Incompatible field type for a parameter's provider (%s / %s)"), 
				     gnome_db_server_data_type_get_sqlname (dt),
				     gnome_db_server_data_type_get_sqlname (field->priv->srv_type));
			return FALSE;
		}
		
		field->priv->value_prov_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (field))));
		gnome_db_ref_base_set_ref_object_type (field->priv->value_prov_ref, GNOME_DB_BASE (provider), GNOME_DB_FIELD_TYPE);
	}
	
	return TRUE;
}

/*
 * gnome_db_qf_value_restrict_with_field_xml
 * @field: a #GnomeDbQfValue object
 * @prov_xml_id: the XML Id of a #GnomeDbDfield object, or NULL
 * @error: location to store error, or %NULL
 *
 * Same as gnome_db_qf_value_restrict_with_field() but using an XML Id instead of a direct pointer
 *
 * Returns: TRUE if no error occurred
 */
static gboolean 
gnome_db_qf_value_restrict_with_field_xml (GnomeDbQfValue *field, const gchar *prov_xml_id, GError **error)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), FALSE);
	g_return_val_if_fail (field->priv, FALSE);

	if (field->priv->value_prov_ref) {
		g_object_unref (G_OBJECT (field->priv->value_prov_ref));
		field->priv->value_prov_ref = NULL;
	}

	if (prov_xml_id) {
		/* No test possible, really */
		field->priv->value_prov_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (field))));
		gnome_db_ref_base_set_ref_name (field->priv->value_prov_ref, GNOME_DB_FIELD_TYPE, REFERENCE_BY_XML_ID,
						prov_xml_id);
	}

	return TRUE;	
}

/**
 * gnome_db_qf_value_set_server_data_type
 * @field: a #GnomeDbQfValue object
 *
 * Set the #GnomeDbserverDataType type of @field
 */
void
gnome_db_qf_value_set_server_data_type (GnomeDbQfValue *field, GnomeDbServerDataType *type)
{
	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	g_return_if_fail (field->priv);
	g_return_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type));

	if (type == field->priv->srv_type)
		return;

	/* get rid of the old data type */
	g_signal_handlers_disconnect_by_func (field->priv->srv_type,
					      G_CALLBACK (nullified_object_cb), field);
	
	/* setting the new data type */
	field->priv->srv_type = type;
	gnome_db_base_connect_nullify (type,
				 G_CALLBACK (nullified_object_cb), field);
	
	field->priv->gda_type = gnome_db_server_data_type_get_gda_type (type);

	/* signal a change */
	gnome_db_base_changed (GNOME_DB_BASE (field));
}

/**
 * gnome_db_qf_value_get_server_data_type
 * @field: a #GnomeDbQfValue object
 *
 * Get the #GnomeDbserverDataType type of @field
 *
 * Returns: the #GnomeDbserverDataType type
 */
GnomeDbServerDataType *
gnome_db_qf_value_get_server_data_type (GnomeDbQfValue *field)
{
	g_return_val_if_fail (field && IS_GNOME_DB_QF_VALUE (field), NULL);
	g_return_val_if_fail (field->priv, NULL);

	return field->priv->srv_type;
}

#ifdef debug
static void
gnome_db_qf_value_dump (GnomeDbQfValue *field, guint offset)
{
	gchar *str;
	gint i;
	GnomeDbDataHandler *dh;

	g_return_if_fail (field && IS_GNOME_DB_QF_VALUE (field));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (field->priv) {
		gchar *val;
		dh = gnome_db_server_data_type_get_handler (field->priv->srv_type);
                g_print ("%s" D_COL_H1 "GnomeDbQfValue" D_COL_NOR " \"%s\" (%p, id=%d) ",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (field)), field, gnome_db_base_get_id (GNOME_DB_BASE (field)));
		if (field->priv->is_parameter) 
			g_print ("is param, ");

		if (gnome_db_qf_value_is_active (GNOME_DB_REFERER (field)))
			g_print ("Active, ");
		else
			g_print (D_COL_ERR "Inactive" D_COL_NOR ", ");

		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
			g_print ("Visible, ");
		if (field->priv->value) {
			val = gnome_db_data_handler_get_sql_from_value (dh, field->priv->value);
			g_print ("Value: %s ", val);
			g_free (val);
		}

		if (field->priv->default_value) {
			GnomeDbServer *srv = gnome_db_dict_get_server (GNOME_DB_DICT (gnome_db_base_get_dict (GNOME_DB_BASE (field->priv->srv_type))));
			GnomeDbDataHandler *dhd = gnome_db_server_get_handler_by_gda (srv, gda_value_get_type (field->priv->default_value));
			val = gnome_db_data_handler_get_sql_from_value (dhd, field->priv->default_value);
			g_print ("Default: %s ", val);
			g_free (val);
		}
		g_print ("\n");

		if (field->priv->value_prov_ref) {
			g_print ("%sPossible values:\n", str);
			gnome_db_base_dump (GNOME_DB_BASE (field->priv->value_prov_ref), offset+5);
		}
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, field);
}
#endif


/* 
 * GnomeDbField interface implementation
 */
static GnomeDbEntity *
gnome_db_qf_value_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_QF_VALUE (iface)->priv->query);
}

static GnomeDbServerDataType *
gnome_db_qf_value_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);

	return GNOME_DB_QF_VALUE (iface)->priv->srv_type;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_qf_value_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *q_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);

	q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_QF_VALUE (iface)->priv->query));
	xml_id = g_strdup_printf ("%s:QF%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
	g_free (q_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_qf_value_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbQfValue *field;
	gchar *str;
	GnomeDbDataHandler *dh;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);

	field = GNOME_DB_QF_VALUE (iface);

	node = xmlNewNode (NULL, "GNOME_DB_QF");
	
	str = gnome_db_qf_value_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);

	xmlSetProp (node, "type", "VAL");
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (field)));
	if (gnome_db_base_get_description (GNOME_DB_BASE (field)) && *gnome_db_base_get_description (GNOME_DB_BASE (field)))
		xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (field)));
	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_visible",  "f");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (field)))
		xmlSetProp (node, "is_internal", "t");

	xmlSetProp (node, "is_param", field->priv->is_parameter ? "t" : "f");
	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (field->priv->srv_type));
	xmlSetProp (node, "srv_type",  str);
	g_free (str);

	dh = gnome_db_server_data_type_get_handler (field->priv->srv_type);
	if (field->priv->value) {
		str = gnome_db_data_handler_get_str_from_value (dh, field->priv->value);
		xmlSetProp (node, "value", str);
		g_free (str);
	}
	if (field->priv->default_value) {
		GnomeDbServer *srv = gnome_db_dict_get_server (GNOME_DB_DICT (gnome_db_base_get_dict (GNOME_DB_BASE (field->priv->srv_type))));
		GnomeDbDataHandler *dhd = gnome_db_server_get_handler_by_gda (srv, gda_value_get_type (field->priv->default_value));
		GdaValueType vtype;
		
		str = gnome_db_data_handler_get_str_from_value (dhd, field->priv->default_value);
		xmlSetProp (node, "default", str);
		g_free (str);
		vtype = gda_value_get_type (field->priv->default_value);
		xmlSetProp (node, "default_gda_type", gda_type_to_string (vtype));
	}

	xmlSetProp (node, "null_ok", field->priv->is_null_allowed ? "t" : "f");
	if (field->priv->value_prov_ref)
		xmlSetProp (node, "value_prov", 
			    gnome_db_ref_base_get_ref_name (field->priv->value_prov_ref, NULL, NULL));

	str = gnome_db_qfield_get_alias (GNOME_DB_QFIELD (field));
	if (str && *str) 
		xmlSetProp (node, "alias", str);
	
	if (field->priv->plugin)
		xmlSetProp (node, "plugin", field->priv->plugin);

	return node;
}

static gboolean
gnome_db_qf_value_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQfValue *field;
	gchar *prop;
	GnomeDbDataHandler *dh = NULL;
	gboolean err = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	field = GNOME_DB_QF_VALUE (iface);
	if (strcmp (node->name, "GNOME_DB_QF")) {
		g_set_error (error,
			     GNOME_DB_QF_VALUE_ERROR,
			     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QF>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		if (strcmp (prop, "VAL")) {
			g_set_error (error,
				     GNOME_DB_QF_VALUE_ERROR,
				     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
				     _("Wrong type of field in <GNOME_DB_QF>"));
			return FALSE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "id");
	if (prop) {
		gchar *ptr, *tok;
		ptr = strtok_r (prop, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (strlen (ptr) < 3) {
			g_set_error (error,
				     GNOME_DB_QF_VALUE_ERROR,
				     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
				     _("Wrong 'id' attribute in <GNOME_DB_QF>"));
			return FALSE;
		}
		gnome_db_base_set_id (GNOME_DB_BASE (field), atoi (ptr+2));
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (field), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_visible");
	if (prop) {
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_internal");
	if (prop) {
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (field), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "srv_type");
	if (prop) {
		field->priv->srv_type = gnome_db_server_get_data_type_by_xml_id (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (field))), prop);
		if (field->priv->srv_type) {
			dh = gnome_db_server_data_type_get_handler (field->priv->srv_type);
			field->priv->gda_type = gnome_db_server_data_type_get_gda_type (field->priv->srv_type);
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "value");
	if (prop) {
		if (dh) {
			field->priv->value = gnome_db_data_handler_get_value_from_str (dh, prop, field->priv->gda_type);
			if (!field->priv->value) {
				g_set_error (error,
					     GNOME_DB_QF_VALUE_ERROR,
					     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
					     _("Can't interpret '%s' as a value"), prop);
				g_free (prop);
				return FALSE;
			}
		}

		g_free (prop);
	}

	prop = xmlGetProp (node, "default");
	if (prop) {
		gchar *str2;

		str2 = xmlGetProp (node, "default_gda_type");
		if (str2) {
			if (field->priv->srv_type) {
				GnomeDbServer *srv = gnome_db_dict_get_server (GNOME_DB_DICT (gnome_db_base_get_dict (GNOME_DB_BASE (field->priv->srv_type))));
				GnomeDbDataHandler *dh2;
				GdaValueType vtype;
				GdaValue *value;
				
				vtype = gda_type_from_string (str2);			
				dh2 = gnome_db_server_get_handler_by_gda (srv, vtype);
				value = gnome_db_data_handler_get_value_from_str (dh2, prop, vtype);
				field->priv->default_value = value;
			}

			g_free (str2);
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_param");
	if (prop) {
		field->priv->is_parameter = (*prop == 't') ? TRUE : FALSE;
		g_free (prop);
	}

	prop = xmlGetProp (node, "null_ok");
	if (prop) {
		field->priv->is_null_allowed = (*prop == 't') ? TRUE : FALSE;
		g_free (prop);
	}

	prop = xmlGetProp (node, "plugin");
	if (prop) 
		field->priv->plugin = prop;

	prop = xmlGetProp (node, "value_prov");
	if (prop) {
		if (!gnome_db_qf_value_restrict_with_field_xml (field, prop, error))
			err = TRUE;
		g_free (prop);
	}

	prop = xmlGetProp (node, "alias");
	if (prop) {
		gnome_db_qfield_set_alias (GNOME_DB_QFIELD (field), prop);
		g_free (prop);
	}

	if (!dh) {
		err = TRUE;
		g_set_error (error,
			     GNOME_DB_QF_VALUE_ERROR,
			     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
			     _("Missing required gda_type for <GNOME_DB_QF>"));
	}

	if (!err) {
		if (!field->priv->is_parameter) {
			if (!field->priv->value) {
				err = TRUE;
				g_set_error (error,
					     GNOME_DB_QF_VALUE_ERROR,
					     GNOME_DB_QF_VALUE_XML_LOAD_ERROR,
					     _("Value field '%s' does not have a value!"),
					     gnome_db_base_get_name (GNOME_DB_BASE (field)));
			}
		}
	}

	return !err;
}


/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_qf_value_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	GdaXqlItem *node = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return node;
}

static gboolean
gnome_db_qf_value_render_find_value (GnomeDbQfValue *field, GnomeDbDataSet *context, 
			       const GdaValue **value_found, GnomeDbParameter **param_source)
{
	const GdaValue *cvalue = NULL;
	gboolean found = FALSE;

	if (param_source)
		*param_source = NULL;
	if (value_found)
		*value_found = NULL;

	/* looking for a value into the context first */
	if (context) {
		GSList *list = context->parameters;
		GSList *for_fields;
		
		while (list && !found) {
			for_fields = gnome_db_parameter_get_dest_fields (GNOME_DB_PARAMETER (list->data));
			if (g_slist_find (for_fields, field)) {
				if (param_source)
					*param_source = GNOME_DB_PARAMETER (list->data);
				found = TRUE;
				cvalue = gnome_db_parameter_get_value (GNOME_DB_PARAMETER (list->data));
			}
			list = g_slist_next (list);
		}
	}
	
	/* using the field's value, if available */
	if (!cvalue && field->priv->value) {
		found = TRUE;
		cvalue = field->priv->value;
	}

	if (value_found)
		*value_found = cvalue;

	return found;
}

static gchar *
gnome_db_qf_value_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;
	GnomeDbQfValue *field;
	const GdaValue *value = NULL;
	gboolean sqlext = options & GNOME_DB_RENDERER_EXTRA_VAL_ATTRS;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);
	field = GNOME_DB_QF_VALUE (iface);

	if (field->priv->is_parameter) {
		gboolean value_found;
		GnomeDbParameter *param_source = NULL;

		value_found = gnome_db_qf_value_render_find_value (field, context, &value, &param_source);
	
		/* actual rendering */
		if (value_found) {
			if (param_source && ! gnome_db_parameter_is_valid (param_source)) {
				gchar *str, *str2;
				
				str2 = value ? gda_value_stringify (value) : g_strdup ("NULL");
				str = g_strdup_printf (_("Invalid parameter '%s' (value: %s)"),
						       gnome_db_base_get_name (GNOME_DB_BASE (param_source)), str2);
				g_free (str2);
				
				g_set_error (error,
					     GNOME_DB_QF_VALUE_ERROR,
					     GNOME_DB_QF_VALUE_RENDER_ERROR,
					     str);
				g_free (str);
				
				/*g_print ("Param %p (%s) is invalid!\n", param_source, 
				  gnome_db_base_get_name (GNOME_DB_BASE (param_source)));*/
				return NULL;
			}
		
			str = NULL;
			if (param_source) {
				gboolean use_default;
				g_object_get (G_OBJECT (param_source), "use_default_value", &use_default, NULL);
				if (use_default) {
					if (options & GNOME_DB_RENDERER_ERROR_IF_DEFAULT) {
						g_set_error (error,
							     GNOME_DB_QF_VALUE_ERROR,
							     GNOME_DB_QF_VALUE_DEFAULT_PARAM_ERROR,
							     "Default value requested");
						return NULL;
					}
					else
						str = g_strdup ("DEFAULT");
				}
			}
			if (!str) {
				if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL)) {
					GnomeDbDataHandler *dh;
					
					dh = gnome_db_server_data_type_get_handler (field->priv->srv_type);
					str = gnome_db_data_handler_get_sql_from_value (dh, value);
				}
				else
					str = g_strdup ("NULL");	
			}
		}
		else {
			if (field->priv->is_null_allowed)
				str = g_strdup ("##");
			else {
				if (context) {
					g_set_error (error,
						     GNOME_DB_QF_VALUE_ERROR,
						     GNOME_DB_QF_VALUE_RENDER_ERROR,
						     _("No specified value"));
				}
				else
					str = g_strdup ("##");
			}
		}
		
	}
	else {
		value = field->priv->value;
		if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL)) {
			GnomeDbDataHandler *dh;
			
			dh = gnome_db_server_data_type_get_handler (field->priv->srv_type);
			str = gnome_db_data_handler_get_sql_from_value (dh, value);
		}
		else
			str = g_strdup ("NULL");	
	}

	if (sqlext && field->priv->is_parameter) {
		GString *extra = g_string_new ("");
		const gchar *tmpstr;
		gchar *str2;
		gboolean isfirst = TRUE;

		/* add extra information about the value, as an extension of SQL */
		if (! field->priv->is_parameter) {
			g_string_append (extra, ":isparam=\"FALSE\"");
			isfirst = FALSE;
		}

		if (field->priv->srv_type) {
			if (isfirst)
				isfirst = FALSE;
			else
				g_string_append (extra, " ");
			g_string_append_printf (extra, ":type=\"%s\"", 
						gnome_db_base_get_name (GNOME_DB_BASE (field->priv->srv_type)));
		}

		tmpstr = gnome_db_base_get_name (GNOME_DB_BASE (field));
		if (tmpstr && *tmpstr) {
			if (isfirst)
				isfirst = FALSE;
			else
				g_string_append (extra, " ");

			g_string_append_printf (extra, ":name=\"%s\"", tmpstr);
		}


		tmpstr = gnome_db_base_get_description (GNOME_DB_BASE (field));
		if (tmpstr && *tmpstr) {
			if (isfirst)
				isfirst = FALSE;
			else
				g_string_append (extra, " ");
			g_string_append_printf (extra, ":descr=\"%s\"", tmpstr);
		}

		if (field->priv->is_null_allowed) {
			if (isfirst)
				isfirst = FALSE;
			else
				g_string_append (extra, " ");
			g_string_append (extra, ":nullok=\"TRUE\"");
		}

		str2 = g_strdup_printf ("%s [%s]", str, extra->str);
		g_free (str);
		str = str2;

		g_string_free (extra, TRUE);
	}

	return str;
}

static gchar *
gnome_db_qf_value_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);
	
	str = gnome_db_qf_value_render_as_sql (iface, context, 0, NULL);
	if (!str)
		str = g_strdup ("???");
	return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_qf_value_activate (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, FALSE);

	if (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref)
		return gnome_db_ref_base_activate (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref);
	else
		return TRUE;
}

static void
gnome_db_qf_value_deactivate (GnomeDbReferer *iface)
{
	g_return_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface));
	g_return_if_fail (GNOME_DB_QF_VALUE (iface)->priv);

	if (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref)
		gnome_db_ref_base_deactivate (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref);
}

static gboolean
gnome_db_qf_value_is_active (GnomeDbReferer *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, FALSE);

	if (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref)
		return gnome_db_ref_base_is_active (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref);
	else
		return TRUE;
}

static GSList *
gnome_db_qf_value_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_VALUE (iface)->priv, NULL);

	if (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref) {
		GnomeDbBase *base = gnome_db_ref_base_get_ref_object (GNOME_DB_QF_VALUE (iface)->priv->value_prov_ref);
		if (base)
			list = g_slist_append (list, base);
	}

        return list;
}

static void
gnome_db_qf_value_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbQfValue *field;

        g_return_if_fail (iface && IS_GNOME_DB_QF_VALUE (iface));
        g_return_if_fail (GNOME_DB_QF_VALUE (iface)->priv);

        field = GNOME_DB_QF_VALUE (iface);	
        if (field->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, field->priv->query);
                if (query) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (field->priv->query),
                                                              G_CALLBACK (nullified_object_cb), field);
                        field->priv->query = query;
			gnome_db_base_connect_nullify (query,
						 G_CALLBACK (nullified_object_cb), field);
                }
        }

	if (field->priv->value_prov_ref)
		gnome_db_ref_base_replace_ref_object (field->priv->value_prov_ref, replacements);
}
