/* gnome-db-sql-parser.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <glib.h>
#include <strings.h>
#include <string.h>

#include "gnome-db-sql-parser.h"
#include "gnome-db-sql-tree.h"

#ifndef _
#define _(x) (x)
#endif

extern char *gnome_db_sqltext;
extern void gnome_db_sql_switch_to_buffer (void *buffer);
extern void *gnome_db_sql_scan_string (const char *string);

void gnome_db_sqlerror (char *error);

GnomeDbSqlStatement *sql_result;
GError **sql_error;

int gnome_db_sqlparse (void);

static void sql_destroy_expr (GnomeDbSqlExpr *expr);
static void sql_destroy_param_spec (GnomeDbParamSpec *pspec);

/**
 * sqlerror:
 * 
 * Internal function for displaying error messages used by the lexer parser.
 */
void
gnome_db_sqlerror (char *string)
{
	if (sql_error) {
		if (!strcmp (string, "parse error"))
			g_set_error (sql_error, 0, 0, _("Parse error near `%s'"), gnome_db_sqltext);
		if (!strcmp (string, "syntax error"))
			g_set_error (sql_error, 0, 0, _("Syntax error near `%s'"), gnome_db_sqltext);
	}
	else
		fprintf (stderr, "SQL Parser error: %s near `%s'\n", string, gnome_db_sqltext);
}



static void
sql_destroy_param_spec (GnomeDbParamSpec *pspec)
{
	if (!pspec)
		return;

	g_free (pspec->content);
	g_free (pspec);
}

static void
sql_destroy_expr (GnomeDbSqlExpr *expr)
{
	if (expr->sql_text)
		g_free (expr->sql_text);
	if (expr->pspec_list) {
		GList *pspecs = expr->pspec_list;
		while (pspecs) {
			sql_destroy_param_spec ((GnomeDbParamSpec *)(pspecs->data));
			pspecs = g_list_next (pspecs);
		}
		g_list_free (expr->pspec_list);
	}

	g_free (expr);
}

static void
sql_destroy_statement (GnomeDbSqlStatement *statement)
{
	GList *list;
	g_free (statement->full_query);
	list = statement->expr_list;
	while (list) {
		sql_destroy_expr ((GnomeDbSqlExpr *)(list->data));
		list = g_list_next (list);
	}
	g_list_free (statement->expr_list);
	g_list_free (statement->params_specs);
	g_free (statement);
}


/**
 * sql_destroy:
 * @statement: Sql statement generated by sql_parse()
 * 
 * Free up a GnomeDbSqlStatement generated by sql_parse().
 */
int
gnome_db_sql_destroy (GnomeDbSqlStatement *statement)
{
	if (!statement)
		return 0;

	sql_destroy_statement (statement);
	return 0;
}

/**
 * gnome_db_sql_parse_with_error:
 * @sqlquery: A SQL query string. ie SELECT * FROM FOO
 * @error: a place where to store an error, or %NULL
 * 
 * Generate in memory a structure of the @sqlquery in an easy
 * to view way.  You can also modify the returned structure and
 * regenerate the sql query using sql_stringify().  The structure
 * contains information on what type of sql statement it is, what
 * tables its getting from, what fields are selected, the where clause,
 * joins etc.
 * 
 * Returns: A generated GnomeDbSqlStatement or %NULL on error.
 */
GnomeDbSqlStatement *
gnome_db_sql_parse_with_error (const char *sqlquery, GError ** error)
{
	if (!sqlquery) {
		if (error)
			g_set_error (error, 0, 0, _("Empty query to parse"));
		return NULL;
	}
	
	sql_error = error;
	gnome_db_sql_switch_to_buffer (gnome_db_sql_scan_string (g_strdup (sqlquery)));
	if (! gnome_db_sqlparse ()) {
		sql_result->full_query = g_strdup (sqlquery);
		return sql_result;
	}

	return NULL;
}

/**
 * gnome_db_sql_parse:
 * @sqlquery: A SQL query string. ie SELECT * FROM FOO
 *
 * Generate in memory a structure of the @sqlquery in an easy
 * to view way.  You can also modify the returned structure and
 * regenerate the sql query using sql_stringify().  The structure
 * contains information on what type of sql statement it is, what
 * tables its getting from, what fields are selected, the where clause,
 * joins etc.
 *
 * Returns: A generated GnomeDbSqlStatement or %NULL on error.
 */
GnomeDbSqlStatement *
gnome_db_sql_parse (const char *sqlquery)
{
	return gnome_db_sql_parse_with_error (sqlquery, NULL);
}

static void sql_display_expr (GnomeDbSqlExpr *expr);
static void sql_display_pspec_list (GList *pspecs);
void
gnome_db_sql_display (GnomeDbSqlStatement *statement)
{
	GList *list;

	if (!statement)
		return;

	switch (statement->type) {
	case GNOME_DB_SQL_SELECT:
		g_print ("Select statement:\n");
		break;
	case GNOME_DB_SQL_INSERT:
		g_print ("Insert statement:\n");
		break;
	case GNOME_DB_SQL_UPDATE:
		g_print ("Update statement:\n");
		break;
	case GNOME_DB_SQL_DELETE:
		g_print ("Delete statement:\n");
		break;
	default:
		g_print ("Unknown statement:\n");
		break;
	}

	g_print ("Original SQL: %s\n", statement->full_query);
	g_print ("Parsed SQL:\n");
	list = statement->expr_list;
	while (list) {
		sql_display_expr ((GnomeDbSqlExpr *)(list->data));
		list = g_list_next (list);
	}
	g_print ("Parsed parameters:\n");
	list = statement->params_specs;
	while (list) {
		sql_display_pspec_list ((GList *)(list->data));
		list = g_list_next (list);
	}
}

static void
sql_display_expr (GnomeDbSqlExpr *expr)
{
	if (expr->sql_text)
		g_print ("\t%s\n", expr->sql_text);
	if (expr->pspec_list)
		sql_display_pspec_list (expr->pspec_list);
}

static void
sql_display_pspec_list (GList *pspecs)
{
	GList *list;

	list = pspecs;
	g_print ("\t## [");
	while (list) {
		GnomeDbParamSpec *pspec = (GnomeDbParamSpec *)(list->data);

		if (list != pspecs)
			g_print (" ");

		switch (pspec->type) {
		case GNOME_DB_PARAM_NAME:
			g_print (":name=\"%s\"", pspec->content);
			break;
		case GNOME_DB_PARAM_DESCR:
			g_print (":descr=\"%s\"", pspec->content);
			break;
		case GNOME_DB_PARAM_TYPE:
			g_print (":type=\"%s\"", pspec->content);
			break;
		case GNOME_DB_PARAM_ISPARAM:
			g_print (":isparam=\"%s\"", pspec->content);
			break;
		case GNOME_DB_PARAM_NULLOK:
			g_print (":nullok=\"%s\"", pspec->content);
			break;
		default:
			g_print (":?? =\"%s\"", pspec->content);
			break;
		}
		list = g_list_next (list);
	}
	g_print ("]\n");
}


static GnomeDbSqlExpr *
copy_expr (GnomeDbSqlExpr *orig)
{
	GList *list;
	GnomeDbSqlExpr *expr = g_new0 (GnomeDbSqlExpr, 1);

	list = orig->pspec_list;
	while (list) {
		GnomeDbParamSpec *pspec = g_new0 (GnomeDbParamSpec, 1);
		pspec->type = ((GnomeDbParamSpec *)(list->data))->type;
		pspec->content = g_strdup (((GnomeDbParamSpec *)(list->data))->content);
		expr->pspec_list = g_list_prepend (expr->pspec_list, pspec);
		list = g_list_next (list);
	}
	if (expr->pspec_list)
		expr->pspec_list = g_list_reverse (expr->pspec_list);
	if (orig->sql_text)
		expr->sql_text = g_strdup (orig->sql_text);

	return expr;
}


/**
 * gnome_db_sql_parse_copy_statement
 *
 * makes a copy of @statement
 */
GnomeDbSqlStatement *
gnome_db_sql_parse_copy_statement (GnomeDbSqlStatement *statement)
{
	GnomeDbSqlStatement *stm;
	GList *list;

	if (!statement)
		return NULL;

	stm = g_new0 (GnomeDbSqlStatement, 1);
	stm->type = statement->type;
	stm->full_query = g_strdup (statement->full_query);
	list = statement->expr_list;
	while (list) {
		GnomeDbSqlExpr *tmp = copy_expr ((GnomeDbSqlExpr *)(list->data));
		stm->expr_list = g_list_prepend (stm->expr_list, tmp);
		
		list = g_list_next (list);
	}
	stm->expr_list = g_list_reverse (stm->expr_list);

	/* make a list of the para specs */
	list = stm->expr_list;
	while (list) {
		if (((GnomeDbSqlExpr *)(list->data))->pspec_list)
			stm->params_specs = g_list_append (stm->params_specs, ((GnomeDbSqlExpr *)(list->data))->pspec_list);
		list = g_list_next (list);
	}

	return stm;
}
