/* gnome-db-wrapper-field.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-wrapper-field.h"
#include "gnome-db-model-wrapper.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-server.h"
#include <libgda/libgda.h>

/* 
 * Main static functions 
 */
static void gnome_db_wrapper_field_class_init (GnomeDbWrapperFieldClass *class);
static void gnome_db_wrapper_field_init (GnomeDbWrapperField *wfield);
static void gnome_db_wrapper_field_dispose (GObject *object);
static void gnome_db_wrapper_field_finalize (GObject *object);

/* Field interface */
static void                   gnome_db_wrapper_field_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_wrapper_field_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_wrapper_field_get_data_type   (GnomeDbField *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

struct _GnomeDbWrapperFieldPrivate
{
	GnomeDbEntity          *entity;
	gint                    col_no;
	GnomeDbField           *field;
};

/* module error */
GQuark gnome_db_wrapper_field_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_wrapper_field_error");
	return quark;
}


guint
gnome_db_wrapper_field_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbWrapperFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_wrapper_field_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbWrapperField),
			0,
			(GInstanceInitFunc) gnome_db_wrapper_field_init
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_wrapper_field_field_init,
			NULL,
			NULL
		};

		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbWrapperField", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_FIELD_TYPE, &field_info);
	}
	return type;
}

static void
gnome_db_wrapper_field_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_wrapper_field_get_entity;
	iface->get_data_type = gnome_db_wrapper_field_get_data_type;
}

static void
gnome_db_wrapper_field_class_init (GnomeDbWrapperFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_wrapper_field_dispose;
	object_class->finalize = gnome_db_wrapper_field_finalize;
}

static void
gnome_db_wrapper_field_init (GnomeDbWrapperField *wfield)
{
	wfield->priv = g_new0 (GnomeDbWrapperFieldPrivate, 1);
	wfield->priv->col_no = -1;
}

static void nullified_object_cb (GObject *obj, GnomeDbWrapperField *wfield);

/**
 * gnome_db_wrapper_field_new_in_model
 * @model: a #GnomeDbModelWrapper object
 * @col: the column number of the #GdaDataModel @model holds.
 *
 * Creates a new #GnomeDbWrapperField object. The new object will implement the #GnomeDbField
 * interface for the @col column of the @model model.
 * 
 * Returns: the newly created object
 */
GObject *
gnome_db_wrapper_field_new_in_model (GnomeDbModelWrapper *model, gint col)
{
	GObject *obj;
	GnomeDbWrapperField *wfield;
	GdaDataModelColumnAttributes *col_attrs;

	g_return_val_if_fail (model && IS_GNOME_DB_MODEL_WRAPPER (model), NULL);
	g_return_val_if_fail (col >= 0, NULL);

	obj = g_object_new (GNOME_DB_WRAPPER_FIELD_TYPE, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (model)), NULL);
	wfield = GNOME_DB_WRAPPER_FIELD (obj);

	wfield->priv->entity = GNOME_DB_ENTITY (model);
	wfield->priv->col_no = col;

	col_attrs = gda_data_model_describe_column (GDA_DATA_MODEL (wfield->priv->entity), wfield->priv->col_no);
	gnome_db_base_set_name (GNOME_DB_BASE (wfield), col_attrs->name);

	gnome_db_base_connect_nullify (model,
				       G_CALLBACK (nullified_object_cb), wfield);

	return obj;
}

/**
 * gnome_db_wrapper_field_new_in_entity
 * @entity: a #GnomeDbEntity object
 * @field: a #GnomeDbField object
 *
 * Creates a new #GnomeDbWrapperField object. The new object is a wrapper of the @field field
 * which will say it belongs to the @entity #GnomeDbEntity.
 * 
 * Returns: the newly created object
 */
GObject *
gnome_db_wrapper_field_new_in_entity (GnomeDbEntity *entity, GnomeDbField *field)
{
	GObject *obj;
	GnomeDbWrapperField *wfield;

	g_return_val_if_fail (entity && IS_GNOME_DB_ENTITY (entity), NULL);
	g_return_val_if_fail (field && IS_GNOME_DB_FIELD (field), NULL);

	obj = g_object_new (GNOME_DB_WRAPPER_FIELD_TYPE, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (entity)), NULL);
	wfield = GNOME_DB_WRAPPER_FIELD (obj);

	wfield->priv->entity = entity;
	wfield->priv->field = field;
	gnome_db_base_set_name (GNOME_DB_BASE (wfield), gnome_db_base_get_name (GNOME_DB_BASE (field)));

	gnome_db_base_connect_nullify (entity,
				       G_CALLBACK (nullified_object_cb), wfield);
	gnome_db_base_connect_nullify (field,
				       G_CALLBACK (nullified_object_cb), wfield);

	return obj;
}

static void
nullified_object_cb (GObject *obj, GnomeDbWrapperField *wfield)
{
	gnome_db_base_nullify (GNOME_DB_BASE (wfield));
}

static void
gnome_db_wrapper_field_dispose (GObject *object)
{
	GnomeDbWrapperField *wfield;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_WRAPPER_FIELD (object));

	wfield = GNOME_DB_WRAPPER_FIELD (object);
	if (wfield->priv) {
		if (wfield->priv->entity) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (wfield->priv->entity),
                                                              G_CALLBACK (nullified_object_cb), wfield);
			wfield->priv->entity = NULL;
		}
		if (wfield->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (wfield->priv->field),
                                                              G_CALLBACK (nullified_object_cb), wfield);
			wfield->priv->field = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_wrapper_field_finalize (GObject   * object)
{
	GnomeDbWrapperField *wfield;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_WRAPPER_FIELD (object));

	wfield = GNOME_DB_WRAPPER_FIELD (object);
	if (wfield->priv) {
		g_free (wfield->priv);
		wfield->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

/*
 * GnomeDbField interface
 */
static GnomeDbEntity *
gnome_db_wrapper_field_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (IS_GNOME_DB_WRAPPER_FIELD (iface), NULL);

	return GNOME_DB_WRAPPER_FIELD (iface)->priv->entity;
}

static GnomeDbServerDataType *
gnome_db_wrapper_field_get_data_type (GnomeDbField *iface)
{
	GdaDataModelColumnAttributes *col_attrs;
	GnomeDbWrapperField *wfield;

	g_return_val_if_fail (IS_GNOME_DB_WRAPPER_FIELD (iface), NULL);
	wfield = GNOME_DB_WRAPPER_FIELD (iface);
	
	if (wfield->priv->field) 
		return gnome_db_field_get_data_type (wfield->priv->field);
	else {
		gchar *type = NULL;
		GnomeDbServerDataType *retval = NULL;
		GnomeDbServer *srv;

		srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (iface)));
		col_attrs = gda_data_model_describe_column (GDA_DATA_MODEL (wfield->priv->entity), 
							    wfield->priv->col_no);
		
		/* FIXME: find the GnomeDbDataServer data type for col_attrs->gda_type, or modify col_attrs to also
		   return the actual data type name of the column */
		if (col_attrs->dbms_type)
			type = col_attrs->dbms_type;
		else {
			GnomeDbServerInfo *info;
			gint i;

			info = gnome_db_server_get_server_info (srv);
			if (info) {
				for (i = 0; (i < info->value_nb_tests_types) && !type; i ++) {
					if (info->value_test_gda_types [i] == col_attrs->gda_type)
						type = info->value_test_data_types [i];
				}
			}
		}

		if (type) {
			retval = gnome_db_server_get_data_type_by_name (srv, type);
			if (!retval) {
				/* create a GnomeDbServerDataType for that 'type' */
				retval = GNOME_DB_SERVER_DATA_TYPE (gnome_db_server_data_type_new (srv));
				gnome_db_server_data_type_set_sqlname (retval, type);
				gnome_db_server_data_type_set_gda_type (retval, col_attrs->gda_type);
				
				gnome_db_server_add_data_type (srv, retval);
				g_object_unref (retval);
			}
		}
		else
			g_warning ("Could not determine the DBMS data type for Gda type %d", col_attrs->gda_type);
		
		return retval;
	}
}
