/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkpaned.h>
#include <libgnomedb/gnome-db-browser.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>
#include <bonobo/bonobo-i18n.h>
#include "gnome-db-browser-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbBrowserPrivate {
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *paned;
	GtkWidget *object_notebook;
	GtkWidget *detail_notebook;

	/* objects management */
	struct {
		GdaConnectionSchema schema;
		GtkWidget *object_list;
		GtkWidget *object_detail;
		gboolean data_retrieved;
		const gchar *title;

		/* function pointers for each operation */
		GtkWidget * (* new_function) (void);
		void        (* show_function) (GtkWidget *widget,
					       GdaConnection *cnc,
					       const gchar *str);
		void        (* clear_function) (GtkWidget *widget);
	} objects_info [7];
};

static void gnome_db_browser_class_init   (GnomeDbBrowserClass *klass);
static void gnome_db_browser_init         (GnomeDbBrowser *browser,
					   GnomeDbBrowserClass *klass);
static void gnome_db_browser_set_property (GObject *object,
					   guint paramid,
					   const GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_get_property (GObject *object,
					   guint param_id,
					   GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_CONNECTION
};

enum {
	PROGRESS_MESSAGE,
	LAST_SIGNAL
};

static gint browser_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
notebook_page_changed_cb (GtkNotebook *notebook,
			  GtkNotebookPage *page,
			  guint page_num,
			  gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	gtk_notebook_set_current_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				       page_num);
}

static void
object_selected_cb (GnomeDbList *list, gint row, gpointer user_data)
{
	GdaDataModel *model;
	const GdaValue *value;
	const gchar *str;
	gint current_page;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	current_page = gtk_notebook_current_page (
		GTK_NOTEBOOK (browser->priv->object_notebook));

	model = gnome_db_list_get_model (
		GNOME_DB_LIST (browser->priv->objects_info[current_page].object_list));
	value = gda_data_model_get_value_at (model, 0, row);
	if (!value)
		return;

	str = gda_value_get_string ((GdaValue *) value);
	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE],
		       0, _("Loading..."));
	if (browser->priv->objects_info[current_page].show_function) {
		browser->priv->objects_info[current_page].show_function (
			browser->priv->objects_info[current_page].object_detail,
			browser->priv->cnc,
			str);
	}

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE], 0, NULL);
}

static void
object_selection_cleared_cb (GnomeDbList *list, gpointer user_data)
{
	gint current_page;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	current_page = gtk_notebook_current_page (
		GTK_NOTEBOOK (browser->priv->object_notebook));
	if (browser->priv->objects_info[current_page].clear_function) {
		browser->priv->objects_info[current_page].clear_function (
			browser->priv->objects_info[current_page].object_detail);
	}
}

/*
 * Private functions
 */

#define POS_DATABASES  0
#define POS_TABLES     1
#define POS_VIEWS      2
#define POS_TYPES      3
#define POS_TRIGGERS   4
#define POS_PROCEDURES 5
#define POS_AGGREGATES 6

static void
initialize_object_widgets (GnomeDbBrowser *browser)
{
	gint i;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* initialize function pointers */
	browser->priv->objects_info[POS_DATABASES].schema = GDA_CONNECTION_SCHEMA_DATABASES;
	browser->priv->objects_info[POS_DATABASES].data_retrieved = FALSE;
	browser->priv->objects_info[POS_DATABASES].title = _("Databases");
	browser->priv->objects_info[POS_DATABASES].new_function = NULL;
	browser->priv->objects_info[POS_DATABASES].show_function = NULL;
	browser->priv->objects_info[POS_DATABASES].clear_function = NULL;

	browser->priv->objects_info[POS_TABLES].schema = GDA_CONNECTION_SCHEMA_TABLES;
	browser->priv->objects_info[POS_TABLES].data_retrieved = FALSE;
	browser->priv->objects_info[POS_TABLES].title = _("Tables");
	browser->priv->objects_info[POS_TABLES].new_function = gnome_db_browser_tables_new;
	browser->priv->objects_info[POS_TABLES].show_function = gnome_db_browser_tables_show;
	browser->priv->objects_info[POS_TABLES].clear_function = gnome_db_browser_tables_clear;

	browser->priv->objects_info[POS_TYPES].schema = GDA_CONNECTION_SCHEMA_TYPES;
	browser->priv->objects_info[POS_TYPES].data_retrieved = FALSE;
	browser->priv->objects_info[POS_TYPES].title = _("Types");
	browser->priv->objects_info[POS_TYPES].new_function = gnome_db_browser_types_new;
	browser->priv->objects_info[POS_TYPES].show_function = NULL;
	browser->priv->objects_info[POS_TYPES].clear_function = NULL;

	browser->priv->objects_info[POS_VIEWS].schema = GDA_CONNECTION_SCHEMA_VIEWS;
	browser->priv->objects_info[POS_VIEWS].data_retrieved = FALSE;
	browser->priv->objects_info[POS_VIEWS].title = _("Views");
	browser->priv->objects_info[POS_VIEWS].new_function = gnome_db_browser_views_new;
	browser->priv->objects_info[POS_VIEWS].show_function = gnome_db_browser_views_show;
	browser->priv->objects_info[POS_VIEWS].clear_function = gnome_db_browser_views_clear;

	browser->priv->objects_info[POS_PROCEDURES].schema = GDA_CONNECTION_SCHEMA_PROCEDURES;
	browser->priv->objects_info[POS_PROCEDURES].data_retrieved = FALSE;
	browser->priv->objects_info[POS_PROCEDURES].title = _("Procedures");
	browser->priv->objects_info[POS_PROCEDURES].new_function = gnome_db_browser_procedures_new;
	browser->priv->objects_info[POS_PROCEDURES].show_function = NULL;
	browser->priv->objects_info[POS_PROCEDURES].clear_function = NULL;

	browser->priv->objects_info[POS_AGGREGATES].schema = GDA_CONNECTION_SCHEMA_AGGREGATES;
	browser->priv->objects_info[POS_AGGREGATES].data_retrieved = FALSE;
	browser->priv->objects_info[POS_AGGREGATES].title = _("Aggregates");
	browser->priv->objects_info[POS_AGGREGATES].new_function = NULL;
	browser->priv->objects_info[POS_AGGREGATES].show_function = NULL;
	browser->priv->objects_info[POS_AGGREGATES].clear_function = NULL;

	browser->priv->objects_info[POS_TRIGGERS].schema = GDA_CONNECTION_SCHEMA_TRIGGERS;
	browser->priv->objects_info[POS_TRIGGERS].data_retrieved = FALSE;
	browser->priv->objects_info[POS_TRIGGERS].title = _("Triggers");
	browser->priv->objects_info[POS_TRIGGERS].new_function = NULL;
	browser->priv->objects_info[POS_TRIGGERS].show_function = NULL;
	browser->priv->objects_info[POS_TRIGGERS].clear_function = NULL;

	/* create widgets */
	for (i = 0;
	     i < sizeof (browser->priv->objects_info) / sizeof (browser->priv->objects_info[0]);
	     i++) {
		GtkWidget *detail;

		browser->priv->objects_info[i].object_list = gnome_db_list_new ();
		g_signal_connect (G_OBJECT (browser->priv->objects_info[i].object_list),
				  "row_selected", G_CALLBACK (object_selected_cb),
				  browser);
		g_signal_connect (G_OBJECT (browser->priv->objects_info[i].object_list),
				  "selection_cleared",
				  G_CALLBACK (object_selection_cleared_cb), browser);
		gtk_widget_show (browser->priv->objects_info[i].object_list);
		gtk_notebook_append_page (
			GTK_NOTEBOOK (browser->priv->object_notebook),
			browser->priv->objects_info[i].object_list,
			gtk_label_new (browser->priv->objects_info[i].title));

		if (browser->priv->objects_info[i].new_function != NULL) {
			detail = browser->priv->objects_info[i].new_function ();
			gtk_widget_show (detail);
		}
		else {
			detail = gnome_db_new_label_widget ("TODO");
		}

		gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
					  detail, gtk_label_new ("TAB"));
		browser->priv->objects_info[i].object_detail = detail;
	}
}

/*
 * GnomeDbBrowser class implementation
 */

static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_browser_set_property;
	object_class->get_property = gnome_db_browser_get_property;
	object_class->finalize = gnome_db_browser_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_CONNECTION,
		g_param_spec_object ("connection", NULL, NULL,
				     GDA_TYPE_CONNECTION,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* add class signals */
	browser_signals[PROGRESS_MESSAGE] =
		g_signal_new ("progress_message",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbBrowserClass, progress_message),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__STRING,
			      G_TYPE_NONE, 1, G_TYPE_STRING);
}

static void
gnome_db_browser_init (GnomeDbBrowser *browser, GnomeDbBrowserClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* allocate private structure */
	browser->priv = g_new0 (GnomeDbBrowserPrivate, 1);
	browser->priv->cnc = NULL;

	/* set up widgets */
	browser->priv->paned = gnome_db_new_hpaned_widget ();
	gtk_box_pack_start (GTK_BOX (browser), browser->priv->paned, TRUE, TRUE, 0);

	browser->priv->object_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_popup_disable (GTK_NOTEBOOK (browser->priv->object_notebook));
	g_signal_connect (G_OBJECT (browser->priv->object_notebook), "switch_page",
			  G_CALLBACK (notebook_page_changed_cb), browser);
	gtk_paned_add1 (GTK_PANED (browser->priv->paned), browser->priv->object_notebook);

	browser->priv->detail_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_popup_disable (GTK_NOTEBOOK (browser->priv->detail_notebook));
	gtk_paned_add2 (GTK_PANED (browser->priv->paned), browser->priv->detail_notebook);

	initialize_object_widgets (browser);
}

static void
gnome_db_browser_set_property (GObject *object,
			       guint param_id,
			       const GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		gnome_db_browser_set_connection (browser,
						 GDA_CONNECTION (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_get_property (GObject *object,
			       guint param_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		g_value_set_object (value, G_OBJECT (browser->priv->cnc));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_finalize (GObject *object)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* free memory */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	g_free (browser->priv);
	browser->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_browser_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBrowserClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_browser_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBrowser),
			0,
			(GInstanceInitFunc) gnome_db_browser_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbBrowser", &info, 0);
	}
	return type;
}

/**
 * gnome_db_browser_new
 *
 * Create a new #GnomeDbBrowser widget. This widget lets you present to your
 * users a complete database browser, which lets him/her navigate through
 * his/her database objects (tables, views, procedures, etc), in a
 * visual way.
 *
 * Returns: the newly created object.
 */
GtkWidget *
gnome_db_browser_new (void)
{
	GnomeDbBrowser *browser;

	browser = g_object_new (GNOME_DB_TYPE_BROWSER, NULL);
	return GTK_WIDGET (browser);
}

/**
 * gnome_db_browser_get_connection
 * @browser: a #GnomeDbBrowser widget.
 *
 * Retrieve the #GdaConnection object being used by the given browser
 * widget. The connection object associated with the browser makes it
 * show the objects for that connection.
 *
 * Returns: a #GdaConnection object.
 */
GdaConnection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser)
{
	g_return_val_if_fail (GNOME_DB_IS_BROWSER (browser), NULL);
	return browser->priv->cnc;
}

/*
 * gnome_db_browser_set_connection
 * @browser: a #GnomeDbBrowser widget.
 * @cnc: a #GdaConnection object.
 *
 * Associates a #GdaConnection object with the given browser widget. This will
 * make the browser widget refresh its widgets and immediately display
 * the objects contained in the new connection (@cnc).
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, GdaConnection *cnc)
{
	gint n;
	gint object_count;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE],
		       0, _("Loading database schemas..."));

	/* clear the current setup */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	object_count = sizeof (browser->priv->objects_info) / sizeof (browser->priv->objects_info[0]);
	for (n = 0; n < object_count; n++) {
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     NULL, 0);
		browser->priv->objects_info[n].data_retrieved = FALSE;

		if (browser->priv->objects_info[n].clear_function != NULL) {
			browser->priv->objects_info[n].clear_function (
				browser->priv->objects_info[n].object_detail);
		}
	}

	/* refresh views */
	if (GDA_IS_CONNECTION (cnc))
		g_object_ref (G_OBJECT (cnc));
	browser->priv->cnc = cnc;

	/* refresh current view */
	for (n = 0; n < object_count; n++) {
		GdaDataModel *recset;

		recset = gda_connection_get_schema (
			browser->priv->cnc,
			browser->priv->objects_info[n].schema,
			NULL);
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     recset, 0);
		browser->priv->objects_info[n].data_retrieved = TRUE;

		g_object_unref (G_OBJECT (recset));
	}

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE], 0, NULL);
}

/**
 * gnome_db_browser_refresh
 * @browser: a #GnomeDbBrowser widget.
 *
 * Make the browser widget refresh its currently displayed data.
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *browser)
{
	gint current;
	GdaDataModel *recset;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE],
		       0, _("Refreshing data..."));

	current = gtk_notebook_get_current_page (
		GTK_NOTEBOOK (browser->priv->object_notebook));

	/* refresh the object list */
	recset = gda_connection_get_schema (
			browser->priv->cnc,
			browser->priv->objects_info[current].schema,
			NULL);
	gnome_db_list_set_model (
		GNOME_DB_LIST (browser->priv->objects_info[current].object_list),
		recset, 0);
	browser->priv->objects_info[current].data_retrieved = TRUE;
	g_object_unref (G_OBJECT (recset));

	/* clear the detail widget */
	if (browser->priv->objects_info[current].clear_function != NULL) {
		browser->priv->objects_info[current].clear_function (
			browser->priv->objects_info[current].object_detail);
	}

	g_signal_emit (G_OBJECT (browser), browser_signals[PROGRESS_MESSAGE], 0, NULL);
}
