/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtklabel.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkpaned.h>
#include <libgnomedb/gnome-db-browser.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-browser-private.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbBrowserPrivate {
	GdaConnection *cnc;

	/* widgets */
	GtkWidget *paned;
	GtkWidget *object_notebook;
	GtkWidget *detail_notebook;

	/* objects management */
	struct {
		GdaConnectionSchema schema;
		GtkWidget *object_list;
		GtkWidget *object_detail;
		gboolean data_retrieved;
	} objects_info [6];
};

static void gnome_db_browser_class_init   (GnomeDbBrowserClass *klass);
static void gnome_db_browser_init         (GnomeDbBrowser *browser,
					   GnomeDbBrowserClass *klass);
static void gnome_db_browser_set_property (GObject *object,
					   guint paramid,
					   const GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_get_property (GObject *object,
					   guint param_id,
					   GValue *value,
					   GParamSpec *pspec);
static void gnome_db_browser_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_CONNECTION
};

static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
notebook_page_changed_cb (GtkNotebook *notebook,
			  GtkNotebookPage *page,
			  guint page_num,
			  gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	gtk_notebook_set_current_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				       page_num);
}

static void
table_selected_cb (GnomeDbList *list, gint row, gpointer user_data)
{
	GdaDataModel *model;
	const GdaValue *value;
	const gchar *str;
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	model = gnome_db_list_get_model (
		GNOME_DB_LIST (browser->priv->objects_info[0].object_list));
	value = gda_data_model_get_value_at (model, 0, row);
	if (!value)
		return;

	str = gda_value_get_string ((GdaValue *) value);
	gnome_db_browser_tables_show (browser->priv->objects_info[0].object_detail,
				      browser->priv->cnc, str);
}

static void
table_selection_cleared_cb (GnomeDbList *list, gpointer user_data)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) user_data;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	gnome_db_browser_tables_clear (browser->priv->objects_info[0].object_detail);
}

/*
 * Private functions
 */

static void
initialize_object_widgets (GnomeDbBrowser *browser)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* initialize tables widgets */
	browser->priv->objects_info[0].schema = GDA_CONNECTION_SCHEMA_TABLES;
	browser->priv->objects_info[0].data_retrieved = FALSE;

	browser->priv->objects_info[0].object_list = gnome_db_list_new ();
	g_signal_connect (G_OBJECT (browser->priv->objects_info[0].object_list),
			  "row_selected", G_CALLBACK (table_selected_cb), browser);
	g_signal_connect (G_OBJECT (browser->priv->objects_info[0].object_list),
			  "selection_cleared", G_CALLBACK (table_selection_cleared_cb),
			  browser);
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[0].object_list,
				  gtk_label_new (_("Tables")));
	gtk_widget_show (browser->priv->objects_info[0].object_list);

	browser->priv->objects_info[0].object_detail = gnome_db_browser_tables_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				  browser->priv->objects_info[0].object_detail,
				  gtk_label_new (_("Tables")));

	/* initialize types widgets */
	browser->priv->objects_info[1].schema = GDA_CONNECTION_SCHEMA_TYPES;
	browser->priv->objects_info[1].data_retrieved = FALSE;

	browser->priv->objects_info[1].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[1].object_list,
				  gtk_label_new (_("Types")));
	gtk_widget_show (browser->priv->objects_info[1].object_list);

	browser->priv->objects_info[1].object_detail = gnome_db_browser_types_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				  browser->priv->objects_info[1].object_detail,
				  gtk_label_new (_("Types")));

	/* initialize views widgets */
	browser->priv->objects_info[2].schema = GDA_CONNECTION_SCHEMA_VIEWS;
	browser->priv->objects_info[2].data_retrieved = FALSE;

	browser->priv->objects_info[2].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[2].object_list,
				  gtk_label_new (_("Views")));
	gtk_widget_show (browser->priv->objects_info[2].object_list);

	browser->priv->objects_info[2].object_detail = gnome_db_browser_views_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				  browser->priv->objects_info[2].object_detail,
				  gtk_label_new (_("Views")));

	/* initialize procedures widgets */
	browser->priv->objects_info[3].schema = GDA_CONNECTION_SCHEMA_PROCEDURES;
	browser->priv->objects_info[3].data_retrieved = FALSE;

	browser->priv->objects_info[3].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[3].object_list,
				  gtk_label_new (_("Procedures")));
	gtk_widget_show (browser->priv->objects_info[3].object_list);

	browser->priv->objects_info[3].object_detail = gnome_db_browser_procedures_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
				  browser->priv->objects_info[3].object_detail,
				  gtk_label_new (_("Procedures")));

	/* initialize aggregate widgets */
	browser->priv->objects_info[4].schema = GDA_CONNECTION_SCHEMA_AGGREGATES;
	browser->priv->objects_info[4].data_retrieved = FALSE;

	browser->priv->objects_info[4].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[4].object_list,
				  gtk_label_new (_("Aggregates")));
	gtk_widget_show (browser->priv->objects_info[4].object_list);

	browser->priv->objects_info[4].object_detail = gnome_db_new_label_widget ("TODO");
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
	                          browser->priv->objects_info[4].object_detail,
				  gtk_label_new (_("Aggregates")));

	/* initialize trigger widgets */
	browser->priv->objects_info[5].schema = GDA_CONNECTION_SCHEMA_TRIGGERS;
	browser->priv->objects_info[5].data_retrieved = FALSE;

	browser->priv->objects_info[5].object_list = gnome_db_list_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->object_notebook),
				  browser->priv->objects_info[5].object_list,
				  gtk_label_new (_("Triggers")));
	gtk_widget_show (browser->priv->objects_info[5].object_list);

	browser->priv->objects_info[5].object_detail = gnome_db_new_label_widget ("TODO");
	gtk_notebook_append_page (GTK_NOTEBOOK (browser->priv->detail_notebook),
	                          browser->priv->objects_info[5].object_detail,
				  gtk_label_new (_("Triggers")));
}

/*
 * GnomeDbBrowser class implementation
 */

static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_browser_set_property;
	object_class->get_property = gnome_db_browser_get_property;
	object_class->finalize = gnome_db_browser_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_CONNECTION,
		g_param_spec_object ("connection", NULL, NULL,
				     GDA_TYPE_CONNECTION,
				     (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_browser_init (GnomeDbBrowser *browser, GnomeDbBrowserClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* allocate private structure */
	browser->priv = g_new0 (GnomeDbBrowserPrivate, 1);
	browser->priv->cnc = NULL;

	/* set up widgets */
	browser->priv->paned = gnome_db_new_hpaned_widget ();
	gtk_box_pack_start (GTK_BOX (browser), browser->priv->paned, TRUE, TRUE, 0);

	browser->priv->object_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_popup_disable (GTK_NOTEBOOK (browser->priv->object_notebook));
	g_signal_connect (G_OBJECT (browser->priv->object_notebook), "switch_page",
			  G_CALLBACK (notebook_page_changed_cb), browser);
	gtk_paned_add1 (GTK_PANED (browser->priv->paned), browser->priv->object_notebook);

	browser->priv->detail_notebook = gnome_db_new_notebook_widget ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (browser->priv->detail_notebook), FALSE);
	gtk_notebook_popup_disable (GTK_NOTEBOOK (browser->priv->detail_notebook));
	gtk_paned_add2 (GTK_PANED (browser->priv->paned), browser->priv->detail_notebook);

	initialize_object_widgets (browser);
}

static void
gnome_db_browser_set_property (GObject *object,
			       guint param_id,
			       const GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		gnome_db_browser_set_connection (browser,
						 GDA_CONNECTION (g_value_get_object (value)));
		break;
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_get_property (GObject *object,
			       guint param_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	switch (param_id) {
	case PROP_CONNECTION :
		g_value_set_object (value, G_OBJECT (browser->priv->cnc));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_browser_finalize (GObject *object)
{
	GnomeDbBrowser *browser = (GnomeDbBrowser *) object;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* free memory */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	g_free (browser->priv);
	browser->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_browser_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbBrowserClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_browser_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbBrowser),
			0,
			(GInstanceInitFunc) gnome_db_browser_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbBrowser", &info, 0);
	}
	return type;
}

/**
 * gnome_db_browser_new
 *
 * Create a new #GnomeDbBrowser widget. This widget lets you present to your
 * users a complete database browser, which lets him/her navigate through
 * his/her database objects (tables, views, procedures, etc), in a
 * visual way.
 *
 * Returns: the newly created object.
 */
GtkWidget *
gnome_db_browser_new (void)
{
	GnomeDbBrowser *browser;

	browser = g_object_new (GNOME_DB_TYPE_BROWSER, NULL);
	return GTK_WIDGET (browser);
}

/**
 * gnome_db_browser_get_connection
 * @browser: a #GnomeDbBrowser widget.
 *
 * Retrieve the #GdaConnection object being used by the given browser
 * widget. The connection object associated with the browser makes it
 * show the objects for that connection.
 *
 * Returns: a #GdaConnection object.
 */
GdaConnection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser)
{
	g_return_val_if_fail (GNOME_DB_IS_BROWSER (browser), NULL);
	return browser->priv->cnc;
}

/*
 * gnome_db_browser_set_connection
 * @browser: a #GnomeDbBrowser widget.
 * @cnc: a #GdaConnection object.
 *
 * Associates a #GdaConnection object with the given browser widget. This will
 * make the browser widget refresh its widgets and immediately display
 * the objects contained in the new connection (@cnc).
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, GdaConnection *cnc)
{
	gint n;
	gint object_count;

	g_return_if_fail (GNOME_DB_IS_BROWSER (browser));

	/* clear the current setup */
	if (GDA_IS_CONNECTION (browser->priv->cnc)) {
		g_object_unref (G_OBJECT (browser->priv->cnc));
		browser->priv->cnc = NULL;
	}

	object_count = sizeof (browser->priv->objects_info) / sizeof (browser->priv->objects_info[0]);
	for (n = 0; n < object_count; n++) {
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     NULL, 0);
		browser->priv->objects_info[n].data_retrieved = FALSE;

		switch (browser->priv->objects_info[n].schema) {
		case GDA_CONNECTION_SCHEMA_TABLES :
			gnome_db_browser_tables_clear (browser->priv->objects_info[n].object_detail);
			break;
		default :
		}
	}

	/* refresh views */
	if (GDA_IS_CONNECTION (cnc))
		g_object_ref (G_OBJECT (cnc));
	browser->priv->cnc = cnc;

	/* refresh current view */
	for (n = 0; n < object_count; n++) {
		GdaDataModel *recset;

		recset = gda_connection_get_schema (
			browser->priv->cnc,
			browser->priv->objects_info[n].schema,
			NULL);
		gnome_db_list_set_model (
		     GNOME_DB_LIST (browser->priv->objects_info[n].object_list),
		     recset, 0);

		g_object_unref (G_OBJECT (recset));
	}
}
