/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libgda/gda-config.h>
#include <libgnomedb/gnome-db-login.h>
#include <libgnomedb/gnome-db-util.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtktable.h>

#include "gnome-db-intl.h"

struct _GnomeDbLoginPrivate {
	gchar *dsn_name;
	GtkWidget *dsn_entry;
	GtkWidget *username_entry;
	GtkWidget *password_entry;
};

static void gnome_db_login_class_init   (GnomeDbLoginClass *klass);
static void gnome_db_login_init         (GnomeDbLogin *login, GnomeDbLoginClass *klass);
static void gnome_db_login_set_property (GObject *object,
					 guint paramid,
					 const GValue *value,
					 GParamSpec *pspec);
static void gnome_db_login_get_property (GObject *object,
					 guint param_id,
					 GValue *value,
					 GParamSpec *pspec);
static void gnome_db_login_finalize     (GObject *object);

enum {
	PROP_0,
	PROP_DSN,
	PROP_USERNAME,
	PROP_PASSWORD
};

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
load_data_sources (GtkCombo *combo)
{
	GList *dsn_list;
	GList *l;
	GList *strings = NULL;

	g_return_if_fail (GTK_IS_COMBO (combo));

	dsn_list = gda_config_get_data_source_list ();
	for (l = dsn_list; l != NULL; l = l->next) {
		GdaDataSourceInfo *dsn_info = (GdaDataSourceInfo *) l->data;
		strings = g_list_append (strings, dsn_info->name);
	}

	gtk_combo_set_popdown_strings (combo, strings);

	/* free memory */
	g_list_free (strings);
	gda_config_free_data_source_list (dsn_list);
}

/*
 * GnomeDbLogin class implementation
 */

static void
gnome_db_login_class_init (GnomeDbLoginClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_login_set_property;
	object_class->get_property = gnome_db_login_get_property;
	object_class->finalize = gnome_db_login_finalize;

	/* add class properties */
	g_object_class_install_property (
		object_class, PROP_DSN,
		g_param_spec_string ("dsn", NULL, NULL, NULL, G_PARAM_READABLE));
	g_object_class_install_property (
		object_class, PROP_USERNAME,
		g_param_spec_string ("username", NULL, NULL, NULL, G_PARAM_READABLE));
	g_object_class_install_property (
		object_class, PROP_PASSWORD,
		g_param_spec_string ("password", NULL, NULL, NULL, G_PARAM_READABLE));
}

static void
gnome_db_login_init (GnomeDbLogin *login, GnomeDbLoginClass *klass)
{
	/* allocate the internal structure */
	login->priv = g_new0 (GnomeDbLoginPrivate, 1);
	login->priv->dsn_name = NULL;
}

static void
gnome_db_login_set_property (GObject *object,
			     guint param_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	switch (param_id) {
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_login_get_property (GObject *object,
			       guint param_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	switch (param_id) {
	case PROP_DSN :
		g_value_set_string (value, gnome_db_login_get_dsn (login));
		break;
	case PROP_USERNAME :
		g_value_set_string (value, gnome_db_login_get_username (login));
		break;
	case PROP_PASSWORD :
		g_value_set_string (value, gnome_db_login_get_password (login));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_login_finalize (GObject *object)
{
	GnomeDbLogin *login = (GnomeDbLogin *) object;

	g_return_if_fail (GNOME_DB_IS_LOGIN (login));

	/* free memory */
	if (login->priv->dsn_name != NULL) {
		g_free (login->priv->dsn_name);
		login->priv->dsn_name = NULL;
	}

	g_free (login->priv);
	login->priv = NULL;

	parent_class->finalize (object);
}

GType
gnome_db_login_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbLoginClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_login_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbLogin),
			0,
			(GInstanceInitFunc) gnome_db_login_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbLogin", &info, 0);
	}
	return type;
}

static GtkWidget *
gnome_db_login_construct (GnomeDbLogin *login, const gchar *dsn)
{
	GtkWidget *table;
	GtkWidget *label;

	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), GTK_WIDGET (login));

	if (dsn != NULL)
		login->priv->dsn_name = g_strdup (dsn);
	else
		login->priv->dsn_name = NULL;

	table = gnome_db_new_table_widget (3, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (login), table, TRUE, TRUE, 0);

	label = gnome_db_new_label_widget (_("Data Source"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
	if (dsn != NULL) {
		login->priv->dsn_entry = gnome_db_new_entry_widget (0, FALSE);
		gtk_entry_set_text (GTK_ENTRY (login->priv->dsn_entry), dsn);
		gtk_editable_set_editable (GTK_EDITABLE (login->priv->dsn_entry), FALSE);
	}
	else {
		login->priv->dsn_entry = gnome_db_new_combo_widget ();
		gtk_combo_set_value_in_list (GTK_COMBO (login->priv->dsn_entry), TRUE, FALSE);
		load_data_sources (GTK_COMBO (login->priv->dsn_entry));
	}
	gtk_table_attach (GTK_TABLE (table), login->priv->dsn_entry, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Username"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	login->priv->username_entry = gnome_db_new_entry_widget (0, TRUE);
	gtk_table_attach (GTK_TABLE (table), login->priv->username_entry, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 2, 2);

	label = gnome_db_new_label_widget (_("Password"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	login->priv->password_entry = gnome_db_new_entry_widget (0, TRUE);
	gtk_entry_set_visibility (GTK_ENTRY (login->priv->password_entry), FALSE);
	gtk_table_attach (GTK_TABLE (table), login->priv->password_entry, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 2, 2);

	return GTK_WIDGET (login);
}

/**
 * gnome_db_login_new
 */
GtkWidget *
gnome_db_login_new (const gchar *dsn)
{
	GnomeDbLogin *login;

	login = g_object_new (GNOME_DB_TYPE_LOGIN, NULL);
	return gnome_db_login_construct (login, dsn);
}

/**
 * gnome_db_login_get_dsn
 */
const gchar *
gnome_db_login_get_dsn (GnomeDbLogin *login)
{
	GtkEntry *entry;

	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);

	if (login->priv->dsn_name)
		return (const gchar *) login->priv->dsn_name;

	entry = GTK_ENTRY (GTK_COMBO (login->priv->dsn_entry)->entry);
	g_return_val_if_fail (GTK_IS_ENTRY (entry), NULL);

	return gtk_entry_get_text (entry);
}

/**
 * gnome_db_login_get_username
 */
const gchar *
gnome_db_login_get_username (GnomeDbLogin *login)
{
	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);
	return gtk_entry_get_text (GTK_ENTRY (login->priv->username_entry));
}

/**
 * gnome_db_login_get_password
 */
const gchar *
gnome_db_login_get_password (GnomeDbLogin *login)
{
	g_return_val_if_fail (GNOME_DB_IS_LOGIN (login), NULL);
	return gtk_entry_get_text (GTK_ENTRY (login->priv->password_entry));
}
