/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgnomedb/gnome-db-model.h>
#include <libgnomedb/gnome-db-util.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeviewcolumn.h>
#include "libgnomedb-private.h"

#define DATA_MODEL_INFO_TYPE data_model_info_get_type ()

/*
 * Private functions
 */

static void
data_model_info_free (DataModelInfo *info)
{
	g_object_unref (G_OBJECT (info->model));
	g_free (info);
}

static DataModelInfo *
data_model_info_copy (DataModelInfo *src)
{
	DataModelInfo *info;

	g_return_val_if_fail (src != NULL, NULL);

	info = g_new0 (DataModelInfo, 1);
	info->row = src->row;
	info->model = src->model;
	if (info->model)
		g_object_ref (G_OBJECT (info->model));

	return info;
}

static GType
data_model_info_get_type (void)
{
	static GType type = 0;

	if (!type) {
		type = g_boxed_type_register_static (
			"GNOME_DB_DataModelInfo",
			(GBoxedCopyFunc) data_model_info_copy,
			(GBoxedFreeFunc) data_model_info_free);
	}
	return type;
}

/*
 * Callbacks
 */

static void
value_set_func (GtkTreeViewColumn *tree_column,
                GtkCellRenderer *cell,
                GtkTreeModel *model,
                GtkTreeIter *iter,
                gpointer user_data)
{
	DataModelInfo *info = NULL;
	GdaValue *value;
	gchar *txt;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return;

	value = gda_data_model_get_value_at (info->model,
					     GPOINTER_TO_INT (user_data),
					     info->row);
	if (!value)
		return;

	txt = gda_value_stringify (value);
	g_object_set (G_OBJECT (cell), "text", txt, NULL);

	g_free (txt);
}

/**
 * gnome_db_model_to_gtk_tree_model
 */
GtkTreeView *
gnome_db_model_to_gtk_tree_view (GdaDataModel *model)
{
	gint i;
	gint row_count;
	gint col_count;
	GtkTreeView *tree_view;
	GtkListStore *tree_model = NULL;
	GtkCellRenderer *text_renderer;

	tree_model = gtk_list_store_new (1, DATA_MODEL_INFO_TYPE);
	row_count = gda_data_model_get_n_rows (model);
	for (i = 0; i < row_count; i++) {
		DataModelInfo info;
		GtkTreeIter iter;

		info.row = i;
		info.model = model;
		gtk_list_store_append (tree_model, &iter);
		gtk_list_store_set (tree_model, &iter, 0, &info, -1);
	}

	/* create the tree view widget */
	tree_view = gnome_db_new_tree_view_widget (GTK_TREE_MODEL (tree_model));
	g_object_unref (G_OBJECT (tree_model));

	text_renderer = gtk_cell_renderer_text_new ();

	col_count = gda_data_model_get_n_columns (model);
	for (i = 0; i < col_count; i++) {
		gtk_tree_view_insert_column_with_data_func (
			GTK_TREE_VIEW (tree_view),
			-1,
			gda_data_model_get_column_title (model, i),
			text_renderer,
			value_set_func,
			GINT_TO_POINTER (i),
			NULL);
	}

	gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree_view), TRUE);

	return tree_view;
}
