/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gnome;

import org.gnu.glib.Handle;
import org.gnu.glib.Type;
import org.gnu.gtk.Button;
import org.gnu.gtk.Container;
import org.gnu.gtk.Widget;
import org.gnu.gtk.Window;

/**
 * The GNOME druid is a system for assisting the user with installing a
 * service. It is roughly equivalent in functionality to the Wizards
 * available in Windows.
 */
public class Druid extends Container {
	
	/**
	 * Construct a new Druid object.
	 */
	public Druid() {
		super(gnome_druid_new());
	}
	
	/**
	 * Construct a new Druid from a handle to a native resource.
	 */
	public Druid(Handle handle) {
	    super(handle);
	}
	
	/**
	 * Construct a new Druid object that already contains a window.
	 */
	public Druid(String title, Window parent, boolean closeOnCancel, Widget window) {
		super(gnome_druid_new_with_window(title, parent.getHandle(), closeOnCancel, window.getHandle()));
	}
	
	/**
	 * Sets the sensitivity of the buttons on the Druid dialog.
	 * @param backSensitive Determines if the "Back" button is sensitive.
	 * @param nextSensitive Determines if the "Next" button is sensitive.
	 * @param cancelSensitive Determines if the "Cancel" button is sensitive.
	 * @param helpSensitive Determines if the "Help" button is sensitive.
	 */
	public void setButtonsSensitive(boolean backSensitive, boolean nextSensitive, boolean cancelSensitive, boolean helpSensitive) {
		gnome_druid_set_buttons_sensitive(getHandle(), backSensitive, nextSensitive, cancelSensitive, helpSensitive);
	}
	
	/**
	 * Sets the text on the last button of the dialog.  If <i>showFinish</i> is true the text
	 * becomes "Finish".  Otherwise the text is set to "Cancel".
	 * @param showFinish Determines the text of the last button on the dialog.
	 */
	public void setShowFinish(boolean showFinish) {
		gnome_druid_set_show_finish(getHandle(), showFinish);
	}

	/**
	 * Sets whether the help button should be displayed 
	 */
	public void setShowHelp( boolean showHelp ){
		gnome_druid_set_show_help( getHandle(), showHelp );
	}
	
	/**
	 * Prepends a DruidPage into the list of pages for this dialog.
	 * @param page The page to prepend.
	 */
	public void prependPage(DruidPage page) {
		gnome_druid_prepend_page(getHandle(), page.getHandle());
	}
	
	/**
	 * Append a DruidPage into the list of pages for this dialog.
	 * @param page The page to append.
	 */
	public void appendPage(DruidPage page) {
		gnome_druid_append_page(getHandle(), page.getHandle());
	}
	
	/**
	 * Insert <i>page</i> after <i>backPage</i> into the list of pages for
	 * this dialog.  If </i>backPage is not present or null, </i>page</i> will
	 * be prepended.
	 * @param backPage
	 * @param page
	 */
	public void insertPage(DruidPage backPage, DruidPage page) {
		gnome_druid_insert_page(getHandle(), backPage.getHandle(), page.getHandle());
	}
	
	/**
	 * Set the currently displayed page to <i>page</i>.
	 * @param page The page to make the current page.
	 */
	public void setPage(DruidPage page) {
		gnome_druid_set_page(getHandle(), page.getHandle());
	}
	
	/**
	 * Retrieve the "Help" button.
	 */
	public Button getHelpButton() {
		Handle hndl = getHelp(getHandle());
        return Button.getButton(hndl);
	}
	
	/**
	 * Retrieve the "Back" button.
	 */
	public Button getBackButton() {
		Handle hndl = getBack(getHandle());
		return Button.getButton(hndl);
	}
	
	/**
	 * Retrieve the "Next" button.
	 */
	public Button getNextButton() {
		Handle hndl = getNext(getHandle());
		return Button.getButton(hndl);
	}
	
	/**
	 * Retrieve the "Cancel" button.
	 */
	public Button getCancelButton() {
		Handle hndl = getCancel(getHandle());
        return Button.getButton(hndl);
	}
	
	/**
	 * Retrieve the "Finish" button.
	 */
	public Button getFinishButton() {
		Handle hndl = getFinish(getHandle());
        return Button.getButton(hndl);
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gnome_druid_get_type());
	}

	native static final protected Handle getHelp(Handle cptr);
	native static final protected Handle getBack(Handle cptr);
	native static final protected Handle getNext(Handle cptr);
	native static final protected Handle getCancel(Handle cptr);
	native static final protected Handle getFinish(Handle cptr);
	native static final protected int gnome_druid_get_type();
	native static final protected Handle gnome_druid_new();
	native static final protected void gnome_druid_set_buttons_sensitive(Handle druid, boolean backSensitive, boolean nextSensitive, boolean cancelSensitive, boolean helpSensitive);
	native static final protected void gnome_druid_set_show_finish(Handle druid, boolean showFinish);
	native static final protected void gnome_druid_set_show_help(Handle druid, boolean showHelp);
	native static final protected void gnome_druid_prepend_page(Handle druid, Handle page);
	native static final protected void gnome_druid_insert_page(Handle druid, Handle backPage, Handle page);
	native static final protected void gnome_druid_append_page(Handle druid, Handle page);
	native static final protected void gnome_druid_set_page(Handle druid, Handle page);
	native static final protected Handle gnome_druid_new_with_window(String title, Handle parent, boolean closeOnCancel, Handle window);

}
