/* $Id: userserver.c,v 1.14 2001/01/21 23:55:45 menesis Exp $ */

#include "common.h"

#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <pwd.h>

void show_contact_message_client( GSList *contact, gint sock );
void icq_msgbox_client( STORED_MESSAGE_PTR message_text, UIN_T uin, gint sock );

int userserver_socket;
gchar *userserver_name;
gchar *userserver_sym;

gint userserver_gdk_input;

#define USERSERVER_MAX_LENGTH 8192

typedef enum {
	USERSERVER_CMD_NOOP = 0,
	USERSERVER_CMD_DO  = 1,
	USERSERVER_CMD_READMSG = 2,
} UserServerCmd;

void userserver_handler(gint sock)
{
	UserServerCmd cmdtype;
	unsigned long cmdlen;
	gchar *cmdbuf;
	gchar c;
	gchar *cmd, *args;
	GSList *contact;

	/* Read in the packet */

	if (read(sock,&c,1) != 1)
	{
		fprintf(stderr,"Fatal error reading from control socke, ignoring request.\n");
		return;
	}

	cmdtype = c;

	if (read(sock,&cmdlen,sizeof(unsigned long)) != sizeof(unsigned long))
	{
		fprintf(stderr,"Unable to read length information from control socket, ignoring request.\n");
		return;
	}
	
	cmdlen = ntohl(cmdlen);

	if (cmdlen > USERSERVER_MAX_LENGTH)
	{
		fprintf(stderr,"Requested length of %ld exceeds hardcoded maximum of %ld, ignoring request.\n",
			cmdlen, (unsigned long)USERSERVER_MAX_LENGTH);
		return;
	}
	
	cmdbuf=g_malloc0(cmdlen);

	if (!cmdbuf)
	{
		fprintf(stderr,"Unable to allocated %ld bytes for request, ignoring request, and boy you're in trouble.\n",
			cmdlen);
		return;
	}


	if (read(sock,cmdbuf,cmdlen) != cmdlen)
	{
		fprintf(stderr,"Unable to read all %ld requested bytes from socket, ignoring request.\n",cmdlen);
		g_free(cmdbuf);
		return;
	}

	switch (cmdtype)
	{
	case USERSERVER_CMD_NOOP:
	  break;
		case USERSERVER_CMD_READMSG:
			contact = Contacts;
			while( contact != NULL && g_slist_length( kontakt->stored_messages ) == 0 )
				contact = contact->next;

			if( contact == NULL )
			{
				g_free( cmdbuf );
				cmdbuf = g_strdup( _("No messages available.\n\n") );
				cmdlen = g_htonl( strlen( cmdbuf + 1 ) );
				write( sock, &cmdlen, sizeof( unsigned long ) );
				write( sock, cmdbuf, strlen( cmdbuf ) + 1 );
				break;
			}

			show_contact_message_client( contact, sock );
			break;
		
		case USERSERVER_CMD_DO:
			cmd=strtok(cmdbuf," \t");
			if (!cmd) break;

			/* COMMAND: QUIT */
			if (!strcasecmp(cmd,"quit"))
				icq_quit(NULL,NULL);

			/* COMMAND: AWAY <MESSAGE> */
			else if (!strcasecmp(cmd,"away"))
			{
				char *msg;
	
				msg=strtok(NULL,"");
				if (!msg)
					break;

				g_free( Away_Message );
				Away_Message = g_strdup( msg );
			}

			/* COMMAND: USERADD <UIN> */
			else if (!strcasecmp(cmd,"adduser") || !strcasecmp(cmd,"useradd"))
			{
				char *uin;

				uin=strtok(NULL," ");
				if (!uin) break;
	
				Add_User(atol(uin), uin, TRUE);
			}

			/* COMMAND: MSG <UIN> <MSG> */
			else if (!strcasecmp(cmd,"msg"))
			{
				char *uin,*msg;
	
				uin=strtok(NULL," ");
				if (!uin || atoi(uin)==0) break;
				msg=strtok(NULL,"");
				if (!msg) break;

				icq_sendmsg(atoi(uin),msg,TRUE);
				break;
			}

			/* COMMAND: STATUS <offline|online|na|invisible|freechat|occupied|away|dnd> */
			else if (!strcasecmp(cmd,"status"))
			{
				args=strtok(NULL," \r\n");
				if (!args) break;

				if (!strcasecmp(args,"offline"))
					icq_set_status_offline( NULL, NULL );
				if (!strcasecmp(args,"online"))
					icq_set_status_online( NULL, NULL );
				if (!strcasecmp(args,"na"))
					icq_set_status_na( NULL, NULL );
				if (!strcasecmp(args,"invisible"))
					icq_set_status_invisible( NULL, NULL );
				if (!strcasecmp(args,"freechat"))
					icq_set_status_ffc( NULL, NULL );
				if (!strcasecmp(args,"occupied"))
					icq_set_status_occ( NULL, NULL );
				if (!strcasecmp(args,"away"))
					icq_set_status_away( NULL, NULL );
				if (!strcasecmp(args,"dnd"))
					icq_set_status_dnd( NULL, NULL );
			}

            /* COMMAND: READEVENT */
            else if(!strcasecmp(cmd,"readevent"))
            {
                contact = Contacts;

                while( contact != NULL && g_slist_length( kontakt->stored_messages ) == 0 )
                    contact = contact->next;

                if( contact != NULL )
                    show_contact_message( contact );
            }
		break;
	}

	free(cmdbuf);
	return;
}


void userserver_accept(gpointer data,gint sock,GdkInputCondition cond)
{
	int fd;

	fd = accept(sock, NULL,NULL);

	if (fd >= 0)
	{
		userserver_handler(fd);
		close(fd);
	}

}

/* This code is borrowed heavily (read: cut&paste) from sawmill.
 * Thanks John!
 */
void userserver_init(void)
{
	gchar namebuf[1024];
	gchar *user;
	gchar *uinstr;
	gchar *sym;

	struct passwd *pw;

	userserver_socket = -1;
	userserver_name = NULL;

	pw = getpwuid(getuid());

	if (!pw)
	  {
	    fprintf(stderr,"Unable to get passwd information for uid %d, naming service problem?\n",
		    getuid());
	    return;
	  }


	user = g_strdup( pw->pw_name );

	if (!user)
	{
	  fprintf(stderr,"No memory to duplicate username.\n");
	  return;
	}

	sprintf(namebuf,"/tmp/.gnomeicu-%s",user);

	g_free( user );
	
	if (mkdir(namebuf,0700) != 0)
	{
		if (errno == EEXIST)
		{
			struct stat st;

			if (stat(namebuf,&st) == 0)
			{
				if ((st.st_uid != getuid()) || (st.st_mode & (S_IRWXG | S_IRWXO)))
				{
					fprintf(stderr,"The %s directory is not owned by uid %d, or permissions are not secure enough.\n",
					        namebuf,getuid());
				}
			} else {
				perror(namebuf);
				return;
			}
		} else {
			perror(namebuf);
			return;
		}
	}
	strcat(namebuf, "/ctl" );
	sym = g_strdup( namebuf );
	userserver_sym = g_strdup( namebuf );
	strcat(namebuf,"-");

	uinstr = g_strdup_printf( "%d", our_info->uin );

	strcat(namebuf,uinstr);

	symlink( namebuf, sym );

	g_free( sym );
	g_free( uinstr );
	
	/* Delete the socket if it exists */
	if(access(namebuf, F_OK) == 0)
	{
		/* Socket already exists. Delete it */
		unlink(namebuf);
			
		if (access (namebuf, F_OK) == 0)
		{
			fprintf (stderr, "Can't delete %s\n", namebuf);
			return;
		}
	}
	
	userserver_socket = socket(AF_UNIX, SOCK_STREAM, 0);

	if(userserver_socket >= 0)
	{
		struct sockaddr_un addr;

		addr.sun_family = AF_UNIX;
		strcpy(addr.sun_path, namebuf);
		
		if(bind(userserver_socket, (struct sockaddr *)&addr,
			sizeof(addr.sun_family) + strlen(addr.sun_path) + 1) == 0)
		{
			chmod (namebuf, 0700);

			if( listen(userserver_socket, 5) == 0 )
			{
				userserver_gdk_input = gdk_input_add( userserver_socket,
								GDK_INPUT_READ|GDK_INPUT_EXCEPTION,
								(GdkInputFunction)userserver_accept,
								NULL);
				userserver_name = g_strdup(namebuf);
				return;
			} else {
				perror ("listen");
				close(userserver_socket);
				unlink(namebuf);
			}
		} else {
			perror ("bind");
			close(userserver_socket);
		}
	} else
		perror ("socket");

	userserver_socket = -1;
}

void userserver_kill (void)
{
	gchar tmp[1024];
	gint size;

	if(userserver_socket > 0)
	{
		size = readlink( userserver_sym, tmp, 1023 );
		tmp[size] = 0;
		if( !strcmp( userserver_name, tmp ) )
			unlink( userserver_sym );
		gdk_input_remove( userserver_gdk_input );
		close(userserver_socket);
		userserver_socket = -1;
		unlink(userserver_name);
		g_free(userserver_name);
		g_free(userserver_sym);
	}
}

void show_contact_message_client( GSList *contact, gint sock )
{
	STORED_MESSAGE_PTR msg = (STORED_MESSAGE_PTR)kontakt->stored_messages->data;

	if( kontakt->read_next &&
	    GTK_IS_WIDGET( kontakt->read_next ) &&
	    GTK_WIDGET_VISIBLE( kontakt->read_next ) )
		gtk_signal_emit_by_name( GTK_OBJECT( kontakt->read_next ), "clicked", MainData );

	if( g_slist_length( kontakt->stored_messages ) )
	{
		icq_msgbox_client( (msg), kontakt->uin, sock );
		g_free(msg->chatsessionname);
		if (msg->chatcontact && ((ChatContact *)msg->chatcontact)->contact == NULL)
		     g_free(msg->chatcontact);
		g_free( msg->message );
		g_free( kontakt->stored_messages->data );
		kontakt->stored_messages = g_slist_remove( kontakt->stored_messages, kontakt->stored_messages->data );

		if( g_slist_length( kontakt->stored_messages ) )
		{
			kontakt->icon_p = GetIcon_p( kontakt->status );
			kontakt->icon_b = GetIcon_b( kontakt->status );
			kontakt->need_update = 1;
		}

		Update_Contact_Style( Which_List( kontakt ), kontakt );
		applet_update( Current_Status, NULL );
		Shortcut_Update( kontakt );
	}
}

void icq_msgbox_client( STORED_MESSAGE_PTR message_text, UIN_T uin, gint sock )
{
	gint type = message_text->type;
	gchar *text = message_text->message;
	gchar *retmsg;
	GSList *contact;
	unsigned long u;
	gchar *ptr;

	contact = Find_User( uin );
	if( contact == NULL )
		return;

	if( g_slist_length( kontakt->stored_messages ) == 1 )
	{
		kontakt->icon_p = GetIcon_p( kontakt->status );
		kontakt->icon_b = GetIcon_b( kontakt->status );
		kontakt->need_update = 1;
	}

	switch( type )
	{
		case MESSAGE_TEXT:
			retmsg = g_strdup_printf( "Message from %s (%d):\n%s\n",
			                          kontakt->nick,
			                          kontakt->uin,
			                          text );
			break;
		case MESSAGE_URL:
			ptr = strchr( text, '\xFE' );
			ptr[0] = '\n';
			retmsg = g_strdup_printf( "URL from %s (%d):\n%s\n",
			                          kontakt->nick,
			                          kontakt->uin,
			                          text );
			break;
		case MESSAGE_FILE_REQ:
			retmsg = g_strdup_printf( "File request from %s (%d):\nAuto refusing",
			                          kontakt->nick,
			                          kontakt->uin );
			TCPRefuseFile( message_text->xfer );
			break;
		case MESSAGE_CHAT_REQ:
			retmsg = g_strdup_printf( "Chat request from %s (%d):\nAuto refusing",
			                          kontakt->nick,
			                          kontakt->uin );
			TCPRefuseChat( kontakt->sok,
				       ((STORED_MESSAGE_PTR)message_text)->chatcontact );
			break;
		case MESSAGE_AUTH_REQ:
			retmsg = g_strdup_printf( "Authorization request from %s (%d):\nAuto refusing",
			                          kontakt->nick,
			                          kontakt->uin );
			break;
		case MESSAGE_USER_ADD:
			retmsg = g_strdup_printf( "%s (%d) has added you to his/her list.\n",
			                          kontakt->nick,
			                          kontakt->uin );
			break;
		case MESSAGE_USER_AUTHD:
			retmsg = g_strdup_printf( "%s (%d) has authorized you to add him/her to your list.\n",
			                          kontakt->nick,
			                          kontakt->uin );
			break;
		case MESSAGE_CONT_LIST:
			retmsg = g_strdup_printf( "%s (%d) sent you a contact list.\n",
			                          kontakt->nick,
			                          kontakt->uin );
			break;
		default:
			retmsg = g_strdup_printf( "Unknown message from %s (%d).\n",
			                          kontakt->nick,
			                          kontakt->uin );
			break;
	}

	u = g_htonl( strlen( retmsg ) + 1 );
	write( sock, &u, sizeof( unsigned long ) );
	write( sock, retmsg, strlen( retmsg ) + 1 );
}
