/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*  This file is part of gnome-spell bonobo component
    copied from echo.c written by Miguel de Icaza and updated for Spell.idl needs

    Copyright (C) 1999, 2000 Helix Code, Inc.
    Authors:                 Radek Doulik <rodo@helixcode.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <bonobo.h>

#include "Spell.h"
#include "dictionary.h"

static BonoboObjectClass                  *dictionary_parent_class;
static POA_GNOME_Spell_Dictionary__vepv    dictionary_vepv;

/* #define DICT_DEBUG(x) x */
#define DICT_DEBUG(x)

static void
dictionary_object_init (GtkObject *object)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (object);

	dict->changed = TRUE;
	dict->config  = new_pspell_config ();
	dict->manager = NULL;
	dict->active  = TRUE;
}

static void
dictionary_object_destroy (GtkObject *object)
{
	GNOMESpellDictionary *dictionary = GNOME_SPELL_DICTIONARY (object);

	if (dictionary->manager)
		delete_pspell_manager (dictionary->manager);
	if (dictionary->config)
		delete_pspell_config (dictionary->config);

	GTK_OBJECT_CLASS (dictionary_parent_class)->destroy (object);
}

static void
impl_gnome_spell_dictionary_set_active (PortableServer_Servant  servant,
					const CORBA_boolean     active,
					CORBA_Environment      *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));

	dict->active = active;
}

static CORBA_boolean
impl_gnome_spell_dictionary_get_active (PortableServer_Servant  servant,
					CORBA_Environment      *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));

	return dict->active;
}

static void
impl_gnome_spell_dictionary_set_tag (PortableServer_Servant  servant,
				     const CORBA_char       *key,
				     const CORBA_char       *value,
				     CORBA_Environment      *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
									 
	DICT_DEBUG (printf ("Dictionary set: %s <- %s\n", key, value));
	pspell_config_replace (dict->config, key, value);
}

static void
raise_error (CORBA_Environment * ev, const gchar *s)
{
	GNOME_Spell_Dictionary_Error *exception;
	exception = GNOME_Spell_Dictionary_Error__alloc ();
		
	exception->error = CORBA_string_dup (s);
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GNOME_Spell_Dictionary_Error,
			     exception);
}

static void
update_manager (GNOMESpellDictionary *dict, CORBA_Environment * ev)
{
	PspellCanHaveError *err;

	g_assert (IS_GNOME_SPELL_DICTIONARY (dict));
	g_assert (dict->config);

	if (dict->changed) {

		DICT_DEBUG (printf ("Dictionary: creating new pspell manager\n"));

		if (dict->manager)
			delete_pspell_manager (dict->manager);
		err = new_pspell_manager (dict->config);
		if (pspell_error_number (err) != 0) {
			DICT_DEBUG (g_warning ("pspell error: %s\n", pspell_error_message (err)));
			dict->manager = NULL;
			raise_error (ev, pspell_error_message (err));
		} else {
			dict->manager = to_pspell_manager (err);
			dict->changed = FALSE;
		}
	}
}

static CORBA_boolean
impl_gnome_spell_dictionary_check_word (PortableServer_Servant servant, const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	gint pspell_result;
	CORBA_boolean result = CORBA_TRUE;
									 
	DICT_DEBUG (printf ("Dictionary check_word: %s\n", word));
	update_manager (dict, ev);
	if (dict->manager) {
		pspell_result = pspell_manager_check (dict->manager, word, strlen (word));
		if (pspell_result == 0)
			result = CORBA_FALSE;
		if (pspell_result == -1) {
			DICT_DEBUG (g_warning ("pspell error: %s\n", pspell_manager_error_message (dict->manager)));
			raise_error (ev, pspell_manager_error_message (dict->manager));
		}
	}

	return result || !dict->active;
}

static void
impl_gnome_spell_dictionary_add_word_to_session (PortableServer_Servant servant, const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));

	DICT_DEBUG (printf ("Dictionary add_word_to_session: %s\n", word));
	pspell_manager_add_to_session (dict->manager, word, strlen (word));
}

static void
impl_gnome_spell_dictionary_add_word_to_personal (PortableServer_Servant servant,
						  const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));

	DICT_DEBUG (printf ("Dictionary add_word_to_personal: %s\n", word));
	pspell_manager_add_to_personal (dict->manager, word, strlen (word));
	pspell_manager_save_all_word_lists (dict->manager);
}

static void
impl_gnome_spell_dictionary_set_correction (PortableServer_Servant servant,
					    const CORBA_char *word, const CORBA_char *replacement, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));

	DICT_DEBUG (printf ("Dictionary correction: %s <-- %s\n", word, replacement));
	pspell_manager_store_replacement (dict->manager, word, strlen (word), replacement, strlen (replacement));
	pspell_manager_save_all_word_lists (dict->manager);
}

static GNOME_Spell_StringSeq *
impl_gnome_spell_dictionary_get_suggestions (PortableServer_Servant servant,
					     const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary  *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	const PspellWordList  *suggestions;
	PspellStringEmulation *elements;
	GNOME_Spell_StringSeq *seq = NULL;
	gint i;
	DICT_DEBUG (printf ("Dictionary correction: %s\n", word));

	update_manager (dict, ev);
	if (dict->manager) {
		suggestions  = pspell_manager_suggest (dict->manager, word, strlen (word));
		seq          = GNOME_Spell_StringSeq__alloc ();
		seq->_length = pspell_word_list_size (suggestions);

		if (seq->_length == 0)
			return seq;

		seq->_buffer = CORBA_sequence_CORBA_string_allocbuf (seq->_length);
		elements     = pspell_word_list_elements (suggestions);
		for (i=0; i<seq->_length; i++)
			seq->_buffer [i] = CORBA_string_dup (pspell_string_emulation_next (elements));
		CORBA_sequence_set_release (seq, CORBA_TRUE);
		delete_pspell_string_emulation (elements);
	}

	return seq;
}

/*
 * If you want users to derive classes from your implementation
 * you need to support this method.
 */
POA_GNOME_Spell_Dictionary__epv *
gnome_spell_dictionary_get_epv (void)
{
	POA_GNOME_Spell_Dictionary__epv *epv;

	epv = g_new0 (POA_GNOME_Spell_Dictionary__epv, 1);

	epv->_set_active          = impl_gnome_spell_dictionary_set_active;
	epv->_get_active          = impl_gnome_spell_dictionary_get_active;
	epv->setTag               = impl_gnome_spell_dictionary_set_tag;
	epv->checkWord            = impl_gnome_spell_dictionary_check_word;
	epv->addWordToSession     = impl_gnome_spell_dictionary_add_word_to_session;
	epv->addWordToPersonal    = impl_gnome_spell_dictionary_add_word_to_personal;
	epv->getSuggestions       = impl_gnome_spell_dictionary_get_suggestions;
	epv->setCorrection        = impl_gnome_spell_dictionary_set_correction;

	return epv;
}

static void
init_dictionary_corba_class (void)
{
	dictionary_vepv.Bonobo_Unknown_epv         = bonobo_object_get_epv ();
	dictionary_vepv.GNOME_Spell_Dictionary_epv = gnome_spell_dictionary_get_epv ();
}

static void
dictionary_class_init (GNOMESpellDictionaryClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	dictionary_parent_class  = gtk_type_class (bonobo_object_get_type ());
	object_class->destroy    = dictionary_object_destroy;

	init_dictionary_corba_class ();
}

GtkType
gnome_spell_dictionary_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"Dictionary",
			sizeof (GNOMESpellDictionary),
			sizeof (GNOMESpellDictionaryClass),
			(GtkClassInitFunc) dictionary_class_init,
			(GtkObjectInitFunc) dictionary_object_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (bonobo_object_get_type (), &info);
	}

	return type;
}

BonoboObject *
dictionary_construct (BonoboObject *dictionary, GNOME_Spell_Dictionary corba_dictionary)
{
	CORBA_Environment ev;
	static int i;
	
	g_return_val_if_fail (dictionary != NULL, NULL);
	g_return_val_if_fail (IS_GNOME_SPELL_DICTIONARY (dictionary), NULL);
	g_return_val_if_fail (corba_dictionary != CORBA_OBJECT_NIL, NULL);

	if (!bonobo_object_construct (dictionary, (CORBA_Object) corba_dictionary))
		return NULL;

	CORBA_exception_init (&ev);
	CORBA_exception_free (&ev);

	return dictionary;
}

/*
 * This routine creates the ORBit CORBA server and initializes the
 * CORBA side of things
 */
static GNOME_Spell_Dictionary
create_dictionary (BonoboObject *dictionary)
{
	POA_GNOME_Spell_Dictionary *servant;
	CORBA_Environment ev;

	servant       = (POA_GNOME_Spell_Dictionary *) g_new0 (BonoboObjectServant, 1);
	servant->vepv = &dictionary_vepv;

	CORBA_exception_init (&ev);
	POA_GNOME_Spell_Dictionary__init ((PortableServer_Servant) servant, &ev);
	ORBIT_OBJECT_KEY (servant->_private)->object = NULL;

	if (ev._major != CORBA_NO_EXCEPTION){
		g_free (servant);
		CORBA_exception_free (&ev);
		return CORBA_OBJECT_NIL;
	}

	CORBA_exception_free (&ev);

	return (GNOME_Spell_Dictionary) bonobo_object_activate_servant (dictionary, servant);
}

BonoboObject *
gnome_spell_dictionary_new (void)
{
	BonoboObject *dictionary;
	GNOME_Spell_Dictionary corba_dictionary;

              dictionary = gtk_type_new (gnome_spell_dictionary_get_type ());
	corba_dictionary = create_dictionary (dictionary);

	if (corba_dictionary == CORBA_OBJECT_NIL) {
		bonobo_object_unref (BONOBO_OBJECT (dictionary));
		return NULL;
	}
	
	return dictionary_construct (dictionary, corba_dictionary);
}
